/*
 *	Device handling code
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/netpoll.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/list.h>
#include <linux/netfilter_bridge.h>

#include <asm/uaccess.h>
#include "br_private.h"

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP

extern char bridge_name[IFNAMSIZ];

typedef void (*add_member_cb)(unsigned char *, unsigned char *);
typedef void (*del_member_cb)(unsigned char *,unsigned char *);
typedef void (*clear_group_cb)(unsigned char *);
typedef void (*snoop_init_cb)(void);
typedef void (*snoop_deinit_cb)(void);
add_member_cb add_member=NULL;
del_member_cb del_member=NULL;
clear_group_cb clear_group=NULL;
snoop_init_cb snoop_init=NULL;
snoop_deinit_cb snoop_deinit=NULL;

void register_igmp_callbacks(add_member_cb fun1, del_member_cb fun2, clear_group_cb fun3)
{
	add_member = fun1;
	del_member = fun2;
	clear_group = fun3;	
}

void unregister_igmp_callbacks(void)
{	
	add_member = NULL;
	del_member = NULL;
	clear_group = NULL;		
}

void register_snoop_init_callback (snoop_init_cb funa,snoop_deinit_cb funb)
{
	snoop_init = funa;
	snoop_deinit = funb;
}	

void unregister_snoop_init_callback(void)
{
	snoop_init = NULL;
	snoop_deinit =NULL;
}

EXPORT_SYMBOL(register_snoop_init_callback);
EXPORT_SYMBOL(unregister_snoop_init_callback);
EXPORT_SYMBOL(register_igmp_callbacks);
EXPORT_SYMBOL(unregister_igmp_callbacks);

#endif

/* net device transmit always called with BH disabled */
netdev_tx_t br_dev_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);
	const unsigned char *dest = skb->data;
	struct net_bridge_fdb_entry *dst;
	struct net_bridge_mdb_entry *mdst;
	struct br_cpu_netstats *brstats = this_cpu_ptr(br->stats);
	struct net_bridge_port *pdst;
	br_get_dst_hook_t *get_dst_hook;

	rcu_read_lock();
#ifdef CONFIG_BRIDGE_NETFILTER
	if (skb->nf_bridge && (skb->nf_bridge->mask & BRNF_BRIDGED_DNAT)) {
		br_nf_pre_routing_finish_bridge_slow(skb);
		rcu_read_unlock();
		return NETDEV_TX_OK;
	}
#endif

	u64_stats_update_begin(&brstats->syncp);
	brstats->tx_packets++;
	brstats->tx_bytes += skb->len;
	u64_stats_update_end(&brstats->syncp);

	BR_INPUT_SKB_CB(skb)->brdev = dev;

	skb_reset_mac_header(skb);
	skb_pull(skb, ETH_HLEN);

	get_dst_hook = rcu_dereference(br_get_dst_hook);
	if (is_broadcast_ether_addr(dest))
		br_flood_deliver(br, skb);
	else if (is_multicast_ether_addr(dest)) {
		br_multicast_handle_hook_t *multicast_handle_hook = rcu_dereference(br_multicast_handle_hook);
		if (!__br_get(multicast_handle_hook, true, NULL, skb))
			goto out;

		if (unlikely(netpoll_tx_running(dev))) {
			br_flood_deliver(br, skb);
			goto out;
		}
		if (br_multicast_rcv(br, NULL, skb)) {
			kfree_skb(skb);
			goto out;
		}

		mdst = br_mdb_get(br, skb);
		if (mdst || BR_INPUT_SKB_CB_MROUTERS_ONLY(skb))
			br_multicast_deliver(mdst, skb);
		else
			br_flood_deliver(br, skb);
	} else {
		pdst = __br_get(get_dst_hook, NULL, NULL, &skb);
	    if (pdst) {
			if (!skb)
			   goto out;
			br_deliver(pdst, skb);
		}
        else {
           dst = __br_fdb_get(br, dest);
			if (dst)
				br_deliver(dst->dst, skb);
			else
                br_flood_deliver(br, skb);
           }
        }


out:
	rcu_read_unlock();
	return NETDEV_TX_OK;
}

static int br_dev_init(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);

	br->stats = alloc_percpu(struct br_cpu_netstats);
	if (!br->stats)
		return -ENOMEM;

	return 0;
}

static int br_dev_open(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);

	netdev_update_features(dev);
	netif_start_queue(dev);
	br_stp_enable_bridge(br);
	br_multicast_open(br);

	return 0;
}

static void br_dev_set_multicast_list(struct net_device *dev)
{
}

static int br_dev_stop(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);

	br_stp_disable_bridge(br);
	br_multicast_stop(br);

	netif_stop_queue(dev);

	return 0;
}

static struct rtnl_link_stats64 *br_get_stats64(struct net_device *dev,
						struct rtnl_link_stats64 *stats)
{
	struct net_bridge *br = netdev_priv(dev);
	struct br_cpu_netstats tmp, sum = { 0 };
	unsigned int cpu;

	for_each_possible_cpu(cpu) {
		unsigned int start;
		const struct br_cpu_netstats *bstats
			= per_cpu_ptr(br->stats, cpu);
		do {
			start = u64_stats_fetch_begin(&bstats->syncp);
			memcpy(&tmp, bstats, sizeof(tmp));
		} while (u64_stats_fetch_retry(&bstats->syncp, start));
		sum.tx_bytes   += tmp.tx_bytes;
		sum.tx_packets += tmp.tx_packets;
		sum.rx_bytes   += tmp.rx_bytes;
		sum.rx_packets += tmp.rx_packets;
	}

	stats->tx_bytes   = sum.tx_bytes;
	stats->tx_packets = sum.tx_packets;
	stats->rx_bytes   = sum.rx_bytes;
	stats->rx_packets = sum.rx_packets;

	return stats;
}

static int br_change_mtu(struct net_device *dev, int new_mtu)
{
	struct net_bridge *br = netdev_priv(dev);
	if (new_mtu < 68 || new_mtu > br_min_mtu(br))
		return -EINVAL;

	dev->mtu = new_mtu;

#ifdef CONFIG_BRIDGE_NETFILTER
	/* remember the MTU in the rtable for PMTU */
	dst_metric_set(&br->fake_rtable.dst, RTAX_MTU, new_mtu);
#endif

	return 0;
}

/* Allow setting mac address to any valid ethernet address. */
static int br_set_mac_address(struct net_device *dev, void *p)
{
	struct net_bridge *br = netdev_priv(dev);
	struct sockaddr *addr = p;

	if (!is_valid_ether_addr(addr->sa_data))
		return -EINVAL;

	spin_lock_bh(&br->lock);
	if (compare_ether_addr(dev->dev_addr, addr->sa_data)) {
		memcpy(dev->dev_addr, addr->sa_data, ETH_ALEN);
		br_fdb_change_mac_address(br, addr->sa_data);
		br_stp_change_bridge_id(br, addr->sa_data);
	}
	br->flags |= BR_SET_MAC_ADDR;
	spin_unlock_bh(&br->lock);

	return 0;
}

static void br_getinfo(struct net_device *dev, struct ethtool_drvinfo *info)
{
	strcpy(info->driver, "bridge");
	strcpy(info->version, BR_VERSION);
	strcpy(info->fw_version, "N/A");
	strcpy(info->bus_info, "N/A");
}

static netdev_features_t br_fix_features(struct net_device *dev,
	netdev_features_t features)
{
	struct net_bridge *br = netdev_priv(dev);

	return br_features_recompute(br, features);
}

#ifdef CONFIG_NET_POLL_CONTROLLER
static void br_poll_controller(struct net_device *br_dev)
{
}

static void br_netpoll_cleanup(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);
	struct net_bridge_port *p, *n;

	list_for_each_entry_safe(p, n, &br->port_list, list) {
		br_netpoll_disable(p);
	}
}

static int br_netpoll_setup(struct net_device *dev, struct netpoll_info *ni)
{
	struct net_bridge *br = netdev_priv(dev);
	struct net_bridge_port *p, *n;
	int err = 0;

	list_for_each_entry_safe(p, n, &br->port_list, list) {
		if (!p->dev)
			continue;

		err = br_netpoll_enable(p);
		if (err)
			goto fail;
	}

out:
	return err;

fail:
	br_netpoll_cleanup(dev);
	goto out;
}

int br_netpoll_enable(struct net_bridge_port *p)
{
	struct netpoll *np;
	int err = 0;

	np = kzalloc(sizeof(*p->np), GFP_KERNEL);
	err = -ENOMEM;
	if (!np)
		goto out;

	np->dev = p->dev;
	strlcpy(np->dev_name, p->dev->name, IFNAMSIZ);

	err = __netpoll_setup(np);
	if (err) {
		kfree(np);
		goto out;
	}

	p->np = np;

out:
	return err;
}

void br_netpoll_disable(struct net_bridge_port *p)
{
	struct netpoll *np = p->np;

	if (!np)
		return;

	p->np = NULL;

	/* Wait for transmitting packets to finish before freeing. */
	synchronize_rcu_bh();

	__netpoll_cleanup(np);
	kfree(np);
}

#endif

static int br_add_slave(struct net_device *dev, struct net_device *slave_dev)

{
	struct net_bridge *br = netdev_priv(dev);

	return br_add_if(br, slave_dev);
}

static int br_del_slave(struct net_device *dev, struct net_device *slave_dev)
{
	struct net_bridge *br = netdev_priv(dev);

	return br_del_if(br, slave_dev);
}

static const struct ethtool_ops br_ethtool_ops = {
	.get_drvinfo    = br_getinfo,
	.get_link	= ethtool_op_get_link,
};

static const struct net_device_ops br_netdev_ops = {
	.ndo_open		 = br_dev_open,
	.ndo_stop		 = br_dev_stop,
	.ndo_init		 = br_dev_init,
	.ndo_start_xmit		 = br_dev_xmit,
	.ndo_get_stats64	 = br_get_stats64,
	.ndo_set_mac_address	 = br_set_mac_address,
	.ndo_set_rx_mode	 = br_dev_set_multicast_list,
	.ndo_change_mtu		 = br_change_mtu,
	.ndo_do_ioctl		 = br_dev_ioctl,
#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_netpoll_setup	 = br_netpoll_setup,
	.ndo_netpoll_cleanup	 = br_netpoll_cleanup,
	.ndo_poll_controller	 = br_poll_controller,
#endif
	.ndo_add_slave		 = br_add_slave,
	.ndo_del_slave		 = br_del_slave,
	.ndo_fix_features        = br_fix_features,
};

static void br_dev_free(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);

	free_percpu(br->stats);
	free_netdev(dev);
}

static struct device_type br_type = {
	.name	= "bridge",
};

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
static int check_mac(char *s)
{
	char * accept = MAC_ACCEPT_CHAR;
	if(!s || !(*s))	return (-1);
	if ( strlen(s) == strspn(s, accept) )
		return 0;
	return (-1);
}

/* search device'name that matched */
/* called under bridge lock */
static struct net_bridge_port * search_device(struct net_bridge * br, char* name)
{
	struct net_bridge_port *p;
	list_for_each_entry(p, &br->port_list, list) {
		if (strcmp(p->dev->name, name) == 0 ){
			return p;
		}
	}
	return NULL;
}

static uint8_t * hex2dec(char *ch)
{
	if ( *ch >= '0' && *ch <= '9') *ch=*ch-'0';
	else if ( *ch >= 'A' && *ch <= 'F')  *ch=*ch-'A' +10;
	else if ( *ch >= 'a' && *ch <= 'f')  *ch=*ch-'a' +10;
	return ch;
}

static void split_MAC(unsigned char * mac_addr, char * token)
{
	char *macDelim = MAC_DELIM;
	char **pMAC = &token;
	char * macField_char[6];
	int i;
	for (i=0; i<6; ++i)
	{
		int j;
		char temp[2];
		macField_char[i] = strsep(pMAC, macDelim);
		/* copy each char byte and convert to dec number */
		for(j=0; j<2; ++j) {
			memcpy(&temp[j],macField_char[i]+j, sizeof(char));
			hex2dec(&temp[j]);
		}

		temp[0] = temp[0] << 4;
		*(mac_addr + i)= (temp[0]^temp[1]);
	}
}

/* called under bridge lock */
#ifdef CONFIG_BRIDGE_MLD
static int table_setmode(struct net_bridge *br, int type)
{
	switch(type)
	{
		case RT_MODE :
			br->mCastMode = 1;
			break;
		case BR_MODE :
			br->mCastMode = 2;
			break;
	}
	return 0;
}
#endif

/* called under bridge lock */
#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
static int table_setmf(struct net_bridge *br, char * name, int type)
{
	struct net_bridge_port *hit_port;
	hit_port = search_device(br, name);
	if (hit_port != NULL){
		if(type==ENABLE)
			atomic_set(&hit_port->multicast_forward, 1);
		else 
			atomic_set(&hit_port->multicast_forward, 0);
		return 0;
	}else
		return (-1);
}
#endif

/* called under bridge lock */
static int table_setsnoop(struct net_bridge *br, int type)
{
	switch(type)
	{
		case ENABLE :
			br->snooping = 1;
			if(snoop_init) 
				snoop_init();
			else 
			{
				printk("No snooping implementation. Please check !! \n");
				return (-1);
			}
			break;
		case DISABLE : 
			br->snooping = 0;
			if(snoop_deinit) 
				snoop_deinit();
			else
			{
				printk("No snooping implementation. Please check !! \n");
				return (-1);
			}
			break;
	}
	return 0;
}

/* set wireless identifier */
/* called under bridge lock */
static int table_setwl(struct net_bridge *br, char * name, int type)
{
	struct net_bridge_port *hit_port;
	hit_port = search_device(br, name);
	if (hit_port != NULL){
		if(type==ENABLE)
			atomic_set(&hit_port->wireless_interface, 1);
		else 
			atomic_set(&hit_port->wireless_interface, 0);
		return 0;
	}else
		return (-1);
}

#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
static void member_add_ordered(struct list_head *new, struct list_head *head, unsigned char * macaddr)
{
	struct port_member_mac *pmentry = NULL;

	list_for_each_entry(pmentry, head, list)
	{
		if(!memcmp(pmentry->member_mac,macaddr, 6))
			break;
	}

	__list_add(new, pmentry->list.prev, &pmentry->list);
}
#endif

#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
static void table_add(  struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr, int ip_version, uint32 *member_ip)
#else
static void table_add(  struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr)
#endif
{
	int found=0;
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	int found_member=0;
#endif
	//unsigned long flags;
	//spinlock_t lock;
	struct port_group_mac *pgentry;
	struct port_member_mac *pcentry;

#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	if(group_addr==NULL || member_addr==NULL || member_ip==NULL)
		return;	
#else
	if(group_addr==NULL || member_addr==NULL)
		return;	
#endif

	//spin_lock_irqsave(&lock,flags);

	//1. find old group if exist
	list_for_each_entry(pgentry, &p->igmp_group_list, list)
	{
		if(!memcmp(pgentry->grpmac,group_addr, 6))
		{
			found = 1;
			break;
		}
	}

	if(!found)	//create new group
	{
		pgentry = (struct port_group_mac *)kmalloc(sizeof(struct port_group_mac), GFP_ATOMIC);
		INIT_LIST_HEAD(&pgentry->list);
		INIT_LIST_HEAD(&pgentry->member_list);
		list_add(&pgentry->list, &p->igmp_group_list);
		memcpy(pgentry->grpmac , group_addr , 6);
		print_debug("brg : Create new group 0x%02x%02x%02x%02x%02x%02x\n", 
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5]);
	}
	//2. find old client mac if exist
	found = 0;
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	found_member = 0;
#endif
	list_for_each_entry(pcentry, &pgentry->member_list, list)
	{
		if(!memcmp(pcentry->member_mac,member_addr, 6))
		{	/* member already exist, do nothing ~*/
#ifdef CONFIG_BRIDGE_MLD
			pcentry->aging_time = groupMemberAgingTime;
#endif
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			found_member = 1;
			if(!memcmp(pcentry->member_ipaddr,member_ip, 16))
			{
				found = 1;
				break;
			}
#else
			found = 1;
			break;
#endif
		}
	}
	if(!found)
	{	/* member NOT exist, create NEW ONE and attached it to this group-mac linked list ~*/
		pcentry	= (struct port_member_mac *)kmalloc(sizeof(struct port_member_mac), GFP_ATOMIC);
		INIT_LIST_HEAD(&pcentry->list);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		if(found_member)
		{
			member_add_ordered(&pcentry->list, &pgentry->member_list, member_addr);
		}
		else
		{
#endif
		list_add(&pcentry->list, &pgentry->member_list);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		}
#endif		
		memcpy( pcentry->member_mac ,member_addr , 6);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		pcentry->ipVersion = ip_version;
		memcpy( pcentry->member_ipaddr ,member_ip , 16);
#endif
#ifdef CONFIG_BRIDGE_MLD
		pcentry->aging_time = groupMemberAgingTime;
#endif
		print_debug("brg : Added client mac 0x%02x%02x%02x%02x%02x%02x to group 0x%02x%02x%02x%02x%02x%02x\n", 
				pcentry->member_mac[0],pcentry->member_mac[1],pcentry->member_mac[2],pcentry->member_mac[3],pcentry->member_mac[4],pcentry->member_mac[5],
				pgentry->grpmac[0],pgentry->grpmac[1],pgentry->grpmac[2],pgentry->grpmac[3],pgentry->grpmac[4],pgentry->grpmac[5]
				);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		print_debug("brg : Added client ipaddr %08x-%08x-%08x-%08x\n", 
				pcentry->member_ipaddr[0],pcentry->member_ipaddr[1],pcentry->member_ipaddr[2],pcentry->member_ipaddr[3]);
#endif
	}
	//spin_unlock_irqrestore (&lock, flags);
}

/*
   1. find old group 
   2. find old client mac 
   3. if group is empty, delete group
   4. snooping : update the group port list 
 */
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
static void table_remove(struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr, int ip_version, uint32 *member_ip)
#else
static void table_remove(struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr)
#endif
{
	struct port_group_mac *pgentry;
	struct port_member_mac *pcentry;
	int found = 0;

	//0. sanity check
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	if(group_addr==NULL || member_addr==NULL || member_ip==NULL)
		return;
#else
	if(group_addr==NULL || member_addr==NULL)
		return;
#endif

	//1. find old group 
	list_for_each_entry(pgentry, &p->igmp_group_list, list)
	{
		if(!memcmp(pgentry->grpmac,group_addr, 6))
		{
			found = 1;
			break;
		}
	}
	if(!found)
	{
		print_debug("dbg : Can't delete 0x%02x%02x%02x%02x%02x%02x, group NOT FOUND.\n", 
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );
		return;
	}

	//2. find old client mac 
	found = 0;
	list_for_each_entry(pcentry, &pgentry->member_list, list)
	{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		if(!memcmp(pcentry->member_mac,member_addr, 6) && !memcmp(pcentry->member_ipaddr,member_ip, 16))
#else
		if(!memcmp(pcentry->member_mac,member_addr, 6))
#endif
		{	
			found = 1;
			break;
		}
	}

	if(found)
	{
		/* member to be deleted FOUND, DELETE IT ! */
		list_del(&pcentry->list);
		kfree(pcentry);
		print_debug("dbg : Delete client 0x%02x%02x%02x%02x%02x%02x in group 0x%02x%02x%02x%02x%02x%02x\n", 
				member_addr[0],member_addr[1],member_addr[2],member_addr[3],member_addr[4],member_addr[5],
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );	
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		print_debug("brg : Delete client ipaddr %08x-%08x-%08x-%08x\n", 
				member_ip[0],member_ip[1],member_ip[2],member_ip[3]);
#endif
	}else
	{	/* do nothing, just debug */
		print_debug("dbg : Can't delete client 0x%02x%02x%02x%02x%02x%02x, client NOT FOUND in group 0x%02x%02x%02x%02x%02x%02x\n", 
				member_addr[0],member_addr[1],member_addr[2],member_addr[3],member_addr[4],member_addr[5],
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );
	}

	//3. if group is empty, delete group
	if(list_empty(&pgentry->member_list))
	{
		list_del(&pgentry->member_list);
		list_del(&pgentry->list);
		kfree(pgentry);
		//remove group mac from port_list
		print_debug("dbg : Delete group 0x%02x%02x%02x%02x%02x%02x since its empty \n", 
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );	
		return;
	}
}

static int proc_read_alpha_multicast (char *buf, char **start, off_t offset,
		int len, int *eof, void *data)
{
	int count =0;
	struct net_bridge *br = (struct net_bridge *) data;
	struct net_bridge_port *p;
	struct port_group_mac *pgentry;
	struct port_member_mac *pmentry;	

	spin_lock_bh(&br->lock); // bridge lock
	printk( "**********************************************************************\n");
	printk( "* bridge name    : %s\n",br->dev->name);
	printk( "* mCastMode      : %d\n",br->mCastMode);
	printk( "* snooping       : %d\n",br->snooping);
	printk( "**********************************************************************\n");
	list_for_each_entry_rcu(p, &br->port_list, list) {
		printk( "* ==============================================================\n");
		printk( "* <%d> port name : %s\n", p->port_no, p->dev->name);
		printk( "* <%d> wireless_interface -> %d\n", p->port_no, atomic_read(&p->wireless_interface) );
#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
		printk( "* <%d> multicast_forward -> %d\n", p->port_no, atomic_read(&p->multicast_forward) );
#endif
		//traverse through all group list, list all the members inside 
		list_for_each_entry(pgentry, &p->igmp_group_list, list)
		{
			printk(" Group Mac  0x%02x%02x%02x%02x%02x%02x\n",pgentry->grpmac[0],
					pgentry->grpmac[1],
					pgentry->grpmac[2],
					pgentry->grpmac[3],
					pgentry->grpmac[4],
					pgentry->grpmac[5]);

			list_for_each_entry(pmentry, &pgentry->member_list, list)
			{
#ifdef CONFIG_BRIDGE_MLD
#ifdef CONFIG_BRIDGE_IGMP
				if(1)
#else
				if(IPV6_MULTICAST_MAC(pgentry->grpmac))
#endif
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
					printk("   membermac 0x%02x%02x%02x%02x%02x%02x\n   ipaddr %08x-%08x-%08x-%08x aging:%ds\n",pmentry->member_mac[0],
						pmentry->member_mac[1],
						pmentry->member_mac[2],
						pmentry->member_mac[3],
						pmentry->member_mac[4],
						pmentry->member_mac[5],
						pmentry->member_ipaddr[0],
						pmentry->member_ipaddr[1],
						pmentry->member_ipaddr[2],
						pmentry->member_ipaddr[3],
						pmentry->aging_time);
#else
					printk("   membermac 0x%02x%02x%02x%02x%02x%02x aging:%ds\n",pmentry->member_mac[0],
						pmentry->member_mac[1],
						pmentry->member_mac[2],
						pmentry->member_mac[3],
						pmentry->member_mac[4],
						pmentry->member_mac[5],
						pmentry->aging_time);
#endif
				else
#endif
				printk("   membermac 0x%02x%02x%02x%02x%02x%02x\n",pmentry->member_mac[0],
						pmentry->member_mac[1],
						pmentry->member_mac[2],
						pmentry->member_mac[3],
						pmentry->member_mac[4],
						pmentry->member_mac[5]);
			}
		}
		printk( "* ==============================================================\n");
	} // list_for_each_entry_rcu() - END
	printk( "**********************************************************************\n");
	spin_unlock_bh(&br->lock); // bridge unlock

	*eof = 1;
	return count;
}

static int proc_write_alpha_multicast (struct file *file, const char *buf,
		unsigned long count, void *data)
{
	int len = MESSAGE_LENGTH+1;
	char message[len];
	char *msgDelim = MESSAGE_DELIM;
	char *pmesg;
	char *action_token, *action;
	struct net_bridge *br;

	if(count > MESSAGE_LENGTH) {len = MESSAGE_LENGTH;}
	else {len = count; }
	if(copy_from_user(message, buf, len))
		return -EFAULT;
	message[len-1] = '\0';

	/* split input message that get from user space
	 * token[0] => action token --> add or remove
	 * token[1] => multicast group mac address
	 * token[2] => member MAC address of host
	 */
	pmesg = message ;

	action_token = strsep(&pmesg, msgDelim);

	br = (struct net_bridge *) data;

#ifdef CONFIG_BRIDGE_MLD
	/* ============================  set bridge mode =====================*/
	action = ACTION_RT_MODE;
	if (memcmp(action_token, action, sizeof(ACTION_RT_MODE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmode(br, RT_MODE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SET mode FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}
	
	action = ACTION_BR_MODE;
	if (memcmp(action_token, action, sizeof(ACTION_BR_MODE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmode(br, BR_MODE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SET mode FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}
#endif

#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
	/* ============================  set multicastforward  =====================*/
	action = ACTION_SET_MCASTFWD;
	if (memcmp(action_token, action, sizeof(ACTION_SET_MCASTFWD) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmf(br,pmesg, ENABLE) != 0){
			print_debug("[BR_IGMPP_PROC]->WARNING SETMF FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  unset multicastforward  ===================*/
	action = ACTION_UNSET_MCASTFWD;
	if ( memcmp(action_token, action, sizeof(ACTION_UNSET_MCASTFWD) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmf(br,pmesg, DISABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SETMF FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}
#endif
	
	/* ============================  set wireless enhance =====================*/
	action = ACTION_SET_ENHANCE;
	if (memcmp(action_token, action, sizeof(ACTION_SET_ENHANCE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setwl(br,pmesg, ENABLE) != 0){
			print_debug("[BR_IGMPP_PROC]->WARNING SETWL FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  unset wireless enhance  ===================*/
	action = ACTION_UNSET_ENHANCE;
	if ( memcmp(action_token, action, sizeof(ACTION_UNSET_ENHANCE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setwl(br,pmesg, DISABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SETWL FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  set snooping ============================*/
	action = ACTION_SET_SNOOPING;
	if (memcmp(action_token, action, sizeof(ACTION_SET_SNOOPING) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setsnoop(br, ENABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SET snooping FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  unset snooping ==========================*/
	action = ACTION_UNSET_SNOOPING;
	if ( memcmp(action_token, action, sizeof(ACTION_UNSET_SNOOPING) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setsnoop(br, DISABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING UNSET snooping FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  add - START =====================================*/
#ifndef CONFIG_BRIDGE_SNOOPING_IP_LIST
	action = ACTION_ADD;
	if ( memcmp(action_token, action, sizeof(ACTION_ADD) )== 0){
		/********** add - START of processing input string **********/
		char *token[2]={0,0};
		int i;
		unsigned char mac_addr[6];
		unsigned char grp_mac_addr[6];	
		struct net_bridge_fdb_entry *hit_fdb_entry;

		for(i=0; i<2; ++i)
			token[i] = strsep(&pmesg, msgDelim);

		/* Only accept MAC, split host MAC address */
		if ( check_mac(token[0]) == -1 || check_mac(token[1]) == -1)
		{
			print_debug(KERN_INFO "[BR_IGMPP_PROC]-> Host MAC address: %s,%s is illegal !!\n",
					(token[0])?(token[0]):"null",
					(token[1])?(token[1]):"null");
			goto proc_write_br_igmpp_out;
		}

		split_MAC(grp_mac_addr, token[0]);
		split_MAC(mac_addr, token[1]);

		print_debug("brg : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);

		spin_lock_bh(&br->lock); // bridge lock
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
		hit_fdb_entry = __br_fdb_get(br, mac_addr);

		if (hit_fdb_entry != NULL)
		{
			table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}

		rcu_read_unlock();
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		//do snoop if implemented in switch
		if(add_member) add_member(grp_mac_addr,mac_addr);

		goto proc_write_br_igmpp_out;
	}


	action = ACTION_REMOVE;
	if ( memcmp(action_token, action, sizeof(ACTION_REMOVE) ) == 0)
	{
		char *token[2]={0,0};
		int i;
		unsigned char mac_addr[6];
		struct net_bridge_fdb_entry *hit_fdb_entry;
		unsigned char grp_mac_addr[6];

		for(i=0; i<2; ++i)
			token[i] = strsep(&pmesg, msgDelim);

		/* Only accept MAC, split host MAC address */
		if (check_mac(token[0]) == -1 || check_mac(token[1]) == -1)
		{
			print_debug(KERN_INFO "[BR_IGMPP_PROC]-> Host MAC address: %s,%s is illegal !!\n",
					(token[0])?(token[0]):"null",
					(token[1])?(token[1]):"null");
			goto proc_write_br_igmpp_out;
		}

		split_MAC(grp_mac_addr, token[0]);
		split_MAC(mac_addr, token[1]);

		print_debug("brg : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);

		spin_lock_bh(&br->lock); // bridge lock			
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
		hit_fdb_entry = __br_fdb_get(br, mac_addr);

		if (hit_fdb_entry != NULL)
		{
			table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}
		rcu_read_unlock();
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out

		//do snoop if implemented in switch
		if(del_member) del_member(grp_mac_addr, mac_addr);

		goto proc_write_br_igmpp_out;
	}
#endif
	/* ============================= remove - END ======================================*/

proc_write_br_igmpp_out:
	return len;
}

#ifdef CONFIG_BRIDGE_MLD
void write_alpha_multicast (unsigned char op, struct net_bridge *br, unsigned char *dest, struct sk_buff *skb)
{
	unsigned char *src;
	src=(unsigned char*)(skb_mac_header(skb)+ETH_ALEN);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	struct ethhdr *ethh = eth_hdr(skb);
	struct iphdr *iph;
	struct ipv6hdr *ipv6h;
#endif
	struct net_bridge_fdb_entry *hit_fdb_entry;
	unsigned char mac_addr[6];
	unsigned char grp_mac_addr[6];
	memcpy(grp_mac_addr, dest, 6);
	memcpy(mac_addr, src, 6);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	uint32 ip_addr[4]={0,0,0,0};
#endif
	
	if (op == 1) /* add */
	{
		print_debug("brg : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);
		
		spin_lock_bh(&br->lock); // bridge lock
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
		hit_fdb_entry = __br_fdb_get(br, mac_addr);

		if (hit_fdb_entry != NULL)
		{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			if(ethh->h_proto == htons(ETH_P_IP))
			{
				iph = ip_hdr(skb);
				ip_addr[0] = ntohl(iph->saddr);
				table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION4, ip_addr);
			}
			else if(ethh->h_proto == htons(ETH_P_IPV6))
			{
				ipv6h = ipv6_hdr(skb);
				ip_addr[0] = ntohl(ipv6h->saddr.s6_addr32[0]);
				ip_addr[1] = ntohl(ipv6h->saddr.s6_addr32[1]);
				ip_addr[2] = ntohl(ipv6h->saddr.s6_addr32[2]);
				ip_addr[3] = ntohl(ipv6h->saddr.s6_addr32[3]);
				table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION6, ip_addr);
			}
#else
			table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
#endif
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}

		rcu_read_unlock();
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		//do snoop if implemented in switch
		if(add_member) add_member(grp_mac_addr,mac_addr);

		goto write_br_igmpp_out;	
	}
	else if (op == 2) /* delete */
	{
		print_debug("brg : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);

		spin_lock_bh(&br->lock); // bridge lock			
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
		hit_fdb_entry = __br_fdb_get(br, mac_addr);

		if (hit_fdb_entry != NULL)
		{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			if(ethh->h_proto == htons(ETH_P_IP))
			{
				iph = ip_hdr(skb);
				ip_addr[0] = ntohl(iph->saddr);
				table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION4, ip_addr);
			}
			else if(ethh->h_proto == htons(ETH_P_IPV6))
			{
				ipv6h = ipv6_hdr(skb);
				ip_addr[0] = ntohl(ipv6h->saddr.s6_addr32[0]);
				ip_addr[1] = ntohl(ipv6h->saddr.s6_addr32[1]);
				ip_addr[2] = ntohl(ipv6h->saddr.s6_addr32[2]);
				ip_addr[3] = ntohl(ipv6h->saddr.s6_addr32[3]);
				table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION6, ip_addr);
			}
#else
			table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
#endif
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}
		rcu_read_unlock();
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out

		//do snoop if implemented in switch
		if(del_member) del_member(grp_mac_addr, mac_addr);

		goto write_br_igmpp_out;
	}

write_br_igmpp_out:
	return;
}

void br_mCastAgingTimerExpired(unsigned long arg)
{
	struct net_bridge *br = (struct net_bridge*) arg;
	struct net_bridge_port *p;
	struct port_group_mac *pgentry;
	struct port_member_mac *pmentry;
	struct port_group_mac *next_pgentry;
	struct port_member_mac *next_pmentry;
		
	spin_lock_bh(&br->lock); // bridge lock
	list_for_each_entry_rcu(p, &br->port_list, list) {
		list_for_each_entry_safe(pgentry, next_pgentry, &p->igmp_group_list, list)
		{
#ifndef CONFIG_BRIDGE_IGMP
			if(IPV6_MULTICAST_MAC(pgentry->grpmac))
			{
#endif
				list_for_each_entry_safe(pmentry, next_pmentry, &pgentry->member_list, list)
				{
					/* member to be deleted TIMEOUT, DELETE IT ! */
					if(pmentry->aging_time <= 0)
					{
						/* member to be deleted FOUND, DELETE IT ! */
						print_debug("dbg : Delete client 0x%02x%02x%02x%02x%02x%02x in group 0x%02x%02x%02x%02x%02x%02x\n", 
							pmentry->member_mac[0],pmentry->member_mac[1],pmentry->member_mac[2],pmentry->member_mac[3],pmentry->member_mac[4],pmentry->member_mac[5],
							pgentry->grpmac[0],pgentry->grpmac[1],pgentry->grpmac[2],pgentry->grpmac[3],pgentry->grpmac[4],pgentry->grpmac[5] );	
						list_del(&pmentry->list);
						kfree(pmentry);
					}
					else
						pmentry->aging_time = pmentry->aging_time - 1;
				}
#ifndef CONFIG_BRIDGE_IGMP
			}
#endif
			// if group is empty, delete group
			if(list_empty(&pgentry->member_list))
			{
				//remove group mac from port_list
				print_debug("dbg : Delete group 0x%02x%02x%02x%02x%02x%02x since its empty \n", 
					pgentry->grpmac[0],pgentry->grpmac[1],pgentry->grpmac[2],pgentry->grpmac[3],pgentry->grpmac[4],pgentry->grpmac[5] );	
				list_del(&pgentry->member_list);
				list_del(&pgentry->list);
				kfree(pgentry);
			}
		}
	}

	spin_unlock_bh(&br->lock); // bridge unlock
	mod_timer(&br->mCastAgingtimer, jiffies+HZ);
	
	return;
}

int32 getGroupNum(struct net_bridge *br, uint32 ipVersion)
{
	int32 groupCnt=0;
	struct net_bridge_port *p;
	struct port_group_mac *pgentry;
	
#ifdef CONFIG_BRIDGE_IGMP_QUERY	
	if(ipVersion ==IP_VERSION4)
	{
		groupCnt=0;
		
		spin_lock_bh(&br->lock); // bridge lock
		list_for_each_entry_rcu(p, &br->port_list, list) {
			list_for_each_entry(pgentry, &p->igmp_group_list, list)
			{
				if(MULTICAST_MAC(pgentry->grpmac))
				{
					groupCnt++; 
				}
			}
		}
		spin_unlock_bh(&br->lock); // bridge unlock
	}
	else if(ipVersion ==IP_VERSION6)
#else
	if(ipVersion ==IP_VERSION6)
#endif
	{
		groupCnt=0;
		
		spin_lock_bh(&br->lock); // bridge lock
		list_for_each_entry_rcu(p, &br->port_list, list) {
			list_for_each_entry(pgentry, &p->igmp_group_list, list)
			{
				if(IPV6_MULTICAST_MAC(pgentry->grpmac))
				{
					groupCnt++; 
				}
			}
		}
		spin_unlock_bh(&br->lock); // bridge unlock
	}
	
	return groupCnt;
}
#endif

#endif

void br_dev_setup(struct net_device *dev)
{
#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	char alpha_proc_name[32];
#endif
	struct net_bridge *br = netdev_priv(dev);

	random_ether_addr(dev->dev_addr);
	ether_setup(dev);

	dev->netdev_ops = &br_netdev_ops;
	dev->destructor = br_dev_free;
	SET_ETHTOOL_OPS(dev, &br_ethtool_ops);
	SET_NETDEV_DEVTYPE(dev, &br_type);
	dev->tx_queue_len = 0;
	dev->priv_flags = IFF_EBRIDGE;

	dev->features = NETIF_F_SG | NETIF_F_FRAGLIST | NETIF_F_HIGHDMA |
			NETIF_F_GSO_MASK | NETIF_F_HW_CSUM | NETIF_F_LLTX |
			NETIF_F_NETNS_LOCAL | NETIF_F_HW_VLAN_TX;
	dev->hw_features = NETIF_F_SG | NETIF_F_FRAGLIST | NETIF_F_HIGHDMA |
			   NETIF_F_GSO_MASK | NETIF_F_HW_CSUM |
			   NETIF_F_HW_VLAN_TX;

	br->dev = dev;
	spin_lock_init(&br->lock);
	INIT_LIST_HEAD(&br->port_list);
	spin_lock_init(&br->hash_lock);

	br->bridge_id.prio[0] = 0x80;
	br->bridge_id.prio[1] = 0x00;

	memcpy(br->group_addr, br_group_address, ETH_ALEN);

	br->stp_enabled = BR_NO_STP;
	br->group_fwd_mask = BR_GROUPFWD_DEFAULT;

	br->designated_root = br->bridge_id;
	br->bridge_max_age = br->max_age = 20 * HZ;
	br->bridge_hello_time = br->hello_time = 2 * HZ;
	br->bridge_forward_delay = br->forward_delay = 15 * HZ;
	br->ageing_time = 300 * HZ;

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	snprintf(alpha_proc_name, sizeof(alpha_proc_name), "alpha/multicast_%s", bridge_name);
	br->alpha_multicast_proc = create_proc_entry (alpha_proc_name, 0644, NULL);
	if (br->alpha_multicast_proc == NULL)
	{
		printk("create proc FAILED %s\n", bridge_name);
		return ERR_PTR(-ENOMEM);
	}		
	br->alpha_multicast_proc->data = (void*)br;
	br->alpha_multicast_proc->read_proc = proc_read_alpha_multicast;
	br->alpha_multicast_proc->write_proc = proc_write_alpha_multicast;
	br->snooping = 0;
#endif
#ifdef CONFIG_BRIDGE_MLD
	br->mCastMode = 0;
#endif	
	br_netfilter_rtable_init(br);
	br_stp_timer_init(br);
	br_multicast_init(br);
}
