/*
 *	Handle incoming frames
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/netfilter_bridge.h>
#include <linux/export.h>
#include "br_private.h"

#ifdef CONFIG_NET_PPPOE_IPV6_PTHROUGH
extern int ipv6_pt_enable;
extern int pppoe_pt_enable;
extern int pt_dmac_search(unsigned char *dmac);
#endif

#ifdef CONFIG_BRIDGE_IGMP
#include <linux/igmp.h>
#endif

#ifdef CONFIG_BRIDGE_MLD
#include <linux/in6.h>
#include <net/mld.h>
#include <linux/icmpv6.h>

#define IPV6_ROUTER_ALTER_OPTION 0x05020000
#define HOP_BY_HOP_OPTIONS_HEADER 0
#define ROUTING_HEADER 43
#define FRAGMENT_HEADER 44
#define DESTINATION_OPTION_HEADER 60

#define PIM_PROTOCOL 103
#define MOSPF_PROTOCOL 89
#define TCP_PROTOCOL 6
#define UDP_PROTOCOL 17
#define NO_NEXT_HEADER 59
#define ICMP_PROTOCOL 58

#define MLD_QUERY 130
#define MLDV1_REPORT 131
#define MLDV1_DONE 132
#define MLDV2_REPORT 143

#define IS_IPV6_PIM_ADDR(ipv6addr) ((ipv6addr[0] == 0xFF020000)&&(ipv6addr[1] == 0x00000000)&&(ipv6addr[2] == 0x00000000)&&(ipv6addr[3] ==0x0000000D))
#define IS_IPV6_MOSPF_ADDR1(ipv6addr) ((ipv6addr[0] == 0xFF020000)&&(ipv6addr[1] == 0x00000000)&&(ipv6addr[2] == 0x00000000)&&(ipv6addr[3] ==0x00000005))
#define IS_IPV6_MOSPF_ADDR2(ipv6addr) ((ipv6addr[0] == 0xFF020000)&&(ipv6addr[1] == 0x00000000)&&(ipv6addr[2] == 0x00000000)&&(ipv6addr[3] ==0x00000006))

extern void write_alpha_multicast (unsigned char op, struct net_bridge *br, unsigned char *dest, struct sk_buff *skb);
#endif

/* Bridge group multicast address 802.1d (pg 51). */
const u8 br_group_address[ETH_ALEN] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x00 };

/* Hook for brouter */
br_should_route_hook_t __rcu *br_should_route_hook __read_mostly;
EXPORT_SYMBOL(br_should_route_hook);

/* Hook for external Multicast handler */
br_multicast_handle_hook_t __rcu *br_multicast_handle_hook __read_mostly;
EXPORT_SYMBOL_GPL(br_multicast_handle_hook);

/* Hook for external forwarding logic */
br_get_dst_hook_t __rcu *br_get_dst_hook __read_mostly;
EXPORT_SYMBOL_GPL(br_get_dst_hook);

#ifdef CONFIG_BRIDGE_GUEST_ZONE
typedef int (*gz_skb_mark_cb)(const struct net_bridge *, struct sk_buff *);
gz_skb_mark_cb gz_skb_mark=NULL;

void register_gz_skb_mark_init_callback(gz_skb_mark_cb funa)
{
	gz_skb_mark = funa;
}	

void unregister_gz_skb_mark_init_callback(void)
{
	gz_skb_mark = NULL;
}

EXPORT_SYMBOL(register_gz_skb_mark_init_callback);
EXPORT_SYMBOL(unregister_gz_skb_mark_init_callback);
#endif

static int br_pass_frame_up(struct sk_buff *skb)
{
	struct net_device *indev, *brdev = BR_INPUT_SKB_CB(skb)->brdev;
	struct net_bridge *br = netdev_priv(brdev);
	struct br_cpu_netstats *brstats = this_cpu_ptr(br->stats);

	u64_stats_update_begin(&brstats->syncp);
	brstats->rx_packets++;
	brstats->rx_bytes += skb->len;
	u64_stats_update_end(&brstats->syncp);

#ifdef CONFIG_BRIDGE_GUEST_ZONE
	if (gz_skb_mark)
	{
		if (!gz_skb_mark(br, skb))
		{
			kfree_skb(skb);
			return;
		}
	}
#endif
	
	indev = skb->dev;
	skb->dev = brdev;

	return BR_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN, skb, indev, NULL,
		       netif_receive_skb);
}

#ifdef CONFIG_BRIDGE_IGMP
int ConvertMulticatIPtoMacAddr(__u32 group, unsigned char *gmac)
{
	__u32 u32tmp, tmp;
	int i;
	int ret=1;

	if(group == 0xEFFFFFFA || group == 0xEFFFFC84 || (group&0xFFFFFF00)==0xE0000000)
	{
		//upnp:239.255.255.250, nec adp deamon:239.255.252.132, 224.0.0.0/24
		ret = 0;
	}
	
	u32tmp = group & 0x007FFFFF;
	gmac[0]=0x01; gmac[1]=0x00; gmac[2]=0x5e;
	for (i=5; i>=3; i--) {
		tmp=u32tmp&0xFF;
		gmac[i]=tmp;
		u32tmp >>= 8;
	}
	return ret;
}

//#define DBG_IGMP	//enable it to debug igmp check
//#define NIPQUAD(addr) \
        ((unsigned char *)&addr)[0], \
        ((unsigned char *)&addr)[1], \
        ((unsigned char *)&addr)[2], \
        ((unsigned char *)&addr)[3]

static char igmp_type_check(struct sk_buff *skb, unsigned char *gmac,unsigned int *gIndex,unsigned int *moreFlag)
{
	struct iphdr *iph;
	__u8 hdrlen;
	struct igmphdr *igmph;
	int i;
	unsigned int groupAddr=0;// add for fit igmp v3
	*moreFlag=0;

	iph=(struct iphdr *)skb_network_header(skb);
	hdrlen = iph->ihl << 2;
	if ((iph->version != 4) &&  (hdrlen < 20))
		return -1;
	if (ip_fast_csum((u8 *)iph, iph->ihl) != 0)
		return -1;
	{ /* check the length */
		__u32 len = ntohs(iph->tot_len);
		if (skb->len < len || len < hdrlen)
			return -1; 
	}
	/* parsing the igmp packet */
	igmph = (struct igmphdr *)((u8*)iph+hdrlen);
	
	if ((igmph->type==IGMP_HOST_MEMBERSHIP_REPORT) ||
		(igmph->type==IGMPV2_HOST_MEMBERSHIP_REPORT)) 
	{
		groupAddr = ntohl(igmph->group);
		if(!IN_MULTICAST(groupAddr))
		{			
			return -1;
		}
		
		if(ConvertMulticatIPtoMacAddr(groupAddr, gmac))
			return 1; /* report and add it */
		else
			return -1;
	}
	else if (igmph->type==IGMPV3_HOST_MEMBERSHIP_REPORT)
	{ 
		//for support igmp v3   	
		struct igmpv3_report *igmpv3report=(struct igmpv3_report * )igmph;
		struct igmpv3_grec	*igmpv3grec=NULL; 
		//printk("%s:%d,*gIndex is %d,igmpv3report->ngrec is %d\n",__FUNCTION__,__LINE__,*gIndex,igmpv3report->ngrec);
		if(*gIndex>=ntohs(igmpv3report->ngrec))
		{
			*moreFlag=0;
			return -1;
		}
		for(i=0;i<ntohs(igmpv3report->ngrec);i++)
		{
			if(i==0)
			{
				igmpv3grec = (struct igmpv3_grec *)(&(igmpv3report->grec)); /*first igmp group record*/
			}
			else
			{
				igmpv3grec=(struct igmpv3_grec *)((unsigned char*)igmpv3grec+8+ntohs(igmpv3grec->grec_nsrcs)*4+(igmpv3grec->grec_auxwords)*4);
				
			}
			
			if(i!=*gIndex)
			{	
				continue;
			}
			if(i==(ntohs(igmpv3report->ngrec)-1))
			{
				/*last group record*/
				*moreFlag=0;
			}
			else
			{
				*moreFlag=1;
			}
			
			/*gIndex move to next group*/
			*gIndex=*gIndex+1;	
			groupAddr= ntohl(igmpv3grec->grec_mca);
			//printk("%s:%d,groupAddr is %d.%d.%d.%d\n",__FUNCTION__,__LINE__,NIPQUAD(groupAddr));
			if(!IN_MULTICAST(groupAddr))
			{			
				return -1;
			}
			
			if(!ConvertMulticatIPtoMacAddr(groupAddr, gmac))
				return -1;
			if(((igmpv3grec->grec_type == IGMPV3_CHANGE_TO_INCLUDE) || (igmpv3grec->grec_type == IGMPV3_MODE_IS_INCLUDE))&& (ntohs(igmpv3grec->grec_nsrcs)==0))
			{	
				return 2; /* leave and delete it */	
			}
			else if((igmpv3grec->grec_type == IGMPV3_CHANGE_TO_EXCLUDE) ||
				(igmpv3grec->grec_type == IGMPV3_MODE_IS_EXCLUDE) ||
				(igmpv3grec->grec_type == IGMPV3_ALLOW_NEW_SOURCES))
			{
				return 1;
			}
			else
			{
				return 1; //unknown igmp report is regarded as join
			}
			
			return -1;
		}
		
		/*avoid dead loop in case of initial gIndex is too big*/
		if(i>=(ntohs(igmpv3report->ngrec)-1))
		{
			/*last group record*/
			*moreFlag=0;
			return -1;
		}

	}
	else if (igmph->type==IGMP_HOST_LEAVE_MESSAGE){
		groupAddr = ntohl(igmph->group);
		if(!IN_MULTICAST(groupAddr))
		{			
				return -1;
		}
		
		if(ConvertMulticatIPtoMacAddr(groupAddr, gmac))
			return 2; /* leave and delete it */
		else
			return -1;
	}	
	
	return -1;
}
#endif

#ifdef CONFIG_BRIDGE_MLD
int getIpv6TransportProtocol(struct ipv6hdr* ipv6h)
{
	unsigned char *ptr=NULL;
	unsigned char *startPtr=NULL;
	unsigned char *lastPtr=NULL;
	unsigned char nextHeader=0;
	unsigned short extensionHdrLen=0;

	unsigned char  optionDataLen=0;
	unsigned char  optionType=0;
	unsigned int ipv6RAO=0;
	unsigned int ipv6addr[4] = {0};

	if(ipv6h==NULL)
	{
		return -1;
	}

	if(ipv6h->version!=6)
	{
		return -1;
	}

	startPtr= (unsigned char *)ipv6h;
	lastPtr=startPtr+sizeof(struct ipv6hdr)+ntohs(ipv6h->payload_len);
	nextHeader= ipv6h ->nexthdr;
	ptr=startPtr+sizeof(struct ipv6hdr);

	ipv6addr[0] = ntohl(ipv6h->daddr.s6_addr32[0]);
	ipv6addr[1] = ntohl(ipv6h->daddr.s6_addr32[1]);
	ipv6addr[2] = ntohl(ipv6h->daddr.s6_addr32[2]);
	ipv6addr[3] = ntohl(ipv6h->daddr.s6_addr32[3]);

	while(ptr<lastPtr)
	{
		switch(nextHeader)
		{
			case HOP_BY_HOP_OPTIONS_HEADER:
				/*parse hop-by-hop option*/
				nextHeader=ptr[0];
				extensionHdrLen=((uint16)(ptr[1])+1)*8;
				ptr=ptr+2;

				while(ptr<(startPtr+extensionHdrLen+sizeof(struct ipv6hdr)))
				{
					optionType=ptr[0];
					/*pad1 option*/
					if(optionType==0)
					{
						ptr=ptr+1;
						continue;
					}

					/*padN option*/
					if(optionType==1)
					{
						optionDataLen=ptr[1];
						ptr=ptr+optionDataLen+2;
						continue;
					}

					/*router altert option*/
					if(ntohl(*(uint32 *)(ptr))==IPV6_ROUTER_ALTER_OPTION)
					{
						ipv6RAO=IPV6_ROUTER_ALTER_OPTION;
						ptr=ptr+4;
						continue;
					}

					/*other TLV option*/
					if((optionType!=0) && (optionType!=1))
					{
						optionDataLen=ptr[1];
						ptr=ptr+optionDataLen+2;
						continue;
					}
				}

				break;

			case ROUTING_HEADER:
				nextHeader=ptr[0];
				extensionHdrLen=((uint16)(ptr[1])+1)*8;
				ptr=ptr+extensionHdrLen;
				break;

			case FRAGMENT_HEADER:
				nextHeader=ptr[0];
				ptr=ptr+8;
				break;

			case DESTINATION_OPTION_HEADER:
				nextHeader=ptr[0];
				extensionHdrLen=((uint16)(ptr[1])+1)*8;
				ptr=ptr+extensionHdrLen;
				break;

			case ICMP_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				if((ptr[0]==MLD_QUERY) ||(ptr[0]==MLDV1_REPORT) ||(ptr[0]==MLDV1_DONE) ||(ptr[0]==MLDV2_REPORT))
				{
					return ICMP_PROTOCOL;

				}
				break;

			case PIM_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				if(IS_IPV6_PIM_ADDR(ipv6addr))
				{
					return PIM_PROTOCOL;
				}

				break;

			case MOSPF_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				if(IS_IPV6_MOSPF_ADDR1(ipv6addr) || IS_IPV6_MOSPF_ADDR2(ipv6addr))
				{
					return MOSPF_PROTOCOL;
				}
				break;

			case TCP_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				return TCP_PROTOCOL;

				break;

			case UDP_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				return UDP_PROTOCOL;

				break;

			default:
				/*not ipv6 multicast protocol*/
				return -1;
				break;
		}
	}
	return -1;
}

/*Convert  MultiCatst IPV6_Addr to MAC_Addr*/
void CIPV6toMac(unsigned char* icmpv6_McastAddr, unsigned char *gmac )
{
	/*ICMPv6 valid addr 2^32 -1*/
	gmac[0] = 0x33;
	gmac[1] = 0x33;
	gmac[2] = icmpv6_McastAddr[12];
	gmac[3] = icmpv6_McastAddr[13];
	gmac[4] = icmpv6_McastAddr[14];
	gmac[5] = icmpv6_McastAddr[15];			
}

//#define DBG_ICMPv6	//enable it to debug icmpv6 check

//add mld v2 support
char ICMPv6_check(struct sk_buff *skb , unsigned char *gmac, unsigned int *gIndex,unsigned int *moreFlag)
{
	struct ipv6hdr *ipv6h;
	char* protoType;
	int i = 0;
	*moreFlag=0;

	/* check IPv6 header information */
	//ipv6h = skb->nh.ipv6h;
	ipv6h = (struct ipv6hdr *)skb_network_header(skb);
	if(ipv6h->version != 6){
		//printk("ipv6h->version != 6\n");
		return -1;
	}
 
	/*Next header: IPv6 hop-by-hop option (0x00)*/
	if(ipv6h->nexthdr == 0) 
	{
		protoType = (unsigned char*)( (unsigned char*)ipv6h + sizeof(struct ipv6hdr) );
	}
	else
	{
		//printk("ipv6h->nexthdr != 0\n");
		return -1;
	}

	if(protoType[0] == 0x3a)
	{
		//printk("recv icmpv6 packet\n");
		struct icmp6hdr* icmpv6h = (struct icmp6hdr*)(protoType + 8);
		unsigned char *icmpv6_McastAddr;
 
		if(icmpv6h->icmp6_type == ICMPV6_MGM_REPORT)
		{
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8);
			
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (Multicast listener report) \n",icmpv6h->icmp6_type);
#endif
			if(icmpv6_McastAddr[0] != 0xFF)
				return -1;
			CIPV6toMac(icmpv6_McastAddr, gmac);
#ifdef DBG_ICMPv6
			printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
				icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
				icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
				icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
				icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
 
			printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
				gmac[0],gmac[1],gmac[2],
				gmac[3],gmac[4],gmac[5]);
#endif
			return 1;//add

		}
		else if(icmpv6h->icmp6_type == ICMPV6_MLD2_REPORT)
		{
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8 + 4);
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (Multicast listener report v2) \n",icmpv6h->icmp6_type);
#endif
        	struct mld2_report *mldv2report = (struct mld2_report * )icmpv6h;
           	struct mld2_grec *mldv2grec = NULL;
#ifdef DBG_ICMPv6
			printk("%s:%d,*gIndex is %d,mldv2report->ngrec is %d\n",__FUNCTION__,__LINE__,*gIndex, ntohs(mldv2report->mld2r_ngrec));
#endif
			if(*gIndex >= ntohs(mldv2report->mld2r_ngrec))
			{
				*moreFlag=0;
				return -1;
			}

			for(i=0;i< ntohs(mldv2report->mld2r_ngrec);i++)
			{
				if(i==0)
				{
					mldv2grec = (struct mld2_grec *)(&(mldv2report->mld2r_grec)); /*first igmp group record*/
				}
				else
				{
					mldv2grec = (struct mld2_grec *)((unsigned char*)mldv2grec+4+16+ntohs(mldv2grec->grec_nsrcs)*16+(mldv2grec->grec_auxwords)*4);
				}

				if(i!=*gIndex)
				{
					continue;
				}

				if(i==(ntohs(mldv2report->mld2r_ngrec)-1))
				{
					/*last group record*/
					*moreFlag=0;
				}
				else
				{
					*moreFlag=1;
				}

				/*gIndex move to next group*/
				*gIndex=*gIndex+1;

				icmpv6_McastAddr = (unsigned char *)&mldv2grec->grec_mca;
				//printk("%s:%d,groupAddr is %d.%d.%d.%d\n",__FUNCTION__,__LINE__,NIPQUAD(groupAddr));
				if(icmpv6_McastAddr[0] != 0xFF)
					return -1;
				if(icmpv6_McastAddr[0] == 0xFF && icmpv6_McastAddr[1]==0x02) //flooding ff02::/16
				{
#ifdef DBG_ICMPv6
					printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
						icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
						icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
						icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
						icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
#endif
					return -1;
				}
				CIPV6toMac(icmpv6_McastAddr, gmac);
#ifdef DBG_ICMPv6
				printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
					icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
					icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
					icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
					icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
 
				printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
					gmac[0],gmac[1],gmac[2],
					gmac[3],gmac[4],gmac[5]);
 
				printk("grec_type = %d, src_num = %d\n", mldv2grec->grec_type, mldv2grec->grec_nsrcs);
#endif
 
				if(((mldv2grec->grec_type== MLD2_CHANGE_TO_INCLUDE) || (mldv2grec->grec_type == MLD2_MODE_IS_INCLUDE))&& (ntohs(mldv2grec->grec_nsrcs)==0))
				{
					return 2; /* leave and delete it */
				}
				else if((mldv2grec->grec_type == MLD2_CHANGE_TO_EXCLUDE) ||
						(mldv2grec->grec_type == MLD2_MODE_IS_EXCLUDE) ||
						(mldv2grec->grec_type == MLD2_ALLOW_NEW_SOURCES))
				{
					return 1;
				}
				else
				{
					return 1; //unknown mld report is regarded as join
				}
 
				return -1;
			}
 
			/*avoid dead loop in case of initial gIndex is too big*/
			if(i>=(ntohs(mldv2report->mld2r_ngrec)-1))
			{
				/*last group record*/
				*moreFlag=0;
				return -1;
			}
		}
		else if(icmpv6h->icmp6_type == ICMPV6_MGM_REDUCTION)
		{
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8 );
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (Multicast listener done ) \n",icmpv6h->icmp6_type);
#endif
			if(icmpv6_McastAddr[0] != 0xFF)
				return -1;
			CIPV6toMac(icmpv6_McastAddr, gmac);
#ifdef DBG_ICMPv6
			printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
				icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
				icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
				icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
				icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
 
			printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
				gmac[0],gmac[1],gmac[2],
				gmac[3],gmac[4],gmac[5]);
#endif
			return 2;//del
		}
		else
		{
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (unknow type)\n",icmpv6h->icmp6_type);
#endif
			return -1;
		}
	}
	else
	{
		//printk("protoType[0] != 0x3a\n");
		return -1;//not icmpv6 type
	}
 
	return -1;
}
#endif

/* note: already called with rcu_read_lock */
int br_handle_frame_finish(struct sk_buff *skb)
{
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);
	struct net_bridge *br;
	struct net_bridge_fdb_entry *dst;
	struct net_bridge_mdb_entry *mdst;
	struct sk_buff *skb2;
	struct net_bridge_port *pdst = NULL;
	br_get_dst_hook_t *get_dst_hook = rcu_dereference(br_get_dst_hook);

	if (!p || p->state == BR_STATE_DISABLED)
		goto drop;

	/* insert into forwarding database after filtering to avoid spoofing */
	br = p->br;
	br_fdb_update(br, p, eth_hdr(skb)->h_source);

	if (!is_broadcast_ether_addr(dest) && is_multicast_ether_addr(dest) &&
	    br_multicast_rcv(br, p, skb))
		goto drop;

	if ((p->state == BR_STATE_LEARNING) && skb->protocol != htons(ETH_P_PAE))
		goto drop;

	BR_INPUT_SKB_CB(skb)->brdev = br->dev;

	/* The packet skb2 goes to the local host (NULL to skip). */
	skb2 = NULL;

	if (br->dev->flags & IFF_PROMISC)
		skb2 = skb;

#ifdef CONFIG_NET_PPPOE_IPV6_PTHROUGH
	{
		// default let pppoe packet to send up to support pppoe passthrough.
		unsigned short proto;
		struct ethhdr *eth = eth_hdr(skb);

		proto = eth->h_proto;
		if ((proto == htons(ETH_P_PPP_SES)) || (proto == htons(ETH_P_PPP_DISC))) {
			skb2 = skb;
		}
	}
#endif

	dst = NULL;

	if (skb->protocol == htons(ETH_P_PAE)) {
		skb2 = skb;
		/* Do not forward 802.1x/EAP frames */
		skb = NULL;
	} else if (is_broadcast_ether_addr(dest))
		skb2 = skb;
	else if (is_multicast_ether_addr(dest)) {
		br_multicast_handle_hook_t *multicast_handle_hook = rcu_dereference(br_multicast_handle_hook);
		if (!__br_get(multicast_handle_hook, true, p, skb))
			goto out;

		mdst = br_mdb_get(br, skb);
		if (mdst || BR_INPUT_SKB_CB_MROUTERS_ONLY(skb)) {
			if ((mdst && mdst->mglist) ||
			    br_multicast_is_router(br))
				skb2 = skb;
			br_multicast_forward(mdst, skb, skb2);
			skb = NULL;
			if (!skb2)
				goto out;
		} else
			skb2 = skb;

		br->dev->stats.multicast++;
	} else if ((pdst = __br_get(get_dst_hook, NULL, p, &skb))) {
		if (!skb) goto out;
	} else if ((p->flags & BR_ISOLATE_MODE) ||
		   ((dst = __br_fdb_get(br, dest)) && dst->is_local)) {
		skb2 = skb;
		/* Do not forward the packet since it's local. */
		skb = NULL;
	}

	if (skb) {
#ifdef CONFIG_BRIDGE_MLD
		if (is_multicast_ether_addr(dest))
		{
#ifdef CONFIG_BRIDGE_IGMP
			struct iphdr *iph=NULL;
#endif
			struct ipv6hdr *ipv6h=NULL;
			unsigned char proto=0;
			unsigned char macAddr[6];
			unsigned char operation;
			char tmpOp;
			unsigned int gIndex=0;
			unsigned int moreFlag=1;

#ifdef CONFIG_BRIDGE_IGMP
			if (MULTICAST_MAC(dest) 
				&& (eth_hdr(skb)->h_proto == htons(ETH_P_IP)))
			{
				iph=(struct iphdr *)skb_network_header(skb);
				proto =  iph->protocol;
				
				if (proto == IPPROTO_IGMP) 
				{
					//printk("[%s:%d]process IGMP\n",__FUNCTION__,__LINE__);
					while(moreFlag)
					{
						tmpOp=igmp_type_check(skb, macAddr, &gIndex, &moreFlag);
						if(tmpOp>0)
						{
							operation=(unsigned char)tmpOp;
#ifdef DBG_IGMP
							if( operation == 1)
								printk("igmp add from frame finish\n");
							else if(operation == 2)
								printk("igmp del from frame finish\n");
#endif
							write_alpha_multicast(operation, br, macAddr, skb);
						}
					}
				}
			}
			else if(IPV6_MULTICAST_MAC(dest)
				&& (eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)))
#else
			if(IPV6_MULTICAST_MAC(dest)
				&& (eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)))
#endif
			{
				ipv6h=(struct ipv6hdr *)skb_network_header(skb);
				proto =  getIpv6TransportProtocol(ipv6h);
				/*icmp protocol*/
				if (proto == IPPROTO_ICMPV6) 
				{
#ifdef DBG_ICMPv6
					if(net_ratelimit())
						printk("[%s:%d]ipv6 ICMPV6,\n",__FUNCTION__,__LINE__);		
#endif
					while(moreFlag)
					{
						tmpOp=ICMPv6_check(skb , macAddr, &gIndex, &moreFlag);
						if(tmpOp > 0)
						{
							operation=(unsigned char)tmpOp;
#ifdef DBG_ICMPv6
							if( operation == 1)
								printk("icmpv6 add from frame finish\n");
							else if(operation == 2)
								printk("icmpv6 del from frame finish\n");
#endif
							write_alpha_multicast(operation, br, macAddr, skb);
						}
					}
				}
			}
		}
#endif
		if (dst) {
			dst->used = jiffies;
			pdst = dst->dst;
		}

		if (pdst)
			br_forward(pdst, skb, skb2);
		else
		{
#ifdef CONFIG_NET_PPPOE_IPV6_PTHROUGH
			if (    
					(pt_dmac_search(dest) == 1 ) && 
					(
						( ipv6_pt_enable  == 1 && (eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)) && !IPV6_MULTICAST_MAC(dest) ) || 
			  			( pppoe_pt_enable == 1 && (eth_hdr(skb)->h_proto == htons(ETH_P_PPP_SES) || eth_hdr(skb)->h_proto == htons(ETH_P_PPP_DISC))  )
			  		)
		  		)
			{
				// if dst mac at WAN, and ipv6/pppoe passthrough enabled, then no flooding
			}
			else	
#endif
				br_flood_forward(br, skb, skb2);
		}
	}

	if (skb2)
		return br_pass_frame_up(skb2);

out:
	return 0;
drop:
	kfree_skb(skb);
	goto out;
}

/* note: already called with rcu_read_lock */
static int br_handle_local_finish(struct sk_buff *skb)
{
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);

	br_fdb_update(p->br, p, eth_hdr(skb)->h_source);
	return 0;	 /* process further */
}

/* Does address match the link local multicast address.
 * 01:80:c2:00:00:0X
 */
static inline int is_link_local(const unsigned char *dest)
{
	__be16 *a = (__be16 *)dest;
	static const __be16 *b = (const __be16 *)br_group_address;
	static const __be16 m = cpu_to_be16(0xfff0);

	return ((a[0] ^ b[0]) | (a[1] ^ b[1]) | ((a[2] ^ b[2]) & m)) == 0;
}

/*
 * Return NULL if skb is handled
 * note: already called with rcu_read_lock
 */
rx_handler_result_t br_handle_frame(struct sk_buff **pskb)
{
	struct net_bridge_port *p;
	struct sk_buff *skb = *pskb;
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	br_should_route_hook_t *rhook;

	if (unlikely(skb->pkt_type == PACKET_LOOPBACK))
		return RX_HANDLER_PASS;

	if (!is_valid_ether_addr(eth_hdr(skb)->h_source))
		goto drop;

	skb = skb_share_check(skb, GFP_ATOMIC);
	if (!skb)
		return RX_HANDLER_CONSUMED;

	p = br_port_get_rcu(skb->dev);

	if (unlikely(is_link_local(dest))) {
		/*
		 * See IEEE 802.1D Table 7-10 Reserved addresses
		 *
		 * Assignment		 		Value
		 * Bridge Group Address		01-80-C2-00-00-00
		 * (MAC Control) 802.3		01-80-C2-00-00-01
		 * (Link Aggregation) 802.3	01-80-C2-00-00-02
		 * 802.1X PAE address		01-80-C2-00-00-03
		 *
		 * 802.1AB LLDP 		01-80-C2-00-00-0E
		 *
		 * Others reserved for future standardization
		 */
		switch (dest[5]) {
		case 0x00:	/* Bridge Group Address */
			/* If STP is turned off,
			   then must forward to keep loop detection */
			if (p->br->stp_enabled == BR_NO_STP)
				goto forward;
			break;

		case 0x01:	/* IEEE MAC (Pause) */
			goto drop;

		default:
			/* Allow selective forwarding for most other protocols */
			if (p->br->group_fwd_mask & (1u << dest[5]))
				goto forward;
		}

		/* Deliver packet to local host only */
		if (BR_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN, skb, skb->dev,
			    NULL, br_handle_local_finish)) {
			return RX_HANDLER_CONSUMED; /* consumed by filter */
		} else {
			*pskb = skb;
			return RX_HANDLER_PASS;	/* continue processing */
		}
	}

forward:
	switch (p->state) {
	case BR_STATE_FORWARDING:
		rhook = rcu_dereference(br_should_route_hook);
		if (rhook) {
			if ((*rhook)(skb)) {
				*pskb = skb;
				return RX_HANDLER_PASS;
			}
			dest = eth_hdr(skb)->h_dest;
		}
		/* fall through */
	case BR_STATE_LEARNING:
		if (!compare_ether_addr(p->br->dev->dev_addr, dest))
			skb->pkt_type = PACKET_HOST;

		BR_HOOK(NFPROTO_BRIDGE, NF_BR_PRE_ROUTING, skb, skb->dev, NULL,
			br_handle_frame_finish);
		break;
	default:
drop:
		kfree_skb(skb);
	}
	return RX_HANDLER_CONSUMED;
}
