/*
 * passthrough 
 * This module is used to pass through the specific packets
 * 
 * Peter Wu 20050804
 */

#include <linux/netdevice.h>
#include <linux/proc_fs.h>
#include "pthrough.h"

extern int pppoe_pt_enable;
extern int ipv6_pt_enable;
extern int pppoe_pthrough(struct sk_buff *skb);
extern int ipv6_pthrough(struct sk_buff *skb);
extern int pt_dmac_add(unsigned char *dmac);
extern void pt_dmac_delete(void);
extern int pt_dmac_search(unsigned char *dmac);
extern void pt_dmac_print(void);

int pt_dmac_idx = 0;
static unsigned long g_pt_dmac_jiffies=0;
pt_dmac_entry_t pt_dmac_table[PT_DMAC_TABLE_SIZE];

int pt_dmac_add(unsigned char *dmac)
{
	int i=0;

	//printk("%s(%d):  into \n", __func__, __LINE__);

	for (i=0; i<pt_dmac_idx; i++) {
		if ( pt_dmac_table[i].aging_time > 0 && !memcmp(pt_dmac_table[i].dmac, dmac, ETH_ALEN)) {
			break;
		}
	}

	//printk("%s(%d):  i=%d \n", __func__, __LINE__, i);
	
	if (i == pt_dmac_idx) {
		//printk("%s(%d):  %02x:%02x:%02x:%02x:%02x:%02x \n", __func__, __LINE__, dmac[0], dmac[1], dmac[2], dmac[3], dmac[4], dmac[5]);
		memcpy(pt_dmac_table[i].dmac, dmac, ETH_ALEN);
		pt_dmac_table[i].aging_time = jiffies;
		pt_dmac_idx++;
		if (pt_dmac_idx >= PT_DMAC_TABLE_SIZE) {
			printk("PeTeR: pt_dmac_table full!! (%d)\n", pt_dmac_idx);
			pt_dmac_idx--;
		}
	}
	else if (i < pt_dmac_idx) 
		pt_dmac_table[i].aging_time = jiffies;

	return 0;
}

void pt_dmac_delete()
{
	int i=0, j=0;
	
	for (i=0; i<pt_dmac_idx; i++) {
		if ( pt_dmac_table[i].aging_time > 0 &&  time_before(pt_dmac_table[i].aging_time, jiffies - PT_DMAC_AGING_TIME*HZ ) ) 
			memset(&pt_dmac_table[i], 0 , sizeof(pt_dmac_entry_t));
	}

	for (i=pt_dmac_idx; i<PT_DMAC_TABLE_SIZE; i++) 
			memset(&pt_dmac_table[i], 0 , sizeof(pt_dmac_entry_t));
	
	for (i=0; i<PT_DMAC_TABLE_SIZE-1 ;i++) 
	{
		if ( pt_dmac_table[i].aging_time != 0 )
			continue;
			
		for (j=i+1; j<PT_DMAC_TABLE_SIZE; j++) 
		{
			if ( pt_dmac_table[j].aging_time == 0 )
				continue;
				
			memcpy( &pt_dmac_table[i], &pt_dmac_table[j], sizeof(pt_dmac_entry_t));		
			memset( &pt_dmac_table[j], 0 , sizeof(pt_dmac_entry_t));
			break;
		}
	}

	for (i=0; i<PT_DMAC_TABLE_SIZE; i++) {
		if ( pt_dmac_table[i].aging_time == 0 )
		{
			pt_dmac_idx = i;
			break;
		}
	}
	
	g_pt_dmac_jiffies = jiffies;
}


int pt_dmac_search(unsigned char *dmac)
{
	int i=0;
	
	for (i=0; i<pt_dmac_idx; i++) {
		if ( pt_dmac_table[i].aging_time > 0 && !memcmp(pt_dmac_table[i].dmac, dmac, ETH_ALEN)) 
		{
			pt_dmac_table[i].aging_time = jiffies;
			return 1;
		}
	}

	return 0;
}

void pt_dmac_print()
{
	int i=0;
	
	pt_dmac_delete();
	
    printk("\n\nCurrent time: %lu  pt_dmac_table: %d \n",  jiffies, pt_dmac_idx );
	
	for (i=0; i<pt_dmac_idx; i++) 
		printk("%03d:  %02x:%02x:%02x:%02x:%02x:%02x   %lu\n", 
				i+1,
				pt_dmac_table[i].dmac[0], 
				pt_dmac_table[i].dmac[1],
				pt_dmac_table[i].dmac[2], 
				pt_dmac_table[i].dmac[3], 
				pt_dmac_table[i].dmac[4], 
				pt_dmac_table[i].dmac[5],
				pt_dmac_table[i].aging_time );	
			
	return;
}


/* return 1, if we want to handle this packet, or
 * return 0, let other ones do this	 */
int private_pthrough(struct sk_buff *skb)
{
	int ret = 0;

	if (pppoe_pt_enable || ipv6_pt_enable) {
		if ( g_pt_dmac_jiffies == 0 )
		{
			int i=0;
			
			for (i=0; i<PT_DMAC_TABLE_SIZE; i++) 
				memset( &pt_dmac_table[i], 0 , sizeof(pt_dmac_entry_t));
			pt_dmac_idx =0;
			g_pt_dmac_jiffies = jiffies;
		}
				
				
		if ( (pt_dmac_idx > (PT_DMAC_TABLE_SIZE * 3 / 4 )) && time_before(g_pt_dmac_jiffies, jiffies - 1 * HZ ) )
			pt_dmac_delete();
		else if ( (pt_dmac_idx > (PT_DMAC_TABLE_SIZE * 1 / 2 )) && time_before(g_pt_dmac_jiffies, jiffies - PT_DMAC_PERIOD_CHECK_TIME * HZ ) )
			pt_dmac_delete();
		else if ( (pt_dmac_idx > (PT_DMAC_TABLE_SIZE * 1 / 4 )) && time_before(g_pt_dmac_jiffies, jiffies - 10 * HZ ) )
			pt_dmac_delete();
		else if ( time_before(g_pt_dmac_jiffies, jiffies - 60 * HZ ) )
			pt_dmac_delete();
			
	}
	if (pppoe_pt_enable) {
		ret = pppoe_pthrough(skb);
	}
	if (!ret && ipv6_pt_enable) {
		ret = ipv6_pthrough(skb);
	}
	
	return ret;
}

#define PT_ROOT		"pthrough"
static struct proc_dir_entry * root = NULL;
static struct proc_dir_entry * pppoe = NULL;
static struct proc_dir_entry * ipv6 = NULL;

extern int proc_pppoe_read(char *page, char **start, off_t off, int count, int *eof, void *data);
extern int proc_pppoe_write(struct file *file, const char * buffer, unsigned long count, void *data);
extern int proc_ipv6_read(char *page, char **start, off_t off, int count, int *eof, void *data);
extern int proc_ipv6_write(struct file *file, const char * buffer, unsigned long count, void *data);

void pthrough_remove_proc_entry(void)
{
	if (pppoe) {
		remove_proc_entry("pppoe", root);
		pppoe = NULL;
	}
	if (ipv6) {
		remove_proc_entry("ipv6", root);
		ipv6 = NULL;
	}
	if (root) {
		remove_proc_entry(PT_ROOT, NULL);
		root = NULL;
	}
}

int pthrough_create_proc_entry(void)
{
	/* create directory */
	root = proc_mkdir(PT_ROOT, NULL);
	if (root == NULL) {
		printk("proc_mkdir return NULL!\n");
		goto pt_out;
	}

	/* create entries */
	pppoe = create_proc_entry("pppoe", 0644, root);
	if (pppoe == NULL) {
		printk("create_proc_entry (pppoe) return NULL!\n");
		goto pt_out;
	}
	pppoe->read_proc = proc_pppoe_read;
	pppoe->write_proc = proc_pppoe_write;
	
	ipv6 = create_proc_entry("ipv6", 0644, root);
	if (ipv6 == NULL) {
		printk("create_proc_entry (ipv6) return NULL!\n");
		goto pt_out;
	}
	ipv6->read_proc = proc_ipv6_read;
	ipv6->write_proc = proc_ipv6_write;
	
	return 0;
	
pt_out:
	pthrough_remove_proc_entry();
	printk("Unable to create %s !!\n", PT_ROOT);
	return -1;
}

