/* vi: set sw=4 ts=4: */


static char const RCSID[] =
"$Id: pppoe.c,v 1.1.1.1 2005/05/19 10:53:06 r01122 Exp $";

#include "pppoe.h"

#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <errno.h>

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef USE_LINUX_PACKET

/* TIOCSETD is defined in asm/ioctls.h ,
 * but in broadcom's toolchain, the 'tIOC' is used to define TIOCSETD,
 * which is defined only when __USE_MISC or __KERNEL__.
 * But __USE_MISC is undefined, so define tIOC here, only for sentry5.
 * 				David Hsieh (david_hsieh@alphanetworks.com)
 */
#ifdef _sentry5_
#define tIOC ('t' << 8)
#endif

#include <sys/ioctl.h>
#include <fcntl.h>
#endif

#include <signal.h>

#ifdef HAVE_N_HDLC
#ifndef N_HDLC
#include <linux/termios.h>
#endif
#endif

#include "dtrace.h"

#include "detect.h"

static PPPoEConnection conn;

/* Ethernet I/F name (example : nas10_0) */
char	devnam[NAME_MAX];

int PPPOEConnectDevice(void)
{
	int rlt=discovery(&conn);
	
	/* Do discovery only. */
	if(rlt==0)
		sendPADT(&conn, "Discovery finished");
	return rlt;
}

int PPPOEInitDevice(void)
{
	d_dbg("pppoe: >>> PPPOEInitDevice()\n");
	
	memset(&conn, 0, sizeof(conn));

	conn.discoverySocket = -1;
	conn.sessionSocket = -1;
	conn.ifName = devnam;
	conn.serviceName = NULL;
	conn.acName = NULL;
	conn.useHostUniq = 0;
	conn.synchronous = 0;
	return 1;
}

/***********************************************************************
*%FUNCTION: sendPADT
*%ARGUMENTS:
* conn -- PPPoE connection
* msg -- if non-NULL, extra error message to include in PADT packet.
*%RETURNS:
* Nothing
*%DESCRIPTION:
* Sends a PADT packet
***********************************************************************/
int sendPADT(PPPoEConnection *conn, char const *msg)
{
	PPPoEPacket packet;
	unsigned char *cursor = packet.payload;
	UINT16_t plen = 0;

	d_dbg("pppoe: >>> sendPADT(%s)\n", msg);
	
	/* Do nothing if no session established yet */
	if (!conn->session) return 0;

	/* Do nothing if no discovery socket */
	if (conn->discoverySocket < 0) return 0;

	memcpy(packet.ethHdr.h_dest, conn->peerEth, ETH_ALEN);
	memcpy(packet.ethHdr.h_source, conn->myEth, ETH_ALEN);

	packet.ethHdr.h_proto = htons(Eth_PPPOE_Discovery);
	packet.ver = 1;
	packet.type = 1;
	packet.code = CODE_PADT;
	packet.session = conn->session;

	/* Reset Session to zero so there is no possibility of
	 * recursive calls to this function by any signal handler */
	conn->session = 0;

	/* If we're using Host-Uniq, copy it over */
	if (conn->useHostUniq)
	{
		PPPoETag hostUniq;
		pid_t pid = getpid();
		hostUniq.type = htons(TAG_HOST_UNIQ);
		hostUniq.length = htons(sizeof(pid));
		memcpy(hostUniq.payload, &pid, sizeof(pid));
		memcpy(cursor, &hostUniq, sizeof(pid) + TAG_HDR_SIZE);
		cursor += sizeof(pid) + TAG_HDR_SIZE;
		plen += sizeof(pid) + TAG_HDR_SIZE;
	}

	/* Copy error message */
	if (msg)
	{
		PPPoETag err;
		size_t elen = strlen(msg);
		err.type = htons(TAG_GENERIC_ERROR);
		err.length = htons(elen);
		strcpy((char *)err.payload, msg);
		memcpy(cursor, &err, elen + TAG_HDR_SIZE);
		cursor += elen + TAG_HDR_SIZE;
		plen += elen + TAG_HDR_SIZE;
	}

	/* Copy cookie and relay-ID if needed */
	if (conn->cookie.type)
	{
		CHECK_ROOM(cursor, packet.payload, ntohs(conn->cookie.length) + TAG_HDR_SIZE);
		memcpy(cursor, &conn->cookie, ntohs(conn->cookie.length) + TAG_HDR_SIZE);
		cursor += ntohs(conn->cookie.length) + TAG_HDR_SIZE;
		plen += ntohs(conn->cookie.length) + TAG_HDR_SIZE;
	}

	if (conn->relayId.type)
	{
		CHECK_ROOM(cursor, packet.payload, ntohs(conn->relayId.length) + TAG_HDR_SIZE);
		memcpy(cursor, &conn->relayId, ntohs(conn->relayId.length) + TAG_HDR_SIZE);
		cursor += ntohs(conn->relayId.length) + TAG_HDR_SIZE;
		plen += ntohs(conn->relayId.length) + TAG_HDR_SIZE;
	}

	packet.length = htons(plen);
	sendPacket(conn, conn->discoverySocket, &packet, (int) (plen + HDR_SIZE));
/*
	if (conn->debugFile)
	{
		dumpPacket(conn->debugFile, &packet, "SENT");
		fprintf(conn->debugFile, "\n");
		fflush(conn->debugFile);
	}
*/
	d_info("pppoe: Sent PADT!\n");
	return 0;
}

/**********************************************************************
*%FUNCTION: parsePacket
*%ARGUMENTS:
* packet -- the PPPoE discovery packet to parse
* func -- function called for each tag in the packet
* extra -- an opaque data pointer supplied to parsing function
*%RETURNS:
* 0 if everything went well; -1 if there was an error
*%DESCRIPTION:
* Parses a PPPoE discovery packet, calling "func" for each tag in the packet.
* "func" is passed the additional argument "extra".
***********************************************************************/
int parsePacket(PPPoEPacket *packet, ParseFunc *func, void *extra)
{
	UINT16_t len = ntohs(packet->length);
	unsigned char *curTag;
	UINT16_t tagType, tagLen;

	if (packet->ver != 1)
	{
		d_warn("pppoe: Invalid PPPoE version (%d)", (int) packet->ver);
		return -1;
	}
	if (packet->type != 1)
	{
		d_warn("pppoe: Invalid PPPoE type (%d)", (int) packet->type);
		return -1;
	}

	/* Do some sanity checks on packet */
	if (len > ETH_DATA_LEN - 6)
	{	/* 6-byte overhead for PPPoE header */
		d_warn("pppoe: Invalid PPPoE packet length (%u)\n", len);
		return -1;
    }

	/* Step through the tags */
	curTag = packet->payload;
	while (curTag - packet->payload < len)
	{	/* Alignment is not guaranteed, so do this by hand... */
		tagType = (((UINT16_t) curTag[0]) << 8) + (UINT16_t) curTag[1];
		tagLen = (((UINT16_t) curTag[2]) << 8) + (UINT16_t) curTag[3];
		if (tagType == TAG_END_OF_LIST)
		{
			return 0;
		}
		if ((curTag - packet->payload) + tagLen + TAG_HDR_SIZE > len)
		{
			d_warn("Invalid PPPoE tag length (%u)", tagLen);
			return -1;
		}
		if (func(tagType, tagLen, curTag+TAG_HDR_SIZE, extra) < 0) return -1;
		curTag = curTag + TAG_HDR_SIZE + tagLen;
	}
	return 0;
}

/**********************************************************************
 * %FUNCTION: PPPoEDevnameHook
 * %ARGUMENTS:
 * cmd -- the command (actually, the device name
 * argv -- argument vector
 * doit -- if non-zero, set device name.  Otherwise, just check if possible
 * %RETURNS:
 * 1 if we will handle this device; 0 otherwise.
 * %DESCRIPTION:
 * Checks if name is a valid interface name; if so, returns 1.  Also
 * sets up devnam (string representation of device).
 ***********************************************************************/
static int
PPPoEPrepareDevice( char*  arg_devname )
{
    int r = 1;
    int fd;
    struct ifreq ifr;
	
	d_dbg("pppoe: >>> PPPoEPrepareDevice()\n");

    /* Open a socket */
    if ((fd = socket(PF_PACKET, SOCK_RAW, 0)) < 0) {
	r = 0;
    }

    /* Try getting interface index */
    if (r) {
	strncpy(ifr.ifr_name, arg_devname, sizeof(ifr.ifr_name));
	if (ioctl(fd, SIOCGIFINDEX, &ifr) < 0) {
	    r = 0;
	} else {
	    if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) {
		r = 0;
	    } else {
#ifdef ARPHRD_ETHER
		if (ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER) {
			error("Interface %s not Ethernet", arg_devname);
			r = 0;
		}
#endif
	    }
	}
    }

    /* Close socket */
    close(fd);
    if ( r ) {
	strncpy(devnam, arg_devname, sizeof(devnam));
	PPPOEInitDevice();

	return 1;
    }

    return r;
}

/**********************************************************************
 * %FUNCTION: plugin_init
 * %ARGUMENTS:
 * None
 * %RETURNS:
 * Nothing
 * %DESCRIPTION:
 * Initializes hooks for pppd plugin
 ***********************************************************************/
int  pppoe_init( char*  arg_devname )
{
	int	res;
	
	d_dbg("pppoe: >>> pppoe_init()\n");
	
	res = PPPoEPrepareDevice( arg_devname );

	return res;
}
