#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/file.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>
#include <sysexits.h>
#include <limits.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/wait.h>

extern char *optarg;
extern int optind;

static sig_atomic_t timeout_expired = 0;

#define PROG_TEXT		"NEC flock"
#define PROG_VERSION		"0.0.1"
#define USAGE_OPTIONS		"Options:\n"

static void
usage(const char *pname)
{

	printf("%s version %s\n"
	       "Usage: %s [options] <file>|<directory> <command> [<argument>...]\n\n", PROG_TEXT, PROG_VERSION, pname);

	fputs(USAGE_OPTIONS, stdout);
	fputs((  " -s          get a shared lock\n"), stdout);
	fputs((  " -x          get an exclusive lock (default)\n"), stdout);
	fputs((  " -u          remove a lock\n"), stdout);
	fputs((  " -n          fail rather than wait\n"), stdout);
	fputs((  " -w <secs>   wait for a limited amount of time\n"), stdout);
	fputs((  " -o          close file descriptor before running command\n"), stdout);
	fputs((  " -F          execute command without forking\n"), stdout);

	exit(EXIT_FAILURE);
}

static void 
run_program(char **cmd_argv)
{
	execvp(cmd_argv[0], cmd_argv);

	warn(("failed to execute %s"), cmd_argv[0]);
	_exit((errno == ENOMEM) ? EX_OSERR : EX_UNAVAILABLE);
}

void timer_handler(int signum)
{
	timeout_expired = 1;
}

static int 
open_file(const char *filename, int *flags)
{

	int fd;
	int fl = *flags == 0 ? O_RDONLY : *flags;

	errno = 0;
	fl |= O_NOCTTY | O_CREAT;
	fd = open(filename, fl, 0666);

	/* Linux doesn't like O_CREAT on a directory, even though it
	 * should be a no-op; POSIX doesn't allow O_RDWR or O_WRONLY
	 */
	if (fd < 0 && errno == EISDIR) {
		fl = O_RDONLY | O_NOCTTY;
		fd = open(filename, fl);
	}
	if (fd < 0) {
		printf("cannot open lock file %s\n", filename);
		if (errno == ENOMEM || errno == EMFILE || errno == ENFILE)
			exit(EX_OSERR);
		if (errno == EROFS || errno == ENOSPC)
			exit(EX_CANTCREAT);
		exit(EX_NOINPUT);
	}
	*flags = fl;
	return fd;
}

static int 
e_atoi(const char *nptr)
{
	long val;
	char *endptr;

	errno = 0;
	val = strtol(nptr, &endptr, 10);

	if ((errno == ERANGE && (val == LONG_MAX || val == LONG_MIN))
	    || (errno != 0 && val == 0)) {
		perror("strtol");
		exit(EXIT_FAILURE);
	}

	if (endptr == nptr) {
		fprintf(stderr, "No digits were found\n");
		exit(EXIT_FAILURE);
	}

	return val;
}

int 
main(int argc, char** argv)
{
	int i;
	struct sigaction act, oldact;
	timer_t tid;
	struct itimerspec itval = {0};
	int opt;
	int have_timeout = 0;
	int type = LOCK_EX;
	int block = 0;
	int timeout = 0;
	int fd, open_flags = 0;
	int do_close = 0;
	int no_fork = 0;
	int status;
	const char *filename = NULL;
	char **cmd_argv = NULL;

	char *progname;

	progname = basename(argv[0]);

	while ((opt = getopt(argc, argv, "sexuoFnw:")) != -1) {
		switch (opt) {
		case 's':
			type = LOCK_SH;
			break;
		case 'e':
		case 'x':
			type = LOCK_EX;
			break;
		case 'u':
			type = LOCK_UN;
			break;
		case 'o':
			do_close = 1;
			break;
		case 'F':
			no_fork = 1;
			break;
		case 'n':
			block = LOCK_NB;
			break;
		case 'w':
			have_timeout = 1;
			timeout = e_atoi(optarg);
			break;
		default: /* '?' */
			usage(progname);
			/* NOTREACHED */
		}
	}

	if (no_fork && do_close) {
		errx(EX_USAGE,
			("the --no-fork and --close options are incompatible"));
	}

	if (argc > optind + 1) {
		/* Run command */
		cmd_argv = &argv[optind + 1];

		filename = argv[optind];
		fd = open_file(filename, &open_flags);

	} else {
		usage(progname);
		/* NOTREACHED */
	}

	if (timeout == 0) {
		have_timeout = 0;
		block = LOCK_NB;
	}

	if (have_timeout) {
		memset(&act, 0, sizeof(struct sigaction));
		memset(&oldact, 0, sizeof(struct sigaction));

		// シグナルハンドラの登録
		act.sa_handler = timer_handler;
		act.sa_flags = SA_RESETHAND;
		if(sigaction(SIGALRM, &act, &oldact) < 0) {
		    perror("sigaction()");
		    exit(EXIT_FAILURE);
		}

		// タイマ割り込みを発生させる
		itval.it_value.tv_sec = timeout;
		itval.it_value.tv_nsec = 0;

		// タイマの作成
		if(timer_create(CLOCK_REALTIME, NULL, &tid) < 0) {
		    perror("timer_create");
		    exit(EXIT_FAILURE);
		}

		//  タイマのセット
		if(timer_settime(tid, 0, &itval, NULL) < 0) {
		    perror("timer_settime");
		    exit(EXIT_FAILURE);
		}

	}

	while (flock(fd, type | block)) {
		switch (errno) {
		case EWOULDBLOCK:
			/* -n option set and failed to lock. */
			warnx(("failed to get lock"));
			exit(EXIT_FAILURE);
		case EINTR:
			/* Signal received */
			if (timeout_expired) {
				/* -w option set and failed to lock. */
				warnx(("timeout while waiting to get lock"));
				exit(EXIT_FAILURE);
			}
			/* otherwise try again */
			continue;
		case EIO:
		case EBADF:		/* since Linux 3.4 (commit 55725513) */
			/* Probably NFSv4 where flock() is emulated by fcntl().
			 * Let's try to reopen in read-write mode.
			 */
			if (!(open_flags & O_RDWR) &&
			    type != LOCK_SH &&
			    filename &&
			    access(filename, R_OK | W_OK) == 0) {

				close(fd);
				open_flags = O_RDWR;
				fd = open_file(filename, &open_flags);

				if (open_flags & O_RDWR) {
					break;
				}
			}
			/* fallthrough */
		default:
			/* Other errors */
			if (filename) {
				warn("%s", filename);
			} else {
				warn("%d", fd);
			}
			exit((errno == ENOLCK || errno == ENOMEM) ? EX_OSERR : EX_DATAERR);
		}
	}

	if (have_timeout) {
		// タイマの解除
		timer_delete(tid);

		// シグナルハンドラの解除
		sigaction(SIGALRM, &oldact, NULL);
	}

	status = EX_OK;

	if (cmd_argv) {
		pid_t w, f;
		/* Clear any inherited settings */
		signal(SIGCHLD, SIG_DFL);

		if (!no_fork) {
			f = fork();
			if (f < 0) {
				err(EX_OSERR, ("fork failed"));
			} else if (f == 0) {
				/* child */
				if (do_close)
					close(fd);
				run_program(cmd_argv);
			} else {
				/* parent */
				do {
					w = waitpid(f, &status, 0);
					if (w == -1 && errno != EINTR)
						break;
				} while (w != f);

				if (w == -1) {
					status = EXIT_FAILURE;
					warn(("waitpid failed"));
				} else if (WIFEXITED(status)) {
					status = WEXITSTATUS(status);
				} else if (WIFSIGNALED(status)) {
					status = WTERMSIG(status) + 128;
				} else {
					status = EX_OSERR;
				}
			}

		} else {
			/* no-fork execution */
			run_program(cmd_argv);
		}
	}

	return status;
}
