/*
 *	IPv6 tunneling device
 *	Linux INET6 implementation
 *
 *	Authors:
 *	nothing
 *
 *      Based on:
 *      linux/net/ipv6/ip6_tunnel.c
 *
 *      RFC 2473
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *  This file is modified by NEC Platforms, Ltd at 2018.
 */

#include <linux/module.h>
#include <linux/capability.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/sockios.h>
#include <linux/icmp.h>
#include <linux/if.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <linux/net.h>
#include <linux/inet.h>
#include <linux/in6.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/icmpv6.h>
#include <linux/init.h>
#include <linux/route.h>
#include <linux/rtnetlink.h>
#include <linux/netfilter_ipv6.h>
#include <linux/slab.h>

#include <asm/uaccess.h>
#include <linux/atomic.h>

#include <net/icmp.h>
#include <net/ip.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/addrconf.h>
#include <net/xfrm.h>
#include <net/dsfield.h>
#include <net/inet_ecn.h>
#include <net/net_namespace.h>
#include <net/netns/generic.h>
#include <linux/delay.h>

#include "map_e.h"

#ifdef CONFIG_RTL_MAP_E_FASTPATH_SUPPORT_DPI_ENGINE
#include <net/rtl/fastpath/fastpath_core.h>
#endif


MODULE_LICENSE("GPL");
MODULE_ALIAS_NETDEV("map0");

#define IPV6_TLV_TNL_ENCAP_LIMIT 4
#define IPV6_DEFAULT_TNL_ENCAP_LIMIT 4

/* don't add encapsulation limit if one isn't present in inner packet */
#define IP6_TNL_F_IGN_ENCAP_LIMIT 0x1
/* copy the traffic class field from the inner packet */
#define IP6_TNL_F_USE_ORIG_TCLASS 0x2
/* copy fwmark from inner packet */
#define IP6_TNL_F_USE_ORIG_FWMARK 0x20
/* copy the flowlabel from the inner packet */
#define IP6_TNL_F_USE_ORIG_FLOWLABEL 0x4

/* capable of sending packets */
#define IP6_TNL_F_CAP_XMIT 0x10000
/* capable of receiving packets */
#define IP6_TNL_F_CAP_RCV 0x20000

#define IPV6_TCLASS_MASK (IPV6_FLOWINFO_MASK & ~IPV6_FLOWLABEL_MASK)
#define IPV6_TCLASS_SHIFT 20

#define HASH_SIZE  32

#define HASH(addr) ((__force u32)((addr)->s6_addr32[0] ^ (addr)->s6_addr32[1] ^ \
		     (addr)->s6_addr32[2] ^ (addr)->s6_addr32[3]) & \
		    (HASH_SIZE - 1))

/* Tunnel encapsulation limit destination sub-option */

struct ipv6_tlv_tnl_enc_lim {
	__u8 type;		/* type-code for option         */
	__u8 length;		/* option length                */
	__u8 encap_limit;	/* tunnel encapsulation limit   */
} __packed;

/*
 * map_e.cEXPORTƤؿ򻲾Ȥ뤿Ρextern
 */
extern int map_e_rcv(struct sk_buff *, struct ip6_tnl_map_e *);
extern int map_e_frag_rcv(struct sk_buff *, struct ip6_tnl_map_e *);
extern int map_e_err_rcv(struct sk_buff *, struct iphdr *, struct ip6_tnl_map_e *);
extern int map_e_xmit(struct sk_buff *, struct flowi6 *, u8, struct ip6_tnl_map_e *, __u8);
extern int map_e_portset_init(uint16_t, uint16_t, struct ip6_tnl_map_e *);
extern int map_e_ioctl(struct net_device *, struct ifreq *, int);
extern int map_e_port_range_check(uint16_t, struct ip6_tnl_map_e *);
extern void map_e_session_init(struct ip6_tnl_map_e *t);
extern void map_e_session_destroy(struct ip6_tnl_map_e *t);
extern void map_e_session_clear(struct ip6_tnl_map_e *t);
extern const char *map_e_ntop(int af, const void *src, char *dst, size_t size);
extern int map_e_show_staticnapt(struct ip6_tnl_map_e *, struct map_e_parm_staticnapt *);
extern int map_e_add_staticnapt(struct ip6_tnl_map_e *, struct map_e_parm_staticnapt *);
extern int map_e_del_staticnapt(struct ip6_tnl_map_e *, struct map_e_parm_staticnapt *);
extern int map_e_flush_staticnapt(struct ip6_tnl_map_e *);
extern int set_session_timer(int expire, int protocol);

static inline int ip4ip6_tnl_xmit_map_e(struct sk_buff *skb, struct net_device *dev);
static int ip6_tnl_dev_init_map_e(struct net_device *dev);
static void ip6_tnl_dev_setup_map_e(struct net_device *dev);

/* LOCAL_INѥåȤν */
static struct net_device *g_xmap_dev = NULL;
int g_map_e_deinitializing = 0;
uint8_t g_debug = 1;

static int ip6_tnl_net_id_map_e __read_mostly;
struct ip6_tnl_net_map_e {
	/* the IPv6 tunnel fallback device */
	struct net_device *fb_tnl_dev;
	/* lists for storing tunnels in use */
	struct ip6_tnl_map_e __rcu *tnls_r_l[HASH_SIZE];
	struct ip6_tnl_map_e __rcu *tnls_wc[1];
	struct ip6_tnl_map_e __rcu **tnls[2];
};

/* often modified stats are per cpu, other are shared (netdev->stats) */
struct pcpu_tstats {
	unsigned long	rx_packets;
	unsigned long	rx_bytes;
	unsigned long	tx_packets;
	unsigned long	tx_bytes;
};

static struct
net_device_stats *ip6_get_stats_map_e(struct net_device *dev)
{
	struct pcpu_tstats sum = { 0 };
	int i;

	for_each_possible_cpu(i) {
		const struct pcpu_tstats *tstats = per_cpu_ptr(dev->tstats, i);

		sum.rx_packets += tstats->rx_packets;
		sum.rx_bytes   += tstats->rx_bytes;
		sum.tx_packets += tstats->tx_packets;
		sum.tx_bytes   += tstats->tx_bytes;
	}
	dev->stats.rx_packets = sum.rx_packets;
	dev->stats.rx_bytes   = sum.rx_bytes;
	dev->stats.tx_packets = sum.tx_packets;
	dev->stats.tx_bytes   = sum.tx_bytes;
	return &dev->stats;
}

/*
 * Locking : hash tables are protected by RCU and RTNL
 */

static inline struct dst_entry *
ip6_tnl_dst_check_map_e(struct ip6_tnl_map_e *t)
{
	struct dst_entry *dst = t->dst_cache;

	if (dst && dst->obsolete &&
	    dst->ops->check(dst, t->dst_cookie) == NULL) {
		t->dst_cache = NULL;
		dst_release(dst);
		return NULL;
	}

	return dst;
}

static inline void
ip6_tnl_dst_reset_map_e(struct ip6_tnl_map_e *t)
{
	dst_release(t->dst_cache);
	t->dst_cache = NULL;
}

static inline void
ip6_tnl_dst_store_map_e(struct ip6_tnl_map_e *t, struct dst_entry *dst)
{
	struct rt6_info *rt = (struct rt6_info *) dst;
	t->dst_cookie = rt->rt6i_node ? rt->rt6i_node->fn_sernum : 0;
	dst_release(t->dst_cache);
	t->dst_cache = dst;
}

/**
 * ip6_tnl_lookup_map_e - fetch tunnel matching the end-point addresses
 *   @local: the address of the tunnel entry-point
 *
 * Return:
 *   tunnel matching given end-points if found,
 *   else fallback tunnel if its device is up,
 *   else %NULL
 **/

#define for_each_ip6_tunnel_rcu(start) \
	for (t = rcu_dereference(start); t; t = rcu_dereference(t->next))

static struct ip6_tnl_map_e *
ip6_tnl_lookup_map_e(struct net *net, const struct in6_addr *local)
{
	unsigned int h1 = HASH(local);
	struct ip6_tnl_map_e *t;
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);

	for_each_ip6_tunnel_rcu(ip6n->tnls_r_l[h1]) {
		if (t->dev->flags & IFF_UP && t->map_e_info) {
			if (ipv6_addr_equal(local, &t->map_e_info->wan_addr)) {
				return t;
			}
		}
	}
	t = rcu_dereference(ip6n->tnls_wc[0]);
	if (t && (t->dev->flags & IFF_UP))
		return t;

	return NULL;
}

/**
 * ip6_tnl_bucket_map_e - get head of list matching given tunnel parameters
 *   @p: parameters containing tunnel end-points
 *
 * Description:
 *   ip6_tnl_bucket_map_e() returns the head of the list matching the
 *   &struct in6_addr entries laddr and raddr in @p.
 *
 * Return: head of IPv6 tunnel list
 **/

static struct ip6_tnl_map_e __rcu **
ip6_tnl_bucket_map_e(struct ip6_tnl_net_map_e *ip6n, const struct map_e_parm *p)
{
	const struct in6_addr *local = &p->map_e_v6;
	unsigned h = 0;
	int prio = 0;

	local = &p->map_e_v6;
	if (!ipv6_addr_any(local)) {
		prio = 1;
		h = HASH(local);
	}
	return &ip6n->tnls[prio][h];
}

/**
 * ip6_tnl_link_map_e - add tunnel to hash table
 *   @t: tunnel to be added
 **/

static void
ip6_tnl_link_map_e(struct ip6_tnl_net_map_e *ip6n, struct ip6_tnl_map_e *t)
{
	struct ip6_tnl_map_e __rcu **tp = ip6_tnl_bucket_map_e(ip6n, &t->parms);

	rcu_assign_pointer(t->next , rtnl_dereference(*tp));
	rcu_assign_pointer(*tp, t);
}

/**
 * ip6_tnl_unlink_map_e - remove tunnel from hash table
 *   @t: tunnel to be removed
 **/

static void
ip6_tnl_unlink_map_e(struct ip6_tnl_net_map_e *ip6n, struct ip6_tnl_map_e *t)
{
	struct ip6_tnl_map_e __rcu **tp;
	struct ip6_tnl_map_e *iter;

	for (tp = ip6_tnl_bucket_map_e(ip6n, &t->parms);
	     (iter = rtnl_dereference(*tp)) != NULL;
	     tp = &iter->next) {
		if (t == iter) {
			rcu_assign_pointer(*tp, t->next);
			break;
		}
	}
}

static void
ip6_dev_free_map_e(struct net_device *dev)
{
	free_percpu(dev->tstats);
	free_netdev(dev);
}

/**
 * ip6_tnl_create_map_e() - create a new tunnel
 *   @p: tunnel parameters
 *   @pt: pointer to new tunnel
 *
 * Description:
 *   Create tunnel matching given parameters.
 *
 * Return:
 *   created tunnel or NULL
 **/

static struct ip6_tnl_map_e *
ip6_tnl_create_map_e(struct net *net, struct map_e_parm *p)
{
	struct net_device *dev;
	struct ip6_tnl_map_e *t;
	char name[IFNAMSIZ];
	int err;
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);

	if (p->name[0])
		strlcpy(name, p->name, IFNAMSIZ);
	else
		snprintf(name, IFNAMSIZ, "map%%d");

	dev = alloc_netdev(sizeof (*t), name, ip6_tnl_dev_setup_map_e);
	if (dev == NULL)
		goto failed;

	dev_net_set(dev, net);

	t = netdev_priv(dev);
	t->parms = *p;
	err = ip6_tnl_dev_init_map_e(dev);
	if (err < 0)
		goto failed_free;

	if ((err = register_netdevice(dev)) < 0)
		goto failed_free;

	strlcpy(t->parms.name, dev->name, sizeof(t->parms.name));

	dev_hold(dev);
	ip6_tnl_link_map_e(ip6n, t);
	return t;

failed_free:
	ip6_dev_free_map_e(dev);
failed:
	return NULL;
}

/**
 * ip6_tnl_locate_map_e - find or create tunnel matching given parameters
 *   @p: tunnel parameters
 *   @create: != 0 if allowed to create new tunnel if no match found
 *
 * Description:
 *   ip6_tnl_locate_map_e() first tries to locate an existing tunnel
 *   based on @parms. If this is unsuccessful, but @create is set a new
 *   tunnel device is created and registered for use.
 *
 * Return:
 *   matching tunnel or NULL
 **/

static struct ip6_tnl_map_e *
ip6_tnl_locate_map_e(struct net *net, struct map_e_parm *p, int create)
{
	const struct in6_addr *local = &p->map_e_v6;
	struct ip6_tnl_map_e __rcu **tp;
	struct ip6_tnl_map_e *t;
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);

	for (tp = ip6_tnl_bucket_map_e(ip6n, p);
	     (t = rtnl_dereference(*tp)) != NULL;
	     tp = &t->next) {

		if (ipv6_addr_equal(local, &t->parms.map_e_v6)) {
			if (create) {
				return NULL;
			} else {
				return t;
			}
		}
	}
	if (!create)
		return NULL;
	return ip6_tnl_create_map_e(net, p);
}

static struct ip6_tnl_map_e *
ip6_tnl_locate_map_e2(struct net *net, const char *ifname)
{
	int hash;
	struct ip6_tnl_map_e *t;
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);

	if (ip6n == NULL || ifname == NULL) {
		MAP_TRACE("ifname or ip6n is null");
		return NULL;
	}
	MAP_TRACE("ifname : %s", ifname);

	for (hash = 0; hash < HASH_SIZE; hash++) {
		for_each_ip6_tunnel_rcu(ip6n->tnls_r_l[hash]) {
			if (strcmp(t->dev->name, ifname) == 0 && t->map_e_info) {
				MAP_TRACE("return t(0x%x)", (int)t);
				return t;
			}
		}
	}

	MAP_TRACE("return NULL");
	return NULL;
}

/**
 * ip6_tnl_dev_uninit_map_e - tunnel device uninitializer
 *   @dev: the device to be destroyed
 *
 * Description:
 *   ip6_tnl_dev_uninit_map_e() removes tunnel from its list
 **/

static void
ip6_tnl_dev_uninit_map_e(struct net_device *dev)
{
	struct ip6_tnl_map_e *t = netdev_priv(dev);
	struct net *net = dev_net(dev);
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);

	if (dev == ip6n->fb_tnl_dev)
		RCU_INIT_POINTER(ip6n->tnls_wc[0], NULL);
	else
		ip6_tnl_unlink_map_e(ip6n, t);
	ip6_tnl_dst_reset_map_e(t);
	dev_put(dev);
}

/**
 * parse_tvl_tnl_enc_lim - handle encapsulation limit option
 *   @skb: received socket buffer
 *
 * Return:
 *   0 if none was found,
 *   else index to encapsulation limit
 **/

static __u16
parse_tlv_tnl_enc_lim_map_e(struct sk_buff *skb, __u8 * raw)
{
	const struct ipv6hdr *ipv6h = (const struct ipv6hdr *) raw;
	__u8 nexthdr = ipv6h->nexthdr;
	__u16 off = sizeof (*ipv6h);

	while (ipv6_ext_hdr(nexthdr) && nexthdr != NEXTHDR_NONE) {
		__u16 optlen = 0;
		struct ipv6_opt_hdr *hdr;
		if (raw + off + sizeof (*hdr) > skb->data &&
		    !pskb_may_pull(skb, raw - skb->data + off + sizeof (*hdr)))
			break;

		hdr = (struct ipv6_opt_hdr *) (raw + off);
		if (nexthdr == NEXTHDR_FRAGMENT) {
			struct frag_hdr *frag_hdr = (struct frag_hdr *) hdr;
			if (frag_hdr->frag_off)
				break;
			optlen = 8;
		} else if (nexthdr == NEXTHDR_AUTH) {
			optlen = (hdr->hdrlen + 2) << 2;
		} else {
			optlen = ipv6_optlen(hdr);
		}
		if (nexthdr == NEXTHDR_DEST) {
			__u16 i = off + 2;
			while (1) {
				struct ipv6_tlv_tnl_enc_lim *tel;

				/* No more room for encapsulation limit */
				if (i + sizeof (*tel) > off + optlen)
					break;

				tel = (struct ipv6_tlv_tnl_enc_lim *) &raw[i];
				/* return index of option if found and valid */
				if (tel->type == IPV6_TLV_TNL_ENCAP_LIMIT &&
				    tel->length == 1)
					return i;
				/* else jump to next option */
				if (tel->type)
					i += tel->length + 2;
				else
					i++;
			}
		}
		nexthdr = hdr->nexthdr;
		off += optlen;
	}
	return 0;
}

/**
 * ip6_tnl_err_map_e - tunnel error handler
 *
 * Description:
 *   ip6_tnl_err_map_e() should handle errors in the tunnel according
 *   to the specifications in RFC 2473.
 **/

static int
ip6_tnl_err_map_e(struct sk_buff *skb, __u8 ipproto, struct inet6_skb_parm *opt,
	    u8 *type, u8 *code, int *msg, __u32 *info, int offset)
{
	const struct ipv6hdr *ipv6h = (const struct ipv6hdr *) skb->data;
	struct ip6_tnl_map_e *t;
	int rel_msg = 0;
	u8 rel_type = ICMPV6_DEST_UNREACH;
	u8 rel_code = ICMPV6_ADDR_UNREACH;
	__u32 rel_info = 0;
	__u16 len;
	int err = -ENOENT;

	/* If the packet doesn't contain the original IPv6 header we are
	   in trouble since we might need the source address for further
	   processing of the error. */

	rcu_read_lock();
	if ((t = ip6_tnl_lookup_map_e(dev_net(skb->dev), &ipv6h->saddr)) == NULL)
		goto out;

	if (t->parms.proto != ipproto && t->parms.proto != 0)
		goto out;

	err = 0;

	switch (*type) {
		__u32 teli;
		struct ipv6_tlv_tnl_enc_lim *tel;
		__u32 mtu;
	case ICMPV6_DEST_UNREACH:
		if (net_ratelimit())
			printk(KERN_WARNING
			       "%s: Path to destination invalid "
			       "or inactive!\n", t->parms.name);
		rel_msg = 1;
		break;
	case ICMPV6_TIME_EXCEED:
		if ((*code) == ICMPV6_EXC_HOPLIMIT) {
			if (net_ratelimit())
				printk(KERN_WARNING
				       "%s: Too small hop limit or "
				       "routing loop in tunnel!\n",
				       t->parms.name);
			rel_msg = 1;
		}
		break;
	case ICMPV6_PARAMPROB:
		teli = 0;
		if ((*code) == ICMPV6_HDR_FIELD)
			teli = parse_tlv_tnl_enc_lim_map_e(skb, skb->data);

		if (teli && teli == *info - 2) {
			tel = (struct ipv6_tlv_tnl_enc_lim *) &skb->data[teli];
			if (tel->encap_limit == 0) {
				if (net_ratelimit())
					printk(KERN_WARNING
					       "%s: Too small encapsulation "
					       "limit or routing loop in "
					       "tunnel!\n", t->parms.name);
				rel_msg = 1;
			}
		} else if (net_ratelimit()) {
			printk(KERN_WARNING
			       "%s: Recipient unable to parse tunneled "
			       "packet!\n ", t->parms.name);
		}
		break;
	case ICMPV6_PKT_TOOBIG:
		mtu = *info - offset;
		if (mtu < IPV6_MIN_MTU)
			mtu = IPV6_MIN_MTU;

		if ((len = sizeof (*ipv6h) + ntohs(ipv6h->payload_len)) > mtu) {
			rel_type = ICMPV6_PKT_TOOBIG;
			rel_code = 0;
			rel_info = mtu;
			rel_msg = 1;
		}
		break;
	}

	*type = rel_type;
	*code = rel_code;
	*info = rel_info;
	*msg = rel_msg;

out:
	rcu_read_unlock();
	return err;
}

static int
ip4ip6_err_map_e(struct sk_buff *skb, struct inet6_skb_parm *opt,
	   u8 type, u8 code, int offset, __be32 info)
{
	int rel_msg = 0;
	u8 rel_type = type;
	u8 rel_code = code;
	__u32 rel_info = ntohl(info);
	int err;
	struct sk_buff *skb2;
	struct iphdr *eiph;
	struct rtable *rt;
	struct flowi4 fl4;
	struct ip6_tnl_map_e *t;
	const struct ipv6hdr *ipv6h = (const struct ipv6hdr *) skb->data;

	err = ip6_tnl_err_map_e(skb, IPPROTO_IPIP, opt, &rel_type, &rel_code,
			  &rel_msg, &rel_info, offset);
	if (err < 0)
		return err;

	if (rel_msg == 0)
		return 0;

	switch (rel_type) {
	case ICMPV6_DEST_UNREACH:
		if (rel_code != ICMPV6_ADDR_UNREACH)
			return 0;
		rel_type = ICMP_DEST_UNREACH;
		rel_code = ICMP_HOST_UNREACH;
		break;
	case ICMPV6_PKT_TOOBIG:
		if (rel_code != 0)
			return 0;
		rel_type = ICMP_DEST_UNREACH;
		rel_code = ICMP_FRAG_NEEDED;
		break;
	default:
		return 0;
	}

	if (!pskb_may_pull(skb, offset + sizeof(struct iphdr)))
		return 0;

	skb2 = skb_clone(skb, GFP_ATOMIC);
	if (!skb2)
		return 0;

	skb_dst_drop(skb2);

	skb_pull(skb2, offset);
	skb_reset_network_header(skb2);
	eiph = ip_hdr(skb2);

	rcu_read_lock();
	if ((t = ip6_tnl_lookup_map_e(dev_net(skb->dev), &ipv6h->saddr)) == NULL)
		goto out;

	if (t->parms.proto != IPPROTO_IPIP && t->parms.proto != 0)
		goto out;

	if (map_e_err_rcv(skb, eiph, t) < 0)
		goto out;
	rcu_read_unlock();

	/* Try to guess incoming interface */
	rt = ip_route_output_ports(dev_net(skb->dev), &fl4, NULL,
				   eiph->saddr, 0,
				   0, 0,
				   IPPROTO_IPIP, RT_TOS(eiph->tos), 0);
	if (IS_ERR(rt))
		goto out;

	skb2->dev = rt->dst.dev;

	/* route "incoming" packet */
	if (rt->rt_flags & RTCF_LOCAL) {
		ip_rt_put(rt);
		rt = NULL;
		rt = ip_route_output_ports(dev_net(skb->dev), &fl4, NULL,
					   eiph->daddr, eiph->saddr,
					   0, 0,
					   IPPROTO_IPIP,
					   RT_TOS(eiph->tos), 0);
		if (IS_ERR(rt) ||
		    rt->dst.dev->type != ARPHRD_TUNNEL) {
			if (!IS_ERR(rt))
				ip_rt_put(rt);
			goto out;
		}
		skb_dst_set(skb2, &rt->dst);
	} else {
		ip_rt_put(rt);
		if (ip_route_input(skb2, eiph->daddr, eiph->saddr, eiph->tos,
				   skb2->dev) ||
		    ((skb_dst(skb2)->dev->type != ARPHRD_TUNNEL) &&
		     (skb_dst(skb2)->dev->type != ARPHRD_TUNNEL6)))
			goto out;
	}

	/* change mtu on this route */
	if (rel_type == ICMP_DEST_UNREACH && rel_code == ICMP_FRAG_NEEDED) {
		if (rel_info > dst_mtu(skb_dst(skb2)))
			goto out;

		skb_dst(skb2)->ops->update_pmtu(skb_dst(skb2), NULL, skb2, rel_info);
	}

	icmp_send(skb2, rel_type, rel_code, htonl(rel_info));

out:
	rcu_read_unlock();
	kfree_skb(skb2);
	return 0;
}

/**
 * ip6_tnl_rcv_map_e - decapsulate IPv6 packet and retransmit it locally
 *   @skb: received socket buffer
 *   @protocol: ethernet protocol ID
 *   @dscp_ecn_decapsulate: the function to decapsulate DSCP code and ECN
 *
 * Return: 0
 **/

static int
ip6_tnl_rcv_map_e(struct sk_buff *skb, __u16 protocol,
       __u8 ipproto,
       void (*dscp_ecn_decapsulate)(const struct ip6_tnl_map_e *t,
				    const struct ipv6hdr *ipv6h,
				    struct sk_buff *skb))
{
	struct ip6_tnl_map_e *t;
	const struct ipv6hdr *ipv6h = ipv6_hdr(skb);
	int err;

	rcu_read_lock();

	if ((t = ip6_tnl_lookup_map_e(dev_net(skb->dev), &ipv6h->daddr)) != NULL) {
		struct pcpu_tstats *tstats;

		if (t->parms.proto != ipproto && t->parms.proto != 0) {
			rcu_read_unlock();
			goto discard;
		}

		secpath_reset(skb);

		struct ipv6hdr *tip6h = ipv6_hdr(skb);
		struct iphdr *tip4h = (struct iphdr *)(tip6h + 1);

		if (tip4h->frag_off & htons(IP_MF | IP_OFFSET)) {
			err = map_e_frag_rcv(skb, t);
			if (err < 0 ) {
				rcu_read_unlock();
				goto discard;
			}else if( 1 == err ){
				rcu_read_unlock();
				return 0;
			}
		}else{
			err = map_e_rcv(skb, t);	/* MAP-E Υϥɥ */
			if (err < 0) {
				rcu_read_unlock();
				goto discard;
			}
		}

		memset(skb->cb, 0, sizeof(struct inet6_skb_parm));

		tstats = this_cpu_ptr(t->dev->tstats);
		tstats->rx_packets++;
		tstats->rx_bytes += skb->len;

		__skb_tunnel_rx(skb, t->dev);

	#ifdef CONFIG_RTL_MAP_E_FASTPATH_SUPPORT_DPI_ENGINE
		skb->mape_pkt = MAPE_PCK_FLAG;
	#endif

		netif_rx(skb);
		rcu_read_unlock();


		return 0;
	}
	rcu_read_unlock();
	return 1;

discard:
	kfree_skb(skb);
	return 0;
}

static int
ip4ip6_rcv_map_e(struct sk_buff *skb)
{
	return ip6_tnl_rcv_map_e(skb, ETH_P_IP, IPPROTO_IPIP,
			   NULL);
}

struct ipv6_tel_txoption_map_e {
	struct ipv6_txoptions ops;
	__u8 dst_opt[8];
};

static void
init_tel_txopt_map_e(struct ipv6_tel_txoption_map_e *opt, __u8 encap_limit)
{
	memset(opt, 0, sizeof(struct ipv6_tel_txoption_map_e));

	opt->dst_opt[2] = IPV6_TLV_TNL_ENCAP_LIMIT;
	opt->dst_opt[3] = 1;
	opt->dst_opt[4] = encap_limit;
	opt->dst_opt[5] = IPV6_TLV_PADN;
	opt->dst_opt[6] = 1;

	opt->ops.dst0opt = (struct ipv6_opt_hdr *) opt->dst_opt;
	opt->ops.opt_nflen = 8;
}

/**
 * ip6_tnl_xmit2_map_e - encapsulate packet and send
 *   @skb: the outgoing socket buffer
 *   @dev: the outgoing tunnel device
 *   @dsfield: dscp code for outer header
 *   @fl: flow of tunneled packet
 *   @encap_limit: encapsulation limit
 *   @pmtu: Path MTU is stored if packet is too big
 *
 * Description:
 *   Build new header and do some sanity checks on the packet before sending
 *   it.
 *
 * Return:
 *   0 on success
 *   -1 fail
 *   %-EMSGSIZE message too big. return mtu in this case.
 **/

static int
ip6_tnl_xmit2_map_e(struct sk_buff *skb,
	 struct net_device *dev,
	 __u8 dsfield,
	 struct flowi6 *fl6,
	 int encap_limit,
	 __u32 *pmtu)
{
	struct net *net = dev_net(dev);
	struct ip6_tnl_map_e *t = netdev_priv(dev);
	struct net_device_stats *stats = &t->dev->stats;
	struct ipv6_tel_txoption_map_e opt;
	struct dst_entry *dst = NULL, *ndst = NULL;
	struct net_device *tdev;
	int mtu;
	unsigned int max_headroom = sizeof(struct ipv6hdr);
	u8 proto;
	int err = -1;
	int pkt_len;

	if (!fl6->flowi6_mark)
		dst = ip6_tnl_dst_check_map_e(t);
	if (!dst) {
		ndst = ip6_route_output(net, NULL, fl6);

		if (ndst->error)
			goto tx_err_link_failure;
		ndst = xfrm_lookup(net, ndst, flowi6_to_flowi(fl6), NULL, 0);
		if (IS_ERR(ndst)) {
			err = PTR_ERR(ndst);
			ndst = NULL;
			goto tx_err_link_failure;
		}
		dst = ndst;
	}

	tdev = dst->dev;

	if (tdev == dev) {
		stats->collisions++;
		if (net_ratelimit())
			printk(KERN_WARNING
			       "%s: Local routing loop detected!\n",
			       t->parms.name);
		goto tx_err_dst_release;
	}
	mtu = dst_mtu(dst) - sizeof(struct ipv6hdr);
#if 0
	if (encap_limit >= 0) {
		max_headroom += 8;
		mtu -= 8;
	}
#endif
	if (mtu < IPV6_MIN_MTU)
		mtu = IPV6_MIN_MTU;
	if (skb_dst(skb))
		skb_dst(skb)->ops->update_pmtu(skb_dst(skb), NULL, skb, mtu);

	/* DFeth1mtuĶäѹ롪 */
	MAP_TRACE("dst->dev->name : %s, mtu : %d, DF : 0x%08x",
	    dst->dev->name, mtu, ip_hdr(skb)->frag_off & htons(IP_DF));
	if (skb->len > mtu && ip_hdr(skb)->frag_off & htons(IP_DF)) {
		*pmtu = mtu;
		err = -EMSGSIZE;
		goto tx_err_dst_release;
	}

	/*
	 * Okay, now see if we can stuff it in the buffer as-is.
	 */
	max_headroom += LL_RESERVED_SPACE(tdev);

	if (skb_headroom(skb) < max_headroom || skb_shared(skb) ||
	    (skb_cloned(skb) && !skb_clone_writable(skb, 0))) {
		struct sk_buff *new_skb;

		if (!(new_skb = skb_realloc_headroom(skb, max_headroom)))
			goto tx_err_dst_release;

		if (skb->sk)
			skb_set_owner_w(new_skb, skb->sk);
		kfree_skb(skb);
		skb = new_skb;
	}
	skb_dst_drop(skb);
	if (fl6->flowi6_mark) {
		skb_dst_set(skb, dst);
		ndst = NULL;
	} else {
		skb_dst_set_noref(skb, dst);
	}
	skb->transport_header = skb->network_header;

	proto = fl6->flowi6_proto;
	if (encap_limit >= 0) {
		init_tel_txopt_map_e(&opt, encap_limit);
		ipv6_push_nfrag_opts(skb, &opt.ops, &proto, NULL);
	}

	MAP_TRACE("reassemble?, ip_len:%u, skb_len:%d", ntohs(ip_hdr(skb)->tot_len), skb->len);
	if (ip_hdr(skb)->frag_off & htons(IP_MF | IP_OFFSET)) {
		MAP_TRACE("reassemble start");
		if (ip_defrag(skb, IP_DEFRAG_LOCAL_DELIVER)) {
			MAP_TRACE("Will be discard? or reassemble");
			dst_release(ndst);
			return 0;
		}
		MAP_TRACE("reassemble OK!");
	}

	err = map_e_xmit(skb, fl6, proto, t, dsfield);	/* MAP-E Υϥɥ */
	if (err < 0) {
		goto tx_err_dst_release;
	}

	nf_reset(skb);
	pkt_len = skb->len;
	err = ip6_local_out(skb);

	if (net_xmit_eval(err) == 0) {
		struct pcpu_tstats *tstats = this_cpu_ptr(t->dev->tstats);

		tstats->tx_bytes += pkt_len;
		tstats->tx_packets++;
	} else {
		stats->tx_errors++;
		stats->tx_aborted_errors++;
	}
	if (ndst)
		ip6_tnl_dst_store_map_e(t, ndst);
	return 0;
tx_err_link_failure:
	stats->tx_carrier_errors++;
	dst_link_failure(skb);
tx_err_dst_release:
	dst_release(ndst);
	return err;
}

static inline int
ip4ip6_tnl_xmit_map_e(struct sk_buff *skb, struct net_device *dev)
{
	struct ip6_tnl_map_e *t = netdev_priv(dev);
	const struct iphdr  *iph = ip_hdr(skb);
	int encap_limit = -1;
	struct flowi6 fl6;
	__u8 dsfield;
	__u32 mtu = 0;
	int err;


	if (!(t->parms.flags & IP6_TNL_F_IGN_ENCAP_LIMIT))
		encap_limit = t->parms.encap_limit;

	memcpy(&fl6, &t->fl.u.ip6, sizeof (fl6));
	fl6.flowi6_proto = IPPROTO_IPIP;

	dsfield = ipv4_get_dsfield(iph);

	if (t->parms.flags & IP6_TNL_F_USE_ORIG_TCLASS)
		fl6.flowlabel |= htonl((__u32)iph->tos << IPV6_TCLASS_SHIFT)
					  & IPV6_TCLASS_MASK;
	if (t->parms.flags & IP6_TNL_F_USE_ORIG_FWMARK)
		fl6.flowi6_mark = skb->mark;

	err = ip6_tnl_xmit2_map_e(skb, dev, dsfield, &fl6, encap_limit, &mtu);
	if (err != 0) {
		/* XXX: send ICMP error even if DF is not set. */
		if (err == -EMSGSIZE)
			icmp_send(skb, ICMP_DEST_UNREACH, ICMP_FRAG_NEEDED,
				  htonl(mtu));
		return -1;
	}

	return 0;
}

static netdev_tx_t
ip6_tnl_xmit_map_e(struct sk_buff *skb, struct net_device *dev)
{
	struct ip6_tnl_map_e *t = netdev_priv(dev);
	struct net_device_stats *stats = &t->dev->stats;
	int ret;

	if (g_map_e_deinitializing)
		goto tx_err;

	switch (skb->protocol) {
	case htons(ETH_P_IP):
		ret = ip4ip6_tnl_xmit_map_e(skb, dev);
		break;
	default:
		goto tx_err;
	}

	if (ret < 0)
		goto tx_err;

	return NETDEV_TX_OK;

tx_err:
	stats->tx_errors++;
	stats->tx_dropped++;
	kfree_skb(skb);
	return NETDEV_TX_OK;
}

static void
ip6_tnl_set_cap_map_e(struct ip6_tnl_map_e *t)
{
	struct map_e_parm *p = &t->parms;

	p->flags &= ~(IP6_TNL_F_CAP_XMIT|IP6_TNL_F_CAP_RCV);

	p->flags |= IP6_TNL_F_CAP_XMIT;
	p->flags |= IP6_TNL_F_CAP_RCV;
}

static void
ip6_tnl_link_config_map_e(struct ip6_tnl_map_e *t)
{
	struct net_device *dev = t->dev;
	struct map_e_parm *p = &t->parms;
	struct flowi6 *fl6 = &t->fl.u.ip6;

	memcpy(dev->dev_addr, &p->map_e_v6, sizeof(struct in6_addr));
	memcpy(dev->broadcast, &p->dmr_v6, sizeof(struct in6_addr));

	/* Set up flowi template */
	memcpy(&fl6->saddr, &p->map_e_v6, sizeof(struct in6_addr));
	memcpy(&fl6->daddr, &p->dmr_v6, sizeof(struct in6_addr));

	fl6->flowi6_oif = p->link;
	fl6->flowlabel = 0;

	if (!(p->flags&IP6_TNL_F_USE_ORIG_TCLASS))
		fl6->flowlabel |= IPV6_TCLASS_MASK & p->flowinfo;
	if (!(p->flags&IP6_TNL_F_USE_ORIG_FLOWLABEL))
		fl6->flowlabel |= IPV6_FLOWLABEL_MASK & p->flowinfo;

	ip6_tnl_set_cap_map_e(t);

	if (p->flags&IP6_TNL_F_CAP_XMIT && p->flags&IP6_TNL_F_CAP_RCV)
		dev->flags |= IFF_POINTOPOINT;
	else
		dev->flags &= ~IFF_POINTOPOINT;

	dev->iflink = p->link;

	if (p->flags & IP6_TNL_F_CAP_XMIT) {
		int strict = (ipv6_addr_type(&p->dmr_v6) &
		    (IPV6_ADDR_MULTICAST|IPV6_ADDR_LINKLOCAL));

		struct rt6_info *rt = rt6_lookup(dev_net(dev),
		    &p->dmr_v6, &p->map_e_v6, p->link, strict);

		if (rt == NULL)
			return;

		if (rt->dst.dev) {
			dev->hard_header_len = rt->dst.dev->hard_header_len +
				sizeof (struct ipv6hdr);

			if (!(t->parms.flags & IP6_TNL_F_IGN_ENCAP_LIMIT))
				dev->mtu-=8;

			if (dev->mtu < IPV6_MIN_MTU)
				dev->mtu = IPV6_MIN_MTU;
		}
		dst_release(&rt->dst);
	}
}

/**
 * ip6_tnl_change_map_e - update the tunnel parameters
 *   @t: tunnel to be changed
 *   @p: tunnel configuration parameters
 *
 * Description:
 *   ip6_tnl_change_map_e() updates the tunnel parameters
 **/

static int
ip6_tnl_change_map_e(struct ip6_tnl_map_e *t, struct map_e_parm *p)
{
	struct map_e_parm *rcv_p = NULL;
	struct map_e_portset *ps = NULL;
	struct map_e_rule *rule = NULL;
	struct map_e_rule *new_rule = NULL;
	struct map_e_rule *new_rule_ptr = NULL;
	int idx;
	int ret;

	MAP_TRACE("called");

	/*
	 * MAP-Eѥ롼βϡ
	 */
	rcv_p = p;

        memcpy(&t->parms.map_e_v6, &rcv_p->map_e_v6, sizeof(struct in6_addr));
        memcpy(&t->parms.dmr_v6, &rcv_p->dmr_v6, sizeof(struct in6_addr));

	t->parms.flags = rcv_p->flags;
	t->parms.hop_limit = rcv_p->hop_limit;
	t->parms.encap_limit = rcv_p->encap_limit;
	t->parms.flowinfo = rcv_p->flowinfo;
	t->parms.link = rcv_p->link;
	t->parms.proto = rcv_p->proto;
	t->parms.rule_max = rcv_p->rule_max;
	t->parms.session_max = rcv_p->session_max;

	/*  t->map_e_info β */
	if (t->map_e_info) {
		/* portsetõ */
		while (!list_empty((struct list_head *)&t->map_e_info->portset_list.list)) {
			ps = list_first_entry(&t->map_e_info->portset_list.list, struct map_e_portset, list);
			list_del((struct list_head *)ps);
			KFREE(ps);
		}

		if (!list_empty((struct list_head *)&t->map_e_info->fmr_list)) {
			rule = list_first_entry((struct list_head *)(&t->map_e_info->fmr_list),
			    struct map_e_rule, list);
			if (rule) {
				/* ruleϤޤȤƼƤΤǡǽΤҤȤĤfreeɤ */
				KFREE(rule);
			}
		}

		KFREE(t->map_e_info);
		t->map_e_info = NULL;
	}

	t->map_e_info = KMALLOC(sizeof(struct map_e_info), GFP_KERNEL);
	if (t->map_e_info == NULL) {
		return -ENOMEM;
	}

	memset(t->map_e_info, 0x00, sizeof(struct map_e_info));
	INIT_LIST_HEAD(&t->map_e_info->fmr_list.list);
	INIT_LIST_HEAD(&t->map_e_info->portset_list.list);
	for (idx = 0; idx < MAP_E_HASH_SIZE; idx++) {
		INIT_LIST_HEAD(&t->map_e_info->ses_list[idx].list);
	}
	t->map_e_info->ses_num = 0;
	INIT_LIST_HEAD(&t->map_e_info->staticnapt_list.list);
	t->map_e_info->staticnapt_num = 0;

	memcpy(&t->map_e_info->wan_addr, &rcv_p->map_e_v6, sizeof(struct in6_addr));
	memcpy(&t->map_e_info->shared_addr4, &rcv_p->shared_v4, sizeof(struct in_addr));
	memcpy(&t->map_e_info->dmr_addr, &rcv_p->dmr_v6, sizeof(struct in6_addr));

	for (idx = 0; idx < rcv_p->rule_num; idx++) {
		if (idx == 0) {
			/* ǽ1malloc */
			new_rule_ptr = KMALLOC(sizeof(struct map_e_rule) * rcv_p->rule_num, GFP_KERNEL);
			if (new_rule_ptr == NULL) {
				KFREE(t->map_e_info);
				t->map_e_info = NULL;
				return -ENOMEM;
			}
			MAP_TRACE("new_rule_ptr : 0x%08x", (int)new_rule_ptr);
		}

		rule = (struct map_e_rule *)(&rcv_p->rule[idx]);
		new_rule = new_rule_ptr + idx;

		memcpy(&new_rule->v6_prefix, &rule->v6_prefix, sizeof(new_rule->v6_prefix));
		new_rule->v6_prefix_len = rule->v6_prefix_len;
		memcpy(&new_rule->v4_prefix, &rule->v4_prefix, sizeof(new_rule->v4_prefix));
		new_rule->v4_prefix_len = rule->v4_prefix_len;
		new_rule->eabits_len = rule->eabits_len;
		new_rule->psid_offset = rule->psid_offset;

		if (rule->bmr) {
			MAP_TRACE("select bmr [%d]", idx);
			t->map_e_info->bmr = new_rule;
		}

		list_add_tail((struct list_head *)new_rule, &t->map_e_info->fmr_list.list);
	}

	/*
	 * ݡȥåȤη׻򤹤
	 */
	ret = map_e_portset_init(p->ps_id, p->ps_len, t);	/* MAP-EΥϥɥ */
	if (ret < 0) {
		if (!list_empty((struct list_head *)&t->map_e_info->fmr_list)) {
			rule = list_first_entry((struct list_head *)(&t->map_e_info->fmr_list),
			    struct map_e_rule, list);
			if (rule) {
				/* ruleϤޤȤƼƤΤǡǽΤҤȤĤfreeɤ */
				KFREE(rule);
			}
		}
		KFREE(t->map_e_info);
		t->map_e_info = NULL;
		return -EINVAL;
	}

	ip6_tnl_dst_reset_map_e(t);
	ip6_tnl_link_config_map_e(t);

	return 0;
}

/**
 * ip6_tnl_ioctl_map_e - configure ipv6 tunnels from userspace
 *   @dev: virtual device associated with tunnel
 *   @ifr: parameters passed from userspace
 *   @cmd: command to be performed
 *
 * Description:
 *   ip6_tnl_ioctl_map_e() is used for managing IPv6 tunnels
 *   from userspace.
 *
 *   The possible commands are the following:
 *     %SIOCGETTUNNEL: get tunnel parameters for device
 *     %SIOCADDTUNNEL: add tunnel matching given tunnel parameters
 *     %SIOCCHGTUNNEL: change tunnel parameters to those given
 *     %SIOCDELTUNNEL: delete tunnel
 *
 *   The fallback device "map0", created during module
 *   initialization, can be used for creating other tunnel devices.
 *
 * Return:
 *   0 on success,
 *   %-EFAULT if unable to copy data to or from userspace,
 *   %-EPERM if current process hasn't %CAP_NET_ADMIN set
 *   %-EINVAL if passed tunnel parameters are invalid,
 *   %-EEXIST if changing a tunnel's parameters would cause a conflict
 *   %-ENODEV if attempting to change or delete a nonexisting device
 **/

static int
ip6_tnl_ioctl_map_e(struct net_device *dev, struct ifreq *ifr, int cmd)
{
	int err = 0;
	struct map_e_parm *p = NULL;
	struct map_e_parm_staticnapt *p_staticnapt = NULL;
	struct ip6_tnl_map_e *t = NULL;
	struct net *net = dev_net(dev);
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);

	MAP_TRACE("dev->name : %s", dev ? dev->name : " ");
	switch (cmd) {
	case SIOCGET_MAPE:
		p = KMALLOC(sizeof(struct map_e_parm), GFP_KERNEL);
		if (p == NULL) {
			printk(KERN_ERR "KMALLOC error.");
			return -ENOMEM;
		}
		p_staticnapt = KMALLOC(sizeof(struct map_e_parm_staticnapt), GFP_KERNEL);
		if (p_staticnapt == NULL) {
			printk(KERN_ERR "KMALLOC error.");
			if (p) {
				KFREE(p);
			}
			return -ENOMEM;
		}
		break;
	case SIOCADD_MAPE:
	case SIOCDEL_MAPE:
		p = KMALLOC(sizeof(struct map_e_parm), GFP_KERNEL);
		if (p == NULL) {
			printk(KERN_ERR "KMALLOC error.");
			return -ENOMEM;
		}
		break;
	case SIOCGET_STATICNAPT:
	case SIOCADD_STATICNAPT:
	case SIOCDEL_STATICNAPT:
	case SIOCFLSH_STATICNAPT:
		p_staticnapt = KMALLOC(sizeof(struct map_e_parm_staticnapt), GFP_KERNEL);
		if (p_staticnapt == NULL) {
			printk(KERN_ERR "KMALLOC error.");
			return -ENOMEM;
		}
		break;
	default:
		return -EINVAL;
	}

	switch (cmd) {
	case SIOCGET_MAPE:
		if (dev == ip6n->fb_tnl_dev) {
			if (copy_from_user(p, ifr->ifr_ifru.ifru_data, sizeof(*p))) {
				err = -EFAULT;
				MAP_TRACE("copy_from_user() error (1)");
				break;
			}
			t = ip6_tnl_locate_map_e(net, p, 0);
		}
		if (t == NULL) {
			if (copy_from_user(p, ifr->ifr_ifru.ifru_data, sizeof(*p))) {
				err = -EFAULT;
				MAP_TRACE("copy_from_user() error (2)");
				break;
			}
			t = netdev_priv(dev);
		}

		if (t && t->map_e_info) {
			/* ͥν(MAP-E) */
			struct list_head *hd;
			struct map_e_rule *rule = NULL;
			struct map_e_portset *ps = NULL;
			struct map_e_session *ses = NULL;
			int count, sleep_count=0;
			uint32_t hash_index;
			char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];
			char buf3[INET6_ADDRSTRLEN + 1], buf4[INET6_ADDRSTRLEN + 1];
			char buf5[INET6_ADDRSTRLEN + 1];
			char proto_str[10];

			MAP_TRACE("start show command, t->dev->name:%s, t->map_e_info:0x%08x",
			    t->dev ? t->dev->name : " ", (int)t->map_e_info);
			if (p->show_cmd & MAP_E_SHOW_USERLAND) {
				if (p->show_cmd & MAP_E_SHOW_PORTSET) {
					t->parms.portset_num = t->map_e_info->portset_num;
					memcpy(p, &t->parms, sizeof(*p));
					if (copy_to_user(ifr->ifr_ifru.ifru_data, p, sizeof(*p))) {
						err = -EFAULT;
						break;
					}
				} else if (p->show_cmd & MAP_E_SHOW_PORTSET2) {
					map_e_portset_array port_set;
					map_e_portset_array *user_port_set_buf =
						((struct map_e_parm *)ifr->ifr_ifru.ifru_data)->port_set;
					list_for_each(hd, &t->map_e_info->portset_list.list) {
						ps = list_entry(hd, struct map_e_portset, list);
						port_set.port_min = ps->port_min;
						port_set.port_max = ps->port_max;
						if (copy_to_user(user_port_set_buf, &port_set, sizeof(port_set))) {
							err = -EFAULT;
							break;
						}
						user_port_set_buf++;
					}
				} else {
					printk("not support show cmd.\n");
				}
				break;
			}
			switch (p->show_cmd) {
			case MAP_E_SHOW_RULE: {
				printk("debug : %s/%s\n", t->map_e_info->debug ? "on" : "off", g_debug ? "on" : "off");

				map_e_ntop(AF_INET6, &t->map_e_info->wan_addr, buf1, sizeof(buf1));
				map_e_ntop(AF_INET,  &t->map_e_info->shared_addr4, buf2, sizeof(buf2));
				map_e_ntop(AF_INET6, &t->map_e_info->dmr_addr, buf3, sizeof(buf3));
				printk("MAP IPv6 address(WAN-IF address) : %s\n", buf1);
				printk("Shared IPv4 address              : %s\n", buf2);
				printk("default mapping rule(BR address) : %s\n", buf3);

				count = 0;
				list_for_each(hd, &t->map_e_info->fmr_list.list) {
					rule = list_entry(hd, struct map_e_rule, list);

					map_e_ntop(AF_INET6, &rule->v6_prefix, buf1, sizeof(buf1));
					map_e_ntop(AF_INET,  &rule->v4_prefix, buf2, sizeof(buf2));

					printk("[Rule%03d%s]\n", count, t->map_e_info->bmr == rule ? "*" : "");
					printk("  IPv6 Prefix/Len : %s/%u\n", buf1, rule->v6_prefix_len & 0xFF);
					printk("  IPv4 Prefix/Len : %s/%u\n", buf2, rule->v4_prefix_len & 0xFF);
					printk("  EAbits Len      : %u\n", rule->eabits_len & 0xFF);
					printk("  PSID Offset     : %u\n", rule->psid_offset & 0xFF);
					count++;
				}
			}
				break;
			case MAP_E_SHOW_PORTSET: {
				count = 0;
				list_for_each(hd, &t->map_e_info->portset_list.list) {
					ps = list_entry(hd, struct map_e_portset, list);
					printk("[%03d] %5u - %5u\n", count, ps->port_min, ps->port_max);
					count++;
					sleep_count++;
					if (sleep_count%100 == 0) {
						msleep(10);
					}
				}
			}
				break;
			case MAP_E_SHOW_SESSION: {
				for (hash_index = 0; hash_index < MAP_E_HASH_SIZE; hash_index++) {
					count = 0;
					list_for_each(hd, &t->map_e_info->ses_list[hash_index].list) {
						ses = list_entry(hd, struct map_e_session, list);
						if (count == 0) {
							printk("[hash : %04x]\n", hash_index);
						}
						switch (ses->protocol) {
						case IPPROTO_TCP:
							strlcpy(proto_str, "TCP", sizeof(proto_str));
							break;
						case IPPROTO_UDP:
							strlcpy(proto_str, "UDP", sizeof(proto_str));
							break;
						case IPPROTO_ICMP:
							strlcpy(proto_str, "ICMP", sizeof(proto_str));
							break;
						default:
							strlcpy(proto_str, "TCP", sizeof(proto_str));
							snprintf(proto_str, sizeof(proto_str), "0x%02x", ses->protocol & 0xFF);
							break;
						}
						map_e_ntop(AF_INET6, &ses->dst6, buf1, sizeof(buf1));
						map_e_ntop(AF_INET6, &ses->src6, buf2, sizeof(buf2));
						map_e_ntop(AF_INET, &ses->src4, buf3, sizeof(buf3));
						map_e_ntop(AF_INET, &ses->c_src4, buf4, sizeof(buf4));
						map_e_ntop(AF_INET, &ses->dst4, buf5, sizeof(buf5));
						printk("  <%d> IPv6:%s/%s, IPv4:%s->%s/%s, %s:%u->%u/%u, expire:%d\n",
						    count, buf2, buf1, buf3, buf4, buf5,
						    proto_str, ntohs(ses->sport), ntohs(ses->c_sport), ntohs(ses->dport), ses->expire);
						count++;
						sleep_count++;
						if (sleep_count%100 == 0) {
							msleep(10);
						}
					}
				}

			}
				break;
			case MAP_E_SHOW_PARAM: {
				printk("Protocol       : 0x%02x\n", t->parms.proto);
				printk("Encap Limit    : %u\n", t->parms.encap_limit & 0xFF);
				printk("Hop Limit      : %u\n", t->parms.hop_limit & 0xFF);
				printk("Rule MAX       : %u\n", t->parms.rule_max);
				printk("Session MAX    : %u\n", t->parms.session_max);
				printk("Portset-Id     : 0x%04x\n", t->parms.ps_id);
				printk("Portset-Id Len : %u\n", t->parms.ps_len);
			}
				break;
			case MAP_E_SHOW_STATUS: {
				count = 0;
				for (hash_index = 0; hash_index < MAP_E_HASH_SIZE; hash_index++) {
					list_for_each(hd, &t->map_e_info->ses_list[hash_index].list) {
						count++;
					}
				}
				printk("Session Status : %d/%u\n", count, t->parms.session_max);
			}
				break;
			case MAP_E_SHOW_STATICNAPT: {
				if (copy_from_user(p_staticnapt, ifr->ifr_ifru.ifru_data, sizeof(*p_staticnapt))) {
					err = -EFAULT;
					printk("MAP_E_SHOW_STATICNAPT: copy_from_user\n");
					break;
				}
				map_e_show_staticnapt(t, p_staticnapt);
			}
				break;
			default:
				MAP_TRACE("## unknown(0x%08x) ##", p->show_cmd);
				memcpy(p, &t->parms, sizeof(*p));
				if (copy_to_user(ifr->ifr_ifru.ifru_data, p, sizeof(*p))) {
					err = -EFAULT;
				}
				break;
			}
		} else {
			MAP_TRACE("t is null");
			memcpy(p, &t->parms, sizeof(*p));
			if (copy_to_user(ifr->ifr_ifru.ifru_data, p, sizeof(*p))) {
				err = -EFAULT;
			}
		}

		break;
	case SIOCADD_MAPE:
		err = -EPERM;
		if (!capable(CAP_NET_ADMIN)) {
			MAP_TRACE("SIOCADD_MAPE: capable(CAP_NET_ADMIN)");
			break;
		}
		err = -EFAULT;
		if (copy_from_user(p, ifr->ifr_ifru.ifru_data, sizeof(*p))) {
			MAP_TRACE("SIOCADD_MAPE: copy_from_user");
			break;
		}
		err = -EINVAL;
		if (p->proto != IPPROTO_IPIP && p->proto != 0) {
			MAP_TRACE("SIOCADD_MAPE: protocol");
			break;
		}

		/* MAP-E⥸塼ɬפ꤬Сǽ */
		if (p->debug) {
			MAP_TRACE("p->debug = 1");
			map_e_ioctl(dev, ifr, 1);
			g_debug = 1;
		} else {
			MAP_TRACE("p->debug = 0");
			map_e_ioctl(dev, ifr, 0);
			g_debug = 0;
		}

		/* set TCP and UDP session timer */
		if (set_session_timer(p->tcp_expire, IPPROTO_TCP) < 0) {
			MAP_TRACE("SIOCADD_SESSION_TIMER: fail to set session timer.");
		}
		if (set_session_timer(p->udp_expire, IPPROTO_UDP) < 0) {
			MAP_TRACE("SIOCADD_SESSION_TIMER: fail to set session timer.");
		}

		/* ȥͥǥХθȡ */
		t = ip6_tnl_locate_map_e(net, p, cmd == SIOCADD_MAPE);

		if (t) {
			MAP_TRACE("t->map_e_info : 0x%08x", (int)t->map_e_info);
			g_xmap_dev = t->dev;
			t->map_e_info = NULL;

			ip6_tnl_unlink_map_e(ip6n, t);
			err = ip6_tnl_change_map_e(t, p);
			if (err < 0) {
				printk("[%s] ip6_tnl_change_map_e() error\n", __func__);
			} else {
				map_e_session_init(t);
			}
			ip6_tnl_link_map_e(ip6n, t);
			netdev_state_change(dev);
		#ifdef CONFIG_RTL_FAST_MAP_E	
		#ifdef CONFIG_RTL_MAP_E_FASTPATH_SUPPORT_DPI_ENGINE
			strncpy(map_e_tunnel_name, t->dev->name, IFNAMSIZ);
		#endif
			map_e_fastpath_t = t;
		#endif

		} else {
			printk("[%s] It has already started.\n", __func__);
			err = (cmd == SIOCADD_MAPE ? -ENOBUFS : -ENOENT);
		}
		break;

	case SIOCDEL_MAPE:
		err = -EPERM;
		if (!capable(CAP_NET_ADMIN)) {
			MAP_TRACE("SIOCDEL_MAPE: capable(CAP_NET_ADMIN)");
			break;
		}

		if (dev == ip6n->fb_tnl_dev) {
			err = -EFAULT;
			if (copy_from_user(p, ifr->ifr_ifru.ifru_data, sizeof(*p))) {
				MAP_TRACE("SIOCDEL_MAPE: copy_from_user");
				break;
			}
			err = -ENOENT;
			if ((t = ip6_tnl_locate_map_e(net, p, 0)) == NULL) {
				MAP_TRACE("SIOCDEL_MAPE: ip6_tnl_locate_map_e");
				break;
			}
			err = -EPERM;
			if (t->dev == ip6n->fb_tnl_dev) {
				MAP_TRACE("SIOCDEL_MAPE: different device");
				break;
			}
			dev = t->dev;
		}
		err = 0;
		synchronize_net();
		unregister_netdevice(dev);
		break;
	case SIOCGET_STATICNAPT:
		if (copy_from_user(p_staticnapt, ifr->ifr_ifru.ifru_data, sizeof(*p_staticnapt))) {
			err = -EFAULT;
			MAP_TRACE("SIOCGET_STATICNAPT: copy_from_user");
			break;
		}
		t = ip6_tnl_locate_map_e2(net, ifr->ifr_name);
		if (t == NULL) {
			err = -EINVAL;
			break;
		}
		map_e_show_staticnapt(t, p_staticnapt);
		break;
	case SIOCADD_STATICNAPT:
		if (copy_from_user(p_staticnapt, ifr->ifr_ifru.ifru_data, sizeof(*p_staticnapt))) {
			err = -EFAULT;
			MAP_TRACE("SIOCADD_STATICNAPT: copy_from_user");
			break;
		}
		t = ip6_tnl_locate_map_e2(net, ifr->ifr_name);
		if (t) {
			synchronize_net();
			map_e_flush_staticnapt(t);
			map_e_session_clear(t);

			map_e_add_staticnapt(t, p_staticnapt);
		}

		break;
	case SIOCDEL_STATICNAPT:
		if (copy_from_user(p_staticnapt, ifr->ifr_ifru.ifru_data, sizeof(*p_staticnapt))) {
			err = -EFAULT;
			MAP_TRACE("SIOCDEL_STATICNAPT: copy_from_user");
			break;
		}
		t = ip6_tnl_locate_map_e2(net, ifr->ifr_name);
		if (t == NULL) {
			err = -EINVAL;
			break;
		}
		synchronize_net();
		map_e_del_staticnapt(t, p_staticnapt);
		break;
	case SIOCFLSH_STATICNAPT:
		t = ip6_tnl_locate_map_e2(net, ifr->ifr_name);
		if (t == NULL) {
			err = -EINVAL;
			break;
		}
		synchronize_net();
		map_e_flush_staticnapt(t);
		map_e_session_clear(t);
		break;
	}

	if (p) {
		KFREE(p);
	}
	if (p_staticnapt) {
		KFREE(p_staticnapt);
	}
	return err;
}

/**
 * ip6_tnl_change_mtu_map_e - change mtu manually for tunnel device
 *   @dev: virtual device associated with tunnel
 *   @new_mtu: the new mtu
 *
 * Return:
 *   0 on success,
 *   %-EINVAL if mtu too small
 **/

static int
ip6_tnl_change_mtu_map_e(struct net_device *dev, int new_mtu)
{
	if (new_mtu < IPV6_MIN_MTU) {
		return -EINVAL;
	}
	dev->mtu = new_mtu;
	return 0;
}

static const struct net_device_ops xmap_tnl_netdev_ops = {
	.ndo_uninit	= ip6_tnl_dev_uninit_map_e,
	.ndo_start_xmit = ip6_tnl_xmit_map_e,
	.ndo_do_ioctl	= ip6_tnl_ioctl_map_e,
	.ndo_change_mtu = ip6_tnl_change_mtu_map_e,
	.ndo_get_stats	= ip6_get_stats_map_e,
};


/**
 * ip6_tnl_dev_setup_map_e - setup virtual tunnel device
 *   @dev: virtual device associated with tunnel
 *
 * Description:
 *   Initialize function pointers and device parameters
 **/

static void
ip6_tnl_dev_setup_map_e(struct net_device *dev)
{
	struct ip6_tnl_map_e *t;

	MAP_TRACE("called");

	dev->netdev_ops = &xmap_tnl_netdev_ops;
	dev->destructor = ip6_dev_free_map_e;

	dev->type = ARPHRD_TUNNEL6;
	dev->hard_header_len = LL_MAX_HEADER + sizeof (struct ipv6hdr);

#define MAP_E_MTU	(65535)
	dev->mtu = MAP_E_MTU;
	t = netdev_priv(dev);
	if (!(t->parms.flags & IP6_TNL_F_IGN_ENCAP_LIMIT))
		dev->mtu-=8;
	dev->flags |= IFF_NOARP;
	dev->addr_len = sizeof(struct in6_addr);
	dev->features |= NETIF_F_NETNS_LOCAL;
	dev->priv_flags &= ~IFF_XMIT_DST_RELEASE;
}


/**
 * ip6_tnl_dev_init_gen_map_e - general initializer for all tunnel devices
 *   @dev: virtual device associated with tunnel
 **/

static inline int
ip6_tnl_dev_init_gen_map_e(struct net_device *dev)
{
	struct ip6_tnl_map_e *t = netdev_priv(dev);

	MAP_TRACE("called");

	t->dev = dev;
	dev->tstats = alloc_percpu(struct pcpu_tstats);
	if (!dev->tstats)
		return -ENOMEM;
	return 0;
}

/**
 * ip6_tnl_dev_init_map_e - initializer for all non fallback tunnel devices
 *   @dev: virtual device associated with tunnel
 **/

static int
ip6_tnl_dev_init_map_e(struct net_device *dev)
{
	struct ip6_tnl_map_e *t = netdev_priv(dev);
	int err = ip6_tnl_dev_init_gen_map_e(dev);

	MAP_TRACE("called");

	if (err)
		return err;
	ip6_tnl_link_config_map_e(t);
	return 0;
}

/**
 * ip6_fb_tnl_dev_init_map_e - initializer for fallback tunnel device
 *   @dev: fallback device
 *
 * Return: 0
 **/

static int __net_init
ip6_fb_tnl_dev_init_map_e(struct net_device *dev)
{
	struct ip6_tnl_map_e *t = netdev_priv(dev);
	struct net *net = dev_net(dev);
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);
	int err = ip6_tnl_dev_init_gen_map_e(dev);

	MAP_TRACE("called");

	if (err)
		return err;

	t->parms.proto = IPPROTO_IPV6;
	dev_hold(dev);
	rcu_assign_pointer(ip6n->tnls_wc[0], t);
	return 0;
}

static struct xfrm6_tunnel ip4ip6_handler __read_mostly = {
	.handler	= ip4ip6_rcv_map_e,
	.err_handler	= ip4ip6_err_map_e,
	.priority	=	1,
};

static void __net_exit
ip6_tnl_destroy_tunnels_map_e(struct ip6_tnl_net_map_e *ip6n)
{
	int h;
	struct ip6_tnl_map_e *t = NULL;
	struct map_e_portset *ps = NULL;
	struct map_e_rule *rule = NULL;

	MAP_TRACE("called");

	LIST_HEAD(list);

	for (h = 0; h < HASH_SIZE; h++) {
		t = rtnl_dereference(ip6n->tnls_r_l[h]);
		MAP_TRACE("t->map_e_info : 0x%08x", t ? (int)t->map_e_info : 0);

		while (t != NULL) {
			if (t->map_e_info) {
				map_e_session_destroy(t);

				/* portsetõ */
				while (!list_empty((struct list_head *)&t->map_e_info->portset_list.list)) {
					ps = list_first_entry(&t->map_e_info->portset_list.list, struct map_e_portset, list);
					list_del((struct list_head *)ps);
					KFREE(ps);
				}

				/* ruleõ */
				if (!list_empty((struct list_head *)&t->map_e_info->fmr_list)) {
					rule = list_first_entry((struct list_head *)(&t->map_e_info->fmr_list),
					    struct map_e_rule, list);
					if (rule) {
						/* ruleϤޤȤƼƤΤǡǽΤҤȤĤfreeɤ */
						KFREE(rule);
					}
				}
				KFREE(t->map_e_info);
				t->map_e_info = NULL;
			}

			unregister_netdevice_queue(t->dev, &list);
			t = rtnl_dereference(t->next);
			MAP_TRACE("t->map_e_info : 0x%08x", t ? (int)t->map_e_info : 0);
		}
	}

	t = rtnl_dereference(ip6n->tnls_wc[0]);
	MAP_TRACE("t->map_e_info : 0x%08x", t ? (int)t->map_e_info : 0);
	unregister_netdevice_queue(t->dev, &list);
	unregister_netdevice_many(&list);
}

static int __net_init
ip6_tnl_init_net_map_e(struct net *net)
{
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);
	struct ip6_tnl_map_e *t = NULL;
	int err;

	MAP_TRACE("start");
	ip6n->tnls[0] = ip6n->tnls_wc;
	ip6n->tnls[1] = ip6n->tnls_r_l;

	err = -ENOMEM;
	ip6n->fb_tnl_dev = alloc_netdev(sizeof(struct ip6_tnl_map_e), "map0",
				      ip6_tnl_dev_setup_map_e);

	if (!ip6n->fb_tnl_dev)
		goto err_alloc_dev;
	dev_net_set(ip6n->fb_tnl_dev, net);

	err = ip6_fb_tnl_dev_init_map_e(ip6n->fb_tnl_dev);
	if (err < 0)
		goto err_register;

	err = register_netdev(ip6n->fb_tnl_dev);
	if (err < 0)
		goto err_register;

	t = netdev_priv(ip6n->fb_tnl_dev);

	strlcpy(t->parms.name, ip6n->fb_tnl_dev->name, sizeof(t->parms.name));
	return 0;

err_register:
	ip6_dev_free_map_e(ip6n->fb_tnl_dev);
err_alloc_dev:
	return err;
}

static void __net_exit
ip6_tnl_exit_net_map_e(struct net *net)
{
	struct ip6_tnl_net_map_e *ip6n = net_generic(net, ip6_tnl_net_id_map_e);
	g_map_e_deinitializing = 1;
	rtnl_lock();
	MAP_TRACE("start");
	ip6_tnl_destroy_tunnels_map_e(ip6n);
	rtnl_unlock();
}

static struct pernet_operations ip6_tnl_net_ops_map_e = {
	.init = ip6_tnl_init_net_map_e,
	.exit = ip6_tnl_exit_net_map_e,
	.id   = &ip6_tnl_net_id_map_e,
	.size = sizeof(struct ip6_tnl_net_map_e),
};

/**
 * ip6_tunnel_init_map_e - register protocol and reserve needed resources
 *
 * Return: 0 on success
 **/

static int __init
ip6_tunnel_init_map_e(void)
{
	int  err;

	MAP_TRACE("start");
	err = register_pernet_device(&ip6_tnl_net_ops_map_e);
	if (err < 0)
		goto out_pernet;

	err = xfrm6_tunnel_register(&ip4ip6_handler, AF_INET);
	if (err < 0) {
		printk(KERN_ERR "ip6_tunnel init: can't register ip4ip6\n");
		goto out_ip4ip6;
	}

	return 0;

out_ip4ip6:
	unregister_pernet_device(&ip6_tnl_net_ops_map_e);
out_pernet:
	return err;
}

/**
 * ip6_tunnel_cleanup_map_e - free resources and unregister protocol
 **/

static void __exit
ip6_tunnel_cleanup_map_e(void)
{
	MAP_TRACE("start");
	if (xfrm6_tunnel_deregister(&ip4ip6_handler, AF_INET))
		printk(KERN_INFO "ip6_tunnel close: can't deregister ip4ip6\n");

	unregister_pernet_device(&ip6_tnl_net_ops_map_e);
}

module_init(ip6_tunnel_init_map_e);
module_exit(ip6_tunnel_cleanup_map_e);
