/*

 *	Copyright(c) NEC Platforms, Ltd. 2001-2018

 */
#include <linux/errno.h>
#include <linux/icmp.h>
#include <linux/if_packet.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/ipv6.h>
#include <linux/module.h>
#include <linux/rwlock_types.h>
#include <linux/spinlock.h>

#include <net/ip.h>
#include <net/ipv6.h>
#include <net/tcp.h>
#include <net/dsfield.h>
#include <net/inet_ecn.h>
#include "map_e.h"

#ifdef CONFIG_RTL_FAST_MAP_E
#include <net/rtl/fastpath/fastpath_core.h>
#include <net/netfilter/nf_conntrack.h>
#include <linux/netfilter/nf_conntrack_tcp.h>
void rtl_add_mape_fastpath_entry(struct ip6_tnl_map_e *t, struct sk_buff *skb, struct ipv6hdr *ip6h, struct iphdr *ip4h, 
				uint32 old_addr, uint16_t old_port, int direction);

static int rtl_search_map_e_session(struct iphdr *ip4h, struct tcphdr *tcphupuh, u32 *ext_ip, u16 *ext_port, \
	struct in6_addr *src6, struct in6_addr *dst6, void **session, int direction);

static void rtl_delete_map_e_session(void *session);

struct ip6_tnl_map_e *map_e_fastpath_t = NULL;
EXPORT_SYMBOL(map_e_fastpath_t);
#endif

#define MAP_E_EXPIRE_DEFAULT_TCP	(300)
#define MAP_E_EXPIRE_DEFAULT_UDP	(300)
#define MAP_E_EXPIRE_DEFAULT_ICMP	(300)
#define MAP_E_EXPIRE_TCP_CLOSE_WAIT	(20)

#define FTP_CTL_PORT	(21)
#define random32() prandom_u32()	
#define srandom32(seed) prandom_seed(seed)

static int g_rlock_count = 0;
static int g_wlock_count = 0;

static int map_e_expire_tcp;
static int map_e_expire_udp;
rwlock_t map_e_session_lock;
rwlock_t staicnapt_list_lock;

uint8_t g_debug = 1;

#define LOCK_R_SESSION()	\
do {				\
	g_rlock_count++;	\
/*printk("[%s:%d] lock, %d\n", __func__, __LINE__, g_rlock_count);*/\
	read_lock(&map_e_session_lock);	\
} while (0)

#define ULOCK_R_SESSION()	\
do {				\
	g_rlock_count--;			\
/*printk("[%s:%d] unlock, %d\n", __func__, __LINE__, g_rlock_count);*/\
	read_unlock(&map_e_session_lock);	\
} while (0)

#define LOCK_W_SESSION()	\
do {				\
	g_wlock_count++;			\
/*printk("[%s:%d] lock, %d\n", __func__, __LINE__, g_wlock_count);*/\
	write_lock(&map_e_session_lock);	\
} while (0)

#define ULOCK_W_SESSION()	\
do {				\
	g_wlock_count--;			\
/*printk("[%s:%d] unlock, %d\n", __func__, __LINE__, g_wlock_count);*/\
	write_unlock(&map_e_session_lock);	\
} while (0)



#define LOCK_R_STATICNAPT()	\
do {				\
	read_lock(&staicnapt_list_lock);	\
} while (0)

#define ULOCK_R_STATICNAPT()	\
do {				\
	read_unlock(&staicnapt_list_lock);	\
} while (0)

#define LOCK_W_STATICNAPT()	\
do {				\
	write_lock(&staicnapt_list_lock);	\
} while (0)

#define ULOCK_W_STATICNAPT()	\
do {				\
	write_unlock(&staicnapt_list_lock);	\
} while (0)


static struct map_e_rule *map_e_domainv6_to_rule(struct in6_addr *, struct ip6_tnl_map_e *);
static int map_e_dstv4_to_dstv6(struct map_e_rule *,
    struct in_addr *, struct in6_addr *, uint16_t);
static uint8_t *map_e_bit_shiftleft(uint8_t *, int, int);
static uint8_t *map_e_bit_shiftright(uint8_t *, int, int);
static uint8_t *map_e_bit_or(uint8_t *, uint8_t *, int);
static uint16_t map_e_in4_checksum(uint16_t *, int);
static uint16_t map_e_pow16(uint16_t);

static const char *map_e_ntop(int, const void *, char *, size_t);
static int map_e_ce_state_napt46(struct sk_buff *, struct ip6_tnl_map_e *,
    struct in6_addr *, struct flowi6 *, __u8);
static int map_e_ce_state_napt46_err(struct sk_buff *, struct ip6_tnl_map_e *,
    struct in6_addr *, struct flowi6 *, __u8);
static int map_e_create_state_napt46(struct sk_buff *, struct ip6_tnl_map_e *,
    struct in6_addr *, struct in_addr *, uint16_t);
static int map_e_add_v6hdr(struct sk_buff *, struct in6_addr *, struct in6_addr *, struct ip6_tnl_map_e *, struct flowi6 *, __u8);

static struct net_device *map_e_ce_state_napt64(struct sk_buff *, struct ip6_tnl_map_e *);
static struct net_device *map_e_ce_state_napt64_err(struct sk_buff *, struct ip6_tnl_map_e *);
static struct net_device *map_e_ce_state_frag_napt64(struct sk_buff *, struct ip6_tnl_map_e *, struct ipv6hdr *);
static struct net_device *map_e_ce_state_frag_napt64_err(struct sk_buff *, struct ip6_tnl_map_e * , struct ipv6hdr *);

static int match_icmp_type(uint8_t, uint8_t, uint16_t, uint16_t);
static int map_e_dstv4_currentv4addr_check(struct in_addr *, struct ip6_tnl_map_e *);
static int map_e_port_range_check(uint16_t, struct ip6_tnl_map_e *);
static struct map_e_rule *map_e_dstv4_to_rule(struct in_addr *, struct ip6_tnl_map_e *);
static uint16_t map_e_ipid_get(struct ip6_tnl_map_e *);
static uint32_t map_e_get_hash_index(const struct in_addr *, uint16_t, uint8_t);

static struct map_e_session *map_e_search_session46(struct in_addr *, struct in_addr *,
    uint16_t, uint16_t, uint8_t, struct ip6_tnl_map_e *);
static struct map_e_session *map_e_search_session64(struct in6_addr *, struct in6_addr *,
    struct in_addr *, uint16_t, uint16_t, uint8_t, struct ip6_tnl_map_e *);

static int map_e_portset_init(uint16_t ps_id, uint16_t ps_len, struct ip6_tnl_map_e *t);
static uint16_t map_e_napt_port_get(struct in_addr *, struct in_addr *,
    uint16_t, uint16_t, uint8_t, struct ip6_tnl_map_e *);
static void map_e_session_init(struct ip6_tnl_map_e *);
static void map_e_session_destroy(struct ip6_tnl_map_e *t);
static void map_e_session_clear(struct ip6_tnl_map_e *t);
static void map_e_session_timer(unsigned long);
static void map_e_update_session_expire(struct sk_buff *, struct map_e_session *);

static int sam_ce_static_napt64(struct sk_buff *, struct ip6_tnl_map_e *);
static int sam_ce_static_frag_napt64(struct sk_buff *, struct ip6_tnl_map_e *,struct ipv6hdr *);
static struct map_e_staticnapt *sam_ce_static_napt_search64(struct sk_buff *, struct ip6_tnl_map_e *);
static struct map_e_staticnapt *sam_ce_static_napt_frag_search64(struct sk_buff *, struct ip6_tnl_map_e *);
static int sam_ce_static_napt_create64(struct sk_buff *, struct ip6_tnl_map_e *, struct ipv6hdr *, struct in_addr *, uint16_t);
static int map_e_ftp_alg_46(struct sk_buff *skb, struct iphdr *ip4h, struct tcphdr *th,
	struct ip6_tnl_map_e *t, struct map_e_session *ses);
static int map_e_atoi(const char *s);
static int map_e_ftp_alg_update_skb(struct sk_buff *skb, unsigned int iph_off,
	unsigned int ipadr_pos, unsigned int before_len, const char *after_buf,
	unsigned int after_len, struct map_e_session *ses);
static int map_e_enlarge_skb(struct sk_buff *skb, unsigned int diffsz);
static void map_e_ftp_alg_update_skb_data(struct sk_buff *skb, unsigned int dataoff,
	unsigned int ipadr_pos, unsigned int before_len, const char *after_buf,
	unsigned int after_len);
static int map_e_ftp_alg_create_session(struct map_e_session *ftp_data_ses, struct ip6_tnl_map_e *t);
static void _map_e_tcpmss(struct tcphdr *th, int reverse, struct sk_buff *skb);
static void map_e_tcpmss(struct tcphdr *th, struct sk_buff *skb);
static void map_e_tcpmss_reverse(struct tcphdr *th);
static void map_e_update_ses_tcp_flag(struct tcphdr *th, struct map_e_session *ses);
static struct timer_list map_e_session_callout;


#define MAP_E_ICMPDATA_OFFSET     (8)
#define MAP_E_PORT_BITLEN         (16)
#define MAP_E_PORT_MIN            (4096)
#define MAP_E_IPADDR_BITLEN       (32)
#define MAP_E_IPADDR6_BITLEN      (128)
#define MAP_E_PADDINGPOS_SUFFIX_HEAD      (8)
#define MAP_E_PREFIX_LIMIT        (32)
#define MAP_E_BIT_PER_BYTES       (8)

#define MAP_E_EXPIRE_INTERVAL     (1)
#define MAP_E_EXPIRE_DEFAULT      (3600)

#define MAP_E_FTP_ALG_PORT_OFFSET (5)		/* "PORT 192,168,..."IPɥ쥹ϰ */
#define MAP_E_FTP_ALG_EPRT_OFFSET (5)		/* "EPRT |1|192...."Ρ|׳ϰ */
#define MAP_E_FTP_PORT            (1)
#define MAP_E_FTP_EPRT            (2)


/*
 * å󥿥
 */
static int
set_session_timer(int expire, int protocol)
{
	switch(protocol){
	case IPPROTO_TCP:
		map_e_expire_tcp = expire;
		break;
	case IPPROTO_UDP:
		map_e_expire_udp = expire;
		break;
	default:
		break;
	}
	return 0;
}
EXPORT_SYMBOL(set_session_timer);

/* ICMP Error process 
 * @skb: original packet(after v6 encap)
 * @ip4h: original packet v4 header
 * @t: tunnel info
 */
static int
map_e_err_rcv(struct sk_buff *skb, struct iphdr *ip4h, struct ip6_tnl_map_e *t)
{
	struct map_e_session *ses = NULL;
	struct icmphdr *ih = NULL;
	struct udphdr *uh = NULL;
	struct tcphdr *th = NULL;
	struct ipv6hdr *ip6h = skb->data;
	uint16_t dport, sport;
	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		th = (struct tcphdr *)(ip4h + 1);
		dport = th->source;
		sport = th->dest;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		uh = (struct udphdr *)(ip4h + 1);
		dport = uh->source;
		sport = uh->dest;
		break;
	case IPPROTO_ICMP: //Only support the format include id field
		MAP_TRACE("IPPROTO_ICMP");
		ih = (struct icmphdr *)(ip4h + 1);
		if (ih->type != ICMP_ECHO &&
		    ih->type != ICMP_TIMESTAMP &&
		    ih->type != ICMP_ADDRESS &&
		    ih->type != ICMP_INFO_REQUEST) {
			return -1;
		}
		dport = 0;
		sport = ih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		return -1;
	}

	ses = map_e_search_session64(&ip6h->saddr, &ip6h->daddr,
					(struct in_addr *)&ip4h->daddr,
					dport, sport, ip4h->protocol, t);
	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("NOT session");
		ULOCK_R_SESSION();
		return -1;
	}
	if (memcmp(&ses->src4, &ses->c_src4, sizeof(struct in_addr)) ||
	     (ses->sport != ses->c_sport)) {

		ip4h->saddr = ses->src4.s_addr;
		switch (ip4h->protocol) {
		case IPPROTO_TCP:
			th->source = ses->sport;

			/* MSS */
			map_e_tcpmss_reverse(th);

			th->check = 0;
			len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)th, len, 0);
			sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
			    len, ip4h->protocol, sum1);
			th->check = sum2;
			MAP_TRACE("checksum = 0x%08x", th->check);
			break;
		case IPPROTO_UDP:
			uh->source = ses->sport;
			if (uh->check) {
				uh->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)uh, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);
				uh->check = sum2;
			}
			MAP_TRACE("checksum = 0x%08x", uh->check);
			break;
		case IPPROTO_ICMP:
			ih->un.echo.id = ses->sport;
			ih->checksum = 0;
			len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
			ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
			break;
		default:
			break;
		}
	}
	ULOCK_R_SESSION();
	return 0;
}
EXPORT_SYMBOL(map_e_err_rcv);
/*
 * ѥåȼMAP-Eͭ
 */
static int
map_e_frag_rcv(struct sk_buff *skb, struct ip6_tnl_map_e *t)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct ipv6hdr tmp_ip6h;
	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	uint16_t sport = 0;
	int ret;
	struct map_e_rule *rule;
	struct in6_addr ip6addr_bak;
	struct net_device *dev;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];
	int from_br = 0;
	int icmp_err = 0;
#ifdef CONFIG_RTL_FAST_MAP_E
	uint32 old_daddr;
	uint16 old_dport;
#endif

	MAP_TRACE("called");
	rcu_read_lock();

	if (t == NULL) {
		MAP_TRACE("t is null goto discard(0)");
		goto frag_discard;
	}

	if (t && t->map_e_info == NULL) {
		MAP_TRACE("t->map_e_info is null goto discard(1)");
		goto frag_discard;
	}

	if (ip6h->nexthdr != IPPROTO_IPIP) {
		MAP_TRACE("protocol is not ipip.  goto discard(1.1)");
		goto frag_discard;
	}

	// Reassemble Start
	// Reassembleλ³ΤˡIPv6إåϺäĤ褦
	memset(&tmp_ip6h , 0 , sizeof(tmp_ip6h));
	memcpy(&tmp_ip6h , ip6h , sizeof(struct ipv6hdr)); 

	/* IPv6إåκ */
	skb->mac_header = skb->network_header;
	skb_reset_network_header(skb);
	skb->protocol = htons(ETH_P_IP);
	skb->pkt_type = PACKET_HOST;

	// Reassemble Start
	int err = ip_defrag(skb, IP_DEFRAG_LOCAL_DELIVER);
	MAP_TRACE("ip_defrag = %d",err);
	if (err) {
		MAP_TRACE("Will be discard? or reassemble");
		rcu_read_unlock();
		return 1 ; // ʤ
        }
	MAP_TRACE("reassemble OK!");

	ip4h = (struct iphdr *)skb_network_header(skb);

#ifdef CONFIG_RTL_FAST_MAP_E
	old_daddr = ip4h->daddr;
#endif

	if (ip4h->protocol == IPPROTO_TCP) {
		th = (struct tcphdr *)(ip4h + 1);
		sport = ntohs(th->source);
#ifdef CONFIG_RTL_FAST_MAP_E
		old_dport = th->dest;
#endif
		
	} else if (ip4h->protocol == IPPROTO_UDP) {
		uh = (struct udphdr *)(ip4h + 1);
		sport = ntohs(uh->source);
#ifdef CONFIG_RTL_FAST_MAP_E
		old_dport = uh->dest;
#endif
	} else if (ip4h->protocol == IPPROTO_ICMP) {
		ih = (struct icmphdr *)(ip4h + 1);
		MAP_TRACE("ICMP_ECHOREPLY(%d)  ih->type=%d", ICMP_ECHOREPLY,ih->type);
		if (ih->type == ICMP_ECHOREPLY ||
		    ih->type == ICMP_TIMESTAMPREPLY ||
		    ih->type == ICMP_ADDRESSREPLY ||
		    ih->type == ICMP_INFO_REPLY) {
			/* just through */

		} else if (ih->type == ICMP_DEST_UNREACH ||
			   ih->type == ICMP_SOURCE_QUENCH ||
			   ih->type == ICMP_TIME_EXCEEDED ||
			   ih->type == ICMP_PARAMETERPROB){
			icmp_err = 1;
			MAP_TRACE("icmp_err!");
		} else {
			goto frag_discard;
		}
	} else {
		MAP_TRACE("goto discard(4), protocol=%d", ip4h->protocol);
		goto frag_discard;
	}

	/* BRv6ɥ쥹 */
	if (memcmp(&t->map_e_info->dmr_addr, &(tmp_ip6h.saddr), sizeof(struct in6_addr)) == 0) {
		from_br = 1;
		MAP_TRACE("from_br = 1");
		goto frag_remove_v6hdr;
	}

	/*
	 * CE̿ΡICMP Message̤ݡ
	 */
	if (ip4h->protocol == IPPROTO_ICMP) {
		MAP_TRACE("goto discard(5)");
		goto frag_discard;
	}

frag_remove_v6hdr:
	/*
	 * sessionθV6إåNAPT
	 */
	if (icmp_err == 0)
		dev = map_e_ce_state_frag_napt64(skb, t, &tmp_ip6h);
	else
		dev = map_e_ce_state_frag_napt64_err(skb, t , &tmp_ip6h);
	if (dev) {
		rcu_read_unlock();
		MAP_TRACE("end, t->dev->name:%s", t->dev ? t->dev->name : " ");
		return 0;
	}

	if (!from_br) {
		/* ¦v6ɥ쥹롼 */
		//rule = map_e_domainv6_to_rule(&ip6h->saddr, t); 
		rule = map_e_domainv6_to_rule( &(tmp_ip6h.saddr), t); 
		if (rule == NULL) {
			MAP_TRACE("goto discard(6)");
			goto frag_discard;
		}

		/*
		 * 롼ȡv4ɥ쥹ݡȤv6ɥ쥹롣
		 */
		ret = map_e_dstv4_to_dstv6(rule, (struct in_addr *)&ip4h->saddr, &ip6addr_bak, sport);
		if (ret == 0) {
			map_e_ntop(AF_INET6, &ip6addr_bak, buf1, sizeof(buf1));
			//map_e_ntop(AF_INET6, &ip6h->saddr, buf2, sizeof(buf2));
			map_e_ntop(AF_INET6, &(tmp_ip6h.saddr), buf2, sizeof(buf2));
			MAP_TRACE("(ip6addr_bak, ip6h->saddr) = (%s, %s)", buf1, buf2);
			//if (memcmp(&ip6addr_bak, &ip6h->saddr, sizeof(struct in6_addr))) {
			if (memcmp(&ip6addr_bak, &(tmp_ip6h.saddr), sizeof(struct in6_addr))) {
				/* Фɥ쥹ȳ¦v6ɥ쥹㤦 */
				/* Ʊ4rd Domain μʬʳCEž줿Ȥ */
				/* ΤΥå */
				MAP_TRACE("goto discard(7)");
				goto frag_discard;
			}
		}
	}

	/**
	 * ǡŪNAPTθ
	 * ͳʤξϡdiscard
	 **/
	ret = sam_ce_static_frag_napt64(skb, t, &tmp_ip6h);
	if (ret != 0) {
		MAP_TRACE("goto discard(8)");
		goto frag_discard;
	}

	rcu_read_unlock();
	MAP_TRACE("end, t->dev->name:%s", t->dev ? t->dev->name : " ");

#ifdef CONFIG_RTL_FAST_MAP_E	
	rtl_add_mape_fastpath_entry(t, skb, &tmp_ip6h, ip4h, old_daddr, old_dport, MAPE_IN_BOUND);
#endif 

	return 0;

frag_discard:
	rcu_read_unlock();
	MAP_TRACE("discard end");
	return -1;


}
EXPORT_SYMBOL(map_e_frag_rcv);
/*
 * ѥåȼMAP-Eͭ
 */
static int
map_e_rcv(struct sk_buff *skb, struct ip6_tnl_map_e *t)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	uint16_t sport = 0;
	int ret;
	struct map_e_rule *rule;
	struct in6_addr ip6addr_bak;
	struct net_device *dev;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];
	int from_br = 0;
	int icmp_err = 0;
#ifdef CONFIG_RTL_FAST_MAP_E
	struct ipv6hdr tmp_ip6h;
	uint32 old_daddr;
	uint16 old_dport;
#endif


	MAP_TRACE("called");
	rcu_read_lock();

	if (t == NULL) {
		MAP_TRACE("t is null");
		MAP_TRACE("goto discard(0)");
		goto discard;
	}

	if (t && t->map_e_info == NULL) {
		MAP_TRACE("t->map_e_info is null");
		MAP_TRACE("goto discard(1)");
		goto discard;
	}

	if (ip6h->nexthdr != IPPROTO_IPIP) {
		MAP_TRACE("protocol is not ipip.");
		MAP_TRACE("goto discard(1.1)");
		goto discard;
	}
	ip4h = (struct iphdr *)(ip6h + 1);

#ifdef CONFIG_RTL_FAST_MAP_E
	memset(&tmp_ip6h, 0x00, sizeof(struct ipv6hdr));
	memcpy(&tmp_ip6h, ip6h, sizeof(struct ipv6hdr));
	old_daddr = ip4h->daddr;
#endif

	if (ip4h->protocol == IPPROTO_TCP) {
		th = (struct tcphdr *)(ip4h + 1);
		sport = ntohs(th->source);
#ifdef CONFIG_RTL_FAST_MAP_E
		old_dport = th->dest;
#endif
		
	} else if (ip4h->protocol == IPPROTO_UDP) {
		uh = (struct udphdr *)(ip4h + 1);
		sport = ntohs(uh->source);
#ifdef CONFIG_RTL_FAST_MAP_E
		old_dport = uh->dest;
#endif
	} else if (ip4h->protocol == IPPROTO_ICMP) {
		ih = (struct icmphdr *)(ip4h + 1);
		if (ih->type == ICMP_ECHOREPLY ||
		    ih->type == ICMP_TIMESTAMPREPLY ||
		    ih->type == ICMP_ADDRESSREPLY ||
		    ih->type == ICMP_INFO_REPLY) {
			/* just through */

		} else if (ih->type == ICMP_DEST_UNREACH ||
			   ih->type == ICMP_SOURCE_QUENCH ||
			   ih->type == ICMP_TIME_EXCEEDED ||
			   ih->type == ICMP_PARAMETERPROB){
			icmp_err = 1;
		} else {
			goto discard;
		}
	} else {
		MAP_TRACE("goto discard(4), protocol=%d", ip4h->protocol);
		goto discard;
	}

	/* BRv6ɥ쥹 */
	if (memcmp(&t->map_e_info->dmr_addr, &ip6h->saddr, sizeof(struct in6_addr)) == 0) {
		from_br = 1;
		goto remove_v6hdr;
	}

	/*
	 * CE̿ΡICMP Message̤ݡ
	 */
	if (ip4h->protocol == IPPROTO_ICMP) {
		MAP_TRACE("goto discard(5)");
		goto discard;
	}

remove_v6hdr:
	/*
	 * sessionθV6إåNAPT
	 */
	if (icmp_err == 0)
		dev = map_e_ce_state_napt64(skb, t);
	else
		dev = map_e_ce_state_napt64_err(skb, t);
	if (dev) {
		rcu_read_unlock();
		MAP_TRACE("end, t->dev->name:%s", t->dev ? t->dev->name : " ");
		return 0;
	}

	if (!from_br) {
		/* ¦v6ɥ쥹롼 */
		rule = map_e_domainv6_to_rule(&ip6h->saddr, t);
		if (rule == NULL) {
			MAP_TRACE("goto discard(6)");
			goto discard;
		}

		/*
		 * 롼ȡv4ɥ쥹ݡȤv6ɥ쥹롣
		 */
		ret = map_e_dstv4_to_dstv6(rule, (struct in_addr *)&ip4h->saddr, &ip6addr_bak, sport);
		if (ret == 0) {
			map_e_ntop(AF_INET6, &ip6addr_bak, buf1, sizeof(buf1));
			map_e_ntop(AF_INET6, &ip6h->saddr, buf2, sizeof(buf2));
			MAP_TRACE("(ip6addr_bak, ip6h->saddr) = (%s, %s)", buf1, buf2);
			if (memcmp(&ip6addr_bak, &ip6h->saddr, sizeof(struct in6_addr))) {
				/* Фɥ쥹ȳ¦v6ɥ쥹㤦 */
				/* Ʊ4rd Domain μʬʳCEž줿Ȥ */
				/* ΤΥå */
				MAP_TRACE("goto discard(7)");
				goto discard;
			}
		}
	}

	/**
	 * ǡŪNAPTθ
	 * ͳʤξϡdiscard
	 **/
	ret = sam_ce_static_napt64(skb, t);
	if (ret != 0) {
		MAP_TRACE("goto discard(8)");
		goto discard;
	}

	rcu_read_unlock();
	MAP_TRACE("end, t->dev->name:%s", t->dev ? t->dev->name : " ");

#ifdef CONFIG_RTL_FAST_MAP_E
        rtl_add_mape_fastpath_entry(t, skb, &tmp_ip6h, ip4h, old_daddr, old_dport, MAPE_IN_BOUND);	
#endif 

	return 0;

discard:
	rcu_read_unlock();
	MAP_TRACE("discard end");
	return -1;
}
EXPORT_SYMBOL(map_e_rcv);

/*
 * ŪNAPTž
 */
static int
sam_ce_static_napt64(struct sk_buff *skb, struct ip6_tnl_map_e *t)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct map_e_staticnapt *staticnapt = NULL;

	struct in_addr old_ip4dst;
	uint16_t old_dport;
	struct ipv6hdr ip6_bak;

	int ret;
	unsigned int len = 0;
	__wsum  sum1 = 0;
	__sum16 sum2 = 0;

	struct skb_shared_info *skb_inf = skb_shinfo(skb);
	__sum16 sum_old = 0;
	uint32_t sum_buf = 0;

	MAP_TRACE("called");

	/* IPv6إåХååפ */
	memcpy(&ip6_bak, ip6h, sizeof(struct ipv6hdr));

	/* ŪNAPT긡 */
	staticnapt = sam_ce_static_napt_search64(skb, t);

	LOCK_R_STATICNAPT();
	if (staticnapt == NULL) {
		MAP_TRACE("There is no static napt.");
		ULOCK_R_STATICNAPT();
		return -1;
	}

	/*
	 * V6إå
	 */
	skb->mac_header = skb->network_header;
	skb_reset_network_header(skb);
	skb->protocol = htons(ETH_P_IP);
	skb->pkt_type = PACKET_HOST;

	ip4h = (struct iphdr *)skb_network_header(skb);
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		break;
	case IPPROTO_ICMP:
	default:
		MAP_TRACE("ignore transport protocol.");
		ULOCK_R_STATICNAPT();
		return -1;
	}

	/*
	 * IPv4ɥ쥹񤭡ݡȽ񤭴
	 */
	old_ip4dst.s_addr = ip4h->daddr;
	if (ip4h->protocol == IPPROTO_TCP) {
		old_dport = th->dest;
	} else if (ip4h->protocol == IPPROTO_UDP) {
		old_dport = uh->dest;
	}

	ip4h->daddr = staticnapt->lan_addr.s_addr;
	/* 1:1 modeΤȤsam_port=0Ȥʤ뤿ᡢ1:1modeξȽǤѤ */
	if (staticnapt->sam_port != 0 && staticnapt->lan_port != ntohs(old_dport)) {
		if (ip4h->protocol == IPPROTO_TCP) {
			th->dest = htons(staticnapt->lan_port);

			/* MSSĴ */
			map_e_tcpmss(th,skb);

			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				th->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)th, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);
			} else {
				/* fragment */
				sum_old = th->check;
				sum_buf = ((uint16_t)~sum_old + (uint16_t)~old_dport
					+ (uint16_t)~(old_ip4dst.s_addr>>16) + (uint16_t)~(0xffff&old_ip4dst.s_addr)
					+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + th->dest);
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				sum2 = (uint16_t)~sum_buf;
			}
			th->check = sum2;
			MAP_TRACE("checksum = 0x%08x", th->check);
		}
		else if (ip4h->protocol == IPPROTO_UDP) {
			uh->dest = htons(staticnapt->lan_port);

			if (uh->check) {
				if (skb_inf->frag_list == NULL) {
					/* not fragmet */
					uh->check = 0;
					len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
					sum1 = csum_partial((char *)uh, len, 0);
					sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
					    len, ip4h->protocol, sum1);
				} else {
					/* fragment */
					sum_old = uh->check;
					sum_buf = ((uint16_t)~sum_old + (uint16_t)~old_dport
						+ (uint16_t)~(old_ip4dst.s_addr>>16) + (uint16_t)~(0xffff&old_ip4dst.s_addr)
						+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + uh->dest);
					sum_buf = (sum_buf>>16) + sum_buf&0xffff;
					sum2 = (uint16_t)~sum_buf;
				}
				uh->check = sum2;
			}
			MAP_TRACE("checksum = 0x%08x", uh->check);
		}
	}
	else {
		/* 1:1 mode  (ݡѴϤʤ) */
		/* 1:1ǥݡֹ椬Ʊ⤳MSSĴ */
		if (ip4h->protocol == IPPROTO_TCP) {
			/* MSSĴ */
			map_e_tcpmss(th,skb);

			th->check = 0;
			len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)th, len, 0);
			sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
			    len, ip4h->protocol, sum1);
			th->check = sum2;
		}
	}
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	/* åϿ */
	ret = sam_ce_static_napt_create64(skb, t, &ip6_bak, &old_ip4dst, old_dport);
	if (ret != 0) {
		/* ϿˤϼԤžϼ¹Ԥ */
		MAP_TRACE("sam_ce_static_napt_create64() error");
		ULOCK_R_STATICNAPT();
		return 0;
	}
	ULOCK_R_STATICNAPT();
	return 0;
}
/*
 * ŪNAPTž
 */
static int
sam_ce_static_frag_napt64(struct sk_buff *skb, struct ip6_tnl_map_e *t,struct ipv6hdr *ip6h)
{
	//struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct map_e_staticnapt *staticnapt = NULL;

	struct in_addr old_ip4dst;
	uint16_t old_dport;
	struct ipv6hdr ip6_bak;

	int ret;
	unsigned int len = 0;
	__wsum  sum1 = 0;
	__sum16 sum2 = 0;

	struct skb_shared_info *skb_inf = skb_shinfo(skb);
	__sum16 sum_old = 0;
	uint32_t sum_buf = 0;

	MAP_TRACE("called");

	/* IPv6إåХååפ */
	memcpy(&ip6_bak, ip6h, sizeof(struct ipv6hdr));

	/* ŪNAPT긡 */
	staticnapt = sam_ce_static_napt_frag_search64(skb, t);

	LOCK_R_STATICNAPT();
	if (staticnapt == NULL) {
		MAP_TRACE("There is no static napt.");
		ULOCK_R_STATICNAPT();
		return -1;
	}

	ip4h = (struct iphdr *)skb_network_header(skb);
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		break;
	case IPPROTO_ICMP:
	default:
		MAP_TRACE("ignore transport protocol.");
		ULOCK_R_STATICNAPT();
		return -1;
	}

	/*
	 * IPv4ɥ쥹񤭡ݡȽ񤭴
	 */
	old_ip4dst.s_addr = ip4h->daddr;
	if (ip4h->protocol == IPPROTO_TCP) {
		old_dport = th->dest;
	} else if (ip4h->protocol == IPPROTO_UDP) {
		old_dport = uh->dest;
	}

	ip4h->daddr = staticnapt->lan_addr.s_addr;
	/* 1:1 modeΤȤsam_port=0Ȥʤ뤿ᡢ1:1modeξȽǤѤ */
	if (staticnapt->sam_port != 0 && staticnapt->lan_port != ntohs(old_dport)) {
		if (ip4h->protocol == IPPROTO_TCP) {
			th->dest = htons(staticnapt->lan_port);

			/* MSSĴ */
			map_e_tcpmss(th,skb);

			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				th->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)th, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);
			} else {
				/* fragment */
				sum_old = th->check;
				sum_buf = ((uint16_t)~sum_old + (uint16_t)~old_dport
					+ (uint16_t)~(old_ip4dst.s_addr>>16) + (uint16_t)~(0xffff&old_ip4dst.s_addr)
					+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + th->dest);
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				sum2 = (uint16_t)~sum_buf;
			}
			th->check = sum2;
			MAP_TRACE("checksum = 0x%08x", th->check);
		}
		else if (ip4h->protocol == IPPROTO_UDP) {
			uh->dest = htons(staticnapt->lan_port);

			if (uh->check) {
				if (skb_inf->frag_list == NULL) {
					/* not fragmet */
					uh->check = 0;
					len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
					sum1 = csum_partial((char *)uh, len, 0);
					sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
					    len, ip4h->protocol, sum1);
				} else {
					/* fragment */
					sum_old = uh->check;
					sum_buf = ((uint16_t)~sum_old + (uint16_t)~old_dport
						+ (uint16_t)~(old_ip4dst.s_addr>>16) + (uint16_t)~(0xffff&old_ip4dst.s_addr)
						+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + uh->dest);
					sum_buf = (sum_buf>>16) + sum_buf&0xffff;
					sum2 = (uint16_t)~sum_buf;
				}
				uh->check = sum2;
			}
			MAP_TRACE("checksum = 0x%08x", uh->check);
		}
	}
	else {
		/* 1:1 mode  (ݡѴϤʤ) */
		/* 1:1ǥݡֹ椬Ʊ⤳MSSĴ */
		if (ip4h->protocol == IPPROTO_TCP) {
			/* MSSĴ */
			map_e_tcpmss(th,skb);

			th->check = 0;
			len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)th, len, 0);
			sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
			    len, ip4h->protocol, sum1);
			th->check = sum2;
		}
	}
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	/* åϿ */
	ret = sam_ce_static_napt_create64(skb, t, &ip6_bak, &old_ip4dst, old_dport);
	if (ret != 0) {
		/* ϿˤϼԤžϼ¹Ԥ */
		MAP_TRACE("sam_ce_static_napt_create64() error");
		ULOCK_R_STATICNAPT();
		return 0;
	}
	ULOCK_R_STATICNAPT();
	return 0;
}

/*
 * ŪNAPT긡
 */
static struct map_e_staticnapt *
sam_ce_static_napt_search64(
    struct sk_buff *skb,
    struct ip6_tnl_map_e *t)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct map_e_staticnapt *staticnapt;
	struct list_head *p;
	uint16_t dport;

	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;

	ip4h = (struct iphdr *)(ip6h + 1);
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		dport = ntohs(th->dest);
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		dport = ntohs(uh->dest);
		break;
	default:
		printk(KERN_ERR "[%s] ignore protocol.\n", __func__);
		return NULL;
	}

	LOCK_R_STATICNAPT();
	list_for_each(p, &t->map_e_info->staticnapt_list.list) {
		staticnapt = list_entry(p, struct map_e_staticnapt, list);

		if (t->map_e_info->shared_addr4.s_addr == ip4h->daddr &&
		    staticnapt->protocol == ip4h->protocol &&
		    staticnapt->sam_port == dport) {
			ULOCK_R_STATICNAPT();
			return staticnapt;
		}
		/* 1:1 mode */
		/* 1:1 modeΤȤsam_port=0Ȥʤ뤿ᡢ1:1modeξȽǤѤ */
		if (t->map_e_info->shared_addr4.s_addr == ip4h->daddr &&
		    staticnapt->protocol == ip4h->protocol &&
		    staticnapt->sam_port == 0 &&
			(staticnapt->lan_port <= dport && staticnapt->lan_toport >= dport)) {
			ULOCK_R_STATICNAPT();
			return staticnapt;
		}
	}
	ULOCK_R_STATICNAPT();
	return NULL;
}

/*
 * ŪNAPT긡
 */
static struct map_e_staticnapt *
sam_ce_static_napt_frag_search64(
    struct sk_buff *skb,
    struct ip6_tnl_map_e *t)
{
//	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct map_e_staticnapt *staticnapt;
	struct list_head *p;
	uint16_t dport;

	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;

	ip4h = (struct iphdr *)skb_network_header(skb);
	//ip4h = (struct iphdr *)(ip6h + 1);
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		dport = ntohs(th->dest);
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		dport = ntohs(uh->dest);
		break;
	default:
		printk(KERN_ERR "[%s] ignore protocol.\n", __func__);
		return NULL;
	}

	LOCK_R_STATICNAPT();
	list_for_each(p, &t->map_e_info->staticnapt_list.list) {
		staticnapt = list_entry(p, struct map_e_staticnapt, list);

		if (t->map_e_info->shared_addr4.s_addr == ip4h->daddr &&
		    staticnapt->protocol == ip4h->protocol &&
		    staticnapt->sam_port == dport) {
			ULOCK_R_STATICNAPT();
			return staticnapt;
		}
		/* 1:1 mode */
		/* 1:1 modeΤȤsam_port=0Ȥʤ뤿ᡢ1:1modeξȽǤѤ */
		if (t->map_e_info->shared_addr4.s_addr == ip4h->daddr &&
		    staticnapt->protocol == ip4h->protocol &&
		    staticnapt->sam_port == 0 &&
			(staticnapt->lan_port <= dport && staticnapt->lan_toport >= dport)) {
			ULOCK_R_STATICNAPT();
			return staticnapt;
		}
	}
	ULOCK_R_STATICNAPT();
	return NULL;
}
/*
 * ŪNAPTΥå
 */
static int
sam_ce_static_napt_create64(
    struct sk_buff *skb,
    struct ip6_tnl_map_e *t,
    struct ipv6hdr *ip6_bak,
    struct in_addr *old_ip4dst,
    uint16_t old_dport)
{
	struct iphdr *ip4h = NULL;
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct map_e_session *new_ses = NULL;
	uint32_t hash_index;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1], buf3[INET6_ADDRSTRLEN + 1], buf4[INET6_ADDRSTRLEN + 1], buf5[INET6_ADDRSTRLEN + 1];

	LOCK_R_SESSION();
	if (t->map_e_info->ses_num >= t->parms.session_max) {
		printk(KERN_WARNING "The number of sessions reached the maximum.\n");
		ULOCK_R_SESSION();
		goto create_failed;
	}
	ULOCK_R_SESSION();

	ip4h = (struct iphdr *)skb_network_header(skb);

	new_ses = KMALLOC(sizeof(struct map_e_session), GFP_ATOMIC);
	if (new_ses == NULL) {
		printk(KERN_ERR "[%s] KMALLOC() error.\n", __func__);
		goto create_failed;
	}
	memset(new_ses, 0x00, sizeof(struct map_e_session));

	memcpy(&new_ses->dst6, &ip6_bak->saddr, sizeof(struct in6_addr));
	memcpy(&new_ses->src6, &ip6_bak->daddr, sizeof(struct in6_addr));

	new_ses->c_src4.s_addr = old_ip4dst->s_addr;
	new_ses->src4.s_addr   = ip4h->daddr;
	new_ses->dst4.s_addr   = ip4h->saddr;
	new_ses->protocol      = ip4h->protocol;

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		new_ses->sport   = th->dest;
		new_ses->c_sport = old_dport;
		new_ses->dport   = th->source;
		new_ses->expire  = map_e_expire_tcp;
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		new_ses->sport   = uh->dest;
		new_ses->c_sport = old_dport;
		new_ses->dport   = uh->source;
		new_ses->expire  = map_e_expire_udp;
		break;
	default:
		printk(KERN_ERR "[%s] ignore protocol.\n", __func__);
		goto create_failed;
	}

	if (g_debug) {
		map_e_ntop(AF_INET6, &new_ses->src6, buf1, sizeof(buf1));
		map_e_ntop(AF_INET6, &new_ses->dst6, buf2, sizeof(buf2));
		map_e_ntop(AF_INET, &new_ses->src4, buf3, sizeof(buf3));
		map_e_ntop(AF_INET, &new_ses->c_src4, buf4, sizeof(buf4));
		map_e_ntop(AF_INET, &new_ses->dst4, buf5, sizeof(buf5));
		MAP_TRACE("<<new session>>");
		MAP_TRACE("  .src6    :%s", buf1);
		MAP_TRACE("  .dst6    :%s", buf2);
		MAP_TRACE("  .src4    :%s", buf3);
		MAP_TRACE("  .c_src4  :%s", buf4);
		MAP_TRACE("  .dst4    :%s", buf5);
		MAP_TRACE("  .protocol:0x%04x", new_ses->protocol);
		MAP_TRACE("  .expire  :%d", new_ses->expire);
		MAP_TRACE("  .sport   :%u", ntohs(new_ses->sport));
		MAP_TRACE("  .c_sport :%u", ntohs(new_ses->c_sport));
		MAP_TRACE("  .dport   :%u", ntohs(new_ses->dport));
	}

	/* åϿ */
	hash_index = map_e_get_hash_index(&new_ses->dst4, new_ses->dport, new_ses->protocol);
	MAP_TRACE("hash_index = %u", hash_index);
	LOCK_W_SESSION();
	list_add_tail((struct list_head *)new_ses, &t->map_e_info->ses_list[hash_index].list);
	t->map_e_info->ses_num++;

	ULOCK_W_SESSION();
	return 0;

create_failed:
	if (new_ses) {
		KFREE(new_ses);
	}
	return -1;
}

/*
 * ѥåMAP-Eͭ
 */
static int
map_e_xmit(struct sk_buff *skb, struct flowi6 *fl, u8 proto, struct ip6_tnl_map_e *t, __u8 dsfield)
{
	struct in6_addr src_addr, dst_addr;
	struct in_addr old_saddr;
	struct iphdr *ip4h = (struct iphdr *)skb_network_header(skb);
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	uint16_t dport = 0, sport_old = 0, sport_new = 0;
	struct map_e_rule *rule;
	int prefix_length = 0;
	int err = -EFAULT;
	int ret;
	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;
	int icmp_err = 0;

	MAP_TRACE("start");

	if (t == NULL) {
		MAP_TRACE("t is null");
		goto tx_err;
	}

	if (t->dev) {
		MAP_TRACE("t->dev->name : %s", t->dev->name);
	}

	if (t->map_e_info == NULL) {
		MAP_TRACE("t->map_e_info is null, t=%p",t);
		goto tx_err;
	}

	/*
	 * ǥեȤBR뤿᰸V6ɥ쥹BRˤ
	 */
	memcpy(&src_addr, &t->map_e_info->wan_addr, sizeof(struct in6_addr));
	memcpy(&dst_addr, &t->map_e_info->dmr_addr, sizeof(struct in6_addr));

	MAP_TRACE("dst_addr : %04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x",
	    ntohs(dst_addr.s6_addr16[0]),
	    ntohs(dst_addr.s6_addr16[1]),
	    ntohs(dst_addr.s6_addr16[2]),
	    ntohs(dst_addr.s6_addr16[3]),
	    ntohs(dst_addr.s6_addr16[4]),
	    ntohs(dst_addr.s6_addr16[5]),
	    ntohs(dst_addr.s6_addr16[6]),
	    ntohs(dst_addr.s6_addr16[7]));

	if (t->map_e_info->bmr) {
		MAP_TRACE("bmr : %08x", (int)t->map_e_info->bmr);
		MAP_TRACE("  v4_prefix_len:%d, eabits_len:%d",
		    t->map_e_info->bmr->v4_prefix_len, t->map_e_info->bmr->eabits_len);
	} else {
		MAP_TRACE("bmr : NULL");
	}

	if ((t->map_e_info->bmr ?
	    (t->map_e_info->bmr->v4_prefix_len + t->map_e_info->bmr->eabits_len) : 0) < MAP_E_PREFIX_LIMIT) {
		MAP_TRACE("4rd prefix len < 32");
		goto tx_err;
	}

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		dport = th->dest;
		sport_old = th->source;
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		dport = uh->dest;
		sport_old = uh->source;
		break;
	case IPPROTO_ICMP:
		ih = (struct icmphdr *)(ip4h + 1);
		if (ih->type == ICMP_ECHO ||
		    ih->type == ICMP_TIMESTAMP ||
		    ih->type == ICMP_ADDRESS ||
		    ih->type == ICMP_INFO_REQUEST) {
			/* just through */

		} else if (ih->type == ICMP_DEST_UNREACH ||
			   ih->type == ICMP_SOURCE_QUENCH ||
			   ih->type == ICMP_TIME_EXCEEDED ||
			   ih->type == ICMP_PARAMETERPROB){
			icmp_err = 1;
		} else {
			goto tx_err;
		}
		dport = 0;
		sport_old = ih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		goto tx_err;
	}

#ifdef CONFIG_RTL_FAST_MAP_E
        old_saddr.s_addr = ip4h->saddr;
#endif

	if (icmp_err == 0) {
		/* å󤬴¸ߤƤ顢ξѤž */
		err = map_e_ce_state_napt46(skb, t, &dst_addr, fl, dsfield);
		if (err >= 0) {
			goto do_tx;
		}
	} else {
		/* ICMP errorϥå̵žԲ */
		err = map_e_ce_state_napt46_err(skb, t, &dst_addr, fl, dsfield);
		if (err >= 0) {
			goto do_tx;
		} else {
			goto tx_err;
		}                            	
	}
	/* ICMPξϡɥ쥹åʤ */
	if (ip4h->protocol == IPPROTO_ICMP) {
		goto get_sport;
	}

	if (map_e_dstv4_currentv4addr_check((struct in_addr *)&ip4h->daddr, t)) {
		if(!map_e_port_range_check(ntohs(dport), t)) {
			/*
			 * ʬΥݡϰϤʤΤ¾CE¾CEV6ɥ쥹
			 * 뤫ɤȥ饤¾Υ롼뻲
			 */
			err = map_e_dstv4_to_dstv6(t->map_e_info->bmr,
			    (struct in_addr *)&ip4h->daddr, &dst_addr, ntohs(dport));
			if (err < 0) {
				MAP_TRACE("map_e_dstv4_to_dstv6() error");
				goto tx_err;
			}
		} else {
			MAP_TRACE("port is out of range");
			goto tx_err;
		}
	} else if ((rule = map_e_dstv4_to_rule((struct in_addr *)&ip4h->daddr, t)) != NULL) {
		prefix_length = rule->v4_prefix_len + rule->eabits_len;
		if (prefix_length < MAP_E_PREFIX_LIMIT) {
			MAP_TRACE("prefix_length < 32");
			goto tx_err;
		} else {
			err = map_e_dstv4_to_dstv6(rule, (struct in_addr *)&ip4h->daddr,
			    &dst_addr, ntohs(dport));
			if (err < 0) {
				MAP_TRACE("map_e_dstv4_to_dstv6() error");
				goto tx_err;
			}
		}
	}

get_sport:
	/* MAP-EΥݡȤ */
	sport_new = map_e_napt_port_get((struct in_addr *)&ip4h->daddr, (struct in_addr *)&ip4h->saddr,
	    dport, sport_old, ip4h->protocol, t);
	if (sport_new == 0) {
		MAP_TRACE("[%s] map_e_napt_port_get() error.\n", __func__);
		goto tx_err;
	}

	/* ip_idݡȥåȤϰϤ˽񤭴 */
	ip4h->id = htons(map_e_ipid_get(t));
	old_saddr.s_addr = ip4h->saddr;
	ip4h->saddr = t->map_e_info->shared_addr4.s_addr;
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	/* ݡֹݡȥåȤϰϤ˽񤭴 */
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th->source = htons(sport_new);

		/* MSSĴ */
		map_e_tcpmss(th,skb);

		th->check = 0;
		len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
		sum1 = csum_partial((char *)th, len, 0);
		sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
		    len, ip4h->protocol, sum1);
		th->check = sum2;
		MAP_TRACE("checksum = 0x%08x", th->check);
		break;
	case IPPROTO_UDP:
		uh->source = htons(sport_new);
		if (uh->check) {
			uh->check = 0;
			len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)uh, len, 0);
			sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
			    len, ip4h->protocol, sum1);
			uh->check = sum2;
		}
		MAP_TRACE("checksum = 0x%08x", uh->check);
		break;
	case IPPROTO_ICMP:
		ih->un.echo.id = htons(sport_new);
		ih->checksum = 0;
		len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
		ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
		break;
	default:
		break;
	}

	/* IPv6إåɲ */
	ret = map_e_add_v6hdr(skb, &t->map_e_info->wan_addr, &dst_addr, t, fl, dsfield);
	if (ret < 0) {
		goto tx_err;
	}

	/* ݡȡݡȤǥåơž */
	err = map_e_create_state_napt46(skb, t, &dst_addr, &old_saddr, sport_old);
	if (err < 0) {
		MAP_TRACE("map_e_ce_state_napt46() error");
		goto tx_err;
	}
	if (g_debug) {
		debug_skb_printk(__func__, skb);
	}

do_tx:
#ifdef CONFIG_RTL_FAST_MAP_E
        rtl_add_mape_fastpath_entry(t, skb, ipv6_hdr(skb), ip4h, old_saddr.s_addr, sport_old, MAPE_OUT_BOUND);
#endif 

	nf_reset(skb);
	skb->local_df = 1; /* IPv6TOOBIG֤ʤ褦ˤơե饰Ƚ¹Ԥ */

	MAP_TRACE("success");
	return 0;

tx_err:
	MAP_TRACE("failed");
	return err;
}
EXPORT_SYMBOL(map_e_xmit);

#ifdef CONFIG_RTL_FAST_MAP_E
static int rtl_check_protocol_valid( struct sk_buff *skb, struct iphdr *ip4h, int direction)
{
        struct nf_conn *ct = NULL;
        u16 src_port, dst_port, check_port;

        if (!skb ||!ip4h) return 0;
        if ((ip4h->protocol != IPPROTO_TCP) && (ip4h->protocol != IPPROTO_UDP)) return 0;

        if ((ip4h->protocol == IPPROTO_TCP)){
                ct = (struct nf_conn*)skb->nfct;
                if( ct && (ct->proto.tcp.state != TCP_CONNTRACK_ESTABLISHED) )
                        return 0;

                /* FTP control packet */
                src_port = ((struct tcphdr *)(ip4h + 1))->source;
                dst_port = ((struct tcphdr *)(ip4h + 1))->dest;
                check_port = ((direction == MAPE_OUT_BOUND)?dst_port:src_port);
                if(ntohs(check_port) == FTP_CTL_PORT)
                        return 0;
        }
        return 1;
}

/* RTL: Set portset info and DPI hook func for fast-path */
static int rtl_init_map_e_fastpath_info(struct ip6_tnl_map_e *t)
{
        uint16_t index_fp_min ;

        if(!t) return 0;
        if(init_map_e_portset)   return 1;

        init_map_e_portset = 1;
        memset(&map_e_fastpath_info, 0x00, sizeof(struct map_e_fp_info));
        memcpy(&map_e_fastpath_info.portset_num, &(t->map_e_info->portset_num), sizeof(uint16_t)); /*8*/
        memcpy(&map_e_fastpath_info.portset_id, &(t->map_e_info->portset_id), sizeof(uint16_t));        /*0x34*/
        index_fp_min = (t->map_e_info->portset_num >> 1);
        memcpy(&map_e_fastpath_info.current_portset_index, &index_fp_min, sizeof(uint16_t));/*fastpath min: 0x8340*/
        memcpy(&map_e_fastpath_info.portset_list, &(t->map_e_info->portset_list), sizeof(map_e_fp_portset_list));

	rtl_search_mape_session_hook = rtl_search_map_e_session;   // 2018.8.15
	rtl_delete_mape_session_hook = rtl_delete_map_e_session;
 
        return 1;
}

void rtl_add_mape_fastpath_entry(struct ip6_tnl_map_e *t, struct sk_buff *skb, struct ipv6hdr *ip6h, struct iphdr *ip4h,
                                uint32 old_addr, uint16_t old_port, int direction)
{
        rtl_fp_napt_entry *rtlFpNaptEntryP = NULL, rtlFpNaptEntry = {0};
        struct in6_addr src6, dst6;

        MAP_TRACE("rtl_add_mape_fastpath_entry called.");

        if (!fast_map_e_fw) return;
        if (!t || !skb || !ip6h || !ip4h) return;
        if ((direction != MAPE_OUT_BOUND) && (direction != MAPE_IN_BOUND)) return;
        if (!rtl_check_protocol_valid(skb, ip4h, direction)) return;
        if (!rtl_init_map_e_fastpath_info(t)) return;

        memcpy(&src6, &ip6h->saddr, sizeof(struct in6_addr));
        memcpy(&dst6, &ip6h->daddr, sizeof(struct in6_addr));

        rtlFpNaptEntryP = rtl_getTunnelNaptEntry(&rtlFpNaptEntry, ip4h, old_addr, old_port, src6, dst6, 0, direction);
        if(rtlFpNaptEntryP)
                rtl_addTunnelNaptConn(skb, rtlFpNaptEntryP);
	return;
}

static int rtl_search_map_e_session
(struct iphdr *ip4h, struct tcphdr *tcphupuh, u32 *ext_ip, u16 *ext_port, 
	struct in6_addr *src6, struct in6_addr *dst6, void **session, int direction)
{
        struct tcphdr *th;
        struct udphdr *uh;
	u16 s_port, d_port;
	struct map_e_session *ses = NULL;
	
	if (!ip4h || !tcphupuh || !ext_ip || !ext_port || !src6 || !dst6 || !map_e_fastpath_t) return 0;
        if (direction != MAPE_DPI_OUT_BOUND && direction != MAPE_DPI_IN_BOUND) return 0;	
		
	s_port = tcphupuh->source;
	d_port = tcphupuh->dest;

        if(direction == MAPE_DPI_OUT_BOUND)
                ses = map_e_search_session46((struct in_addr *)&ip4h->daddr, (struct in_addr *)&ip4h->saddr,
	    		d_port, s_port, ip4h->protocol, map_e_fastpath_t);
        else
                ses = map_e_search_session46((struct in_addr *)&ip4h->saddr, (struct in_addr *)&ip4h->daddr,
				s_port, d_port, ip4h->protocol, map_e_fastpath_t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("not found session: s_port=%d, d_port=%d",s_port, d_port);
		ULOCK_R_SESSION(); // 2018.8.15
		return 0;
	}

	*ext_ip = ses->c_src4.s_addr;
	*ext_port = ses->c_sport;
	
        if(direction == MAPE_DPI_OUT_BOUND){
                memcpy(src6, &ses->src6, sizeof(struct in6_addr));
                memcpy(dst6, &ses->dst6, sizeof(struct in6_addr));
        }else{
                memcpy(src6, &ses->dst6, sizeof(struct in6_addr));
                memcpy(dst6, &ses->src6, sizeof(struct in6_addr));
        }
 
	if(session)
		*session = (void *)ses;
#if 0
	printk("[%s:%d](%s) ses=%p,s_port=%d,nat_port=%d, d_port=%d\n",__FUNCTION__,__LINE__,
		direction==MAPE_DPI_OUT_BOUND?"Lan->Wan":"Wan->Lan",
		ses, s_port, *ext_port, d_port);
#endif

	ULOCK_R_SESSION();
	return 1;
}

int rtl_construct_mape_fp_session(struct map_e_session *ses, rtl_fp_napt_entry* fpNaptEntryP){
	if (!fast_map_e_fw) 
		return 0;	
	if(!ses|| !fpNaptEntryP)
		return 0;

	fpNaptEntryP->p_flag = MAPE_PCK_FLAG;
	fpNaptEntryP->protocol = ((ses->protocol == IPPROTO_TCP) ? NP_TCP : NP_UDP);
	fpNaptEntryP->intIp = ses->src4.s_addr;
	fpNaptEntryP->intPort = ses->sport;
	fpNaptEntryP->extIp = ses->c_src4.s_addr;
	fpNaptEntryP->extPort = ses->c_sport;
	fpNaptEntryP->remIp = ses->dst4.s_addr;
	fpNaptEntryP->remPort = ses->dport;
	
	return 1;
}

void rtl_delete_map_e_session(void *session){
	struct map_e_session *ses;
	
	if(!session) return;
	
	LOCK_W_SESSION();
	ses = (struct map_e_session*)session;
	list_del((struct list_head *)ses);
	KFREE(ses);
	map_e_fastpath_t->map_e_info->ses_num--;
//	printk("[%s:%d] map_e_num=%d\n",__FUNCTION__,__LINE__, map_e_fastpath_t->map_e_info->ses_num);
	ULOCK_W_SESSION();
}

void rtl_flush_map_e_fastpath_connection(void)
{
	init_map_e_portset = 0;
	rtl_search_mape_session_hook = NULL;
	rtl_delete_mape_session_hook = NULL;

	rtl_flushMAPEConnection();
}
#endif /* CONFIG_RTL_FAST_MAP_E */

/*
 * IOCTLMAP-Eͭ
 */
static int
map_e_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
	MAP_TRACE("start, cmd=%d", cmd);
	g_debug = cmd;
	g_debug = 1;
	return 0;
}
EXPORT_SYMBOL(map_e_ioctl);

/**
 * map_e_init - register protocol and reserve needed resources
 *
 * Return: 0 on success
 **/
static int __init
map_e_init(void)
{
	rwlock_init(&map_e_session_lock);
	return 0;
}

/**
 * map_e_cleanup - free resources and unregister protocol
 **/
static void __exit
map_e_cleanup(void)
{
#ifdef CONFIG_RTL_FAST_MAP_E
	rtl_flush_map_e_fastpath_connection();
#else
	;
#endif
}

module_init(map_e_init);
module_exit(map_e_cleanup);


/* ****************************************** *
 *
 * MAP-EϢδؿ
 *
 * ****************************************** */
/*
 * Domain IPv6 prefix 롼
 * longest match
 */
static struct map_e_rule *
map_e_domainv6_to_rule(struct in6_addr *ipv6, struct ip6_tnl_map_e *t)
{
	struct list_head *p;
	struct map_e_rule *krule = NULL;
	struct map_e_rule *krule_longest = NULL;
	int bits_longest = 0;

	if (ipv6 == NULL) {
		MAP_TRACE("ipv6 is null, return NULL");
		return NULL;
	}

	list_for_each(p, &t->map_e_info->fmr_list.list) {
		krule = list_entry(p, struct map_e_rule, list);

		if (ipv6_prefix_equal((const struct in6_addr *)&krule->v6_prefix.s6_addr[0],
		    (const struct in6_addr *)&ipv6->s6_addr[0], krule->v6_prefix_len)) {
			if (bits_longest < krule->v6_prefix_len) {
				krule_longest = krule;
				bits_longest = krule->v6_prefix_len;
			} else if (bits_longest == krule->v6_prefix_len) {
				MAP_TRACE("dupulicate IPv6 prefix.");
			}
		}
	}

	if (krule_longest == NULL) {
		MAP_TRACE("BMR search Error.");
		return NULL;
	}

	MAP_TRACE("return krule_longest(0x%08x)", (int)krule_longest);
	return krule_longest;
}

/*
 * v4ɥ쥹ʤȥ롼ˤ鰸v6ɥ쥹κߤؿ
 *
 * ͡
 * 0 
 * <0 
 */
static int
map_e_dstv4_to_dstv6(struct map_e_rule *rule,
    struct in_addr *dstv4, struct in6_addr *ipv6, uint16_t dport)
{
	uint64_t eabits;
	uint32_t host_id;
	uint32_t shared_ipv4;
	uint32_t portset_id32;
	uint16_t portset_length;
	uint16_t dport_temp;
	uint32_t mask_reverse;
	uint8_t res[16], res1[16];
	int idx;
	uint8_t addr_string[INET6_ADDRSTRLEN + 1];

	MAP_TRACE("start");

	if (rule == NULL) {
		MAP_TRACE("rule is null, return -1");
		goto failed;
	}
	if (dstv4 == NULL) {
		MAP_TRACE("dstv4 is null, return -1");
		goto failed;
	}
	if (ipv6 == NULL) {
		MAP_TRACE("ipv6 is null, return -1");
		goto failed;
	}
	if ((rule->v4_prefix_len + rule->eabits_len) < MAP_E_PREFIX_LIMIT) {
		MAP_TRACE("cannot create shared ipv4 address(<32)");
		goto failed;
	}

	/* IPv4ɥ쥹ΥۥФ */
	mask_reverse = 0xFFFFFFFF >> rule->v4_prefix_len;
	host_id = ntohl(dstv4->s_addr) & mask_reverse;
	MAP_TRACE("dstv4        : 0x%x", ntohl(dstv4->s_addr));
	MAP_TRACE("mask_reverse : 0x%x", mask_reverse);
	MAP_TRACE("host_id      : 0x%x", host_id);

	portset_length = rule->eabits_len - (MAP_E_IPADDR_BITLEN - rule->v4_prefix_len);

	/* EA-bits ͤ */
	eabits = ((uint64_t)host_id) << (rule->eabits_len - (MAP_E_IPADDR_BITLEN - rule->v4_prefix_len));
	if ((rule->v4_prefix_len + rule->eabits_len) != MAP_E_PREFIX_LIMIT) {
		dport_temp = dport << rule->psid_offset;
		portset_id32 = dport_temp >> (MAP_E_PORT_BITLEN - portset_length);
		eabits = eabits | portset_id32;
	} else {
		portset_id32 = 0;
	}
	MAP_TRACE("portset_id32 : 0x%x", portset_id32);
	MAP_TRACE("EA-bits : 0x%llx", eabits);

	memset(&res1[0], 0x00, sizeof(res1));
#ifdef MAP_BIG_ENDIAN
	res1[0] = *(((uint8_t *)&eabits) + 0) & 0xff;
	res1[1] = *(((uint8_t *)&eabits) + 1) & 0xff;
	res1[2] = *(((uint8_t *)&eabits) + 2) & 0xff;
	res1[3] = *(((uint8_t *)&eabits) + 3) & 0xff;
	res1[4] = *(((uint8_t *)&eabits) + 4) & 0xff;
	res1[5] = *(((uint8_t *)&eabits) + 5) & 0xff;
	res1[6] = *(((uint8_t *)&eabits) + 6) & 0xff;
	res1[7] = *(((uint8_t *)&eabits) + 7) & 0xff;
#else
	res1[0] = *(((uint8_t *)&eabits) + 7) & 0xff;
	res1[1] = *(((uint8_t *)&eabits) + 6) & 0xff;
	res1[2] = *(((uint8_t *)&eabits) + 5) & 0xff;
	res1[3] = *(((uint8_t *)&eabits) + 4) & 0xff;
	res1[4] = *(((uint8_t *)&eabits) + 3) & 0xff;
	res1[5] = *(((uint8_t *)&eabits) + 2) & 0xff;
	res1[6] = *(((uint8_t *)&eabits) + 1) & 0xff;
	res1[7] = *(((uint8_t *)&eabits) + 0) & 0xff;
#endif /* MAP_BIG_ENDIAN */
	map_e_bit_shiftleft(&res1[0], MAP_E_IPADDR6_BITLEN, MAP_E_IPADDR6_BITLEN - rule->eabits_len - 64);
	map_e_bit_shiftright(&res1[0], MAP_E_IPADDR6_BITLEN, rule->v6_prefix_len);

	/* Rule IPv6 prefix Ѱդ */
	memset(&res[0], 0x00, sizeof(res));
	for (idx = 0; idx < 16; idx++) {
		res[idx] = *(((uint8_t *)&rule->v6_prefix.s6_addr[0]) + idx) & 0xff;
	}

	/* Rule IPv6 prefix  EA-bits  */
	map_e_bit_or(&res[0], &res1[0], MAP_E_IPADDR6_BITLEN);

	/* ˲64bitshared IPv4 address */
	shared_ipv4 = rule->v4_prefix.s_addr | htonl(host_id);
	map_e_ntop(AF_INET, &shared_ipv4, addr_string, sizeof(addr_string));
	MAP_TRACE("shared_ipv4 : %s", addr_string);

	res[9]  = *(((uint8_t *)&shared_ipv4) + 0);
	res[10] = *(((uint8_t *)&shared_ipv4) + 1);
	res[11] = *(((uint8_t *)&shared_ipv4) + 2);
	res[12] = *(((uint8_t *)&shared_ipv4) + 3);

	/* ˲64bit˥ݡȥåȤ */
#ifdef MAP_BIG_ENDIAN
	res[13] = *(((uint8_t *)&portset_id32) + 2);
	res[14] = *(((uint8_t *)&portset_id32) + 3);
#else
	res[13] = *(((uint8_t *)&portset_id32) + 1);
	res[14] = *(((uint8_t *)&portset_id32) + 0);
#endif /* MAP_BIG_ENDIAN */

	for (idx = 0; idx < 16; idx++) {
		ipv6->s6_addr[idx] = res[idx];
	}
	map_e_ntop(AF_INET6, ipv6, addr_string, sizeof(addr_string));
	MAP_TRACE("dst IPv6 : %s", addr_string);

	MAP_TRACE("end");
	return 0;

failed:
	MAP_TRACE("end");
	return -1;
}

/*
 * ǤեѿǡեȤ
 */
static uint8_t *
map_e_bit_shiftleft(uint8_t *src, int bit, int shift)
{
	int byte_pos, pos_max;
	uint8_t t8 = 0, t8_bak = 0, overflow = 0;

	pos_max = bit / MAP_E_BIT_PER_BYTES;
	if (pos_max * MAP_E_BIT_PER_BYTES != bit) {
		/* 黻оݤѿbyteñ̤ǻꤷߤ */
		MAP_TRACE("It is necessary to specify an 8-bit boundary.");
		return src;
	}

	while (shift > 0) {
		if (shift >= MAP_E_BIT_PER_BYTES) {
			/* 1byte=8bitshift */
			for (byte_pos = 0; byte_pos < pos_max - 1; byte_pos++) {
				*(src + byte_pos) = *(src + byte_pos + 1);
			}
			*(src + pos_max - 1) = 0;
			shift -= MAP_E_BIT_PER_BYTES;
		} else {
			/* üshift */
			for (byte_pos = pos_max - 1; 0 <= byte_pos; byte_pos--) {
				t8_bak = t8 = *(src + byte_pos);
				t8 = t8 << shift;

				/* 黻ΥСեȹ */
				*(src + byte_pos) = t8 | overflow;

				/* Сեʬη׻ */
				overflow = t8_bak >> (MAP_E_BIT_PER_BYTES - shift);
			}
			shift = 0;
			break;
		}
	}
	return src;
}

/*
 * ǤեѿǡեȤ
 */
static uint8_t *
map_e_bit_shiftright(uint8_t *src, int bit, int shift)
{
	int i, j, j_max;

	j_max = bit / MAP_E_BIT_PER_BYTES;
	for (i = 0; i < bit && i < shift; i++) {
		for (j = j_max - 1; j >= 0; j--) {
			if ((*(src + j)) & 0x01) {
				*(src + j) = *(src + j) & ~0x01;
				if (j < j_max - 1) {
					*(src + j + 1) = *(src + j + 1) | 0x80;
				}
			}
			*(src + j) = *(src + j) >> 1;
			*(src + j) = *(src + j) & ~0x80;
		}
	}
	return src;
}

/*
 * ǤեѿǡOR黻
 */
static uint8_t *
map_e_bit_or(uint8_t *target, uint8_t *src, int bit)
{
	int i, i_max, j_max;
	uint8_t mask;

	/* 1ХȤȤν */
	i_max = bit / MAP_E_BIT_PER_BYTES;
	for (i = 0; i < i_max; i++) {
		*(target + i) = (*(target + i)) | (*(src + i));
	}

	/* üӥåȤȤ˽ */
	j_max = bit - i_max * MAP_E_BIT_PER_BYTES;
	if (j_max > 0) {
		mask = 0xFF;
		mask = mask << (MAP_E_BIT_PER_BYTES - j_max);
		*(target + i_max) = (*(target + i_max) & mask) | (*(src + i_max) & mask);
	}

	return target;
}

/*
 * IPv4/IPv6 ɥ쥹ХʥƥȷѴ
 */
static const char *
map_e_ntop(int af, const void *src, char *dst, size_t size)
{
	uint8_t *pitem;

	if (af == AF_INET) {
		pitem = (uint8_t *)src;
		snprintf(dst, size, "%d.%d.%d.%d",
		    *(pitem + 0),
		    *(pitem + 1),
		    *(pitem + 2),
		    *(pitem + 3)
		);

		return dst;
	} else if (af == AF_INET6) {
		pitem = (uint8_t *)src;
		snprintf(dst, size, "%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x",
		    *(pitem + 0), *(pitem + 1),
		    *(pitem + 2), *(pitem + 3),
		    *(pitem + 4), *(pitem + 5),
		    *(pitem + 6), *(pitem + 7),
		    *(pitem + 8), *(pitem + 9),
		    *(pitem + 10), *(pitem + 11),
		    *(pitem + 12), *(pitem + 13),
		    *(pitem + 14), *(pitem + 15)
		);

		return dst;
	}

	return NULL;
}
EXPORT_SYMBOL(map_e_ntop);

/*
 * session򸡺ơHit餽sessionѤž
 * ICMP error
 */
static int
map_e_ce_state_napt46_err(struct sk_buff *skb, struct ip6_tnl_map_e *t, struct in6_addr *daddr,
	struct flowi6 *fl, __u8 dsfield)
{
	struct iphdr *ip4h = (struct iphdr *)skb_network_header(skb);
	struct icmphdr *ih = (struct icmphdr *)(ip4h + 1);
	struct iphdr *ip_o4h =(struct iphdr *)(ih + 1); /* ICMP errorv4إå */
	struct tcphdr *oth = NULL;   /* ICMP errorTCPإå */
	struct udphdr *ouh = NULL;   /* ICMP errorUDPإå */
	struct icmphdr *oih = NULL;  /* ICMP errorICMPإå */
	struct map_e_session *ses = NULL;

	uint16_t dport = 0, sport_old = 0;
	int ret;

	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	MAP_TRACE("start");
	if (t == NULL) {
		MAP_TRACE("t is null");
		goto tx_err_link_failure;
	}
	if (t->dev) {
		MAP_TRACE("t->dev->name : %s", t->dev->name);
	}

	switch (ip_o4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		oth = (struct tcphdr *)(ip_o4h + 1);
		dport = oth->source;
		sport_old = oth->dest;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		ouh = (struct udphdr *)(ip_o4h + 1);
		dport = ouh->source;
		sport_old = ouh->dest;
		break;
	case IPPROTO_ICMP: //Only support the format include id field
		MAP_TRACE("IPPROTO_ICMP");
		oih = (struct icmphdr *)(ip_o4h + 1);
		if (oih->type != ICMP_ECHO &&
		    oih->type != ICMP_TIMESTAMP &&
		    oih->type != ICMP_ADDRESS &&
		    oih->type != ICMP_INFO_REQUEST) {
			return -1;
		}
		dport = 0;
		sport_old = oih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		return -1;
	}
	/*
	 * åθä餽ȤäƥݡѴ
	 */
	ses = map_e_search_session46((struct in_addr *)&ip4h->daddr, (struct in_addr *)&ip4h->saddr,
	    dport, sport_old, ip_o4h->protocol, t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("not found session");
		ULOCK_R_SESSION();
		goto tx_err_link_failure;
	}
	MAP_TRACE("found session");

	/* ip_idݡȥåȤϰϤ˽񤭴 */
	ip4h->id = htons(map_e_ipid_get(t));
	ip4h->saddr = t->map_e_info->shared_addr4.s_addr;
	ip_o4h->daddr = t->map_e_info->shared_addr4.s_addr;

	/* ICMP顼ǥå󥿥޹Ϥʤ */

	switch (ip_o4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP %u -> %u", ntohs(oth->source), ntohs(ses->c_sport));

		oth->source = ses->dport;
		oth->dest = ses->c_sport;
		/* MSSĴ */
		map_e_tcpmss(oth,skb);

		oth->check = 0;
		len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
		sum1 = csum_partial((char *)oth, len, 0);
		sum2 = csum_tcpudp_magic(ip_o4h->saddr, ip_o4h->daddr,
		    len, ip_o4h->protocol, sum1);
		oth->check = sum2;
		MAP_TRACE("checksum = 0x%08x", oth->check);
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP %u -> %u", ntohs(ouh->source), ntohs(ses->c_sport));
		ouh->source = ses->dport;
		ouh->dest = ses->c_sport;
		if (ouh->check) {
			ouh->check = 0;
			len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)ouh, len, 0);
			sum2 = csum_tcpudp_magic(ip_o4h->saddr, ip_o4h->daddr,
			    len, ip_o4h->protocol, sum1);
			ouh->check = sum2;
		}
		MAP_TRACE("checksum = 0x%08x", ouh->check);
		break;
	case IPPROTO_ICMP:
		MAP_TRACE("IPPROTO_ICMP %u -> %u", ntohs(ih->un.echo.id), ntohs(ses->c_sport));
		oih->un.echo.id = ses->c_sport;

		oih->checksum = 0;
		len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
		oih->checksum = map_e_in4_checksum((uint16_t *)oih, len);
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		MAP_TRACE("end");
		ULOCK_R_SESSION();
		return -1;
	}
	/* ICMPǡIPإåå񤭴 */
	ip_o4h->check = 0;
	ip_o4h->check = map_e_in4_checksum((uint16_t *)ip_o4h, sizeof(struct iphdr));
	/* ICMPإåå񤭴 */
	ih->checksum = 0;
	len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
	ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
	/* IPإåå񤭴 */
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	/* IPv6إåɲ */
	ret = map_e_add_v6hdr(skb, &ses->src6, &ses->dst6, t, fl, dsfield);

	ULOCK_R_SESSION();

	MAP_TRACE("end");
	return ret;

tx_err_link_failure:
	MAP_TRACE("end");
	return -1;
}



/*
 * session򸡺ơHit餽sessionѤž
 */
static int
map_e_ce_state_napt46(struct sk_buff *skb, struct ip6_tnl_map_e *t, struct in6_addr *daddr,
	struct flowi6 *fl, __u8 dsfield)
{
	struct iphdr *ip4h = (struct iphdr *)skb_network_header(skb);
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	struct map_e_session *ses = NULL;

	uint16_t dport = 0, sport_old = 0;
	int ret;

	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	MAP_TRACE("start");
	if (t == NULL) {
		MAP_TRACE("t is null");
		goto tx_err_link_failure;
	}
	if (t->dev) {
		MAP_TRACE("t->dev->name : %s", t->dev->name);
	}

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		th = (struct tcphdr *)(ip4h + 1);
		dport = th->dest;
		sport_old = th->source;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		uh = (struct udphdr *)(ip4h + 1);
		dport = uh->dest;
		sport_old = uh->source;
		break;
	case IPPROTO_ICMP:
		MAP_TRACE("IPPROTO_ICMP");
		ih = (struct icmphdr *)(ip4h + 1);
		if (ih->type != ICMP_ECHO &&
		    ih->type != ICMP_TIMESTAMP &&
		    ih->type != ICMP_ADDRESS &&
		    ih->type != ICMP_INFO_REQUEST) {
			goto tx_err_link_failure;
		}
		dport = 0;
		sport_old = ih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		goto tx_err_link_failure;
	}

	/*
	 * åθä餽ȤäƥݡѴ
	 */
	ses = map_e_search_session46((struct in_addr *)&ip4h->daddr, (struct in_addr *)&ip4h->saddr,
	    dport, sport_old, ip4h->protocol, t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("not found session");
		ULOCK_R_SESSION();
		goto tx_err_link_failure;
	}
	MAP_TRACE("found session");
	ULOCK_R_SESSION();

	/* TCPξ祻å󥿥޴Τflag򹹿 */
	if (ip4h->protocol == IPPROTO_TCP)
		map_e_update_ses_tcp_flag(th, ses);

	/* FTP ALGΥå¾ϺΤᡢsession򹹿,
	   FTP ALG˼Ԥ륱Ǥ⹹뤳Ȥˤʤ뤬۾Ϥ
	   ΥåΥ顼Ǥ뤿ʤ */
	map_e_update_session_expire(skb, ses);

	if (map_e_ftp_alg_46(skb, ip4h, th, t, ses) != 0) {
		MAP_TRACE("ftp alg err");
		goto tx_err_link_failure;
	}

	LOCK_R_SESSION();
	/* ip_idݡȥåȤϰϤ˽񤭴 */
	ip4h->id = htons(map_e_ipid_get(t));
	ip4h->saddr = t->map_e_info->shared_addr4.s_addr;
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP %u -> %u", ntohs(th->source), ntohs(ses->c_sport));

		th->source = ses->c_sport;

		/* MSSĴ */
		map_e_tcpmss(th,skb);

		th->check = 0;
		len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
		sum1 = csum_partial((char *)th, len, 0);
		sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
		    len, ip4h->protocol, sum1);
		th->check = sum2;
		MAP_TRACE("checksum = 0x%08x", th->check);
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP %u -> %u", ntohs(uh->source), ntohs(ses->c_sport));
		uh->source = ses->c_sport;
		if (uh->check) {
			uh->check = 0;
			len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)uh, len, 0);
			sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
			    len, ip4h->protocol, sum1);
			uh->check = sum2;
		}
		MAP_TRACE("checksum = 0x%08x", uh->check);
		break;
	case IPPROTO_ICMP:
		MAP_TRACE("IPPROTO_ICMP %u -> %u", ntohs(ih->un.echo.id), ntohs(ses->c_sport));
		ih->un.echo.id = ses->c_sport;

		ih->checksum = 0;
		len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
		ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		MAP_TRACE("end");
		ULOCK_R_SESSION();
		return -1;
	}

	/* IPv6إåɲ */
	ret = map_e_add_v6hdr(skb, &ses->src6, &ses->dst6, t, fl, dsfield);

	ULOCK_R_SESSION();

	MAP_TRACE("end");
	return ret;

tx_err_link_failure:
	MAP_TRACE("end");
	return -1;
}

/*
 * sessionơž
 */
static int
map_e_create_state_napt46(struct sk_buff *skb, struct ip6_tnl_map_e *t, struct in6_addr *daddr, struct in_addr *old_saddr, uint16_t sport_old)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct iphdr *ip4h = (struct iphdr *)(ip6h + 1);
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	struct map_e_session *new_ses = NULL;
	uint32_t hash_index;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1], buf3[INET6_ADDRSTRLEN + 1], buf4[INET6_ADDRSTRLEN + 1], buf5[INET6_ADDRSTRLEN + 1];

	int err = -1;

	MAP_TRACE("start");

	LOCK_R_SESSION();
	if (t->map_e_info->ses_num >= t->parms.session_max) {
		MAP_TRACE("The number of sessions reached the maximum.");
		ULOCK_R_SESSION();
		goto tx_err_link_failure;
	}
	ULOCK_R_SESSION();

	new_ses = KMALLOC(sizeof(struct map_e_session), GFP_ATOMIC);
	if (new_ses == NULL) {
		printk(KERN_ERR "[%s] KMALLOC() error.\n", __func__);
		goto tx_err_link_failure;
	}
	memset(new_ses, 0x00, sizeof(struct map_e_session));

	memcpy(&new_ses->src6, &t->map_e_info->wan_addr, sizeof(struct in6_addr));
	memcpy(&new_ses->dst6, daddr, sizeof(struct in6_addr));
	new_ses->src4.s_addr   = old_saddr->s_addr;
	new_ses->c_src4.s_addr = ip4h->saddr;
	new_ses->dst4.s_addr   = ip4h->daddr;
	new_ses->protocol      = ip4h->protocol;
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		new_ses->sport = sport_old;
		new_ses->c_sport = th->source;
		new_ses->dport = th->dest;
		new_ses->expire        = map_e_expire_tcp;
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		new_ses->sport = sport_old;
		new_ses->c_sport = uh->source;
		new_ses->dport = uh->dest;
		new_ses->expire        = map_e_expire_udp;
		break;
	case IPPROTO_ICMP:
		/*
		 * ICMPid fieldĤΤդ褦ˤδؿ
		 * ƤФ˥ե륿󥰤ƤϤʤΤǡ
		 * ˤʤˤ⤷ʤ
		 */
		ih = (struct icmphdr *)(ip4h + 1);
		new_ses->sport = sport_old;
		new_ses->c_sport = ih->un.echo.id;
		new_ses->dport = 0;
		new_ses->expire        = MAP_E_EXPIRE_DEFAULT_ICMP;
		break;
	default:
		new_ses->expire        = MAP_E_EXPIRE_DEFAULT;
		break;
	}

	if (g_debug) {
		map_e_ntop(AF_INET6, &new_ses->src6, buf1, sizeof(buf1));
		map_e_ntop(AF_INET6, &new_ses->dst6, buf2, sizeof(buf2));
		map_e_ntop(AF_INET, &new_ses->src4, buf3, sizeof(buf3));
		map_e_ntop(AF_INET, &new_ses->c_src4, buf4, sizeof(buf4));
		map_e_ntop(AF_INET, &new_ses->dst4, buf5, sizeof(buf5));
		MAP_TRACE("<<new session>>");
		MAP_TRACE("  .src6    :%s", buf1);
		MAP_TRACE("  .dst6    :%s", buf2);
		MAP_TRACE("  .src4    :%s", buf3);
		MAP_TRACE("  .c_src4  :%s", buf4);
		MAP_TRACE("  .dst4    :%s", buf5);
		MAP_TRACE("  .protocol:0x%04x", new_ses->protocol);
		MAP_TRACE("  .expire  :%d", new_ses->expire);
		MAP_TRACE("  .sport   :%u", ntohs(new_ses->sport));
		MAP_TRACE("  .c_sport :%u", ntohs(new_ses->c_sport));
		MAP_TRACE("  .dport   :%u", ntohs(new_ses->dport));
	}

	/* åϿ */
	hash_index = map_e_get_hash_index(&new_ses->dst4, new_ses->dport, new_ses->protocol);
	MAP_TRACE("hash_index = %u", hash_index);
	LOCK_W_SESSION();
	list_add_tail((struct list_head *)new_ses, &t->map_e_info->ses_list[hash_index].list);
	t->map_e_info->ses_num++;
	ULOCK_W_SESSION();

	MAP_TRACE("end");
	return 0;

tx_err_link_failure:
	MAP_TRACE("end");
	return err;
}

/*
 * IPv6إåɲä
 */
static int
map_e_add_v6hdr(struct sk_buff *skb, struct in6_addr *src6,
	struct in6_addr *dst6, struct ip6_tnl_map_e *t, struct flowi6 *fl, __u8 dsfield)
{
	struct ipv6hdr *ip6h = NULL;

	skb_push(skb, sizeof(struct ipv6hdr));

	skb_reset_network_header(skb);
	ip6h = ipv6_hdr(skb);
	if (ip6h == NULL) {
		MAP_TRACE("ip6h is null");
		return -1;
	}
	ip6h->version = 6;
	ip6h->priority = 0;
	*(__be32*)ip6h = fl->flowlabel | htonl(0x60000000);
	//dsfield = INET_ECN_encapsulate(0, dsfield);
	ipv6_change_dsfield(ip6h, ~INET_ECN_MASK, dsfield);
	ip6h->payload_len = skb->len;
	ip6h->hop_limit = t->parms.hop_limit;
	ip6h->nexthdr = IPPROTO_IPIP;
	memcpy(&ip6h->saddr, src6, sizeof(struct in6_addr));
	memcpy(&ip6h->daddr, dst6, sizeof(struct in6_addr));

	return 0;
}

/*
 * sessionθV6إåNAPT
 * ICMP err, ꥸʥѥåȤν񤭴Ԥ
 */
static struct net_device *
map_e_ce_state_napt64_err(struct sk_buff *skb, struct ip6_tnl_map_e *t)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct iphdr *ip4h = (struct iphdr *)(ip6h + 1);
	struct icmphdr *ih = (struct icmphdr *)(ip4h + 1);
	struct iphdr *ip_o4h =(struct iphdr *)(ih + 1); /* ICMP errorv4إå */
	struct tcphdr *oth = NULL;   /* ICMP errorTCPإå */
	struct udphdr *ouh = NULL;   /* ICMP errorUDPإå */
	struct icmphdr *oih = NULL;  /* ICMP errorICMPإå */
	struct map_e_session *ses = NULL;
	struct net_device *dev = NULL;
	uint16_t dport, sport;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];

	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	MAP_TRACE("start");

	if (t->map_e_info->bmr == NULL) {
		MAP_TRACE("BMR is NULL.");
		goto failed;
	}
	if (ip6h->version  != 6) {
		MAP_TRACE("ignore ip version.");
		goto failed;
	}
	if (ip6h->nexthdr != IPPROTO_IPIP) {
		MAP_TRACE("ignore network protocol.");
		goto failed;
	}
 
	switch (ip_o4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		oth = (struct tcphdr *)(ip_o4h + 1);
		dport = oth->source;
		sport = oth->dest;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		ouh = (struct udphdr *)(ip_o4h + 1);
		dport = ouh->source;
		sport = ouh->dest;
		break;
	case IPPROTO_ICMP: //Only support the format include id field
		MAP_TRACE("IPPROTO_ICMP");
		oih = (struct icmphdr *)(ip_o4h + 1);
		if (oih->type != ICMP_ECHO &&
		    oih->type != ICMP_TIMESTAMP &&
		    oih->type != ICMP_ADDRESS &&
		    oih->type != ICMP_INFO_REQUEST) {
			goto failed;
		}
		dport = 0;
		sport = oih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		goto failed;
	}
	/*
	 * åθ
	 */
	MAP_TRACE("call map_e_search_session64()");
	ses = map_e_search_session64(&ip6h->daddr, &ip6h->saddr, (struct in_addr *)&ip_o4h->daddr, dport, sport, ip_o4h->protocol, t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("NOT session");
		ULOCK_R_SESSION();
		return NULL;
	}

	MAP_TRACE("session found");

	/* IPv6إåκ */
	skb->mac_header = skb->network_header;
	skb_reset_network_header(skb);
	skb->protocol = htons(ETH_P_IP);
	skb->pkt_type = PACKET_HOST;

	ip4h = (struct iphdr *)skb_network_header(skb);

	if (g_debug) {
		map_e_ntop(AF_INET, &ses->src4, buf1, sizeof(buf1));
		map_e_ntop(AF_INET, &ses->c_src4, buf2, sizeof(buf2));
		MAP_TRACE("ses->src4:%s, ses->c_src4:%s, ses->sport:%u, ses->c_sport:%u",
		    buf1, buf2, ntohs(ses->sport), ntohs(ses->c_sport));
	}

	/* IPv4ɥ쥹񤭡ݡȽ񤭴 */
	if (memcmp(&ses->src4, &ses->c_src4, sizeof(struct in_addr)) ||
	     (ses->sport != ses->c_sport)) {
		ip4h->daddr = ses->src4.s_addr;
		ip_o4h->saddr = ses->src4.s_addr;
		switch (ip_o4h->protocol) {
		case IPPROTO_TCP:
			oth->source = ses->sport;
			/* MSS */
			map_e_tcpmss_reverse(oth);

			/* ICMPǡTCPå񤭴 */
			oth->check = 0;
			len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)oth, len, 0);
			sum2 = csum_tcpudp_magic(ip_o4h->saddr, ip_o4h->daddr,
			    len, ip_o4h->protocol, sum1);
			oth->check = sum2;
			break;
		case IPPROTO_UDP:
			ouh->source = ses->sport;

			/* ICMPǡUDPå񤭴 */
			if (ouh->check) {
				ouh->check = 0;
				len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)ouh, len, 0);
				sum2 = csum_tcpudp_magic(ip_o4h->saddr, ip_o4h->daddr,
				    len, ip_o4h->protocol, sum1);
				ouh->check = sum2;
			}
			break;
		case IPPROTO_ICMP:
			oih->un.echo.id = ses->sport;
			oih->checksum = 0;
			len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
			oih->checksum = map_e_in4_checksum((uint16_t *)oih, len);
			break;
		default:
			break;
		}
	}
	/* ICMPǡIPإåå񤭴 */
	ip_o4h->check = 0;
	ip_o4h->check = map_e_in4_checksum((uint16_t *)ip_o4h, sizeof(struct iphdr));
	/* ICMPإåå񤭴 */
	ih->checksum = 0;
	len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
	ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
	/* IPإåå񤭴 */
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	dev = skb->dev;

	ULOCK_R_SESSION();

	/* ICMP顼ǥå󥿥޹Ϥʤ */

failed:
	MAP_TRACE("end");
	return dev;
}

/*
 * sessionθNAPT
 * ICMP err, ꥸʥѥåȤν񤭴Ԥ
 */
static struct net_device *
map_e_ce_state_frag_napt64_err(struct sk_buff *skb, struct ip6_tnl_map_e *t, struct ipv6hdr *ip6h)
{
//	struct ipv6hdr *ip6h = ipv6_hdr(skb);
//	struct iphdr *ip4h = (struct iphdr *)(ip6h + 1);
	struct iphdr *ip4h = (struct iphdr *)skb_network_header(skb);
	struct icmphdr *ih = (struct icmphdr *)(ip4h + 1);
	struct iphdr *ip_o4h =(struct iphdr *)(ih + 1); /* ICMP errorv4إå */
	struct tcphdr *oth = NULL;   /* ICMP errorTCPإå */
	struct udphdr *ouh = NULL;   /* ICMP errorUDPإå */
	struct icmphdr *oih = NULL;  /* ICMP errorICMPإå */
	struct map_e_session *ses = NULL;
	struct net_device *dev = NULL;
	uint16_t dport, sport;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];

	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	MAP_TRACE("start");

	if (t->map_e_info->bmr == NULL) {
		MAP_TRACE("BMR is NULL.");
		goto failed;
	}
	if (ip6h->version  != 6) {
		MAP_TRACE("ignore ip version.");
		goto failed;
	}
	if (ip6h->nexthdr != IPPROTO_IPIP) {
		MAP_TRACE("ignore network protocol.");
		goto failed;
	}
 
	switch (ip_o4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		oth = (struct tcphdr *)(ip_o4h + 1);
		dport = oth->source;
		sport = oth->dest;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		ouh = (struct udphdr *)(ip_o4h + 1);
		dport = ouh->source;
		sport = ouh->dest;
		break;
	case IPPROTO_ICMP: //Only support the format include id field
		MAP_TRACE("IPPROTO_ICMP");
		oih = (struct icmphdr *)(ip_o4h + 1);
		if (oih->type != ICMP_ECHO &&
		    oih->type != ICMP_TIMESTAMP &&
		    oih->type != ICMP_ADDRESS &&
		    oih->type != ICMP_INFO_REQUEST) {
			return -1;
		}
		dport = 0;
		sport = oih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		return -1;
	}
	/*
	 * åθ
	 */
	MAP_TRACE("call map_e_search_session64()");
	ses = map_e_search_session64(&ip6h->daddr, &ip6h->saddr, (struct in_addr *)&ip_o4h->daddr, dport, sport, ip_o4h->protocol, t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("NOT session");
		ULOCK_R_SESSION();
		return NULL;
	}

	MAP_TRACE("session found");

	ip4h = (struct iphdr *)skb_network_header(skb);

	if (g_debug) {
		map_e_ntop(AF_INET, &ses->src4, buf1, sizeof(buf1));
		map_e_ntop(AF_INET, &ses->c_src4, buf2, sizeof(buf2));
		MAP_TRACE("ses->src4:%s, ses->c_src4:%s, ses->sport:%u, ses->c_sport:%u",
		    buf1, buf2, ntohs(ses->sport), ntohs(ses->c_sport));
	}

	/* IPv4ɥ쥹񤭡ݡȽ񤭴 */
	if (memcmp(&ses->src4, &ses->c_src4, sizeof(struct in_addr)) ||
	     (ses->sport != ses->c_sport)) {
		ip4h->daddr = ses->src4.s_addr;
		ip_o4h->saddr = ses->src4.s_addr;
		switch (ip_o4h->protocol) {
		case IPPROTO_TCP:
			oth->source = ses->sport;
			/* MSS */
			map_e_tcpmss_reverse(oth);

			/* ICMPǡTCPå񤭴 */
			oth->check = 0;
			len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
			sum1 = csum_partial((char *)oth, len, 0);
			sum2 = csum_tcpudp_magic(ip_o4h->saddr, ip_o4h->daddr,
			    len, ip_o4h->protocol, sum1);
			oth->check = sum2;
			break;
		case IPPROTO_UDP:
			ouh->source = ses->sport;

			/* ICMPǡUDPå񤭴 */
			if (ouh->check) {
				ouh->check = 0;
				len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)ouh, len, 0);
				sum2 = csum_tcpudp_magic(ip_o4h->saddr, ip_o4h->daddr,
				    len, ip_o4h->protocol, sum1);
				ouh->check = sum2;
			}
			break;
		case IPPROTO_ICMP:
			oih->un.echo.id = ses->sport;
			oih->checksum = 0;
			len = ntohs(ip_o4h->tot_len) - sizeof(struct iphdr);
			oih->checksum = map_e_in4_checksum((uint16_t *)oih, len);
			break;
		default:
			break;
		}
	}
	/* ICMPǡIPإåå񤭴 */
	ip_o4h->check = 0;
	ip_o4h->check = map_e_in4_checksum((uint16_t *)ip_o4h, sizeof(struct iphdr));
	/* ICMPإåå񤭴 */
	ih->checksum = 0;
	len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
	ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
	/* IPإåå񤭴 */
	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));

	dev = skb->dev;

	ULOCK_R_SESSION();

	/* ICMP顼ǥå󥿥޹Ϥʤ */

failed:
	MAP_TRACE("end");
	return dev;
}

/*
 * sessionθNAPT
 */
static struct net_device *
map_e_ce_state_frag_napt64(struct sk_buff *skb, struct ip6_tnl_map_e *t , struct ipv6hdr *ip6h)
{
//	struct ipv6hdr *ip6h = ipv6_hdr(skb);
//	struct iphdr *ip4h = (struct iphdr *)(ip6h + 1);
	struct iphdr *ip4h = (struct iphdr *)skb_network_header(skb);
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	struct map_e_session *ses = NULL;
	struct net_device *dev = NULL;
	uint16_t dport, sport;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];

	struct skb_shared_info *skb_inf = skb_shinfo(skb);
	struct in_addr daddr;
	__sum16 sum_old = 0;
	uint32_t sum_buf = 0;

	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	MAP_TRACE("start");

	if (t->map_e_info->bmr == NULL) {
		MAP_TRACE("BMR is NULL.");
		goto failed;
	}
	if (ip6h->version  != 6) {
		MAP_TRACE("ignore ip version.");
		goto failed;
	}
	if (ip6h->nexthdr != IPPROTO_IPIP) {
		MAP_TRACE("ignore network protocol.");
		goto failed;
	}

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		th = (struct tcphdr *)(ip4h + 1);
		dport = th->dest;
		sport = th->source;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		uh = (struct udphdr *)(ip4h + 1);
		dport = uh->dest;
		sport = uh->source;
		break;
	case IPPROTO_ICMP:
		MAP_TRACE("IPPROTO_ICMP");
		ih = (struct icmphdr *)(ip4h + 1);
		if (ih->type != ICMP_ECHOREPLY &&
		    ih->type != ICMP_TIMESTAMPREPLY &&
		    ih->type != ICMP_ADDRESSREPLY) {
			goto failed;
		}
		dport = 0;
		sport = ih->un.echo.id;
		MAP_TRACE("ICMP sport=%d sequence=%d",sport,ih->un.echo.sequence);
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		goto failed;
	}

	/*
	 * åθ
	 */
	MAP_TRACE("call map_e_search_session64()");
	ses = map_e_search_session64(&ip6h->daddr, &ip6h->saddr, (struct in_addr *)&ip4h->saddr, dport, sport, ip4h->protocol, t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("NOT session");
		ULOCK_R_SESSION();
		return NULL;
	}

	MAP_TRACE("session found");

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		break;
	case IPPROTO_ICMP:
		ih = (struct icmphdr *)(ip4h + 1);
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		ULOCK_R_SESSION();
		goto failed;
	}

	if (g_debug) {
		map_e_ntop(AF_INET, &ses->src4, buf1, sizeof(buf1));
		map_e_ntop(AF_INET, &ses->c_src4, buf2, sizeof(buf2));
		MAP_TRACE("ses->src4:%s, ses->c_src4:%s, ses->sport:%u, ses->c_sport:%u",
		    buf1, buf2, ntohs(ses->sport), ntohs(ses->c_sport));
	}

	/* IPv4ɥ쥹񤭡ݡȽ񤭴 */
	if (memcmp(&ses->src4, &ses->c_src4, sizeof(struct in_addr)) ||
	     (ses->sport != ses->c_sport)) {

		memcpy(&daddr, &ip4h->daddr, sizeof(struct in_addr));
		ip4h->daddr = ses->src4.s_addr;
		switch (ip4h->protocol) {
		case IPPROTO_TCP:

			/* FTP ALG: ѥåȤackֹѴ */
			if (ntohs(th->source) == 21) {
				th->ack_seq = htonl(ntohl(th->ack_seq) + ses->offset_after);
				MAP_TRACE("FTP ALG: new ack=0x%x(%u)", th->ack_seq, ntohl(th->ack_seq));

				if (th->syn || th->fin) {
					MAP_TRACE("Find SYN/FIN");
					ses->correction_pos = 0;
					ses->offset_before	= 0;
					ses->offset_after	= 0;
				}

			}

			th->dest = ses->sport;

			/* MSSĴ */
			map_e_tcpmss(th,skb);

			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				th->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)th, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);

			} else {
				/* fragment */
				sum_old = th->check;
				sum_buf = ((uint16_t)~sum_old + (uint16_t)~dport
					+ (uint16_t)~(daddr.s_addr>>16) + (uint16_t)~(0xffff&daddr.s_addr)
					+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + th->dest);
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				sum2 = (uint16_t)~sum_buf;
			}
			th->check = sum2;
			MAP_TRACE("checksum = 0x%08x", th->check);
			break;
		case IPPROTO_UDP:
			uh->dest = ses->sport;
			if (uh->check ==0)
				break;
			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				uh->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)uh, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);
			} else {
				/* fragment */
				sum_old = uh->check;
				sum_buf = ((uint16_t)~sum_old + (uint16_t)~dport
					+ (uint16_t)~(daddr.s_addr>>16) + (uint16_t)~(0xffff&daddr.s_addr)
					+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + uh->dest);
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				sum2 = (uint16_t)~sum_buf;
			}
			uh->check = sum2;
			MAP_TRACE("checksum = 0x%08x", uh->check);
			break;
		case IPPROTO_ICMP:
			if (skb_inf->frag_list == NULL) {
			        MAP_TRACE("not fragmet");
				/* not fragmet */
				ih->un.echo.id = ses->sport;
				ih->checksum = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
			} else {
			        MAP_TRACE("fragmet");
				/* fragment */
				sum_old = ih->checksum;
				sum_buf = (uint16_t)~sum_old + (uint16_t)~ih->un.echo.id + ses->sport;
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				ih->un.echo.id = ses->sport;
				ih->checksum = (uint16_t)~sum_buf;
			}
			MAP_TRACE("checksum = 0x%08x", ih->checksum);
			break;
		default:
			break;
		}
	}

	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));
	dev = skb->dev;

	ULOCK_R_SESSION();

	/* TCPξ祻å󥿥޴Τflag򹹿 */
	if (ip4h->protocol == IPPROTO_TCP)
		map_e_update_ses_tcp_flag(th, ses);

	/* session򹹿 */
	map_e_update_session_expire(skb, ses);

failed:
	MAP_TRACE("end");
	return dev;
}

/*
 * sessionθV6إåNAPT
 */
static struct net_device *
map_e_ce_state_napt64(struct sk_buff *skb, struct ip6_tnl_map_e *t)
{
	struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct iphdr *ip4h = (struct iphdr *)(ip6h + 1);
	struct tcphdr *th = NULL;
	struct udphdr *uh = NULL;
	struct icmphdr *ih = NULL;
	struct map_e_session *ses = NULL;
	struct net_device *dev = NULL;
	uint16_t dport, sport;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];

	struct skb_shared_info *skb_inf = skb_shinfo(skb);
	struct in_addr daddr;
	__sum16 sum_old = 0;
	uint32_t sum_buf = 0;

	__wsum  sum1 = 0;
	__sum16 sum2 = 0;
	unsigned int len = 0;

	MAP_TRACE("start");

	if (t->map_e_info->bmr == NULL) {
		MAP_TRACE("BMR is NULL.");
		goto failed;
	}
	if (ip6h->version  != 6) {
		MAP_TRACE("ignore ip version.");
		goto failed;
	}
	if (ip6h->nexthdr != IPPROTO_IPIP) {
		MAP_TRACE("ignore network protocol.");
		goto failed;
	}

	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		th = (struct tcphdr *)(ip4h + 1);
		dport = th->dest;
		sport = th->source;
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		uh = (struct udphdr *)(ip4h + 1);
		dport = uh->dest;
		sport = uh->source;
		break;
	case IPPROTO_ICMP:
		MAP_TRACE("IPPROTO_ICMP");
		ih = (struct icmphdr *)(ip4h + 1);
		if (ih->type != ICMP_ECHOREPLY &&
		    ih->type != ICMP_TIMESTAMPREPLY &&
		    ih->type != ICMP_ADDRESSREPLY) {
			goto failed;
		}
		dport = 0;
		sport = ih->un.echo.id;
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		goto failed;
	}

	/*
	 * åθ
	 */
	MAP_TRACE("call map_e_search_session64()");
	ses = map_e_search_session64(&ip6h->daddr, &ip6h->saddr, (struct in_addr *)&ip4h->saddr, dport, sport, ip4h->protocol, t);

	LOCK_R_SESSION();
	if (ses == NULL) {
		MAP_TRACE("NOT session");
		ULOCK_R_SESSION();
		return NULL;
	}

	MAP_TRACE("session found");

	/* IPv6إåκ */
	skb->mac_header = skb->network_header;
	skb_reset_network_header(skb);
	skb->protocol = htons(ETH_P_IP);
	skb->pkt_type = PACKET_HOST;

	ip4h = (struct iphdr *)skb_network_header(skb);
	switch (ip4h->protocol) {
	case IPPROTO_TCP:
		th = (struct tcphdr *)(ip4h + 1);
		break;
	case IPPROTO_UDP:
		uh = (struct udphdr *)(ip4h + 1);
		break;
	case IPPROTO_ICMP:
		ih = (struct icmphdr *)(ip4h + 1);
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		ULOCK_R_SESSION();
		goto failed;
	}

	if (g_debug) {
		map_e_ntop(AF_INET, &ses->src4, buf1, sizeof(buf1));
		map_e_ntop(AF_INET, &ses->c_src4, buf2, sizeof(buf2));
		MAP_TRACE("ses->src4:%s, ses->c_src4:%s, ses->sport:%u, ses->c_sport:%u",
		    buf1, buf2, ntohs(ses->sport), ntohs(ses->c_sport));
	}

	/* IPv4ɥ쥹񤭡ݡȽ񤭴 */
	if (memcmp(&ses->src4, &ses->c_src4, sizeof(struct in_addr)) ||
	     (ses->sport != ses->c_sport)) {

		memcpy(&daddr, &ip4h->daddr, sizeof(struct in_addr));
		ip4h->daddr = ses->src4.s_addr;
		switch (ip4h->protocol) {
		case IPPROTO_TCP:

			/* FTP ALG: ѥåȤackֹѴ */
			if (ntohs(th->source) == 21) {
				th->ack_seq = htonl(ntohl(th->ack_seq) + ses->offset_after);
				MAP_TRACE("FTP ALG: new ack=0x%x(%u)", th->ack_seq, ntohl(th->ack_seq));

				if (th->syn || th->fin) {
					MAP_TRACE("Find SYN/FIN");
					ses->correction_pos = 0;
					ses->offset_before	= 0;
					ses->offset_after	= 0;
				}

			}

			th->dest = ses->sport;

			/* MSSĴ */
			map_e_tcpmss(th,skb);

			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				th->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)th, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);

			} else {
				/* fragment */
				sum_old = th->check;
				sum_buf = ((uint16_t)~sum_old + (uint16_t)~dport
					+ (uint16_t)~(daddr.s_addr>>16) + (uint16_t)~(0xffff&daddr.s_addr)
					+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + th->dest);
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				sum2 = (uint16_t)~sum_buf;
			}
			th->check = sum2;
			MAP_TRACE("checksum = 0x%08x", th->check);
			break;
		case IPPROTO_UDP:
			uh->dest = ses->sport;
			if (uh->check == 0)
				break;
			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				uh->check = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				sum1 = csum_partial((char *)uh, len, 0);
				sum2 = csum_tcpudp_magic(ip4h->saddr, ip4h->daddr,
				    len, ip4h->protocol, sum1);
			} else {
				/* fragment */
				sum_old = uh->check;
				sum_buf = ((uint16_t)~sum_old + (uint16_t)~dport
					+ (uint16_t)~(daddr.s_addr>>16) + (uint16_t)~(0xffff&daddr.s_addr)
					+ (ip4h->daddr>>16) + (0xffff&ip4h->daddr) + uh->dest);
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				sum2 = (uint16_t)~sum_buf;
			}
			uh->check = sum2;
			MAP_TRACE("checksum = 0x%08x", uh->check);
			break;
		case IPPROTO_ICMP:
			if (skb_inf->frag_list == NULL) {
				/* not fragmet */
				ih->un.echo.id = ses->sport;
				ih->checksum = 0;
				len = ntohs(ip4h->tot_len) - sizeof(struct iphdr);
				ih->checksum = map_e_in4_checksum((uint16_t *)ih, len);
			} else {
				/* fragment */
				sum_old = ih->checksum;
				sum_buf = (uint16_t)~sum_old + (uint16_t)~ih->un.echo.id + ses->sport;
				sum_buf = (sum_buf>>16) + sum_buf&0xffff;
				ih->un.echo.id = ses->sport;
				ih->checksum = (uint16_t)~sum_buf;
			}
			MAP_TRACE("checksum = 0x%08x", ih->checksum);
			break;
		default:
			break;
		}
	}

	ip4h->check = 0;
	ip4h->check = map_e_in4_checksum((uint16_t *)ip4h, sizeof(struct iphdr));
	dev = skb->dev;

	ULOCK_R_SESSION();

	/* TCPξ祻å󥿥޴Τflag򹹿 */
	if (ip4h->protocol == IPPROTO_TCP)
		map_e_update_ses_tcp_flag(th, ses);

	/* session򹹿 */
	map_e_update_session_expire(skb, ses);

failed:
	MAP_TRACE("end");
	return dev;
}

/*
 * ICMPΥꥯȤȱɳդ
 */
static int 
match_icmp_type(uint8_t src_type, uint8_t  ses_type, uint16_t src_id, uint16_t ses_id)
{
	switch (src_type) {
	case ICMP_ECHOREPLY:
		if (ses_type != ICMP_ECHO)
			return 0;
		break;
	case ICMP_TIMESTAMPREPLY:
		if (ses_type != ICMP_TIMESTAMP)
			return 0;
		break;
	case ICMP_ADDRESSREPLY:
		if (ses_type != ICMP_ADDRESS)
			return 0;
		break;
	case ICMP_ECHO:
		if (ses_type != ICMP_ECHOREPLY)
			return 0;
		break;
	case ICMP_TIMESTAMP:
		if (ses_type != ICMP_TIMESTAMPREPLY)
			return 0;
		break;
	case ICMP_ADDRESS:
		if (ses_type != ICMP_ADDRESSREPLY)
			return 0;
		break;
	case ICMP_DEST_UNREACH:
	case ICMP_REDIRECT:
	case ICMP_TIME_EXCEEDED:
	case ICMP_PARAMETERPROB:
	default:
		MAP_TRACE("ICMP Type not supported.");
		return 0;
	}

	if (src_id == ses_id) {
		return 1;
	}

	MAP_TRACE("Session ID is different.");
	return 0;
}

/*
 * åη׻
 */
static uint16_t
map_e_in4_checksum(uint16_t *ip4h, int buf_size)
{
	uint16_t *buf = ip4h;
	uint32_t sum = 0;

	while(buf_size > 1){
		sum += *buf; 
		if(sum & 0x80000000) {
			sum = (sum&0xFFFF) + (sum>>16);
		}
		buf++; 
		buf_size -= 2; 
	} 
	if (buf_size == 1){
		u_int16_t val = 0;
		memcpy(&val, buf, sizeof(u_int8_t));
		sum += val;
	}

	sum = (sum & 0xffff) + (sum >> 16); 
	sum = (sum & 0xffff) + (sum >> 16); 

	return ~sum;
}

/*
 * IPv4ɥ쥹ͭIPv4ɥ쥹ɤå.
 */
static int
map_e_dstv4_currentv4addr_check(struct in_addr *dstv4, struct ip6_tnl_map_e *t)
{
	if (dstv4 == NULL) {
		MAP_TRACE("dstv4 is null.");
		return 0;
	}
	if (!t->map_e_info->bmr) {
		MAP_TRACE("bmr is null.");
		return 0;
	}

	if (t->map_e_info->shared_addr4.s_addr == dstv4->s_addr) {
		return 1;
	}

	return 0;
}

/*
 * ꤷݡֹ椬ݡȥåȤϰ⤫Ƚꤹ.
 */
static int
map_e_port_range_check(uint16_t port, struct ip6_tnl_map_e *t)
{
	struct list_head *p;
	struct map_e_portset *ps;

	MAP_TRACE("start, port=%u", port);
	list_for_each(p, &t->map_e_info->portset_list.list) {
		ps = list_entry(p, struct map_e_portset, list);
		MAP_TRACE("ps->port_min=%u, ps->port_max=%u",
		    ps->port_min, ps->port_max);
		if (ps->port_min <= port && port <= ps->port_max) {
			MAP_TRACE("return 1");
			return 1;
		}
	}

	return 0;
}
EXPORT_SYMBOL(map_e_port_range_check);

/*
 * ʰIPv4ɥ쥹ʬƱDomain 4rd prefixǤʤˡ
 * äƤmapping ruleΥꥹȤå
 * IPv4ɥ쥹ƱDomain 4rd prefixΤ(롼)õؿ
 *
 * ͡
 * struct map_e_rule * 
 * NULL 
 */
static struct map_e_rule *
map_e_dstv4_to_rule(struct in_addr *dstv4, struct ip6_tnl_map_e *t)
{
	struct list_head *p;
	struct map_e_rule *rule;
	struct map_e_rule *rule_longest = NULL;
	uint32_t mask;
	int bits_longest = 0;

	MAP_TRACE("start");
	if (dstv4 == NULL) {
		MAP_TRACE("dstv4 is null.");
		return NULL;
	}

	list_for_each(p, &t->map_e_info->fmr_list.list) {
		rule = list_entry(p, struct map_e_rule, list);

		mask = 0xFFFFFFFF << (32 - rule->v4_prefix_len);
		MAP_TRACE("(rule->v4_prefix, dstv4, mask)=(0x%08x, 0x%08x, 0x%08x)",
		    ntohl(rule->v4_prefix.s_addr), ntohl(dstv4->s_addr), mask);
		if ((ntohl(rule->v4_prefix.s_addr) & mask) == (ntohl(dstv4->s_addr) & mask)) {
			if (bits_longest < rule->v4_prefix_len) {
				rule_longest = rule;
				bits_longest = rule->v4_prefix_len;
			} else if (bits_longest == rule->v4_prefix_len) {
				MAP_TRACE("dupulicate IPv4 prefix.");
			}
		}
	}

	if (rule_longest == NULL) {
		MAP_TRACE("rule_longest is null.");
		return NULL;
	}

	return rule_longest;
}

/*
 * ip_idμ.
 */
static uint16_t
map_e_ipid_get(struct ip6_tnl_map_e *t)
{
	struct list_head *p;
	struct map_e_portset *ps = NULL;
	int index;

	do {
		/* ݡȥåȤ */
		index = 0;
		list_for_each(p, &t->map_e_info->portset_list.list) {
			ps = list_entry(p, struct map_e_portset, list);
			if (index == t->map_e_info->current_portset_index) {
				break;
			}
		}
		if (ps == NULL) {
			break;
		}

		t->map_e_info->current_portset_id++;
		if (t->map_e_info->current_portset_id <= ps->port_max) {
			return t->map_e_info->current_portset_id;
		} else {
			if (t->map_e_info->portset_list.list.next) {
				/* Υݡȥå */
				t->map_e_info->current_portset_index++;

			#ifdef CONFIG_RTL_FAST_MAP_E
				if(fast_map_e_fw){	//ip->id
					if ( (t->map_e_info->current_portset_index) & (t->map_e_info->portset_num >> 1))
						break;
				}
			#endif
			
				continue;
			} else {
				break;
			}
		}
	} while (0);

	/* ֺǽΥȥ */
	if (list_empty((struct list_head *)&t->map_e_info->portset_list.list)) {
		goto failed;
	}
	ps = list_first_entry(&t->map_e_info->portset_list.list, struct map_e_portset, list);
	t->map_e_info->current_portset_id = ps->port_min;
	t->map_e_info->current_portset_index = 0;
	return t->map_e_info->current_portset_id;

failed:
	/* ۾ʾ */
	printk(KERN_WARNING "[%s] ipid not assigned.\n", __func__);
	return 0;
}

/*
 * hash index
 */
static uint32_t
map_e_get_hash_index(const struct in_addr *daddr, uint16_t dport, uint8_t proto)
{
	return (daddr->s_addr ^ (daddr->s_addr>>16) ^ dport ^ proto) % MAP_E_HASH_MASK;
}

/*
 * ϿƤ륻å򸡺(TCP/UDPǡIPv4->IPv6ΤȤ˻Ѥ븡)
 */
static struct map_e_session *
map_e_search_session46(
	struct in_addr *daddr, struct in_addr *saddr,
	uint16_t dport, uint16_t sport,
	uint8_t protocol, struct ip6_tnl_map_e *t)
{
	uint32_t hash_index;
	struct list_head *p;
	struct map_e_session *ses;
	char buf[INET6_ADDRSTRLEN + 1];

	if (daddr == NULL) {
		MAP_TRACE("daddr is null.");
		return NULL;
	}
	if (saddr == NULL) {
		MAP_TRACE("saddr is null.");
		return NULL;
	}

	hash_index = map_e_get_hash_index((const struct in_addr *)daddr, dport, protocol);
	if (g_debug) {
		map_e_ntop(AF_INET, daddr, buf, sizeof(buf));
		MAP_TRACE("hash_index:%u(daddr:%s, dport:%u)", hash_index, buf, ntohs(dport));
	}

	LOCK_R_SESSION();
	list_for_each(p, &t->map_e_info->ses_list[hash_index].list) {
		ses = list_entry(p, struct map_e_session, list);

		if (daddr->s_addr == ses->dst4.s_addr &&
		    saddr->s_addr == ses->src4.s_addr &&
		    protocol == ses->protocol) {
			switch (protocol) {
			case IPPROTO_TCP:
			case IPPROTO_UDP:
				if (dport == ses->dport && sport == ses->sport) {
					ULOCK_R_SESSION();
					return ses;
				}
				break;
			case IPPROTO_ICMP:
				if (sport == ses->sport) {
					ULOCK_R_SESSION();
					return ses;
				}
				break;
			default:
				continue;
			}
		}
	}
	ULOCK_R_SESSION();

	return NULL;
}

/*
 * ϿƤ륻å򸡺(TCP/UDPǡIPv6->IPv4ΤȤ˻Ѥ븡)
 */
static struct map_e_session *
map_e_search_session64(
	struct in6_addr *daddr6,
	struct in6_addr *saddr6,
	struct in_addr *saddr4,
	uint16_t dport,
	uint16_t sport,
	uint8_t protocol,
	struct ip6_tnl_map_e *t)
{
	struct list_head *p = NULL;
	struct map_e_session *ses = NULL;
	uint32_t hash_index;

	int count;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1], buf3[INET6_ADDRSTRLEN + 1];

	MAP_TRACE("start");

	switch (protocol) {
	case IPPROTO_TCP:
		MAP_TRACE("IPPROTO_TCP");
		hash_index = map_e_get_hash_index(saddr4, sport, protocol);
		break;
	case IPPROTO_UDP:
		MAP_TRACE("IPPROTO_UDP");
		hash_index = map_e_get_hash_index(saddr4, sport, protocol);
		break;
	case IPPROTO_ICMP:
		MAP_TRACE("IPPROTO_ICMP");
		hash_index = map_e_get_hash_index(saddr4, dport, protocol);
		break;
	default:
		MAP_TRACE("ignore transport protocol.");
		return NULL;
	}

	/*
	 * ꥹȤθ
	 */
	count = 0;
	if (g_debug) {
		map_e_ntop(AF_INET6, saddr6, buf1, sizeof(buf1));
		map_e_ntop(AF_INET6, daddr6, buf2, sizeof(buf2));
		map_e_ntop(AF_INET,  saddr4, buf3, sizeof(buf3));
		MAP_TRACE("hash_index:%u", hash_index);
		MAP_TRACE("saddr6:%s, daddr6:%s, saddr4:%s, proto:0x%04x", buf1, buf2, buf3, protocol);
	}

	LOCK_R_SESSION();
	list_for_each(p, &t->map_e_info->ses_list[hash_index].list) {
		ses = list_entry(p, struct map_e_session, list);

		if (g_debug) {
			map_e_ntop(AF_INET6, &ses->dst6, buf1, sizeof(buf1));
			map_e_ntop(AF_INET6, &ses->src6, buf2, sizeof(buf2));
			map_e_ntop(AF_INET, &ses->dst4.s_addr, buf3, sizeof(buf3));
			MAP_TRACE("  [%d] ses.saddr:%s, ses.daddr:%s, ses.ip4_src:%s, proto:0x%04x",
			    count, buf1, buf2, buf3, ses->protocol);
		}
		count++;

		if (ipv6_addr_equal(saddr6, &ses->dst6) &&
		    ipv6_addr_equal(daddr6, &ses->src6) &&
		    saddr4->s_addr == ses->dst4.s_addr &&
		    protocol == ses->protocol) {
			MAP_TRACE("eaual !!");
			switch (protocol) {
			case IPPROTO_TCP:
				MAP_TRACE("IPPROTO_TCP th->source:%u, ses->dport:%u, th->dest:%u, ses->c_sport:%u",
				    ntohs(sport), ntohs(ses->dport), ntohs(dport), ntohs(ses->c_sport));
				if (sport == ses->dport &&
				    dport == ses->c_sport) {
					MAP_TRACE("IPPROTO_TCP Get!!");
					ULOCK_R_SESSION();
					return ses;
				}
				break;
			case IPPROTO_UDP:
				MAP_TRACE("IPPROTO_UDP uh->source:%u, ses->dport:%u, uh->dest:%u, ses->c_sport:%u",
				    ntohs(sport), ntohs(ses->dport), ntohs(dport), ntohs(ses->c_sport));
				if (sport == ses->dport &&
				    dport == ses->c_sport) {
					MAP_TRACE("IPPROTO_UDP Get!!");
					ULOCK_R_SESSION();
					return ses;
				}
				break;
			case IPPROTO_ICMP:
				MAP_TRACE("IPPROTO_ICMP ih->un.echo.id:%u, ses->dport:%u, ses->c_sport:%u, (dport:%u)",
				    ntohs(sport), ntohs(ses->dport), ntohs(ses->c_sport), ntohs(dport));
				if (sport == ses->c_sport) {
					MAP_TRACE("IPPROTO_ICMP Get!!");
					ULOCK_R_SESSION();
					return ses;
				}
				break;
			}
		}
	}
	ULOCK_R_SESSION();

	return NULL;
}

/*
 * ݡȤμ
 */
static uint16_t
map_e_napt_port_get(
	struct in_addr *daddr, struct in_addr *saddr,
	uint16_t dport, uint16_t sport,
	uint8_t protocol, struct ip6_tnl_map_e *t)
{
	struct list_head *p, *q;
	struct map_e_portset *ps = NULL;
	struct map_e_portset *cur_ps = NULL;
	struct map_e_staticnapt *sn = NULL;
	uint16_t result_sport = 0;
	uint16_t cur_sport = 0;
	uint16_t host_sport = ntohs(sport);
	uint16_t net_sport = 0;
	int ps_index, cur_ps_index, index;
	int found = 0;
	int ps_index_increment = 1;

	struct map_e_session *ses;
	uint32_t hash_index;
	uint32_t eim_search_index;

	MAP_TRACE("start");
	hash_index = map_e_get_hash_index((const struct in_addr *)daddr, dport, protocol);

	/*
	 * TCPޤUDPǡݡֹ椬ŪNAPTΤΤȰפϡΥݡֹѤ
	 */
	if (protocol == IPPROTO_TCP || protocol == IPPROTO_UDP) {

		LOCK_R_STATICNAPT();
		list_for_each(p, &t->map_e_info->staticnapt_list.list) {
			sn = list_entry(p, struct map_e_staticnapt, list);
			/* 1:1 modeΤȤsam_port=0Ȥʤ뤿ᡢ1:1modeξȽǤѤ */
			if (sn->sam_port != 0 && sn->lan_port == host_sport && sn->lan_addr.s_addr == saddr->s_addr && sn->protocol == protocol) {
				found = 0;
				list_for_each(q, &t->map_e_info->portset_list.list) {
					ps = list_entry(q, struct map_e_portset, list);
					if (ps->port_min <= sn->sam_port && sn->sam_port <= ps->port_max) {
						found = 1;
						break;
					}
				}
				if (found) {
					MAP_TRACE("static napt port : %u -> %u", host_sport, sn->sam_port);
					ULOCK_R_STATICNAPT();
					return sn->sam_port;
				} else {
					MAP_TRACE("static napt port(but, out of range) : %u", host_sport);
					ULOCK_R_STATICNAPT();
					return 0;
				}
			}
			/* 1:1 mode */
			else if (sn->sam_port == 0 &&
					 sn->lan_addr.s_addr == saddr->s_addr &&
					 sn->protocol == protocol &&
					 (sn->lan_port <= host_sport && sn->lan_toport >= host_sport)) {
				MAP_TRACE("static napt port : %u isn't translated (1:1 mode)", host_sport);
				ULOCK_R_STATICNAPT();
				return host_sport;
			}
		}
		ULOCK_R_STATICNAPT();
	}
	/*
	 * NAT-EIM implemetation.
	 * search saddr, sport, protocol from session,
	 * if session entry match, transform sport is used.
	 */
	LOCK_R_SESSION();

	for (eim_search_index = 0; eim_search_index < MAP_E_HASH_MASK; eim_search_index++) {
		list_for_each(p, &t->map_e_info->ses_list[eim_search_index].list) {
			ses = list_entry(p, struct map_e_session, list);
			if (saddr->s_addr == ses->src4.s_addr &&
			    protocol == ses->protocol &&
			    sport == ses->sport) {
				ULOCK_R_SESSION();
				return ntohs(ses->c_sport);
			}
		}
	}
	ULOCK_R_SESSION();

	/*
	 * ѴΥݡֹ椬ݡȥåȤϰϤäƤ
	 * ġΥݡֹ椬ѤƤʤä顢Ѥ褦ˤ롣
	 */
	LOCK_R_SESSION();
	list_for_each(p, &t->map_e_info->portset_list.list) {
		ps = list_entry(p, struct map_e_portset, list);
		if (ps->port_min <= host_sport && host_sport <= ps->port_max) {
			found = 0;
			list_for_each(q, &t->map_e_info->ses_list[hash_index].list) {
				ses = list_entry(q, struct map_e_session, list);
				if ((ses->c_sport == sport) && (ses->dport == dport) &&
				    (ses->dst4.s_addr == daddr->s_addr) && (ses->protocol == protocol)) {
					found = 1;
					break;
				}
			}
			if (found == 0) {
				MAP_TRACE("not change port : %u", host_sport);
				ULOCK_R_SESSION();
				return host_sport;
			}
		}
	}
	ULOCK_R_SESSION();

	/*
	 * ˥ݡȥåȤ򤹤
	 */
	srandom32((uint32_t)jiffies);
	ps_index = random32() % t->map_e_info->portset_num;
	cur_ps_index = ps_index;

	LOCK_R_SESSION();
	while (ps_index >= 0) {
		/* ݡȥåȤ */
		index = 0;
		list_for_each(p, &t->map_e_info->portset_list.list) {
			ps = list_entry(p, struct map_e_portset, list);
			if (index == ps_index) {
				cur_ps = ps;
				break;
			}
			index++;
		}
		if (cur_ps == NULL) {
			MAP_TRACE("cur_ps == NULL??,  ps_index=%d", ps_index);
			ULOCK_R_SESSION();
			return 0;
		}

		/* ˥ݡֹ򤹤 */
		result_sport = (random32() % (ps->port_max - ps->port_min + 1)) + ps->port_min;
		cur_sport = result_sport;
		found = 0;
		net_sport = htons(result_sport);
		list_for_each(p, &t->map_e_info->ses_list[hash_index].list) {
			ses = list_entry(p, struct map_e_session, list);
			if ((ses->c_sport) == net_sport && (ses->dport == dport) &&
			    (ses->dst4.s_addr == daddr->s_addr) && (ses->protocol == protocol)) {
				found = 1;
				break;
			}
		}
		if (found == 0) {
			MAP_TRACE("get port : %u", result_sport);
			ULOCK_R_SESSION();
			return result_sport;
		}

		/* 򤷤ݡֹ겼ͤ򸡾 */
		for (result_sport = cur_sport - 1; result_sport >= cur_ps->port_min; result_sport--) {
			found = 0;
			net_sport = htons(result_sport);
			list_for_each(p, &t->map_e_info->ses_list[hash_index].list) {
				ses = list_entry(p, struct map_e_session, list);
				if ((ses->c_sport) == net_sport && (ses->dport == dport) &&
				    (ses->dst4.s_addr == daddr->s_addr) && (ses->protocol == protocol)) {
					found = 1;
					break;
				}
			}
			if (found == 0) {
				MAP_TRACE("get port : %u", result_sport);
				ULOCK_R_SESSION();
				return result_sport;
			}
		}

		/* 򤷤ݡֹͤ򸡾 */
		for (result_sport = cur_sport + 1; result_sport <= cur_ps->port_max; result_sport++) {
			found = 0;
			net_sport = htons(result_sport);
			list_for_each(p, &t->map_e_info->ses_list[hash_index].list) {
				ses = list_entry(p, struct map_e_session, list);
				if ((ses->c_sport) == net_sport && (ses->dport == dport) &&
				    (ses->dst4.s_addr == daddr->s_addr) && (ses->protocol == protocol)) {
					found = 1;
					break;
				}
			}
			if (found == 0) {
				MAP_TRACE("get port : %u", result_sport);
				ULOCK_R_SESSION();
				return result_sport;
			}
		}

		/* оݤΥݡȥåȤ򤹤 */
		if (ps_index_increment == 1) {
			ps_index++;
			if (ps_index >= t->map_e_info->portset_num) {
				ps_index_increment = 0;
				ps_index = cur_ps_index - 1;
			}
		} else {
			ps_index--;
		}
	}
	ULOCK_R_SESSION();

	/* ݡȥåȤ˶̵ */
	MAP_TRACE("[%s] NAPT port not assigned.\n", __func__);
	return 0;
}

/*
 * åν.
 * ߤҤȤĤIFˤΤб
 */
static void
map_e_session_init(struct ip6_tnl_map_e *t)
{
	LOCK_W_SESSION();

	init_timer(&map_e_session_callout);
	map_e_session_callout.expires = jiffies + MAP_E_EXPIRE_INTERVAL * HZ;
	map_e_session_callout.data = (unsigned long)t;
	map_e_session_callout.function = map_e_session_timer;
	add_timer(&map_e_session_callout);

	ULOCK_W_SESSION();
}
EXPORT_SYMBOL(map_e_session_init);

static void
map_e_session_destroy(struct ip6_tnl_map_e *t)
{
	uint8_t hash_index;
	struct map_e_session *ses;

	LOCK_W_SESSION();

	/* ޡ */
	del_timer(&map_e_session_callout);

	/* å */
	for (hash_index = 0; hash_index < MAP_E_HASH_MASK; hash_index++) {
		while (!list_empty((struct list_head *)&t->map_e_info->ses_list[hash_index].list)) {
			ses = list_first_entry(&t->map_e_info->ses_list[hash_index].list, struct map_e_session, list);
			list_del((struct list_head *)ses);
			KFREE(ses);
			t->map_e_info->ses_num--;
		}
	}

	ULOCK_W_SESSION();
}
EXPORT_SYMBOL(map_e_session_destroy);

static void
map_e_session_clear(struct ip6_tnl_map_e *t)
{
	uint8_t hash_index;
	struct map_e_session *ses;

	LOCK_W_SESSION();

	/* å */
	for (hash_index = 0; hash_index < MAP_E_HASH_MASK; hash_index++) {
		while (!list_empty((struct list_head *)&t->map_e_info->ses_list[hash_index].list)) {
			ses = list_first_entry(&t->map_e_info->ses_list[hash_index].list, struct map_e_session, list);
			list_del((struct list_head *)ses);
			KFREE(ses);
		}
	}
	t->map_e_info->ses_num = 0;

	ULOCK_W_SESSION();
}
EXPORT_SYMBOL(map_e_session_clear);
/*
 * åδƻ륿.
 */
static void
map_e_session_timer(unsigned long data)
{
	uint8_t hash_index;
	struct list_head *p, *n;
	struct map_e_session *ses;
	struct ip6_tnl_map_e *t = (struct ip6_tnl_map_e *)data;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1];
	char buf3[INET6_ADDRSTRLEN + 1], buf4[INET6_ADDRSTRLEN + 1];
	char buf5[INET6_ADDRSTRLEN + 1];

#ifdef CONFIG_RTL_FAST_MAP_E	
	rtl_fp_napt_entry fpNaptEntry={0};
#endif

	LOCK_W_SESSION();

	if (!t || !t->map_e_info) {
		MAP_TRACE("t->map_e_info is null, goto end_timer");
		goto end_timer;
	}

	for (hash_index = 0; hash_index < MAP_E_HASH_MASK; hash_index++) {
		list_for_each(p, &t->map_e_info->ses_list[hash_index].list) {
			ses = list_entry(p, struct map_e_session, list);
		#ifdef CONFIG_RTL_FAST_MAP_E
			//If the seesion has been added in fastpath, it will be deleted by IPv4 nf_conn instead
			if(rtl_construct_mape_fp_session(ses, &fpNaptEntry) && rtl_existInMAPEFastpath(&fpNaptEntry)){
				MAP_TRACE("Added in fastpath!! (fpNaptEntry->intPort:%d)\n",fpNaptEntry.intPort);
				continue;
			}
		#endif
			ses->expire -= MAP_E_EXPIRE_INTERVAL;
			if (ses->expire < 0) {
				if (g_debug) {
					MAP_TRACE("*** session expire ***");
					map_e_ntop(AF_INET6, &ses->dst6, buf1, sizeof(buf1));
					map_e_ntop(AF_INET6, &ses->src6, buf2, sizeof(buf2));
					map_e_ntop(AF_INET, &ses->src4, buf3, sizeof(buf3));
					map_e_ntop(AF_INET, &ses->c_src4, buf4, sizeof(buf4));
					map_e_ntop(AF_INET, &ses->dst4, buf5, sizeof(buf5));
					MAP_TRACE("IPv6:%s/%s, IPv4:%s->%s/%s, %d:%u->%u/%u, expire:%d",
					    buf2, buf1, buf3, buf4, buf5, ses->protocol,
					    ntohs(ses->sport), ntohs(ses->c_sport), ntohs(ses->dport), ses->expire);
				}

				n = p->prev;
				list_del(p);
				KFREE(p);
				p = n;
				t->map_e_info->ses_num--;
				MAP_TRACE("t->map_e_info->ses_num=%d\n",t->map_e_info->ses_num);
			}
		}
	}

end_timer:
	// ޤκ
	map_e_session_callout.expires = jiffies + MAP_E_EXPIRE_INTERVAL * HZ;
	add_timer(&map_e_session_callout);

	ULOCK_W_SESSION();
	return;
}

static void
map_e_update_ses_tcp_flag(struct tcphdr *th, struct map_e_session *ses)
{
	LOCK_W_SESSION();
	ses->tcp_flag = 0;
	if (th->syn)
		ses->tcp_flag |= F_TCP_SYN;
	if (th->fin)
		ses->tcp_flag |= F_TCP_FIN;
	if (th->rst)
		ses->tcp_flag |= F_TCP_RST;
	if (th->ack)
		ses->tcp_flag |= F_TCP_ACK;
	ULOCK_W_SESSION();
}

/*
 * å󥿥ॢȤι
 * TCPϥݡȸϳкΤFINե饰ޤ̿ξ祿ޤ20äˤƹ
 */
static void
map_e_update_session_expire(struct sk_buff *skb, struct map_e_session *ses)
{
	if (!skb || !ses) return;

	LOCK_W_SESSION();
	switch (ses->protocol) {
	case IPPROTO_TCP:
		if (ses->tcp_flag & F_TCP_RST) {
			/* 0ˤmap_e_ftp_alg_46
			   ߤꥻå
			   줬뤿1äˤƤ*/
			ses->expire = 1;
		} else if (ses->tcp_flag & F_TCP_FIN) {
			if (ses->expire > MAP_E_EXPIRE_TCP_CLOSE_WAIT)
				ses->expire = MAP_E_EXPIRE_TCP_CLOSE_WAIT;
		} else if (ses->tcp_flag & F_TCP_ACK) {
			struct nf_conn *ct;
			ct = (struct nf_conn*)skb->nfct;
			if((ct != NULL) && (ct->proto.tcp.state == TCP_CONNTRACK_ESTABLISHED)) {
				ses->expire = map_e_expire_tcp;
			} else {
				/* don't update when originate, terminate sequence */
			}
		} else {
			ses->expire = map_e_expire_tcp;
		}
		break;
	case IPPROTO_UDP:
		ses->expire = map_e_expire_udp;
		break;
	case IPPROTO_ICMP:
		ses->expire = MAP_E_EXPIRE_DEFAULT_ICMP;
		break;
	default:
		ses->expire = MAP_E_EXPIRE_DEFAULT;
		break;
 	}
	ULOCK_W_SESSION();
}

/*
 * 16bitѿΣγ׻
 */
static uint16_t
map_e_pow16(uint16_t factorial)
{
	if (factorial == 0) {
		return 1;
	}
	if (factorial >= 16) {
		MAP_TRACE("very overflow?");
		return 0;
	}

	return 1 << factorial;
}

/*
 * Port-set Ϣν.
 */
int
map_e_portset_init(uint16_t ps_id, uint16_t ps_len, struct ip6_tnl_map_e *t)
{
	struct map_e_rule *bmr;
	struct map_e_portset *ps = NULL;
	struct map_e_portset *ps_bak = NULL;
	uint16_t anybits_len;
	uint16_t anybits_max;
	uint16_t port_tmp;
	int ps_num;
	int idx;
	uint16_t portset_min;
	uint16_t R, M;

	MAP_TRACE("start, ps_id=0x%08x, ps_len=0x%08x", ps_id, ps_len);
	bmr = t->map_e_info->bmr;
	if (!bmr) {
		MAP_TRACE("bmr is null.");
		return -1;
	}
	if (MAP_E_PORT_BITLEN < (bmr->psid_offset + ps_len)) {
		MAP_TRACE("psid-offset + psid-len is over 16.");
		return -1;
	}

	/* portsetõ */
	while (!list_empty((struct list_head *)&t->map_e_info->portset_list.list)) {
		ps = list_first_entry(&t->map_e_info->portset_list.list, struct map_e_portset, list);
		list_del((struct list_head *)ps);
		KFREE(ps);
	}

	anybits_len = MAP_E_PORT_BITLEN - (bmr->psid_offset + ps_len);
	anybits_max = map_e_pow16(anybits_len) - 1;

	if (ps_id == 0 && ps_len == 0) {	/* 1:1 mode */
		R = 0;
		M = 16;
		portset_min = 1;
		ps_num = 1;
	} else {
		R = map_e_pow16(ps_len);
		if (R <= 0) {
			MAP_TRACE("R is under zero.");
			return -1;
		}
		M = map_e_pow16(anybits_len);
		if (M <= 0) {
			MAP_TRACE("M is under zero.");
			return -1;
		}
		portset_min = 1; /* draft03 5.1.   A>0 */
		ps_num = ((65536 / M) / R) - 1;
	}

	printk("anybits_len : %u\n", anybits_len);
	printk("anybits_max : %u\n", anybits_max);
	printk("R           : %u\n", R);
	printk("M           : %u\n", M);
	printk("portset_min : %u\n", portset_min);
	printk("portset_max : %u\n", ps_num);

	for (idx = portset_min; idx <= ps_num; idx++) {
		port_tmp = idx << (ps_len + anybits_len);
		port_tmp = port_tmp | (ps_id << anybits_len);

		ps = KMALLOC(sizeof(struct map_e_portset), GFP_KERNEL);
		if (ps == NULL) {
			printk(KERN_ERR "alloc error\n");
			while (!list_empty((struct list_head *)&t->map_e_info->portset_list.list)) {
				ps = list_first_entry(&t->map_e_info->portset_list.list, struct map_e_portset, list);
				list_del((struct list_head *)ps);
				KFREE(ps);
			}
			return -1;
		}
		ps->id = (int)idx;
		ps->port_min = port_tmp;
		ps->port_max = port_tmp + anybits_max;
		if (ps_bak) {
			if (ps_bak->port_max == ps->port_min) {
				printk(KERN_WARNING "duplicate port(%u -> %u)\n",
					ps->port_min, ps->port_min + 1);
				ps->port_min++;
			}
		}

		list_add_tail((struct list_head *)ps, &t->map_e_info->portset_list.list);
		ps_bak = ps;
	}

	/* ޥåԥ󥰥롼˺ǽ̤ȿǤ */
	t->map_e_info->portset_num = ps_num - portset_min + 1;
	t->map_e_info->portset_id = ps_id;
	if (!list_empty((struct list_head *)&t->map_e_info->portset_list.list)) {
		ps = list_first_entry(&t->map_e_info->portset_list.list, struct map_e_portset, list);
		t->map_e_info->current_portset_id = ps->port_min;
	} else {
		t->map_e_info->current_portset_id = 0;
	}
	t->map_e_info->current_portset_index = 0;

	return 0;
}
EXPORT_SYMBOL(map_e_portset_init);

static int
map_e_show_staticnapt(struct ip6_tnl_map_e *t, struct map_e_parm_staticnapt *p)
{
	struct map_e_staticnapt *staticnapt;
	struct list_head *lh;
	char buf[INET6_ADDRSTRLEN + 1];
	char str_protocol[14];

	if (p == NULL) {
		MAP_TRACE("struct map_e_parm_staticnapt is null");
		return -1;
	}
	if (t == NULL) {
		MAP_TRACE("struct ip6_tnl_map_e is null");
		return -1;
	}

	printk("<<static napt setting>>\n");

	LOCK_R_STATICNAPT();
	list_for_each(lh, &t->map_e_info->staticnapt_list.list) {
		staticnapt = list_entry(lh, struct map_e_staticnapt, list);

		switch (staticnapt->protocol) {
		case IPPROTO_TCP:
			snprintf(str_protocol, sizeof(str_protocol), "%s", "TCP");
			break;
		case IPPROTO_UDP:
			snprintf(str_protocol, sizeof(str_protocol), "%s", "UDP");
			break;
		default:
			snprintf(str_protocol, sizeof(str_protocol), "0x%08x", staticnapt->protocol);
		}
		map_e_ntop(AF_INET, &staticnapt->lan_addr, buf, sizeof(buf));
		printk("%s/%u-%u->%u, %s\n", buf, staticnapt->lan_port, staticnapt->lan_toport, staticnapt->sam_port, str_protocol);
	}
	ULOCK_R_STATICNAPT();
	return 0;
}
EXPORT_SYMBOL(map_e_show_staticnapt);

static int
map_e_add_staticnapt(struct ip6_tnl_map_e *t, struct map_e_parm_staticnapt *p)
{
	struct map_e_staticnapt *staticnapt;
	int idx;

	if (p == NULL) {
		MAP_TRACE("struct map_e_parm_staticnapt is null");
		return -1;
	}

	LOCK_R_STATICNAPT();
	if (p->staticnapt_num <= 0) {
		MAP_TRACE("invalid staticnapt_num");
		ULOCK_R_STATICNAPT();
		return -1;
	}
	ULOCK_R_STATICNAPT();

	if (t == NULL) {
		MAP_TRACE("struct ip6_tnl_map_e is null");
		return -1;
	}
	if (t->map_e_info == NULL) {
		MAP_TRACE("map_e_info is null");
		return -1;
	}

	LOCK_W_STATICNAPT();
	for (idx = 0; idx < p->staticnapt_num; idx++) {
		staticnapt = KMALLOC(sizeof(struct map_e_staticnapt), GFP_ATOMIC);
		if (staticnapt == NULL) {
			printk(KERN_ERR "[%s] KMALLOC() error.\n", __func__);
			ULOCK_W_STATICNAPT();
			return -1;
		}
		memcpy(staticnapt, &p->staticnapt[idx], sizeof(struct map_e_staticnapt));
		list_add_tail((struct list_head *)staticnapt, &t->map_e_info->staticnapt_list.list);
		t->map_e_info->staticnapt_num++;
	}
	ULOCK_W_STATICNAPT();

	return 0;
}
EXPORT_SYMBOL(map_e_add_staticnapt);

static int
map_e_del_staticnapt(struct ip6_tnl_map_e *t, struct map_e_parm_staticnapt *p)
{
	struct map_e_staticnapt *staticnapt;
	struct list_head *lh;
	int idx;

	MAP_TRACE("called");

	if (p == NULL) {
		MAP_TRACE("struct map_e_parm_staticnapt is null");
		return -1;
	}
	if (t == NULL) {
		MAP_TRACE("struct ip6_tnl_map_e is null");
		return -1;
	}
	LOCK_W_STATICNAPT();
	list_for_each(lh, &t->map_e_info->staticnapt_list.list) {
		staticnapt = list_entry(lh, struct map_e_staticnapt, list);
		for (idx = 0; idx < p->staticnapt_num; idx++) {
			if (staticnapt->lan_addr.s_addr == p->staticnapt[idx].lan_addr.s_addr &&
			    staticnapt->lan_port == p->staticnapt[idx].lan_port &&
			    staticnapt->lan_toport == p->staticnapt[idx].lan_toport &&
			    staticnapt->sam_port == p->staticnapt[idx].sam_port &&
			    staticnapt->protocol == p->staticnapt[idx].protocol) {
				list_del((struct list_head *)staticnapt);
				KFREE(staticnapt);
				t->map_e_info->staticnapt_num--;
				break;
			}
		}
	}
	ULOCK_W_STATICNAPT();
	return -1;
}
EXPORT_SYMBOL(map_e_del_staticnapt);

static int
map_e_flush_staticnapt(struct ip6_tnl_map_e *t)
{
	struct map_e_staticnapt *staticnapt;

	MAP_TRACE("called");

	if (t == NULL) {
		MAP_TRACE("struct ip6_tnl_map_e is null");
		return -1;
	}

	LOCK_W_STATICNAPT();
	while (!list_empty((struct list_head *)&t->map_e_info->staticnapt_list.list)) {
		staticnapt = list_first_entry(&t->map_e_info->staticnapt_list.list, struct map_e_staticnapt, list);
		list_del((struct list_head *)staticnapt);
		KFREE(staticnapt);
	}
	t->map_e_info->staticnapt_num = 0;
	ULOCK_W_STATICNAPT();
	return 0;
}
EXPORT_SYMBOL(map_e_flush_staticnapt);

/* FTP ALG: PORT,EPRTѥåȤѴԤ */
static int
map_e_ftp_alg_46(struct sk_buff *skb,
				 struct iphdr *ip4h,
				 struct tcphdr *th,
				 struct ip6_tnl_map_e *t,
				 struct map_e_session *ses)
{
	int ftp_off, ftp_data_off;
	int i, cnt;
	int matchlen, buflen;
	int ftp_type = 0;
	uint16_t sport_new = 0;
	uint16_t ftp_port = 0;
	char port_buf[256];
	char write_buf[256];
	char addr[INET_ADDRSTRLEN+1];
	char *port1, *port2, *tmp;

	struct map_e_session ftp_data_ses;

	MAP_TRACE("start");

	if (ip4h->protocol == IPPROTO_TCP && htons(th->dest) == FTP_CTL_PORT) {
		ftp_off = sizeof(struct iphdr) + th->doff*4;

		MAP_TRACE("FTP ALG: oldseq=0x%x(%u)", ntohl(th->seq), ntohl(th->seq));
		th->seq = htonl(ntohl(th->seq) - ses->offset_after);
		MAP_TRACE("FTP ALG: newseq=0x%x(%u)", ntohl(th->seq), ntohl(th->seq));

		if (strncmp(&skb->data[ftp_off],  "PORT", strlen("PORT")) == 0) {
			ftp_type = MAP_E_FTP_PORT;
			ftp_data_off = MAP_E_FTP_ALG_PORT_OFFSET;

			memset(port_buf, 0, sizeof(port_buf));

			/* PORTʹߤIPɥ쥹ݡֹΥǡĹ */
			for (i=ftp_off+ftp_data_off, matchlen=0, cnt=0; i<skb->len; i++, matchlen++) {
				if (skb->data[i] == '\r') {
					break;
				}
				if (skb->data[i] == ',') {
					cnt++;
					if(cnt == 4) {
						memcpy(port_buf, &skb->data[i+1], skb->len - i - 2);
					}
				}
			}
//			MAP_TRACE("matchlen=%d", matchlen);

			/* PORTֹ򥢥Ѵ */
			port1 = port_buf;
			for (i=0; ; i++) {
				if (port_buf[i] == ',') {
					port_buf[i] = '\0';
					port2 = &port_buf[i+1];
					break;
				}
			}
			ftp_port = (uint16_t)map_e_atoi(port1) * 256 + (uint16_t)map_e_atoi(port2);
			MAP_TRACE("ftp_port=%u", ftp_port);
		}
		else if (strncmp(&skb->data[ftp_off],  "EPRT", strlen("EPRT")) == 0) {
			ftp_type = MAP_E_FTP_EPRT;
			ftp_data_off = MAP_E_FTP_ALG_EPRT_OFFSET;

			memset(port_buf, 0, sizeof(port_buf));

			/* EPRTʹߤIPɥ쥹ݡֹΥǡĹ */
			for (i=ftp_off+ftp_data_off, matchlen=0, cnt=0; i<skb->len; i++, matchlen++) {
				if (skb->data[i] == '\r') {
					break;
				}
				if (skb->data[i] == '|') {
					cnt++;
					if(cnt == 3) {
						memcpy(port_buf, &skb->data[i+1], skb->len - i - 3);
					}
				}
			}
			//MAP_TRACE("matchlen=%d", matchlen);
			ftp_port = (uint16_t)map_e_atoi(port_buf);
			MAP_TRACE("ftp_port=%u", ftp_port);
		}

		if (ftp_port != 0) {
			/* FTP-dataѤMAP-EΥݡȤ */
			sport_new = map_e_napt_port_get((struct in_addr *)&ip4h->daddr,
				(struct in_addr *)&ip4h->saddr, htons(20), htons(ftp_port), ip4h->protocol, t);
			if (sport_new == 0) {
				MAP_TRACE("[%s] map_e_napt_port_get() error");
				return -1;
			}

			/* FTP-dataѤΥå󸡺ʤк */
			if (NULL == map_e_search_session46((struct in_addr *)&ip4h->daddr,
							(struct in_addr *)&ip4h->saddr,
							htons(20), htons(ftp_port), ip4h->protocol, t))
			{
				memcpy(&ftp_data_ses.src6, &t->map_e_info->wan_addr, sizeof(struct in6_addr));
				memcpy(&ftp_data_ses.dst6, &t->map_e_info->dmr_addr, sizeof(struct in6_addr));
				ftp_data_ses.src4.s_addr   = ip4h->saddr;
				ftp_data_ses.c_src4.s_addr = t->map_e_info->shared_addr4.s_addr;
				ftp_data_ses.dst4.s_addr   = ip4h->daddr;
				ftp_data_ses.protocol      = ip4h->protocol;
				ftp_data_ses.sport         = htons(ftp_port); 	// PORT,EPRTޥɤǻ
				ftp_data_ses.c_sport       = htons(sport_new);
				ftp_data_ses.dport         = htons(20);			// FTP-data

				/* FTP-dataѤΥå */
				if (map_e_ftp_alg_create_session(&ftp_data_ses, t) != 0) {
					MAP_TRACE("map_e_ftp_alg_create_session() error");
					return -1;
				}
			}

			/* NAPTΥǡ */
			memset(addr, 0, sizeof(addr));
			memset(write_buf, 0, sizeof(write_buf));
			map_e_ntop(AF_INET, &t->map_e_info->shared_addr4.s_addr, addr, sizeof(addr));

			if (ftp_type == MAP_E_FTP_PORT) {
				while ((tmp = strchr(addr, '.')) != NULL) {
					*tmp = ',';
				}
				buflen = snprintf(write_buf, sizeof(write_buf), "%s,%u,%u",
					addr, sport_new>>8, sport_new&0xFF);
				MAP_TRACE("FTP ALG(PORT): %s", write_buf);
			}
			else if (ftp_type == MAP_E_FTP_EPRT) {
				buflen = snprintf(write_buf, sizeof(write_buf), "|1|%s|%d|", addr, sport_new);
				MAP_TRACE("FTP ALG(EPRT): %s", write_buf);
			}

			/* sk_buff */
			if (!map_e_ftp_alg_update_skb(skb, sizeof(struct iphdr), ftp_data_off,
				matchlen, write_buf, buflen, ses)) {
				return -1;
			}
		}
	}

	MAP_TRACE("end");

	return 0;
}

/* ʸѴ */
static int map_e_atoi(const char *s)
{
	int val = 0;

	for (;; s++) {
		switch (*s) {
			case '0'...'9':
			val = 10*val+(*s-'0');
			break;
		default:
			return val;
		}
	}
}

/* FTP ALG: sk_buff */
static int map_e_ftp_alg_update_skb(struct sk_buff *skb,
				   unsigned int iph_off,
				   unsigned int ipadr_pos,
				   unsigned int before_len,
				   const char *after_buf,
					unsigned int after_len,
					struct map_e_session *ses)
{
	struct tcphdr *tcph;

	MAP_TRACE("start");
	MAP_TRACE("iph_off=%d ipadr_pos=%d before_len=%d after_buf=%s after_len=%d",
			iph_off, ipadr_pos, before_len, after_buf, after_len);

	if (!skb_make_writable(skb, skb->len))
		return 0;

	if (after_len > before_len &&
		after_len - before_len > skb_tailroom(skb) &&
		!map_e_enlarge_skb(skb, after_len - before_len))
		return 0;

	SKB_LINEAR_ASSERT(skb);

	tcph = (void *)skb->data + iph_off;

	map_e_ftp_alg_update_skb_data(skb, iph_off + tcph->doff*4,
			ipadr_pos, before_len, after_buf, after_len);

	/* ѴΥǡĹۤʤ硢seq/ackֹѴɬ */
	if (after_len != before_len) {
		if (ses->offset_before == ses->offset_after ||
			before(ses->correction_pos, ntohl(tcph->seq))) {
			ses->correction_pos = ntohl(tcph->seq);
			ses->offset_before = ses->offset_after;
			ses->offset_after += (before_len - after_len);
		}
	}

	MAP_TRACE("end");
	return 1;
}

/* Unusual, but possible case. */
static int map_e_enlarge_skb(struct sk_buff *skb, unsigned int diffsz)
{
	if (skb->len + diffsz > 65535)
		return 0;

	if (pskb_expand_head(skb, 0, diffsz - skb_tailroom(skb), GFP_ATOMIC))
		return 0;

	return 1;
}

/* FTP ALG: sk_buff             */
/* TCPcksum͹napt46ǹԤ */
static void map_e_ftp_alg_update_skb_data(struct sk_buff *skb,
				unsigned int dataoff,
				unsigned int ipadr_pos,
				unsigned int before_len,
				const char *after_buf,
				unsigned int after_len)
{
	unsigned char *data;

	BUG_ON(skb_is_nonlinear(skb));
	data = skb_network_header(skb) + dataoff;

	memmove(data + ipadr_pos + after_len,
		data + ipadr_pos + before_len,
		skb->tail - (skb->network_header + dataoff +
				 ipadr_pos + before_len));

	memcpy(data + ipadr_pos, after_buf, after_len);

	if (after_len > before_len) {
		skb_put(skb, after_len - before_len);
	} else {
		__skb_trim(skb, skb->len + after_len - before_len);
	}

	/* IPإå */
	ip_hdr(skb)->tot_len = htons(skb->len);
	ip_send_check(ip_hdr(skb));
}

/*
 * FTP ALG: FTP-dataѤsession
 */
static int
map_e_ftp_alg_create_session(struct map_e_session *ftp_data_ses, struct ip6_tnl_map_e *t)
{
	struct map_e_session *new_ses = NULL;
	uint32_t hash_index;
	char buf1[INET6_ADDRSTRLEN + 1], buf2[INET6_ADDRSTRLEN + 1], buf3[INET6_ADDRSTRLEN + 1], buf4[INET6_ADDRSTRLEN + 1], buf5[INET6_ADDRSTRLEN + 1];

	MAP_TRACE("start");

	LOCK_R_SESSION();
	if (t->map_e_info->ses_num >= t->parms.session_max) {
		MAP_TRACE("The number of sessions reached the maximum.");
		ULOCK_R_SESSION();
		goto tx_err_link_failure;
	}
	ULOCK_R_SESSION();

	new_ses = KMALLOC(sizeof(struct map_e_session), GFP_ATOMIC);
	if (new_ses == NULL) {
		printk(KERN_ERR "[%s] KMALLOC() error.\n", __func__);
		goto tx_err_link_failure;
	}
	memset(new_ses, 0x00, sizeof(struct map_e_session));

	memcpy(&new_ses->src6, &ftp_data_ses->src6, sizeof(struct in6_addr));
	memcpy(&new_ses->dst6, &ftp_data_ses->dst6, sizeof(struct in6_addr));
	new_ses->src4.s_addr   = ftp_data_ses->src4.s_addr;
	new_ses->c_src4.s_addr = ftp_data_ses->c_src4.s_addr;
	new_ses->dst4.s_addr   = ftp_data_ses->dst4.s_addr;
	new_ses->protocol      = ftp_data_ses->protocol;
	new_ses->sport         = ftp_data_ses->sport;
	new_ses->c_sport       = ftp_data_ses->c_sport;
	new_ses->dport         = ftp_data_ses->dport;
	new_ses->expire        = map_e_expire_tcp;

	if (g_debug) {
		map_e_ntop(AF_INET6, &new_ses->src6, buf1, sizeof(buf1));
		map_e_ntop(AF_INET6, &new_ses->dst6, buf2, sizeof(buf2));
		map_e_ntop(AF_INET, &new_ses->src4, buf3, sizeof(buf3));
		map_e_ntop(AF_INET, &new_ses->c_src4, buf4, sizeof(buf4));
		map_e_ntop(AF_INET, &new_ses->dst4, buf5, sizeof(buf5));
		MAP_TRACE("<<new session>>");
		MAP_TRACE("  .src6    :%s", buf1);
		MAP_TRACE("  .dst6    :%s", buf2);
		MAP_TRACE("  .src4    :%s", buf3);
		MAP_TRACE("  .c_src4  :%s", buf4);
		MAP_TRACE("  .dst4    :%s", buf5);
		MAP_TRACE("  .protocol:0x%04x", new_ses->protocol);
		MAP_TRACE("  .expire  :%d", new_ses->expire);
		MAP_TRACE("  .sport   :%u", ntohs(new_ses->sport));
		MAP_TRACE("  .c_sport :%u", ntohs(new_ses->c_sport));
		MAP_TRACE("  .dport   :%u", ntohs(new_ses->dport));
	}

	/* åϿ */
	hash_index = map_e_get_hash_index(&new_ses->dst4, new_ses->dport, new_ses->protocol);
	MAP_TRACE("hash_index = %u", hash_index);
	LOCK_W_SESSION();
	list_add_tail((struct list_head *)new_ses, &t->map_e_info->ses_list[hash_index].list);
	t->map_e_info->ses_num++;
	ULOCK_W_SESSION();

	MAP_TRACE("end");
	return 0;

tx_err_link_failure:
	MAP_TRACE("end");
	return -1;
}

/* MSS auto setting */
static void _map_e_tcpmss(struct tcphdr *th, int reverse , struct sk_buff *skb)
{
	int optlen, i;
	u_int8_t *op;
	u_int16_t mss;
	u_int16_t d_mtu;

	if (th->syn) {
		optlen = th->doff*4 - sizeof(struct tcphdr);
		if (optlen != 0) {
			op = (u_int8_t *)(th + 1);
			for (i = 0; i < optlen; ) {
				if (op[i] == TCPOPT_MSS && (optlen - i) >= TCPOLEN_MSS && op[i+1] == TCPOLEN_MSS) {

					mss = (op[i+2] << 8) | op[i+3];

					MAP_TRACE("old mss: %u", mss);

					if (skb != NULL) {
						/* map1 ǥХMTUĥƤ١*/
						/* "Path MTU"ǤϤʤ"dst MTU"Τ߻       */

						d_mtu = dst_mtu(skb_dst(skb));
						MAP_TRACE("dst mtu: %u", d_mtu);

						if ( d_mtu == 0) {
							MAP_TRACE("d_mtu = 0. Skip mss adjustment");
							return ;
						}

						/* dst mtu(1500) - ipv6hdr(40) - iphdr(20) - tcphdr(20) */
						d_mtu = d_mtu - sizeof(struct ipv6hdr) - sizeof(struct tcphdr) - sizeof(struct iphdr);

						MAP_TRACE("dst mtu - ipv6hdr - iphdr - tcphdr : %u", d_mtu);

						if ( d_mtu >= mss ) {
							MAP_TRACE("d_mtu >= mss. Skip mss adjustment");
							return ;
						} else {
							mss = d_mtu;
						}
					}

					if (reverse)
						mss += sizeof(struct ipv6hdr);

					MAP_TRACE("new mss: %u", mss);

					op[i+2] = (mss & 0xff00) >> 8;
					op[i+3] = mss & 0x00ff;

					break;
				}

				if (op[i] < 2)
					i++;
				else
					i += op[i+1] ? : 1;
			}
		}
	}
}

/* MSS auto setting */
static void map_e_tcpmss(struct tcphdr *th ,struct sk_buff *skb)
{
	_map_e_tcpmss(th, 0 ,skb);
}

/* MSS auto setting when ICMP Error */
static void map_e_tcpmss_reverse(struct tcphdr *th)
{
	_map_e_tcpmss(th, 1 ,NULL);
}

