/*
 * Dual licensed under the ISC license and the GPL2.
 */


#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#include "dom_sock.h"




static struct sockaddr_un	zm_peer_addr;


#define	NUM_BACKLOG	(5)


/*
 * path :	socket's path
 * sel :	specifiy CLIENT(active) or SERVER(passive)
 *
 * result:	>= 0	file descriptor of opened socket
 *		< 0	(error code) * (-1)
 */
int  dom_sock_open( const char*  path, int  sel )
{
	struct stat	statbuf;
	int		ctrl_sock = -ENODEV;
	int		res;

	if( sel == DOM_SOCK_CLI ){
		/* check whether socket file exists or not */
		res = stat( path, &statbuf );
		if( res ){
			ctrl_sock = -errno;
			goto l_exit;
		}
	}

	if( (ctrl_sock = socket(PF_UNIX, SOCK_STREAM, 0)) < 0 ) {
		ctrl_sock = -errno;
		goto l_exit;
	}

	memset(&zm_peer_addr, 0, sizeof(zm_peer_addr));
	zm_peer_addr.sun_family = AF_UNIX;
	strncpy(zm_peer_addr.sun_path, path, sizeof(zm_peer_addr.sun_path));

	if( sel == DOM_SOCK_SERV ){
		if( bind(ctrl_sock, (struct sockaddr*)&zm_peer_addr, sizeof(zm_peer_addr)) < 0 ) {
			res = -errno;
			close( ctrl_sock );
			ctrl_sock = res;
			goto l_exit;
		}

		if( listen( ctrl_sock, NUM_BACKLOG ) < 0 ){
			res = -errno;
			close( ctrl_sock );
			ctrl_sock = res;
			goto l_exit;
		}
	}

  l_exit:;
	return ctrl_sock;
}


/*
 */
int  dom_sock_close( int  ctrl_sock )
{
	close( ctrl_sock );
	return 0;
}


/*
 * ctrl_sock :	listening master socket
 */
int  dom_sock_accept( int  ctrl_sock )
{
	int	conn_sock;

	conn_sock = accept( ctrl_sock, NULL, NULL );
	if( conn_sock < 0 ){
		conn_sock = -errno;
	}
	return conn_sock;
}

/*
 */
static dom_sock_set_timeout( int  sock_fd, int  type, unsigned int  tmout_msec )
{
	struct timeval	tv;
	int		res = 0;

	if( tmout_msec ){
		tv.tv_sec = tmout_msec / 1000;
		tv.tv_usec = (tmout_msec - tv.tv_sec * 1000) * 1000;
	} else{
		tv.tv_sec = 0;
		tv.tv_usec = 0;
	}

	if( (res = setsockopt( sock_fd, SOL_SOCKET, type, &tv, sizeof(tv) )) != 0 ){
		res = errno;
	}

	return res;
}

/*
 */
int  dom_sock_connect( int  ctrl_sock, unsigned int  tmout_msec )
{
	int	res;

	if( (res = dom_sock_set_timeout( ctrl_sock, SO_SNDTIMEO, tmout_msec )) ||
			(res = dom_sock_set_timeout( ctrl_sock, SO_RCVTIMEO, tmout_msec )) ){
		res = errno;
		goto l_exit;
	}

	if ( (res = connect(ctrl_sock, (struct sockaddr*)&zm_peer_addr, sizeof(zm_peer_addr))) < 0 ) {
		res = errno;
		goto l_exit;
	}

  l_exit:;
	dom_sock_set_timeout( ctrl_sock, SO_SNDTIMEO, DOM_SOCK_NO_TMOUT );
	dom_sock_set_timeout( ctrl_sock, SO_RCVTIMEO, DOM_SOCK_NO_TMOUT );

	return res;
}


int  dom_sock_recv_fd_msg( int  sock_fd, int*  p_remote_fd, void*  message, size_t  message_len, unsigned int  tmout_msec )
{
	struct msghdr	msg;
	struct iovec	iov;
	char		cmsgbuf[CMSG_SPACE(sizeof(int))];
	struct cmsghdr*	cmsg;
	int		len;
	int		res = 0;

	if( (res = dom_sock_set_timeout( sock_fd, SO_SNDTIMEO, tmout_msec )) ||
			(res = dom_sock_set_timeout( sock_fd, SO_RCVTIMEO, tmout_msec )) ){
		res = errno;
		goto l_exit;
	}

	if( p_remote_fd ){
		iov.iov_base	= message;
		iov.iov_len	= message_len;

		msg.msg_name		= NULL;
		msg.msg_namelen		= 0;
		msg.msg_iov		= &iov;
		msg.msg_iovlen		= 1;
		msg.msg_control		= cmsgbuf;
		msg.msg_controllen	= sizeof(cmsgbuf);
		msg.msg_flags		= MSG_WAITALL;

		if ( (len = recvmsg(sock_fd, &msg, 0)) < 0 ) {
			res = errno;
			goto l_exit;
		}

		cmsg = (struct cmsghdr*)cmsgbuf;

		*p_remote_fd = *((int *)CMSG_DATA(cmsg));
	} else{
		if( (len = recv( sock_fd, message, message_len, MSG_WAITALL )) < 0 ){
			res = errno;
			goto l_exit;
		}
	}
	if( len != message_len ){
		res = ENODEV;
		goto l_exit;
	}

  l_exit:;
	dom_sock_set_timeout( sock_fd, SO_SNDTIMEO, DOM_SOCK_NO_TMOUT );
	dom_sock_set_timeout( sock_fd, SO_RCVTIMEO, DOM_SOCK_NO_TMOUT );

	return res;
}


int  dom_sock_send_fd_msg( int  sock_fd, int  fd_target, void*  message, size_t  message_len, unsigned int  tmout_msec )
{
	struct msghdr	msg;
	struct iovec	iov;
	char		cmsgbuf[CMSG_SPACE(sizeof(int))];
	struct cmsghdr*	cmsg;
	int		len;
	int		res = 0;

	if( (res = dom_sock_set_timeout( sock_fd, SO_SNDTIMEO, tmout_msec )) ||
			(res = dom_sock_set_timeout( sock_fd, SO_RCVTIMEO, tmout_msec )) ){
		res = errno;
		goto l_exit;
	}

	if( fd_target >= 0 ){
		iov.iov_base	= message;
		iov.iov_len	= message_len;

		cmsg			  = (struct cmsghdr*)cmsgbuf;
		cmsg->cmsg_len		  = CMSG_LEN(sizeof(int));
		cmsg->cmsg_level	  = SOL_SOCKET;
		cmsg->cmsg_type		  = SCM_RIGHTS;
		*((int *)CMSG_DATA(cmsg)) = fd_target;

		memset(&msg, 0, sizeof(msg));
		msg.msg_name		= NULL;
		msg.msg_namelen		= 0;
		msg.msg_iov		= &iov;
		msg.msg_iovlen		= 1;
		msg.msg_control		= cmsgbuf;
		msg.msg_controllen	= sizeof(cmsgbuf);
		msg.msg_flags		= 0;

		if( (len = sendmsg(sock_fd, &msg, 0)) < 0 ){
			res = errno;
			goto l_exit;
		}
	} else{
		if( (len = send( sock_fd, message, message_len, 0 )) < 0 ){
			res = errno;
			goto l_exit;
		}
	}
	if( len != message_len ){
		res = ENODEV;
		goto l_exit;
	}

  l_exit:;
	dom_sock_set_timeout( sock_fd, SO_SNDTIMEO, DOM_SOCK_NO_TMOUT );
	dom_sock_set_timeout( sock_fd, SO_RCVTIMEO, DOM_SOCK_NO_TMOUT );

	return res;
}


/*
 * EOF
 */
