

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/socket.h>

#include "pppd.h"
#include "fsm.h"
#include "lcp.h"
#include "ccp.h"
#include "ipcp.h"

#include "ctl_comm.h"




/********************************************************************/
/********************************************************************/
static char*	g_ipc_command	= NULL;

static int	g_pppox_pid	= -1;
static int	g_ipc_sock	= -1;

static int	g_pppoe_sock	= -1;

static comm_msg_t*	g_ptxbuf;
static comm_msg_t*	g_prxbuf;

static int	g_check_opt_done;


static int	ipc_debug;


#define	DBGOUT	!ipc_debug ? (void)0 : printf




/********************************************************************/
/********************************************************************/

#define	TMOUT_MSEC_IPC_CONNECT		(1000)

#define	TMOUT_MSEC_CMD_TX		(1000)
#define	TMOUT_MSEC_CMD_RX_PING		(1000)
#define	TMOUT_MSEC_CMD_RX_CONNECT	(CTL_COMM_NO_TMOUT)	/* ??? appropriate value ??? */
#define	TMOUT_MSEC_CMD_RX_DISCONNECT	(5000)
#define	TMOUT_MSEC_CMD_RX_CHK_OPTIONS	(1000)


/*
 */
static int  pppox_ipc_stop_ipc( void )
{
	if( g_ptxbuf ){
		free( g_ptxbuf );
		g_ptxbuf = NULL;
	}
	if( g_prxbuf ){
		free( g_prxbuf );
		g_prxbuf = NULL;
	}

	ctl_comm_mst_exit();
	g_ipc_sock = -1;

	/* terminate client here */
	if( g_pppox_pid >= 0 ){
		kill( g_pppox_pid, SIGTERM );
		g_pppox_pid = -1;
	}
	g_check_opt_done = 0;

	return 0;
}


/*
 */
static void  pppox_ipc_child_done( void*  arg )
{
	g_pppox_pid = -1;
	g_check_opt_done = 0;

	ctl_comm_mst_exit();
	g_ipc_sock = -1;

	/* also close PPPoE socket mapped from client */
	if( g_pppoe_sock >= 0 ){
		close( g_pppoe_sock );
		g_pppoe_sock = -1;
	}

	DBGOUT("%s:\n",__FUNCTION__);
}

/*
 */
static int  pppox_ipc_start_ipc( void )
{
	int	pid;
	char*	argv[32];
	char*	buf = NULL;
	char*	bp_e;
	char*	bp_s;
	char	c;
	int	ix;
	int	res = 0;

	if( g_ipc_command == NULL ){
		fprintf( stderr, "IPC command is required.\n" );
		res = EINVAL;
		goto l_exit;
	}

	/********************************************/
	/*
	 * execute PPPoX client program
	 */
	if( g_pppox_pid < 0 ){
		if( (buf = strdup( g_ipc_command )) == NULL ){
			res = errno;
			goto l_exit;
		}

		bp_s = buf;
		bp_e = buf;
		ix = 0;
		while( (c = *bp_e) != '\0' ){
			if( c == ' ' ){
				*bp_e = '\0';
				argv[ix++] = bp_s;
				bp_e++;
				bp_s = bp_e;
				if( ix >= (32 - 2) ){
					res = EINVAL;
					goto l_exit;
				}
			} else{
				bp_e++;
			}
		}
		argv[ix++] = bp_s;
		argv[ix] = NULL;

		pid = run_program( argv[0], argv, 1, pppox_ipc_child_done, NULL, 0 );
		if( pid < 0 ){
			fprintf( stderr, "IPC command executing failed.\n" );
			res = ENOENT;
			goto l_exit;
		}
		free( buf );
		buf = NULL;

		g_pppox_pid = pid;
	}

	/********************************************/
	if( g_ipc_sock < 0 ){
		const int	RETRY_WAIT_USEC = (1000 * 1000);
		int		retry	= 10;

		while( --retry >= 0 ){
			/*
			 * open IPC channel
			 */
			res = ctl_comm_mst_init( g_pppox_pid, &g_ipc_sock );
			if( res ){
				g_ipc_sock = -1;
				DBGOUT("cannot open IPC channel: res = %d\n", res );
				goto l_loop_tail;
			}

			res = ctl_comm_mst_connect( g_ipc_sock, TMOUT_MSEC_IPC_CONNECT );
			if( res ){
				DBGOUT("ctl_comm_mst_connect() res = %d\n", res );
				goto l_loop_tail;
			}

		    l_loop_tail:;
			if( res == 0 ){
				break;
			} else{
				if( g_ipc_sock >= 0 ){
					ctl_comm_mst_exit();
					g_ipc_sock = -1;
				}
				usleep( RETRY_WAIT_USEC );
			}
		}

		if( retry < 0 ){
			DBGOUT("cannot connect IPC channel\n");
			if( g_ipc_sock >= 0 ){
				ctl_comm_mst_exit();
			}
			g_ipc_sock = -1;
			res = ENODEV;
			goto l_exit;
		}
	}

	if( g_ptxbuf == NULL ){
		if( (g_ptxbuf = malloc( sizeof(*g_ptxbuf) )) == NULL ){
			res = errno;
			goto l_exit;
		}
	}
	if( g_prxbuf == NULL ){
		if( (g_prxbuf = malloc( sizeof(*g_prxbuf) )) == NULL ){
			res = errno;
			goto l_exit;
		}
	}

  l_exit:;
	if( res ){
		if( (g_ipc_sock >= 0) || (g_pppox_pid >= 0) ){
			pppox_ipc_stop_ipc();
		}
	}
	if( buf != NULL ){
		free( buf );
	}

	return res;
}

static option_t	pppox_ipc_options[]	= {
	{	"ipc_debug",	o_int,		&ipc_debug,
		"increase debug level for IPC I/F",
		OPT_INC | OPT_INC | OPT_NOARG | 1 },

	/* this is terminator */
	{ NULL },
};

/*
 */
static int  pppox_ipc_ping( void )
{
	comm_msg_t*	txbuf;
	comm_msg_t*	rxbuf;
	int		fd;
	int		randval;
	int		res;

	res = pppox_ipc_start_ipc();
	if( res ){
		goto l_exit;
	}

	/* must refer after pppox_ipc_start_ipc() */
	txbuf = g_ptxbuf;
	rxbuf = g_prxbuf;

	txbuf->cmd = COMM_CMD_IPC_PING;
	randval = rand();
	txbuf->u.ping.val = randval;

	res = ctl_comm_send_msg( g_ipc_sock, CTL_COMM_NO_FD, txbuf, TMOUT_MSEC_CMD_TX );
	if( res ){
		goto l_exit;
	}

	res = ctl_comm_recv_msg( g_ipc_sock, &fd, rxbuf, TMOUT_MSEC_CMD_RX_PING );
	if( res ){
		goto l_exit;
	}

	if( (rxbuf->cmd == COMM_CMD_IPC_PING) && (rxbuf->u.ping.val == randval) ){
		/* OK */
	} else{
		res = ENODEV;
	}

  l_exit:;
	return res;
}


/*
 */
static void  pppox_ipc_process_extra_options( void )
{
	/* NOP */
}


/*
 */
static int  pppox_ipc_do_check_options( void )
{
	comm_msg_t*	txbuf;
	comm_msg_t*	rxbuf;
	int		tmp;
	int		res;

	res = pppox_ipc_ping();
	if( res ){
		goto l_exit;
	}

	if( g_check_opt_done ){
		goto l_exit;
	}

	/* must refer after pppox_ipc_ping() */
	txbuf = g_ptxbuf;
	rxbuf = g_prxbuf;

	/**************************************/
	/*
	 * set current values
	 */
	txbuf->u.chkopt.lcp.neg_accompression = 
			lcp_wantoptions[0].neg_accompression;

	txbuf->u.chkopt.lcp.neg_asyncmap =
			lcp_wantoptions[0].neg_asyncmap;

	txbuf->u.chkopt.lcp.neg_pcompression =
			lcp_wantoptions[0].neg_pcompression;

	txbuf->u.chkopt.lcp.mru =
			lcp_wantoptions[0].mru;

	txbuf->u.chkopt.ccp.deflate =
			ccp_wantoptions[0].deflate;

	txbuf->u.chkopt.ccp.bsd_compress =
			ccp_wantoptions[0].bsd_compress;

	txbuf->u.chkopt.ipcp.neg_vj =
			ipcp_wantoptions[0].neg_vj;

	/**************************************/

	txbuf->cmd = COMM_CMD_CHK_OPTIONS;

	res = ctl_comm_send_msg( g_ipc_sock, CTL_COMM_NO_FD, txbuf, TMOUT_MSEC_CMD_TX );
	if( res ){
		DBGOUT("%s: send_msg : res = %d\n", __FUNCTION__, res );
		goto l_exit;
	}

	res = ctl_comm_recv_msg( g_ipc_sock, NULL, rxbuf, TMOUT_MSEC_CMD_RX_CHK_OPTIONS );
	if( res ){
		DBGOUT("%s: recv_msg : res = %d\n", __FUNCTION__, res );
		goto l_exit;
	}

	if( rxbuf->cmd != COMM_CMD_CHK_OPTIONS ){
		res = ENODEV;
		goto l_exit;
	}

	if( rxbuf->err != 0 ){
		/* maybe not supported. skip updating values */
		goto l_done;
	}

	/**************************************/
	/*
	 * update with received values
	 */
	lcp_allowoptions[0].neg_accompression =
	lcp_wantoptions[0].neg_accompression =
			rxbuf->u.chkopt.lcp.neg_accompression;

	lcp_allowoptions[0].neg_asyncmap =
	lcp_wantoptions[0].neg_asyncmap =
			rxbuf->u.chkopt.lcp.neg_asyncmap;

	lcp_allowoptions[0].neg_pcompression =
	lcp_wantoptions[0].neg_pcompression =
			rxbuf->u.chkopt.lcp.neg_pcompression;

	tmp = rxbuf->u.chkopt.lcp.mru;
	if( lcp_allowoptions[0].mru > tmp ){
		lcp_allowoptions[0].mru = tmp;
		DBGOUT("%s:new MRU = %d\n", __FUNCTION__, tmp );
	}
	if( lcp_wantoptions[0].mru > tmp ){
		lcp_wantoptions[0].mru = tmp;
		DBGOUT("%s:new MRU = %d\n", __FUNCTION__, tmp );
	}

	ccp_allowoptions[0].deflate =
	ccp_wantoptions[0].deflate =
			rxbuf->u.chkopt.ccp.deflate;

	ccp_allowoptions[0].bsd_compress =
	ccp_wantoptions[0].bsd_compress =
			rxbuf->u.chkopt.ccp.bsd_compress;

	ipcp_allowoptions[0].neg_vj =
	ipcp_wantoptions[0].neg_vj =
			rxbuf->u.chkopt.ipcp.neg_vj;
	
	/**************************************/

  l_done:;
	g_check_opt_done = 1;

  l_exit:;
	return res;
}


/*
 */
static void  pppox_ipc_check_options( void )
{
}


/*
 */
static int  pppox_ipc_connect( void )
{
	comm_msg_t*	txbuf;
	comm_msg_t*	rxbuf;
	int		fd;
	int		res;

	res = pppox_ipc_do_check_options();
	if( res ){
		goto l_exit;
	}

	/* must refer after pppox_ipc_ping() */
	txbuf = g_ptxbuf;
	rxbuf = g_prxbuf;

	txbuf->cmd = COMM_CMD_CONNECT;

	res = ctl_comm_send_msg( g_ipc_sock, CTL_COMM_NO_FD, txbuf, TMOUT_MSEC_CMD_TX );
	if( res ){
		DBGOUT("%s: send_msg : res = %d\n", __FUNCTION__, res );
		goto l_exit;
	}

	res = ctl_comm_recv_msg( g_ipc_sock, &fd, rxbuf, TMOUT_MSEC_CMD_RX_CONNECT );
	if( res ){
		DBGOUT("%s: recv_msg : res = %d\n", __FUNCTION__, res );
		goto l_exit;
	}

	if( rxbuf->cmd != COMM_CMD_CONNECT ){
		close( fd );
		res = ENODEV;
		goto l_exit;
	}

	memset( remote_number, 0, sizeof(remote_number) );
	strlcpy( remote_number, rxbuf->u.conn.remote_number, sizeof(remote_number) - 1 );

	memset( ppp_devnam, 0, sizeof(ppp_devnam) );
	strlcpy( ppp_devnam, rxbuf->u.conn.ppp_devname, sizeof(ppp_devnam) - 1 );

	script_setenv("MACREMOTE", remote_number, 0);

	DBGOUT("%s: command result = %d\n", __FUNCTION__, rxbuf->err );
	DBGOUT("%s: received fd = %d\n", __FUNCTION__, fd );
	DBGOUT("%s: remote number = %s\n", __FUNCTION__, remote_number );

	g_pppoe_sock = fd;

  l_exit:;
	return ((res == 0) ? fd : -1);
}


/*
 */
static void  pppox_ipc_disconnect( void )
{
	comm_msg_t*	txbuf;
	comm_msg_t*	rxbuf;
	int		res = 0;

	if( g_pppoe_sock < 0 ){
		goto l_exit;
	}

	res = pppox_ipc_do_check_options();
	if( res ){
		goto l_exit;
	}

	/* must refer after pppox_ipc_ping() */
	txbuf = g_ptxbuf;
	rxbuf = g_prxbuf;

	/* socket handle of this side */
	close( g_pppoe_sock );
	g_pppoe_sock = -1;

	txbuf->cmd = COMM_CMD_DISCONNECT;

	res = ctl_comm_send_msg( g_ipc_sock, CTL_COMM_NO_FD, txbuf, TMOUT_MSEC_CMD_TX );
	if( res ){
		DBGOUT("%s: send_msg : res = %d\n", __FUNCTION__, res );
		goto l_exit;
	}

	res = ctl_comm_recv_msg( g_ipc_sock, NULL, rxbuf, TMOUT_MSEC_CMD_RX_DISCONNECT );
	if( res ){
		DBGOUT("%s: recv_msg : res = %d\n", __FUNCTION__, res );
		goto l_exit;
	}

	DBGOUT("%s: command result = %d\n", __FUNCTION__, rxbuf->err );

  l_exit:;
	return;
}


/*
 */
static void  pppox_ipc_cleanup( void )
{
}


/*
 */
static void  pppox_ipc_close( void )
{
}


/********************************************************************/
/********************************************************************/

/*
 */
struct channel	pppox_ipc_channel = {
	pppox_ipc_options,			/* options */
	pppox_ipc_process_extra_options,	/* process_extra_options */
	pppox_ipc_check_options,		/* check_options */
	pppox_ipc_connect,			/* connect */
	pppox_ipc_disconnect,			/* disconnect */
	generic_establish_ppp,			/* establish_ppp */	
	generic_disestablish_ppp,		/* disestablish_ppp */	
	NULL,					/* send_config */
	NULL,					/* recv_config */
	pppox_ipc_cleanup,			/* cleanup */
	pppox_ipc_close,			/* close */
};


/*
 */
void  pppox_ipc_init( void )
{
}


int  pppox_ipc_parser( char**  argv )
{
	int	res = 1;

	DBGOUT("%s: command line = \"%s\"\n", __FUNCTION__, argv[0] );
	g_ipc_command = strdup( argv[0] );

	the_channel = &pppox_ipc_channel;

	return res;
}


/*
 * EOF
 */
