/* NTP client
 *
 * Copyright (C) 1997-2015  Larry Doolittle <larry@doolittle.boa.org>
 * Copyright (C) 2010-2017  Joachim Nilsson <troglobit@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (Version 2,
 * June 1991) as published by the Free Software Foundation.  At the
 * time of writing, that license was published by the FSF with the URL
 * http://www.gnu.org/copyleft/gpl.html, and is incorporated herein by
 * reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * Possible future improvements:
 *    - Support leap second processing
 *    - Support multiple (interleaved) servers
 *
 * If the compile gives you any flak, check below in the section
 * labelled "XXX fixme - non-automatic build configuration".
 */

#define _POSIX_C_SOURCE 199309

#ifdef USE_OBSOLETE_GETTIMEOFDAY
#define _BSD_SOURCE
#endif

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>		/* getaddrinfo -> gethostbyname */
#include <arpa/inet.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#ifdef PRECISION_SIOCGSTAMP
#include <sys/ioctl.h>
#endif
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
//Brad add sync system time for dhcpd 2008/01/28 
#include <sys/stat.h>
//-------------------------
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */
#ifdef USE_OBSOLETE_GETTIMEOFDAY
#include <sys/time.h>
#endif

#include "ntpclient.h"

/* XXX fixme - non-automatic build configuration */
#ifdef __linux__
#include <sys/utsname.h>
#include <sys/time.h>
#include <sys/timex.h>
#else
extern struct hostent *gethostbyname(const char *name);
extern int h_errno;
char __hstrerror_buf[10];

#define hstrerror(errnum) \
	snprintf(__hstrerror_buf, sizeof(__hstrerror_buf), "Error %d", errnum)
#endif

/* Default to the RFC-4330 specified value */
#ifndef MIN_INTERVAL
#define MIN_INTERVAL 15
#endif

#ifdef ENABLE_DEBUG
#define DEBUG_OPTION "d"
int debug=0;
#else
#define DEBUG_OPTION
#endif

#ifdef ENABLE_REPLAY
#define REPLAY_OPTION   "r"
#else
#define REPLAY_OPTION
#endif

/* Some NTP Client custom default parameter values */
#define SYNCHRONIZATION_INTERVAL        86400 /* Time between 2 NTP Client Requests (default is 24hrs) */
#define NO_RESPONSE_TIME                60 /* NTP Client Request Timeout (default is 60 s) */
#define RETRY_INTERVAL                  "60,120,240,480,960,.," /* Interval between 2 entire NTP Message Cycles in case of failure (default is '1min-2min-4-min-8-min-16min-16 after that) */

/* Harmless way to get around the 'implicit declaration of function daemon()' compiler warning */
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
//void daemon(void);
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */
/* Flag to denote whether the Client should run as Daemon (if set to 1); 0 by default */
int run_as_daemon=0;
/* Number of Probes sent and responded to */
int probes_sent=0; 
#define JAN_1970        0x83aa7e80	/* 2208988800 1970 - 1900 in seconds */
#define NTP_PORT (123)

/* How to multiply by 4294.967296 quickly (and not quite exactly)
 * without using floating point or greater than 32-bit integers.
 * If you want to fix the last 12 microseconds of error, add in
 * (2911*(x))>>28)
 */
#define NTPFRAC(x) ( 4294*(x) + ( (1981*(x))>>11 ) )

/* The reverse of the above, needed if we want to set our microsecond
 * clock (via clock_settime) based on the incoming time in NTP format.
 * Basically exact.
 */
#define USEC(x) ( ( (x) >> 12 ) - 759 * ( ( ( (x) >> 10 ) + 32768 ) >> 16 ) )

/* Converts NTP delay and dispersion, apparently in seconds scaled
 * by 65536, to microseconds.  RFC-1305 states this time is in seconds,
 * doesn't mention the scaling.
 * Should somehow be the same as 1000000 * x / 65536
 */
#define sec2u(x) ( (x) * 15.2587890625 )

struct ntptime {
	unsigned int coarse;
	unsigned int fine;
};

struct ntp_control {
	uint32_t time_of_send[2];
	int usermode;
	int live;
	int set_clock;		/* non-zero presumably needs root privs */
	int probe_count;
	int synchronization_interval;
	int goodness;
	int cross_check;

	uint16_t udp_port[2];	/* [0]:primary, [1]:secondary */
	char *hostname[2];		/* must be set *//* [0]:primary, [1]:secondary */
	char serv_addr[4];
	int no_response_time;
	char retry_interval[1025];
};

int log_level = LOG_NOTICE;
const char *prognm = PACKAGE_NAME;
static int sighup = 0;
static int sigterm = 0;

static int ip_version = 0;

struct {
	int enable;
	char *hostname[2];
} v4fallback = {
	.enable = 0,
	.hostname[0] = "ntp.nict.jp", /* primary */
	.hostname[1] = "ntp.jst.mfeed.ad.jp", /* secondary */
};

extern char *optarg;		/* according to man 2 getopt */

/* prototypes for some local routines */
static void send_packet(int usd, uint32_t time_sent[2]);
static int rfc1305print(uint32_t *data, struct ntptime *arrival, struct ntp_control *ntpc, int server_index, int *error);
static int getaddrbyname(char *host, struct sockaddr_storage *ss);

void logit(int severity, int syserr, const char *format, ...)
{
	va_list ap;
	char buf[200];

	if (log_level < severity)
		return;

	va_start(ap, format);
	vsnprintf(buf, sizeof(buf), format, ap);
	va_end(ap);

	if (debug) {
		if (syserr) {
			syslog(severity, "%s: %s", buf, strerror(syserr));
		} else {
			syslog(severity, "%s", buf);
		}

		if (!run_as_daemon) {
			if (severity == LOG_WARNING) {
				fputs("Warning - ", stderr);
			} else if (severity == LOG_ERR) {
				fputs("ERROR - ", stderr);
			}

			if (syserr) {
				fprintf(stderr, "%s: %s\n", buf, strerror(errno));
			} else {
				fprintf(stderr, "%s\n", buf);
			}
		}
	}
}

/* OS dependent routine to get the current value of clock frequency */
static int get_current_freq(void)
{
#ifdef __linux__
	struct timex txc;

	txc.modes = 0;
	if (adjtimex(&txc) < 0) {
		PRINT(LOG_ERR, errno, "Failed adjtimex(GET)");
		exit(EXIT_FAILURE);
	}
	return txc.freq;
#else
	return 0;
#endif
}

/* OS dependent routine to set a new value of clock frequency */
static int set_freq(int new_freq)
{
#ifdef __linux__
	struct timex txc;

	txc.modes = ADJ_FREQUENCY;
	txc.freq = new_freq;
	if (adjtimex(&txc) < 0) {
		PRINT(LOG_ERR, errno, "Failed adjtimex(SET)");
		exit(EXIT_FAILURE);
	}
	return txc.freq;
#else
	return 0;
#endif
}

static void set_time(struct ntptime *new)
{
#ifndef USE_OBSOLETE_GETTIMEOFDAY
	/* POSIX 1003.1-2001 way to set the system clock */
	struct timespec tv_set;

	/* it would be even better to subtract half the slop */
	tv_set.tv_sec = new->coarse - JAN_1970;
	/* divide xmttime.fine by 4294.967296 */
	tv_set.tv_nsec = USEC(new->fine) * 1000;
	if (clock_settime(CLOCK_REALTIME, &tv_set) < 0) {
		PRINT(LOG_ERR, errno, "Failed clock_settime()");
		exit(EXIT_FAILURE);
	}
	if (debug)
		PRINT(LOG_DEBUG, 0, "Set time to %lu.%.9lu", tv_set.tv_sec, tv_set.tv_nsec);
#else
	/* Traditional Linux way to set the system clock */
	struct timeval tv_set;

	/* it would be even better to subtract half the slop */
	tv_set.tv_sec = new->coarse - JAN_1970;
	/* divide xmttime.fine by 4294.967296 */
	tv_set.tv_usec = USEC(new->fine);
	if (settimeofday(&tv_set, NULL) < 0) {
		PRINT(LOG_ERR, errno, "Failed settimeofday()");
		exit(EXIT_FAILURE);
	}
	if (debug)
		PRINT(LOG_DEBUG, 0, "set time to %lu.%.6lu", tv_set.tv_sec, tv_set.tv_usec);
#endif
}

static void ntpc_gettime(uint32_t *time_coarse, uint32_t *time_fine)
{
#ifndef USE_OBSOLETE_GETTIMEOFDAY
	/* POSIX 1003.1-2001 way to get the system time */
	struct timespec now;

	clock_gettime(CLOCK_REALTIME, &now);
	*time_coarse = now.tv_sec + JAN_1970;
	*time_fine = NTPFRAC(now.tv_nsec / 1000);
#else
	/* Traditional Linux way to get the system time */
	struct timeval now;

	gettimeofday(&now, NULL);
	*time_coarse = now.tv_sec + JAN_1970;
	*time_fine = NTPFRAC(now.tv_usec);
#endif
}

static void send_packet(int usd, uint32_t time_sent[2])
{
	uint32_t data[12];

#define LI 0
#define VN 3
#define MODE 3
#define STRATUM 0
#define POLL 4
#define PREC -6

#ifdef ENABLE_DEBUG
	if (debug)
		PRINT(LOG_DEBUG, 0, "Sending packet ...");
#endif
	if (sizeof(data) != 48) {
		PRINT(LOG_ERR, 0, "Packet size error");
		return;
	}

	memset(data, 0, sizeof data);
	data[0] = htonl((LI << 30) | (VN << 27) | (MODE << 24) | (STRATUM << 16) | (POLL << 8) | (PREC & 0xff));
	data[1] = htonl(1 << 16);	/* Root Delay (seconds) */
	data[2] = htonl(1 << 16);	/* Root Dispersion (seconds) */
	ntpc_gettime(time_sent, time_sent + 1);

	data[10] = htonl(time_sent[0]);	/* Transmit Timestamp coarse */
	data[11] = htonl(time_sent[1]);	/* Transmit Timestamp fine   */
	send(usd, data, 48, 0);
}

static void get_packet_timestamp(int usd, struct ntptime *udp_arrival_ntp)
{
#ifdef PRECISION_SIOCGSTAMP
	struct timeval udp_arrival;

	if (ioctl(usd, SIOCGSTAMP, &udp_arrival) < 0) {
		PRINT(LOG_ERR, errno, "Failed ioctl(SIOCGSTAMP)");
		ntpc_gettime(&udp_arrival_ntp->coarse, &udp_arrival_ntp->fine);
	} else {
		udp_arrival_ntp->coarse = udp_arrival.tv_sec + JAN_1970;
		udp_arrival_ntp->fine = NTPFRAC(udp_arrival.tv_usec);
	}
#else
	(void)usd;		/* not used */
	ntpc_gettime(&udp_arrival_ntp->coarse, &udp_arrival_ntp->fine);
#endif
}

static int check_source(int data_len, struct sockaddr_storage *sa_source, struct ntp_control *ntpc)
{
	struct sockaddr_in6 *ipv6;
	struct sockaddr_in *ipv4;
	uint16_t port;

	(void)data_len;
	(void)ntpc;		/* not used */
	PRINT(LOG_DEBUG, 0, "packet of length %d received", data_len);

	if (sa_source->ss_family == AF_INET) {
		ipv4 = (struct sockaddr_in *)(sa_source);
		port = ntohs(ipv4->sin_port);
	} else if (sa_source->ss_family == AF_INET6) {
		ipv6 = (struct sockaddr_in6 *)(sa_source);
		port = ntohs(ipv6->sin6_port);
	} else {
		PRINT(LOG_DEBUG, 0, "%s: Unsupported address family", __func__);
		return 1;
	}

	/*
	 * we could check that the source is the server we expect, but
	 * Denys Vlasenko recommends against it: multihomed hosts get it
	 * wrong too often.
	 */
	if (NTP_PORT != port) {
		PRINT(LOG_INFO, 0, "%s: invalid port: %u", __func__, port);
		return 1;
	}

	return 0;
}

static double ntpdiff(struct ntptime *start, struct ntptime *stop)
{
	int a;
	unsigned int b;

	a = stop->coarse - start->coarse;
	if (stop->fine >= start->fine) {
		b = stop->fine - start->fine;
	} else {
		b = start->fine - stop->fine;
		b = ~b;
		a -= 1;
	}

	return a * 1.e6 + b * (1.e6 / 4294967296.0);
}

/* Does more than print, so this name is bogus.
 * It also makes time adjustments, both sudden (-s)
 * and phase-locking (-l).
 * sets *error to the number of microseconds uncertainty in answer
 * returns 0 normally, 1 if the message fails sanity checks
 */
static int rfc1305print(uint32_t *data, struct ntptime *arrival, struct ntp_control *ntpc, int server_index, int *error)
{
	/* straight out of RFC-1305 Appendix A */
	int li, vn, mode, stratum, prec;
	int delay, disp;

#ifdef ENABLE_DEBUG
	int poll, refid;
	struct ntptime reftime;
#endif
	struct ntptime orgtime, rectime, xmttime;
	double el_time, st_time, skew1, skew2;
	int freq;
	const char *drop_reason = NULL;

#define Data(i) ntohl(((uint32_t *)data)[i])
	li      = Data(0) >> 30 & 0x03;
	vn      = Data(0) >> 27 & 0x07;
	mode    = Data(0) >> 24 & 0x07;
	stratum = Data(0) >> 16 & 0xff;
#ifdef ENABLE_DEBUG
	poll    = Data(0) >>  8 & 0xff;
#endif
	prec    = Data(0) & 0xff;
	if (prec & 0x80)
		prec |= 0xffffff00;
	delay   = Data(1);
	disp    = Data(2);

#ifdef ENABLE_DEBUG
	refid          = Data(3);
	reftime.coarse = Data(4);
	reftime.fine   = Data(5);
#endif

	orgtime.coarse = Data(6);
	orgtime.fine   = Data(7);
	rectime.coarse = Data(8);
	rectime.fine   = Data(9);
	xmttime.coarse = Data(10);
	xmttime.fine   = Data(11);
#undef Data

#ifdef ENABLE_DEBUG
	if (debug) {
		PRINT(LOG_DEBUG, 0, "LI=%d  VN=%d  Mode=%d  Stratum=%d  Poll=%d  Precision=%d", li, vn, mode, stratum, poll, prec);
		PRINT(LOG_DEBUG, 0, "Delay=%.1f  Dispersion=%.1f  Refid=%u.%u.%u.%u", sec2u(delay), sec2u(disp),
		      refid >> 24 & 0xff, refid >> 16 & 0xff, refid >> 8 & 0xff, refid & 0xff);
		PRINT(LOG_DEBUG, 0, "Reference %u.%.6u", reftime.coarse, USEC(reftime.fine));
		PRINT(LOG_DEBUG, 0, "(sent)    %u.%.6u", ntpc->time_of_send[0], USEC(ntpc->time_of_send[1]));
		PRINT(LOG_DEBUG, 0, "Originate %u.%.6u", orgtime.coarse, USEC(orgtime.fine));
		PRINT(LOG_DEBUG, 0, "Receive   %u.%.6u", rectime.coarse, USEC(rectime.fine));
		PRINT(LOG_DEBUG, 0, "Transmit  %u.%.6u", xmttime.coarse, USEC(xmttime.fine));
		PRINT(LOG_DEBUG, 0, "Our recv  %u.%.6u", arrival->coarse, USEC(arrival->fine));
	}
#endif

	el_time = ntpdiff(&orgtime, arrival);	/* elapsed */
	st_time = ntpdiff(&rectime, &xmttime);	/* stall */
	skew1 = ntpdiff(&orgtime, &rectime);
	skew2 = ntpdiff(&xmttime, arrival);
	freq = get_current_freq();

#ifdef ENABLE_DEBUG
	if (debug) {
		PRINT(LOG_DEBUG, 0, "Total elapsed: %9.2f", el_time);
		PRINT(LOG_DEBUG, 0, "Server stall:  %9.2f", st_time);
		PRINT(LOG_DEBUG, 0, "Slop:          %9.2f", el_time - st_time);
		PRINT(LOG_DEBUG, 0, "Skew:          %9.2f", (skew1 - skew2) / 2);
		PRINT(LOG_DEBUG, 0, "Frequency:     %9d", freq);
		PRINT(LOG_DEBUG, 0, " Day   Second     Elapsed    Stall     Skew  Dispersion  Freq");
	}
#endif

	/* error checking, see RFC-4330 section 5 */
#define FAIL(x) do { drop_reason=(x); goto fail;} while (0)
	if (ntpc->cross_check) {
		if (li == 3)
			FAIL("LI==3");	/* unsynchronized */
		if (vn < 3)
			FAIL("VN<3");	/* RFC-4330 documents SNTP v4, but we interoperate with NTP v3 */
		if (mode != 4)
			FAIL("MODE!=3");
		if (orgtime.coarse != ntpc->time_of_send[0] || orgtime.fine != ntpc->time_of_send[1])
			FAIL("ORG!=sent");
		if (xmttime.coarse == 0 && xmttime.fine == 0)
			FAIL("XMT==0");
		if (delay > 65536 || delay < -65536)
			FAIL("abs(DELAY)>65536");
		if (disp > 65536 || disp < -65536)
			FAIL("abs(DISP)>65536");
		if (stratum == 0)
			FAIL("STRATUM==0");	/* kiss o' death */
#undef FAIL
	}

	PRINT(LOG_DEBUG, 0, "ntpc->set_clock:%d", ntpc->set_clock);
	/* XXX should I do this if debug flag is set? */
	if (ntpc->set_clock) {	/* you'd better be root, or ntpclient will exit here! */
		set_time(&xmttime);
		PRINT(LOG_DEBUG, 0, "/bin/accesstimer &");
		system("/bin/accesstimer &");
//WNC-NMR2202-D2R030-YUAN-I-CHOU-20161220-WiFi schedule would work after 1 minute from system time updated
		PRINT(LOG_DEBUG, 0, "touch /tmp/startwifisch");
		system("touch /tmp/startwifisch");
		/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */
		// WNC-NM01108-D2R031-HENG-I-CHEN-2015-11-18: execute accesstimer after set time #END
	}

	/* Not the ideal order for printing, but we want to be sure
	 * to do all the time-sensitive thinking (and time setting)
	 * before we start the output, especially fflush() (which
	 * could be slow).  Of course, if debug is turned on, speed
	 * has gone down the drain anyway. */
	if (ntpc->live) {
		int new_freq;

		new_freq = contemplate_data(arrival->coarse, (skew1 - skew2) / 2, el_time + sec2u(disp), freq);
		if (!debug && new_freq != freq)
			set_freq(new_freq);
	}

	/* Display by default for regular users, root users need to supply -d. */
	if (debug || ntpc->usermode) {
		PRINT(LOG_NOTICE, 0, "%d %.5d.%.3d  %8.1f %8.1f  %8.1f %8.1f %9d",
		      arrival->coarse / 86400, arrival->coarse % 86400,
		      arrival->fine / 4294967, el_time, st_time,
		      (skew1 - skew2) / 2, sec2u(disp), freq);
	}
	*error = el_time - st_time;

	return 0;
 fail:
	if (debug) {
		PRINT(LOG_ERR, 0, "%d %.5d.%.3d rejected packet: %s",
		      arrival->coarse / 86400, arrival->coarse % 86400,
		      arrival->fine / 4294967, drop_reason);
	}

	return 1;
}

static void setup_receive(int usd, uint16_t port)
{
	struct sockaddr_in6 sin6;
	int opt = 0;

	/*
	 * Setting this means the socket only accepts IPv6 connections
	 * unset it accepts both IPv6 and IPv4 (mapped address)
	 * connections.  Must disable it before calling bind()
	 */
	if (setsockopt(usd, IPPROTO_IPV6, IPV6_V6ONLY, &opt, sizeof(opt))) {
		PRINT(LOG_ERR, errno, "%s: failed setsockopt", __func__);
		exit(1);
	}

	memset(&sin6, 0, sizeof(struct sockaddr_in6));
	sin6.sin6_family = AF_INET6;
	sin6.sin6_port = htons(port);
	sin6.sin6_addr = in6addr_any;
	if (bind(usd, (struct sockaddr *)&sin6, sizeof(sin6)) == -1) {
		PRINT(LOG_ERR, errno, "%s: Failed binding to UDP port %u", __func__, port);
		exit(1);
	}
	/* listen(usd,3); this isn't TCP; thanks Alexander! */
}

static int setup_transmit(int usd, struct ntp_control *ntpc, int server_index)
{
	struct sockaddr_storage ss;
	struct sockaddr_in6 *ipv6;
	struct sockaddr_in *ipv4;
	socklen_t len = 0;

	if (v4fallback.enable) {
		if (getaddrbyname(v4fallback.hostname[server_index], &ss)) {
			return 1;
		}
	} else {
		while (getaddrbyname(ntpc->hostname[server_index], &ss)) {
#if 0 /* No infinite retry of the primary(Retry on primary and secondary). */
		if (EINVAL != errno && ntpc->live) {
			/* Wait here a while, networking is probably not up yet. */
			sleep(1);
			continue;
		}
#endif
			return 1;
		}
	}

	/* Prefer IPv4 over IPv6, for now */
	if (ss.ss_family == AF_INET) {
		ipv4 = (struct sockaddr_in *)(&ss);
		ipv4->sin_port = htons(ntpc->udp_port[server_index]);
		len = sizeof(struct sockaddr_in);
	} else if (ss.ss_family == AF_INET6) {
		ipv6 = (struct sockaddr_in6 *)(&ss);
		ipv6->sin6_port = htons(ntpc->udp_port[server_index]);
		len = sizeof(struct sockaddr_in6);
	} else {
		PRINT(LOG_ERR, 0, "%s: Unsupported address family", __func__);
		return 1;
	}

	while (connect(usd, (struct sockaddr *)&ss, len) == -1) {
		if (ntpc->live) {
			/* Wait here a while, networking is probably not up yet. */
			sleep(1);
			continue;
		}

		PRINT(LOG_ERR, errno, "Failed connecting to NTP server");
		return 1;
	}

	return 0;
}

static int setup_socket(struct ntp_control *ntpc, int server_index)
{
	int sd;

	/* using IPV6 socket for IPV4 and IPV6 */
	if ((sd = socket(AF_INET6, SOCK_DGRAM, IPPROTO_UDP)) == -1)
		return -1;

	setup_receive(sd, ntpc->udp_port[server_index]);
	setup_transmit(sd, ntpc, server_index);

	/*
	 * Every day: reopen socket and perform a new DNS lookup.
	 */
	alarm(60 * 60 * 24);

	return sd;
}

/*
 * Signal handler.  Take note of the fact that the signal arrived
 * so that the main loop can take care of it.
 */
static void handler(int sig)
{
	switch (sig) {
	case SIGHUP:
	case SIGALRM:
		/* Trigger NTP sync */
		sighup = 1;
		break;

	case SIGINT:
	case SIGQUIT:
	case SIGTERM:
	case SIGUSR1:
	case SIGUSR2:
		sigterm = 1;
		break;
	}
}

static void setup_signals(void)
{
	struct sigaction sa;

	sa.sa_handler = handler;
	sa.sa_flags = 0;	/* Interrupt system calls */
	sigemptyset(&sa.sa_mask);
	sigaction(SIGHUP, &sa, NULL);
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGUSR1, &sa, NULL);
	sigaction(SIGUSR2, &sa, NULL);
	sigaction(SIGALRM, &sa, NULL);
}

static int primary_loop(int usd, struct ntp_control *ntpc, int server_index)
{
	fd_set fds;
	struct sockaddr_storage sa_xmit;
	/* Flag to denote whether NTP Response was received or not */  
	int no_response_received;
	/* Flag to Return from 'primary_loop' if loop has run once for Secondary Server already */
	int loop_end;
	int i, pack_len, probes_sent, error;
	socklen_t sa_xmit_len;
	struct timeval to;
	struct ntptime udp_arrival_ntp;
	static uint32_t incoming_word[325];

#define incoming ((char *) incoming_word)
#define sizeof_incoming (sizeof incoming_word)

#ifdef ENABLE_DEBUG
	if (debug)
		PRINT(LOG_DEBUG, 0, "Listening...");
#endif
	probes_sent = 0;
	sa_xmit_len = sizeof(sa_xmit);
	to.tv_sec = 0;
	to.tv_usec = 0;
	no_response_received=loop_end=0;

	while (1) {
		if (sigterm) {
			ntpc->live = 0;
			break;
		}
		if (sighup) {
			sighup = 0;
			to.tv_sec = 0;
			to.tv_usec = 0;
			close(usd);
			usd = setup_socket(ntpc, server_index);
			if (-1 == usd) {
				PRINT(LOG_ERR, errno, "Failed reopening NTP socket");
				return 2;
			}
			PRINT(LOG_DEBUG, 0, "Got SIGHUP, triggering resync with NTP server.");
		}

		FD_ZERO(&fds);
		FD_SET(usd, &fds);
		PRINT(LOG_DEBUG, 0, "server_index:%d select to.tv_sec %d", server_index, to.tv_sec);
		i = select(usd + 1, &fds, NULL, NULL, &to);	/* Wait on read or error */
		PRINT(LOG_DEBUG, 0, "select return %d", i);
		if ((i != 1) || (!FD_ISSET(usd, &fds))) {
			PRINT(LOG_DEBUG, 0, "server_index=%d (i != 1) || (!FD_ISSET(usd, &fds)", server_index);
			if (i < 0) {
				if (errno != EINTR)
					PRINT(LOG_ERR, errno, "Failed select()");
				continue;
			}
			if (to.tv_sec == 0) {
				PRINT(LOG_DEBUG, 0, "server_index:%d no_response_received %d, loop_end %d", server_index, no_response_received, loop_end);
				/* If the NTP request has Timed-out, return error */
				if (no_response_received) return 1;
				/* If there are more probes to be sent, have the Primary Server do so */
				if (loop_end) {
					if ((ntpc->probe_count != 0) && (probes_sent >= ntpc->probe_count))
						break;
					else
						return 2;
				}
				PRINT(LOG_DEBUG, 0, "server_index:%d bf send_packet() [0]:%u, [1]:%u", server_index, ntpc->time_of_send[0], ntpc->time_of_send[1]);
				send_packet(usd, ntpc->time_of_send);
				++probes_sent;
				to.tv_sec = ntpc->no_response_time;
				to.tv_usec = 0;
				no_response_received=1;
			}
			continue;
		}
		pack_len = recvfrom(usd, incoming, sizeof_incoming, 0, (struct sockaddr *)&sa_xmit, &sa_xmit_len);
		PRINT(LOG_DEBUG, 0, "server_index:%d pack_len %d", server_index, pack_len);
		error = ntpc->goodness;
		if (pack_len < 0) {
			PRINT(LOG_ERR, errno, "Failed recvfrom()");
		} else if (pack_len > 0 && (unsigned)pack_len < sizeof_incoming) {
			get_packet_timestamp(usd, &udp_arrival_ntp);
			if (check_source(pack_len, &sa_xmit, ntpc)) {
				continue;
			}
			if (rfc1305print(incoming_word, &udp_arrival_ntp, ntpc, server_index, &error) != 0) {
				continue;
			}
			/* udp_handle(usd,incoming,pack_len,&sa_xmit,sa_xmit_len); */
			no_response_received=0;
			++probes_sent;
			/* Return if the 'Probe Count' limit has been reached */
			if ((ntpc->probe_count != 0) && (probes_sent >= ntpc->probe_count)) {
				break;
			}
			/* The Secondary Server is Fall-Back only; Set the loop to return after one Send-Recv Cycle in that case */
			/* If done IPv4 Fall-Back, Set the loop to return after one Send-Recv Cycle in that case */
			if (server_index != 0 || v4fallback.enable != 0) {
				loop_end=1;
				v4fallback.enable = 0;
			}
			/* Now that we have received a successful response, have the next NTP probe sent after 'Synchronization Interval' */
			to.tv_sec=ntpc->synchronization_interval;
			to.tv_usec=0;
		} else {
			PRINT(LOG_ERR, 0, "Ooops.  pack_len=%d", pack_len);
		}

		/*
		 * best rollover option: specify -g, -s, and -l.
		 * simpler rollover option: specify -s and -l, which
		 * triggers a magic -c 1
		 */
		if ((error < ntpc->goodness && ntpc->goodness != 0) ||
		    (probes_sent >= ntpc->probe_count && ntpc->probe_count != 0)) {
			ntpc->set_clock = 0;
			if (!ntpc->live)
				break;
		}
	}
#undef incoming
#undef sizeof_incoming
	PRINT(LOG_DEBUG, 0, "primary_loop() end");
	return 0;
}

#ifdef ENABLE_REPLAY
static int do_replay(void)
{
	char line[100];
	int n, day, freq, absolute;
	float sec, el_time, st_time, disp;
	double skew, errorbar;
	int simulated_freq = 0;
	unsigned int last_fake_time = 0;
	double fake_delta_time = 0.0;

	while (fgets(line, sizeof line, stdin)) {
		n = sscanf(line, "%d %f %f %f %lf %f %d", &day, &sec, &el_time, &st_time, &skew, &disp, &freq);
		if (n == 7) {
			PRINT(LOG_DEBUG, 0, "%s", line);
			absolute = day * 86400 + (int)sec;
			errorbar = el_time + disp;
#ifdef ENABLE_DEBUG
			if (debug)
				PRINT(LOG_DEBUG, 0, "Contemplate %u %.1f %.1f %d", absolute, skew, errorbar, freq);
#endif
			if (last_fake_time == 0)
				simulated_freq = freq;
			fake_delta_time += (absolute - last_fake_time) * ((double)(freq - simulated_freq)) / 65536;
#ifdef ENABLE_DEBUG
			if (debug)
				PRINT(LOG_DEBUG, 0, "Fake %f %d", fake_delta_time, simulated_freq);
#endif
			skew += fake_delta_time;
			freq = simulated_freq;
			last_fake_time = absolute;
			simulated_freq = contemplate_data(absolute, skew, errorbar, freq);
		} else {
			PRINT(LOG_ERR, 0, "Replay input error");
			return 2;
		}
	}

	return 0;
}
#endif

static int usage(int code)
{
	fprintf(stderr,
		"Usage: %s [-dl"
#ifdef ENABLE_REPLAY
		"r"
#endif
	"stV] [-c count]"
	" [-f frequency] [-g goodness] [-q min delay]\n"
	"\t[-H primary hostname] [-P primary port] [-h secondary hostname] [-p secondary port]\n"
	"\t[-i synchronization interval] [-n no response judgment timeout] [-R retry interval cycle]\n"
	"\t"
	"\n",
	prognm);

	fprintf(stderr, "Options:\n"
		" -c count      Stop after count time measurements. Default: 0 (forever)\n"
		" -d            Debug, or diagnostics mode  Possible to enable more at compile\n"
		" -f frequency  Initialize frequency offset.  Linux only, requires root\n"
		" -g goodness   Stop after getting a result more accurate than goodness msec,\n"
		"               microseconds. Default: 0 (forever)\n"
		" -H hostname   primary hostname\n"
		" -P port       primary port\n"
		" -h hostname   secondary hostname\n"
		" -p port       secondary port\n"
		" -i interval   Check time every interval seconds.  Default: 600\n"
		" -l            Attempt to lock local clock to server using adjtimex(2)\n"
		" -n timeout    no response judgment timeout\n"
		" -R interval   retry interval cycle\n"
		" -q min_delay  Minimum packet delay for transaction (default 800 microseconds)\n"
#ifdef ENABLE_REPLAY
		" -r            Replay analysis code based on stdin\n"
#endif
		" -s            Simple clock set, implies -c 1 unless -l is also set\n"
		" -t            Trust network and server, no RFC-4330 recommended validation\n"
		" -V            Display version and copyright information\n"
		"\n");

	return code;
}

static int version(void)
{
	fprintf(stderr, "Larry Doolittle's ntpclient v" PACKAGE_VERSION "\n\n");
	fprintf(stderr, "Copyright (C) 1997-2015  Larry Doolittle <larry@doolittle.boa.org>\n"
		"Copyright (C) 2010-2017  Joachim Nilsson <troglobit@gmail.com>\n\n");
	fprintf(stderr, "License GPLv2: GNU GPL version 2 <http://gnu.org/licenses/gpl2.html>\n"
		"This is free software: you are free to change and redistribute it.\n"
		"There is NO WARRANTY, to the extent permitted by law.\n");

	return 1;
}

static const char *progname(const char *arg0)
{
	const char *nm;

	nm = strrchr(arg0, '/');
	if (nm)
		nm++;
	else
		nm = arg0;

	return nm;
}
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
//Brad add sync system time for dhcpd 2008/01/28 
int getPid(char *filename)
{
	struct stat status;
	char buff[100];
	FILE *fp;

	if ( stat(filename, &status) < 0)
		return -1;
	fp = fopen(filename, "r");
	if (!fp) {
        	printf("Can not open file :%s\n", filename);
		return -1;
   	}
	fgets(buff, 100, fp);
	fclose(fp);

	return (atoi(buff));
}

void Inform_DHCPD(void)
{
	char tmpBuf[100];
	int pid;
	memset(tmpBuf, '\0', 100);
	sprintf(tmpBuf, "%s/%s.pid", "/var/run", "udhcpd");
	pid = getPid(tmpBuf);
	if ( pid > 0)
		kill(pid, SIGUSR2);
	usleep(1000);
}
//-----------------------------------
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */

int main(int argc, char *argv[])
{
	int c, usd;		/* socket */
	int ret_val=EXIT_SUCCESS;

	/* These parameters are settable from the command line
	   the initializations here provide default behavior */
	int initial_freq = 0;	/* initial freq value to use */
	struct ntp_control ntpc;

/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
//Brad add sync system time for dhcpd 2008/01/28 
	unsigned long sys_time_orig=0;
	char tmpBuf[100];
	time(&sys_time_orig);
//-----------------------------------------------------
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */
	ntpc.live=0;
	ntpc.set_clock=0;
	ntpc.probe_count=0;		/* default of 0 means loop forever */
	ntpc.synchronization_interval=SYNCHRONIZATION_INTERVAL;	/* 'Synchronization Interval' in seconds (default is 24hrs) */
	ntpc.no_response_time=NO_RESPONSE_TIME;	/* 'No Response Time' in seconds (default is 60s) */
	strcpy(ntpc.retry_interval, RETRY_INTERVAL);	/* 'Retry Interval' Cycle Range (default is '1min-2min-4-min-8-min-16min-16 after that') */
	ntpc.goodness=0;
	ntpc.cross_check=1;
	ntpc.udp_port[0]=NTP_PORT;
	ntpc.udp_port[1]=NTP_PORT;
	ntpc.hostname[0]=NULL;
	ntpc.hostname[1]=NULL;

	/* Use 'syslog' mechanism for logging messages */
	openlog("ntpclient", LOG_PID|LOG_CONS, LOG_DAEMON);
	/* Arrange for the Syslog Connected to be closed on program termination */
	atexit(closelog);

	prognm = progname(argv[0]);
	while (1) {
		char opts[] = "c:" DEBUG_OPTION "f:g:q:H:P:h:p:i:R:ln:" REPLAY_OPTION "46stDV?";

		c = getopt(argc, argv, opts);
		if (c == EOF)
			break;

		switch (c) {
		case 'c':
			ntpc.probe_count = atoi(optarg);
			break;

		case 'd':
			debug++;
			log_level = LOG_DEBUG;
			break;

		case 'f':
			initial_freq = atoi(optarg);
			break;

		case 'g':
			ntpc.goodness = atoi(optarg);
			break;

		case 'H':
			ntpc.hostname[0] = optarg;
			break;
		case 'P':
			ntpc.udp_port[0] = atoi(optarg);
			break;
		case 'h':
			ntpc.hostname[1] = optarg;
			break;
		case 'p':
			ntpc.udp_port[1] = atoi(optarg);
			break;

		case 'i':
			ntpc.synchronization_interval = atoi(optarg);
			break;

		case 'l':
			ntpc.live++;
			break;

		case 'n':
			ntpc.no_response_time = atoi(optarg);
			break;

		/* The 'Retry Interval' Cycle needs to be delimited by ',';
		 * A '.' at the end means that the previous 'Retry Interval' value is used forever */
		case 'R':
			if (!strcmp(optarg, ".,")) {
				usage(1);
				exit(EXIT_FAILURE);
			}
			strncpy(ntpc.retry_interval, optarg, sizeof(ntpc.retry_interval)-1);
			ntpc.retry_interval[strlen(ntpc.retry_interval)-1]='\0';
			break;

		case 'q':
			min_delay = atof(optarg);
			break;

#ifdef ENABLE_REPLAY
		case 'r':
			return do_replay();
#endif

		case 's':
			ntpc.set_clock++;
			break;

		case 't':
			ntpc.cross_check = 0;
			break;

		case 'D':
			run_as_daemon=1;
			/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
			//daemon();
			daemon(0, 1);
			/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */
			break;

		case '4':
			ip_version = 4;
			break;

		case '6':
			ip_version = 6;
			break;

		case 'V':
			return version();

		case '?':
		default:
			return usage(0);
		}
	}

	/* Bail Out if the Primary Server or Secondary Server Information is absent */
	if ((ntpc.hostname[0] == NULL) || (ntpc.hostname[1] == NULL)) {
		fprintf(stderr, "Missing NTP server argument.\n");
		usage(1);
		exit(EXIT_FAILURE);
	}

	if (initial_freq) {
#ifdef ENABLE_DEBUG
		PRINT(LOG_DEBUG, 0, "Initial frequency %d", initial_freq);
#endif
		set_freq(initial_freq);
	}

	if (ntpc.set_clock && !ntpc.live && !ntpc.goodness && !ntpc.probe_count)
		ntpc.probe_count = 1;

	/* If user gives a probe count, then assume non-live run and verbose reporting. */
	if (ntpc.probe_count > 0) {
		ntpc.live = 0;
	}

	/* respect only applicable MUST of RFC-4330 */
	if (ntpc.probe_count != 1 && ntpc.synchronization_interval < MIN_INTERVAL) {
		ntpc.synchronization_interval = MIN_INTERVAL;
	}

#ifdef ENABLE_DEBUG
	if (debug) {
		PRINT(LOG_DEBUG, 0, "Configuration:");
		PRINT(LOG_DEBUG, 0, "  -c probe_count                  %d", ntpc.probe_count);
		PRINT(LOG_DEBUG, 0, "  -d (debug)                      %d", debug);
		PRINT(LOG_DEBUG, 0, "  -g goodness                     %d", ntpc.goodness);
		PRINT(LOG_DEBUG, 0, "  -H primary hostname             %s", ntpc.hostname[0]);
		PRINT(LOG_DEBUG, 0, "  -P primary port                 %d", ntpc.udp_port[0]);
		PRINT(LOG_DEBUG, 0, "  -h secondary hostname           %s", ntpc.hostname[1]);
		PRINT(LOG_DEBUG, 0, "  -p secondary port               %d", ntpc.udp_port[1]);
		PRINT(LOG_DEBUG, 0, "  -i synchronization interval     %d", ntpc.synchronization_interval);
		PRINT(LOG_DEBUG, 0, "  -n no response judgment timeout %d", ntpc.no_response_time);
		PRINT(LOG_DEBUG, 0, "  -R retry interval cycle         %s", ntpc.retry_interval);
		PRINT(LOG_DEBUG, 0, "  -l live                         %d", ntpc.live);
		PRINT(LOG_DEBUG, 0, "  -q min_delay                    %f", min_delay);
		PRINT(LOG_DEBUG, 0, "  -s set_clock                    %d", ntpc.set_clock);
		PRINT(LOG_DEBUG, 0, "  -t cross_check                  %d", ntpc.cross_check);
		PRINT(LOG_DEBUG, 0, "  -D daemon                       %d", run_as_daemon);
	}
#endif /* ENABLE_DEBUG */

	/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
	// create the pid file
	{
		FILE *old_fp = NULL;
		FILE *fp = NULL;
		pid_t mypid;
		pid_t kill_pid;
		char *pid_file = "/var/run/ntpclient.pid";
		char buf[256] = {0};
		
		if ((old_fp = fopen(pid_file, "r")) != NULL) {
			if(fgets(buf, sizeof(buf), old_fp) != NULL) {
				kill_pid = strtoul(buf, NULL, 10);
				if(kill_pid != 0){
					kill(kill_pid, SIGKILL);
				}
			}

			fclose(old_fp);
		}

		if ((fp = fopen(pid_file, "w")) == NULL) {
			fprintf(stderr, "could not create pid file: %s \n", pid_file);
			exit(1);
		}

		mypid = getpid();
		fprintf(fp, "%d\n", (int)mypid);
		fclose(fp);
	}
	/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */

	/* This NTP Client works as follows:
	 * The Client first tries the Primary Server, if the connection fails or no NTP response is received, it tries the Secondary Server after 'No Response Judgment Time' elapses;
	 * If the Secondary Server does not respond too, the whole Cycle is repeated after 'Retry Interval' amount of time;
	 * After sending an NTP Request to an NTP Server, the Client waits for 'No Response Judgment' Time;
	 * If it receives a response within that timeframe, the Client sends the next NTP Request 'Synchronization Interval' time after the previously sent request. */
	unsigned int retry_interval=0;
	char *retry_interval_str=alloca(1); /* Make Compiler Happy by allocating the string on Stack using "alloca" */
	memset(retry_interval_str, '\0', 1);    /* Initialize it to be Empty. */
	char *retry_interval_cycle=ntpc.retry_interval;
	while (1) {
		ret_val=EXIT_SUCCESS;
		int i, ret;
		for (i=0; i < 2; i++) {
			const char *server=(i == 0) ? "Primary":"Secondary";

			PRINT(LOG_INFO, 0, "Attempting to fetch Time from %s Server [%s]...", server, ntpc.hostname[i]);

			if ((usd=setup_socket(&ntpc, i)) == -1) {
				PRINT(LOG_ERR, errno, "Failed creating UDP socket() to SNTP server");
				if (i == 0) {
					continue;
				} else {
					ret_val=EXIT_FAILURE;
					break;
				}
			}

			setup_signals();

			if (run_as_daemon) {
				PRINT(LOG_NOTICE, 0, "Using time sync server: %s", ntpc.hostname[i]);
			}

			ret = primary_loop(usd, &ntpc, i);

			close(usd);

			/* If there are pending probes, reset the Loop so that the next request goes to the Primary Server */
			if (ret == 2) {
				i=-1;
				continue;
			} else if (ret == 0) {
				break;
			} else if (i == 0) {
				PRINT(LOG_ERR, errno, "Primary Server [%s] NTP Request Timeout! Trying Secondary Server [%s]...", ntpc.hostname[0], ntpc.hostname[1]);
			} else {
				PRINT(LOG_ERR, errno, "Secondary Server [%s] NTP Request Timeout!", ntpc.hostname[1]);
				ret_val=EXIT_FAILURE;
				break;
			}
		}

		if (ret_val == EXIT_SUCCESS) {
			break;
		}

		if (ip_version == 6 && v4fallback.enable == 0) {
			/* If IPv6 server all fail, Fall-Back IPv4 mode */
			v4fallback.enable = 1; /* ON */
			continue;
		} else if (ip_version == 6 && v4fallback.enable == 1) {
			/* If IPv4 server all fail, Roll-Back IPv6 mode*/
			v4fallback.enable = 0; /* OFF */
		}

		if (strcmp(retry_interval_str, ".")) {
			retry_interval_str=strtok(retry_interval_cycle, ",");
			if (retry_interval_str == NULL) {
				break;
			}

			if (strcmp(retry_interval_str, ".")) {
				retry_interval=(unsigned int)atol(retry_interval_str);
			}

			retry_interval_cycle=NULL;
		}

		if (retry_interval > 0) {
			PRINT(LOG_ERR, errno, "Will Retry NTP Connection Request after [%u] seconds...", retry_interval);
			/* Sleep for 'Retry Interval' before trying again */
			sleep(retry_interval);
		}
	}

/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-start */
//Brad add sync system time for dhcpd 2008/01/28 	
	sprintf(tmpBuf, "echo \"%ld\" > /tmp/dhcpd_unix", sys_time_orig);
	system(tmpBuf);
	Inform_DHCPD();
//-----------------------------------------------	
/* NMR0000-20160314-Porting NTP client from Laser codebase to Spider codebase-end */

	return ret_val;
}

static int getaddrbyname(char *host, struct sockaddr_storage *ss)
{
	int err;
	static int netdown = 0;
	struct addrinfo hints;
	struct addrinfo *result;
	struct addrinfo *rp;

	if (!host || !ss) {
		errno = EINVAL;
		return 1;
	}

	memset(&hints, 0, sizeof(struct addrinfo));
	hints.ai_family = AF_UNSPEC;
	if (ip_version == 4) {
		hints.ai_family = AF_INET;
	} else if (ip_version == 6) {
		if (v4fallback.enable) {
			hints.ai_family = AF_INET;
		} else {
			hints.ai_family = AF_INET6;
		}
	}
	hints.ai_socktype = 0;
	hints.ai_flags = AI_PASSIVE;
	hints.ai_protocol = 0;
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;

	memset(ss, 0, sizeof(struct sockaddr_storage));
	err = getaddrinfo(host, NULL, &hints, &result);
	if (err) {
		PRINT(LOG_ERR, 0, "Failed resolving address to hostname %s: %s", host, gai_strerror(err));
		netdown = errno = ENETDOWN;
		return 1;
	}

	/* The first result will be used. IPV4 has higher priority */
	err = 1;
	for (rp = result; rp; rp = rp->ai_next) {
		if (rp->ai_family == AF_INET) {
			memcpy(ss, (struct sockaddr_in *)(rp->ai_addr), sizeof(struct sockaddr_in));
			err = 0;
			break;
		}
		if (rp->ai_family == AF_INET6) {
			memcpy(ss, (struct sockaddr_in6 *)(rp->ai_addr), sizeof(struct sockaddr_in6));
			err = 0;
			break;
		}
	}
	freeaddrinfo(result);

	if (err) {
		errno = EAGAIN;
		return 1;
	}

	if (netdown) {
		PRINT(LOG_NOTICE, 0, "Network up, resolved address to hostname %s", host);
		netdown = 0;
	}

	return 0;
}

/**
 * Local Variables:
 *  compile-command: "make ntpclient"
 *  indent-tabs-mode: t
 *  c-file-style: "linux"
 * End:
 */
