/*	$KAME: dhcp6c_ia.c,v 1.33 2005/07/22 08:50:05 jinmei Exp $	*/

/*
 * Copyright (C) 2003 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/queue.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in.h>

#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dhcp6.h"
#include "config.h"
#include "common.h"
#include "timer.h"
#include "dhcp6c.h"
#include "dhcp6c_ia.h"
#include "prefixconf.h"
#include "addrconf.h"

#ifdef DHCPV6_LOGO
#include <errno.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <unistd.h>
#include "libnetlink.h"
#endif


char *xifname = NULL;  //rbj
extern int ra_pfxlen;
typedef enum {IAS_ACTIVE, IAS_RENEW, IAS_REBIND , IAS_CNF , IAS_DECLINE} iastate_t;
typedef enum {IA_OK, IA_NG, IA_NOBINDING} iacheck_t;

struct ia {
	TAILQ_ENTRY(ia) link;

	/* back pointer to configuration */
	struct ia_conf *conf;

	/* common parameters of IA */
	u_int32_t t1;		/* duration for renewal */
	u_int32_t t2;		/* duration for rebind  */

	/* internal parameters for renewal/rebinding */
	iastate_t state;
	struct dhcp6_timer *timer;
	struct dhcp6_eventdata *evdata;

	/* DHCP related parameters */
	struct dhcp6_if *ifp;	/* DHCP interface */
	struct duid serverid;	/* the server ID that provided this IA */

	/* control information shared with each particular config routine */
	struct iactl *ctl;

	/* authentication parameters for transaction with servers on this IA */
	struct authparam *authparam;
};

static iacheck_t check_update_ia __P((struct ia **, iatype_t, struct siteprefix **, struct statefuladdr **,
    struct dhcp6_list *, struct dhcp6_if *, struct duid *, struct authparam *));
static int update_authparam __P((struct ia *, struct authparam *));
static void reestablish_ia __P((struct ia *));
static void reestablish_ia_pd_na __P((struct ia *));
static void callback __P((struct ia *));
static int release_ia __P((struct ia *));
static void remove_ia __P((struct ia *));
static struct ia *get_ia __P((iatype_t, struct dhcp6_if *, struct ia_conf *,
    struct dhcp6_listval *, struct duid *));
static struct ia *find_ia __P((struct ia_conf *, iatype_t, u_int32_t));
static struct dhcp6_timer *ia_timo __P((void *));
static struct dhcp6_timer *ia_pd_na_timo __P((void *));
static struct dhcp6_timer *sa_sp_timo __P((void *));

static void check_binding_ia_pd __P((struct dhcp6_optinfo *, struct dhcp6_if *));
static void check_binding_ia_na __P((struct dhcp6_optinfo *, struct dhcp6_if *));

static char *iastr __P((iatype_t));
static char *statestr __P((iastate_t));

extern int iana_addr_nothing;
extern int iana_addr_delete;
extern int ngn_pd_mode;

struct siteprefix *g_sp;
struct statefuladdr *g_sa;

#ifdef DHCPV6_LOGO
void release_choose_ia(iatype_t iatype, struct dhcp6_list *ialist, struct dhcp6_if *ifp,struct duid *serverid)
{
	struct ia_conf *iac;
	struct dhcp6_listval *iav;
	struct ia *ia;

	for (iav = TAILQ_FIRST(ialist); iav; iav = TAILQ_NEXT(iav, link)) 
	{
		if ((iac = find_iaconf(&ifp->iaconf_list, iatype, iav->val_ia.iaid)) == NULL) 
			continue;	

		ia = get_ia(iatype, ifp, iac, iav, serverid);
		if(ia)
			remove_ia(ia);
	}
}

//for UNH-IOL (tom, 20110502)
static void renew_ia(struct ia *ia)
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;
	int dhcpstate;
	struct timeval timo;

	debug_printf(LOG_DEBUG, FNAME, "renew IA: %s-%lu", iastr(ia->conf->type), ia->conf->iaid);

	/* cancel the current event for the prefix. */
	if (ia->evdata) 
	{
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

	ia->state = IAS_RENEW;
	dhcpstate = DHCP6S_RENEW;
	timo.tv_sec = ia->t1 < ia->t2 ? ia->t2 - ia->t1 : 0;
	timo.tv_usec = 0;
	dhcp6_set_timer(&timo, ia->timer);

	if ((ev = dhcp6_create_event(ia->ifp, dhcpstate)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	evd->event = ev;
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	if (duidcpy(&ev->serverid, &ia->serverid)) {
		debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
		goto fail;
	}

	iaparam.iaid = ia->conf->iaid;
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;

	if (ia->ctl && ia->ctl->renew_data) {
		if ((*ia->ctl->renew_data)(ia->ctl, &iaparam,
		    &ia->evdata, evd)) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to make renew data");
			goto fail;
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	ia->evdata = evd;
	client6_send(ev);

	return;
fail:
	if (ev)
		dhcp6_remove_event(ev);

	return;
}

void renew_choose_ia(iatype_t iatype, struct dhcp6_list *ialist, struct dhcp6_if *ifp,struct duid *serverid)
{
	struct ia_conf *iac;
	struct dhcp6_listval *iav;
	struct ia *ia;

	for (iav = TAILQ_FIRST(ialist); iav; iav = TAILQ_NEXT(iav, link)) 
	{
		if ((iac = find_iaconf(&ifp->iaconf_list, iatype, iav->val_ia.iaid)) == NULL) 
			continue;	

		ia = get_ia(iatype, ifp, iac, iav, serverid);
		if(ia)
			renew_ia(ia);
	}
}

typedef struct match_cond_t
{
	struct in6_addr addr;
	int if_idx;
	int found;
}match_cond;

#define IFA_F_DADFAILED (0x08)

static int check_address_object(const struct sockaddr_nl *who , struct nlmsghdr *n , void *arg)
{
	match_cond *target = (match_cond*)arg;
	struct ifaddrmsg *ifa = NLMSG_DATA(n);
	struct rtattr * rta_tb[IFA_MAX+1];

	if(n->nlmsg_len < NLMSG_LENGTH(sizeof(ifa)))
		return 0; //skip this

	if(ifa->ifa_index != target->if_idx)
		return 0; //skip this

	if(ifa->ifa_family != AF_INET6)
		return 0; //skip this

	parse_rtattr(rta_tb, IFA_MAX, IFA_RTA(ifa), n->nlmsg_len - NLMSG_LENGTH(sizeof(*ifa)));

	if(rta_tb[IFA_ADDRESS])
	{
		if(memcmp(target->addr.s6_addr , ((struct in6_addr*)RTA_DATA(rta_tb[IFA_ADDRESS]))->s6_addr , 16) == 0)
		{
			//is DAD ok?
			if((ifa->ifa_flags & IFA_F_DADFAILED) == 0)
				target->found = 1;
		}
	}

	return 0;
}

static int check_address_status(const char *ifname , struct in6_addr *addr)
{
	match_cond target;
	int result = 0;
	struct rtnl_handle rth;

	memset(&target , 0 , sizeof(target));
	//prepare our target object
	memcpy(target.addr.s6_addr , addr->s6_addr , 16);
	target.if_idx = if_nametoindex(ifname);
	if(target.if_idx == 0)
	{
		result = -ENODATA;
		goto out;
	}

	if((result = rtnl_open(&rth , 0)) < 0)
		goto out;

	if((result = rtnl_wilddump_request(&rth, AF_INET6, RTM_GETADDR)) < 0)
		goto out_close_connection;

	if((result = rtnl_dump_filter(&rth, check_address_object, &target, NULL, NULL)) < 0)
		goto out_close_connection;
	
	if(target.found == 0)
		result = -ENODATA;

out_close_connection:
	rtnl_close(&rth);
out:
	return result;
}

static void decline_ia(struct ia *ia)
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;

	ia->state = IAS_DECLINE;
	//console_printf(LOC_FMT"declining IA: %s-%lu\n" , LOC_ARG , iastr(ia->conf->type), ia->conf->iaid);
	debug_printf(LOG_DEBUG, FNAME, "declining IA: %s-%lu", iastr(ia->conf->type), ia->conf->iaid);

    /* cancel the current event for the prefix. */
	if (ia->evdata) 
	{
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

    /* we don't need a timer for the IA (see comments in a_timo()) */
	if (ia->timer)
		dhcp6_remove_timer(&ia->timer);

    if ((ev = dhcp6_create_event(ia->ifp, DHCP6S_DECLINE)) == NULL) {
		//console_printf(LOC_FMT"failed to create a new event\n" , LOC_ARG);
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);
	//console_printf(LOC_FMT"new event: %p\n" , LOC_ARG , ev);

    if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		//console_printf(LOC_FMT"failed to create a new event timer\n" , LOC_ARG);
		debug_printf(LOG_NOTICE, FNAME,"failed to create a new event timer");
		goto fail;
	}

	if (duidcpy(&ev->serverid, &ia->serverid)) {
		debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) 
	{
		//console_printf(LOC_FMT"failed to create a new event data\n" , LOC_ARG);
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	evd->event = ev;
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	iaparam.iaid = ia->conf->iaid;
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;

	if (ia->ctl && ia->ctl->reestablish_data) 
	{
		if ((*ia->ctl->reestablish_data)(ia->ctl, &iaparam, &ia->evdata, evd)) 
		{
			//console_printf(LOC_FMT"failed to make reestablish data\n" , LOC_ARG);
			debug_printf(LOG_NOTICE, FNAME, "failed to make reestablish data");
			goto fail;
		}
	}

	if (ia->authparam != NULL) 
	{
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) 
		{
			//console_printf(LOC_FMT"failed to copy authparam\n" , LOC_ARG);
			debug_printf(LOG_WARNING, FNAME, "failed to copy authparam");
			goto fail;
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	ia->evdata = evd;

	client6_send(ev);
	return;

fail:
	if (ev)
		dhcp6_remove_event(ev);
	return;
}

void remove_declined_address(struct iactl **ctlp , struct dhcp6_statefuladdr *addr);

void reset_ia_timer(struct dhcp6_listval *iav , iatype_t iatype , struct dhcp6_if *ifp , struct duid *serverid)
{
	struct ia_conf *iac;
	struct ia *ia;
	struct timeval timo;

	if((iac = find_iaconf(&ifp->iaconf_list, iatype, iav->val_ia.iaid)) == NULL)
		return;

	if((ia = get_ia(iatype, ifp, iac, iav, serverid)) == NULL)
		return;

	if(ia->timer == NULL)
		ia->timer = dhcp6_add_timer(ia_timo, ia);

	if(ia->timer == NULL)
		return;

	timo.tv_sec = ia->t2 - ia->t1;
	timo.tv_usec = 0;
	dhcp6_set_timer(&timo, ia->timer);

	ia->state = IAS_ACTIVE;
	//console_printf(LOC_FMT"reset timer (t2: %d, t1: %d)\n" , LOC_ARG , ia->t2 , ia->t1);
}
#endif //DHCPV6_LOGO

#ifdef DHCPV6_LOGO
int
#else
void
#endif
update_ia(iatype, ialist, ifp, serverid, authparam)
	iatype_t iatype;
	struct dhcp6_list *ialist;
	struct dhcp6_if *ifp;
	struct duid *serverid;
	struct authparam *authparam;
{
	struct ia *ia;
	struct ia_conf *iac;
	struct iapd_conf *iapdc;
	struct iana_conf *ianac;
	struct dhcp6_listval *iav, *siav;
	struct timeval timo;
	struct statefuladdr *sa;
	struct siteprefix *sp;

	for (iav = TAILQ_FIRST(ialist); iav; iav = TAILQ_NEXT(iav, link)) {
		/* if we're not interested in this IA, ignore it. */
		if ((iac = find_iaconf(&ifp->iaconf_list, iatype,
		    iav->val_ia.iaid)) == NULL) {
			continue;
		}

		/* validate parameters */
		/*
		 * If a client receives an IA_NA with T1 greater than T2, and
		 * both T1 and T2 are greater than 0, the client discards the
		 * IA_NA option and processes the remainder of the message as
		 * though the server had not included the invalid IA_NA option.
		 * [RFC3315 22.4]
		 * We apply the same rule to IA_PD as well.
		 */
		if (iav->val_ia.t2 != 0 && iav->val_ia.t1 > iav->val_ia.t2) {
			debug_printf(LOG_INFO, FNAME,
			    "invalid IA: T1(%lu) > T2(%lu)",
			    iav->val_ia.t1, iav->val_ia.t2);
			continue;
		}

		/* locate the local IA or make a new one */
		ia = get_ia(iatype, ifp, iac, iav, serverid);
		if (ia == NULL) {
			debug_printf(LOG_WARNING, FNAME, "failed to get an IA "
			    "type: %s, ID: %u", iastr(iac->type), iac->iaid);
			continue;
		}

		/* update authentication parameters */
		if (update_authparam(ia, authparam)) {
			debug_printf(LOG_WARNING, FNAME, "failed to update "
			    "authentication param for IA "
			    "type: %s, ID: %u", iastr(iac->type), iac->iaid);
			remove_ia(ia);
			client6_start(ifp);
			continue;
		}

		/* update IA configuration information */
		for (siav = TAILQ_FIRST(&iav->sublist); siav;
		    siav = TAILQ_NEXT(siav, link)) {
			switch (siav->type) {
			case DHCP6_LISTVAL_PREFIX6:
				/* add or update the prefix */
				iapdc = (struct iapd_conf *)iac;
				if (update_prefix(ia, &siav->val_prefix6,
				    &iapdc->iapd_pif_list, ifp, &ia->ctl,
				    callback, &sp, PREFIX_TIMER_SET_NOW)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to update a prefix %s/%d",
					    in6addr2str(&siav->val_prefix6.addr, 0),
					    siav->val_prefix6.plen);
#ifdef DHCPV6_LOGO
					return -EINVAL;
#endif
				}
				ra_pfxlen = siav->val_prefix6.plen;	/* not used */
				break;
			case DHCP6_LISTVAL_STATEFULADDR6:
				ianac = (struct iana_conf *)iac;
				if (update_address(ia, &siav->val_statefuladdr6,
				    ifp, &ia->ctl, callback, &sa, ADDR_TIMER_SET_NOW)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to update an address %s",
					    in6addr2str(&siav->val_statefuladdr6.addr, 0));
				}
#ifdef DHCPV6_LOGO
				//wait for kernel DAD procedure, 1 sec should be enough
				sleep(1);
				{
				int result;
				if((result = check_address_status(ifp->ifname , &siav->val_statefuladdr6.addr)) < 0)
				{
					remove_declined_address(&ia->ctl , &siav->val_statefuladdr6);
					decline_ia(ia);
					goto nextia;
				}
				else
				{
				}
				}
#endif
				break;
			case DHCP6_LISTVAL_STCODE:
				debug_printf(LOG_INFO, FNAME,
				    "status code for %s-%lu: %s",
				    iastr(iatype), iav->val_ia.iaid,
				    dhcp6_stcodestr(siav->val_num16));
				if ((ia->state == IAS_RENEW ||
				    ia->state == IAS_REBIND) &&
				    siav->val_num16 == DH6OPT_STCODE_NOBINDING) {
					/*
					 * For each IA in the original Renew or
					 * Rebind message, the client
					 * sends a Request message if the IA
					 * contained a Status Code option
					 * with the NoBinding status.
					 * [RFC3315 18.1.8]
					 * XXX: what about the PD case?
					 */
					debug_printf(LOG_INFO, FNAME,
					    "receive NoBinding against "
					    "renew/rebind for %s-%lu",
					    iastr(ia->conf->type),
					    ia->conf->iaid);
					reestablish_ia(ia);
					goto nextia;
				}
				break;
			default:
				debug_printf(LOG_ERR, FNAME, "impossible case");
				goto nextia;
			}
		}

		/* see if this IA is still valid.  if not, remove it. */
		if (ia->ctl == NULL || !(*ia->ctl->isvalid)(ia->ctl)) {
			debug_printf(LOG_DEBUG, FNAME, "IA %s-%lu is invalidated",
			    iastr(ia->conf->type), ia->conf->iaid);
			remove_ia(ia);
			client6_start(ifp);
			continue;
		}

		/* if T1 or T2 is 0, determine appropriate values locally. */
		if (ia->t1 == 0 || ia->t2 == 0) {
			u_int32_t duration;

			if (ia->ctl && ia->ctl->duration)
				duration = (*ia->ctl->duration)(ia->ctl);
			else
				duration = 1800; /* 30min. XXX: no rationale */

			if (ia->t1 == 0) {
				if (duration == DHCP6_DURATION_INFINITE)
					ia->t1 = DHCP6_DURATION_INFINITE;
				else
					ia->t1 = duration / 2;
			}
			if (ia->t2 == 0) {
				if (duration == DHCP6_DURATION_INFINITE)
					ia->t2 = DHCP6_DURATION_INFINITE;
				else
					ia->t2 = duration * 4 / 5;
			}

			/* make sure T1 <= T2 */
			if (ia->t1 > ia->t2)
				ia->t1 = ia->t2 * 5 / 8;

			debug_printf(LOG_INFO, FNAME, "T1(%lu) and/or T2(%lu) "
			    "is locally determined",  ia->t1, ia->t2);
		}

		/*
		 * Be proactive for too-small timeout values.  Note that
		 * the adjusted values may make some information expire
		 * without renewal.
		 */
		if (ia->t2 < DHCP6_DURATION_MIN) {
			debug_printf(LOG_INFO, FNAME, "T1 (%lu) or T2 (%lu) "
			    "is too small", ia->t1, ia->t2);
			ia->t2 = DHCP6_DURATION_MIN;
			ia->t1 = ia->t2 * 5 / 8;
			debug_printf(LOG_INFO, "", "  adjusted to %lu and %lu",
			    ia->t1, ia->t2);
		}

		/* set up a timer for this IA. */
		if (ia->t1 == DHCP6_DURATION_INFINITE) {
			if (ia->timer)
				dhcp6_remove_timer(&ia->timer);
		} else {
			if (ia->timer == NULL)
				ia->timer = dhcp6_add_timer(ia_timo, ia);
			if (ia->timer == NULL) {
				debug_printf(LOG_ERR, FNAME,
				    "failed to add IA timer");
				remove_ia(ia); /* XXX */
				continue;
			}
			timo.tv_sec = ia->t1;
			timo.tv_usec = 0;
			dhcp6_set_timer(&timo, ia->timer);
		}

		ia->state = IAS_ACTIVE;

	  nextia:
		;
	}

#ifdef DHCPV6_LOGO
	return 0;
#endif
}

void
update_ia_pdna(optinfo, ifp, authparam)
	struct dhcp6_optinfo *optinfo;
	struct dhcp6_if *ifp;
	struct authparam *authparam;
{
	iacheck_t rtn;
	struct ia *ia_pd = NULL, *ia_na = NULL;
	struct timeval timo;
	iatype_t iatype;
	int binding_flg = 0;

	/* IA-NA check -> IA-NA Address nothing    -> IA-PD Prefix (LAN & WAN) */
	/* IA-NA check -> IA-NA Address exist(WAN) -> IA-PD Prefix (LAN) */
	iatype = IATYPE_NA;
	rtn = check_update_ia(&ia_na, iatype, &g_sp, &g_sa, &optinfo->iana_list, ifp, &optinfo->serverID, authparam);
	/* IA-NA return check */
	if (rtn == IA_NOBINDING) {
		binding_flg = 1;
	} else if (rtn == IA_NG) {
		/* WAN: PD Prefix route */
		debug_printf(LOG_NOTICE, FNAME, "IA-NA prefix nothing");
		iana_addr_nothing = 1;
	} else {
		iana_addr_nothing = 0;
	}

	iatype = IATYPE_PD;
	rtn = check_update_ia(&ia_pd, iatype, &g_sp, &g_sa, &optinfo->iapd_list, ifp, &optinfo->serverID, authparam);

	/* IA-PD return check */
	if ((rtn == IA_NOBINDING) || (binding_flg)) {
		reestablish_ia_pd_na(ia_pd);
		/* "g_sa" and "g_sp" is released when Reply is received. */
		return;
	} else if (rtn == IA_NG) {
		debug_printf(LOG_NOTICE, FNAME, "IA-PD prefix nothing");
	}

	if ((ia_pd) && (ia_na)) {
		/* set up a timer for this IA. */
		if ((ia_pd->t1 == DHCP6_DURATION_INFINITE) &&
		    (ia_na->t1 == DHCP6_DURATION_INFINITE)) {
			if (ia_pd->timer)
				dhcp6_remove_timer(&ia_pd->timer);
		} else {
			/* timer set ia_pd only */
			if (ia_pd->timer == NULL) {
				ia_pd->timer = dhcp6_add_timer(ia_pd_na_timo, ia_pd);

				if (ia_pd->timer == NULL) {
					debug_printf(LOG_ERR, FNAME,
					    "failed to add IA timer");
					goto update_ia_err;
				}

			}
			if (ia_pd->t1 > ia_na->t1)
				timo.tv_sec = ia_na->t1;
			else
				timo.tv_sec = ia_pd->t1;

			timo.tv_usec = 0;

			dhcp6_set_timer(&timo, ia_pd->timer);
		}
		ia_pd->state = IAS_ACTIVE;
		ia_na->state = IAS_ACTIVE;

		if ((g_sp->prefix.vltime == DHCP6_DURATION_INFINITE) ||
		    (g_sa->addr.vltime == DHCP6_DURATION_INFINITE)) {
			if (g_sp->timer)
				dhcp6_remove_timer(&g_sp->timer);
			return;
		}

		/* timer set g_sp(prefix) only */
		if (g_sp->timer == NULL) {
			g_sp->timer = dhcp6_add_timer(sa_sp_timo, g_sp);
			if (g_sp->timer == NULL) {
				debug_printf(LOG_ERR, FNAME,
				    "failed to add prefix timer");
				goto ia_vl_timer_err;
			}
		}

		if (g_sp->prefix.vltime < g_sa->addr.vltime)
			timo.tv_sec = g_sa->addr.vltime;
		else
			timo.tv_sec = g_sp->prefix.vltime;

		timo.tv_usec = 0;

		dhcp6_set_timer(&timo, g_sp->timer);

	} else if (ia_pd) {
		debug_printf(LOG_INFO, FNAME, "IA-NA nothing. IA-PD only update.");
		update_ia(IATYPE_PD, &optinfo->iapd_list, ifp,
		    &optinfo->serverID, authparam);
	} else if (ia_na) {
		debug_printf(LOG_INFO, FNAME, "IA-PD nothing. IA-NA only update.");
		update_ia(IATYPE_NA, &optinfo->iana_list, ifp,
		    &optinfo->serverID, authparam);
	} else {
		debug_printf(LOG_ERR, FNAME, "update IA nothing.");
		goto update_ia_err;
	}

	return;

ia_vl_timer_err:
	if (ia_pd->timer)
		dhcp6_remove_timer(&ia_pd->timer);

update_ia_err:
	if (g_sp) {
		call_remove_siteprefix(g_sp);
		g_sp = NULL;
	}
	if (g_sa) {
		call_remove_addr(g_sa);
		g_sa = NULL;
	}

	if (ia_pd)
		remove_ia(ia_pd);

	if (ia_na)
		remove_ia(ia_na);

	client6_start(ifp);

	return;
}

/*
 * check_update_ia(): Use update_ia_pdna() only
 * return 0: Success (get IA)
 *        1: Failure (no get IA: Solicit Message send)
 *        2: Failure (Request Message send)
 */
static iacheck_t
check_update_ia(p_ia, iatype, p_sp, p_sa, ialist, ifp, serverid, authparam)
	struct ia **p_ia;
	iatype_t iatype;
	struct siteprefix **p_sp;
	struct statefuladdr **p_sa;
	struct dhcp6_list *ialist;
	struct dhcp6_if *ifp;
	struct duid *serverid;
	struct authparam *authparam;
{
	struct ia *ia;
	struct dhcp6_listval *iav, *siav;
	struct ia_conf *iac;
	struct iapd_conf *iapdc;
	struct siteprefix *sp = NULL;
	struct statefuladdr *sa = NULL;

	int nobinding_flg = 0;
	int ia_cnt = 0;

	/* Cattion: Both IA-PD / IA-NA do not support multiple data.*/
	for (iav = TAILQ_FIRST(ialist); iav; iav = TAILQ_NEXT(iav, link)) {
		/* if we're not interested in this IA, ignore it. */
		if ((iac = find_iaconf(&ifp->iaconf_list, iatype,
		    iav->val_ia.iaid)) == NULL) {
			continue;
		}

		/* locate the local IA or make a new one */
		ia = get_ia(iatype, ifp, iac, iav, serverid);
		if (ia == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to get an IA type: %s, ID: %u",
			    iastr(iac->type), iac->iaid);
			continue;
		}

		/* validate parameters */

		/*
		 * If a client receives an IA_NA with T1 greater than T2, and
		 * both T1 and T2 are greater than 0, the client discards the
		 * IA_NA option and processes the remainder of the message as
		 * though the server had not included the invalid IA_NA option.
		 * [RFC3315 22.4]
		 * If a requesting router receives an IA_PD with T1 greater than T2, and
		 * both T1 and T2 are greater than 0, the client discards the IA_PD
		 * option and processes the remainder of the message as though the
		 * delegating router had not included the IA_PD option.
		 * [RFC3633 9]
		 */
		if (iav->val_ia.t2 != 0 && iav->val_ia.t1 > iav->val_ia.t2) {
			debug_printf(LOG_INFO, FNAME,
			    "invalid IA: type: %s T1(%lu) > T2(%lu)",
			    iastr(iac->type), iav->val_ia.t1, iav->val_ia.t2);
			remove_ia(ia);
			continue;
		}

		/* update authentication parameters */
		if (update_authparam(ia, authparam)) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to update authentication param for IA "
			    "type: %s, ID: %u", iastr(iac->type), iac->iaid);
			remove_ia(ia);
			continue;
		}

		/* update IA configuration information */
		for (siav = TAILQ_FIRST(&iav->sublist); siav;
		    siav = TAILQ_NEXT(siav, link)) {
			switch (siav->type) {
			case DHCP6_LISTVAL_PREFIX6:
				/* add or update the prefix */
				iapdc = (struct iapd_conf *)iac;
				if (update_prefix(ia, &siav->val_prefix6,
				    &iapdc->iapd_pif_list, ifp, &ia->ctl,
				    callback, &sp, PREFIX_TIMER_SET_LATER)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to update a prefix %s/%d",
					    in6addr2str(&siav->val_prefix6.addr, 0),
					    siav->val_prefix6.plen);
				}
				break;
			case DHCP6_LISTVAL_STATEFULADDR6:
				if (update_address(ia, &siav->val_statefuladdr6,
				    ifp, &ia->ctl, callback, &sa, ADDR_TIMER_SET_LATER)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to update an address %s",
					    in6addr2str(&siav->val_statefuladdr6.addr, 0));
				}
				break;
			case DHCP6_LISTVAL_STCODE:
				debug_printf(LOG_INFO, FNAME,
				    "status code for %s-%lu: %s",
				    iastr(iatype), iav->val_ia.iaid,
				    dhcp6_stcodestr(siav->val_num16));

				if ((ia->state == IAS_RENEW || ia->state == IAS_REBIND) &&
				    siav->val_num16 == DH6OPT_STCODE_NOBINDING) {
					/*
					 * For each IA in the original Renew or Rebind message,
					 * the client sends a Request message if the IA
					 * contained a Status Code option with the NoBinding status.
					 * [RFC3315 18.1.8]
					 */
					debug_printf(LOG_INFO, FNAME,
					    "receive NoBinding against "
					    "renew/rebind for %s-%lu",
					    iastr(ia->conf->type),
					    ia->conf->iaid);

					/* Request Message send flg */
					nobinding_flg = 1;
					goto nextia;
				}
				break;
			default:
				debug_printf(LOG_ERR, FNAME, "impossible case");
				break;
			}
		}

		/* see if this IA is still valid.  if not, remove it. */
		if (ia->ctl == NULL || !(*ia->ctl->isvalid)(ia->ctl)) {
			debug_printf(LOG_DEBUG, FNAME, "IA %s-%lu is invalidated",
			    iastr(ia->conf->type), ia->conf->iaid);
			remove_ia(ia);
			continue;
		}

		/* if T1 or T2 is 0, determine appropriate values locally. */
		if (ia->t1 == 0 || ia->t2 == 0) {
			u_int32_t duration;

			if (ia->ctl && ia->ctl->duration)
				duration = (*ia->ctl->duration)(ia->ctl);
			else
				duration = 1800; /* 30min. XXX: no rationale */

			if (ia->t1 == 0) {
				if (duration == DHCP6_DURATION_INFINITE)
					ia->t1 = DHCP6_DURATION_INFINITE;
				else
					ia->t1 = duration / 2;
			}
			if (ia->t2 == 0) {
				if (duration == DHCP6_DURATION_INFINITE)
					ia->t2 = DHCP6_DURATION_INFINITE;
				else
					ia->t2 = duration * 4 / 5;
			}

			/* make sure T1 <= T2 */
			if (ia->t1 > ia->t2)
				ia->t1 = ia->t2 * 5 / 8;

			debug_printf(LOG_INFO, FNAME, "T1(%lu) and/or T2(%lu) "
			    "is locally determined",  ia->t1, ia->t2);
		}

		/*
		 * Be proactive for too-small timeout values.  Note that
		 * the adjusted values may make some information expire
		 * without renewal.
		 */
		if (ia->t2 < DHCP6_DURATION_MIN) {
			debug_printf(LOG_INFO, FNAME, "T1 (%lu) or T2 (%lu) "
			    "is too small", ia->t1, ia->t2);
			ia->t2 = DHCP6_DURATION_MIN;
			ia->t1 = ia->t2 * 5 / 8;
			debug_printf(LOG_INFO, "", "  adjusted to %lu and %lu",
			    ia->t1, ia->t2);
		}

nextia:
		/* Cattion: Both IA-PD / IA-NA do not support multiple data.*/
		*p_ia = ia;

		if (iatype == IATYPE_NA) {
			if (sa != NULL)
				*p_sa = sa;
		} else {
			if (sp != NULL)
				*p_sp = sp;
		}
		break;
	}

	if (nobinding_flg) {
		return(IA_NOBINDING);
	}
	else if (*p_ia == NULL) {
		return(IA_NG);
	}

	return(IA_OK);
}

static int
update_authparam(ia, authparam)
	struct ia *ia;
	struct authparam *authparam;
{
	if (authparam == NULL)
		return (0);

	if (ia->authparam == NULL) {
		if ((ia->authparam = copy_authparam(authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			return (-1);
		}
		return (0);
	}

	/* update the previous RD value and flags */
	ia->authparam->prevrd = authparam->prevrd;
	ia->authparam->flags = authparam->flags;

	return (0);
}

static void
reestablish_ia(ia)
	struct ia *ia;
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;

	debug_printf(LOG_DEBUG, FNAME, "re-establishing IA: %s-%lu", 
	    iastr(ia->conf->type), ia->conf->iaid);

	if (ia->state != IAS_RENEW && ia->state != IAS_REBIND) {
		debug_printf(LOG_ERR, FNAME, "internal error (invalid IA status)");
		exit(1);	/* XXX */
	}

	/* cancel the current event for the prefix. */
	if (ia->evdata) {
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

	/* we don't need a timer for the IA (see comments in ia_timo()) */
	if (ia->timer)
		dhcp6_remove_timer(&ia->timer);

	if ((ev = dhcp6_create_event(ia->ifp, DHCP6S_REQUEST)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	evd->event = ev;
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	if (duidcpy(&ev->serverid, &ia->serverid)) {
		debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
		goto fail;
	}

	iaparam.iaid = ia->conf->iaid;
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;

	if (ia->ctl && ia->ctl->reestablish_data) {
		if ((*ia->ctl->reestablish_data)(ia->ctl, &iaparam,
		    &ia->evdata, evd)) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to make reestablish data");
			goto fail;
		}
	}

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	ia->evdata = evd;

	client6_send(ev);

	return;

  fail:
	if (ev)
		dhcp6_remove_event(ev);

	return;
}

static void
reestablish_ia_pd_na(ia)
	struct ia *ia;
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;
	struct ia_conf *iac;
	struct ia *iaconf_ia, *next_ia;

	debug_printf(LOG_DEBUG, FNAME, "re-establishing IA: PD/NA-%lu",
	    ia->conf->iaid);

	if (ia->state != IAS_RENEW && ia->state != IAS_REBIND) {
		debug_printf(LOG_ERR, FNAME, "internal error (invalid IA status)");
		exit(1);	/* XXX */
	}

	/* we don't need a timer for the IA (see comments in ia_timo()) */
	if (ia->timer)
		dhcp6_remove_timer(&ia->timer);

	if ((ev = dhcp6_create_event(ia->ifp, DHCP6S_REQUEST)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if (duidcpy(&ev->serverid, &ia->serverid)) {
		debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
		goto fail;
	}

	for (iac = TAILQ_FIRST(&ia->ifp->iaconf_list); iac; iac = TAILQ_NEXT(iac, link)) {
		if (TAILQ_EMPTY(&iac->iadata)) {
			debug_printf(LOG_NOTICE, FNAME, "failed to empty iadata");
			continue;
		}

		for (iaconf_ia = TAILQ_FIRST(&iac->iadata); iaconf_ia; iaconf_ia = next_ia) {
			next_ia = TAILQ_NEXT(iaconf_ia, link);

			/* cancel the current event for the prefix. */
			if (iaconf_ia->evdata) {
				TAILQ_REMOVE(&iaconf_ia->evdata->event->data_list, iaconf_ia->evdata, link);
				if (iaconf_ia->evdata->destructor)
					iaconf_ia->evdata->destructor(iaconf_ia->evdata);
				free(iaconf_ia->evdata);
				iaconf_ia->evdata = NULL;
			}

			memset(&iaparam, 0, sizeof(iaparam));
			iaparam.iaid = iaconf_ia->conf->iaid;
			iaparam.t1 = iaconf_ia->t1;
			iaparam.t2 = iaconf_ia->t2;

			if ((evd = malloc(sizeof(*evd))) == NULL) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to create a new event data");
				goto fail;
			}
			memset(evd, 0, sizeof(*evd));
			evd->event = ev;
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

			if (iaconf_ia->ctl && iaconf_ia->ctl->reestablish_data) {
				if ((*iaconf_ia->ctl->reestablish_data)(iaconf_ia->ctl, &iaparam,
				    &iaconf_ia->evdata, evd)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to make reestablish data(%s)", iastr(iaconf_ia->conf->type));
					goto fail;
				}
			}

			iaconf_ia->evdata = evd;
		}
	}

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	client6_send(ev);

	return;

  fail:
	if (ev)
		dhcp6_remove_event(ev);

	return;
}

#ifdef DHCPV6_LOGO
static void rebind_ia(struct ia *ia);
static void confirm_ia(struct ia *ia , int type)
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;

	if(ia->state != IAS_ACTIVE)
		return; //this ia not active, do nothing

	if(type == IATYPE_PD)
	{
		//Hans asks us that do not wait T2 timeout for PD, just rebind immediately. (tom, 20130625)
		rebind_ia(ia);
		return;
	}

	ia->state = IAS_CNF;

	//console_printf(LOC_FMT"confirming IA: %s-%lu\n" , LOC_ARG , iastr(ia->conf->type), ia->conf->iaid);
	debug_printf(LOG_DEBUG, FNAME, "confirming IA: %s-%lu", iastr(ia->conf->type), ia->conf->iaid);

    /* cancel the current event for the prefix. */
	if (ia->evdata) 
	{
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

    /* we don't need a timer for the IA (see comments in ia_timo()) */
	if (ia->timer)
		dhcp6_remove_timer(&ia->timer);

    if ((ev = dhcp6_create_event(ia->ifp, DHCP6S_CNF)) == NULL) {
		//console_printf(LOC_FMT"failed to create a new event\n" , LOC_ARG);
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}

	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);
	//console_printf(LOC_FMT"new event: %p\n" , LOC_ARG , ev);

    if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		//console_printf(LOC_FMT"failed to create a new event timer\n" , LOC_ARG);
		debug_printf(LOG_NOTICE, FNAME,"failed to create a new event timer");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) 
	{
		//console_printf(LOC_FMT"failed to create a new event data\n" , LOC_ARG);
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	evd->event = ev;
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	iaparam.iaid = ia->conf->iaid;
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;

	if (ia->ctl && ia->ctl->reestablish_data) 
	{
		if ((*ia->ctl->reestablish_data)(ia->ctl, &iaparam, &ia->evdata, evd)) 
		{
			//console_printf(LOC_FMT"failed to make reestablish data\n" , LOC_ARG);
			debug_printf(LOG_NOTICE, FNAME, "failed to make reestablish data");
			goto fail;
		}
	}

	if (ia->authparam != NULL) 
	{
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) 
		{
			//console_printf(LOC_FMT"failed to copy authparam\n" , LOC_ARG);
			debug_printf(LOG_WARNING, FNAME, "failed to copy authparam");
			goto fail;
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	ia->evdata = evd;

	client6_send(ev);
	return;

fail:
	if (ev)
		dhcp6_remove_event(ev);

	//console_printf(LOC_FMT"\n" , LOC_ARG);
	return;
}

void confirm_all_ia(struct dhcp6_if *ifp);
void confirm_all_ia(struct dhcp6_if *ifp)
{   
	struct ia_conf *iac;
	struct ia *ia, *ia_next;

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac ; iac = TAILQ_NEXT(iac, link)) 
	{
		for (ia = TAILQ_FIRST(&iac->iadata); ia; ia = ia_next) 
		{
			ia_next = TAILQ_NEXT(ia, link);

			//console_printf(LOC_FMT">>confirm_ia\n" , LOC_ARG);
			confirm_ia(ia , iac->type);
			//console_printf(LOC_FMT"<<confirm_ia\n" , LOC_ARG);
		}
	}
}
#endif

void
check_binding_ia(optinfo, ifp)
	struct dhcp6_optinfo *optinfo;
	struct dhcp6_if *ifp;
{
	/* IA-NA check */
	check_binding_ia_na(optinfo, ifp);

	/* IA-PD check */
	check_binding_ia_pd(optinfo, ifp);
}

static void
check_binding_ia_pd(optinfo, ifp)
	struct dhcp6_optinfo *optinfo;
	struct dhcp6_if *ifp;
{
	struct ia *ia;
	struct ia_conf *iac;
	struct iactl_pd *iac_pd;

	struct dhcp6_listval *iav, *siav;

	struct siteprefix *sp, *next_sp;

	int remove_prefix_flg;	/* 1: remove_prefix() run */
	int remove_ia_cnt;      /* 0: remove_ia() run */


	/* caution: If multiple IAIDs are notified, it is not subject to operation */
	for (iav = TAILQ_FIRST(&optinfo->iapd_list); iav; iav = TAILQ_NEXT(iav, link)) {

		/* Get "ia_conf" matching iatype and iaid */
		if ((iac = find_iaconf(&ifp->iaconf_list, IATYPE_PD,
		    iav->val_ia.iaid)) == NULL) {
			continue;
		}

		if ((ia = find_ia(iac, IATYPE_PD, iav->val_ia.iaid)) == NULL) {
			debug_printf(LOG_INFO, FNAME, "IA-%s not found", iastr(IATYPE_PD));
			continue;
		}

		iac_pd = (struct iactl_pd *)ia->ctl;

		remove_ia_cnt = 0;
		for (sp = TAILQ_FIRST(&iac_pd->siteprefix_head); sp; sp = next_sp) {
			next_sp = TAILQ_NEXT(sp, link);
			remove_prefix_flg = 1;
			remove_ia_cnt++;

			/* IA-NA: WAN IPv6 address all delete -> IA-PD: LAN IPv6 address delete */
			if (iana_addr_delete == 0) {
				for (siav = TAILQ_FIRST(&iav->sublist); siav;
				    siav = TAILQ_NEXT(siav, link)) {
					if (siav->type != DHCP6_LISTVAL_PREFIX6) {
						continue;
					}
					if (siav->val_prefix6.vltime != DHCP6_DURATION_INFINITE &&
					    (siav->val_prefix6.pltime == DHCP6_DURATION_INFINITE ||
					    siav->val_prefix6.pltime > siav->val_prefix6.vltime)) {
						debug_printf(LOG_INFO, FNAME, "invalid prefix %s/%d: "
						    "pltime (%lu) is larger than vltime (%lu)",
						    in6addr2str(&siav->val_prefix6.addr, 0), siav->val_prefix6.plen,
						    siav->val_prefix6.pltime, siav->val_prefix6.vltime);
						continue;
					}
					if ((IN6_ARE_ADDR_EQUAL(&sp->prefix.addr, &siav->val_prefix6.addr)) &&
					    (sp->prefix.plen == siav->val_prefix6.plen)) {
						remove_prefix_flg = 0;
						break;
					}
				}
			}
			if (remove_prefix_flg) {
				if(g_sp == sp)
					g_sp = NULL;
				call_remove_siteprefix(sp);
				remove_ia_cnt--;
			}
		}
		if (remove_ia_cnt == 0) {
			remove_ia(ia);
		}
	}

	return;
}

static void
check_binding_ia_na(optinfo, ifp)
	struct dhcp6_optinfo *optinfo;
	struct dhcp6_if *ifp;
{
	struct ia *ia;
	struct ia_conf *iac;
	struct iactl_na *iac_na;

	struct dhcp6_listval *iav, *siav;

	struct statefuladdr *sa, *next_sa;

	int remove_addr_flg;	/* 1: remove_addr() run */
	int remove_ia_cnt;	/* 0: remove_ia() run */

	/* caution: If multiple IAIDs are notified, it is not subject to operation */
	for (iav = TAILQ_FIRST(&optinfo->iana_list); iav; iav = TAILQ_NEXT(iav, link)) {

		/* Get "ia_conf" matching iatype and iaid */
		if ((iac = find_iaconf(&ifp->iaconf_list, IATYPE_NA,
		    iav->val_ia.iaid)) == NULL) {
			continue;
		}

		if ((ia = find_ia(iac, IATYPE_NA, iav->val_ia.iaid)) == NULL) {
			debug_printf(LOG_INFO, FNAME, "IA-%s not found", iastr(IATYPE_NA));
			continue;
		}

		iac_na = (struct iactl_na *)ia->ctl;

		remove_ia_cnt = 0;
		for (sa = TAILQ_FIRST(&iac_na->statefuladdr_head); sa; sa = next_sa) {
			next_sa = TAILQ_NEXT(sa, link);
			remove_addr_flg = 1;
			remove_ia_cnt++;
			for (siav = TAILQ_FIRST(&iav->sublist); siav;
			    siav = TAILQ_NEXT(siav, link)) {
				if (siav->type != DHCP6_LISTVAL_STATEFULADDR6) {
					continue;
				}
				if (siav->val_statefuladdr6.vltime != DHCP6_DURATION_INFINITE &&
				    (siav->val_statefuladdr6.pltime == DHCP6_DURATION_INFINITE ||
				     siav->val_statefuladdr6.pltime > siav->val_statefuladdr6.vltime)) {
					debug_printf(LOG_INFO, FNAME, "invalid address %s: "
					    "pltime (%lu) is larger than vltime (%lu)",
					    in6addr2str(&siav->val_statefuladdr6.addr, 0),
					    siav->val_statefuladdr6.pltime, siav->val_statefuladdr6.vltime);
					continue;
				}
				if (IN6_ARE_ADDR_EQUAL(&sa->addr.addr, &siav->val_statefuladdr6.addr)) {
					remove_addr_flg = 0;
					break;
				}
			}
			if (remove_addr_flg) {
				if(g_sa == sa)
					g_sa = NULL;
				call_remove_addr(sa);
				remove_ia_cnt--;
			}
		}
		if (remove_ia_cnt == 0) {
			remove_ia(ia);
			iana_addr_delete = 1;
		}
	}

	return;
}

static void
callback(ia)
	struct ia *ia;
{
	struct dhcp6_if *ifp;

	/* see if this IA is still valid.  if not, remove it. */
	if (ia->ctl == NULL || !(*ia->ctl->isvalid)(ia->ctl)) {
		debug_printf(LOG_DEBUG, FNAME, "IA %s-%lu is invalidated",
		    iastr(ia->conf->type), ia->conf->iaid);
		ifp = ia->ifp;
		remove_ia(ia);
		if (ifp != NULL)
			client6_start(ifp);
	}
}

#ifdef DHCPV6_LOGO
//add by rbj
//>>>> 
void
renew_all_ia(ifp)
	struct dhcp6_if *ifp;
{
	struct ia_conf *iac;
	struct ia *ia, *ia_next;

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac;
	    iac = TAILQ_NEXT(iac, link)) {
		for (ia = TAILQ_FIRST(&iac->iadata); ia; ia = ia_next) {
			ia_next = TAILQ_NEXT(ia, link);

			renew_ia(ia);
		}
	}
}
#endif

static void rebind_ia(struct ia *ia)
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;
	int dhcpstate;
	struct timeval timo;

	debug_printf(LOG_DEBUG, FNAME, "rebind IA: %s-%lu", iastr(ia->conf->type), ia->conf->iaid);

	/* cancel the current event for the prefix. */
	if (ia->evdata) 
	{
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

	ia->state = IAS_REBIND;
	dhcpstate = DHCP6S_REBIND;
	//timo.tv_sec = ia->t1 < ia->t2 ? ia->t2 - ia->t1 : 0;
	//timo.tv_usec = 0;
	//dhcp6_set_timer(&timo, ia->timer);
	dhcp6_remove_timer(&ia->timer);

	if ((ev = dhcp6_create_event(ia->ifp, dhcpstate)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	evd->event = ev;
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	//if (duidcpy(&ev->serverid, &ia->serverid)) {
	//	debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
	//	goto fail;
	//}

	iaparam.iaid = ia->conf->iaid;
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;

	if (ia->ctl && ia->ctl->rebind_data) {
		if ((*ia->ctl->rebind_data)(ia->ctl, &iaparam,
		    &ia->evdata, evd)) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to make renew data");
			goto fail;
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	ia->evdata = evd;
	client6_send(ev);

	return;
fail:
	if (ev)
		dhcp6_remove_event(ev);

	return;
}

void
rebind_all_ia(ifp)
	struct dhcp6_if *ifp;
{
	struct ia_conf *iac;
	struct ia *ia, *ia_next;

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac;
	    iac = TAILQ_NEXT(iac, link)) {
		for (ia = TAILQ_FIRST(&iac->iadata); ia; ia = ia_next) {
			ia_next = TAILQ_NEXT(ia, link);

			rebind_ia(ia);
		}
	}
}
//<<<<

void
release_all_ia(ifp)
	struct dhcp6_if *ifp;
{
	struct ia_conf *iac;
	struct ia *ia, *ia_next;
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;
	int ia_copy_flg = 0;		/* 1: ia copy process skip  */

	if ((ev = dhcp6_create_event(ifp, DHCP6S_RELEASE)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event timer");
		goto fail;
	}

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac;
	    iac = TAILQ_NEXT(iac, link)) {
		for (ia = TAILQ_FIRST(&iac->iadata); ia; ia = ia_next) {
			ia_next = TAILQ_NEXT(ia, link);

			/*
			 * release_ia() not used
			 *   Reason:
			 *     To send one IA-PD and one IA-NA Release message to the DHCPv6 server.
			 */
#if 0
			(void)release_ia(ia);
#else
			if (!ia_copy_flg) {
				if (duidcpy(&ev->serverid, &ia->serverid)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to copy server ID");
					goto fail;
				}
			}

			if ((evd = malloc(sizeof(*evd))) == NULL) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to create a new event data");
				goto fail;
			}
			memset(evd, 0, sizeof(*evd));
			memset(&iaparam, 0, sizeof(iaparam));
			iaparam.iaid = ia->conf->iaid;
			/* XXX: should we set T1/T2 to 0?  spec is silent on this. */
			iaparam.t1 = ia->t1;
			iaparam.t2 = ia->t2;

			if (ia->ctl && ia->ctl->release_data) {
				if ((*ia->ctl->release_data)(ia->ctl, &iaparam, NULL, evd)) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to make release data");
					goto fail;
				}
			}
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

			if ((ia->authparam != NULL) && (!ia_copy_flg)) {
				if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
					debug_printf(LOG_WARNING, FNAME,
					    "failed to copy authparam");
					goto fail;
				}
			}
			ia_copy_flg = 1;
#endif

			/*
			 * The client MUST stop using all of the addresses
			 * being released as soon as the client begins the
			 * Release message exchange process.
			 * [RFC3315 Section 18.1.6]
			 */
			remove_ia(ia);
		}
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	client6_send(ev);

	client6_start(ifp);

fail:
	if (ev)
		dhcp6_remove_event(ev);
}

static int
release_ia(ia)
	struct ia *ia;
{
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;

	debug_printf(LOG_DEBUG, FNAME, "release an IA: %s-%lu",
	    iastr(ia->conf->type), ia->conf->iaid);

	if ((ev = dhcp6_create_event(ia->ifp, DHCP6S_RELEASE))
	    == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if (duidcpy(&ev->serverid, &ia->serverid)) {
		debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	iaparam.iaid = ia->conf->iaid;
	/* XXX: should we set T1/T2 to 0?  spec is silent on this. */
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;

	if (ia->ctl && ia->ctl->release_data) {
		if ((*ia->ctl->release_data)(ia->ctl, &iaparam, NULL, evd)) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to make release data");
			goto fail;
		}
	}
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	client6_send(ev);
	return (0);

  fail:
	if (ev)
		dhcp6_remove_event(ev);

	return (-1);
}

static void
remove_ia(ia)
	struct ia *ia;
{
	struct ia_conf *iac = ia->conf;

	debug_printf(LOG_DEBUG, FNAME, "remove an IA: %s-%lu",
	    iastr(ia->conf->type), ia->conf->iaid);

	TAILQ_REMOVE(&iac->iadata, ia, link);

	duidfree(&ia->serverid);

	if (ia->timer)
		dhcp6_remove_timer(&ia->timer);

	if (ia->evdata) {
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

	if (ia->ctl && ia->ctl->cleanup)
		(*ia->ctl->cleanup)(ia->ctl);

	if (ia->authparam != NULL)
		free(ia->authparam);

	if (ngn_pd_mode == 1) {
		if (remove("/var/dhcp6c_nexthop") == -1) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to remove dhcp6c_nexthop file");
		}
	}

	free(ia);

#if 0
	if(ia->conf->type==0) //rebind
		//system("event WAN-4.DOWN");//IOL test
	{
		char tmp[128];
		sprintf(tmp,"event %s.DOWN",xifname);
		system(tmp);
	}

#endif
}

static struct dhcp6_timer *
ia_timo(arg)
	void *arg;
{
	struct ia *ia = (struct ia *)arg;
	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;
	struct timeval timo;
	int dhcpstate;

	debug_printf(LOG_DEBUG, FNAME, "IA timeout for %s-%lu, state=%s",
	    iastr(ia->conf->type), ia->conf->iaid, statestr(ia->state));

	/* cancel the current event for the prefix. */
	if (ia->evdata) {
		TAILQ_REMOVE(&ia->evdata->event->data_list, ia->evdata, link);
		if (ia->evdata->destructor)
			ia->evdata->destructor(ia->evdata);
		free(ia->evdata);
		ia->evdata = NULL;
	}

	switch (ia->state) {
	case IAS_ACTIVE:
		ia->state = IAS_RENEW;
		dhcpstate = DHCP6S_RENEW;
		timo.tv_sec = ia->t1 < ia->t2 ? ia->t2 - ia->t1 : 0;
		timo.tv_usec = 0;
		dhcp6_set_timer(&timo, ia->timer);
		break;
	case IAS_RENEW:
		ia->state = IAS_REBIND;
		dhcpstate = DHCP6S_REBIND;

		/*
		 * We need keep DUID for sending Release in this state.
		 * But we don't need a timer for the IA.  We'll just wait for a
		 * reply for the REBIND until all associated configuration
		 * parameters for this IA expire.
		 */
		dhcp6_remove_timer(&ia->timer);
		break;
	default:
		debug_printf(LOG_ERR, FNAME, "invalid IA state (%d)",
		    (int)ia->state);
		return (NULL);	/* XXX */
	}

	if ((ev = dhcp6_create_event(ia->ifp, dhcpstate)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if ((evd = malloc(sizeof(*evd))) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event data");
		goto fail;
	}
	memset(evd, 0, sizeof(*evd));
	evd->event = ev;
	TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

	if (ia->state == IAS_RENEW) {
		if (duidcpy(&ev->serverid, &ia->serverid)) {
			debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
			goto fail;
		}
	}

	iaparam.iaid = ia->conf->iaid;
	iaparam.t1 = ia->t1;
	iaparam.t2 = ia->t2;
	switch(ia->state) {
	case IAS_RENEW:
		if (ia->ctl && ia->ctl->renew_data) {
			if ((*ia->ctl->renew_data)(ia->ctl, &iaparam,
			    &ia->evdata, evd)) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to make renew data");
				goto fail;
			}
		}
		break;
	case IAS_REBIND:
		if (ia->ctl && ia->ctl->rebind_data) {
			if ((*ia->ctl->rebind_data)(ia->ctl, &iaparam,
			    &ia->evdata, evd)) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to make rebind data");
				goto fail;
			}
		}
		break;
	default:
		break;
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	ia->evdata = evd;

	switch(ia->state) {
	case IAS_RENEW:
	case IAS_REBIND:
#ifdef DHCPV6_LOGO
	case IAS_CNF:
	case IAS_DECLINE:
#endif
		client6_send(ev);
		break;
	case IAS_ACTIVE:
		/* what to do? */
		break;
	}

	return (ia->timer);

  fail:
	if (ev)
		dhcp6_remove_event(ev);

	return (NULL);
}

static struct dhcp6_timer *
ia_pd_na_timo(arg)
	void *arg;
{
	struct ia *ia = (struct ia *)arg;
	struct ia *iaconf_ia, *next_ia;
	struct ia_conf *iac;

	struct dhcp6_ia iaparam;
	struct dhcp6_event *ev;
	struct dhcp6_eventdata *evd;
	struct timeval timo;
	int dhcpstate;
	long wk_sec = 0;

	 debug_printf(LOG_DEBUG, FNAME, "IA timeout for PD/NA-%lu, state=%s",
		ia->conf->iaid, statestr(ia->state));

	switch (ia->state) {
	case IAS_ACTIVE:
		ia->state = IAS_RENEW;
		dhcpstate = DHCP6S_RENEW;
		break;
	case IAS_RENEW:
		ia->state = IAS_REBIND;
		dhcpstate = DHCP6S_REBIND;

		/*
		 * We need keep DUID for sending Release in this state.
		 * But we don't need a timer for the IA.  We'll just wait for a
		 * reply for the REBIND until all associated configuration
		 * parameters for this IA expire.
		 */
		dhcp6_remove_timer(&ia->timer);
		break;
	default:
		debug_printf(LOG_ERR, FNAME, "invalid IA state (%d)",
		    (int)ia->state);
		return (NULL);  /* XXX */
	}

	timo.tv_sec = 0;
	timo.tv_usec = 0;

	if ((ev = dhcp6_create_event(ia->ifp, dhcpstate)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
		goto fail;
	}
	TAILQ_INSERT_TAIL(&ia->ifp->event_list, ev, link);

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to create a new event timer");
		goto fail;
	}

	if (ia->state == IAS_RENEW) {
		if (duidcpy(&ev->serverid, &ia->serverid)) {
			debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
			goto fail;
		}
	}

	for (iac = TAILQ_FIRST(&ia->ifp->iaconf_list); iac; iac = TAILQ_NEXT(iac, link)) {
		if (TAILQ_EMPTY(&iac->iadata)) {
			debug_printf(LOG_NOTICE, FNAME, "failed to empty iadata");
			continue;
		}

		for (iaconf_ia = TAILQ_FIRST(&iac->iadata); iaconf_ia; iaconf_ia = next_ia) {
			next_ia = TAILQ_NEXT(iaconf_ia, link);

			/* cancel the current event for the prefix. */
			if (iaconf_ia->evdata) {
				TAILQ_REMOVE(&iaconf_ia->evdata->event->data_list, iaconf_ia->evdata, link);
				if (iaconf_ia->evdata->destructor)
					iaconf_ia->evdata->destructor(iaconf_ia->evdata);
				free(iaconf_ia->evdata);
				iaconf_ia->evdata = NULL;
			}

			memset(&iaparam, 0, sizeof(iaparam));
			iaparam.iaid = iaconf_ia->conf->iaid;
			iaparam.t1 = iaconf_ia->t1;
			iaparam.t2 = iaconf_ia->t2;

			if (ia->state == IAS_RENEW) {
				if (timo.tv_sec == 0) {
					timo.tv_sec = iaconf_ia->t1 < iaconf_ia->t2 ? iaconf_ia->t2 - iaconf_ia->t1 : 0;
				} else {
					wk_sec = iaconf_ia->t1 < iaconf_ia->t2 ? iaconf_ia->t2 - iaconf_ia->t1 : 0;

					if (wk_sec < timo.tv_sec) {
						timo.tv_sec = wk_sec;
					}
				}
			}

			if ((evd = malloc(sizeof(*evd))) == NULL) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to create a new event data");
				goto fail;
			}
			memset(evd, 0, sizeof(*evd));
			evd->event = ev;
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);

			switch(ia->state) {
			case IAS_RENEW:
				if (iaconf_ia->ctl && iaconf_ia->ctl->renew_data) {
					if ((*iaconf_ia->ctl->renew_data)(iaconf_ia->ctl, &iaparam,
					     &iaconf_ia->evdata, evd)) {
						debug_printf(LOG_NOTICE, FNAME,
						    "failed to make renew data(%s)",
						    iastr(iaconf_ia->conf->type));
						goto fail;
					}
				} else {
					debug_printf(LOG_NOTICE, FNAME, "renew function nothing");
					goto fail;
				}
				break;
			case IAS_REBIND:
				if (iaconf_ia->ctl && iaconf_ia->ctl->rebind_data) {
					if ((*iaconf_ia->ctl->rebind_data)(iaconf_ia->ctl, &iaparam,
					     &iaconf_ia->evdata, evd)) {
						debug_printf(LOG_NOTICE, FNAME,
						    "failed to make rebind data(%s)",
						    iastr(iaconf_ia->conf->type));
						goto fail;
					}
				} else {
					debug_printf(LOG_NOTICE, FNAME, "rebind function nothing");
					goto fail;
				}
				break;
			default:
				debug_printf(LOG_ERR, FNAME, "state out of range");
			}

			iaconf_ia->evdata = evd;

			/* state check */
			if ((iaconf_ia->state == IAS_ACTIVE) && (ia->state == IAS_RENEW)) {
				iaconf_ia->state = IAS_RENEW;
			} else if ((iaconf_ia->state == IAS_RENEW) && (ia->state == IAS_REBIND)) {
				iaconf_ia->state = IAS_REBIND;
			}
		}
	}

	if (ia->state == IAS_RENEW) {
		dhcp6_set_timer(&timo, ia->timer);
	}

	ev->timeouts = 0;
	dhcp6_set_timeoparam(ev);
	dhcp6_reset_timer(ev);

	if (ia->authparam != NULL) {
		if ((ev->authparam = copy_authparam(ia->authparam)) == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to copy authparam");
			goto fail;
		}
	}

	switch(ia->state) {
	case IAS_RENEW:
	case IAS_REBIND:
		client6_send(ev);
		break;
	case IAS_ACTIVE:
		break;
	}

	return (ia->timer);

fail:
	if (ev)
		dhcp6_remove_event(ev);

	return (NULL);
}

static struct dhcp6_timer *
sa_sp_timo(arg)
	void *arg;
{
	struct ia *ia;
	void (*callback)__P((struct ia *));
	int ifp_flg = 0;

	/* remove check */
	if (g_sp == NULL)
		return (NULL);

	/* sp */
	ia = g_sp->ctl->iacpd_ia;
	callback = g_sp->ctl->iacpd_callback;

	if (g_sp->timer)
		dhcp6_remove_timer(&g_sp->timer);

	call_remove_siteprefix(g_sp);
	g_sp = NULL;

	if (ia->ifp)
		ifp_flg = 1;

	(*callback)(ia);

	/* remove check */
	if (g_sa == NULL) {
		if (ifp_flg)
			ia->ifp = NULL;

		return (NULL);
	}

	/* sa */
	ia = g_sa->ctl->iacpd_ia;
	callback = g_sa->ctl->iacpd_callback;

	call_remove_addr(g_sa);
	g_sa = NULL;

	if (ifp_flg)
		ia->ifp = NULL;

	(*callback)(ia);

	return (NULL);
}

static struct ia *
get_ia(type, ifp, iac, iaparam, serverid)
	iatype_t type;
	struct dhcp6_if *ifp;
	struct ia_conf *iac;
	struct dhcp6_listval *iaparam;
	struct duid *serverid;
{
	struct ia *ia;
	struct duid newserver;
	int create = 0;

	if (duidcpy(&newserver, serverid)) {
		debug_printf(LOG_NOTICE, FNAME, "failed to copy server ID");
		return (NULL);
	}

	if ((ia = find_ia(iac, type, iaparam->val_ia.iaid)) == NULL) {
		if ((ia = malloc(sizeof(*ia))) == NULL) {
			debug_printf(LOG_NOTICE, FNAME, "memory allocation failed");
			duidfree(&newserver); /* XXX */
			return (NULL);
		}
		memset(ia, 0, sizeof(*ia));
		ia->state = IAS_ACTIVE;

		TAILQ_INSERT_TAIL(&iac->iadata, ia, link);
		ia->conf = iac;

		create = 1;
	} else
		duidfree(&ia->serverid);

	ia->t1 = iaparam->val_ia.t1;
	ia->t2 = iaparam->val_ia.t2;
	ia->ifp = ifp;
	ia->serverid = newserver;

	debug_printf(LOG_DEBUG, FNAME, "%s an IA: %s-%lu",
	    create ? "make" : "update", iastr(type), ia->conf->iaid);

	return (ia);
}

static struct ia *
find_ia(iac, type, iaid)
	struct ia_conf *iac;
	iatype_t type;
	u_int32_t iaid;
{
	struct ia *ia;

	for (ia = TAILQ_FIRST(&iac->iadata); ia;
	    ia = TAILQ_NEXT(ia, link)) {
		if (ia->conf->type == type && ia->conf->iaid == iaid)
			return (ia);
	}

	return (NULL);
}

static char *
iastr(type)
	iatype_t type;
{
	switch (type) {
	case IATYPE_PD:
		return ("PD");
	case IATYPE_NA:
		return ("NA");
	default:
		return ("???");	/* should be a bug */
	}
}

static char *
statestr(state)
	iastate_t state;
{
	switch (state) {
	case IAS_ACTIVE:
		return "ACTIVE";
	case IAS_RENEW:
		return "RENEW";
	case IAS_REBIND:
		return "REBIND";
	default:
		return "???";	/* should be a bug */
	}
}
