/* (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2006 Netfilter Core Team <coreteam@netfilter.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/types.h>
#include <linux/init.h>
#include <linux/export.h>
#include <linux/ip.h>
#include <linux/icmp.h>

#include <linux/netfilter.h>
#include <net/netfilter/nf_nat.h>
#include <net/netfilter/nf_nat_core.h>
#include <net/netfilter/nf_nat_l4proto.h>

#ifdef CONFIG_SNAT_MAPE
#include <net/map.h>    //---   2019-0412   ICMP Map-E SNAT
#endif  //---   CONFIG_SNAT_MAPE end

static bool
icmp_in_range(const struct nf_conntrack_tuple *tuple,
	      enum nf_nat_manip_type maniptype,
	      const union nf_conntrack_man_proto *min,
	      const union nf_conntrack_man_proto *max)
{
	return ntohs(tuple->src.u.icmp.id) >= ntohs(min->icmp.id) &&
	       ntohs(tuple->src.u.icmp.id) <= ntohs(max->icmp.id);
}

static void
icmp_unique_tuple(const struct nf_nat_l3proto *l3proto,
		  struct nf_conntrack_tuple *tuple,
		  const struct nf_nat_range *range,
		  enum nf_nat_manip_type maniptype,
		  const struct nf_conn *ct)
{
	static u_int16_t id;
	unsigned int range_size;
	unsigned int i;
#ifdef CONFIG_SNAT_MAPE
    //---   2019-0412   Merge Map-E SNAT flow to here.
    unsigned int j;
    bool mape_result;
#endif  //---   CONFIG_SNAT_MAPE end

	range_size = ntohs(range->max_proto.icmp.id) -
		     ntohs(range->min_proto.icmp.id) + 1;
	/* If no range specified... */
	if (!(range->flags & NF_NAT_RANGE_PROTO_SPECIFIED))
		range_size = 0xFFFF;

#ifdef CONFIG_SNAT_MAPE
    //---   2019-0412   To fix SNAT unstable issue.
    for( i = 0; i < range_size; i++) {
#else
	for (i = 0; ; ++id) {
#endif  //---   CONFIG_SNAT_MAPE end

		tuple->src.u.icmp.id = htons(ntohs(range->min_proto.icmp.id) +
					     (id % range_size));
#ifdef CONFIG_SNAT_MAPE
        //---   2019-0412   To fix SNAT unstable issue.
        id++;
#endif
    
#ifdef CONFIG_SNAT_MAPE
        //---   2019-0412 Move Map-E SNAT process from nf_nat_core to here.
        //---   2019-0201 check Map-E port set.
        if( ct->mark == MAP_NAPT_MARK )
        {
            write_lock_bh(&port_range_for_nf_lock);
            mape_result = false;        
            for( j = 0; (j < port_range_length_for_nf) && !mape_result; j++)
            {   //---   if current port is less then port range, then skip to next port range.
                if( ntohs(tuple->src.u.icmp.id) > port_range_for_nf[j].max)
                {   continue;}
                if( ntohs(tuple->src.u.icmp.id) < port_range_for_nf[j].min)
                {   break;}
                mape_result = true;
            }
            write_unlock_bh(&port_range_for_nf_lock);
            //---   if current port isn't in Map-E port set, skip to next port.
            if( mape_result == false)
                continue;
        }
#endif  //---   CONFIG_SNAT_MAPE end
    
#ifdef CONFIG_SNAT_MAPE
		if (!nf_nat_used_tuple(tuple, ct))
#else
		if (++i == range_size || !nf_nat_used_tuple(tuple, ct))
#endif  //---   CONFIG_SNAT_MAPE end
			return;
	}
	return;
}

static bool
icmp_manip_pkt(struct sk_buff *skb,
	       const struct nf_nat_l3proto *l3proto,
	       unsigned int iphdroff, unsigned int hdroff,
	       const struct nf_conntrack_tuple *tuple,
	       enum nf_nat_manip_type maniptype)
{
	struct icmphdr *hdr;

	if (!skb_make_writable(skb, hdroff + sizeof(*hdr)))
		return false;

	hdr = (struct icmphdr *)(skb->data + hdroff);
	inet_proto_csum_replace2(&hdr->checksum, skb,
				 hdr->un.echo.id, tuple->src.u.icmp.id, 0);
	hdr->un.echo.id = tuple->src.u.icmp.id;
	return true;
}

const struct nf_nat_l4proto nf_nat_l4proto_icmp = {
	.l4proto		= IPPROTO_ICMP,
	.manip_pkt		= icmp_manip_pkt,
	.in_range		= icmp_in_range,
	.unique_tuple		= icmp_unique_tuple,
#if defined(CONFIG_NF_CT_NETLINK) || defined(CONFIG_NF_CT_NETLINK_MODULE)
	.nlattr_to_range	= nf_nat_l4proto_nlattr_to_range,
#endif
};
