/* Copyright(c) NEC Platforms, Ltd. 2001-2015 */
#include <linux/kernel.h>
#include <linux/times.h>
#include <linux/if_pppox.h>

#include "br_private.h"

#define SESSION_IDLECNT (br->session_datamax - br->session_datacnt)

static void br_session_ctable_init(struct net_bridge *);
static struct br_session_ctable *br_session_ctable_lookup(struct net_bridge *, __be16);
static struct br_session_ctable *br_session_ctable_get_idletbl(struct net_bridge *br);
static void br_session_ctable_insert(struct net_bridge *, struct br_session_ctable *, __be16);
static void br_session_ctable_update(struct net_bridge *, struct br_session_ctable *);
static void br_session_ctable_flash(struct net_bridge *);
static void br_session_ctable_flash_anyone(struct net_bridge *, struct br_session_ctable *);

/* pppoe bridge session data static memory */
static struct br_session_ctable br_session_ctables[BR_SESSION_CTABLE_MAX];

/* pppoe bridge session management initialize */
void br_session_init(struct net_bridge *br)
{
	br_debug2(br, "%s :\n", __FUNCTION__);
	br->session_datamax = BR_SESSION_CTABLE_MAX;
	br->session_timeout = BR_SESSION_TIMEOUT;
	setup_timer(&br->session_timer, br_session_timer,
		    (unsigned long)br);
	/* lock */
	spin_lock(&br->session_lock);
	br_session_ctable_init(br);
	/* unlock */
	spin_unlock(&br->session_lock);
}

/* pppoe bridge session management start */
void br_session_start(struct net_bridge *br)
{
	br_debug2(br, "%s :\n", __FUNCTION__);
	mod_timer(&br->session_timer,
		  jiffies + (br->session_timeout * HZ / 4));
}

/* pppoe bridge session management stop */
void br_session_stop(struct net_bridge *br)
{
	br_debug2(br, "%s :\n", __FUNCTION__);
	del_timer(&br->session_timer);
	/* lock */
	spin_lock(&br->session_lock);
	br_session_ctable_flash(br);
	/* unlock */
	spin_unlock(&br->session_lock);
}

/* pppoe bridge session management input */
int br_session_input(struct net_bridge *br, struct sk_buff *skb)
{
#define PPPOE_VER	0x1
#define PPPOE_TYPE	0x1
	struct br_session_ctable *sdc = NULL;
	struct pppoe_hdr *ph = NULL;
	int ret = 0;

	br_debug2(br, "%s :\n", __FUNCTION__);

	if (!pskb_may_pull(skb, sizeof(struct pppoe_hdr))) {
		br_err(br, "%s : could not get PPPoE header\n", __FUNCTION__);
		return ENODATA;
	}

	ph = pppoe_hdr(skb);
	if ((ph->ver != PPPOE_VER) || (ph->type != PPPOE_TYPE)) {
		br_err(br, "%s : unknown version/type packet:"
		       " ver=[0x%x] type=[0x%x]\n",
		       __FUNCTION__, ph->ver, ph->type);
		return EINVAL;
	}

	/* lock */
	spin_lock(&br->session_lock);

	/* search session data */
	sdc = br_session_ctable_lookup(br, ntohs(ph->sid));
	if (sdc == NULL) {
		/* session max counter over */
		if (SESSION_IDLECNT == 0) {
			br_info(br, "%s : max session over\n", __FUNCTION__);
			ret = ENOMEM;
			goto end;
		}
	}
	br_debug2(br, "%s : sdc=[%p] sid=[%d]\n",
		  __FUNCTION__, sdc, ntohs(ph->sid));

	/* PPPoE Discovery Stage */
	if (eth_hdr(skb)->h_proto == htons(ETH_P_PPP_DISC)) {
		switch (ph->code) {
		case PADI_CODE:
		case PADO_CODE:
		case PADR_CODE:
			break;
		case PADS_CODE:
			if (sdc == NULL) {
				/* SESSION_ID is 0 not insert */
				if (ntohs(ph->sid) == 0) {
					br_info(br, "%s : SESSION_ID is 0 "
						"not registered\n",
						__FUNCTION__);
					goto end;
				}

				/* get session data idle table */
				sdc = br_session_ctable_get_idletbl(br);
				if (sdc == NULL) {
					br_info(br, "%s : empty idle table\n",
						__FUNCTION__);
					ret = ENOMEM;
					goto end;
				}
				/* insert session data table */
				br_debug2(br, "%s : PADS INSERT sdc=[%p]\n",
					  __FUNCTION__, sdc);
				br_session_ctable_insert(br, sdc,
							 ntohs(ph->sid));
			} else {
				/* updata session data table */
				br_debug2(br, "%s : PADS UPDATE sdc=[%p]\n",
					  __FUNCTION__, sdc);
				br_session_ctable_update(br, sdc);
			}
			break;
		case PADT_CODE:
			if (sdc != NULL) {
				/* flush session data table */
				br_debug2(br, "%s : PADT flash anyone "
					  "sdc=[%p]\n", __FUNCTION__, sdc);
				br_session_ctable_flash_anyone(br, sdc);
			}
			break;
		default:
			br_err(br, "%s : unknown code (0x%04x) "
			       "session=0x%04x\n",
			       __FUNCTION__, ph->code, ntohs(ph->sid));
			ret = EINVAL;
			goto end;
		}

	/* PPPoE Session Stage */
	} else {
		if (sdc == NULL) {
			/* SESSION_ID is 0 not insert */
			if (ntohs(ph->sid) == 0) {
				br_info(br, "%s : SESSION_ID is 0 "
					"not registered\n",
					__FUNCTION__);
				goto end;
			}

			/* get session data idle table */
			sdc = br_session_ctable_get_idletbl(br);
			if (sdc == NULL) {
				br_info(br, "%s : empty idle table\n",
					__FUNCTION__);
				ret = ENOMEM;
				goto end;
			}
			/* insert session data table */
			br_debug2(br, "%s : SESSION INSERT sdc=[%p]\n",
				  __FUNCTION__, sdc);
			br_session_ctable_insert(br, sdc, ntohs(ph->sid));
		} else {
			/* updata session data table */
			br_debug2(br, "%s : SESSION UPDATE sdc=[%p]\n",
				  __FUNCTION__, sdc);
			br_session_ctable_update(br, sdc);
		}
	}

end:
	/* unlock */
	spin_unlock(&br->session_lock);

	/* passthrough */
	return ret;
}

/* pppoe bridge session management timer */
void br_session_timer(unsigned long arg)
{
	struct net_bridge *br = (struct net_bridge *)arg;
	struct br_session_ctable *sdc, *n;
	ktime_t kt;
	struct timeval now, tv;

	/* lock */
	spin_lock(&br->session_lock);

	kt = ktime_get();
	now = ktime_to_timeval(kt);

	list_for_each_entry_safe(sdc, n, &br->session_data_ctable, pct_list) {
		tv = ktime_to_timeval(sdc->pct_timer);
		br_debug2(br, "%s : sdc=[%p] sid=[%d] "
			  "pct_timer=[%lu]+[%lu]=[%lu] now=[%lu]\n",
			  __FUNCTION__,
			  sdc,
			  sdc->session_id,
			  tv.tv_sec,
			  br->session_timeout,
			  tv.tv_sec + br->session_timeout,
			  now.tv_sec);
		if (now.tv_sec >= (tv.tv_sec + br->session_timeout)) {
			br_debug2(br, "%s : del sdc=[%p] sid=[%d]\n",
				  __FUNCTION__, sdc, sdc->session_id);
			br_session_ctable_flash_anyone(br, sdc);
		} else {
			break;
		}
	}

	/* unlock */
	spin_unlock(&br->session_lock);

	mod_timer(&br->session_timer,
		  jiffies + (br->session_timeout  * HZ / 4));
}

/* pppoe bridge session management ioctl */
int br_set_session_max(struct net_bridge *br, int max)
{
	br_debug2(br, "%s : max=[%d]\n", __FUNCTION__, max);
	/* lock */
	spin_lock(&br->session_lock);

	br_session_ctable_flash(br);
	br->session_datamax = max;

	/* unlock */
	spin_unlock(&br->session_lock);

	return 0;
}

int br_get_session_max(struct net_bridge *br)
{
	br_debug2(br, "%s : max=[%d]\n", __FUNCTION__, br->session_datamax);
	return br->session_datamax;
}

int br_get_session_count(struct net_bridge *br)
{
	br_debug2(br, "%s : count=[%d]\n", __FUNCTION__, br->session_datacnt);
	return br->session_datacnt;
}

/* pppoe bridge session management table */
static void br_session_ctable_init(struct net_bridge *br)
{
	struct br_session_ctable *sdc;
	int i;

	INIT_LIST_HEAD(&br->session_idle_ctable);
	INIT_LIST_HEAD(&br->session_data_ctable);

	br->session_datacnt = 0;

	for (i = 0; i < BR_SESSION_CTABLE_MAX; i++) {
		sdc = &br_session_ctables[i];
		INIT_LIST_HEAD(&sdc->pct_list);
		sdc->session_id = 0;
		sdc->pct_timer.tv64 = 0;
		list_add(&sdc->pct_list, &br->session_idle_ctable);
	}
}

static struct br_session_ctable *br_session_ctable_lookup(struct net_bridge *br, __be16 sid)
{
	struct br_session_ctable *sdc;

	if (sid == 0) {
		return NULL;
	}

	list_for_each_entry(sdc, &br->session_data_ctable, pct_list) {
		br_debug2(br, "%s : sdc=[%p] sid=[%d] lookup_sid=[%d]\n",
			  __FUNCTION__, sdc, sdc->session_id, sid);
		if (sdc->session_id == sid) {
			br_debug2(br, "%s : sdc HIT sdc=[%p] sid=[%d]\n",
				  __FUNCTION__, sdc, sdc->session_id);
			return sdc;
		}
	}

	br_debug2(br, "%s : sdc NULL \n", __FUNCTION__);
	return NULL;
}

static struct br_session_ctable *br_session_ctable_get_idletbl(struct net_bridge *br)
{
	struct br_session_ctable *sdc = NULL;

	br_debug2(br, "%s :\n", __FUNCTION__);
	if (list_empty(&br->session_idle_ctable)) {
		br_debug2(br, "%s : idle table empty\n", __FUNCTION__);
		return NULL;
	}

	list_for_each_entry(sdc, &br->session_idle_ctable, pct_list) {
		break;
	}
	if (sdc) {
		br_debug2(br, "%s : idle table sdc=[%p]\n", __FUNCTION__, sdc);
		list_del_init(&sdc->pct_list);
	}

	return sdc;
}

static void br_session_ctable_insert(struct net_bridge *br, struct br_session_ctable *sdc, __be16 sid)
{
	struct timeval tv;

	INIT_LIST_HEAD(&sdc->pct_list);
	sdc->session_id = sid;
	sdc->pct_timer = ktime_get();
	tv = ktime_to_timeval(sdc->pct_timer);
	list_add_tail(&sdc->pct_list, &br->session_data_ctable);
	br->session_datacnt++;
	br_debug2(br, "%s : sdc=[%p] sid=[%d] pct_timer=[%lu]\n",
		  __FUNCTION__, sdc, sid, tv.tv_sec);
}

static void br_session_ctable_update(struct net_bridge *br, struct br_session_ctable *sdc)
{
	struct timeval tv;

	sdc->pct_timer = ktime_get();
	tv = ktime_to_timeval(sdc->pct_timer);
	list_del_init(&sdc->pct_list);
	list_add_tail(&sdc->pct_list, &br->session_data_ctable);
	br_debug2(br, "%s : sdc=[%p] sid=[%d] pct_timer=[%lu]\n",
		  __FUNCTION__, sdc, sdc->session_id, tv.tv_sec);
}

static void br_session_ctable_flash(struct net_bridge *br)
{
	struct br_session_ctable *sdc, *n;

	br_debug2(br, "%s :\n", __FUNCTION__);
	list_for_each_entry_safe(sdc, n, &br->session_data_ctable, pct_list) {
		list_del_init(&sdc->pct_list);
		list_add(&sdc->pct_list, &br->session_idle_ctable);
	}
	br->session_datacnt = 0;

}

static void br_session_ctable_flash_anyone(struct net_bridge *br, struct br_session_ctable *sdc)
{
	br_debug2(br, "%s : sdc=[%p] sid=[%d]\n",
		  __FUNCTION__, sdc, sdc->session_id);
	list_del_init(&sdc->pct_list);
	list_add(&sdc->pct_list, &br->session_idle_ctable);
	br->session_datacnt--;
}
