/* Copyright(c) NEC Platforms, Ltd. 2001-2016 */
/*
 * $Id: videomgr.c,v 1.1.44.5 2010/11/08 05:37:53 haishi Exp $
 */

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <dirent.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/mount.h>
#include <sys/mman.h>
#include <sys/vfs.h>
#include <mntent.h>
#include <linux/videodev2.h>
#include "videomgr.h"
#include "vmgr_utils.h"
#include "videomgr_conf.h"
#ifdef DEBUG_MTRACE
#include <mcheck.h>
#endif

#define GET_MAX(x, y)   (((x) > (y)) ? (x) : (y))
#define MAX_VIDEO_DEVNAME_LEN 32
#define VMGR_MSG_BUFSIZE 8192

#define GET_PIXFMT(buf, fmt)				\
				sprintf(buf, " Format type: %c%c%c%c\n",\
				fmt.pixelformat & 0xFF,			\
				(fmt.pixelformat >> 8) & 0xFF,		\
				(fmt.pixelformat >> 16) & 0xFF,		\
				(fmt.pixelformat >> 24) & 0xFF)

#define GET_PIXFMT_CDM(buf, fmt)				\
				sprintf(buf, "%c%c%c%c:",		\
				fmt.pixelformat & 0xFF,			\
				(fmt.pixelformat >> 8) & 0xFF,		\
				(fmt.pixelformat >> 16) & 0xFF,		\
				(fmt.pixelformat >> 24) & 0xFF)

#define VMGR_TIMESPEC_TO_TIMEVAL(tv, ts) {		\
				(tv)->tv_sec = (ts)->tv_sec;		\
				(tv)->tv_usec = (ts)->tv_nsec / 1000;	\
}

#define VMGR_CHECK_FORMAT_YUYV(x)	(!strcmp("UVC Camera (046d:081b)", x))
#define VMGR_CHECK_NOT_SUPPORT(x)	(!strcmp("UVC Camera (041e:4058)", x))

#define	STREAM_STOP_NORMAL	0
#define	STREAM_STOP_SKIP	1

#define	UVC_FORMAT_NUM		4 /* UVCドライバの対応フォーマット数 */

#define SHOW_MODE_STRING_OFF	0 /* show videomgr 動作モード名非表示 */
#define SHOW_MODE_STRING_ON	1 /* show videomgr 動作モード名表示 */

#define AUTO_FUNC_OFF		0 /* Auto機能の無効化		*/
#define AUTO_FUNC_ON		1 /* Auto機能の有効化		*/

video_in_t *vd_in = NULL;
int is_changed = 0;
struct vmgr_set_info tmp_info[2];
struct vmgr_cnf_info cnf_info[2]; /* default値 or CLI投入値 */
int videomgr_log_level = VMGR_DEF_LOG_LEVEL;
int videomgr_log_enable = VMGR_DEF_LOG_ENABLE;
unsigned int videomgr_vbuf_count = VMGR_DEF_VBUF_COUNT;

#define	vmgr_sleep(w)		if (signal_flg == 0) sleep((w))
#define	vmgr_usleep(w)		if (signal_flg == 0) usleep((w))

static int videomgr_print_location = 1;
static int videomgr_debug = 0;

static unsigned long save_cycle;

static int vmgr_fd = -1;
static struct sockaddr_un vmgr_saddr;
static fd_set rdset;
static unsigned int signal_flg = 0;
static unsigned char video_str_streaming[128];
static unsigned char browser_access_flg = 0;
static unsigned long browser_access_time;

static struct timeval cap_stime;

/* prototype */
static int vmgr_sock_init(void);
static int vmgr_sock_handler(void);
static int vmgr_cmd_process(struct vmgr_cmd *, int);
static int vmgr_signal_init(void);
static int check_signal(void);
static void signal_handler(int);

static int init_video_in(const char *);
static int open_vdf(void);
static int close_vdf(void);
static int grab_video(void);

static int set_video_param(void);
static int change_video_param(void);
static int check_streaming(void);
static void clear_stream_state(void);
static int streaming_start(void);
static int streaming_stop(int);

static int show_usb_caminfo(char *);
static int show_videomgr(char *);
static int show_usb_devinfo(char *);
static int enum_frame_intervals(uint32_t, uint32_t,
				uint32_t, char *);
static int enum_frame_sizes(uint32_t, char *);
static int enum_def_cur_setting(char *, int);
static int enum_frame_formats(unsigned int *, unsigned int, char *);
static int enum_controls(char *);
static void enum_save_config(int, int, char *);

static u_int32_t change_video_ctrl(int, struct v4l2_queryctrl *);
static void auto_func_ctrl(int, int);
static int video_get_ctrl(int);
static int video_set_ctrl(int, int, struct v4l2_queryctrl *);
static int video_direct_get_ctrl(int, struct vmgr_control *);
static int video_direct_ctrl(int, long);
static int video_tgl_ctrl(int, struct v4l2_queryctrl *);
static int video_rst_ctrl(int);
static unsigned long get_time(void);
static unsigned long get_clock_time(void);
static void caputure_timing_check(int);
static void get_filename(int);
static void save_file(void);
static int timer_check(void);
static int timer_scope_check(time_t, time_t *);
static void delete_old_file(void);
static void save_avifile(int);
static void update_header_avifile(int);
static void close_avifile(int);
static void browser_timer_check(void);
static void usb_caminfo_settable_count(char *);
static void usb_caminfo_settable(char *);
static void usb_caminfo_current_setting(char *);

int
main(void)
{
	struct timeval timeout;
	int ret, max_fd = 0;
	unsigned int formattype[UVC_FORMAT_NUM] = {0};
	int i;
	int j;
	int fchk;
	int mode;
	struct stat st;
	struct v4l2_buffer buf;
	FILE *fp;

	openlog("videomgr", LOG_PID, LOG_USER);

	VMGR_LOG(LOG_INFO, VMGR_LOCATION, "videomgr START\n");

	/* daemon */
	if (daemon(1, 0) == -1) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "failed to fork daemon process: %s\n",
		         strerror(errno));
		_exit(EXIT_FAILURE);
	}

#ifdef DEBUG_MTRACE
        putenv("MALLOC_TRACE=/tmp/mtrace.log");
        mtrace();
#endif

	VMGR_LOG(LOG_INFO, VMGR_LOCATION, "videomgr daemon START\n");

	fp = fopen(VMGR_PID, "w");
	if (fp != NULL) {
		fprintf(fp, "%u", getpid());
		fclose(fp);
	}

	vd_in = (video_in_t *)calloc(1, sizeof(video_in_t));
	if (vd_in == NULL) {
		VMGR_LOG_SYSERR("failed to calloc()");
		goto end;
	}
	/* signal処理初期化 */
	if (vmgr_signal_init() < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "fatal error: %s failed\n", __FUNCTION__);
		goto end;
	}
	/* cli socket初期化 */
	if (vmgr_sock_init() < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "fatal error: %s failed\n", __FUNCTION__);
		goto end;
	}
	/* 内部情報の初期化 */
	if (init_video_in(VMGR_DEF_DEVNAME) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "fatal error: %s failed\n", __FUNCTION__);
		goto end;
	}

	VMGR_LOG(LOG_INFO, VMGR_LOCATION, "videomgr before main loop\n");

	/* config読出し設定 */
	vmgr_read_config();

	/* main big loop */
	for (;;) {
		/* check signal */
		if (signal_flg > 0) {
			if (check_signal()) {
				goto end;
			}
		}

		FD_ZERO(&rdset);
		if (vmgr_fd >= 0) {
			FD_SET(vmgr_fd, &rdset);
			max_fd = GET_MAX(max_fd, vmgr_fd);
		}

		timeout.tv_sec = 0;
	 	timeout.tv_usec = 0;
		ret = select(max_fd + 1, &rdset, NULL, NULL, &timeout);
		if (ret < 0) {
			if (errno == EINTR) {
				goto done;
			}
			VMGR_LOG(LOG_ERR, VMGR_LOCATION, "fail to select\n");
			goto end;
		}

		/* check alive for video0	*/
		if (vd_in->fd >= 0) {
			if ((stat(vd_in->video_devname, &st) == -1) || 
						(!S_ISCHR(st.st_mode))) {
				VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
					"%s is no device\n", 
						vd_in->video_devname);
				/* カメラ未接続検出 */
				close(vd_in->fd);
				vd_in->fd = -1;

				memset(&vd_in->cap, 0,
					sizeof(struct v4l2_capability));

				for (i = 0; i < 2; i++) {
					/* 無条件に default or CLI値 */
					tmp_info[i].width =
						cnf_info[i].width;
					tmp_info[i].height =
						cnf_info[i].height;
					tmp_info[i].fps[0] =
						cnf_info[i].fps[0];
					tmp_info[i].fps[1] =
						cnf_info[i].fps[1];
					tmp_info[i].format =
						vd_in->info[i].format =
							VMGR_DEF_FORMAT;
					tmp_info[i].ftype =
						cnf_info[i].ftype;
					tmp_info[i].cycle =
						 cnf_info[i].cycle;
					tmp_info[i].mode =
						cnf_info[i].mode;
				}
				is_changed = 1;

				close_avifile(get_current_mode());

				clear_stream_state();

				sprintf(video_str_streaming,
					"stop (Device not configured)");

			}
			vmgr_usleep(300*1000);
		}

		if ((vmgr_fd >= 0) && FD_ISSET(vmgr_fd, &rdset)) {
			/* recv */
			vmgr_sock_handler();
			/* エラー時も処理続行 */
		}

		/* If device open failed, retry device open. */
		if (vd_in->fd < 0) {
			
			if (open_vdf() < 0) {
				vmgr_sleep(VMGR_RETRY_OPEN_TIMER);
				continue;
			}
			memset(&vd_in->cap, 0, sizeof(struct v4l2_capability));
			if (ioctl(vd_in->fd, VIDIOC_QUERYCAP, &vd_in->cap) < 0) {
				VMGR_LOG_SYSERR("ioctl(VIDIOC_QUERYCAP) error");
				/* エラー時も処理続行 */
			} else {
				clear_stream_state();
				sprintf(video_str_streaming, "stop");

				/* get support format type	*/
				if (enum_frame_formats(formattype,
							UVC_FORMAT_NUM, NULL)) {
					VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"Unable to enumerate frame formats\n");
					goto done;
				}

				fchk = 0;
				/* check setting format and support format */
				for (i = 0; i < UVC_FORMAT_NUM; i++) {
					for (j = 0; j < 2; j++) {
						if (formattype[i] ==
						    tmp_info[j].format) {
							fchk |= (1 << j);
						}
					}
				}
				/* preview側のみ一致 */
				if (fchk == 0x01) {
					/* schedule側をpreview側にあわせる */
					tmp_info[1].format = tmp_info[0].format;
					is_changed = 1;

				/* schedule側のみ一致 */
				} else if (fchk == 0x02) {
					/* preview側をschedule側にあわせる */
					tmp_info[0].format = tmp_info[1].format;
					is_changed = 1;

				/* 全不一致 */
				} else if (fchk == 0) {
					/* カメラがMJPEG/YUV形式の場合 */
					switch (formattype[0]) {
					case V4L2_PIX_FMT_MJPEG:
					case V4L2_PIX_FMT_YUYV:
						/*
						 * preview/scheduleをカメラの
						 * フォーマットに合わせる
						 */
						tmp_info[0].format =
							tmp_info[1].format =
							formattype[0];
						is_changed = 1;
						break;

					/* その他はエラー出力 */
					default:
						VMGR_LOG(LOG_ERR, VMGR_LOCATION,
					"frame formats nothing or unknow.\n");
						goto done;
						break;
					}

				}

				/* 暫定対応 */
				if (VMGR_CHECK_FORMAT_YUYV(vd_in->cap.card)) {
					tmp_info[0].format =
					tmp_info[1].format = V4L2_PIX_FMT_YUYV;
				}

				/*
				 * カメラが接続された時にサポートフォーマット
				 * に合わせて初期値を変更
				 */
				if (tmp_info[0].format == V4L2_PIX_FMT_YUYV) {
					if ((tmp_info[0].width >=
							VMGR_DEF_WIDTH) ||
					    (tmp_info[0].height >=
							VMGR_DEF_HEIGHT)) {
						tmp_info[0].width =
							VMGR_DEF_YUV_WIDTH;
						tmp_info[0].height =
							VMGR_DEF_YUV_HEIGHT;
					}
					if ((tmp_info[1].width >=
							VMGR_DEF_WIDTH) ||
					    (tmp_info[1].height >=
							VMGR_DEF_HEIGHT)) {
						tmp_info[1].width =
							VMGR_DEF_YUV_WIDTH;
						tmp_info[1].height =
							VMGR_DEF_YUV_HEIGHT;
					}
					/* 画像サイズに関係なくFPSを最低に */
					tmp_info[0].fps[0] = 1;
					tmp_info[0].fps[1] = 1;
					tmp_info[1].fps[0] = 1;
					tmp_info[1].fps[1] = 1;
					is_changed = 1;
				}
			}
		}

		/* 設定タイマのチェック */
		timer_check();

		/* streaminのチェック */
		if (check_streaming()) {
			 goto done;
		}

		/* ブラウザ監視タイマチェック */
		browser_timer_check();
#if 1
		/* steaming中であれば、画像取得を行う */
		if (current_state() != VMGR_STREAM_OFF) {
#else
		if (vd_in->get_pict) { 
#endif
			mode = get_current_mode();

			caputure_timing_check(mode);

			if ((ret = grab_video()) != 0) {
				goto done;
			}

			if (get_picture(mode) < 0) {
				goto done_stream;
			}

			/* 一番最初の画像は捨てる */
			if (vd_in->first_pic == 0) {
				vd_in->first_pic++;
				goto done_stream;
			}

			/* メディアへの保存 */
			save_file();

#if 0
			if (vd_in->get_pict == VMGR_CAP_1SHOT) {
				vd_in->get_pict = VMGR_CAP_OFF;
			}
#endif

done_stream:
			/* 画像データ用バッファをstreamingキューにつなぐ */
			memset(&buf, 0, sizeof(buf));
			buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
			buf.memory = V4L2_MEMORY_MMAP;
			buf.index = vd_in->vbuf_index;
			if (ioctl(vd_in->fd, VIDIOC_QBUF, &buf) < 0) {
				VMGR_LOG_SYSERR("ioctl(VIDIOC_QBUF) error");
			}

		}
done:
		continue;
#if 0
		vmgr_usleep(30000); /* 30msec */
#endif
	}

end:
	if (vmgr_fd >= 0) {
		close(vmgr_fd);
		vmgr_fd = -1;
	}
	if (vd_in != NULL) {
		close_vdf();
		free(vd_in);
		vd_in = NULL;
	}
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "videomgr END\n");

#ifdef DEBUG_MTRACE
        muntrace();
#endif

	return 0;
}

static int
init_video_in(const char *device)
{
	int i;

	if (vd_in == NULL || device == NULL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid parameter\n");
		goto error;
	}

	vd_in->fd = -1;
	vd_in->video_devname = (char *)malloc(MAX_VIDEO_DEVNAME_LEN);
	if (vd_in->video_devname == NULL) {
		VMGR_LOG_SYSERR("failed to malloc()");
		goto error;
	}

	sprintf(video_str_streaming, "stop (Device not configured)");

	strlcpy(vd_in->video_devname, device, MAX_VIDEO_DEVNAME_LEN);
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "video %s \n", vd_in->video_devname);
#if 0
	vd_in->get_pict = VMGR_CAP_CONTINUOUS;
#endif

	vd_in->avisizemax = VMGR_DEF_AVI_SIZE_MAX;
	/* AVIヘッダとIndexテーブルヘッダ分を初期値とする */
	vd_in->avisize[0] = AVI_HEADER + AVI_IDX_HEADER;
	vd_in->avisize[1] = AVI_HEADER + AVI_IDX_HEADER;

	vd_in->media = VMGR_DEF_SAVE_MEDIA;
	vd_in->timer_enable = VMGR_TIMER_DISABLE;

	for (i = 0; i < 2; i++) {
		cnf_info[i].width = tmp_info[i].width =
				vd_in->info[i].width = VMGR_DEF_WIDTH;
		cnf_info[i].height = tmp_info[i].height =
				vd_in->info[i].height = VMGR_DEF_HEIGHT;
		cnf_info[i].fps[0] = tmp_info[i].fps[0] =
				vd_in->info[i].fps[0] = VMGR_DEF_FPS_NUM;
		cnf_info[i].fps[1] = tmp_info[i].fps[1] =
				vd_in->info[i].fps[1] = VMGR_DEF_FPS_DEN;

		tmp_info[i].format = vd_in->info[i].format = VMGR_DEF_FORMAT;

		cnf_info[i].ftype = tmp_info[i].ftype =
				vd_in->info[i].ftype = VMGR_DEF_SAVE_FTYPE;
		cnf_info[i].cycle = tmp_info[i].cycle =
				vd_in->info[i].cycle = VMGR_DEF_SAVE_CYCLE;
		cnf_info[i].mode = tmp_info[i].mode =
				vd_in->info[i].mode = VMGR_DEF_SAVE_MODE;

		vd_in->avifile[i] = NULL;
		vd_in->framecount[i] = 0;
		vd_in->recordstart[i] = 0;
		vd_in->recordtime[i] = 0;
	}
	is_changed = 1;

	vd_in->timer_wakeup = VMGR_BEFORE_TIMER_WAKEUP;
	vd_in->yuv_cap = VMGR_YUV_CAPTURE_TIME;
	vd_in->jpg_cap[0] = VMGR_JPG_CAPTURE_TIME1;
	vd_in->jpg_cap[1] = VMGR_JPG_CAPTURE_TIME2;
	cap_stime.tv_sec = 0;
	cap_stime.tv_usec = 0;

	vd_in->first_pic = 0;

	vd_in->vbuf = NULL;
	vd_in->vbuf_count = 0;
	vd_in->vbuf_index = VMGR_DEF_VBUF_INDEX;
	vd_in->newlen = 0;

	vd_in->jpg_buf = NULL;

	return 0;
error:
	if (vd_in != NULL) {
		if (vd_in->video_devname) {
			free(vd_in->video_devname);
			vd_in->video_devname = NULL;
		}
		free(vd_in);
		vd_in = NULL;
	}
	return -1;
}

static int
open_vdf(void)
{
	if ((vd_in->fd = open(vd_in->video_devname, O_RDWR)) < 0) {
//		int chk_errnum;
//		chk_errnum = (ENXIO == errno) ? LOG_INFO : LOG_ERR;
//		VMGR_LOG(chk_errnum, VMGR_LOCATION, "%s open error:%s",
//			 vd_in->video_devname, strerror(errno));
		goto error;
	}

	return 0;
error:
	if (vd_in->fd >= 0) {
		close(vd_in->fd);
		vd_in->fd = -1;
	}
	return -1;
}

static int
change_video_param(void)
{
	int i;
	int j;
	unsigned int formattype[UVC_FORMAT_NUM] = {0};
	int fchk;

	if (vd_in == NULL) {
		errno = ESRCH;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid pointer\n");
		return -1;
	}

	/* get support format type	*/
	if (enum_frame_formats(formattype, UVC_FORMAT_NUM, NULL)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"Unable to enumerate frame formats\n");
		return -1;
	}

	for (i = 0; i < 2; i++) {
		if (tmp_info[i].format) {
			fchk = 0;
			/* check setting format and support format */
			for (j = 0; j < UVC_FORMAT_NUM; j++) {
				if (formattype[j] == tmp_info[i].format) {
					fchk++;
					break;
				}
			}

			if (fchk) {
				/*
				 * カメラがサポートする画像フォーマットが指定
				 * された場合
				 */
				vd_in->info[i].format = tmp_info[i].format;
			} else {
				/*
				 * カメラがサポートしていない画像フォーマットが
				 * 指定された場合、サポートするフォーマットに
				 * 変更する。
				 */

				/* カメラがMJPEG/YUV形式の場合 */
				switch (formattype[0]) {
				case V4L2_PIX_FMT_MJPEG:
				case V4L2_PIX_FMT_YUYV:
					break;

				/* その他はエラー出力 */
				default:
					errno = EOPNOTSUPP;
					VMGR_LOG(LOG_ERR, VMGR_LOCATION,
					"frame formats nothing or unknow.\n");
					return -1;
					break;
				}
				vd_in->info[i].format = formattype[0];
			}

			/*
			 * サポートフォーマットに合わせて初期値を変更
			 */
			if (vd_in->info[i].format == V4L2_PIX_FMT_YUYV) {
				/* 設定済みの場合 */
				if (tmp_info[i].width == 0 ||
				    tmp_info[i].height == 0) {
					/* 設定済みの値と比較 */
					if ((vd_in->info[i].width >=
							VMGR_DEF_WIDTH) ||
					    (vd_in->info[i].height >=
							VMGR_DEF_HEIGHT)) {
						tmp_info[i].width =
							VMGR_DEF_YUV_WIDTH;
						tmp_info[i].height =
							VMGR_DEF_YUV_HEIGHT;
					}

				/* 未設定の場合、設定する値と比較 */
				} else 
				if ((tmp_info[i].width >= VMGR_DEF_WIDTH) ||
				    (tmp_info[i].height >= VMGR_DEF_HEIGHT)) {
					tmp_info[i].width = VMGR_DEF_YUV_WIDTH;
					tmp_info[i].height =
							VMGR_DEF_YUV_HEIGHT;
				}
				/* 画像サイズに関係なくFPSを最低に */
				tmp_info[i].fps[0] = 1;
				tmp_info[i].fps[1] = 1;
			} else {
				/*
				 * MJPEGの場合、コマンド設定値を
				 * そのまま再設定
				 */
				tmp_info[i].width = cnf_info[i].width;
				tmp_info[i].height = cnf_info[i].height;
				tmp_info[i].fps[0] = cnf_info[i].fps[0];
				tmp_info[i].fps[1] = cnf_info[i].fps[1];
			}
			tmp_info[i].format = 0;	
		}
		if (tmp_info[i].width) {
			vd_in->info[i].width = tmp_info[i].width;
			tmp_info[i].width = 0;	
		}
		if (tmp_info[i].height) {
			vd_in->info[i].height = tmp_info[i].height;
			tmp_info[i].height = 0;	
		}
		if (tmp_info[i].fps[0]) {
			vd_in->info[i].fps[0] = tmp_info[i].fps[0];
			tmp_info[i].fps[0] = 0;	
		}
		if (tmp_info[i].fps[1]) {
			vd_in->info[i].fps[1] = tmp_info[i].fps[1];
			tmp_info[i].fps[1] = 0;	
		}
		if (tmp_info[i].ftype) {
			vd_in->info[i].ftype = tmp_info[i].ftype;
			tmp_info[i].ftype = 0;
		}
		if (tmp_info[i].cycle) {
			vd_in->info[i].cycle = (tmp_info[i].cycle == 1) ? 0 : 
						tmp_info[i].cycle;
			tmp_info[i].cycle = 0; 
		}
		if (tmp_info[i].mode) {
			vd_in->info[i].mode = tmp_info[i].mode;
			tmp_info[i].mode =  0;
		}
	}
	return 0;
}

static int
set_video_param(void)
{
	struct v4l2_streamparm *setfps = NULL;
	int ret = -1;
	int mode;
	struct v4l2_format vf;
	struct v4l2_streamparm vs;

	if (vd_in == NULL) {
		errno = ESRCH;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid pointer\n");
		goto error;
	}
	if (vd_in->fd < 0) {
		errno = ENXIO;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		goto error;
	}
	if (!(vd_in->cap.capabilities & V4L2_CAP_VIDEO_CAPTURE)) {
		errno = ENODEV;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"%s: video capture not supported.\n",
			 vd_in->video_devname);
		goto error;
	}
	if (!(vd_in->cap.capabilities & V4L2_CAP_STREAMING)) {
		errno = EINVAL;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"%s does not support streaming i/o\n",
				vd_in->video_devname);
		goto error;
	}
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "video device:%s\n",
				vd_in->cap.card);

	/* set format in */
	memset(&vd_in->fmt, 0, sizeof(struct v4l2_format));
	if (is_changed) {
		if (change_video_param() < 0) {
			goto error;
		}
		is_changed = 0;
	}

	mode = get_new_mode();

	/* フォーマットの設定 */
	vd_in->fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	vd_in->fmt.fmt.pix.width = vd_in->info[mode].width;
	vd_in->fmt.fmt.pix.height = vd_in->info[mode].height;
	vd_in->fmt.fmt.pix.pixelformat = vd_in->info[mode].format;
	vd_in->fmt.fmt.pix.field = V4L2_FIELD_ANY;
	if (ioctl(vd_in->fd, VIDIOC_S_FMT, &vd_in->fmt) < 0) {
		VMGR_LOG_SYSERR("ioctl(VIDIOC_S_FMT) error");
		goto error;
	}

	/* for confirm setting */
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "set format\n");
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, " width:%d\n",
				vd_in->fmt.fmt.pix.width);
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, " height:%d\n",
				vd_in->fmt.fmt.pix.height);
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, " pixelformat:%c%c%c%c\n",
		 (char)(vd_in->fmt.fmt.pix.pixelformat & 0x000000ff),
		 (char)((vd_in->fmt.fmt.pix.pixelformat & 0x0000ff00) >> 8),
		 (char)((vd_in->fmt.fmt.pix.pixelformat & 0x00ff0000) >> 16),
		 (char)((vd_in->fmt.fmt.pix.pixelformat & 0xff000000) >> 24));

	if ((vd_in->fmt.fmt.pix.width != vd_in->info[mode].width) ||
	    (vd_in->fmt.fmt.pix.height != vd_in->info[mode].height)) {
		VMGR_LOG(LOG_WARNING, VMGR_LOCATION,
			 "format asked unavailable get width %d height %d \n",
			 vd_in->fmt.fmt.pix.width, vd_in->fmt.fmt.pix.height);
		vd_in->info[mode].width = vd_in->fmt.fmt.pix.width;
		vd_in->info[mode].height = vd_in->fmt.fmt.pix.height;
		/* look the format is not part of the deal ??? */
		/* vd_in->format_in = vd_in->fmt.fmt.pix.pixelformat; */
	}

	/* YUV形式の場合 */
	if (vd_in->info[mode].format == V4L2_PIX_FMT_YUYV) {
		vd_in->jpg_buf = (unsigned char *)malloc(
					vd_in->info[mode].width *
					vd_in->info[mode].height * 3);
		if (!vd_in->jpg_buf) {
			VMGR_LOG_SYSERR("malloc() error");
			goto error;
		}
	}

	/* set framerate */
	setfps = (struct v4l2_streamparm *)calloc(1,
					sizeof(struct v4l2_streamparm));
	if (!setfps) {
		VMGR_LOG_SYSERR("calloc() error");
		goto error;
	}
	memset(setfps, 0, sizeof(struct v4l2_streamparm));
	setfps->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	setfps->parm.capture.timeperframe.numerator = vd_in->info[mode].fps[0];
	setfps->parm.capture.timeperframe.denominator =
						vd_in->info[mode].fps[1];
	if (ioctl(vd_in->fd, VIDIOC_S_PARM, setfps) < 0) {
		VMGR_LOG_SYSERR("ioctl(VIDIOC_S_PARM) error");
		goto error;
	}

	/* Current setting */
	memset(&vf, 0, sizeof(vf));
	vf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl(vd_in->fd, VIDIOC_G_FMT, &vf) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "ioctl(VIDIOC_G_FMT) fail(%d)\n", errno);
		goto error;
	}
	memset(&vs, 0, sizeof(vs));
	vs.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl(vd_in->fd, VIDIOC_G_PARM, &vs) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "ioctl(VIDIOC_G_PARM) fail\n");
		goto error;
	}

	vd_in->info[mode].format = vf.fmt.pix.pixelformat;
	vd_in->info[mode].width = vf.fmt.pix.width;
	vd_in->info[mode].height = vf.fmt.pix.height;
	vd_in->info[mode].fps[0] = setfps->parm.capture.timeperframe.numerator;
	vd_in->info[mode].fps[1] = vs.parm.capture.timeperframe.denominator;

	ret = 0;

error:
	if (setfps) {
		free(setfps);
	}
	return ret;
}

static void
clear_stream_state(void)
{
	if (current_state() != VMGR_STREAM_OFF) {
		streaming_stop(STREAM_STOP_NORMAL);
	}

	vd_in->stream_state = VMGR_STREAM_OFF;
	vd_in->stream_state_new = VMGR_STREAM_OFF;
}

static int
streaming_start(void)
{
	int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	int ret = 0;
	int checkflag = 0;
	int mode;
	unsigned int i;
	struct v4l2_requestbuffers req;
	struct v4l2_buffer buf;

	if (is_changed) {
		checkflag = 1;
	}

	/* 変更前のステートからモード設定 */
	mode = get_current_mode();

	/* 現在、streaming中であれば、一度停止する */
	if (current_state() != VMGR_STREAM_OFF) {
		streaming_stop(STREAM_STOP_SKIP);
	}

	/* UVCカメラへ新しいパラメータの設定 */
	if (set_video_param() < 0) {
		ret = -1;
		goto error;
	}

	memset(&req, 0, sizeof(req));

	req.count = videomgr_vbuf_count;
	req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	req.memory = V4L2_MEMORY_MMAP;

	/* 画像データ用バッファ要求 */
	if (ioctl(vd_in->fd, VIDIOC_REQBUFS, &req) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "ioctl(VIDIOC_REQBUFS) fail\n");
		ret = -1;
		goto error;
	}

	if (req.count < 2) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "Insufficient buffer memory\n");
		ret = -1;
		goto error;
	}

	vd_in->vbuf_count = req.count;
	vd_in->vbuf = (struct vmgr_buffer *)calloc(req.count,
					sizeof(struct vmgr_buffer));
	if (vd_in->vbuf == NULL) {
		VMGR_LOG_SYSERR("calloc() error");
		vd_in->vbuf_count = 0;
		ret = -1;
		goto error;
	}

	for (i = 0; i < req.count; i++) {
		memset(&buf, 0, sizeof(buf));

		buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory = V4L2_MEMORY_MMAP;
		buf.index = i;

		/* 画像データ用バッファの情報取得 */
		if (ioctl(vd_in->fd, VIDIOC_QUERYBUF, &buf) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"ioctl(VIDIOC_QUERYBUF) fail\n");
			ret = -1;
			goto error;
		}

		vd_in->vbuf[i].len = buf.length;
		vd_in->vbuf[i].mem = mmap(NULL,
					  buf.length,
					  PROT_READ | PROT_WRITE,
					  MAP_SHARED,
					  vd_in->fd,
					  buf.m.offset);

		if (vd_in->vbuf[i].mem == MAP_FAILED) {
			VMGR_LOG_SYSERR("mmap() error");
			vd_in->vbuf[i].mem = NULL;
			ret = -1;
			goto error;
		}

		/* 画像データ用バッファをstreamingキューにつなぐ */
		if (ioctl(vd_in->fd, VIDIOC_QBUF, &buf) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"ioctl(VIDIOC_QBUF) fail\n");
			ret = -1;
			goto error;
		}
	}

	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
		"ioctl(VIDIOC_REQBUFS req.count=%d\n", req.count);
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
		"ioctl(VIDIOC_QUERYBUF buf.length=%d\n", buf.length);

	usleep(200*1000);

	/* streaming開始 */
	if (ioctl(vd_in->fd, VIDIOC_STREAMON, &type) < 0) {
		VMGR_LOG_SYSERR("ioctl(VIDIOC_STREAMON) error");
		system("echo xhci-hcd.0 > /sys/bus/platform/drivers/xhci-hcd/unbind");
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "unbind");
		ret = -1;
		goto error;
	}

	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "ioctl(STREAMON)\n");

	/* ステートの変更 */
	vd_in->stream_state = vd_in->stream_state_new;

	save_cycle = 0;
	vd_in->first_pic = 0;

	sprintf(video_str_streaming, "start");

	/* 新しいステートでの再設定 */
	mode = get_current_mode();

	if (is_current_state(VMGR_STREAM_TIMER)) {
		strcat(video_str_streaming, " (timer)");
	} else if (is_current_state(VMGR_STREAM_SAVE)) {
		strcat(video_str_streaming, " (save)");
	}

	vd_in->newlen = 0;

	/* UVCコントロールを再設定する */
	for (i = 0; i < VMGR_CMD_SUB_NUM; i++) {
		if (vd_in->control[i].valid) {
			if (video_direct_ctrl(i + 1,
					vd_in->control[i].value) < 0) {
				VMGR_LOG(LOG_ERR, VMGR_LOCATION,
						"UVC control error(%d:%d)\n",
						i + 1,
						vd_in->control[i].value);
			}
		}
	}

	/* AVI保存の場合、開始時間の保持とcounterリセット */
	if (is_current_state(VMGR_STREAM_SAVE)) {
		if (vd_in->info[mode].ftype == VMGR_SAVE_FTYPE_AVI) {
			vd_in->recordstart[mode] = get_time();
			vd_in->framecount[mode] = 0;
		}
	}

error:
	if (ret != 0) {
		sprintf(video_str_streaming, "stop (%s)", strerror(errno));
		vd_in->stream_state = VMGR_STREAM_OFF;
		vd_in->stream_state_new = VMGR_STREAM_OFF;
		streaming_stop(STREAM_STOP_SKIP);
	}

	if (ret != 0 && checkflag) {
		tmp_info[mode].format = vd_in->info[mode].format;
		tmp_info[mode].width = vd_in->info[mode].width;
		tmp_info[mode].height = vd_in->info[mode].height;
		tmp_info[mode].fps[0] = vd_in->info[mode].fps[0];
		tmp_info[mode].fps[1] = vd_in->info[mode].fps[1];
		is_changed = 1;
	}

	return ret;
}

static int
streaming_stop(int flag)
{
	int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	int ret = 0;
	int mode;
	unsigned int i;
	struct v4l2_requestbuffers req;

	mode = get_current_mode();

	/* AVIファイルのCLOSE処理 */
	close_avifile(mode);

	/* streaming停止 */
	if (ioctl(vd_in->fd, VIDIOC_STREAMOFF, &type) < 0) {
		VMGR_LOG_SYSERR("ioctl(VIDIOC_STREAMOFF) error");
		sprintf(video_str_streaming, "stop (%s)", strerror(errno));
		ret = -1;
		vd_in->stream_state = VMGR_STREAM_OFF;
		vd_in->stream_state_new = VMGR_STREAM_OFF;
		goto end;
	}

	sprintf(video_str_streaming, "stop");

end:

	/* YUV形式であれば、CJPEGの後処理を行う */
	if ((vd_in->info[mode].format == V4L2_PIX_FMT_YUYV) && 
						(vd_in->jpg_buf)) {
        	free(vd_in->jpg_buf);
		vd_in->jpg_buf = NULL;
	}

	if (flag == STREAM_STOP_NORMAL) {
		vd_in->stream_state = vd_in->stream_state_new;
	}

	/* 画像データ用バッファ解放 */
	if ((vd_in->vbuf != NULL) && (vd_in->vbuf_count != 0)) {
		for (i = 0; i < vd_in->vbuf_count; i++) {
			if (vd_in->vbuf[i].mem) {
				if ((ret = munmap(vd_in->vbuf[i].mem, 
						vd_in->vbuf[i].len)) == -1) {
					VMGR_LOG_SYSERR("munmap() error");
				}
			}
		}
		free(vd_in->vbuf);
		vd_in->vbuf_count = 0;
		vd_in->vbuf = NULL;

		/* カーネルバッファ解放 */
		memset(&req, 0, sizeof(req)); /* req.count = 0; */
		req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		req.memory = V4L2_MEMORY_MMAP;
		if (ioctl(vd_in->fd, VIDIOC_REQBUFS, &req) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"ioctl(VIDIOC_REQBUFS) fail\n");
		}
	}

	return ret;
}

static int
check_streaming(void)
{
	int ret = 0;
	int i;
	unsigned int bit;
	unsigned int bitmask;

	if (new_state() == VMGR_STREAM_OFF) {
		ret = 1;
		if (current_state() != new_state()) {
			if (streaming_stop(STREAM_STOP_NORMAL) < 0) {
				ret = -1;
			}
		}
	} else {
		bit = 1 << (VMGR_STREAM_NUM - 1);
		bitmask = (bit - 1) << VMGR_STREAM_NUM;

		for (i = 0; i < VMGR_STREAM_NUM; i++) {
			if (is_new_state(bit) != is_current_state(bit) && 
			    is_current_state(bitmask) == 0) {
				if (streaming_start() < 0) {
					ret = -1;
				}
				break;
			}

			bit >>= 1;
			bitmask >>= 1;
		}
	}

	return ret;
}

static int
grab_video(void)
{
	int ret = -1;
	struct v4l2_buffer buf;

	/* 画像バッファのインデックスとサイズを初期化 */
	vd_in->vbuf_index = VMGR_DEF_VBUF_INDEX;
	vd_in->newlen = 0;

	memset(&buf, 0, sizeof(buf));

	buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf.memory = V4L2_MEMORY_MMAP;

	if (ioctl(vd_in->fd, VIDIOC_DQBUF, &buf) < 0) {
		if (errno == EINTR) {
			/* CLIコマンドやカメラが抜かれたSIGNAL受信 */
			VMGR_LOG(LOG_WARNING, VMGR_LOCATION, "Grab abort\n");
		}
		else {
			VMGR_LOG_SYSERR("ioctl(VIDIOC_DQBUF) error");
		}
		goto error;
	}

	if (buf.index >= vd_in->vbuf_count) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "Insufficient buffer memory\n");
		goto error;
	}

	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
		"ioctl(VIDIOC_DQBUF buf.index=%d buf.bytesused=%d\n", 
						buf.index, buf.bytesused);

	/* 画像バッファのインデックスとサイズを設定 */
	vd_in->vbuf_index = buf.index;
	vd_in->newlen = buf.bytesused;

	return 0;
error:
	return ret;
}

static int
close_vdf(void)
{
	int ret = 0;
	unsigned int i;
	struct v4l2_requestbuffers req;

	if (current_state()) {
		streaming_stop(STREAM_STOP_NORMAL);
	}
	if ((vd_in->vbuf != NULL) && (vd_in->vbuf_count != 0)) {
		for (i = 0; i < vd_in->vbuf_count; i++) {
			if (vd_in->vbuf[i].mem) {
				if ((ret = munmap(vd_in->vbuf[i].mem, 
						vd_in->vbuf[i].len)) == -1) {
					VMGR_LOG_SYSERR("munmap() error");
				}
			}
		}
		free(vd_in->vbuf);
		vd_in->vbuf_count = 0;
		vd_in->vbuf = NULL;

		/* カーネルバッファ解放 */
		memset(&req, 0, sizeof(req)); /* req.count = 0; */
		req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		req.memory = V4L2_MEMORY_MMAP;
		if (ioctl(vd_in->fd, VIDIOC_REQBUFS, &req) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"ioctl(VIDIOC_REQBUFS) fail\n");
		}
	}
	if (vd_in->video_devname != NULL) {
		free(vd_in->video_devname);
		vd_in->video_devname = NULL;
	}
	if (vd_in->fd >= 0) {
		close(vd_in->fd);
	}

	return 0;
}

/* signal */
static int
check_signal(void)
{
	int ret = 0;

	switch (signal_flg) {
	case SIGTERM:
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "SIGTERM");
		signal_flg = 0;
		ret = -1;
		break;
	case SIGALRM:
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "SIGALRM");
		signal_flg = 0;
		ret = -1;
		break;
	case SIGHUP:
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "SIGHUP");
		signal_flg = 0;
		ret = -1;
		break;
	case SIGCHLD:
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "SIGCHLD");
		signal_flg = 0;
		break;
	case SIGUSR1:
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "SIGUSR1");
		signal_flg = 0;
		break;
	case SIGUSR2:
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "SIGUSR2");
		signal_flg = 0;
		if (new_state() == VMGR_STREAM_OFF) {
			/* ブラウザ最終アクセス時間を更新 */
			browser_access_flg = 1;
			set_state(VMGR_STREAM_ON);
			if (check_streaming() < 0) {
				VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
					"streaming start error");
			}
		}
		else {
			clear_state(VMGR_STREAM_ON);
			if (check_streaming() < 0) {
				VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
					"streaming stop error");
			}
		}
		break;
	default :
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "unexpected signal [%d]", signal_flg);
		signal_flg = 0;
		break;
	}
	return ret;
}

static void
signal_handler(int sig)
{
	signal_flg = sig;
}

static int
vmgr_signal_init(void)
{
	struct  sigaction sigact;

	/* SIGTERM */
	sigact.sa_handler = signal_handler;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGTERM, &sigact, NULL) != 0 ) {
		return -1;
	}
	/* SIGALRM */
	sigact.sa_handler = signal_handler;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGALRM, &sigact, NULL) != 0 ) {
		return -1;
	}
	/* SIGPIPE */
	sigact.sa_handler = SIG_IGN;    /* ignore */
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGPIPE, &sigact, NULL) != 0 ) {
		return -1;
	}
	/* SIGHUP */
	sigact.sa_handler = signal_handler;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGHUP, &sigact, NULL) != 0 ) {
		return -1;
	}
	/* SIGCHLD */
	sigact.sa_handler = SIG_IGN;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGCHLD, &sigact, NULL) != 0 ) {
		return -1;
	}
	/* SIGUSR1 */
	sigact.sa_handler = signal_handler;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGUSR1, &sigact, NULL) != 0 ) {
		return -1;
	}
	/* SIGUSR2 */
	sigact.sa_handler = signal_handler;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;    /* not set SA_RESTART */
	if (sigaction(SIGUSR2, &sigact, NULL) != 0 ) {
		return -1;
	}
	return 0;
}

static int
vmgr_sock_init(void)
{
	memset(&vmgr_saddr, 0, sizeof(vmgr_saddr));
	vmgr_saddr.sun_family = AF_UNIX;
	snprintf(vmgr_saddr.sun_path, sizeof(vmgr_saddr.sun_path),
		 "%s", VMGR_SOCK_PATH);

	vmgr_fd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (vmgr_fd == -1) {
		VMGR_LOG_SYSERR("failed to socket()");
		return -1;
	}

	unlink(vmgr_saddr.sun_path);
	if (bind(vmgr_fd, (struct sockaddr *)&vmgr_saddr,
		 sizeof(vmgr_saddr)) != 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "bind(sockname:%s): %s\n",
		        vmgr_saddr.sun_path, strerror(errno));
		close(vmgr_fd);
		return -1;
	}

	if (listen(vmgr_fd, 5) != 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "listen(sockname:%s): %s\n",
		       vmgr_saddr.sun_path, strerror(errno));
		close(vmgr_fd);
		return -1;
	}
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION,
		 "open %s as videomgr\n", vmgr_saddr.sun_path);

        return 0;
}

#pragma GCC diagnostic ignored "-Wcast-align"
static int
vmgr_sock_handler(void)
{
	int so, msglen;
	struct sockaddr_storage from;
	socklen_t fromlen = sizeof(from);
	struct vmgr_cmd cmd;
	int ret = -1;
	int len;

	so = accept(vmgr_fd, (struct sockaddr *)&from, &fromlen);
	if (so < 0) {
		VMGR_LOG_SYSERR("failed to accept()");
		return -1;
	}

	/* get buffer length */
	while ((len = recv(
		so, (char *)&cmd, sizeof(struct vmgr_cmd_hdr), MSG_PEEK)) < 0) {
		if (errno == EINTR)
			continue;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"failed to recv(): %s\n", strerror(errno));
		goto end;
	}

	/* sanity check */
	if ((size_t)len < sizeof(cmd.hdr)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid recv header length\n");
		goto end;
	}
	memset(&cmd.conf, 0, sizeof(struct vmgr_config));
	msglen = cmd.hdr.len + sizeof(struct vmgr_cmd_hdr);
	/* get real data */
	while ((len = recv(so, (char *)&cmd, msglen, 0)) < 0) {
	        if (errno == EINTR)
	                continue;
		VMGR_LOG_SYSERR("failed to recv()");
		goto end;
	}
	/* 受信buffer解析処理 */
	if (vmgr_cmd_process(&cmd, so) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "failed to cmd process\n");
		goto end;
	}
	ret = 0;
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "success to recv() len=%d\n", len);

end:
	close(so);

	return ret;
}
#pragma GCC diagnostic warning "-Wcast-align"

#pragma GCC diagnostic ignored "-Wcast-align"
static int
vmgr_cmd_process(struct vmgr_cmd *cmd, int sock)
{
	int ret = 0;
	struct vmgr_cmd_hdr *hdr;
	int sndlen;
	char *buf = NULL, *sndbuf = NULL;
	char tmp[80];
	int i;
	struct vmgr_control control;
	int change = 0;

	if (cmd == NULL || vd_in == NULL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid pointer\n");
		ret = -1;
		goto end;
	}
	VMGR_LOG(LOG_INFO, VMGR_LOCATION,
		"cmd=%d, portno=%d\n", cmd->hdr.kind, cmd->hdr.portno);

	buf = calloc(1, VMGR_MSG_BUFSIZE);
	if (buf == NULL) {
		VMGR_LOG_SYSERR("failed to malloc()");
		ret = -1;
		goto end;
	}

	switch(cmd->hdr.kind) {
	case VMGR_CMD_LOG_ENABLE:
		videomgr_log_enable = cmd->conf.log_enable;
		change = 1;
		break;

	case VMGR_CMD_LOG_LEVEL:
		videomgr_log_level = cmd->conf.log_level;
		change = 1;
		break;
#if 0
	case VMGR_CMD_ENABLE:
		videomgr_log_enable = cmd->conf.log_enable;
		videomgr_log_level = cmd->conf.log_level;
		for (i = 0; i < 2; i++) {
			tmp_info[i].format = vd_in->info[i].format =
						cmd->conf.info[i].format;
			tmp_info[i].width = vd_in->info[i].width =
						cmd->conf.info[i].width;
			tmp_info[i].height = vd_in->info[i].height =
						cmd->conf.info[i].height;
			tmp_info[i].fps = vd_in->info[i].fps =
						cmd->conf.info[i].fps;
			tmp_info[i].ftype = vd_in->info[i].ftype =
						cmd->conf.info[i].ftype;
			tmp_info[i].cycle = vd_in->info[i].cycle =
					cmd->conf.info[i].cycle == 0 ? 1 :
					cmd->conf.info[i].cycle;
			tmp_info[i].mode = vd_in->info[i].mode =
						cmd->conf.info[i].mode;
		}
		is_changed = 1;
		vd_in->media = cmd->conf.media;
		if (vd_in->video_devname != NULL) {
			strlcpy(vd_in->video_devname, cmd->conf.devname,
				MAX_VIDEO_DEVNAME_LEN);
		}
		vd_in->is_recv_conf = 1;
		break;
	case VMGR_CMD_DISABLE:
		break;
#endif
	case VMGR_CMD_STREAMING_START:
		browser_access_flg = 1;	 /* ブラウザ最終アクセス時間を更新 */
		set_state(VMGR_STREAM_ON);
		if (check_streaming() < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "streaming start check_streaming() errno: %s\n", 
				strerror(errno));
		}
		break;

	case VMGR_CMD_STREAMING_STOP:
		clear_state(VMGR_STREAM_ON);
		if (check_streaming() < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "streaming stop check_streaming() errno: %s\n", 
				strerror(errno));
		}
		break;
#if 0
	case VMGR_CMD_CAPTURE_REQ:
		vd_in->get_pict = VMGR_CAP_1SHOT;
		break;
	case VMGR_CMD_CAP_START:
		vd_in->get_pict = VMGR_CAP_CONTINUOUS;
		break;
	case VMGR_CMD_CAP_STOP:
		vd_in->get_pict = VMGR_CAP_OFF;
		break;
	case VMGR_CMD_PARAM_SET_DEFAULT:
		for (i = 0; i < 2; i++) {
			tmp_info[i].format = VMGR_DEF_FORMAT;
			tmp_info[i].width = VMGR_DEF_WIDTH;
			tmp_info[i].height = VMGR_DEF_HEIGHT;
			tmp_info[i].fps = VMGR_DEF_FPS;
			tmp_info[i].ftype = VMGR_DEF_SAVE_FTYPE;
			tmp_info[i].cycle = VMGR_DEF_SAVE_CYCLE;
			tmp_info[i].mode = VMGR_DEF_SAVE_MODE;
		}
		is_changed = 1;
		change = 1;
		break;
#endif
	case VMGR_CMD_PARAM_SET_P_FMT:
		tmp_info[0].format = cmd->conf.info.format;

		/*
		 * 画像フォーマットが指定された場合にフォーマットの
		 * 許容する初期値に変更
		 */
		if (tmp_info[0].format == V4L2_PIX_FMT_YUYV) {
			/* 設定済みの場合 */
			if ((tmp_info[0].width == 0) ||
			    (tmp_info[0].height == 0)) {
				/* 設定済みの値を比較 */
				if ((vd_in->info[0].width >= VMGR_DEF_WIDTH) ||
				    (vd_in->info[0].height >=
							VMGR_DEF_HEIGHT)) {
					tmp_info[0].width = VMGR_DEF_YUV_WIDTH;
					tmp_info[0].height =
							VMGR_DEF_YUV_HEIGHT;
				}

			/* まだ設定していなければ、設定する値と比較 */
			} else if ((tmp_info[0].width >= VMGR_DEF_WIDTH) ||
				   (tmp_info[0].height >= VMGR_DEF_HEIGHT)) {
				tmp_info[0].width = VMGR_DEF_YUV_WIDTH;
				tmp_info[0].height = VMGR_DEF_YUV_HEIGHT;
			}
			/* 画像サイズに関係なくFPSを最低にする */
			tmp_info[0].fps[0] = 1;
			tmp_info[0].fps[1] = 1;
		} else {
			/* MJPEGの場合、コマンド設定値をそのまま再設定 */
			tmp_info[0].width = cnf_info[0].width;
			tmp_info[0].height = cnf_info[0].height;
			tmp_info[0].fps[0] = cnf_info[0].fps[0];
			tmp_info[0].fps[1] = cnf_info[0].fps[1];
		}
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_S_FMT:
		tmp_info[1].format = cmd->conf.info.format;

		/*
		 * 画像フォーマットが指定された場合にフォーマットの
		 * 許容する初期値に変更
		 */
		if (tmp_info[1].format == V4L2_PIX_FMT_YUYV) {
			/* 設定済みの場合 */
			if ((tmp_info[1].width == 0) ||
			    (tmp_info[1].height == 0)) {
				/* 設定済みの値を比較 */
				if ((vd_in->info[1].width >= VMGR_DEF_WIDTH) ||
				    (vd_in->info[1].height >=
							VMGR_DEF_HEIGHT)) {
					tmp_info[1].width = VMGR_DEF_YUV_WIDTH;
					tmp_info[1].height =
							VMGR_DEF_YUV_HEIGHT;
				}

			/* まだ設定していなければ、設定する値と比較 */
			} else if ((tmp_info[1].width >= VMGR_DEF_WIDTH) ||
				   (tmp_info[1].height >= VMGR_DEF_HEIGHT)) {
				tmp_info[1].width = VMGR_DEF_YUV_WIDTH;
				tmp_info[1].height = VMGR_DEF_YUV_HEIGHT;
			}
			/* 画像サイズに関係なくFPSを最低にする */
			tmp_info[1].fps[0] = 1;
			tmp_info[1].fps[1] = 1;
		} else {
			/* MJPEGの場合、コマンド設定値をそのまま再設定 */
			tmp_info[1].width = cnf_info[1].width;
			tmp_info[1].height = cnf_info[1].height;
			tmp_info[1].fps[0] = cnf_info[1].fps[0];
			tmp_info[1].fps[1] = cnf_info[1].fps[1];
		}
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_P_SIZE:
		cnf_info[0].width = tmp_info[0].width = cmd->conf.info.width;
		cnf_info[0].height = tmp_info[0].height = cmd->conf.info.height;
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_S_SIZE:
		cnf_info[1].width = tmp_info[1].width = cmd->conf.info.width;
		cnf_info[1].height = tmp_info[1].height = cmd->conf.info.height;
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_P_FPS:
		cnf_info[0].fps[0] = tmp_info[0].fps[0] = cmd->conf.info.fps[0];
		cnf_info[0].fps[1] = tmp_info[0].fps[1] = cmd->conf.info.fps[1];
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_S_FPS:
		cnf_info[1].fps[0] = tmp_info[1].fps[0] = cmd->conf.info.fps[0];
		cnf_info[1].fps[1] = tmp_info[1].fps[1] = cmd->conf.info.fps[1];
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_DEVICE:
		if (vd_in->video_devname != NULL) {
			strlcpy(vd_in->video_devname, cmd->conf.devname,
				MAX_VIDEO_DEVNAME_LEN);
		} else {
			ret = -1;
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
						"video_devname is NULL\n");
		}
		break;

	case VMGR_CMD_SHOW_VIDEOMGR:
		if ((ret = show_videomgr(buf)) < 0) {
			sprintf(tmp, "%% error - %s\n", strerror(errno));
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
		}
		break;

	case VMGR_CMD_SHOW_USB_DEVINFO:
#if ATERMLX_PENDING
		if (vd_in->cap.portno == cmd->hdr.portno) {
#endif
			if ((ret = show_usb_devinfo(buf)) < 0) {
				sprintf(buf, "%% error - %s\n",
							strerror(errno));
			}
#if ATERMLX_PENDING
		}
#endif
		break;

	case VMGR_CMD_SHOW_CAMINFO:
#if ATERMLX_PENDING
		if (vd_in->cap.portno != 0) {
#endif
			if ((ret = show_usb_caminfo(buf)) < 0) {
				sprintf(buf, "%% error - %s\n",
							strerror(errno));
			}
#if ATERMLX_PENDING
		}
#endif
		break;

	case VMGR_CMD_CONTROL_GET:
		if (video_direct_get_ctrl(cmd->hdr.sub, &control) < 0) {
			sprintf(tmp, "%% error - %s\n", strerror(errno));
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			break;
		}
	
		sndlen = sizeof(*hdr) + sizeof(control);
		sndbuf = malloc(sndlen);
		if (sndbuf == NULL) {
			VMGR_LOG_SYSERR("failed to malloc()");
			ret = -1;
			goto end;
		}
		hdr = (struct vmgr_cmd_hdr *)sndbuf;
		hdr->kind = VMGR_CMD_RESPONSE;
		hdr->len = sizeof(control);
		memcpy((char *)(hdr+1), &control, sizeof(control)); 

		if (send(sock, sndbuf, sndlen, 0) < 0) {
			VMGR_LOG_SYSERR("failed to send()");
		}
		goto end;
		break;

	case VMGR_CMD_CONTROL_RESET:
		if (video_rst_ctrl(cmd->hdr.sub) < 0) {
			sprintf(tmp, "%% error - %s\n", strerror(errno));
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
		}
		if (cmd->hdr.sub >= VMGR_CMD_SUB_BRIGHTNESS && 
		    cmd->hdr.sub <= VMGR_CMD_SUB_EXPOSURE_RELATIVE) {
			vd_in->control[cmd->hdr.sub - 1].valid = 0;
			vd_in->control[cmd->hdr.sub - 1].value = 0;
		}
		change = 1;
		break;

	case VMGR_CMD_CONTROL_DIRECT:
		if (video_direct_ctrl(cmd->hdr.sub, cmd->conf.ctrl) < 0) {
			sprintf(tmp, "%% error - %s\n", strerror(errno));
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
		}
		if (cmd->hdr.sub >= VMGR_CMD_SUB_BRIGHTNESS && 
		    cmd->hdr.sub <= VMGR_CMD_SUB_EXPOSURE_RELATIVE) {
			vd_in->control[cmd->hdr.sub - 1].valid = 1;
			vd_in->control[cmd->hdr.sub - 1].value = cmd->conf.ctrl;
		}
		change = 1;
		break;

	case VMGR_CMD_SAVE_START:
		set_state(VMGR_STREAM_SAVE);
		if (check_streaming() < 0) {
			sprintf(tmp, "%% error - %s\n", strerror(errno));
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
		}

		break;

	case VMGR_CMD_SAVE_STOP:
		clear_state(VMGR_STREAM_SAVE);
		if (check_streaming() < 0) {
			sprintf(tmp, "%% error - %s\n", strerror(errno));
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
		}
		break;

	case VMGR_CMD_PARAM_SET_MEDIA:
		/* 現在値と設定値が異なる場合のみ */
		if (vd_in->media != cmd->conf.media) {
			vd_in->media = cmd->conf.media;

			/* AVIファイルのCLOSE処理 */
			close_avifile(get_current_mode());

			change = 1;
		}
		break;

	case VMGR_CMD_PARAM_SET_AVIMAX:
		/* 現在値と設定値が異なる場合のみ */
		if (vd_in->avisizemax != cmd->conf.avi_max) {
			vd_in->avisizemax = cmd->conf.avi_max;
			change = 1;
		}
		break;

	case VMGR_CMD_PARAM_SET_P_FTYPE:
		cnf_info[0].ftype = tmp_info[0].ftype = cmd->conf.info.ftype;
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_S_FTYPE:
		cnf_info[1].ftype = tmp_info[1].ftype = cmd->conf.info.ftype;
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_P_CYCLE:
		if (cmd->conf.info.cycle == 0) {
			cnf_info[0].cycle = tmp_info[0].cycle = 1;
		} else {
			cnf_info[0].cycle = tmp_info[0].cycle =
					cmd->conf.info.cycle;
		}
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_S_CYCLE:
		if (cmd->conf.info.cycle == 0) {
			cnf_info[1].cycle = tmp_info[1].cycle = 1;
		} else {
			cnf_info[1].cycle = tmp_info[1].cycle =
					cmd->conf.info.cycle;
		}
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_P_MODE:
		cnf_info[0].mode = tmp_info[0].mode = cmd->conf.info.mode;
		is_changed = 1;
		change = 1;
		break;
	case VMGR_CMD_PARAM_SET_S_MODE:
		cnf_info[1].mode = tmp_info[1].mode = cmd->conf.info.mode;
		is_changed = 1;
		change = 1;
		break;

	case VMGR_CMD_PARAM_SET_TIMER:
		for (i = 0; i < VMGR_SAVE_TIMER_ENTRY; i++) {
			if (vd_in->timer[i].dayofweek == VMGR_DAYOFWEEK_NONE) {
				vd_in->timer[i].dayofweek =
						cmd->conf.timer.dayofweek;
				vd_in->timer[i].start.tm_hour =
						cmd->conf.timer.start.tm_hour;
				vd_in->timer[i].start.tm_min =
						cmd->conf.timer.start.tm_min;
				vd_in->timer[i].start.tm_sec =
						cmd->conf.timer.start.tm_sec;
				vd_in->timer[i].end.tm_hour =
						cmd->conf.timer.end.tm_hour;
				vd_in->timer[i].end.tm_min =
						cmd->conf.timer.end.tm_min;
				vd_in->timer[i].end.tm_sec =
						cmd->conf.timer.end.tm_sec;

				vd_in->timer_enable = VMGR_TIMER_ENABLE;
				break;
			}
		}
		change = 1;
		break;

	case VMGR_CMD_PARAM_DELETE_TIMER:
		for (i = 0; i < VMGR_SAVE_TIMER_ENTRY; i++) {
			if (vd_in->timer[i].dayofweek ==
						cmd->conf.timer.dayofweek &&
			    vd_in->timer[i].start.tm_hour ==
						cmd->conf.timer.start.tm_hour &&
			    vd_in->timer[i].start.tm_min ==
						cmd->conf.timer.start.tm_min &&
			    vd_in->timer[i].start.tm_sec ==
						cmd->conf.timer.start.tm_sec &&
			    vd_in->timer[i].end.tm_hour ==
						cmd->conf.timer.end.tm_hour &&
			    vd_in->timer[i].end.tm_min ==
						cmd->conf.timer.end.tm_min &&
			    vd_in->timer[i].end.tm_sec ==
						cmd->conf.timer.end.tm_sec) {
				vd_in->timer[i].dayofweek = VMGR_DAYOFWEEK_NONE;
				vd_in->timer[i].start.tm_hour = 0;
				vd_in->timer[i].start.tm_min = 0;
				vd_in->timer[i].start.tm_sec = 0;
				vd_in->timer[i].end.tm_hour = 0;
				vd_in->timer[i].end.tm_min = 0;
				vd_in->timer[i].end.tm_sec = 0;

				break;
			}
		}
		for (i = 0; i < VMGR_SAVE_TIMER_ENTRY; i++) {
			if (vd_in->timer[i].dayofweek != VMGR_DAYOFWEEK_NONE) {
				break;
			}
		}
		if (i >= VMGR_SAVE_TIMER_ENTRY) {
			vd_in->timer_enable = VMGR_TIMER_DISABLE;
		}
		change = 1;
		break;

	case VMGR_CMD_BROWSER_TIMER:
		vd_in->browser_timer = cmd->conf.browser_timer;
		browser_access_flg = 1;
		change = 1;
		break;

	case VMGR_CMD_BROWSER_ACCESSED:
		browser_access_flg = 1;
		break;
	
	case VMGR_CMD_DISCONNECT_PORT: {
#if ATERMLX_PENDING
		int error = -1;
		int usbdev = -1;
		int addr;
		char usbdevname[10];

		snprintf(usbdevname, sizeof(usbdevname), "/dev/usb%u", 
				vd_in->cap.busno);	
		usbdev = open(usbdevname, O_RDWR);
		if(usbdev < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device %s\n",
					usbdevname);
			break;
		}
		addr = vd_in->cap.dev_addr;

		if((vd_in->cap.portno == cmd->hdr.portno) ||
				(cmd->hdr.portno == -1)){
			error = ioctl(usbdev, USB_DEV_SUSPEND, &addr);
			if (error < 0) {
				VMGR_LOG(LOG_ERR, VMGR_LOCATION,
					 "USB_DEV_SUSPEND %s", strerror(errno));
			} else {
				VMGR_LOG(LOG_INFO, VMGR_LOCATION,
					 "USB_DEV_SUSPEND success");
			}
		}
		if (usbdev >= 0) {
			close(usbdev);
		}
#endif
		break;

	case VMGR_CMD_BUFFER_COUNT:
		videomgr_vbuf_count = cmd->conf.buffer_count;
		break;

	case VMGR_CMD_STREAMING_STATUS:
		snprintf(buf, VMGR_MSG_BUFSIZE, "%s", video_str_streaming);
		break;

	case VMGR_CMD_CAMINFO_SETTABLE_COUNT:
		usb_caminfo_settable_count(buf);
		break;

	case VMGR_CMD_CAMINFO_SETTABLE:
		usb_caminfo_settable(buf);
		break;

	case VMGR_CMD_CAMINFO_CURRENT_SETTING:
		usb_caminfo_current_setting(buf);
		break;
	}

	default :
		break;
	}

	if (strlen(buf) > 0) {
		sndlen = sizeof(*hdr) + strlen(buf) + 1;
		sndbuf = malloc(sndlen);
		if (sndbuf == NULL) {
			VMGR_LOG_SYSERR("failed to malloc()");
			ret = -1;
			goto end;
		}
		hdr = (struct vmgr_cmd_hdr *)sndbuf;
		hdr->kind = cmd->hdr.kind;
		hdr->len = strlen(buf) + 1;
		strlcpy((char *)(hdr + 1), buf, strlen(buf) + 1); 
		VMGR_LOG(LOG_INFO, VMGR_LOCATION, "sndbuf:%s\n", sndbuf);
		VMGR_LOG(LOG_INFO, VMGR_LOCATION, "sndlen:%d\n", sndlen);
		if (send(sock, sndbuf, sndlen, 0) < 0) {
			VMGR_LOG_SYSERR("failed to send()");
		}
	}

end:
	if (buf != NULL) {
		free(buf);
	}
	if (sndbuf != NULL) {
		free(sndbuf);
	}

	if (change) {
		vmgr_save_config();
	}

	return ret;
}
#pragma GCC diagnostic warning "-Wcast-align"

static int
show_videomgr(char *buf)
{
	int	i;

	snprintf(buf, VMGR_MSG_BUFSIZE, "\n streaming %s\n",
				video_str_streaming);

	snprintf(buf, VMGR_MSG_BUFSIZE, "%s browser timer %u\n",
				buf,
				vd_in->browser_timer);

#if 0
	strlcat(buf, " capture ", VMGR_MSG_BUFSIZE);

	switch (vd_in->get_pict) {
	case VMGR_CAP_1SHOT:
		strlcat(buf, "one shot\n", VMGR_MSG_BUFSIZE);
		break;
	case VMGR_CAP_CONTINUOUS:
		strlcat(buf, "continuous\n", VMGR_MSG_BUFSIZE);
		break;
	case VMGR_CAP_OFF:
		strlcat(buf, "stop\n", VMGR_MSG_BUFSIZE);
		break;
	}
#endif

	snprintf(buf, VMGR_MSG_BUFSIZE, "%s\n media server : %s\n",
			buf,
			(vd_in->media == VMGR_SAVE_MEDIA_PRIVATE ?
			"private" : "public"));
	snprintf(buf, VMGR_MSG_BUFSIZE,
			"%s\n AVI max file size : %lu Mbyte\n",
			buf,
			vd_in->avisizemax);

	strlcat(buf, "\n [Current Setting]\n", VMGR_MSG_BUFSIZE);
	if (vd_in->fd < 0) {
		strcat(buf, "   Unknown(Device not configured)\n");
	} else {
		if (enum_def_cur_setting(buf, 2)) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"Unable to enumerate current setting\n");
			goto error;
		}
		enum_save_config(0, SHOW_MODE_STRING_ON, buf);
		enum_save_config(1, SHOW_MODE_STRING_ON, buf);
	}

	strlcat(buf, "\n [Preview Setting]\n", VMGR_MSG_BUFSIZE);
	if (enum_def_cur_setting(buf, 3)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"Unable to enumerate current setting\n");
		goto error;
	}

	enum_save_config(0, SHOW_MODE_STRING_OFF, buf);

	strlcat(buf, " [Schedule Setting]\n", VMGR_MSG_BUFSIZE);
	if (enum_def_cur_setting(buf, 4)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"Unable to enumerate current setting\n");
		goto error;
	}

	enum_save_config(1, SHOW_MODE_STRING_OFF, buf);

        for (i = 0; i < VMGR_SAVE_TIMER_ENTRY; i++) {
                if (vd_in->timer[i].dayofweek != VMGR_DAYOFWEEK_NONE) {
			snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s   timer", buf);
                        switch (vd_in->timer[i].dayofweek) {
                        case VMGR_DAYOFWEEK_EVERYDAY:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s everyday", buf);
                                break;
                        case VMGR_DAYOFWEEK_SUN:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s sun", buf);
                                break;
                        case VMGR_DAYOFWEEK_MON:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s mon", buf);
                                break;
                        case VMGR_DAYOFWEEK_TUE:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s tue", buf);
                                break;
                        case VMGR_DAYOFWEEK_WED:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s wed", buf);
                                break;
                        case VMGR_DAYOFWEEK_THU:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s thu", buf);
                                break;
                        case VMGR_DAYOFWEEK_FRI:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s fri", buf);
                                break;
                        case VMGR_DAYOFWEEK_SAT:
				snprintf(buf, VMGR_MSG_BUFSIZE,
					"%s sat", buf);
                                break;
                        default:
                                continue;
                                break;
                        }
			snprintf(buf, VMGR_MSG_BUFSIZE,
                                   "%s %02d:%02d:%02d %02d:%02d:%02d\n",
                                   buf,
                                   vd_in->timer[i].start.tm_hour,
                                   vd_in->timer[i].start.tm_min,
                                   vd_in->timer[i].start.tm_sec,
                                   vd_in->timer[i].end.tm_hour,
                                   vd_in->timer[i].end.tm_min,
                                   vd_in->timer[i].end.tm_sec);
                }
        }

	return 0;
error:
	return -1;
}

static int
show_usb_devinfo(char *buf)
{
	char tmp[VMGR_CAP_CNSIZE*2+3] = {'\0'};

	if (vd_in->fd < 0) {
		errno = ENXIO;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		goto error;
	}

	snprintf(tmp, sizeof(tmp), "%s %s\n", vd_in->cap.bus_info,
					vd_in->cap.card);
	strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

	return 0;
error:
	return -1;
}

static int
show_usb_caminfo(char *buf)
{
	char tmp[VMGR_CAP_CNSIZE*2+4] = {'\0'};

	if (vd_in->fd < 0) {
		errno = ENXIO;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		goto error;
	}

	sprintf(buf, "[UVC Camera Information]\n");
	snprintf(tmp, sizeof(tmp), " %s %s\n", vd_in->cap.bus_info,
				vd_in->cap.card);
	strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

	strlcat(buf, "[Current Setting Information]\n", VMGR_MSG_BUFSIZE);
	if (enum_def_cur_setting(buf, 1)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"Unable to enumerate current setting\n");
		goto error;
	}

	strlcat(buf, "[Settable Information]\n", VMGR_MSG_BUFSIZE);
	if (enum_frame_formats(NULL, 0, buf)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"Unable to enumerate frame formats\n");
		goto error;
	}

	strlcat(buf, "[Support Control Request]\n", VMGR_MSG_BUFSIZE);
	if (enum_controls(buf)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "Unable to enumerate controls\n");
		goto error;
	}

	return 0;
error:
	return -1;
}

static int
enum_frame_intervals(uint32_t pixfmt, uint32_t width, uint32_t height,
		char *buf)
{
	int ret;
	struct v4l2_frmivalenum fival;
	char tmp[128];

	memset(&fival, 0, sizeof(fival));
	fival.index = 0;
	fival.pixel_format = pixfmt;
	fival.width = width;
	fival.height = height;

	strlcat(buf, " Frame rate:", VMGR_MSG_BUFSIZE);
	while ((ret = ioctl(vd_in->fd, VIDIOC_ENUM_FRAMEINTERVALS,
	       &fival)) == 0) {
		if (fival.type == V4L2_FRMIVAL_TYPE_DISCRETE) {
			sprintf(tmp, "%u/%u, ", fival.discrete.numerator,
				fival.discrete.denominator);
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
		} else if (fival.type == V4L2_FRMIVAL_TYPE_CONTINUOUS) {
			sprintf(tmp, "(%u/%u) --- (%u/%u)",
			       fival.stepwise.min.numerator,
			       fival.stepwise.min.numerator,
			       fival.stepwise.max.denominator,
			       fival.stepwise.max.denominator);
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			break;
		} else if (fival.type == V4L2_FRMIVAL_TYPE_STEPWISE) {
			sprintf(tmp, "(%u/%u) --- (%u/%u)\n"
			       "stepsize %u/%u",
			       fival.stepwise.min.numerator,
			       fival.stepwise.min.denominator,
			       fival.stepwise.max.numerator,
			       fival.stepwise.max.denominator,
			       fival.stepwise.step.numerator,
			       fival.stepwise.step.denominator);
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			break;
		}
		fival.index++;
	}
	buf[strlen(buf)-2] = '\n';
	buf[strlen(buf)-1] = '\0';
		
	if (ret != 0 && errno != EINVAL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"ERROR enumerating frame intervals\n");
		return errno;
	}

	return 0;
}

static int
enum_frame_sizes(uint32_t pixfmt, char *buf)
{
	int ret;
	struct v4l2_frmsizeenum fsize;
	char tmp[256];

	memset(&fsize, 0, sizeof(fsize));
	fsize.index = 0;
	fsize.pixel_format = pixfmt;

	while ((ret = ioctl(vd_in->fd, VIDIOC_ENUM_FRAMESIZES, &fsize)) == 0) {
		if (fsize.type == V4L2_FRMIVAL_TYPE_DISCRETE) {
			sprintf(tmp, "  Size:%4ux%4u",
				fsize.discrete.width, fsize.discrete.height);
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			ret = enum_frame_intervals(pixfmt,
						   fsize.discrete.width,
						   fsize.discrete.height,
						   buf);
			if (ret != 0)
				VMGR_LOG(LOG_ERR, VMGR_LOCATION,
					 "Unable to enumerate frame sizes.\n");
		} else if (fsize.type == V4L2_FRMIVAL_TYPE_CONTINUOUS) {
			sprintf(tmp, " continuous: (width = %u, height = %u) --- "
				 "(width = %u, height = %u)\n",
				 fsize.stepwise.min_width,
				 fsize.stepwise.min_height,
				 fsize.stepwise.max_width,
				 fsize.stepwise.max_height);
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				 "Refusing to enumerate frame intervals.\n");
			break;
		} else if (fsize.type == V4L2_FRMIVAL_TYPE_STEPWISE) {
			sprintf(tmp, " stepwise: (width = %u, height = %u) --- "
				"(width = %u, height = %u)\n"
				"stepsize width = %u, height = %u\n",
				fsize.stepwise.min_width,
				fsize.stepwise.min_height,
				fsize.stepwise.max_width,
				fsize.stepwise.max_height,
				fsize.stepwise.step_width,
				fsize.stepwise.step_height);
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				 "Refusing to enumerate frame intervals.\n");
			break;
		}
		fsize.index++;
	}
	if (ret != 0 && errno != EINVAL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "ERROR enumerating frame sizes\n");
		return errno;
	}

	return 0;
}

static int
enum_def_cur_setting(char *buf, int flag)
{
	struct v4l2_format vf;
	struct v4l2_streamparm vs;
	char tmp[64];
	int mode;

	if (flag == 0) {
		/* Default setting */
		vf.fmt.pix.pixelformat = VMGR_DEF_FORMAT;
		vf.fmt.pix.width = VMGR_DEF_WIDTH;
		vf.fmt.pix.height = VMGR_DEF_HEIGHT;
		vs.parm.capture.timeperframe.numerator = VMGR_DEF_FPS_NUM;
		vs.parm.capture.timeperframe.denominator = VMGR_DEF_FPS_DEN;
	} else if (flag == 1 || flag == 2) {
		/* Current setting */
		memset(&vf, 0, sizeof(vf));
		vf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (ioctl(vd_in->fd, VIDIOC_G_FMT, &vf) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				 "ioctl(VIDIOC_G_FMT) fail(%d)\n", errno);
			return 1;
		}
		memset(&vs, 0, sizeof(vs));
		vs.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (ioctl(vd_in->fd, VIDIOC_G_PARM, &vs) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				 "ioctl(VIDIOC_G_PARM) fail\n");
			return 1;
		}
	} else {
		mode = flag - 3;
		/* Change setting */
		memset(&vf, 0, sizeof(vf));
		memset(&vs, 0, sizeof(vs));

		if (tmp_info[mode].format) {
			vf.fmt.pix.pixelformat = tmp_info[mode].format;
		} else {
			vf.fmt.pix.pixelformat = vd_in->info[mode].format;
		}
		if (tmp_info[mode].width) {
			vf.fmt.pix.width = tmp_info[mode].width;
		} else {
			vf.fmt.pix.width = vd_in->info[mode].width;
		}
		if (tmp_info[mode].height) {
			vf.fmt.pix.height = tmp_info[mode].height;
		} else {
			vf.fmt.pix.height = vd_in->info[mode].height;
		}
		if (tmp_info[mode].fps[0]) {
			vs.parm.capture.timeperframe.numerator = 
						tmp_info[mode].fps[0];
		} else {
			vs.parm.capture.timeperframe.numerator = 
						vd_in->info[mode].fps[0];
		}
		if (tmp_info[mode].fps[1]) {
			vs.parm.capture.timeperframe.denominator =
						tmp_info[mode].fps[1];
		} else {
			vs.parm.capture.timeperframe.denominator =
						vd_in->info[mode].fps[1];
		}
	}

	if (flag == 2 || flag == 3 || flag == 4) {
		strlcat(buf, "  ", VMGR_MSG_BUFSIZE);
	}

	/* Format type */
	GET_PIXFMT(tmp, vf.fmt.pix);
	strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

	if (flag == 2 || flag == 3 || flag == 4) {
		strlcat(buf, "  ", VMGR_MSG_BUFSIZE);
	}

	/* Size */
	sprintf(tmp, " Size:%4ux%4u\n", vf.fmt.pix.width, vf.fmt.pix.height);
	strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

	if (flag == 2 || flag == 3 || flag == 4) {
		strlcat(buf, "  ", VMGR_MSG_BUFSIZE);
	}

	/* Frame rate */
	snprintf(tmp, sizeof(tmp), " Frame rate: %u/%u\n",
		vs.parm.capture.timeperframe.numerator,
		vs.parm.capture.timeperframe.denominator);
	strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

	return 0;
}

static int
enum_frame_formats(unsigned int *sup_fmt, unsigned int max_fmt, char *buf)
{
	int ret;
	char tmp[128];
	struct v4l2_fmtdesc fmt;

	memset(&fmt, 0, sizeof(fmt));
	fmt.index = 0;
	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

	while ((ret = ioctl(vd_in->fd, VIDIOC_ENUM_FMT, &fmt)) == 0) {
		if (sup_fmt == NULL) {
			GET_PIXFMT(tmp, fmt);
			if (buf != NULL) {
				strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
				ret = enum_frame_sizes(fmt.pixelformat, buf);
				if (ret != 0) {
					VMGR_LOG(LOG_ERR, VMGR_LOCATION,
					 "Unable to enumerate frame sizes.\n");
				}
			}
		} else if(fmt.index < max_fmt) {
			sup_fmt[fmt.index] = fmt.pixelformat;
			GET_PIXFMT(tmp, fmt);
			if (buf != NULL) {
				strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			}
		}
		fmt.index++;
	}
	if (errno != EINVAL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			"ERROR enumerating frame formats\n");
		return errno;
	}

        return 0;
}

static void
enum_save_config(int mode, int show, char *buf)
{
	char mstr[10];
	int current = 0;

	memset(mstr, 0, sizeof(mstr));
	if (show == SHOW_MODE_STRING_ON) {
		sprintf(mstr, "%s ", mode == 0 ? "preview" : "schedule");
		current = 1;
	}
	if (tmp_info[mode].ftype == 0 || current) {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s\n   %ssave file type : %s\n",
				buf,
				mstr,
				(vd_in->info[mode].ftype ==
				VMGR_SAVE_FTYPE_AVI ?  "AVI" : "JPEG"));
	} else {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s\n   %ssave file type : %s\n",
				buf,
				mstr,
				(tmp_info[mode].ftype == VMGR_SAVE_FTYPE_AVI ?
				"AVI" : "JPEG"));
	}

	if (tmp_info[mode].cycle == 0 || current) {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s   %ssave cycle : %d\n",
				buf,
				mstr,
				vd_in->info[mode].cycle);
	} else {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s   %ssave cycle : %d\n",
				buf,
				mstr,
				tmp_info[mode].cycle ==
				1 ? 0 : tmp_info[mode].cycle);
	}

	if (tmp_info[mode].mode == 0 || current) {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s   %ssave disk full : %s\n",
				buf,
				mstr,
				(vd_in->info[mode].mode == VMGR_SAVE_MODE_STOP ?
				"stop" : "cyclic"));
	} else {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s   %ssave disk full : %s\n",
				buf,
				mstr,
				(tmp_info[mode].mode == VMGR_SAVE_MODE_STOP ?
				"stop" : "cyclic"));
	}

	if (current == 0) {
		snprintf(buf, VMGR_MSG_BUFSIZE,
				"%s\n",
				buf);
	}
}

void
videomgr_syslog(int pri, const char *func, const char *fmt, ...)
{
	static char buf[800];
	char *newfmt;
	va_list ap;

	if (!videomgr_log_enable)
		return;
	if (videomgr_log_level < pri)
		return;

	va_start(ap, fmt);

	newfmt = buf;
	if (videomgr_print_location)
		sprintf(newfmt, "%s: %s", func, fmt);
	else
		sprintf(newfmt,  "%s", fmt);

	if (videomgr_debug) {
		vprintf(newfmt, ap);
		printf("\n");
	} else {
		vsyslog(pri, newfmt, ap);
	}
	va_end(ap);
}

const char *
videomgr_location(const char *file, int line)
{
	static char buf[1024];
	const char *p;

	/* truncate pathname */
	p = strrchr(file, '/');
	if (p)
		p++;
	else
		p = file;

	snprintf(buf, sizeof(buf), "%s:%d", p, line);

	return buf;
}

static int
enum_controls(char *buf)
{    
	struct v4l2_queryctrl queryctrl;
	struct v4l2_querymenu querymenu;
	struct v4l2_control control_s;
	int id;
	int i;
	char tmp[128];
	int ret = 0;

	sprintf(tmp,
	" control name              default   step      current   scope\n");

	strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
	for (id = V4L2_CID_BASE;id < V4L2_CID_LASTP1; id++) {
		memset(&queryctrl, 0, sizeof (queryctrl));

		queryctrl.id = id;

		if (ioctl(vd_in->fd, VIDIOC_QUERYCTRL, &queryctrl) == 0) {
			if (queryctrl.flags & V4L2_CTRL_FLAG_DISABLED) {
				continue;
			}

			control_s.id = queryctrl.id;
			if (ioctl(vd_in->fd, VIDIOC_G_CTRL, &control_s) < 0) {
				ret = -1;
				goto end;
			}
			usleep(10);
			if (queryctrl.flags & V4L2_CTRL_FLAG_READ_ONLY) {
				sprintf(tmp,
	" %-25s -         -         %-9d (Read only)\n",
					queryctrl.name, 
					control_s.value);
			} else {
				sprintf(tmp, " %-25s", queryctrl.name);

#if ATERMLX_PENDING
				if (queryctrl.flags & VDF_CTRL_FLAG_DEF) {
					sprintf(tmp, "%s %-9d", tmp,
						queryctrl.default_value);
				} else {
					sprintf(tmp, "%s -        ", tmp);
				}
				if (queryctrl.flags & VDF_CTRL_FLAG_STEP) {
					sprintf(tmp, "%s %-9d", tmp,
						queryctrl.step);
				} else {
					sprintf(tmp, "%s -        ", tmp);
				}
#endif

				sprintf(tmp, "%s %-9d", tmp,
						control_s.value);

				if (queryctrl.type == V4L2_CTRL_TYPE_BOOLEAN) {
					sprintf(tmp,
						"%s (0:disable,1:enable)\n",
						tmp);
				} else
#if ATERMLX_PENDING
				if ((queryctrl.flags & VDF_CTRL_FLAG_MIN) || 
				    (queryctrl.flags & VDF_CTRL_FLAG_MAX)) {
					sprintf(tmp, "%s (%5d - %5d)\n", tmp,
						queryctrl.minimum,
						queryctrl.maximum);
				} else {
					sprintf(tmp, "%s\n", tmp);
				}
#endif

				if (queryctrl.type == V4L2_CTRL_TYPE_MENU) {
					memset(&querymenu, 0,
							sizeof (querymenu));

					for (i = 0; i <= queryctrl.maximum;
									i++) {
						querymenu.id = id;
						querymenu.index = i;

						if (ioctl(vd_in->fd,
						    VIDIOC_QUERYMENU,
						    &querymenu) != 0) {
							if (errno == EINVAL ||
							    errno == ENODEV) {
								continue;
							}

							VMGR_LOG(LOG_ERR,
			VMGR_LOCATION, 
			"ERROR getting controls(%x:%d)\n",
			id, errno);
							ret = -1;
							goto end;
						}
						strlcat(buf, tmp,
							VMGR_MSG_BUFSIZE);
						sprintf(tmp,
							"      %d : %s\n",
							querymenu.index,
							querymenu.name);
					}
				}
			}
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

		} else {

			if (errno == EINVAL || errno == ENODEV) {
				continue;
			}

			VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
					"ERROR getting controls(%x:%d)\n",
					id, errno);
			ret = -1;
			goto end;
		}
	}

	/* driver specific controls */
	for (id = V4L2_CID_PRIVATE_BASE; ; id++) {
		memset (&queryctrl, 0, sizeof (queryctrl));

		queryctrl.id = id;

		if (ioctl(vd_in->fd, VIDIOC_QUERYCTRL, &queryctrl) == 0) {
			if (queryctrl.flags & V4L2_CTRL_FLAG_DISABLED) {
				continue;
			}

			control_s.id = queryctrl.id;

			if (ioctl(vd_in->fd, VIDIOC_G_CTRL, &control_s) < 0) {
				ret = -1;
				goto end;
			}
			usleep(10);
			if (queryctrl.flags & V4L2_CTRL_FLAG_READ_ONLY) {
				sprintf(tmp,
	" %-25s -         -         %-9d (Read only)\n",
					queryctrl.name, 
					control_s.value);
			} else {
				sprintf(tmp, " %-25s", queryctrl.name);

#if ATERMLX_PENDING
				if (queryctrl.flags & VDF_CTRL_FLAG_DEF) {
					sprintf(tmp, "%s %-9d", tmp,
						queryctrl.default_value);
				} else {
					sprintf(tmp, "%s -        ", tmp);
				}
				if (queryctrl.flags & VDF_CTRL_FLAG_STEP) {
					sprintf(tmp, "%s %-9d", tmp,
						queryctrl.step);
				} else {
					sprintf(tmp, "%s -        ", tmp);
				}
#endif

				sprintf(tmp, "%s %-9d", tmp,
						control_s.value);

				if (queryctrl.type == V4L2_CTRL_TYPE_BOOLEAN) {
					sprintf(tmp,
						"%s (0:disable,1:enable)\n",
						tmp);
#if ATERMLX_PENDING
				} else
				if ((queryctrl.flags & VDF_CTRL_FLAG_MIN) || 
				    (queryctrl.flags & VDF_CTRL_FLAG_MAX)) {
					sprintf(tmp, "%s (%5d - %5d)\n", tmp,
						queryctrl.minimum,
						queryctrl.maximum);
#endif
				} else {
					sprintf(tmp, "%s\n", tmp);
				}
				if (queryctrl.type == V4L2_CTRL_TYPE_MENU) {
					memset(&querymenu, 0,
							sizeof (querymenu));

					for (i = 0; i <= queryctrl.maximum;
									i++) {
						querymenu.id = id;
						querymenu.index = i;

						if (ioctl(vd_in->fd,
						    VIDIOC_QUERYMENU,
						    &querymenu) != 0) {
							if (errno == EINVAL ||
							    errno == ENODEV) {
								continue;
							}

							VMGR_LOG(LOG_ERR,
			VMGR_LOCATION, 
			"ERROR getting controls(%x:%d)\n",
			id, errno);
							ret = -1;
							goto end;
						}
						strlcat(buf, tmp,
							VMGR_MSG_BUFSIZE);
						sprintf(tmp,
							"      %d : %s\n",
							querymenu.index,
							querymenu.name);
					}
				}
			}
			strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

		} else {
			VMGR_LOG(LOG_INFO, VMGR_LOCATION, 
					"specific controls nothing(%x:%d)\n",
					id, errno);
			ret = -1;
			goto end;
		}
	}

end:

	return ret;
}


static u_int32_t
change_video_ctrl(int control, struct v4l2_queryctrl *queryctrl)
{
	/* Command-IDをControl-IDに変換 */
	switch (control) {
	case VMGR_CMD_SUB_BRIGHTNESS:
		queryctrl->id = V4L2_CID_BRIGHTNESS;
		break;
	case VMGR_CMD_SUB_CONTRAST:
		queryctrl->id = V4L2_CID_CONTRAST;
		break;
	case VMGR_CMD_SUB_SATURATION:
		queryctrl->id = V4L2_CID_SATURATION;
		break;
	case VMGR_CMD_SUB_HUE:
		queryctrl->id = V4L2_CID_HUE;
		break;
	case VMGR_CMD_SUB_BLUE:
		queryctrl->id = V4L2_CID_BLUE_BALANCE;
		break;
	case VMGR_CMD_SUB_RED:
		queryctrl->id = V4L2_CID_RED_BALANCE;
		break;
	case VMGR_CMD_SUB_WHITE:
		queryctrl->id = V4L2_CID_WHITE_BALANCE_TEMPERATURE;
		break;
	case VMGR_CMD_SUB_GAMMA:
		queryctrl->id = V4L2_CID_GAMMA;
		break;
	case VMGR_CMD_SUB_GAIN:
		queryctrl->id = V4L2_CID_GAIN;
		break;
	case VMGR_CMD_SUB_BACKLIGHT:
		queryctrl->id = V4L2_CID_BACKLIGHT_COMPENSATION;
		break;
	case VMGR_CMD_SUB_POWER_LINE:
		queryctrl->id = V4L2_CID_POWER_LINE_FREQUENCY;
		break;
	case VMGR_CMD_SUB_SHARPNESS:
		queryctrl->id = V4L2_CID_SHARPNESS;
		break;
	case VMGR_CMD_SUB_FOCUS:
		queryctrl->id = V4L2_CID_FOCUS_ABSOLUTE;
		break;
	case VMGR_CMD_SUB_FOCUS_RELATIVE:
		queryctrl->id = V4L2_CID_FOCUS_RELATIVE;
		break;
	case VMGR_CMD_SUB_IRIS:
		queryctrl->id = V4L2_CID_IRIS_ABSOLUTE;
		break;
	case VMGR_CMD_SUB_IRIS_RELATIVE:
		queryctrl->id = V4L2_CID_IRIS_RELATIVE;
		break;
	case VMGR_CMD_SUB_ZOOM:
		queryctrl->id = V4L2_CID_ZOOM_ABSOLUTE;
		break;
	case VMGR_CMD_SUB_ZOOM_RELATIVE:
		queryctrl->id = V4L2_CID_ZOOM_RELATIVE;
		break;
	case VMGR_CMD_SUB_PAN:
		queryctrl->id = V4L2_CID_PAN_ABSOLUTE;
		break;
	case VMGR_CMD_SUB_PAN_RELATIVE:
		queryctrl->id = V4L2_CID_PAN_RELATIVE;
		break;
	case VMGR_CMD_SUB_TILT:
		queryctrl->id = V4L2_CID_TILT_ABSOLUTE;
		break;
	case VMGR_CMD_SUB_TILT_RELATIVE:
		queryctrl->id = V4L2_CID_TILT_RELATIVE;
		break;
	case VMGR_CMD_SUB_PRIVACY:
		queryctrl->id = V4L2_CID_PRIVACY;
		break;
	case VMGR_CMD_SUB_EXPOSURE:
		queryctrl->id = V4L2_CID_EXPOSURE_ABSOLUTE;
		break;
	default:
		break;
	}

	if (queryctrl->id != 0) {
		if (ioctl(vd_in->fd, VIDIOC_QUERYCTRL, queryctrl) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
					"ioctl querycontrol error(%x:%s).\n",
					queryctrl->id,
					strerror(errno));
			return 0;
		} else if (queryctrl->flags & V4L2_CTRL_FLAG_DISABLED) {
			return 0;
		} else if (queryctrl->type == V4L2_CTRL_TYPE_BOOLEAN) {
		} else if (queryctrl->type == V4L2_CTRL_TYPE_INTEGER) {
		} else if (queryctrl->type == V4L2_CTRL_TYPE_MENU) {
		} else {
			return 0;
		}

		/* ReadOnlyに対して無効にする */
		if (queryctrl->flags & V4L2_CTRL_FLAG_READ_ONLY) {
			return 0;
		}
	}

	return queryctrl->id;
}

static void
auto_func_ctrl(int control, int flag)
{
	struct v4l2_queryctrl queryctrl;
	int err = 0;

	queryctrl.id = control;

	if (ioctl(vd_in->fd, VIDIOC_QUERYCTRL, &queryctrl) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
				"ioctl querycontrol error(%x:%s).\n",
				control,
				strerror(errno));
	} else if (queryctrl.flags & V4L2_CTRL_FLAG_DISABLED) {
	} else if (queryctrl.type == V4L2_CTRL_TYPE_BOOLEAN) {
		err = video_get_ctrl(control);

		if (err < 0) {
		} else if (err == 1 && flag == AUTO_FUNC_OFF) {
			video_tgl_ctrl(control, &queryctrl);
		} else if (err == 0 && flag == AUTO_FUNC_ON) {
			video_tgl_ctrl(control, &queryctrl);
		}
	} else if (queryctrl.type == V4L2_CTRL_TYPE_INTEGER || 
		   queryctrl.type == V4L2_CTRL_TYPE_MENU) {
		if (control == V4L2_CID_EXPOSURE_AUTO) {
			err = video_get_ctrl(control);
			if (err < 0) {
			} else if (err && flag == AUTO_FUNC_OFF) {
				video_set_ctrl(control, 0, &queryctrl);
			} else if (err == 0 && flag == AUTO_FUNC_ON) {
				video_set_ctrl(control,
					queryctrl.default_value, &queryctrl);
			}
		}
	}
}

static int
video_get_ctrl(int control)
{
	struct v4l2_control control_s;

	control_s.id = control;

	if (ioctl(vd_in->fd, VIDIOC_G_CTRL, &control_s) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"ioctl get control [%x] error.\n", control);
		return -1;
	}

	return control_s.value;
}

static int
video_set_ctrl(int control, int value, struct v4l2_queryctrl *queryctrl)
{
	struct v4l2_control control_s;
	int min, max;

	min = queryctrl->minimum;
	max = queryctrl->maximum;

	if ((value >= min) && (value <= max)) {
		control_s.id = control;
		control_s.value = value;

		if (ioctl(vd_in->fd, VIDIOC_S_CTRL, &control_s) < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"ioctl set control [%x:%d] error.\n", control,
				value);
			return -1;
		}
	}

	return 0;
}

static int
video_direct_get_ctrl(int control, struct vmgr_control *c)
{
	struct v4l2_queryctrl queryctrl;
	u_int32_t id;

	if (vd_in->fd < 0) {
		errno = ENXIO;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		return -1;
	}

	memset(&queryctrl, 0, sizeof(queryctrl));
	if ((id = change_video_ctrl(control, &queryctrl)) == 0) {
		errno = ENODEV;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
				"control not supported(%x)\n", control);
		return -1;
	}

	c->min = queryctrl.minimum;
	c->max = queryctrl.maximum;
	c->step = queryctrl.step;
	c->default_value = queryctrl.default_value;
	c->current = video_get_ctrl(id);
	c->flags = queryctrl.flags;
	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION,
		"min[%d] max[%d] step[%d] default[%d] current[%d]\n",
		c->min,
		c->max,
		c->step,
		c->default_value,
		c->current);

	return 0;
}

static int
video_direct_ctrl(int control, long value)
{
	struct v4l2_queryctrl queryctrl;
	int min, max, current, step, val_def;
	int err;
	u_int32_t id;

	if (vd_in->fd < 0) {
		errno = ENXIO;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		return -1;
	}

	memset(&queryctrl, 0, sizeof(queryctrl));
	if ((id = change_video_ctrl(control, &queryctrl)) == 0) {
		errno = ENODEV;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
				"control not supported(%x)\n", control);
		return -1;
	}

	/* Auto機能があるControlは、Auto機能の無効化を行う */
	switch (control) {
	case VMGR_CMD_SUB_HUE:
		auto_func_ctrl(V4L2_CID_HUE_AUTO, AUTO_FUNC_OFF);
		break;
	case VMGR_CMD_SUB_RED:
	case VMGR_CMD_SUB_BLUE:
		auto_func_ctrl(V4L2_CID_AUTO_WHITE_BALANCE, AUTO_FUNC_OFF);
		break;
	case VMGR_CMD_SUB_FOCUS:
	case VMGR_CMD_SUB_FOCUS_RELATIVE:
		auto_func_ctrl(V4L2_CID_FOCUS_AUTO, AUTO_FUNC_OFF);
		break;
	case VMGR_CMD_SUB_EXPOSURE:
	case VMGR_CMD_SUB_EXPOSURE_RELATIVE:
		auto_func_ctrl(V4L2_CID_EXPOSURE_AUTO, AUTO_FUNC_OFF);
		break;
	default:
		break;
	}

#if ATERMLX_PENDING
	if ((queryctrl.flags & VDF_CTRL_FLAG_MIN) &&
	    queryctrl.minimum > value) {
		value = queryctrl.minimum;
	}
	if ((queryctrl.flags & VDF_CTRL_FLAG_MAX) && 
	    queryctrl.maximum < value) {
		value = queryctrl.maximum;
	}
#endif

	if (queryctrl.type == V4L2_CTRL_TYPE_BOOLEAN) {
		err = video_get_ctrl(id);

		if (err < 0) {
			return -1;
		} else if (err != value) {
			return video_tgl_ctrl(id, &queryctrl);
		}
		return 0;
	}

	min = queryctrl.minimum;
	max = queryctrl.maximum;
	step = queryctrl.step;
	val_def = queryctrl.default_value;
	current = value;

	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, 
		"max %d, min %d, step %d, default %d ,current %d\n",
		max, min, step, val_def, current);

	if (video_set_ctrl(id, current, &queryctrl) < 0) {
		return -1;
	}            

	return 0;
}

static int
video_tgl_ctrl(int control, struct v4l2_queryctrl *queryctrl)
{
	int current;

	current = video_get_ctrl(control);
	if (video_set_ctrl(control, !current, queryctrl) < 0) {
		return -1;
	}            
	return 0;
}

static int
video_rst_ctrl(int control)
{
	struct v4l2_queryctrl queryctrl;
	u_int32_t id;

	if (vd_in->fd < 0) {
		errno = ENXIO;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		return -1;
	}

	memset(&queryctrl, 0, sizeof(queryctrl));
	if ((id = change_video_ctrl(control, &queryctrl)) == 0) {
		errno = ENODEV;
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"control not supported(%x)\n", control);
		return -1;
	}

#if ATERMLX_PENDING
	if (queryctrl.flags & VDF_CTRL_FLAG_DEF) {
		if (video_set_ctrl(id, queryctrl.default_value,
						&queryctrl) < 0) {
			return -1;
		}            
	} else {
		errno = EINVAL;
		return -1;
	}
#endif

	/* Auto機能があるControlは、Auto機能の有効に戻す */
	switch (control) {
       	case VMGR_CMD_SUB_HUE:
		auto_func_ctrl(V4L2_CID_HUE_AUTO, AUTO_FUNC_ON);
		break;
	case VMGR_CMD_SUB_BLUE:
	case VMGR_CMD_SUB_RED:
		auto_func_ctrl(V4L2_CID_AUTO_WHITE_BALANCE, AUTO_FUNC_ON);
		break;
	case VMGR_CMD_SUB_FOCUS:
	case VMGR_CMD_SUB_FOCUS_RELATIVE:
		auto_func_ctrl(V4L2_CID_FOCUS_AUTO, AUTO_FUNC_ON);
		break;
	case VMGR_CMD_SUB_EXPOSURE:
	case VMGR_CMD_SUB_EXPOSURE_RELATIVE:
		auto_func_ctrl(V4L2_CID_EXPOSURE_AUTO, AUTO_FUNC_ON);
		break;
	}

	return 0;
}

static unsigned long
get_time(void)
{
	struct timespec ts;

	(void)clock_gettime(CLOCK_REALTIME, &ts);

	return (unsigned long)ts.tv_sec;
}

static unsigned long
get_clock_time(void)
{
	struct timespec ts;

	(void)clock_gettime(CLOCK_MONOTONIC, &ts);

	return (unsigned long)ts.tv_sec;
}

static void
caputure_timing_check(int mode)
{
	struct timeval etime, diftime;
	struct timespec ts;
	long sec = 0, usec = 0;
	long caputure;

	(void)clock_gettime(CLOCK_REALTIME, &ts);
	VMGR_TIMESPEC_TO_TIMEVAL(&etime, &ts);

	diftime.tv_sec = etime.tv_sec - cap_stime.tv_sec;
	if (etime.tv_usec < cap_stime.tv_usec) {
		diftime.tv_sec--;
		etime.tv_usec += 1000000;
	}
	diftime.tv_usec = etime.tv_usec - cap_stime.tv_usec;

	/*
	 * 既に数秒経過している可能性があるので
	 * 秒の差分もμ秒に加算しておく
	 */
	diftime.tv_usec += diftime.tv_sec * 1000000;

	if (vd_in->info[mode].format == V4L2_PIX_FMT_YUYV) {
		caputure = vd_in->yuv_cap;
	} else  {
		/* 画像が100kを超える時は、刈り取りタイミングを変える) */
		if (vd_in->newlen > (100 * 1024)) {
			caputure = vd_in->jpg_cap[0];
		} else {
			caputure = vd_in->jpg_cap[1];
		}
	}

	if (diftime.tv_usec < caputure) {
		sec = (caputure - diftime.tv_usec) / 1000000;
		usec = (caputure - diftime.tv_usec) % 1000000;
	}

	if (sec > 0) {
		vmgr_sleep(sec);
	}
	if (usec > 0) {
		vmgr_usleep(usec);
	}

	(void)clock_gettime(CLOCK_REALTIME, &ts);
	VMGR_TIMESPEC_TO_TIMEVAL(&cap_stime, &ts);

	return;
}

static void
get_filename(int mode)
{
	time_t t;
	struct tm *tt;
	struct timespec ts;
	char datetime[VMGR_PATH_LEN];
	char tmp[VMGR_PATH_LEN];
	int i;
	struct stat s;

	(void)clock_gettime(CLOCK_REALTIME, &ts);
	t = ts.tv_sec;

	tt = localtime(&t);

	/* ディレクトリ名の作成 */
	sprintf(tmp, "%04d%02d%02d%02d00",
		tt->tm_year + 1900, tt->tm_mon + 1, tt->tm_mday,
		tt->tm_hour);

	/* フォルダが存在しているかチェック */
	if (vd_in->media == VMGR_SAVE_MEDIA_PRIVATE) {
		if (stat(VMGR_SAVE_FOLDER_CLS, &s) != 0) {
			mkdir(VMGR_SAVE_FOLDER_CLS, 0777);
		}
		sprintf(vd_in->filename[mode], "%s/%s",
					VMGR_SAVE_FOLDER_CLS, tmp);
	} else if (vd_in->media == VMGR_SAVE_MEDIA_PUBLIC) {
		if (stat(VMGR_SAVE_FOLDER, &s) != 0) {
			mkdir(VMGR_SAVE_FOLDER, 0777);
		}
		if (stat(VMGR_SAVE_FOLDER_OPN, &s) != 0) {
			mkdir(VMGR_SAVE_FOLDER_OPN, 0777);
		}
		sprintf(vd_in->filename[mode], "%s/%s",
					VMGR_SAVE_FOLDER_OPN, tmp);
	} else {
		return;
	}

	if (stat(vd_in->filename[mode], &s) != 0) {
		mkdir(vd_in->filename[mode], 0777);
	}

	sprintf(datetime, "%04d%02d%02d%02d%02d%02d",
			tt->tm_year + 1900, tt->tm_mon + 1,
			tt->tm_mday, tt->tm_hour, tt->tm_min,
			tt->tm_sec);

	/* フレームレートにより、ファイルが上書きされないようにする */
	for (i = 0; i < vd_in->info[mode].fps[1]; i++) {
		if (vd_in->info[mode].ftype == VMGR_SAVE_FTYPE_AVI) {
			if (i == 0) {
				sprintf(tmp, "%s/%s.avi",
						vd_in->filename[mode],
						datetime);
			} else {
				sprintf(tmp, "%s/%s%d.avi",
						vd_in->filename[mode],
						datetime, i);
			}
		} else {
			if (i == 0) {
				sprintf(tmp, "%s/%s.jpg",
						vd_in->filename[mode],
						datetime);
			} else {
				sprintf(tmp, "%s/%s%d.jpg",
						vd_in->filename[mode],
						datetime, i);
			}
		}

		if (stat(tmp, &s) != 0) {
			break;
		}
	}

	strcpy(vd_in->filename[mode], tmp);
}

static void
save_file(void)
{
	int mode;
	unsigned long	now;
	struct statfs	fs;
	FILE *fp;
	struct mntent *entry;
	int i = 0;

	/* 現在の保存ステートのチェック */
	if (is_current_state(VMGR_STREAM_TIMER)) {
		mode = 1;
		VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "timer check\n");

		if (timer_check() == 0) {
			goto end;
		}
	} else if (is_current_state(VMGR_STREAM_SAVE)) {
		mode = 0;
	} else {
		goto end;
	}

	/* 保存周期のチェック */
	if (vd_in->info[mode].cycle != 0) {
		now = get_time();

		/* 保存時間に到達していない場合 */
		if (now < save_cycle) {
			/* 何もせずに終了 */
			goto end;
		}

		/* 次の保存時間に進める */
		save_cycle = now + vd_in->info[mode].cycle;
	}

	/* USB storageのマウント状態のチェック */
	fp = setmntent("/etc/mtab", "r");

	while ((entry = getmntent(fp)) != NULL) {
		if (strcmp(entry->mnt_dir, VMGR_SAVE_MEDIA) == 0) {
			i++;
			break;
		}
	}

	endmntent(fp);

	if (i == 0) {
		VMGR_LOG(LOG_WARNING, VMGR_LOCATION, "usb storage not mount\n");
		goto end;
	} 

	while (1) {
		/* USB-HDDのファイルシステム情報の取得 */
		if (statfs(VMGR_SAVE_MEDIA, &fs) < 0) {
			clear_state(VMGR_STREAM_SAVE);
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 	"failed to statfs : %s\n", strerror(errno));
			goto end;
		}

		/* 書込み可能サイズのチェック */
		if (fs.f_bavail < ((vd_in->newlen / fs.f_bsize) + 1)) {
			/* 停止モードであれば */
			if (vd_in->info[mode].mode == VMGR_SAVE_MODE_STOP) {
				VMGR_LOG(LOG_WARNING, VMGR_LOCATION,
						"usb storage disk full\n");
				/* 停止状態にする */
				clear_state(VMGR_STREAM_SAVE);
				goto end;

			/* サイクリックモードの場合 */
			} else {
				/* ファイルの削除 */
				delete_old_file();
			}
		} else {
			break;
		}
	}

	if (vd_in->avifile[mode] == NULL) {
		/* ファイル名の取得 */
		get_filename(mode);
	}

	/* ファイルへの格納 */
	if (vd_in->info[mode].ftype == VMGR_SAVE_FTYPE_AVI) {
		save_avifile(mode);
	} else {
		save_picture(mode);
		vd_in->framecount[mode]++;
	}
end:

	return;
}

static int
timer_check(void)
{
	time_t t;
	struct timespec ts;
	int ret = 0;
	time_t slp;

	
	if (vd_in->timer_enable == VMGR_TIMER_DISABLE) {
		clear_state(VMGR_STREAM_TIMER);
		return ret;
	}

	(void)clock_gettime(CLOCK_REALTIME, &ts);
	t = ts.tv_sec;

	if (is_current_state(VMGR_STREAM_TIMER) == 0) {
		/* x秒前からTimerモードにする	*/
		if (timer_scope_check(t + vd_in->timer_wakeup, &slp)) {
			set_state(VMGR_STREAM_TIMER);
			vd_in->framecount[1] = 0;
			VMGR_LOG(LOG_DEBUG, VMGR_LOCATION, "timer start\n");
		} else {
			if (current_state() == VMGR_STREAM_OFF) {
				vmgr_sleep(VMGR_FREE_IDLE_TIMER);
			}
		}
	} else {
		if ((ret = timer_scope_check(t, NULL)) == 0 &&
		    vd_in->framecount[1] != 0) {
			clear_state(VMGR_STREAM_TIMER);

			VMGR_LOG(LOG_DEBUG, VMGR_LOCATION,
						"timer finish\n");
		}
		if (vd_in->framecount[1] == 0 && ret) {
			if (vd_in->info[1].ftype == VMGR_SAVE_FTYPE_AVI) {
				vd_in->recordstart[1] = get_time();
			}
			VMGR_LOG(LOG_DEBUG, VMGR_LOCATION,
						"timer capture start\n");
		}
	}

	return ret;
}

static int
timer_scope_check(time_t t, time_t *slpsec)
{
	struct tm start, end, chgday;
	time_t start_t, end_t, chgday_t;
	int wdayhit = 0;
	int ret = 0;
	struct tm *tt;
	int i;

	tt = localtime(&t);

	if (slpsec != NULL) {
		*slpsec = 0;
	}

	for (i = 0; i < VMGR_SAVE_TIMER_ENTRY; i++) {
		if (vd_in->timer[i].dayofweek == VMGR_DAYOFWEEK_NONE) {
			continue;
		}

		/* 開始時間を総秒数に変換	*/
		start.tm_year = tt->tm_year;
		start.tm_mon  = tt->tm_mon;
		start.tm_mday = tt->tm_mday;
		start.tm_hour = vd_in->timer[i].start.tm_hour;
		start.tm_min  = vd_in->timer[i].start.tm_min;
		start.tm_sec  = vd_in->timer[i].start.tm_sec;
		start.tm_isdst = tt->tm_isdst;

		start_t = mktime(&start);

		/* 終了時間を総秒数に変換	*/
		end.tm_year = tt->tm_year;
		end.tm_mon  = tt->tm_mon;
		end.tm_mday = tt->tm_mday;
		end.tm_hour = vd_in->timer[i].end.tm_hour;
		end.tm_min  = vd_in->timer[i].end.tm_min;
		end.tm_sec  = vd_in->timer[i].end.tm_sec;
		end.tm_isdst = tt->tm_isdst;

		end_t = mktime(&end);

		/* 曜日チェック	*/
		wdayhit = (vd_in->timer[i].dayofweek ==
				VMGR_DAYOFWEEK_EVERYDAY) ? 1 : 0;

		/* 日跨りがある場合	*/
		if (start_t > end_t) {
			/* 既に開始時間	*/
			if (start_t < t) {
				/* 終了時間を1日進める	*/
				end.tm_year = tt->tm_year;
				end.tm_mon  = tt->tm_mon;
				end.tm_mday = tt->tm_mday + 1;
				end.tm_hour = vd_in->timer[i].end.tm_hour;
				end.tm_min  = vd_in->timer[i].end.tm_min;
				end.tm_sec  = vd_in->timer[i].end.tm_sec;
				end.tm_isdst = tt->tm_isdst;
	
				end_t = mktime(&end);

				/* 今日の曜日でチェック	*/
			    	if (vd_in->timer[i].dayofweek ==
							(tt->tm_wday + 2)) {
					wdayhit = 1;
				}

			/* 開始時間前	*/
			} else {
				/* 開始時間を前日にする	*/
				start.tm_year = tt->tm_year;
				start.tm_mon  = tt->tm_mon;
				start.tm_mday = tt->tm_mday - 1;
				start.tm_hour = vd_in->timer[i].start.tm_hour;
				start.tm_min  = vd_in->timer[i].start.tm_min;
				start.tm_sec  = vd_in->timer[i].start.tm_sec;
				start.tm_isdst = tt->tm_isdst;
	
				start_t = mktime(&start);

				/* 昨日の曜日でチェック	*/
			    	if (vd_in->timer[i].dayofweek ==
				    (tt->tm_wday == 6 ? 1 : tt->tm_wday + 1)) {
					wdayhit = 1;
				}
			}
		} else {
			/* 今日の曜日でチェック	*/
			if (vd_in->timer[i].dayofweek == (tt->tm_wday + 2)) {
				wdayhit = 1;
			}
		}

		if (slpsec != NULL) {
			/* まだ開始時間でない */
			if (start_t > t) {
				if (*slpsec == 0) {
					/* 残り時間の保存 */
					*slpsec = start_t - t;

				/* 残り時間の小さい方を保存 */
				} else if (*slpsec > (start_t - t)) {
					*slpsec = start_t - t;
				}
			}
			/* 終了時間を過ぎた */
			if (end_t < t) {
				/* 午前０時を総秒数に変換	*/
				chgday.tm_year = tt->tm_year;
				chgday.tm_mon  = tt->tm_mon;
				chgday.tm_mday = tt->tm_mday + 1;
				chgday.tm_hour = 0;
				chgday.tm_min  = 0;
				chgday.tm_sec  = 0;
				chgday.tm_isdst = tt->tm_isdst;

				chgday_t = mktime(&chgday);

				if (*slpsec == 0) {
					/* 残り時間の保存 */
					*slpsec = chgday_t - t;

				/* 残り時間の小さい方を保存 */
				} else if (*slpsec > (chgday_t - t)) {
					*slpsec = chgday_t - t;
				}
			}
		}

		/* 時間が範囲内であるか	*/
		if (t >= start_t && t <= end_t) {
			/* 曜日が一致	*/
			if (wdayhit) {
				ret = 1;
				if (slpsec == NULL) {
					break;
				}
			}
		}
	}

	return ret;
}

static void
delete_old_file(void)
{
	DIR *dir = NULL;
	struct dirent *dp = NULL;
	char dname[VMGR_PATH_LEN];
	char fname[VMGR_PATH_LEN];
	int fcount;
	int fremove;

	if (vd_in->media == VMGR_SAVE_MEDIA_PRIVATE) {
		snprintf(dname, VMGR_PATH_LEN, "%s", VMGR_SAVE_FOLDER_CLS);
	} else if (vd_in->media == VMGR_SAVE_MEDIA_PUBLIC) {
		snprintf(dname, VMGR_PATH_LEN, "%s", VMGR_SAVE_FOLDER_OPN);
	} else {
		goto end;
	}

	if ((dir = opendir(dname)) == NULL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"directory open error(%s).\n",
				strerror(errno));
		goto end;
	}

	for (dp = readdir(dir); dp != NULL; dp = readdir(dir)) {
		if(strcmp(dp->d_name, ".") == 0) {
			continue;
		} else if(strcmp(dp->d_name, "..") == 0) {
			continue;
		} else {
			if (dp->d_type == DT_DIR) {
				break;
			}
		}
	}

	closedir(dir);
	dir = NULL;

	if (dp == NULL) {
		goto end;
	}

	strlcat(dname, "/", VMGR_PATH_LEN);
	strlcat(dname, dp->d_name, VMGR_PATH_LEN);

	if ((dir = opendir(dname)) == NULL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"directory open error(%s).\n",
				strerror(errno));
		goto end;
	}

	fcount = 0;
	fremove = 0;
	for (dp = readdir(dir); dp != NULL; dp = readdir(dir)) {
		if(strcmp(dp->d_name, ".") == 0) {
			continue;
		} else if(strcmp(dp->d_name, "..") == 0) {
			continue;
		} else {
			if (dp->d_type == DT_REG) {
				if (fcount == 0) {
					snprintf(fname, VMGR_PATH_LEN, "%s/%s",
						dname, dp->d_name);
					remove(fname);
					fremove++;
				}
			}
			fcount++;
		}
	}

	closedir(dir);
	dir = NULL;

	if (fcount == fremove) {
		remove(dname);
	}

end:
	if (dir != NULL) {
		closedir(dir);
	}

	return;
}

static void
save_avifile(int mode)
{
	/* ファイル有り(書込み中) */
	if (vd_in->avifile[mode] != NULL) {
		/* 今回の書込みでMAXサイズより大きい場合 */
		if ((vd_in->avisize[mode] + vd_in->newlen + AVI_ONE_ENTRY) >= 
		    (vd_in->avisizemax * VMGR_MEGA_BYTE)) {
			/* ファイルを閉じる */
			close_avifile(mode);
			/* ファイル名の取得 */
			get_filename(mode);
		}
	}

	/* AVIファイル未登録 */
	if (vd_in->avifile[mode] == NULL) {
		vd_in->avifile[mode] = AVI_open_output_file(
					vd_in->filename[mode]);

		/* AVIヘッダとIndexテーブルヘッダ分を初期値とする */
		vd_in->avisize[mode] = AVI_HEADER + AVI_IDX_HEADER;

		if (vd_in->avifile[mode] == NULL) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION,
				"Error opening avifile"
				" %s\n", 
				vd_in->filename[mode]);
			goto end;
		}

	} 

	/* ファイル有り */
	if (vd_in->avifile[mode] != NULL) {
		AVI_write_frame(vd_in->avifile[mode], 
				vd_in->vbuf[vd_in->vbuf_index].mem,
				vd_in->newlen, vd_in->framecount[mode]);

		vd_in->framecount[mode]++;
		/* 画像データサイズ+TAG+Indexテーブル */
		vd_in->avisize[mode] += (vd_in->newlen + AVI_ONE_ENTRY);

		update_header_avifile(mode);

		/* AVIファイルの分割サイズを超えたらCloseする */
		if (vd_in->avisize[mode] >=
		    (vd_in->avisizemax * VMGR_MEGA_BYTE)) {
			close_avifile(mode);
		}

	}

end:
	return;
}

static void
update_header_avifile(int mode)
{
	float fps;
	char compressor[] = {"MJPG"};

	vd_in->recordtime[mode] = get_time() - vd_in->recordstart[mode];
	if (vd_in->avifile[mode] != NULL) {
		if (vd_in->recordtime[mode] > 0) {
			fps = ((float)vd_in->framecount[mode] /
					vd_in->recordtime[mode]);
		} else {
			fps = 1;
		}
		if (fps < 1) {
			fps = 1;
		}

		/* レート情報の書込み */
		AVI_set_video(vd_in->avifile[mode], vd_in->info[mode].width,
				vd_in->info[mode].height, fps, compressor);
	}
}

static void
close_avifile(int mode)
{
	if (vd_in->avifile[mode] != NULL) {
		AVI_close(vd_in->avifile[mode]);
		vd_in->avifile[mode] = NULL;
	}
}

static void
browser_timer_check(void)
{
	unsigned long current_time;

	if ((current_state() != VMGR_STREAM_ON) ||
		(vd_in->browser_timer == 0)) {
		return;
	}

	/* ブラウザアクセスがあったらbrowser_access_timeをリセット */
	if (browser_access_flg == 1) {
		browser_access_time = get_clock_time();
		browser_access_flg = 0;
		return;
	}

	current_time = get_clock_time();

	VMGR_LOG(LOG_DEBUG, VMGR_LOCATION,
		"time->%lu, browser_timer->%ld \n",
		current_time - browser_access_time, 
		vd_in->browser_timer);

	/* タイマ時間満了チェック */
	if (((current_time > browser_access_time) && 
	 	 (current_time - browser_access_time > 
		  vd_in->browser_timer * 60)) ||	
		((current_time < browser_access_time) &&
	  	(current_time + ULONG_MAX - browser_access_time >
		 vd_in->browser_timer * 60))) {
		/* 動画停止 */
		clear_state(VMGR_STREAM_ON);
		if (check_streaming() < 0) {
			VMGR_LOG(LOG_ERR, VMGR_LOCATION, "ERR");
		}
	}

	return;
}

static void usb_caminfo_settable_count(char *buf)
{
	int ret = -1;
	struct v4l2_fmtdesc fmt;
	int yuv_fmt_support = 0;

	if (!buf) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "buffer empty\n");
		return;
	}

	if (vd_in->fd < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		snprintf(buf, VMGR_MSG_BUFSIZE, "error");
		return;
	}

	if (VMGR_CHECK_NOT_SUPPORT(vd_in->cap.card)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "not support device\n");
		snprintf(buf, VMGR_MSG_BUFSIZE, "error");
		return;
	}

	memset(&fmt, 0, sizeof(fmt));
	fmt.index = 0;
	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

	while ((ret = ioctl(vd_in->fd, VIDIOC_ENUM_FMT, &fmt)) == 0) {
		if (fmt.pixelformat == V4L2_PIX_FMT_YUYV) {
			yuv_fmt_support = 1;
		}
		fmt.index++;
	}

	/* 暫定対応 */
	if (VMGR_CHECK_FORMAT_YUYV(vd_in->cap.card)) {
		if (yuv_fmt_support == 1) {
			fmt.index = 1;
		}
		else {
			fmt.index = 0;
		}
	}

	snprintf(buf, VMGR_MSG_BUFSIZE, "%d", fmt.index);
}

static void usb_caminfo_settable(char *buf)
{
	int ret = -1;
	char tmp[524];
	struct v4l2_fmtdesc fmt;
	struct v4l2_frmsizeenum fsize;

	if (!buf) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "buffer empty\n");
		return;
	}

	if (vd_in->fd < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		snprintf(buf, VMGR_MSG_BUFSIZE, "error");
		return;
	}

	if (VMGR_CHECK_NOT_SUPPORT(vd_in->cap.card)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "not support device\n");
		snprintf(buf, VMGR_MSG_BUFSIZE, "error");
		return;
	}

	memset(tmp, 0, sizeof(tmp));
	memset(&fmt, 0, sizeof(fmt));
	fmt.index = 0;
	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

	/* Format type */
	while ((ret = ioctl(vd_in->fd, VIDIOC_ENUM_FMT, &fmt)) == 0) {
		/* 暫定対応 */
		if (VMGR_CHECK_FORMAT_YUYV(vd_in->cap.card)) {
			if (fmt.pixelformat != V4L2_PIX_FMT_YUYV) {
				fmt.index++;
				continue;
			}
		}
		else if (fmt.index != 0) {
			strlcat(buf, ";", VMGR_MSG_BUFSIZE);
		}

		GET_PIXFMT_CDM(tmp, fmt);
		strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

		memset(&fsize, 0, sizeof(fsize));
		fsize.index = 0;
		fsize.pixel_format = fmt.pixelformat;

		/* Size */
		while ((ret = ioctl(vd_in->fd, VIDIOC_ENUM_FRAMESIZES,
							&fsize)) == 0) {
			if (fsize.index != 0) {
				strlcat(buf, ",", VMGR_MSG_BUFSIZE);
			}
			if (fsize.type == V4L2_FRMIVAL_TYPE_DISCRETE) {
				sprintf(tmp, "%ux%u",
						fsize.discrete.width,
						fsize.discrete.height);
				strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
			}
			fsize.index++;
		}
		fmt.index++;
	}

	if (errno != EINVAL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "ERROR enumerating frame formats\n");
	}
}

static void usb_caminfo_current_setting(char *buf)
{
	struct v4l2_format vf;
	char tmp[64];

	if (vd_in->fd < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "no open device\n");
		if (buf != NULL) {
			snprintf(buf, VMGR_MSG_BUFSIZE, "error");
		}
		return;
	}

	/* Current setting */
	memset(tmp, 0, sizeof(tmp));
	memset(&vf, 0, sizeof(vf));
	vf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl(vd_in->fd, VIDIOC_G_FMT, &vf) < 0) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION,
			 "ioctl(VIDIOC_G_FMT) fail(%d)\n", errno);
		return;
	}

	/* Format type */
	GET_PIXFMT_CDM(tmp, vf.fmt.pix);
	if (buf != NULL)
		strlcat(buf, tmp, VMGR_MSG_BUFSIZE);

	/* Size */
	sprintf(tmp, ":%ux%u", vf.fmt.pix.width, vf.fmt.pix.height);
	if (buf != NULL)
		strlcat(buf, tmp, VMGR_MSG_BUFSIZE);
}
