/*
 * HITUJI USB power driver
 *
 * Copyright (C) 2014 NEC Platforms, Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307USA
 */

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/gpio.h>
#include <asm/atomic.h>
#include <linux/init.h>
#include <linux/genhd.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/htusb.h>

#define DRVNAME		"htusb"
#define DEVNAME		"htusb"

static int dev_major;
static struct class *htusb_class;

#define HTUSB_TICK 5  /* x 10msec */

static struct timer_list htusb_timer;
static int timerstop = 0;

static void htusb_timer_handler( unsigned long data );

struct htusb_port {
	struct htusb_gpio_port port;
	int stat;
#define HTUSB_STAT_ON 0
#define HTUSB_STAT_OFF 1
#define HTUSB_STAT_OVER 2
	int count;
};

struct htusb_port *htusb_ports;
static int htusb_portnum = 0;
static struct proc_dir_entry *p_entry = NULL;

#ifndef CONFIG_HTUSB_COUNT
#define CONFIG_HTUSB_COUNT 2
#endif

//#define HTUSB_DEBUG

#ifdef HTUSB_DEBUG
#define DEBUG_OUT(x) printk x
#else
#define DEBUG_OUT(x)
#endif

static void
htusb_timer_register(void *pdata)
{
	init_timer(&htusb_timer);

	htusb_timer.expires  = jiffies + HTUSB_TICK;
	htusb_timer.data     = (unsigned long)pdata;
	htusb_timer.function = htusb_timer_handler;

	add_timer(&htusb_timer);
}


static void
htusb_timer_handler(unsigned long data)
{
	int i;

	if (timerstop)
		return;

	for (i = 0; i < htusb_portnum; i++) {
		struct htusb_port *p = &htusb_ports[i];

		if (p->stat != HTUSB_STAT_ON)
			continue;
		if (gpio_get_value(p->port.overcurrent) == p->port.ovc_det) {
			if (p->count++ >= CONFIG_HTUSB_COUNT) {
				p->stat = HTUSB_STAT_OVER;  /* over-current */
				p->count = 0;
				gpio_set_value(p->port.power, (p->port.pow_act) ? 0 : 1);
			}
		} else {
			p->count = 0;
		}
	}

	htusb_timer_register((void *)data);
}


static long
htusb_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int retval = 0;

	switch (cmd) {
	default:
		retval = -EINVAL;
		break;
	}

	return retval;
}


static int
htusb_open(struct inode *inode, struct file *file)
{
	unsigned int dev_minor = MINOR(inode->i_rdev);

	if (dev_minor != 0) {
		printk(KERN_ERR DRVNAME ": trying to access unknown minor device -> %d\n",
		       dev_minor);
		return -ENODEV;
	}

	return 0;
}

static int
htusb_close(struct inode * inode, struct file * file)
{
	return 0;
}

struct file_operations htusb_fops = {
	.open = htusb_open,
	.release = htusb_close,
	.unlocked_ioctl = htusb_ioctl,
};


static int
setup_power(struct htusb_gpio_port *p)
{
	int ret;
	int gpio = p->power;

	DEBUG_OUT(("%s: %s port=%d, gpio=%d\n", DRVNAME, __func__, p->usbport, gpio));
	if (!gpio_is_valid(gpio)) {
		printk(KERN_INFO "Skipping unavailable gpio %d\n", gpio);
		return 0;
	}

	ret = gpio_request(gpio, DRVNAME);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_request() error (%d) gpio=%d\n",
		       __func__, ret, gpio);
		return ret;
	}

	ret = gpio_direction_output(gpio, p->pow_act);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_direction_input() error (%d) gpio=%d\n",
		       __func__, ret, gpio);
		gpio_free(gpio);
		return ret;
	}

	return 0;
}

static int
setup_ovc(struct htusb_gpio_port *p)
{
	int ret;
	int gpio = p->overcurrent;

	DEBUG_OUT(("%s: %s port=%d, gpio=%d\n", DRVNAME, __func__, p->usbport, gpio));
	if (!gpio_is_valid(gpio)) {
		printk(KERN_INFO "Skipping unavailable gpio %d\n", gpio);
		return 0;
	}

	ret = gpio_request(gpio, DRVNAME);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_request() error (%d) gpio=%d\n",
		       __func__, ret, gpio);
		return ret;
	}

	ret = gpio_direction_input(gpio);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_direction_input() error (%d) gpio=%d\n",
		       __func__, ret, gpio);
		gpio_free(gpio);
		return ret;
	}

	return 0;
}


static int
htusb_read_proc(char *page, char **start, off_t off,
		int count, int *eof, void *data)
{
	int len;
	struct htusb_port *p = (struct htusb_port *)data;
	char *retstr;

	switch (p->stat) {
	case HTUSB_STAT_ON:
		retstr = "ENABLE\n";
		break;
	case HTUSB_STAT_OFF:
		retstr = "DISABLE\n";
		break;
	case HTUSB_STAT_OVER:
		retstr = "OVERCURRENT\n";
		break;
	default:
		retstr = "UNKNOWN\n";
		break;
	}

	snprintf(page, PAGE_SIZE - 1, "%s", retstr);
	len = strlen(retstr);
	*eof = 1;

	return len;
}


static int
htusb_write_proc(struct file *file, const char __user *buffer,
		 unsigned long count, void *data)
{
	int len;
	char buf[16];
	int stat;
	struct htusb_port *p = (struct htusb_port *)data;

	len = (count >= sizeof(buf) - 1) ? sizeof(buf) - 1 : count;

	if (copy_from_user(buf, buffer, len))
		return -EFAULT;
	buf[len] = '\0';

	stat = simple_strtoul(buf, NULL, 10);

	switch (stat) {
	case HTUSB_STAT_ON:
		if (p->stat != HTUSB_STAT_OVER) {
			gpio_set_value(p->port.power, p->port.pow_act);
			p->stat = stat;
		}
		break;
	case HTUSB_STAT_OFF:
		if (p->stat != HTUSB_STAT_OVER) {
			gpio_set_value(p->port.power, (p->port.pow_act) ? 0 : 1);
			p->stat = stat;
		}
		break;
	case HTUSB_STAT_OVER:
		if (p->stat == HTUSB_STAT_OVER) {
			gpio_set_value(p->port.power, p->port.pow_act);
			p->stat = HTUSB_STAT_ON;
		}
		break;
	default:
		return -EINVAL;
	}

	return len;
}


static int
htusb_probe(struct platform_device *dev)
{
	int result = 0;
	struct htusb_gpio_platform_data *pdata;
	int i;
	struct proc_dir_entry *entry;
	char dev_name[16];

	dev_major = register_chrdev(0, DEVNAME, &htusb_fops);
	if (!dev_major) {
		printk(KERN_ERR DRVNAME ": Error whilst opening %s \n", DEVNAME);
		result = -ENODEV;
		goto out;
	}
	htusb_class = class_create(THIS_MODULE, DRVNAME);
	device_create(htusb_class, NULL, MKDEV(dev_major, 0), dev, DEVNAME);
	printk(KERN_INFO DRVNAME ": device registered with major %d\n", dev_major);

	pdata = dev->dev.platform_data;
	if (!pdata->portnum) {
		result = -EINVAL;
		goto out;
	}

	htusb_ports = kmalloc(sizeof(struct htusb_port) * pdata->portnum, GFP_KERNEL);
	if (htusb_ports == NULL) {
		printk(KERN_ERR DRVNAME ": Error at kmalloc %s \n", DEVNAME);
		result = -ENOMEM;
		goto out;
	}

	p_entry = proc_mkdir(DRVNAME, NULL);
	if (p_entry == NULL)
		printk("%s: create_proc_read_entry() fail %s\n", __func__, DRVNAME);

	htusb_portnum = pdata->portnum;
	for (i = 0; i < pdata->portnum; i++) {
		struct htusb_port *p;

		p = &htusb_ports[i];
		p->port = *(&pdata->port_tbl[i]);
		setup_power(&p->port);
		setup_ovc(&p->port);
		p->stat = HTUSB_STAT_ON;

		snprintf(dev_name, sizeof(dev_name)-1, "port%d", p->port.usbport);
		entry = create_proc_entry(dev_name, S_IRUGO | S_IWUGO, p_entry);
		if (entry == NULL) {
			printk("%s: create_proc_read_entry() fail %s\n", __func__, dev_name);
		} else {
			printk("%s: create /proc/%s/%s\n", DRVNAME, DRVNAME, dev_name);
			entry->data = p;
			entry->read_proc = htusb_read_proc;
			entry->write_proc = htusb_write_proc;
		}
	}

	timerstop = 0;
	htusb_timer_register(pdata);
out:
	return result;
}

static int
htusb_remove(struct platform_device *dev)
{
	struct htusb_gpio_platform_data *pdata;
	char dev_name[16];
	int i;

	timerstop = 1;
	del_timer_sync(&htusb_timer);

	pdata = dev->dev.platform_data;
	for (i = 0; i < pdata->portnum; i++) {
		struct htusb_port *p;

		p = &htusb_ports[i];
		snprintf(dev_name, sizeof(dev_name)-1, "port%d", p->port.usbport);
		remove_proc_entry(dev_name, p_entry);
		gpio_free(p->port.power);
		gpio_free(p->port.overcurrent);
	}
	remove_proc_entry(DRVNAME, NULL);
	device_destroy(htusb_class, MKDEV(dev_major, 0));
	class_destroy(htusb_class);
	unregister_chrdev(dev_major, DEVNAME);

	return 0;
}

static struct
platform_driver htusb_driver = {
	.probe = htusb_probe,
	.remove = htusb_remove,
	.driver = {
		.name = "htusb-gpio",
		.owner = THIS_MODULE,
	},
};

static int __init
htusb_mod_init(void)
{
	int ret = platform_driver_register(&htusb_driver);
	if (ret)
		printk(KERN_INFO DRVNAME ": Error registering platfom driver!\n");

	return ret;
}

static void __exit
htusb_mod_exit(void)
{
	platform_driver_unregister(&htusb_driver);
}

module_init (htusb_mod_init);
module_exit (htusb_mod_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC Platforms, Ltd.");
MODULE_DESCRIPTION("Character device for HITUJI LED w/ gpio");
