#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/export.h>
#include <linux/ctype.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/mtd/mtd.h>
#include <linux/macutil.h>

static int backup_read(void *, const loff_t, int);
static int backup_fetch(void);

typedef struct {
	mac_addr emac;
	mac_addr umac;
	mac_addr wmac;
	mac_addr wmac2;
} __PACKED__ mac_info_t;

typedef struct {
	uint8_t id;
} __PACKED__ configid_info_t;

typedef struct {
	char id[ADDITIONAL_INFO_LENGTH];
} __PACKED__ additional_info_t;

typedef struct {
	uint16_t key;
} __PACKED__ randomkey_info_t;

typedef struct {
	char num[SERIAL_NUM_LENGTH];
} __PACKED__ serial_info_t;

typedef struct {
	char num[PRESET_ID_LENGTH];
} __PACKED__ preset_info_t;

static mac_info_t MacInfo;
static int MacInfoRead;
static configid_info_t ConfigidInfo;
static additional_info_t AdditionalInfo;
static randomkey_info_t RandomkeyInfo;
static serial_info_t SerialInfo;
static preset_info_t PresetInfo;

#ifdef USE_NECPF_FACTORY_2ND
typedef struct {
	char id[SSID_AREA_SIZE];
	uint32_t magic;
	uint32_t magic_n;
} __PACKED__ ssid_info_t;

static ssid_info_t SSIDInfo[SSID_NUM];

typedef struct {
	char key[DEFKEY_AREA_SIZE];
	uint32_t magic;
	uint32_t magic_n;
} __PACKED__ defkey_info_t;

static defkey_info_t DefkeyInfo[DEFKEY_NUM];

typedef struct {
	char pw[WEBPW_AREA_SIZE];
	uint32_t magic;
	uint32_t magic_n;
} __PACKED__ webpw_info_t;

static webpw_info_t WebpwInfo;

typedef struct {
	char pin[WPSPIN_AREA_SIZE];
	uint32_t magic;
	uint32_t magic_n;
} __PACKED__ wpspin_info_t;

static wpspin_info_t WpspinInfo;
#endif	/* USE_NECPF_FACTORY_2ND */

static uint8_t config_backup[CONFIG_FLASH_SIZE];
static int config_fetched = 0;
struct mtd_info *macutil_mtd;

void
macutil_init(struct mtd_info *mtd)
{
	macutil_mtd = mtd;
}

static int
backup_fetch(void)
{
	size_t length;

	if (config_fetched)
		return 0;

	mtd_read(macutil_mtd, CONFIG_MACUTIL_OFFSET,
		 sizeof(config_backup), &length, config_backup);
	config_fetched = 1;
	return 0;
}

static int
backup_read(void *buf, const loff_t flash, int len)
{
	unsigned offset;

	backup_fetch();
	offset =  flash - CONFIG_MACUTIL_OFFSET;
	if (offset >= CONFIG_FLASH_SIZE)
		return -1;
	if ((offset + len) > CONFIG_FLASH_SIZE)
		return -1;
	memcpy(buf, &config_backup[offset], len);
	return 0;
}

int
get_mac_addr(int type, mac_addr *mac)
{
	static const mac_addr local_mac =
		{ {0x02, 0x00, 0x00, 0x00, 0x00, 0x00} };
	static const mac_addr none_mac =
		{ {0xff, 0xff, 0xff, 0xff, 0xff, 0xff} };
	const mac_addr *src = NULL;

	if (!MacInfoRead)
		backup_read(&MacInfo, CONFIG_MAC_INFO_ADDR, sizeof(MacInfo));
	MacInfoRead = 1;
	switch (type) {
	case MAC_ADDR_ETHER:    /* ethernet */
	case MAC_ADDR_ATM:      /* atm */
		src = &MacInfo.emac;
		break;
	case MAC_ADDR_WAN:      /* EoU-PC */
	case MAC_ADDR_USB:      /* usb */
		src = &MacInfo.umac;
		break;
	case MAC_ADDR_WLAN:     /* wlan */
		src = &MacInfo.wmac;
		break;
	case MAC_ADDR_WLAN_2ND: /* wlan 2nd */
		src = &MacInfo.wmac2;
		break;
	}
	if (src == NULL) {
		return 1;       /* FAIL */
	}
	if (compare_mac_addr(src, &none_mac) == 0) {    /* 設定されていない */
		src = &local_mac;
	}
	*mac = *src;
	return 0;               /* ok */
}

/***
 * MACアドレスが記述された文字列から mac に変換する
 * サポート形式: "aabbccddeeff" "aa:bb:cc:dd:ee:ff" "aa-bb-cc-dd-ee-ff"
 * 空白は読み飛ばす
 * parseした文字数を返す
 ***/
int
parse_mac(mac_addr *mac, const char *buf)
{
	int value = 0;
	int column = 0;
	int mac_pos = 0;
	int n = 0;

	while ((n < sizeof(*mac)) && (*buf != '\0')) {
		if (!isxdigit(*buf)) {
			if (isspace(*buf) || (*buf == ':')
			    || (*buf == '-')) {
				buf++;
				column++;
				continue;
			}
			break;  /* サポート文字以外 */
		}
		value <<= 4;
		value +=
			isdigit(*buf) ? *buf - '0' : toupper(*buf) - 'A' +
			0x0a;
		if ((mac_pos % 2) == 1) {
			mac->mac_address[n++] = value;
			value = 0;
		}
		column++;
		mac_pos++;
		buf++;
	}
	if (n != sizeof(*mac))
		return 0;
	return column;
}

/*
 * config id の読み込み
 */
int
get_config_id(void)
{
	backup_read(
		&ConfigidInfo, CONFIG_CONFIG_ID_ADDR, sizeof(ConfigidInfo));
	return ConfigidInfo.id;
}

const char *
get_additional_info(void)
{
	additional_info_t info;

	backup_read(
		&info, CONFIG_ADDITIONAL_INFO_ADDR, sizeof(info));
	if ((u_int8_t)(info.id[0]) == 0xff) {
		memset(AdditionalInfo.id, 0xff, sizeof(AdditionalInfo.id));
		AdditionalInfo.id[0] = '\0';
	} else {
		int i;

		memset(AdditionalInfo.id, 0xff, sizeof(AdditionalInfo.id));
		/* 表示可能な文字だけcopy */
		for (i = 0; i < (sizeof(AdditionalInfo.id) - 1); i++) {
			if (!isprint(info.id[i])) {
				break;
			}
			AdditionalInfo.id[i] = info.id[i];
		}
		AdditionalInfo.id[i] = '\0';
	}
	return AdditionalInfo.id;
}

/*
 * random key の読み込み
 */
int
get_random_key(void)
{
	backup_read(
		&RandomkeyInfo, CONFIG_RANDOM_KEY_ADDR, sizeof(RandomkeyInfo));
	return RandomkeyInfo.key;
}

/*
 * serial の読み込み
 */
const char *
get_serial_num(void)
{
	backup_read(
		&SerialInfo, CONFIG_SERIAL_NUM_ADDR, sizeof(SerialInfo));
	if ((unsigned char) SerialInfo.num[0] == 0xff) {
		SerialInfo.num[0] = '\0';
	}
	return SerialInfo.num;
}

/*
 * presetid の読み込み
 */
const char *
get_preset_id(void)
{
	backup_read(
		&PresetInfo, CONFIG_PRESET_ID_ADDR, sizeof(PresetInfo));
	if ((unsigned char)PresetInfo.num[0] == 0xff)
		PresetInfo.num[0] = '\0';

	return PresetInfo.num;
}

#ifdef USE_NECPF_FACTORY_2ND
/*
 * ssid の読み込み
 */
const char *
get_ssid(int idx)
{
	if (idx >= SSID_NUM)
		return '\0';

	backup_read(&SSIDInfo[idx],
		    (loff_t) (CONFIG_SSID1_ADDR + (SSID_AREA_SIZE + MAGIC_LENGTH) * idx),
		    sizeof(SSIDInfo[idx]));
	if (SSIDInfo[idx].magic != ~(SSIDInfo[idx].magic_n)
	    || SSIDInfo[idx].magic != FACTORY_2ND_MAGIC) {
		SSIDInfo[idx].id[0] = '\0';
	}
	return SSIDInfo[idx].id;
}

/*
 * defkey の読み込み
 */
const char *
get_defkey(int idx)
{
	if (idx >= DEFKEY_NUM)
		return '\0';

	backup_read(&DefkeyInfo[idx],
		    (loff_t) (CONFIG_DEFKEY1_ADDR + (DEFKEY_AREA_SIZE + MAGIC_LENGTH) * idx),
		    sizeof(DefkeyInfo[idx]));
	if (DefkeyInfo[idx].magic != ~(DefkeyInfo[idx].magic_n)
	    || DefkeyInfo[idx].magic != FACTORY_2ND_MAGIC) {
		DefkeyInfo[idx].key[0] = '\0';
	}
	return DefkeyInfo[idx].key;
}

/*
 * webpw の読み込み
 */
const char *
get_webpw(void)
{
	backup_read(
		&WebpwInfo, CONFIG_WEBPW_ADDR, sizeof(WebpwInfo));
	if (WebpwInfo.magic != ~(WebpwInfo.magic_n)
	    || WebpwInfo.magic != FACTORY_2ND_MAGIC) {
		WebpwInfo.pw[0] = '\0';
	}
	return WebpwInfo.pw;
}

/*
 * wpspin の読み込み
 */
const char *
get_wpspin(void)
{
	backup_read(
		&WpspinInfo, CONFIG_WPSPIN_ADDR, sizeof(WpspinInfo));
	if (WpspinInfo.magic != ~(WpspinInfo.magic_n)
	    || WpspinInfo.magic != FACTORY_2ND_MAGIC) {
		WpspinInfo.pin[0] = '\0';
	}
	return WpspinInfo.pin;
}
#endif	/* USE_NECPF_FACTORY_2ND */

EXPORT_SYMBOL(macutil_init);
EXPORT_SYMBOL(get_mac_addr);
EXPORT_SYMBOL(parse_mac);
EXPORT_SYMBOL(get_config_id);
EXPORT_SYMBOL(get_additional_info);
EXPORT_SYMBOL(get_random_key);
EXPORT_SYMBOL(get_serial_num);
EXPORT_SYMBOL(get_preset_id);
#ifdef USE_NECPF_FACTORY_2ND
EXPORT_SYMBOL(get_ssid);
EXPORT_SYMBOL(get_defkey);
EXPORT_SYMBOL(get_webpw);
EXPORT_SYMBOL(get_wpspin);
#endif	/* USE_NECPF_FACTORY_2ND */

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC Platforms, Ltd.");
MODULE_DESCRIPTION("Macutil");
