/*
 * Bridge multicast support.
 *
 * Copyright (c) 2010 Herbert Xu <herbert@gondor.apana.org.au>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 */

#include <linux/err.h>
#include <linux/if_ether.h>
#include <linux/igmp.h>
#include <linux/jhash.h>
#include <linux/kernel.h>
#include <linux/log2.h>
#include <linux/netdevice.h>
#include <linux/netfilter_bridge.h>
#include <linux/random.h>
#include <linux/rculist.h>
#include <linux/skbuff.h>
#include <linux/slab.h>
#include <linux/timer.h>
#include <net/ip.h>
#if IS_ENABLED(CONFIG_IPV6)
#include <net/ipv6.h>
#include <net/mld.h>
#include <net/addrconf.h>
#include <net/ip6_checksum.h>
#endif

#include "br_private.h"

#define mlock_dereference(X, br) \
	rcu_dereference_protected(X, lockdep_is_held(&br->multicast_lock))

#if IS_ENABLED(CONFIG_IPV6)
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_ip6_multicast_leave_group(struct net_bridge *br,
					 struct net_bridge_port *port,
					 const struct in6_addr *group,
					 __be16 nsrcs,
					 struct in6_addr *src,
					 unsigned char *haddr);
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static inline int ipv6_is_transient_multicast(const struct in6_addr *addr)
{
	if (ipv6_addr_is_multicast(addr) && IPV6_ADDR_MC_FLAG_TRANSIENT(addr))
		return 1;
	return 0;
}
#endif

static inline int br_ip_equal(const struct br_ip *a, const struct br_ip *b)
{
	if (a->proto != b->proto)
		return 0;
	switch (a->proto) {
	case htons(ETH_P_IP):
		return a->u.ip4 == b->u.ip4;
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
		return ipv6_addr_equal(&a->u.ip6, &b->u.ip6);
#endif
	}
	return 0;
}

static inline int __br_ip4_hash(struct net_bridge_mdb_htable *mdb, __be32 ip)
{
	return jhash_1word(mdb->secret, (__force u32)ip) & (mdb->max - 1);
}

#if IS_ENABLED(CONFIG_IPV6)
static inline int __br_ip6_hash(struct net_bridge_mdb_htable *mdb,
				const struct in6_addr *ip)
{
	return jhash2((__force u32 *)ip->s6_addr32, 4, mdb->secret) & (mdb->max - 1);
}
#endif

static inline int br_ip_hash(struct net_bridge_mdb_htable *mdb,
			     struct br_ip *ip)
{
	switch (ip->proto) {
	case htons(ETH_P_IP):
		return __br_ip4_hash(mdb, ip->u.ip4);
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
		return __br_ip6_hash(mdb, &ip->u.ip6);
#endif
	}
	return 0;
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static inline int br_multicast_disabled(struct net_bridge *br)
{
	if (br->multicast_disabled && !br->multicast_mld_only)
		return 1;
	return 0;
}

static inline int br_multicast_mld_only(struct net_bridge *br)
{
	if (br->multicast_disabled && br->multicast_mld_only)
		return 1;
	return 0;
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static struct net_bridge_mdb_entry *__br_mdb_ip_get(
	struct net_bridge_mdb_htable *mdb, struct br_ip *dst, int hash)
{
	struct net_bridge_mdb_entry *mp;
	struct hlist_node *p;

	hlist_for_each_entry_rcu(mp, p, &mdb->mhash[hash], hlist[mdb->ver]) {
		if (br_ip_equal(&mp->addr, dst))
			return mp;
	}

	return NULL;
}

static struct net_bridge_mdb_entry *br_mdb_ip_get(
	struct net_bridge_mdb_htable *mdb, struct br_ip *dst)
{
	if (!mdb)
		return NULL;

	return __br_mdb_ip_get(mdb, dst, br_ip_hash(mdb, dst));
}

static struct net_bridge_mdb_entry *br_mdb_ip4_get(
	struct net_bridge_mdb_htable *mdb, __be32 dst)
{
	struct br_ip br_dst;

	br_dst.u.ip4 = dst;
	br_dst.proto = htons(ETH_P_IP);

	return br_mdb_ip_get(mdb, &br_dst);
}

#if IS_ENABLED(CONFIG_IPV6)
static struct net_bridge_mdb_entry *br_mdb_ip6_get(
	struct net_bridge_mdb_htable *mdb, const struct in6_addr *dst)
{
	struct br_ip br_dst;

	br_dst.u.ip6 = *dst;
	br_dst.proto = htons(ETH_P_IPV6);

	return br_mdb_ip_get(mdb, &br_dst);
}
#endif

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
struct net_bridge_mdb_entry *br_mdb_get(struct net_bridge *br,
					struct sk_buff *skb)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
struct net_bridge_mdb_entry *br_mdb_get(struct net_bridge *br,
					struct net_bridge_port *port,
					struct sk_buff *skb)
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
{
	struct net_bridge_mdb_htable *mdb = rcu_dereference(br->mdb);
	struct br_ip ip;

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (br->multicast_disabled)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	if (br_multicast_disabled(br))
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		return NULL;

	if (BR_INPUT_SKB_CB(skb)->igmp)
		return NULL;

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (br->multicast_upstream) {
		if (port != NULL && !port->multicast_upstream)
			return NULL;
	}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	ip.proto = skb->protocol;

	switch (skb->protocol) {
	case htons(ETH_P_IP):
		ip.u.ip4 = ip_hdr(skb)->daddr;
		break;
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
		ip.u.ip6 = ipv6_hdr(skb)->daddr;
		break;
#endif
	default:
		return NULL;
	}

	return br_mdb_ip_get(mdb, &ip);
}

static void br_mdb_free(struct rcu_head *head)
{
	struct net_bridge_mdb_htable *mdb =
		container_of(head, struct net_bridge_mdb_htable, rcu);
	struct net_bridge_mdb_htable *old = mdb->old;

	mdb->old = NULL;
	kfree(old->mhash);
	kfree(old);
}

static int br_mdb_copy(struct net_bridge_mdb_htable *new,
		       struct net_bridge_mdb_htable *old,
		       int elasticity)
{
	struct net_bridge_mdb_entry *mp;
	struct hlist_node *p;
	int maxlen;
	int len;
	int i;

	for (i = 0; i < old->max; i++)
		hlist_for_each_entry(mp, p, &old->mhash[i], hlist[old->ver])
			hlist_add_head(&mp->hlist[new->ver],
				       &new->mhash[br_ip_hash(new, &mp->addr)]);

	if (!elasticity)
		return 0;

	maxlen = 0;
	for (i = 0; i < new->max; i++) {
		len = 0;
		hlist_for_each_entry(mp, p, &new->mhash[i], hlist[new->ver])
			len++;
		if (len > maxlen)
			maxlen = len;
	}

	return maxlen > elasticity ? -EINVAL : 0;
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_multicast_free_hostent(struct rcu_head *head)
{
	struct net_bridge_port_hostent *h =
		container_of(head, struct net_bridge_port_hostent, rcu);

	kfree(h);
}

static void br_multicast_free_srcent(struct rcu_head *head)
{
	struct net_bridge_port_srcent *s =
		container_of(head, struct net_bridge_port_srcent, rcu);

	kfree(s);
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static void br_multicast_free_pg(struct rcu_head *head)
{
	struct net_bridge_port_group *p =
		container_of(head, struct net_bridge_port_group, rcu);

	kfree(p);
}

static void br_multicast_free_group(struct rcu_head *head)
{
	struct net_bridge_mdb_entry *mp =
		container_of(head, struct net_bridge_mdb_entry, rcu);

	kfree(mp);
}

static void br_multicast_group_expired(unsigned long data)
{
	struct net_bridge_mdb_entry *mp = (void *)data;
	struct net_bridge *br = mp->br;
	struct net_bridge_mdb_htable *mdb;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) || timer_pending(&mp->timer))
		goto out;

	mp->mglist = false;

	if (mp->ports)
		goto out;

	mdb = mlock_dereference(br->mdb, br);

	hlist_del_rcu(&mp->hlist[mdb->ver]);
	mdb->size--;

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	call_rcu_bh(&mp->rcu, br_multicast_free_group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	call_rcu(&mp->rcu, br_multicast_free_group);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */

out:
	spin_unlock(&br->multicast_lock);
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_multicast_del_hostent(struct net_bridge *br,
				     struct net_bridge_port_srcent *ps,
				     struct net_bridge_port_hostent *ph)
{
	struct net_bridge_port_hostent *h;
	struct net_bridge_port_hostent __rcu **hh;

	for (hh = &ps->hlist;
	     (h = mlock_dereference(*hh, br)) != NULL;
	     hh = &h->next) {
		if (ph != NULL && h != ph)
			continue;

		rcu_assign_pointer(*hh, h->next);
		del_timer(&h->timer);
		call_rcu(&h->rcu, br_multicast_free_hostent);
	}

	return;
}

static void br_multicast_del_srcent(struct net_bridge *br,
				    struct net_bridge_port_group *pg,
				    struct net_bridge_port_srcent *ps)
{
	struct net_bridge_port_srcent *s;
	struct net_bridge_port_srcent __rcu **ss;

	for (ss = &pg->inlist;
	     (s = mlock_dereference(*ss, br)) != NULL;
	     ss = &s->next) {
		if (ps != NULL && s != ps)
			continue;

		br_multicast_del_hostent(br, s, NULL);

		rcu_assign_pointer(*ss, s->next);
		call_rcu(&s->rcu, br_multicast_free_srcent);
	}

	return;
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static void br_multicast_del_pg(struct net_bridge *br,
				struct net_bridge_port_group *pg)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	struct net_bridge_port_group __rcu **pp;

	mdb = mlock_dereference(br->mdb, br);

	mp = br_mdb_ip_get(mdb, &pg->addr);
	if (WARN_ON(!mp))
		return;

	for (pp = &mp->ports;
	     (p = mlock_dereference(*pp, br)) != NULL;
	     pp = &p->next) {
		if (p != pg)
			continue;

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
		br_multicast_del_srcent(br, pg, NULL);
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

		rcu_assign_pointer(*pp, p->next);
		hlist_del_init(&p->mglist);
		del_timer(&p->timer);
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		call_rcu_bh(&p->rcu, br_multicast_free_pg);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		call_rcu(&p->rcu, br_multicast_free_pg);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */

		if (!mp->ports && !mp->mglist &&
		    netif_running(br->dev))
			mod_timer(&mp->timer, jiffies);

		return;
	}

	WARN_ON(1);
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_multicast_port_group_hostent_expired(unsigned long data)
{
	struct net_bridge_port_hostent *ph = (void *)data;
	struct net_bridge_port_hostent *h;
	struct net_bridge_port_srcent *ps = ph->srcent;
	struct net_bridge_port_srcent *s;
	struct net_bridge_port_group *pg = ps->ports;
	struct net_bridge *br = pg->port->br;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) || timer_pending(&ph->timer))
		goto out;

	br_multicast_del_hostent(br, ps, ph);
	if ((h = mlock_dereference(ps->hlist, br)) == NULL)
		br_multicast_del_srcent(br, pg, ps);
	if ((s = mlock_dereference(pg->inlist, br)) == NULL)
		br_multicast_del_pg(br, pg);

out:
	spin_unlock(&br->multicast_lock);
}
#else  /* USE_NECPF_BRIDGE_SSIS_SSM2S */
static void br_multicast_port_group_expired(unsigned long data)
{
	struct net_bridge_port_group *pg = (void *)data;
	struct net_bridge *br = pg->port->br;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) || timer_pending(&pg->timer) ||
	    hlist_unhashed(&pg->mglist))
		goto out;

	br_multicast_del_pg(br, pg);

out:
	spin_unlock(&br->multicast_lock);
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static int br_mdb_rehash(struct net_bridge_mdb_htable __rcu **mdbp, int max,
			 int elasticity)
{
	struct net_bridge_mdb_htable *old = rcu_dereference_protected(*mdbp, 1);
	struct net_bridge_mdb_htable *mdb;
	int err;

	mdb = kmalloc(sizeof(*mdb), GFP_ATOMIC);
	if (!mdb)
		return -ENOMEM;

	mdb->max = max;
	mdb->old = old;

	mdb->mhash = kzalloc(max * sizeof(*mdb->mhash), GFP_ATOMIC);
	if (!mdb->mhash) {
		kfree(mdb);
		return -ENOMEM;
	}

	mdb->size = old ? old->size : 0;
	mdb->ver = old ? old->ver ^ 1 : 0;

	if (!old || elasticity)
		get_random_bytes(&mdb->secret, sizeof(mdb->secret));
	else
		mdb->secret = old->secret;

	if (!old)
		goto out;

	err = br_mdb_copy(mdb, old, elasticity);
	if (err) {
		kfree(mdb->mhash);
		kfree(mdb);
		return err;
	}

	call_rcu_bh(&mdb->rcu, br_mdb_free);

out:
	rcu_assign_pointer(*mdbp, mdb);

	return 0;
}

static struct sk_buff *br_ip4_multicast_alloc_query(struct net_bridge *br,
						    __be32 group)
{
	struct sk_buff *skb;
	struct igmphdr *ih;
	struct ethhdr *eth;
	struct iphdr *iph;

	skb = netdev_alloc_skb_ip_align(br->dev, sizeof(*eth) + sizeof(*iph) +
						 sizeof(*ih) + 4);
	if (!skb)
		goto out;

	skb->protocol = htons(ETH_P_IP);

	skb_reset_mac_header(skb);
	eth = eth_hdr(skb);

	memcpy(eth->h_source, br->dev->dev_addr, 6);
	eth->h_dest[0] = 1;
	eth->h_dest[1] = 0;
	eth->h_dest[2] = 0x5e;
	eth->h_dest[3] = 0;
	eth->h_dest[4] = 0;
	eth->h_dest[5] = 1;
	eth->h_proto = htons(ETH_P_IP);
	skb_put(skb, sizeof(*eth));

	skb_set_network_header(skb, skb->len);
	iph = ip_hdr(skb);

	iph->version = 4;
	iph->ihl = 6;
	iph->tos = 0xc0;
	iph->tot_len = htons(sizeof(*iph) + sizeof(*ih) + 4);
	iph->id = 0;
	iph->frag_off = htons(IP_DF);
	iph->ttl = 1;
	iph->protocol = IPPROTO_IGMP;
	iph->saddr = 0;
	iph->daddr = htonl(INADDR_ALLHOSTS_GROUP);
	((u8 *)&iph[1])[0] = IPOPT_RA;
	((u8 *)&iph[1])[1] = 4;
	((u8 *)&iph[1])[2] = 0;
	((u8 *)&iph[1])[3] = 0;
	ip_send_check(iph);
	skb_put(skb, 24);

	skb_set_transport_header(skb, skb->len);
	ih = igmp_hdr(skb);
	ih->type = IGMP_HOST_MEMBERSHIP_QUERY;
	ih->code = (group ? br->multicast_last_member_interval :
			    br->multicast_query_response_interval) /
		   (HZ / IGMP_TIMER_SCALE);
	ih->group = group;
	ih->csum = 0;
	ih->csum = ip_compute_csum((void *)ih, sizeof(struct igmphdr));
	skb_put(skb, sizeof(*ih));

	__skb_pull(skb, sizeof(*eth));

out:
	return skb;
}

#if IS_ENABLED(CONFIG_IPV6)
static struct sk_buff *br_ip6_multicast_alloc_query(struct net_bridge *br,
						    const struct in6_addr *group)
{
	struct sk_buff *skb;
	struct ipv6hdr *ip6h;
	struct mld_msg *mldq;
	struct ethhdr *eth;
	u8 *hopopt;
	unsigned long interval;

	skb = netdev_alloc_skb_ip_align(br->dev, sizeof(*eth) + sizeof(*ip6h) +
						 8 + sizeof(*mldq));
	if (!skb)
		goto out;

	skb->protocol = htons(ETH_P_IPV6);

	/* Ethernet header */
	skb_reset_mac_header(skb);
	eth = eth_hdr(skb);

	memcpy(eth->h_source, br->dev->dev_addr, 6);
	eth->h_proto = htons(ETH_P_IPV6);
	skb_put(skb, sizeof(*eth));

	/* IPv6 header + HbH option */
	skb_set_network_header(skb, skb->len);
	ip6h = ipv6_hdr(skb);

	*(__force __be32 *)ip6h = htonl(0x60000000);
	ip6h->payload_len = htons(8 + sizeof(*mldq));
	ip6h->nexthdr = IPPROTO_HOPOPTS;
	ip6h->hop_limit = 1;
	ipv6_addr_set(&ip6h->daddr, htonl(0xff020000), 0, 0, htonl(1));
	if (ipv6_dev_get_saddr(dev_net(br->dev), br->dev, &ip6h->daddr, 0,
			       &ip6h->saddr)) {
		kfree_skb(skb);
		return NULL;
	}
	ipv6_eth_mc_map(&ip6h->daddr, eth->h_dest);

	hopopt = (u8 *)(ip6h + 1);
	hopopt[0] = IPPROTO_ICMPV6;		/* next hdr */
	hopopt[1] = 0;				/* length of HbH */
	hopopt[2] = IPV6_TLV_ROUTERALERT;	/* Router Alert */
	hopopt[3] = 2;				/* Length of RA Option */
	hopopt[4] = 0;				/* Type = 0x0000 (MLD) */
	hopopt[5] = 0;
	hopopt[6] = IPV6_TLV_PAD0;		/* Pad0 */
	hopopt[7] = IPV6_TLV_PAD0;		/* Pad0 */

	skb_put(skb, sizeof(*ip6h) + 8);

	/* ICMPv6 */
	skb_set_transport_header(skb, skb->len);
	mldq = (struct mld_msg *) icmp6_hdr(skb);

	interval = ipv6_addr_any(group) ?
			br->multicast_query_response_interval :
			br->multicast_last_member_interval;

	mldq->mld_type = ICMPV6_MGM_QUERY;
	mldq->mld_code = 0;
	mldq->mld_cksum = 0;
	mldq->mld_maxdelay = htons((u16)jiffies_to_msecs(interval));
	mldq->mld_reserved = 0;
	mldq->mld_mca = *group;

	/* checksum */
	mldq->mld_cksum = csum_ipv6_magic(&ip6h->saddr, &ip6h->daddr,
					  sizeof(*mldq), IPPROTO_ICMPV6,
					  csum_partial(mldq,
						       sizeof(*mldq), 0));
	skb_put(skb, sizeof(*mldq));

	__skb_pull(skb, sizeof(*eth));

out:
	return skb;
}
#endif

static struct sk_buff *br_multicast_alloc_query(struct net_bridge *br,
						struct br_ip *addr)
{
	switch (addr->proto) {
	case htons(ETH_P_IP):
		return br_ip4_multicast_alloc_query(br, addr->u.ip4);
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
		return br_ip6_multicast_alloc_query(br, &addr->u.ip6);
#endif
	}
	return NULL;
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_multicast_add_hostent(struct net_bridge *br,
				    struct net_bridge_port_srcent *s,
				    unsigned char *haddr)
{
	struct net_bridge_port_hostent *h;
	struct net_bridge_port_hostent __rcu **hh;
	unsigned long now = jiffies;
	int err = -ENOENT;

	if (s == NULL || haddr == NULL)
		goto err;

	for (hh = &s->hlist;
	     (h = mlock_dereference(*hh, br)) != NULL;
	     hh = &h->next) {
		if (memcmp(h->haddr, haddr, ETH_ALEN) == 0)
			goto found;
	}

	h = kzalloc(sizeof(*h), GFP_ATOMIC);
	err = -ENOMEM;
	if (unlikely(!h))
		goto err;

	memcpy(h->haddr, haddr, ETH_ALEN);
	h->srcent = s;
	h->next = *hh;
	setup_timer(&h->timer, br_multicast_port_group_hostent_expired,
		    (unsigned long)h);

	rcu_assign_pointer(*hh, h);

found:
	mod_timer(&h->timer, now + br->multicast_membership_interval);
	err = 0;
err:
	return err;
}

static int br_multicast_new_srcent(struct net_bridge *br,
				   struct net_bridge_port_group *p,
				   struct br_ip *src,
				   unsigned char *haddr)
{
	struct net_bridge_port_srcent *s;
	struct net_bridge_port_srcent __rcu **ss;
	int err = 0;

	if (p == NULL || src == NULL)
		return -EINVAL;

	for (ss = &p->inlist;
	     (s = mlock_dereference(*ss, br)) != NULL;
	     ss = &s->next) {
		if (br_ip_equal(&s->addr, src))
			goto found;
	}

	s = kzalloc(sizeof(*s), GFP_ATOMIC);
	if (unlikely(!s))
		return -ENOMEM;

	s->addr = *src;
	s->ports = p;
	s->next = *ss;

	rcu_assign_pointer(*ss, s);

found:
	err = br_multicast_add_hostent(br, s, haddr);
	return err;
}

static int br_ip4_multicast_add_src(struct net_bridge *br,
				    struct net_bridge_port *port,
				    struct br_ip *group,
				    unsigned char *haddr)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	struct net_bridge_port_srcent *s;
	__be32 dummy_src = 0;
	struct br_ip br_src;
	int err = -ENOENT;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	mdb = mlock_dereference(br->mdb, br);
	mp = br_mdb_ip_get(mdb, group);
	if (!mp)
		goto err;

	for (p = mlock_dereference(mp->ports, br);
	     p != NULL;
	     p = mlock_dereference(p->next, br)) {
		if (p->port == port)
			break;
	}
	if (p == NULL)
		goto err;

	br_src.u.ip4 = dummy_src;
	br_src.proto = htons(ETH_P_IP);

	err = br_multicast_new_srcent(br, p, &br_src, haddr);
	if (err)
		goto err;

out:
	err = 0;
err:
	spin_unlock(&br->multicast_lock);
	return err;
}

#if IS_ENABLED(CONFIG_IPV6)
static int br_ip6_multicast_add_src(struct net_bridge *br,
				    struct net_bridge_port *port,
				    struct br_ip *group,
				    __be16 nsrcs,
				    struct in6_addr *src,
				    unsigned char *haddr)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	struct in6_addr dummy_src;
	struct br_ip br_src;
	int i;
	int err = -ENOENT;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	mdb = mlock_dereference(br->mdb, br);
	mp = br_mdb_ip_get(mdb, group);
	if (!mp)
		goto err;

	for (p = mlock_dereference(mp->ports, br);
	     p != NULL;
	     p = mlock_dereference(p->next, br)) {
		if (p->port == port)
			break;
	}
	if (p == NULL)
		goto err;

	if (nsrcs == 0 || src == NULL) {
		/* dummy entry create */
		memset(&dummy_src, 0, sizeof(struct in6_addr));
		src = &dummy_src;
		nsrcs = 1;
	}

	for (i = 0; i < nsrcs; i++) {
		br_src.u.ip6 = src[i];
		br_src.proto = htons(ETH_P_IPV6);

		err = br_multicast_new_srcent(br, p, &br_src, haddr);
		if (err)
			goto err;
	}

out:
	err = 0;
err:
	spin_unlock(&br->multicast_lock);
	return err;
}
#endif
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static struct net_bridge_mdb_entry *br_multicast_get_group(
	struct net_bridge *br, struct net_bridge_port *port,
	struct br_ip *group, int hash)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct hlist_node *p;
	unsigned count = 0;
	unsigned max;
	int elasticity;
	int err;

	mdb = rcu_dereference_protected(br->mdb, 1);
	hlist_for_each_entry(mp, p, &mdb->mhash[hash], hlist[mdb->ver]) {
		count++;
		if (unlikely(br_ip_equal(group, &mp->addr)))
			return mp;
	}

	elasticity = 0;
	max = mdb->max;

	if (unlikely(count > br->hash_elasticity && count)) {
		if (net_ratelimit())
			br_info(br, "Multicast hash table "
				"chain limit reached: %s\n",
				port ? port->dev->name : br->dev->name);

		elasticity = br->hash_elasticity;
	}

	if (mdb->size >= max) {
		max *= 2;
		if (unlikely(max >= br->hash_max)) {
			br_warn(br, "Multicast hash table maximum "
				"reached, disabling snooping: %s, %d\n",
				port ? port->dev->name : br->dev->name, max);
			err = -E2BIG;
disable:
			br->multicast_disabled = 1;
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
			br->multicast_mld_only = 0;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
			goto err;
		}
	}

	if (max > mdb->max || elasticity) {
		if (mdb->old) {
			if (net_ratelimit())
				br_info(br, "Multicast hash table "
					"on fire: %s\n",
					port ? port->dev->name : br->dev->name);
			err = -EEXIST;
			goto err;
		}

		err = br_mdb_rehash(&br->mdb, max, elasticity);
		if (err) {
			br_warn(br, "Cannot rehash multicast "
				"hash table, disabling snooping: %s, %d, %d\n",
				port ? port->dev->name : br->dev->name,
				mdb->size, err);
			goto disable;
		}

		err = -EAGAIN;
		goto err;
	}

	return NULL;

err:
	mp = ERR_PTR(err);
	return mp;
}

static struct net_bridge_mdb_entry *br_multicast_new_group(
	struct net_bridge *br, struct net_bridge_port *port,
	struct br_ip *group)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	int hash;
	int err;

	mdb = rcu_dereference_protected(br->mdb, 1);
	if (!mdb) {
		err = br_mdb_rehash(&br->mdb, BR_HASH_SIZE, 0);
		if (err)
			return ERR_PTR(err);
		goto rehash;
	}

	hash = br_ip_hash(mdb, group);
	mp = br_multicast_get_group(br, port, group, hash);
	switch (PTR_ERR(mp)) {
	case 0:
		break;

	case -EAGAIN:
rehash:
		mdb = rcu_dereference_protected(br->mdb, 1);
		hash = br_ip_hash(mdb, group);
		break;

	default:
		goto out;
	}

	mp = kzalloc(sizeof(*mp), GFP_ATOMIC);
	if (unlikely(!mp))
		return ERR_PTR(-ENOMEM);

	mp->br = br;
	mp->addr = *group;
	setup_timer(&mp->timer, br_multicast_group_expired,
		    (unsigned long)mp);

	hlist_add_head_rcu(&mp->hlist[mdb->ver], &mdb->mhash[hash]);
	mdb->size++;

out:
	return mp;
}

static int br_multicast_add_group(struct net_bridge *br,
				  struct net_bridge_port *port,
				  struct br_ip *group)
{
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	struct net_bridge_port_group __rcu **pp;
	unsigned long now = jiffies;
	int err;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	mp = br_multicast_new_group(br, port, group);
	err = PTR_ERR(mp);
	if (IS_ERR(mp))
		goto err;

	if (!port) {
		mp->mglist = true;
		mod_timer(&mp->timer, now + br->multicast_membership_interval);
		goto out;
	}

	for (pp = &mp->ports;
	     (p = mlock_dereference(*pp, br)) != NULL;
	     pp = &p->next) {
		if (p->port == port)
			goto found;
		if ((unsigned long)p->port < (unsigned long)port)
			break;
	}

	p = kzalloc(sizeof(*p), GFP_ATOMIC);
	err = -ENOMEM;
	if (unlikely(!p))
		goto err;

	p->addr = *group;
	p->port = port;
	p->next = *pp;
	hlist_add_head(&p->mglist, &port->mglist);
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	setup_timer(&p->timer, br_multicast_port_group_expired,
		    (unsigned long)p);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */

	rcu_assign_pointer(*pp, p);

found:
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	mod_timer(&p->timer, now + br->multicast_membership_interval);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
out:
	err = 0;

err:
	spin_unlock(&br->multicast_lock);
	return err;
}

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_ip4_multicast_add_group(struct net_bridge *br,
				      struct net_bridge_port *port,
				      __be32 group)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
static int br_ip4_multicast_add_group(struct net_bridge *br,
				      struct net_bridge_port *port,
				      __be32 group,
				      unsigned char *haddr)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
	struct br_ip br_group;
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	int err;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	if (ipv4_is_local_multicast(group))
		return 0;

	br_group.u.ip4 = group;
	br_group.proto = htons(ETH_P_IP);

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	return br_multicast_add_group(br, port, &br_group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	err = br_multicast_add_group(br, port, &br_group);
	if (err)
		goto out;

	err = br_ip4_multicast_add_src(br, port, &br_group, haddr);

out:
	return err;
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
}

#if IS_ENABLED(CONFIG_IPV6)
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_ip6_multicast_add_group(struct net_bridge *br,
				      struct net_bridge_port *port,
				      const struct in6_addr *group)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
static int br_ip6_multicast_add_group(struct net_bridge *br,
				      struct net_bridge_port *port,
				      const struct in6_addr *group,
				      __be16 nsrcs,
				      struct in6_addr *src,
				      unsigned char *haddr)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
	struct br_ip br_group;
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	int err;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	if (!ipv6_is_transient_multicast(group))
		return 0;

	br_group.u.ip6 = *group;
	br_group.proto = htons(ETH_P_IPV6);

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	return br_multicast_add_group(br, port, &br_group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	err = br_multicast_add_group(br, port, &br_group);
	if (err)
		goto out;

	err = br_ip6_multicast_add_src(br, port, &br_group, nsrcs, src, haddr);

out:
	return err;
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
}
#endif

static void br_multicast_router_expired(unsigned long data)
{
	struct net_bridge_port *port = (void *)data;
	struct net_bridge *br = port->br;

	spin_lock(&br->multicast_lock);
	if (port->multicast_router != 1 ||
	    timer_pending(&port->multicast_router_timer) ||
	    hlist_unhashed(&port->rlist))
		goto out;

	hlist_del_init_rcu(&port->rlist);

out:
	spin_unlock(&br->multicast_lock);
}

static void br_multicast_local_router_expired(unsigned long data)
{
}

static void __br_multicast_send_query(struct net_bridge *br,
				      struct net_bridge_port *port,
				      struct br_ip *ip)
{
	struct sk_buff *skb;

	skb = br_multicast_alloc_query(br, ip);
	if (!skb)
		return;

	if (port) {
		__skb_push(skb, sizeof(struct ethhdr));
		skb->dev = port->dev;
		NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_OUT, skb, NULL, skb->dev,
			dev_queue_xmit);
	} else
		netif_rx(skb);
}

static void br_multicast_send_query(struct net_bridge *br,
				    struct net_bridge_port *port, u32 sent)
{
	unsigned long time;
	struct br_ip br_group;

	if (!netif_running(br->dev) || br->multicast_disabled ||
	    timer_pending(&br->multicast_querier_timer))
		return;

	memset(&br_group.u, 0, sizeof(br_group.u));

	br_group.proto = htons(ETH_P_IP);
	__br_multicast_send_query(br, port, &br_group);

#if IS_ENABLED(CONFIG_IPV6)
	br_group.proto = htons(ETH_P_IPV6);
	__br_multicast_send_query(br, port, &br_group);
#endif

	time = jiffies;
	time += sent < br->multicast_startup_query_count ?
		br->multicast_startup_query_interval :
		br->multicast_query_interval;
	mod_timer(port ? &port->multicast_query_timer :
			 &br->multicast_query_timer, time);
}

static void br_multicast_port_query_expired(unsigned long data)
{
	struct net_bridge_port *port = (void *)data;
	struct net_bridge *br = port->br;

	spin_lock(&br->multicast_lock);
	if (port->state == BR_STATE_DISABLED ||
	    port->state == BR_STATE_BLOCKING)
		goto out;

	if (port->multicast_startup_queries_sent <
	    br->multicast_startup_query_count)
		port->multicast_startup_queries_sent++;

	br_multicast_send_query(port->br, port,
				port->multicast_startup_queries_sent);

out:
	spin_unlock(&br->multicast_lock);
}

void br_multicast_add_port(struct net_bridge_port *port)
{
	port->multicast_router = 1;

	setup_timer(&port->multicast_router_timer, br_multicast_router_expired,
		    (unsigned long)port);
	setup_timer(&port->multicast_query_timer,
		    br_multicast_port_query_expired, (unsigned long)port);
}

void br_multicast_del_port(struct net_bridge_port *port)
{
	del_timer_sync(&port->multicast_router_timer);
}

static void __br_multicast_enable_port(struct net_bridge_port *port)
{
	port->multicast_startup_queries_sent = 0;

	if (try_to_del_timer_sync(&port->multicast_query_timer) >= 0 ||
	    del_timer(&port->multicast_query_timer)) {
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		mod_timer(&port->multicast_query_timer, jiffies);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	}
}

void br_multicast_enable_port(struct net_bridge_port *port)
{
	struct net_bridge *br = port->br;

	spin_lock(&br->multicast_lock);
	if (br->multicast_disabled || !netif_running(br->dev))
		goto out;

	__br_multicast_enable_port(port);

out:
	spin_unlock(&br->multicast_lock);
}

void br_multicast_disable_port(struct net_bridge_port *port)
{
	struct net_bridge *br = port->br;
	struct net_bridge_port_group *pg;
	struct hlist_node *p, *n;

	spin_lock(&br->multicast_lock);
	hlist_for_each_entry_safe(pg, p, n, &port->mglist, mglist)
		br_multicast_del_pg(br, pg);

	if (!hlist_unhashed(&port->rlist))
		hlist_del_init_rcu(&port->rlist);
	del_timer(&port->multicast_router_timer);
	del_timer(&port->multicast_query_timer);
	spin_unlock(&br->multicast_lock);
}

static int br_ip4_multicast_igmp3_report(struct net_bridge *br,
					 struct net_bridge_port *port,
					 struct sk_buff *skb)
{
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	unsigned char *h_source = eth_hdr(skb)->h_source;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
	struct igmpv3_report *ih;
	struct igmpv3_grec *grec;
	int i;
	int len;
	int num;
	int type;
	int err = 0;
	__be32 group;

	if (!pskb_may_pull(skb, sizeof(*ih)))
		return -EINVAL;

	ih = igmpv3_report_hdr(skb);
	num = ntohs(ih->ngrec);
	len = sizeof(*ih);

	for (i = 0; i < num; i++) {
		len += sizeof(*grec);
		if (!pskb_may_pull(skb, len))
			return -EINVAL;

		grec = (void *)(skb->data + len - sizeof(*grec));
		group = grec->grec_mca;
		type = grec->grec_type;

		len += ntohs(grec->grec_nsrcs) * 4;
		if (!pskb_may_pull(skb, len))
			return -EINVAL;

		/* We treat this as an IGMPv2 report for now. */
		switch (type) {
		case IGMPV3_MODE_IS_INCLUDE:
		case IGMPV3_MODE_IS_EXCLUDE:
		case IGMPV3_CHANGE_TO_INCLUDE:
		case IGMPV3_CHANGE_TO_EXCLUDE:
		case IGMPV3_ALLOW_NEW_SOURCES:
		case IGMPV3_BLOCK_OLD_SOURCES:
			break;

		default:
			continue;
		}

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		err = br_ip4_multicast_add_group(br, port, group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		err = br_ip4_multicast_add_group(br, port, group, h_source);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		if (err)
			break;
	}

	return err;
}

#if IS_ENABLED(CONFIG_IPV6)
static int br_ip6_multicast_mld2_report(struct net_bridge *br,
					struct net_bridge_port *port,
					struct sk_buff *skb)
{
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	unsigned char *h_source = eth_hdr(skb)->h_source;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
	struct icmp6hdr *icmp6h;
	struct mld2_grec *grec;
	int i;
	int len;
	int num;
	int err = 0;

	if (!pskb_may_pull(skb, sizeof(*icmp6h)))
		return -EINVAL;

	icmp6h = icmp6_hdr(skb);
	num = ntohs(icmp6h->icmp6_dataun.un_data16[1]);
	len = sizeof(*icmp6h);

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	/* processed first BLOCK. */
	for (i = 0; i < num; i++) {
		__be16 *nsrcs, _nsrcs;

		nsrcs = skb_header_pointer(skb,
					   len + offsetof(struct mld2_grec,
							  grec_nsrcs),
					   sizeof(_nsrcs), &_nsrcs);
		if (!nsrcs)
			return -EINVAL;

		if (!pskb_may_pull(skb,
				   len + sizeof(*grec) +
				   sizeof(struct in6_addr) * ntohs(*nsrcs)))
			return -EINVAL;

		grec = (struct mld2_grec *)(skb->data + len);
		len += sizeof(*grec) +
		       sizeof(struct in6_addr) * ntohs(*nsrcs);

		switch (grec->grec_type) {
		case MLD2_BLOCK_OLD_SOURCES:
			break;

		default:
			continue;
		}

		if (!ntohs(grec->grec_nsrcs))
			continue;

		br_ip6_multicast_leave_group(br, port, &grec->grec_mca,
					     ntohs(grec->grec_nsrcs),
					     grec->grec_src, h_source);
	}

	len = sizeof(*icmp6h);
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	for (i = 0; i < num; i++) {
		__be16 *nsrcs, _nsrcs;

		nsrcs = skb_header_pointer(skb,
					   len + offsetof(struct mld2_grec,
							  grec_nsrcs),
					   sizeof(_nsrcs), &_nsrcs);
		if (!nsrcs)
			return -EINVAL;

		if (!pskb_may_pull(skb,
				   len + sizeof(*grec) +
				   sizeof(struct in6_addr) * ntohs(*nsrcs)))
			return -EINVAL;

		grec = (struct mld2_grec *)(skb->data + len);
		len += sizeof(*grec) +
		       sizeof(struct in6_addr) * ntohs(*nsrcs);

		/* We treat these as MLDv1 reports for now. */
		switch (grec->grec_type) {
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		case MLD2_MODE_IS_INCLUDE:
		case MLD2_MODE_IS_EXCLUDE:
		case MLD2_CHANGE_TO_INCLUDE:
		case MLD2_CHANGE_TO_EXCLUDE:
		case MLD2_ALLOW_NEW_SOURCES:
		case MLD2_BLOCK_OLD_SOURCES:
			break;
		default:
			continue;
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		case MLD2_MODE_IS_INCLUDE:
		case MLD2_ALLOW_NEW_SOURCES:
			break;

		case MLD2_MODE_IS_EXCLUDE:
		case MLD2_CHANGE_TO_INCLUDE:
		case MLD2_CHANGE_TO_EXCLUDE:
		default:
			continue;
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		}

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		err = br_ip6_multicast_add_group(br, port, &grec->grec_mca);
		if (!err)
			break;
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		if (!ntohs(grec->grec_nsrcs))
			continue;

		err = br_ip6_multicast_add_group(br, port, &grec->grec_mca,
						 ntohs(grec->grec_nsrcs),
						 grec->grec_src, h_source);
		if (err)
			break;
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	}

	return err;
}
#endif

/*
 * Add port to rotuer_list
 *  list is maintained ordered by pointer value
 *  and locked by br->multicast_lock and RCU
 */
static void br_multicast_add_router(struct net_bridge *br,
				    struct net_bridge_port *port)
{
	struct net_bridge_port *p;
	struct hlist_node *n, *slot = NULL;

	hlist_for_each_entry(p, n, &br->router_list, rlist) {
		if ((unsigned long) port >= (unsigned long) p)
			break;
		slot = n;
	}

	if (slot)
		hlist_add_after_rcu(slot, &port->rlist);
	else
		hlist_add_head_rcu(&port->rlist, &br->router_list);
}

static void br_multicast_mark_router(struct net_bridge *br,
				     struct net_bridge_port *port)
{
	unsigned long now = jiffies;

	if (!port) {
		if (br->multicast_router == 1)
			mod_timer(&br->multicast_router_timer,
				  now + br->multicast_querier_interval);
		return;
	}

	if (port->multicast_router != 1)
		return;

	if (!hlist_unhashed(&port->rlist))
		goto timer;

	br_multicast_add_router(br, port);

timer:
	mod_timer(&port->multicast_router_timer,
		  now + br->multicast_querier_interval);
}

static void br_multicast_query_received(struct net_bridge *br,
					struct net_bridge_port *port,
					int saddr)
{
	if (saddr)
		mod_timer(&br->multicast_querier_timer,
			  jiffies + br->multicast_querier_interval);
	else if (timer_pending(&br->multicast_querier_timer))
		return;

	br_multicast_mark_router(br, port);
}

static int br_ip4_multicast_query(struct net_bridge *br,
				  struct net_bridge_port *port,
				  struct sk_buff *skb)
{
	const struct iphdr *iph = ip_hdr(skb);
	struct igmphdr *ih = igmp_hdr(skb);
	struct net_bridge_mdb_entry *mp;
	struct igmpv3_query *ih3;
	struct net_bridge_port_group *p;
	struct net_bridge_port_group __rcu **pp;
	unsigned long max_delay;
	unsigned long now = jiffies;
	__be32 group;
	int err = 0;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	br_multicast_query_received(br, port, !!iph->saddr);

	group = ih->group;

	if (skb->len == sizeof(*ih)) {
		max_delay = ih->code * (HZ / IGMP_TIMER_SCALE);

		if (!max_delay) {
			max_delay = 10 * HZ;
			group = 0;
		}
	} else {
		if (!pskb_may_pull(skb, sizeof(struct igmpv3_query))) {
			err = -EINVAL;
			goto out;
		}

		ih3 = igmpv3_query_hdr(skb);
		if (ih3->nsrcs)
			goto out;

		max_delay = ih3->code ?
			    IGMPV3_MRC(ih3->code) * (HZ / IGMP_TIMER_SCALE) : 1;
	}

	if (!group)
		goto out;

	mp = br_mdb_ip4_get(mlock_dereference(br->mdb, br), group);
	if (!mp)
		goto out;

	max_delay *= br->multicast_last_member_count;

	if (mp->mglist &&
	    (timer_pending(&mp->timer) ?
	     time_after(mp->timer.expires, now + max_delay) :
	     try_to_del_timer_sync(&mp->timer) >= 0))
		mod_timer(&mp->timer, now + max_delay);

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	for (pp = &mp->ports;
	     (p = mlock_dereference(*pp, br)) != NULL;
	     pp = &p->next) {
		if (timer_pending(&p->timer) ?
		    time_after(p->timer.expires, now + max_delay) :
		    try_to_del_timer_sync(&p->timer) >= 0)
			mod_timer(&p->timer, now + max_delay);
	}
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */

out:
	spin_unlock(&br->multicast_lock);
	return err;
}

#if IS_ENABLED(CONFIG_IPV6)
static int br_ip6_multicast_query(struct net_bridge *br,
				  struct net_bridge_port *port,
				  struct sk_buff *skb)
{
	const struct ipv6hdr *ip6h = ipv6_hdr(skb);
	struct mld_msg *mld = (struct mld_msg *) icmp6_hdr(skb);
	struct net_bridge_mdb_entry *mp;
	struct mld2_query *mld2q;
	struct net_bridge_port_group *p;
	struct net_bridge_port_group __rcu **pp;
	unsigned long max_delay;
	unsigned long now = jiffies;
	const struct in6_addr *group = NULL;
	int err = 0;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	br_multicast_query_received(br, port, !ipv6_addr_any(&ip6h->saddr));

	/* RFC2710+RFC3810 (MLDv1+MLDv2) require link-local source addresses */
	if (!(ipv6_addr_type(&ip6h->saddr) & IPV6_ADDR_LINKLOCAL)) {
		err = -EINVAL;
		goto out;
	}

	if (skb->len == sizeof(*mld)) {
		if (!pskb_may_pull(skb, sizeof(*mld))) {
			err = -EINVAL;
			goto out;
		}
		mld = (struct mld_msg *) icmp6_hdr(skb);
		max_delay = msecs_to_jiffies(htons(mld->mld_maxdelay));
		if (max_delay)
			group = &mld->mld_mca;
	} else if (skb->len >= sizeof(*mld2q)) {
		if (!pskb_may_pull(skb, sizeof(*mld2q))) {
			err = -EINVAL;
			goto out;
		}
		mld2q = (struct mld2_query *)icmp6_hdr(skb);
		if (!mld2q->mld2q_nsrcs)
			group = &mld2q->mld2q_mca;

		max_delay = max(msecs_to_jiffies(MLDV2_MRC(ntohs(mld2q->mld2q_mrc))), 1UL);
	}

	if (!group)
		goto out;

	mp = br_mdb_ip6_get(mlock_dereference(br->mdb, br), group);
	if (!mp)
		goto out;

	max_delay *= br->multicast_last_member_count;
	if (mp->mglist &&
	    (timer_pending(&mp->timer) ?
	     time_after(mp->timer.expires, now + max_delay) :
	     try_to_del_timer_sync(&mp->timer) >= 0))
		mod_timer(&mp->timer, now + max_delay);

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	for (pp = &mp->ports;
	     (p = mlock_dereference(*pp, br)) != NULL;
	     pp = &p->next) {
		if (timer_pending(&p->timer) ?
		    time_after(p->timer.expires, now + max_delay) :
		    try_to_del_timer_sync(&p->timer) >= 0)
			mod_timer(&p->timer, now + max_delay);
	}
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */

out:
	spin_unlock(&br->multicast_lock);
	return err;
}
#endif

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_multicast_leave_hostent(struct net_bridge *br,
				       struct net_bridge_port_srcent *s,
				       unsigned char *haddr)
{
	struct net_bridge_port_hostent *h;
	unsigned long now;
	unsigned long time;

	if (s == NULL || haddr == NULL)
		return;

	now = jiffies;
	time = now + br->multicast_last_member_count *
		     br->multicast_last_member_interval;

	for (h = mlock_dereference(s->hlist, br);
	     h != NULL;
	     h = mlock_dereference(h->next, br)) {
		if (memcmp(h->haddr, haddr, ETH_ALEN) != 0)
			continue;

		if (timer_pending(&h->timer) ?
		    time_after(h->timer.expires, time) :
		    try_to_del_timer_sync(&h->timer) >= 0) {
			mod_timer(&h->timer, time);
		}
		break;
	}
}

static void br_ip4_multicast_leave_src(struct net_bridge *br,
				       struct net_bridge_port *port,
				       struct br_ip *group,
				       unsigned char *haddr)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	struct net_bridge_port_srcent *s;
	__be32 dummy_src = 0;
	struct br_ip br_src;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	mdb = mlock_dereference(br->mdb, br);
	mp = br_mdb_ip_get(mdb, group);
	if (!mp)
		goto out;

	for (p = mlock_dereference(mp->ports, br);
	     p != NULL;
	     p = mlock_dereference(p->next, br)) {
		if (p->port == port)
			break;
	}
	if (p == NULL)
		goto out;

	br_src.u.ip4 = dummy_src;
	br_src.proto = htons(ETH_P_IP);

	for (s = mlock_dereference(p->inlist, br);
	     s != NULL;
	     s = mlock_dereference(s->next, br)) {
		if (!(br_ip_equal(&s->addr, &br_src)))
			continue;

		br_multicast_leave_hostent(br, s, haddr);
		break;
	}

out:
	spin_unlock(&br->multicast_lock);
}

#if IS_ENABLED(CONFIG_IPV6)
static void br_ip6_multicast_leave_src(struct net_bridge *br,
				       struct net_bridge_port *port,
				       struct br_ip *group,
				       __be16 nsrcs,
				       struct in6_addr *src,
				       unsigned char *haddr)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	struct net_bridge_port_srcent *s;
	struct in6_addr dummy_src;
	struct br_ip br_src;
	int i;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED))
		goto out;

	mdb = mlock_dereference(br->mdb, br);
	mp = br_mdb_ip_get(mdb, group);
	if (!mp)
		goto out;

	for (p = mlock_dereference(mp->ports, br);
	     p != NULL;
	     p = mlock_dereference(p->next, br)) {
		if (p->port == port)
			break;
	}
	if (p == NULL)
		goto out;

	if (nsrcs == 0 || src == NULL) {
		/* dummy entry create */
		memset(&dummy_src, 0, sizeof(struct in6_addr));
		src = &dummy_src;
		nsrcs = 1;
	}

	for (i = 0; i < nsrcs; i++) {
		br_src.u.ip6 = src[i];
		br_src.proto = htons(ETH_P_IPV6);

		for (s = mlock_dereference(p->inlist, br);
		     s != NULL;
		     s = mlock_dereference(s->next, br)) {
			if (!(br_ip_equal(&s->addr, &br_src)))
				continue;

			br_multicast_leave_hostent(br, s, haddr);
			break;
		}
	}

out:
	spin_unlock(&br->multicast_lock);
}
#endif
#else  /* USE_NECPF_BRIDGE_SSIS_SSM2S */
static void br_multicast_leave_group(struct net_bridge *br,
				     struct net_bridge_port *port,
				     struct br_ip *group)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct net_bridge_port_group *p;
	unsigned long now;
	unsigned long time;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) ||
	    (port && port->state == BR_STATE_DISABLED) ||
	    timer_pending(&br->multicast_querier_timer))
		goto out;

	mdb = mlock_dereference(br->mdb, br);
	mp = br_mdb_ip_get(mdb, group);
	if (!mp)
		goto out;

	now = jiffies;
	time = now + br->multicast_last_member_count *
		     br->multicast_last_member_interval;

	if (!port) {
		if (mp->mglist &&
		    (timer_pending(&mp->timer) ?
		     time_after(mp->timer.expires, time) :
		     try_to_del_timer_sync(&mp->timer) >= 0)) {
			mod_timer(&mp->timer, time);
		}

		goto out;
	}

	for (p = mlock_dereference(mp->ports, br);
	     p != NULL;
	     p = mlock_dereference(p->next, br)) {
		if (p->port != port)
			continue;

		if (!hlist_unhashed(&p->mglist) &&
		    (timer_pending(&p->timer) ?
		     time_after(p->timer.expires, time) :
		     try_to_del_timer_sync(&p->timer) >= 0)) {
			mod_timer(&p->timer, time);
		}

		break;
	}

out:
	spin_unlock(&br->multicast_lock);
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_ip4_multicast_leave_group(struct net_bridge *br,
					 struct net_bridge_port *port,
					 __be32 group)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
static void br_ip4_multicast_leave_group(struct net_bridge *br,
					 struct net_bridge_port *port,
					 __be32 group,
					 unsigned char *haddr)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
	struct br_ip br_group;

	if (ipv4_is_local_multicast(group))
		return;

	br_group.u.ip4 = group;
	br_group.proto = htons(ETH_P_IP);

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	br_multicast_leave_group(br, port, &br_group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	br_ip4_multicast_leave_src(br, port, &br_group, haddr);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
}

#if IS_ENABLED(CONFIG_IPV6)
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
static void br_ip6_multicast_leave_group(struct net_bridge *br,
					 struct net_bridge_port *port,
					 const struct in6_addr *group)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
static void br_ip6_multicast_leave_group(struct net_bridge *br,
					 struct net_bridge_port *port,
					 const struct in6_addr *group,
					 __be16 nsrcs,
					 struct in6_addr *src,
					 unsigned char *haddr)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
	struct br_ip br_group;

	if (!ipv6_is_transient_multicast(group))
		return;

	br_group.u.ip6 = *group;
	br_group.proto = htons(ETH_P_IPV6);

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	br_multicast_leave_group(br, port, &br_group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	br_ip6_multicast_leave_src(br, port, &br_group, nsrcs, src, haddr);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
}
#endif

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_multicast_ipv4_rcv_up(struct net_bridge *br,
				    __u8 type,
				    u8 *local_du_fwd)
{
	int snpok = 0;

	switch (type) {
	case IGMPV2_HOST_MEMBERSHIP_REPORT:
	case IGMP_HOST_LEAVE_MESSAGE:
	case IGMP_HOST_MEMBERSHIP_REPORT:
		*local_du_fwd = BR_DUFW_UP_ONLY;
		/* fall through */
	case IGMP_HOST_MEMBERSHIP_QUERY:
	case IGMPV3_HOST_MEMBERSHIP_REPORT:
	default:
		goto skip_snooping;
	}

	snpok = 1;
skip_snooping:
	return snpok;
}

static int br_multicast_ipv4_rcv_dl(struct net_bridge *br,
				    __u8 type,
				    u8 *local_du_fwd)
{
	int snpok = 0;

	switch (type) {
	case IGMPV2_HOST_MEMBERSHIP_REPORT:
	case IGMP_HOST_LEAVE_MESSAGE:
		break;
	case IGMP_HOST_MEMBERSHIP_REPORT:
		if (br->multicast_igmpv1_disabled)
			goto skip_snooping;
		break;
	case IGMPV3_HOST_MEMBERSHIP_REPORT:
		if (br->multicast_igmpv3_disabled)
			goto skip_snooping;
		break;
	case IGMP_HOST_MEMBERSHIP_QUERY:
	default:
		goto skip_snooping;
	}

	snpok = 1;
skip_snooping:
	*local_du_fwd = BR_DUFW_UP_ONLY;
	return snpok;
}

static int br_multicast_ipv4_rcv_addrcheck(struct net_bridge *br,
					   const struct iphdr *iph,
					   struct igmphdr *ih,
					   u8 *local_du_fwd)
{
	int snpok = 0;

	switch (ih->type) {
	case IGMPV2_HOST_MEMBERSHIP_REPORT:
		if (ipv4_is_ssdp_group(ih->group))
			goto all_port;

		if (!ipv4_is_multicast(ih->group))
			goto skip_igmp;

		if (ipv4_is_local_multicast(ih->group))
			goto skip_igmp;

		if (ih->group != iph->daddr)
			goto skip_igmp;

		break;

	case IGMP_HOST_LEAVE_MESSAGE:
		if (ipv4_is_ssdp_group(ih->group))
			goto all_port;

		if (!ipv4_is_allrtrs_group(iph->daddr))
			goto all_port;

		if (!ipv4_is_multicast(ih->group))
			goto skip_igmp;

		break;

	default:
		break;
	}

	snpok = 1;
skip_igmp:
	if (br->multicast_upstream)
		*local_du_fwd = BR_DUFW_UP_ONLY;
	return snpok;
all_port:
	if (br->multicast_upstream)
		*local_du_fwd = BR_DUFW_DEFAULT;
	return snpok;
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_multicast_ipv4_rcv(struct net_bridge *br,
				 struct net_bridge_port *port,
				 struct sk_buff *skb)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
static int br_multicast_ipv4_rcv(struct net_bridge *br,
				 struct net_bridge_port *port,
				 struct sk_buff *skb,
				 u8 *du_forward)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	unsigned char *h_source = eth_hdr(skb)->h_source;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
	struct sk_buff *skb2 = skb;
	const struct iphdr *iph;
	struct igmphdr *ih;
	unsigned len;
	unsigned offset;
	int err;
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	int snp_check;
	u8 local_du_fwd = BR_DUFW_DEFAULT;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	/* We treat OOM as packet loss for now. */
	if (!pskb_may_pull(skb, sizeof(*iph)))
		return -EINVAL;

	iph = ip_hdr(skb);

	if (iph->ihl < 5 || iph->version != 4)
		return -EINVAL;

	if (!pskb_may_pull(skb, ip_hdrlen(skb)))
		return -EINVAL;

	iph = ip_hdr(skb);

	if (unlikely(ip_fast_csum((u8 *)iph, iph->ihl)))
		return -EINVAL;

	if (iph->protocol != IPPROTO_IGMP) {
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		if ((iph->daddr & IGMP_LOCAL_GROUP_MASK) != IGMP_LOCAL_GROUP)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		if (!br->multicast_flooding_ipv4 &&
		    (iph->daddr & IGMP_LOCAL_GROUP_MASK) != IGMP_LOCAL_GROUP)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
			BR_INPUT_SKB_CB(skb)->mrouters_only = 1;
		return 0;
	}

	len = ntohs(iph->tot_len);
	if (skb->len < len || len < ip_hdrlen(skb))
		return -EINVAL;

	if (skb->len > len) {
		skb2 = skb_clone(skb, GFP_ATOMIC);
		if (!skb2)
			return -ENOMEM;

		err = pskb_trim_rcsum(skb2, len);
		if (err)
			goto err_out;
	}

	len -= ip_hdrlen(skb2);
	offset = skb_network_offset(skb2) + ip_hdrlen(skb2);
	__skb_pull(skb2, offset);
	skb_reset_transport_header(skb2);

	err = -EINVAL;
	if (!pskb_may_pull(skb2, sizeof(*ih)))
		goto out;

	switch (skb2->ip_summed) {
	case CHECKSUM_COMPLETE:
		if (!csum_fold(skb2->csum))
			break;
		/* fall through */
	case CHECKSUM_NONE:
		skb2->csum = 0;
		if (skb_checksum_complete(skb2))
			goto out;
	}

	err = 0;

	BR_INPUT_SKB_CB(skb)->igmp = 1;
	ih = igmp_hdr(skb2);

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (port && br->multicast_upstream) {
		if (port->multicast_upstream)
			snp_check = br_multicast_ipv4_rcv_up(br, ih->type,
							     &local_du_fwd);
		else
			snp_check = br_multicast_ipv4_rcv_dl(br, ih->type,
							     &local_du_fwd);
		if (!snp_check)
			goto out;
	}

	if (br->multicast_addrcheck) {
		snp_check = br_multicast_ipv4_rcv_addrcheck(br, iph, ih,
							    &local_du_fwd);
		if (!snp_check)
			goto out;
	}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	switch (ih->type) {
	case IGMP_HOST_MEMBERSHIP_REPORT:
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
		BR_INPUT_SKB_CB(skb)->mrouters_only = 1;
		if (br->multicast_igmpv1_disabled)
			break;
		/* fall through */
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
	case IGMPV2_HOST_MEMBERSHIP_REPORT:
		BR_INPUT_SKB_CB(skb)->mrouters_only = 1;
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		err = br_ip4_multicast_add_group(br, port, ih->group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		err = br_ip4_multicast_add_group(br, port, ih->group, h_source);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		break;
	case IGMPV3_HOST_MEMBERSHIP_REPORT:
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
		if (br->multicast_igmpv3_disabled)
			break;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
		err = br_ip4_multicast_igmp3_report(br, port, skb2);
		break;
	case IGMP_HOST_MEMBERSHIP_QUERY:
		err = br_ip4_multicast_query(br, port, skb2);
		break;
	case IGMP_HOST_LEAVE_MESSAGE:
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		br_ip4_multicast_leave_group(br, port, ih->group);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		br_ip4_multicast_leave_group(br, port, ih->group, h_source);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		break;
	}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (br->multicast_upstream)
		BR_INPUT_SKB_CB(skb)->mrouters_only = 0;

#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
out:
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (du_forward != NULL)
		*du_forward = local_du_fwd;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
	__skb_push(skb2, offset);
err_out:
	if (skb2 != skb)
		kfree_skb(skb2);
	return err;
}

#if IS_ENABLED(CONFIG_IPV6)
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_multicast_ipv6_rcv_up(struct net_bridge *br,
				    __u8 type,
				    u8 *local_du_fwd)
{
	int snpok = 0;

	switch (type) {
	case ICMPV6_MGM_REPORT:
	case ICMPV6_MLD2_REPORT:
	case ICMPV6_MGM_QUERY:
	case ICMPV6_MGM_REDUCTION:
	default:
		goto skip_snooping;
	}

	snpok = 1;
skip_snooping:
	return snpok;
}

static int br_multicast_ipv6_rcv_dl(struct net_bridge *br,
				    __u8 type,
				    u8 *local_du_fwd)
{
	int snpok = 0;

	switch (type) {
	case ICMPV6_MLD2_REPORT:
		break;
	case ICMPV6_MGM_REPORT:
	case ICMPV6_MGM_REDUCTION:
		if (br->multicast_mldv1_disabled)
			goto skip_snooping;
		break;
	case ICMPV6_MGM_QUERY:
	default:
		goto skip_snooping;
	}

	snpok = 1;
skip_snooping:
	return snpok;
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
static int br_multicast_ipv6_rcv(struct net_bridge *br,
				 struct net_bridge_port *port,
				 struct sk_buff *skb)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
static int br_multicast_ipv6_rcv(struct net_bridge *br,
				 struct net_bridge_port *port,
				 struct sk_buff *skb,
				 u8 *du_forward)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	unsigned char *h_source = eth_hdr(skb)->h_source;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
	struct sk_buff *skb2;
	const struct ipv6hdr *ip6h;
	u8 icmp6_type;
	u8 nexthdr;
	__be16 frag_off;
	unsigned len;
	int offset;
	int err;
#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	int snp_check;
	u8 local_du_fwd = BR_DUFW_DEFAULT;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	if (!pskb_may_pull(skb, sizeof(*ip6h)))
		return -EINVAL;

	ip6h = ipv6_hdr(skb);

	/*
	 * We're interested in MLD messages only.
	 *  - Version is 6
	 *  - MLD has always Router Alert hop-by-hop option
	 *  - But we do not support jumbrograms.
	 */
	if (ip6h->version != 6 ||
	    ip6h->nexthdr != IPPROTO_HOPOPTS ||
	    ip6h->payload_len == 0)
		return 0;

	len = ntohs(ip6h->payload_len) + sizeof(*ip6h);
	if (skb->len < len)
		return -EINVAL;

	nexthdr = ip6h->nexthdr;
	offset = ipv6_skip_exthdr(skb, sizeof(*ip6h), &nexthdr, &frag_off);

	if (offset < 0 || nexthdr != IPPROTO_ICMPV6)
		return 0;

	/* Okay, we found ICMPv6 header */
	skb2 = skb_clone(skb, GFP_ATOMIC);
	if (!skb2)
		return -ENOMEM;

	err = -EINVAL;
	if (!pskb_may_pull(skb2, offset + sizeof(struct icmp6hdr)))
		goto out;

	len -= offset - skb_network_offset(skb2);

	__skb_pull(skb2, offset);
	skb_reset_transport_header(skb2);
	skb_postpull_rcsum(skb2, skb_network_header(skb2),
			   skb_network_header_len(skb2));

	icmp6_type = icmp6_hdr(skb2)->icmp6_type;

	switch (icmp6_type) {
	case ICMPV6_MGM_QUERY:
	case ICMPV6_MGM_REPORT:
	case ICMPV6_MGM_REDUCTION:
	case ICMPV6_MLD2_REPORT:
		break;
	default:
		err = 0;
		goto out;
	}

	/* Okay, we found MLD message. Check further. */
	if (skb2->len > len) {
		err = pskb_trim_rcsum(skb2, len);
		if (err)
			goto out;
		err = -EINVAL;
	}

	ip6h = ipv6_hdr(skb2);

	switch (skb2->ip_summed) {
	case CHECKSUM_COMPLETE:
		if (!csum_ipv6_magic(&ip6h->saddr, &ip6h->daddr, skb2->len,
					IPPROTO_ICMPV6, skb2->csum))
			break;
		/*FALLTHROUGH*/
	case CHECKSUM_NONE:
		skb2->csum = ~csum_unfold(csum_ipv6_magic(&ip6h->saddr,
							&ip6h->daddr,
							skb2->len,
							IPPROTO_ICMPV6, 0));
		if (__skb_checksum_complete(skb2))
			goto out;
	}

	err = 0;

	BR_INPUT_SKB_CB(skb)->igmp = 1;

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (port && br->multicast_upstream) {
		if (port->multicast_upstream)
			snp_check = br_multicast_ipv6_rcv_up(br, icmp6_type,
							     &local_du_fwd);
		else
			snp_check = br_multicast_ipv6_rcv_dl(br, icmp6_type,
							     &local_du_fwd);
		if (!snp_check)
			goto out;
	}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	switch (icmp6_type) {
	case ICMPV6_MGM_REPORT:
	    {
		struct mld_msg *mld;
		if (!pskb_may_pull(skb2, sizeof(*mld))) {
			err = -EINVAL;
			goto out;
		}
		mld = (struct mld_msg *)skb_transport_header(skb2);
		BR_INPUT_SKB_CB(skb)->mrouters_only = 1;
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		err = br_ip6_multicast_add_group(br, port, &mld->mld_mca);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		if (br->multicast_mldv1_disabled)
			break;
		err = br_ip6_multicast_add_group(br, port, &mld->mld_mca,
						 0, NULL, h_source);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		break;
	    }
	case ICMPV6_MLD2_REPORT:
		err = br_ip6_multicast_mld2_report(br, port, skb2);
		break;
	case ICMPV6_MGM_QUERY:
		err = br_ip6_multicast_query(br, port, skb2);
		break;
	case ICMPV6_MGM_REDUCTION:
	    {
		struct mld_msg *mld;
		if (!pskb_may_pull(skb2, sizeof(*mld))) {
			err = -EINVAL;
			goto out;
		}
		mld = (struct mld_msg *)skb_transport_header(skb2);
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		br_ip6_multicast_leave_group(br, port, &mld->mld_mca);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		if (br->multicast_mldv1_disabled)
			break;
		br_ip6_multicast_leave_group(br, port, &mld->mld_mca,
					     0, NULL, h_source);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	    }
	}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (br->multicast_upstream)
		BR_INPUT_SKB_CB(skb)->mrouters_only = 0;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

out:
	kfree_skb(skb2);
	return err;
}
#endif

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
int br_multicast_rcv(struct net_bridge *br, struct net_bridge_port *port,
		     struct sk_buff *skb)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
int br_multicast_rcv(struct net_bridge *br, struct net_bridge_port *port,
		     struct sk_buff *skb, u8 *du_forward)
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
{
	BR_INPUT_SKB_CB(skb)->igmp = 0;
	BR_INPUT_SKB_CB(skb)->mrouters_only = 0;

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (br->multicast_disabled)
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	if (br_multicast_disabled(br))
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		return 0;

	switch (skb->protocol) {
	case htons(ETH_P_IP):
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		return br_multicast_ipv4_rcv(br, port, skb);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		if (br_multicast_mld_only(br))
			return 0;
		return br_multicast_ipv4_rcv(br, port, skb, du_forward);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
		return br_multicast_ipv6_rcv(br, port, skb);
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
		return br_multicast_ipv6_rcv(br, port, skb, du_forward);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
#endif
	}

	return 0;
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
int br_multicast_srcent_check(struct net_bridge_port_group *p,
			      struct sk_buff *skb)
{
	struct net_bridge_port_srcent *s;
	struct br_ip ipsrc, dummy_src;

	memset(&dummy_src, 0, sizeof(struct br_ip));

	if ((s = rcu_dereference(p->inlist)) == NULL)
		goto pass;

	dummy_src.proto = skb->protocol;
	ipsrc.proto = skb->protocol;
	switch (skb->protocol) {
	case htons(ETH_P_IP):
		ipsrc.u.ip4 = ip_hdr(skb)->daddr;
		break;
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
		ipsrc.u.ip6 = ipv6_hdr(skb)->saddr;
		break;
#endif
	default:
		goto pass;
	}

	for (s = rcu_dereference(p->inlist);
	     s != NULL;
	     s = rcu_dereference(s->next)) {
		if (br_ip_equal(&s->addr, &dummy_src))
			goto pass;
	}

	for (s = rcu_dereference(p->inlist);
	     s != NULL;
	     s = rcu_dereference(s->next)) {
		if (br_ip_equal(&s->addr, &ipsrc))
			goto pass;
	}

	return 1;
pass:
	return 0;
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

static void br_multicast_query_expired(unsigned long data)
{
	struct net_bridge *br = (void *)data;

	spin_lock(&br->multicast_lock);
	if (br->multicast_startup_queries_sent <
	    br->multicast_startup_query_count)
		br->multicast_startup_queries_sent++;

	br_multicast_send_query(br, NULL, br->multicast_startup_queries_sent);

	spin_unlock(&br->multicast_lock);
}

void br_multicast_init(struct net_bridge *br)
{
	br->hash_elasticity = 4;
	br->hash_max = 512;

	br->multicast_router = 1;
	br->multicast_last_member_count = 2;
	br->multicast_startup_query_count = 2;

	br->multicast_last_member_interval = HZ;
	br->multicast_query_response_interval = 10 * HZ;
	br->multicast_startup_query_interval = 125 * HZ / 4;
	br->multicast_query_interval = 125 * HZ;
	br->multicast_querier_interval = 255 * HZ;
#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	br->multicast_membership_interval = 260 * HZ;
#else  /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
	br->multicast_membership_interval = 500 * HZ;

	br->multicast_disabled = 1;
	br->multicast_mldv1_disabled = 1;
	br->multicast_igmpv1_disabled = 1;
	br->multicast_igmpv3_disabled = 1;
	br->multicast_upstream = 1;
	br->multicast_addrcheck = 1;
	br->multicast_flooding_ipv4 = 1;
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */

	spin_lock_init(&br->multicast_lock);
	setup_timer(&br->multicast_router_timer,
		    br_multicast_local_router_expired, 0);
	setup_timer(&br->multicast_querier_timer,
		    br_multicast_local_router_expired, 0);
	setup_timer(&br->multicast_query_timer, br_multicast_query_expired,
		    (unsigned long)br);
}

void br_multicast_open(struct net_bridge *br)
{
	br->multicast_startup_queries_sent = 0;

	if (br->multicast_disabled)
		return;

#ifndef USE_NECPF_BRIDGE_SSIS_SSM2S
	mod_timer(&br->multicast_query_timer, jiffies);
#endif /* !USE_NECPF_BRIDGE_SSIS_SSM2S */
}

void br_multicast_stop(struct net_bridge *br)
{
	struct net_bridge_mdb_htable *mdb;
	struct net_bridge_mdb_entry *mp;
	struct hlist_node *p, *n;
	u32 ver;
	int i;

	del_timer_sync(&br->multicast_router_timer);
	del_timer_sync(&br->multicast_querier_timer);
	del_timer_sync(&br->multicast_query_timer);

	spin_lock_bh(&br->multicast_lock);
	mdb = mlock_dereference(br->mdb, br);
	if (!mdb)
		goto out;

	br->mdb = NULL;

	ver = mdb->ver;
	for (i = 0; i < mdb->max; i++) {
		hlist_for_each_entry_safe(mp, p, n, &mdb->mhash[i],
					  hlist[ver]) {
			del_timer(&mp->timer);
			call_rcu_bh(&mp->rcu, br_multicast_free_group);
		}
	}

	if (mdb->old) {
		spin_unlock_bh(&br->multicast_lock);
		rcu_barrier_bh();
		spin_lock_bh(&br->multicast_lock);
		WARN_ON(mdb->old);
	}

	mdb->old = mdb;
	call_rcu_bh(&mdb->rcu, br_mdb_free);

out:
	spin_unlock_bh(&br->multicast_lock);
}

int br_multicast_set_router(struct net_bridge *br, unsigned long val)
{
	int err = -ENOENT;

	spin_lock_bh(&br->multicast_lock);
	if (!netif_running(br->dev))
		goto unlock;

	switch (val) {
	case 0:
	case 2:
		del_timer(&br->multicast_router_timer);
		/* fall through */
	case 1:
		br->multicast_router = val;
		err = 0;
		break;

	default:
		err = -EINVAL;
		break;
	}

unlock:
	spin_unlock_bh(&br->multicast_lock);

	return err;
}

int br_multicast_set_port_router(struct net_bridge_port *p, unsigned long val)
{
	struct net_bridge *br = p->br;
	int err = -ENOENT;

	spin_lock(&br->multicast_lock);
	if (!netif_running(br->dev) || p->state == BR_STATE_DISABLED)
		goto unlock;

	switch (val) {
	case 0:
	case 1:
	case 2:
		p->multicast_router = val;
		err = 0;

		if (val < 2 && !hlist_unhashed(&p->rlist))
			hlist_del_init_rcu(&p->rlist);

		if (val == 1)
			break;

		del_timer(&p->multicast_router_timer);

		if (val == 0)
			break;

		br_multicast_add_router(br, p);
		break;

	default:
		err = -EINVAL;
		break;
	}

unlock:
	spin_unlock(&br->multicast_lock);

	return err;
}

int br_multicast_toggle(struct net_bridge *br, unsigned long val)
{
	struct net_bridge_port *port;
	int err = 0;
	struct net_bridge_mdb_htable *mdb;

	spin_lock_bh(&br->multicast_lock);
	if (br->multicast_disabled == !val)
		goto unlock;

	br->multicast_disabled = !val;
	if (br->multicast_disabled)
		goto unlock;

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
	if (br->multicast_mld_only)
		goto unlock;
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

	if (!netif_running(br->dev))
		goto unlock;

	mdb = mlock_dereference(br->mdb, br);
	if (mdb) {
		if (mdb->old) {
			err = -EEXIST;
rollback:
			br->multicast_disabled = !!val;
			goto unlock;
		}

		err = br_mdb_rehash(&br->mdb, mdb->max,
				    br->hash_elasticity);
		if (err)
			goto rollback;
	}

	br_multicast_open(br);
	list_for_each_entry(port, &br->port_list, list) {
		if (port->state == BR_STATE_DISABLED ||
		    port->state == BR_STATE_BLOCKING)
			continue;

		__br_multicast_enable_port(port);
	}

unlock:
	spin_unlock_bh(&br->multicast_lock);

	return err;
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
int br_multicast_mld_only_toggle(struct net_bridge *br, unsigned long val)
{
	struct net_bridge_port *port;
	int err = 0;
	struct net_bridge_mdb_htable *mdb;

	spin_lock_bh(&br->multicast_lock);
	if (br->multicast_mld_only == val)
		goto unlock;

	br->multicast_mld_only = val;
	if (!br->multicast_mld_only)
		goto unlock;

	if (!br->multicast_disabled)
		goto unlock;

	if (!netif_running(br->dev))
		goto unlock;

	mdb = mlock_dereference(br->mdb, br);
	if (mdb) {
		if (mdb->old) {
			err = -EEXIST;
rollback:
			br->multicast_mld_only = !val;
			goto unlock;
		}

		err = br_mdb_rehash(&br->mdb, mdb->max,
				    br->hash_elasticity);
		if (err)
			goto rollback;
	}

	br_multicast_open(br);
	list_for_each_entry(port, &br->port_list, list) {
		if (port->state == BR_STATE_DISABLED ||
		    port->state == BR_STATE_BLOCKING)
			continue;

		__br_multicast_enable_port(port);
	}

unlock:
	spin_unlock_bh(&br->multicast_lock);

	return err;
}

static struct net_bridge_port_hostent *br_multicast_get_hostent(
	const struct net_bridge_port *port,
	struct br_ip *group,
	struct br_ip *src)
{
	struct net_bridge_port_hostent *retlist = NULL;
	struct net_bridge *br;
	struct net_bridge_mdb_htable *mdb;
	struct hlist_node *n;
	struct net_bridge_mdb_entry *mp = NULL;
	struct net_bridge_port_group *pg;
	struct net_bridge_port_srcent *se;
	struct br_ip dummy_src;
	int hash;

	memset(&dummy_src, 0, sizeof(struct br_ip));
	dummy_src.proto = src->proto;

	if (port == NULL)
		goto out;

	br = port->br;
	if (br_multicast_disabled(br))
		goto out;

	mdb = rcu_dereference(br->mdb);
	if (mdb == NULL)
		goto out;

	hash = br_ip_hash(mdb, group);
	hlist_for_each_entry_rcu(mp, n, &mdb->mhash[hash], hlist[mdb->ver]) {
		if (br_ip_equal(&mp->addr, group))
			break;
	}
	if (mp == NULL)
		goto out;

	for (pg = rcu_dereference(mp->ports);
	     pg != NULL;
	     pg = rcu_dereference(pg->next)) {
		if (pg->port == port)
			break;
	}
	if (pg == NULL)
		goto out;

	for (se = rcu_dereference(pg->inlist);
	     se != NULL;
	     se = rcu_dereference(se->next)) {
		if (br_ip_equal(&se->addr, &dummy_src)) {
			retlist = rcu_dereference(se->hlist);
			goto out;
		}
	}
	for (se = rcu_dereference(pg->inlist);
	     se != NULL;
	     se = rcu_dereference(se->next)) {
		if (br_ip_equal(&se->addr, src)) {
			retlist = rcu_dereference(se->hlist);
			goto out;
		}
	}
out:
	return retlist;
}

struct net_bridge_port_hostent *br_multicast_get_hostent_from_skb(
	struct sk_buff * skb)
{
	struct net_bridge_port_hostent *retlist = NULL;
	const struct net_bridge_port *port;
	struct br_ip group, src;

	memset(&group, 0, sizeof(struct br_ip));
	memset(&src, 0, sizeof(struct br_ip));

	if (skb == NULL ||
	    skb->dev == NULL || skb->network_header == NULL)
		goto out;

	port = br_port_get_rcu(skb->dev);

	switch (skb->protocol) {
	case htons(ETH_P_IP):
		if (!pskb_may_pull(skb, sizeof(struct iphdr)) ||
		    !pskb_may_pull(skb, ip_hdrlen(skb)))
			goto out;

		group.u.ip4 = ip_hdr(skb)->daddr;
		src.u.ip4 = ip_hdr(skb)->saddr;
		group.proto = src.proto = htons(ETH_P_IP);
		break;
#if IS_ENABLED(CONFIG_IPV6)
	case htons(ETH_P_IPV6):
		if (!pskb_may_pull(skb, sizeof(struct ipv6hdr)))
			goto out;

		group.u.ip6 = ipv6_hdr(skb)->daddr;
		src.u.ip6 = ipv6_hdr(skb)->saddr;
		group.proto = src.proto = htons(ETH_P_IPV6);
		break;
#endif
	default:
		goto out;
	}

	retlist = br_multicast_get_hostent(port, &group, &src);
out:
	return retlist;
}

EXPORT_SYMBOL_GPL(br_multicast_get_hostent_from_skb);
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */

int br_multicast_set_hash_max(struct net_bridge *br, unsigned long val)
{
	int err = -ENOENT;
	u32 old;
	struct net_bridge_mdb_htable *mdb;

	spin_lock_bh(&br->multicast_lock);
	if (!netif_running(br->dev))
		goto unlock;

	err = -EINVAL;
	if (!is_power_of_2(val))
		goto unlock;

	mdb = mlock_dereference(br->mdb, br);
	if (mdb && val < mdb->size)
		goto unlock;

	err = 0;

	old = br->hash_max;
	br->hash_max = val;

	if (mdb) {
		if (mdb->old) {
			err = -EEXIST;
rollback:
			br->hash_max = old;
			goto unlock;
		}

		err = br_mdb_rehash(&br->mdb, br->hash_max,
				    br->hash_elasticity);
		if (err)
			goto rollback;
	}

unlock:
	spin_unlock_bh(&br->multicast_lock);

	return err;
}

#ifdef USE_NECPF_BRIDGE_SSIS_SSM2S
#define BR_MULTICAST_DBGBUFF_ERRMSG	30
#define BR_MULTICAST_DBGBUFF_SIZE	(PAGE_SIZE - BR_MULTICAST_DBGBUFF_ERRMSG)
char lbuff[BR_MULTICAST_DBGBUFF_SIZE+BR_MULTICAST_DBGBUFF_ERRMSG];

static int br_multicast_show_rlist(struct net_bridge *br,
				   char *buff,
				   int *buff_len)
{
	char tmpbuff[255];
	struct net_bridge_port *port;
	struct hlist_node *node;

	for (node = rcu_dereference(hlist_first_rcu(&br->router_list));
	     node != NULL;
	     node = rcu_dereference(hlist_next_rcu(node))) {
		memset(tmpbuff, 0, sizeof(tmpbuff));
		port = hlist_entry(node, struct net_bridge_port, rlist);
		if ( port == NULL)
			continue;

		sprintf(tmpbuff, "  %s: timer=%lu",
			port->sysfs_name,
			(br_timer_value(&port->multicast_router_timer))/100);
		if (port->multicast_router == 2)
			strcat(tmpbuff, "(permanent)\n");
		else
			strcat(tmpbuff, "\n");

		*buff_len += strlen(tmpbuff);
		if (*buff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
			return -1;
		strcat(buff, tmpbuff);
	}
	return 0;
}

static int br_multicast_show_hostent(struct net_bridge_port_srcent *ps,
				     char *buff,
				     int *buff_len)
{
	char tmpbuff[255];
	char addr_str[64];
	struct net_bridge_port_hostent *ph;

	for (ph = rcu_dereference(ps->hlist);
	     ph != NULL;
	     ph = rcu_dereference(ph->next)) {
		memset(addr_str, 0, sizeof(addr_str));
		memset(tmpbuff, 0, sizeof(tmpbuff));
		sprintf(addr_str, "%02x:%02x:%02x:%02x:%02x:%02x",
			ph->haddr[0], ph->haddr[1], ph->haddr[2],
			ph->haddr[3], ph->haddr[4], ph->haddr[5]);
		sprintf(tmpbuff, "        HOST: %s timer=%lu\n",
			addr_str,
			(br_timer_value(&ph->timer))/100);
		*buff_len += strlen(tmpbuff);
		if (*buff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
			return -1;
		strcat(buff, tmpbuff);
	}
	return 0;
}

static int br_multicast_show_srcent(struct net_bridge_port_group *pg,
				    char *buff,
				    int *buff_len)
{
	char tmpbuff[255];
	char addr_str[64];
	struct br_ip *br_group;
	struct br_ip br_dummy;
	struct net_bridge_port_srcent *ps;

	memset(&br_dummy, 0, sizeof(struct br_ip));

	for (ps = rcu_dereference(pg->inlist);
	     ps != NULL;
	     ps = rcu_dereference(ps->next)) {
		memset(addr_str, 0, sizeof(addr_str));
		memset(tmpbuff, 0, sizeof(tmpbuff));
		br_group = &ps->addr;
		br_dummy.proto = br_group->proto;
		if (br_ip_equal(br_group, &br_dummy)) {
			/* Nothing */
		} else {
			switch(br_group->proto) {
			case htons(ETH_P_IP):
				sprintf(addr_str, "%pI4",
					&br_group->u.ip4);
				break;
#if IS_ENABLED(CONFIG_IPV6)
			case htons(ETH_P_IPV6):
				sprintf(addr_str, "%pI6",
					&br_group->u.ip6.s6_addr);
				break;
#endif
			default:
				continue;
			}

			sprintf(tmpbuff, "      SRC: %s\n", addr_str);
			*buff_len += strlen(tmpbuff);
			if (*buff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
				return -1;
			strcat(buff, tmpbuff);
		}

		if (br_multicast_show_hostent(ps, buff, buff_len) != 0)
			return -1;
	}
	return 0;
}

static int br_multicast_show_port(struct net_bridge_mdb_entry *mp,
				  char *buff,
				  int *buff_len)
{
	char tmpbuff[255];
	char addr_str[64];
	struct net_bridge_port_group *pg;
	struct net_bridge_port *port;

	for (pg = rcu_dereference(mp->ports);
	     pg != NULL;
	     pg = rcu_dereference(pg->next)) {
		memset(addr_str, 0, sizeof(addr_str));
		memset(tmpbuff, 0, sizeof(tmpbuff));
		port = pg->port;
		sprintf(tmpbuff, "    %s: timer=%lu\n",
			port->sysfs_name,
			(br_timer_value(&pg->timer))/100);
		*buff_len += strlen(tmpbuff);
		if (*buff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
			return -1;
		strcat(buff, tmpbuff);
		if (br_multicast_show_srcent(pg, buff, buff_len) != 0)
			return -1;
	}
	return 0;
}

static int br_multicast_show_group(struct net_bridge_mdb_htable *mdb,
				   int hashvalue,
				   char *buff,
				   int *buff_len)
{
	char tmpbuff[255];
	char addr_str[64];
	struct net_bridge_mdb_entry *mp;
	struct hlist_node *node;
	struct br_ip *br_group;

	hlist_for_each_entry_rcu(mp, node, &mdb->mhash[hashvalue], hlist[mdb->ver]) {
		memset(addr_str, 0, sizeof(addr_str));
		memset(tmpbuff, 0, sizeof(tmpbuff));
		br_group = &mp->addr;
		switch(br_group->proto) {
		case htons(ETH_P_IP):
			sprintf(addr_str, "%pI4", &br_group->u.ip4);
			break;
#if IS_ENABLED(CONFIG_IPV6)
		case htons(ETH_P_IPV6):
			sprintf(addr_str, "%pI6", &br_group->u.ip6.s6_addr);
			break;
#endif
		default:
			continue;
		}

		sprintf(tmpbuff, "  group %s\n", addr_str);
		*buff_len += strlen(tmpbuff);
		if (*buff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
			return -1;
		strcat(buff, tmpbuff);

		if (br_multicast_show_port(mp, buff, buff_len) != 0)
			return -1;
	}
	return 0;
}

int br_multicast_show_snooping_entry(struct net_bridge *br, char *buff)
{
	int lbuff_len = 0;
	struct net_bridge_mdb_htable *mdb;
	int hashvalue;

	memset(lbuff, 0, sizeof(lbuff));

	rcu_read_lock();
	lbuff_len += strlen("Snooping Entry:\n");
	if (lbuff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
		goto out;
	strcat(lbuff, "Snooping Entry:\n");

	mdb = rcu_dereference(br->mdb);
	if (!mdb) {
		goto router;
	}

	for (hashvalue = 0; hashvalue < mdb->max; hashvalue++) {
		if (br_multicast_show_group(mdb, hashvalue,
					    lbuff, &lbuff_len) != 0)
			goto out;
	}

router:
	lbuff_len += strlen("\nRouterList Entry:\n");
	if (lbuff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
		goto out;
	strcat(lbuff, "\nRouterList Entry:\n");

	if (br_multicast_show_rlist(br, lbuff, &lbuff_len) != 0)
		goto out;

out:
	if (lbuff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
		strcat(lbuff, "### buffer overflow END ###\n");
	rcu_read_unlock();
	return sprintf(buff, "%s\n", lbuff);
}

static int br_multicast_show_config(struct net_bridge *br,
				    char *buff,
				    int *buff_len)
{
	char tmpbuff[255];
	struct net_bridge_port *port;

	memset(tmpbuff, 0, sizeof(tmpbuff));

	sprintf(tmpbuff, "  IGMP/MLD Function: %s\n",
			br->multicast_disabled ? "Disable" : "Enable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	sprintf(tmpbuff, "  MLD Only Function: %s\n",
			br->multicast_mld_only ? "Enable" : "Disable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	sprintf(tmpbuff, "    IGMP Ver: v1(%s) v2(%s) v3(%s)\n",
			br->multicast_igmpv1_disabled ? "Disable" : "Enable",
			"Enable",
			br->multicast_igmpv3_disabled ? "Disable" : "Enable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	sprintf(tmpbuff, "    MLD  Ver: v1(%s) v2(%s)\n",
			br->multicast_mldv1_disabled ? "Disable" : "Enable",
			"Enable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	sprintf(tmpbuff, "  Flooding: IPv4(%s) IPv6(%s)\n",
			br->multicast_flooding_ipv4 ? "Enable" : "Disable",
			"Enable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	sprintf(tmpbuff, "  Address Check: %s\n",
			br->multicast_addrcheck ? "Enable" : "Disable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	sprintf(tmpbuff, "  Up Stream IF: %s",
			br->multicast_upstream ? "Enable": "Disable");
	*buff_len += strlen(tmpbuff);
	strcat(buff, tmpbuff);

	memset(tmpbuff, 0, sizeof(tmpbuff));
	list_for_each_entry_rcu(port, &br->port_list, list) {
		if (!port->multicast_upstream)
			continue;
		sprintf(tmpbuff, " (%s)", port->sysfs_name);
		*buff_len += strlen(tmpbuff);
		strcat(buff, tmpbuff);
	}
	*buff_len += strlen("\n");
	strcat(buff, "\n");

	return 0;
}

int br_multicast_show_debug(struct net_bridge *br, char *buff)
{
	int lbuff_len = 0;

	memset(lbuff, 0, sizeof(lbuff));

	rcu_read_lock();
	lbuff_len += strlen("Snooping Config:\n");
	if (lbuff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
		goto out;
	strcat(lbuff, "Snooping Config:\n");
	if (br_multicast_show_config(br, lbuff, &lbuff_len) != 0)
		goto out;

out:
	if (lbuff_len >= BR_MULTICAST_DBGBUFF_SIZE-1)
		strcat(lbuff, "### buffer overflow END ###\n");
	rcu_read_unlock();
	return sprintf(buff, "%s\n", lbuff);
}
#endif /* USE_NECPF_BRIDGE_SSIS_SSM2S */
