/***********************************************************************
*
* plugin.c
*
* pppd plugin for kernel-mode PPPoE on Linux
*
* Copyright (C) 2001 by Roaring Penguin Software Inc., Michal Ostrowski
* and Jamal Hadi Salim.
*
* Much code and many ideas derived from pppoe plugin by Michal
* Ostrowski and Jamal Hadi Salim, which carries this copyright:
*
* Copyright 2000 Michal Ostrowski <mostrows@styx.uwaterloo.ca>,
*                Jamal Hadi Salim <hadi@cyberus.ca>
* Borrows heavily from the PPPoATM plugin by Mitchell Blank Jr.,
* which is based in part on work from Jens Axboe and Paul Mackerras.
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version
* 2 of the License, or (at your option) any later version.
*
***********************************************************************/


#define _GNU_SOURCE 1
#include "pppoe.h"

/* NECAT(2013.12.10) -> */

//#include "pppd/pppd.h"

#include <stdio.h>
#include <limits.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/time.h>


typedef unsigned char bool;

enum opt_type {
        o_special_noarg = 0,
        o_special = 1,
        o_bool,
        o_int,
        o_uint32,
        o_string,
        o_wild
};

typedef struct {
        char    *name;
        enum opt_type type;
        void    *addr;
        char    *description;
        unsigned int flags;
        void    *addr2;
        int     upper_limit;
        int     lower_limit;
        const char *source;
        short int priority;
        short int winner;
} option_t;

#define MAX_ENDP_LEN    20
struct epdisc {
	unsigned char       class;
	unsigned char       length;
	unsigned char       value[MAX_ENDP_LEN];
};

struct channel {
        option_t *options;
        void (*process_extra_options) __P((void));
        void (*check_options) __P((void));
        int  (*connect) __P((void));
        void (*disconnect) __P((void));
        int  (*establish_ppp) __P((int));
        void (*disestablish_ppp) __P((int));
        void (*send_config) __P((int, u_int32_t, int, int));
        void (*recv_config) __P((int, u_int32_t, int, int));
        void (*cleanup) __P((void));
        void (*close) __P((void));
};

#define MAXNAMELEN      256
#define OPT_INITONLY    0x4000000 
#define OPT_DEVEQUIV    0x8000000 
#define OPT_DEVNAM      (OPT_INITONLY | OPT_DEVEQUIV)
#define OPT_PRIVFIX     0x2000000
#define OPT_NOARG       0x200
#define OPT_A2STRVAL    0x20000000
#define OPT_STATIC      0x2000


//#include "pppd/fsm.h"

//#include "pppd/lcp.h"

typedef struct lcp_options {
	bool passive;
	bool silent;
	bool restart;
	bool neg_mru;
	bool neg_asyncmap;
	bool neg_upap;
	bool neg_chap;
	bool neg_eap;
	bool neg_magicnumber;
	bool neg_pcompression;
	bool neg_accompression;
	bool neg_lqr;
	bool neg_cbcp;
	bool neg_mrru;
	bool neg_ssnhf;
	bool neg_endpoint;
	int  mru;
	int  mrru;
	u_char chap_mdtype;
	u_int32_t asyncmap;
	u_int32_t magicnumber;
	int  numloops;
	u_int32_t lqr_period;
	struct epdisc endpoint;
} lcp_options;

extern lcp_options lcp_wantoptions[];
extern lcp_options lcp_allowoptions[];

//#include "pppd/ipcp.h"

typedef struct ipcp_options {
	bool neg_addr;
	bool old_addrs;
	bool req_addr;
	bool default_route;
	bool proxy_arp;
	bool neg_vj;
	bool old_vj;
	bool accept_local;
	bool accept_remote;
	bool req_dns1;
	bool req_dns2;
	int  vj_protocol;
	int  maxslotindex;
	bool cflag;
	u_int32_t ouraddr, hisaddr;
	u_int32_t dnsaddr[2];
	u_int32_t winsaddr[2];
} ipcp_options;

extern ipcp_options ipcp_allowoptions[];
extern ipcp_options ipcp_wantoptions[];

//#include "pppd/ccp.h"

typedef struct ccp_options {
	bool bsd_compress;
	bool deflate;
	bool predictor_1;
	bool predictor_2;
	bool deflate_correct;
	bool deflate_draft;
	bool mppe;
	u_short bsd_bits;
	u_short deflate_size;
	short method;
} ccp_options;

extern ccp_options ccp_wantoptions[];
extern ccp_options ccp_allowoptions[];

/* <- NECAT(2013.12.10) */

#include <linux/types.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <net/ethernet.h>
#include <net/if_arp.h>
/* NECAT(2013.12.10) -> */
//#include <linux/ppp_defs.h>
//#include <linux/if_ppp.h>
/* <- NECAT(2013.12.10) */
#include <linux/if_pppox.h>
#include <limits.h>


#include "plugin.h"




#define	EXIT_OPTION_ERROR	(2)




static char *existingSession = NULL;
static char *pppoe_reqd_mac = NULL;
unsigned char pppoe_reqd_mac_addr[6];

static int PPPoEDevnameHook(char *cmd, char **argv, int doit);
static PPPoEConnection *conn = NULL;



/*
 * this variable is not refered in pppd
 */
int	ppp_session_number;


/* Ethernet I/F name (example : nas10_0) */
char	devnam[NAME_MAX];

/* Ethernet I/F name (example : nas10_0) */
char	gm_ppp_devnam[NAME_MAX];


/*
 * MUST return these parameters to pppd
 */
char	gm_remote_number[NAME_MAX];


/* refered in if.c */
int	debug;




/**********************************************************************
 * %FUNCTION: PPPOEInitDevice
 * %ARGUMENTS:
 * None
 * %RETURNS:
 *
 * %DESCRIPTION:
 * Initializes PPPoE device.
 ***********************************************************************/
static int
PPPOEInitDevice(void)
{
    conn = malloc(sizeof(PPPoEConnection));
    if (!conn) {
	novm("PPPoE session data");
    }
    memset(conn, 0, sizeof(PPPoEConnection));

	/*
	 * paramters overriden by imported values from command line arguments in main.c
	 * - ifName
	 * - acName
	 * - serviceName
	 * - useHostUniq
	 * - printACNames
	 * - discoveryTimeout
	 */

    conn->acName = NULL;
    conn->serviceName = NULL;
    conn->ifName = devnam;
    conn->discoverySocket = -1;
    conn->sessionSocket = -1;
    conn->useHostUniq = 1;
    conn->printACNames = 0;
    conn->discoveryTimeout = PADI_TIMEOUT;

    return 1;
}

/**********************************************************************
 * %FUNCTION: PPPOEConnectDevice
 * %ARGUMENTS:
 * None
 * %RETURNS:
 * Non-negative if all goes well; -1 otherwise
 * %DESCRIPTION:
 * Connects PPPoE device.
 ***********************************************************************/
int
PPPOEConnectDevice(void)
{
    struct sockaddr_pppox sp;

    strncpy(gm_ppp_devnam, devnam, sizeof(gm_ppp_devnam));
    if (existingSession) {
	unsigned int mac[ETH_ALEN];
	int i, ses;
	if (sscanf(existingSession, "%d:%x:%x:%x:%x:%x:%x",
		   &ses, &mac[0], &mac[1], &mac[2],
		   &mac[3], &mac[4], &mac[5]) != 7) {
	    fatal("Illegal value for rp_pppoe_sess option");
	}
	conn->session = htons(ses);
	for (i=0; i<ETH_ALEN; i++) {
	    conn->peerEth[i] = (unsigned char) mac[i];
	}
    } else {
	discovery(conn);
	if (conn->discoveryState != STATE_SESSION) {
	    error("Unable to complete PPPoE Discovery");
	    return -1;
	}
    }

    /* Set PPPoE session-number for further consumption */
    ppp_session_number = ntohs(conn->session);

    /* Make the session socket */
    conn->sessionSocket = socket(AF_PPPOX, SOCK_STREAM, PX_PROTO_OE);
    if (conn->sessionSocket < 0) {
	error("Failed to create PPPoE socket: %m");
	goto errout;
    }
    sp.sa_family = AF_PPPOX;
    sp.sa_protocol = PX_PROTO_OE;
    sp.sa_addr.pppoe.sid = conn->session;
    memcpy(sp.sa_addr.pppoe.dev, conn->ifName, IFNAMSIZ);
    memcpy(sp.sa_addr.pppoe.remote, conn->peerEth, ETH_ALEN);

    /* Set gm_remote_number for ServPoET */
    sprintf(gm_remote_number, "%02X:%02X:%02X:%02X:%02X:%02X",
	    (unsigned) conn->peerEth[0],
	    (unsigned) conn->peerEth[1],
	    (unsigned) conn->peerEth[2],
	    (unsigned) conn->peerEth[3],
	    (unsigned) conn->peerEth[4],
	    (unsigned) conn->peerEth[5]);

    warn("Connected to %02X:%02X:%02X:%02X:%02X:%02X via interface %s",
	 (unsigned) conn->peerEth[0],
	 (unsigned) conn->peerEth[1],
	 (unsigned) conn->peerEth[2],
	 (unsigned) conn->peerEth[3],
	 (unsigned) conn->peerEth[4],
	 (unsigned) conn->peerEth[5],
	 conn->ifName);

    if (connect(conn->sessionSocket, (struct sockaddr *) &sp,
		sizeof(struct sockaddr_pppox)) < 0) {
	error("Failed to connect PPPoE socket: %d %m", errno);
	close(conn->sessionSocket);
	goto errout;
    }

    return conn->sessionSocket;

 errout:
    if (conn->discoverySocket >= 0) {
	sendPADT(conn, NULL);
	close(conn->discoverySocket);
	conn->discoverySocket = -1;
    }
    return -1;
}

static void
PPPOERecvConfig(int mru,
		u_int32_t asyncmap,
		int pcomp,
		int accomp)
{
#if 0 /* broken protocol, but no point harrassing the users I guess... */
    if (mru > MAX_PPPOE_MTU)
	warn("Couldn't increase MRU to %d", mru);
#endif
}

/**********************************************************************
 * %FUNCTION: PPPOEDisconnectDevice
 * %ARGUMENTS:
 * None
 * %RETURNS:
 * Nothing
 * %DESCRIPTION:
 * Disconnects PPPoE device
 ***********************************************************************/
void
PPPOEDisconnectDevice(void)
{
    struct sockaddr_pppox sp;

    sp.sa_family = AF_PPPOX;
    sp.sa_protocol = PX_PROTO_OE;
    sp.sa_addr.pppoe.sid = 0;
    memcpy(sp.sa_addr.pppoe.dev, conn->ifName, IFNAMSIZ);
    memcpy(sp.sa_addr.pppoe.remote, conn->peerEth, ETH_ALEN);
    if (connect(conn->sessionSocket, (struct sockaddr *) &sp,
		sizeof(struct sockaddr_pppox)) < 0)
	error("Failed to disconnect PPPoE socket: %d %m", errno);
    close(conn->sessionSocket);
    /* don't send PADT?? */
    if (conn->discoverySocket >= 0) {
        sendPADT(conn, NULL);
	close(conn->discoverySocket);
        conn->discoverySocket = -1;
    }
}


/**********************************************************************
 * %FUNCTION: PPPoEDevnameHook
 * %ARGUMENTS:
 * cmd -- the command (actually, the device name
 * argv -- argument vector
 * doit -- if non-zero, set device name.  Otherwise, just check if possible
 * %RETURNS:
 * 1 if we will handle this device; 0 otherwise.
 * %DESCRIPTION:
 * Checks if name is a valid interface name; if so, returns 1.  Also
 * sets up devnam (string representation of device).
 ***********************************************************************/
static int
PPPoEPrepareDevice( char*  arg_devname )
{
    int r = 1;
    int fd;
    struct ifreq ifr;

    /* Open a socket */
    if ((fd = socket(PF_PACKET, SOCK_RAW, 0)) < 0) {
	r = 0;
    }

    /* Try getting interface index */
    if (r) {
	strncpy(ifr.ifr_name, arg_devname, sizeof(ifr.ifr_name));
	if (ioctl(fd, SIOCGIFINDEX, &ifr) < 0) {
	    r = 0;
	} else {
	    if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) {
		r = 0;
	    } else {
		if (ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER) {
			error("Interface %s not Ethernet", arg_devname);
			r = 0;
		}
	    }
	}
    }

    /* Close socket */
    close(fd);
    if ( r ) {
	strncpy(devnam, arg_devname, sizeof(devnam));
	PPPOEInitDevice();

	return 1;
    }

    return r;
}

/**********************************************************************
 * %FUNCTION: plugin_init
 * %ARGUMENTS:
 * None
 * %RETURNS:
 * Nothing
 * %DESCRIPTION:
 * Initializes hooks for pppd plugin
 ***********************************************************************/
PPPoEConnection*  pppoe_init( char*  arg_devname )
{
	int	res;

	res = PPPoEPrepareDevice( arg_devname );

	return (res == 1) ? conn : NULL;
}

void pppoe_check_options(void)
{
#if 0	/* NOT_YET */
    unsigned int mac[6];
    int i;

    if (pppoe_reqd_mac != NULL) {
	if (sscanf(pppoe_reqd_mac, "%x:%x:%x:%x:%x:%x",
		   &mac[0], &mac[1], &mac[2], &mac[3],
		   &mac[4], &mac[5]) != 6) {
	    option_error("cannot parse pppoe-mac option value");
	    exit(EXIT_OPTION_ERROR);
	}
	for (i = 0; i < 6; ++i)
	    conn->req_peer_mac[i] = mac[i];
	conn->req_peer = 1;
    }
#endif
}
