/*
 * Copyright (c) 2015 The Linux Foundation. All rights reserved.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * Manage the atheros ethernet PHY.
 *
 * All definitions in this file are operating system independent!
 */

#include <common.h>
#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
#include <asm/arch-ipq806x/gpio.h>
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/
#include <miiphy.h>
#include "athrs17_phy.h"
#include "../../../drivers/net/ipq/ipq_gmac.h"
#ifdef CONFIG_CMD_DEBUG_L2SW
#include "../../../common/cmd_debug_l2sw.h"
#endif /* CONFIG_CMD_DEBUG_L2SW */
/******************************************************************************
 * FUNCTION DESCRIPTION: Read switch internal register.
 *                       Switch internal register is accessed through the
 *                       MDIO interface. MDIO access is only 16 bits wide so
 *                       it needs the two time access to complete the internal
 *                       register access.
 * INPUT               : register address
 * OUTPUT              : Register value
 *
 *****************************************************************************/
static uint32_t
athrs17_reg_read(ipq_gmac_board_cfg_t *gmac_cfg, uint32_t reg_addr)
{
	uint32_t reg_word_addr;
	uint32_t phy_addr, reg_val;
	uint16_t phy_val;
	uint16_t tmp_val;
	uint8_t phy_reg;

	/* change reg_addr to 16-bit word address, 32-bit aligned */
	reg_word_addr = (reg_addr & 0xfffffffc) >> 1;

	/* configure register high address */
	phy_addr = 0x18;
	phy_reg = 0x0;
	phy_val = (uint16_t) ((reg_word_addr >> 8) & 0x1ff);  /* bit16-8 of reg address */
	miiphy_write(gmac_cfg->phy_name, phy_addr, phy_reg, phy_val);

	/*
	 * For some registers such as MIBs, since it is read/clear, we should
	 * read the lower 16-bit register then the higher one
	 */

	/* read register in lower address */
	phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
	phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
	miiphy_read(gmac_cfg->phy_name, phy_addr, phy_reg, &phy_val);

	/* read register in higher address */
	reg_word_addr++;
	phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
	phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
	miiphy_read(gmac_cfg->phy_name, phy_addr, phy_reg, &tmp_val);
	reg_val = (tmp_val << 16 | phy_val);

	return reg_val;
}

/******************************************************************************
 * FUNCTION DESCRIPTION: Write switch internal register.
 *                       Switch internal register is accessed through the
 *                       MDIO interface. MDIO access is only 16 bits wide so
 *                       it needs the two time access to complete the internal
 *                       register access.
 * INPUT               : register address, value to be written
 * OUTPUT              : NONE
 *
 *****************************************************************************/
static void
athrs17_reg_write(ipq_gmac_board_cfg_t *gmac_cfg, uint32_t reg_addr,
		   uint32_t reg_val)
{
	uint32_t reg_word_addr;
	uint32_t phy_addr;
	uint16_t phy_val;
	uint8_t phy_reg;

	/* change reg_addr to 16-bit word address, 32-bit aligned */
	reg_word_addr = (reg_addr & 0xfffffffc) >> 1;

	/* configure register high address */
	phy_addr = 0x18;
	phy_reg = 0x0;
	phy_val = (uint16_t) ((reg_word_addr >> 8) & 0x1ff);  /* bit16-8 of reg address */
	miiphy_write(gmac_cfg->phy_name, phy_addr, phy_reg, phy_val);

	/*
	 * For some registers such as ARL and VLAN, since they include BUSY bit
	 * in lower address, we should write the higher 16-bit register then the
	 * lower one
	 */

	/* read register in higher address */
	reg_word_addr++;
	phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
	phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
	phy_val = (uint16_t) ((reg_val >> 16) & 0xffff);
	miiphy_write(gmac_cfg->phy_name, phy_addr, phy_reg, phy_val);

	/* write register in lower address */
	reg_word_addr--;
	phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
	phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
	phy_val = (uint16_t) (reg_val & 0xffff);
	miiphy_write(gmac_cfg->phy_name, phy_addr, phy_reg, phy_val);
}

/*********************************************************************
 * FUNCTION DESCRIPTION: V-lan configuration given by Switch team
			 Vlan 1:PHY0,1,2,3 and Mac 6 of s17c
			 Vlan 2:PHY4 and Mac 0 of s17c
		(Rotary) Vlan 1:PHY1,2,3,4 and Mac 0 of s17c
			 Vlan 2:PHY0 and Mac 6 of s17c
 * INPUT : NONE
 * OUTPUT: NONE
 *********************************************************************/
void athrs17_vlan_config(ipq_gmac_board_cfg_t *gmac_cfg)
{
#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	athrs17_reg_write(gmac_cfg, S17_P0LOOKUP_CTRL_REG, 0x0014003c);
	athrs17_reg_write(gmac_cfg, S17_P0VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P1LOOKUP_CTRL_REG, 0x00140040);
	athrs17_reg_write(gmac_cfg, S17_P1VLAN_CTRL0_REG, 0x20001);

	athrs17_reg_write(gmac_cfg, S17_P2LOOKUP_CTRL_REG, 0x00140039);
	athrs17_reg_write(gmac_cfg, S17_P2VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P3LOOKUP_CTRL_REG, 0x00140035);
	athrs17_reg_write(gmac_cfg, S17_P3VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P4LOOKUP_CTRL_REG, 0x0014002d);
	athrs17_reg_write(gmac_cfg, S17_P4VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P5LOOKUP_CTRL_REG, 0x0014001d);
	athrs17_reg_write(gmac_cfg, S17_P5VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P6LOOKUP_CTRL_REG, 0x00140002);
	athrs17_reg_write(gmac_cfg, S17_P6VLAN_CTRL0_REG, 0x20001);
#else
	athrs17_reg_write(gmac_cfg, S17_P0LOOKUP_CTRL_REG, 0x00140020);
	athrs17_reg_write(gmac_cfg, S17_P0VLAN_CTRL0_REG, 0x20001);

	athrs17_reg_write(gmac_cfg, S17_P1LOOKUP_CTRL_REG, 0x0014005c);
	athrs17_reg_write(gmac_cfg, S17_P1VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P2LOOKUP_CTRL_REG, 0x0014005a);
	athrs17_reg_write(gmac_cfg, S17_P2VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P3LOOKUP_CTRL_REG, 0x00140056);
	athrs17_reg_write(gmac_cfg, S17_P3VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P4LOOKUP_CTRL_REG, 0x0014004e);
	athrs17_reg_write(gmac_cfg, S17_P4VLAN_CTRL0_REG, 0x10001);

	athrs17_reg_write(gmac_cfg, S17_P5LOOKUP_CTRL_REG, 0x00140001);
	athrs17_reg_write(gmac_cfg, S17_P5VLAN_CTRL0_REG, 0x20001);

	athrs17_reg_write(gmac_cfg, S17_P6LOOKUP_CTRL_REG, 0x0014001e);
	athrs17_reg_write(gmac_cfg, S17_P6VLAN_CTRL0_REG, 0x10001);
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/
	printf("%s ...done\n", __func__);
}

/*******************************************************************
* FUNCTION DESCRIPTION: Reset S17 register
* INPUT: NONE
* OUTPUT: NONE
*******************************************************************/
int athrs17_init_switch(ipq_gmac_board_cfg_t *gmac_cfg)
{
#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF)
	/* hardware reset */
	gpio_set_value(63, 1);
	mdelay(100);
	gpio_set_value(63, 0);
	mdelay(100);
#elif defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	/* hardware reset */
	gpio_set_value(45, 1);
	mdelay(100);
	gpio_set_value(45, 0);
	mdelay(100);
#else
	uint32_t data;
	uint32_t i = 0;

	/* Reset the switch before initialization */
	athrs17_reg_write(gmac_cfg, S17_MASK_CTRL_REG, S17_MASK_CTRL_SOFT_RET);
	do {
		udelay(10);
		data = athrs17_reg_read(gmac_cfg, S17_MASK_CTRL_REG);
	} while (data & S17_MASK_CTRL_SOFT_RET);

	do {
		udelay(10);
		data = athrs17_reg_read(gmac_cfg, S17_GLOBAL_INT0_REG);
		i++;
		if (i == 10)
			return -1;
	} while ((data & S17_GLOBAL_INITIALIZED_STATUS) != S17_GLOBAL_INITIALIZED_STATUS);
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

	return 0;
}

/*********************************************************************
 * FUNCTION DESCRIPTION: Configure S17 register
 * INPUT : NONE
 * OUTPUT: NONE
 *********************************************************************/
void athrs17_reg_init(ipq_gmac_board_cfg_t *gmac_cfg)
{
	uint32_t data;

	data = athrs17_reg_read(gmac_cfg, S17_MAC_PWR_REG) | gmac_cfg->mac_pwr0;
	athrs17_reg_write(gmac_cfg, S17_MAC_PWR_REG, data);

#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	athrs17_reg_write(gmac_cfg, S17_P0PAD_MODE_REG, S17_MAC06_EXCHANGE_EN);

	athrs17_reg_write(gmac_cfg, S17_P6STATUS_REG, (S17_SPEED_1000M | S17_TXMAC_EN |
					     S17_RXMAC_EN |
					     S17_DUPLEX_FULL));
#else
	athrs17_reg_write(gmac_cfg, S17_P0STATUS_REG, (S17_SPEED_1000M | S17_TXMAC_EN |
					     S17_RXMAC_EN | S17_TX_FLOW_EN |
					     S17_RX_FLOW_EN | S17_DUPLEX_FULL));
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	athrs17_reg_write(gmac_cfg, S17_GLOFW_CTRL1_REG, (S17_BROAD_DPALL |
						S17_MULTI_FLOOD_DPALL |
						S17_UNI_FLOOD_DPALL));
#else
	athrs17_reg_write(gmac_cfg, S17_GLOFW_CTRL1_REG, (S17_IGMP_JOIN_LEAVE_DPALL |
						S17_BROAD_DPALL |
						S17_MULTI_FLOOD_DPALL |
						S17_UNI_FLOOD_DPALL));
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

	athrs17_reg_write(gmac_cfg, S17_P5PAD_MODE_REG, S17_MAC0_RGMII_RXCLK_DELAY);

#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	athrs17_reg_write(gmac_cfg, S17_P6PAD_MODE_REG, (S17_MAC6_RGMII_EN | \
		S17_MAC6_RGMII_TXCLK_DELAY | \
		(0x0 << S17_MAC6_RGMII_TXCLK_SHIFT) | \
		(0x0 << S17_MAC6_RGMII_RXCLK_SHIFT)));
#else
	athrs17_reg_write(gmac_cfg, S17_P0PAD_MODE_REG, (S17_MAC0_RGMII_EN | \
		S17_MAC0_RGMII_TXCLK_DELAY | S17_MAC0_RGMII_RXCLK_DELAY | \
		(0x1 << S17_MAC0_RGMII_TXCLK_SHIFT) | \
		(0x3 << S17_MAC0_RGMII_RXCLK_SHIFT)));
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

	printf("%s: complete\n", __func__);
}

/*********************************************************************
 * FUNCTION DESCRIPTION: Configure S17 register
 * INPUT : NONE
 * OUTPUT: NONE
 *********************************************************************/
void athrs17_reg_init_lan(ipq_gmac_board_cfg_t *gmac_cfg)
{
	uint32_t reg_val;

#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	athrs17_reg_write(gmac_cfg, S17_P0STATUS_REG, (S17_SPEED_1000M | S17_TXMAC_EN |
					     S17_RXMAC_EN |
					     S17_DUPLEX_FULL));
#else
	athrs17_reg_write(gmac_cfg, S17_P6STATUS_REG, (S17_SPEED_1000M | S17_TXMAC_EN |
					     S17_RXMAC_EN |
					     S17_DUPLEX_FULL));
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

	reg_val = athrs17_reg_read(gmac_cfg, S17_MAC_PWR_REG) | gmac_cfg->mac_pwr1;
	athrs17_reg_write(gmac_cfg, S17_MAC_PWR_REG, reg_val);

#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
	reg_val = athrs17_reg_read(gmac_cfg, S17_P0PAD_MODE_REG);
	athrs17_reg_write(gmac_cfg, S17_P0PAD_MODE_REG, (reg_val | S17_MAC0_SGMII_EN));
#else
	reg_val = athrs17_reg_read(gmac_cfg, S17_P6PAD_MODE_REG);
	athrs17_reg_write(gmac_cfg, S17_P6PAD_MODE_REG, (reg_val | S17_MAC6_SGMII_EN));
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

	reg_val = athrs17_reg_read(gmac_cfg, S17_PWS_REG);
	athrs17_reg_write(gmac_cfg, S17_PWS_REG, (reg_val |
			   S17c_PWS_SERDES_ANEG_DISABLE));

	athrs17_reg_write(gmac_cfg, S17_SGMII_CTRL_REG,(S17c_SGMII_EN_PLL |
					S17c_SGMII_EN_RX |
					S17c_SGMII_EN_TX |
					S17c_SGMII_EN_SD |
					S17c_SGMII_BW_HIGH |
					S17c_SGMII_SEL_CLK125M |
					S17c_SGMII_TXDR_CTRL_600mV |
					S17c_SGMII_CDR_BW_8 |
					S17c_SGMII_DIS_AUTO_LPI_25M |
					S17c_SGMII_MODE_CTRL_SGMII_PHY |
					S17c_SGMII_PAUSE_SG_TX_EN_25M |
					S17c_SGMII_ASYM_PAUSE_25M |
					S17c_SGMII_PAUSE_25M |
					S17c_SGMII_HALF_DUPLEX_25M |
					S17c_SGMII_FULL_DUPLEX_25M));
}

#if defined(CONFIG_CMD_DEBUG_L2SW)

#define strtoul(n, e, b) simple_strtoul((n), (e), (b))

static void
dump_phy_reg(ipq_gmac_board_cfg_t *gmac_cfg)
{
	int port, reg;
	uint16_t phy_reg_value;

	printf("### PHY ###\n");
	printf("\n%-5s: %-5s%-5s%-5s%-5s%-5s\n",
	       "ADDR", "PHY0", "PHY1", "PHY2", "PHY3", "PHY4");
	printf("-----+-------------------------\n");
	for (reg = 0; reg < 0x20; reg++) {
		printf("%4x : ", reg);
		for (port = 0; port < S17_PHY_MAX; port++) {
			miiphy_read(gmac_cfg->phy_name, port, reg, &phy_reg_value);
			printf("%04x ", phy_reg_value);
		}
		printf("\n");
	}
	printf("\n");
}

static void
athrs17_phy_debug_read(ipq_gmac_board_cfg_t *gmac_cfg,int phy, int reg, uint16_t *data)
{
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_DEBUG_PORT_ADDRESS, reg & 0x3f);
	miiphy_read(gmac_cfg->phy_name, phy, ATHR_DEBUG_PORT_DATA, data);
}

static void
athrs17_phy_debug_write(ipq_gmac_board_cfg_t *gmac_cfg, int phy, int reg, uint16_t data)
{
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_DEBUG_PORT_ADDRESS, reg & 0x3f);
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_DEBUG_PORT_DATA, data);
}

static void
dump_phy_debug_reg(ipq_gmac_board_cfg_t *gmac_cfg)
{
	int port, reg;
	uint16_t phy_reg_value;

	printf("### DEBUG ###\n");
	printf("\n%-5s: %-5s%-5s%-5s%-5s%-5s\n",
	       "ADDR", "PHY0", "PHY1", "PHY2", "PHY3", "PHY4");
	printf("-----+-------------------------\n");
	for (reg = 0; reg < 0x40; reg++) {
		printf("%4x : ", reg);
		for (port = 0; port < S17_PHY_MAX; port++) {
			athrs17_phy_debug_read(gmac_cfg, port, reg, &phy_reg_value);
			printf("%04x ", phy_reg_value);
		}
		printf("\n");
	}
	printf("\n");
}

static void
athrs17_phy_mmd_read(ipq_gmac_board_cfg_t *gmac_cfg,int phy, int mmd, uint16_t reg, uint16_t *data)
{
	mmd &= 0x1f;

	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_CONTROL,
		      S17_MMD_FUNC_ADDR | mmd);
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_ADDRESS_DATA, reg);
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_CONTROL,
		      S17_MMD_FUNC_DATA | mmd);
	miiphy_read(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_ADDRESS_DATA, data);
}

struct reg_tbl_t {
	uint32_t addr;
	uint32_t width;
};

static void
dump_phy_mmd3_reg(ipq_gmac_board_cfg_t *gmac_cfg)
{
	static const struct reg_tbl_t reg_tbl[] = {
		{ 0x0,      0x80     },
		{ 0x8000,   0x80     },
		{ 0,        0        },
	};

	int port, offset;
	const struct reg_tbl_t *reg;
	uint16_t phy_reg_value;

	printf("### PHY MMD3 ###\n");
	printf("\n%-5s: %-5s%-5s%-5s%-5s%-5s\n",
	       "ADDR", "PHY0", "PHY1", "PHY2", "PHY3", "PHY4");
	printf("-----+-------------------------\n");
	for (reg = reg_tbl; reg->width != 0; reg++) {
		for (offset = 0; offset < reg->width; offset++) {
			printf("%4x : ", reg->addr + offset);
			for (port = 0; port < S17_PHY_MAX; port++) {
				athrs17_phy_mmd_read(gmac_cfg, port, 3, reg->addr + offset,
						     &phy_reg_value);
				printf("%04x ", phy_reg_value);
			}
			printf("\n");
		}
	}
	printf("\n");
}

static void
dump_phy_mmd7_reg(ipq_gmac_board_cfg_t *gmac_cfg)
{
	static const struct reg_tbl_t reg_tbl[] = {
		{ 0x0,      0x80     },
		{ 0x8000,   0x80     },
		{ 0,        0        },
	};

	int port, offset;
	const struct reg_tbl_t *reg;
	uint16_t phy_reg_value;

	printf("### PHY MMD7 ###\n");
	printf("\n%-5s: %-5s%-5s%-5s%-5s%-5s\n",
	       "ADDR", "PHY0", "PHY1", "PHY2", "PHY3", "PHY4");
	printf("-----+-------------------------\n");
	for (reg = reg_tbl; reg->width != 0; reg++) {
		for (offset = 0; offset < reg->width; offset++) {
			printf("%4x : ", reg->addr + offset);
			for (port = 0; port < S17_PHY_MAX; port++) {
				athrs17_phy_mmd_read(gmac_cfg, port, 7, reg->addr + offset,
						     &phy_reg_value);
				printf("%04x ", phy_reg_value);
			}
			printf("\n");
		}
	}
	printf("\n");
}

static void
dump_l2sw_reg(ipq_gmac_board_cfg_t *gmac_cfg)
{
	static const struct reg_tbl_t reg_tbl[] = {
		{ 0x0,      0xe4     },
		{ 0x100,    0x6c     },
		{ 0x200,    0x74     },
		{ 0x400,    0x58     },
		{ 0x600,    0x11c    },
		{ 0x800,    0x374    },
		{ 0xc00,    0x84     },
		{ 0,        0        },
	};

	int offset;
	const struct reg_tbl_t *reg;
	uint32_t l2sw_reg_value;

	printf("### L2SW ###\n");
	printf("\n%-5s: %-9s%-9s%-9s%-9s\n",
	       "ADDR", "+0", "+4", "+8", "+c");
	printf("-----+------------------------------------");
	for (reg = reg_tbl; reg->width != 0; reg++) {
		for (offset = 0; offset < reg->width; offset += 0x04) {
			if ((offset % 0x10) == 0) {
				printf("\n%4x : ", reg->addr + offset);
			}
			l2sw_reg_value = athrs17_reg_read(gmac_cfg, reg->addr + offset);
			printf("%08x ", l2sw_reg_value);
		}
		printf("\n");
	}
	printf("\n");
}

static int
l2swreg_read(int argc, char * const argv[])
{
	uint32_t reg_addr;
	uint32_t dbg_reg;
	uint32_t l2sw_reg_value;
	uint16_t dbg_reg_value;
	ipq_gmac_board_cfg_t *gmac_tmp_cfg = gboard_param->gmac_cfg;

	switch (argc) {
		/* l2sw reg */
	case 2:
		reg_addr = strtoul(argv[1], NULL, 16);
		l2sw_reg_value = athrs17_reg_read(gmac_tmp_cfg,reg_addr);
		printf("READ REG:0x%x = 0x%08x\n", reg_addr, l2sw_reg_value);
		break;
		/* debug reg */
	case 3:
		reg_addr = strtoul(argv[1], NULL, 16);
		dbg_reg = strtoul(argv[2], NULL, 16);
		athrs17_phy_debug_read(gmac_tmp_cfg, reg_addr, dbg_reg, &dbg_reg_value);
		printf("READ ADR:0x%x DBG_REG:0x%x = 0x%04x\n",
		       reg_addr, dbg_reg, dbg_reg_value);
		break;
	default:
		printf("l2sw read L2SW_REG\n");
		printf("l2sw read ADDR DEBUG_REG\n");
		return 1;
	}

	return 0;
}

static int
l2swreg_write(int argc, char * const argv[])
{
	uint32_t reg_addr;
	uint32_t dbg_reg;
	uint32_t l2sw_reg_value;
	uint16_t dbg_reg_value;
	ipq_gmac_board_cfg_t *gmac_tmp_cfg = gboard_param->gmac_cfg;

	switch (argc) {
		/* l2sw reg */
	case 3:
		reg_addr = strtoul(argv[1], NULL, 16);
		l2sw_reg_value = strtoul(argv[2], NULL, 16);
		printf("WRITE REG:0x%x = 0x%08x\n", reg_addr, l2sw_reg_value);
		athrs17_reg_write(gmac_tmp_cfg, reg_addr, l2sw_reg_value);

		break;
		/* debug reg */
	case 4:
		reg_addr = strtoul(argv[1], NULL, 16);
		dbg_reg = strtoul(argv[2], NULL, 16);
		dbg_reg_value = strtoul(argv[3], NULL, 16);
		printf("WRITE ADR:0x%x DBG_REG:0x%x = 0x%04x\n",
		       reg_addr, dbg_reg, dbg_reg_value);
		athrs17_phy_debug_write(gmac_tmp_cfg, reg_addr, dbg_reg, dbg_reg_value);
		break;
	default:
		printf("l2sw write L2SW_REG DATA\n");
		printf("l2sw write ADDR DEBUG_REG DATA\n");
		return 1;
	}

	return 0;
}

static int
l2swreg_dump(int argc, char * const argv[])
{
	ipq_gmac_board_cfg_t *gmac_tmp_cfg = gboard_param->gmac_cfg;

	dump_phy_reg(gmac_tmp_cfg);
	dump_phy_debug_reg(gmac_tmp_cfg);
	dump_phy_mmd3_reg(gmac_tmp_cfg);
	dump_phy_mmd7_reg(gmac_tmp_cfg);
	dump_l2sw_reg(gmac_tmp_cfg);

	return 0;
}

const l2swreg_access_t ath27_l2swreg = {
	l2swreg_read,
	l2swreg_write,
	l2swreg_dump,
};

#endif /* CONFIG_CMD_DEBUG_L2SW */

struct athrs17_regmap {
	uint32_t start;
	uint32_t end;
};

struct athrs17_regmap regmap[] = {
	{ 0x000, 0x0e0 },
	{ 0x100, 0x168 },
	{ 0x200, 0x270 },
	{ 0x400, 0x454 },
	{ 0x600, 0x718 },
	{ 0x800, 0xb70 },
	{ 0xC00, 0xC80 },
};

int do_ar8xxx_dump(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	int i;

	for (i = 0; i < ARRAY_SIZE(regmap); i++) {
		uint32_t reg;
		struct athrs17_regmap *section = &regmap[i];

		for (reg = section->start; reg <= section->end; reg += sizeof(uint32_t)) {
			ipq_gmac_board_cfg_t *gmac_tmp_cfg = gboard_param->gmac_cfg;
			uint32_t val = athrs17_reg_read(gmac_tmp_cfg, reg);
			printf("%03zu: %08zu\n", reg, val);
		}
	}

	return 0;
};
U_BOOT_CMD(
	ar8xxx_dump,    1,    1,    do_ar8xxx_dump,
	"Dump ar8xxx registers",
	"\n    - print all ar8xxx registers\n"
);

#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
static void
athrs17_phy_mmd_write(ipq_gmac_board_cfg_t *gmac_cfg, int phy, int mmd, uint16_t reg, uint16_t data)
{
	mmd &= 0x1f;

	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_CONTROL,
						S17_MMD_FUNC_ADDR | mmd);
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_ADDRESS_DATA, reg);
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_CONTROL,
						S17_MMD_FUNC_DATA | mmd);
	miiphy_write(gmac_cfg->phy_name, phy, ATHR_MMD_ACCESS_ADDRESS_DATA, data);
}

static void
athrs17_rotary_init(ipq_gmac_board_cfg_t *gmac_cfg)
{
	int phy;

	/* disable EEE */
	athrs17_reg_write(gmac_cfg, S17_EEE_CTRL_REG, 0x00000000);
	for (phy = 0; phy < S17_PHY_MAX ; phy++)
		athrs17_phy_mmd_write(gmac_cfg, phy, 7, ATHR_MMD7_EEE_ADVERTISEMENT,
								0x0000);

	/* change LED blink freq. */
	athrs17_reg_write(gmac_cfg, S17_LED_CTRL0_REG, 0xcc36cc36);
	athrs17_reg_write(gmac_cfg, S17_LED_CTRL1_REG, 0xca36ca36);
	athrs17_reg_write(gmac_cfg, S17_LED_CTRL2_REG, 0xc936c936);
	athrs17_reg_write(gmac_cfg, S17_LED_CTRL3_REG, 0x03ffff00);

	/* set egress VLAN tag mode = untouched */
	athrs17_reg_write(gmac_cfg, S17_ROUTER_EG_VLAN_MODE, 0x03333333);
}
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/

/*********************************************************************
 *
 * FUNCTION DESCRIPTION: This function invokes RGMII,
 * 			SGMII switch init routines.
 * INPUT : ipq_gmac_board_cfg_t *
 * OUTPUT: NONE
 *
**********************************************************************/
int ipq_athrs17_init(ipq_gmac_board_cfg_t *gmac_cfg)
{
	int ret;

	if (gmac_cfg == NULL)
		return -1;

	ret = athrs17_init_switch(gmac_cfg);
	if (ret != -1) {
		athrs17_reg_init(gmac_cfg);
		athrs17_reg_init_lan(gmac_cfg);
#if defined(CONFIG_MACH_IPQ806X_AP148_NECPF) || \
				defined(CONFIG_MACH_IPQ806X_AP145_NECPF)
		athrs17_rotary_init(gmac_cfg);
#endif	/* CONFIG_MACH_IPQ806X_AP148_NECPF CONFIG_MACH_IPQ806X_AP145_NECPF*/
		athrs17_vlan_config(gmac_cfg);
		printf ("S17c init  done\n");
#if defined(CONFIG_L2SW_MIB)
		athrs17_reg_write(gmac_cfg, S17_MODULE_EN_REG, athrs17_reg_read(
				gmac_cfg, S17_MODULE_EN_REG) | S17_MODULE_MIB_EN);
#endif	/* CONFIG_L2SW_MIB */
	}

#ifdef CONFIG_CMD_DEBUG_L2SW
	regist_debug_l2sw_info(&ath27_l2swreg);
#endif /* CONFIG_CMD_DEBUG_L2SW */

	return ret;
}

#if defined(CONFIG_L2SW_MIB)
static const int mib_addr[] = {
	S17_MIB_PORT0, S17_MIB_PORT1, S17_MIB_PORT2, S17_MIB_PORT3,
	S17_MIB_PORT4, S17_MIB_PORT5, S17_MIB_PORT6
};

uint32_t
ipq_mib_tx_broadcast(int mac)
{
	ipq_gmac_board_cfg_t *gmac_cfg = gboard_param->gmac_cfg;

	if (mac >= S17_MAC_MAX)
		return 0;

	return athrs17_reg_read(gmac_cfg, mib_addr[mac] + S17_MIB_TXBROAD);
}

uint32_t
ipq_mib_tx_unicast(int mac)
{
	ipq_gmac_board_cfg_t *gmac_cfg = gboard_param->gmac_cfg;

	if (mac >= S17_MAC_MAX)
		return 0;

	return athrs17_reg_read(gmac_cfg, mib_addr[mac] + S17_MIB_TXUNI);
}

uint32_t
ipq_mib_rx_broadcast(int mac)
{
	ipq_gmac_board_cfg_t *gmac_cfg = gboard_param->gmac_cfg;

	if (mac >= S17_MAC_MAX)
		return 0;

	return athrs17_reg_read(gmac_cfg, mib_addr[mac] + S17_MIB_RXBROAD);
}

uint32_t
ipq_mib_rx_unicast(int mac)
{
	ipq_gmac_board_cfg_t *gmac_cfg = gboard_param->gmac_cfg;

	if (mac >= S17_MAC_MAX)
		return 0;

	return athrs17_reg_read(gmac_cfg, mib_addr[mac] + S17_MIB_RXUNI);
}

uint32_t
ipq_mib_rx_fcserr(int mac)
{
	ipq_gmac_board_cfg_t *gmac_cfg = gboard_param->gmac_cfg;

	if (mac >= S17_MAC_MAX)
		return 0;

	return athrs17_reg_read(gmac_cfg, mib_addr[mac] + S17_MIB_RXFCSERR);
}

void
ipq_mib_flush(int mac)
{
	if (mac >= S17_MAC_MAX)
		return;

	ipq_mib_tx_broadcast(mac);
	ipq_mib_tx_unicast(mac);
	ipq_mib_rx_broadcast(mac);
	ipq_mib_rx_unicast(mac);
	ipq_mib_rx_fcserr(mac);
}
#endif	/* CONFIG_L2SW_MIB */

#if defined(CONFIG_API)

DECLARE_GLOBAL_DATA_PTR;

int
ipq_switch_reinit(void)
{
	ipq_gmac_board_cfg_t *gmac_cfg = gboard_param->gmac_cfg;
	int i, done = 0;

	for (i = 0; gmac_cfg_is_valid(gmac_cfg); gmac_cfg++, i++) {
		if (!done) {
			if (ipq_athrs17_init(gmac_cfg) == 0)
				done = 1;
			else
				return -1;
		}
	}
	return 0;
}
#endif	/* CONFIG_API */
