#include <common.h>
#include <watchdog.h>
#include <command.h>
#include <linux/ctype.h>
#include <linux/flashmng.h>

#define strtoul(n, e, b) simple_strtoul((n), (e), (b))
#define tty_flush()
#define key_abort() ctrlc()

#define FLASH_TOP_ADDR 0x00000000
#define BANK_SIZE FLASH_BLOCK_SIZE
#define DIAG_START_ADDR CONFIG_KERNEL_OFFSET
#define DIAG_FLASH_OPT_FILL (1 << 0)

int do_debug_flash(cmd_tbl_t *, int, int, char * const []);

static int cmd_read_flash(cmd_tbl_t *, int, int, char * const []);
static int cmd_write_flash(cmd_tbl_t *, int, int, char * const []);
static int cmd_erase_flash(cmd_tbl_t *, int, int, char * const []);
static int cmd_info_flash(cmd_tbl_t *, int, int, char * const []);
static int cmd_diag_flash(cmd_tbl_t *, int, int, char * const []);

static cmd_tbl_t cmd_debug_flash_sub[] = {
	U_BOOT_CMD_MKENT(read, 0, 0, (void *)cmd_read_flash,
			 "", ""),
	U_BOOT_CMD_MKENT(write, 0, 0, (void *)cmd_write_flash,
			 "", ""),
	U_BOOT_CMD_MKENT(erase, 0, 0, (void *)cmd_erase_flash,
			 "", ""),
	U_BOOT_CMD_MKENT(info, 0, 0, (void *)cmd_info_flash,
			 "", ""),
	U_BOOT_CMD_MKENT(diag, 0, 0, (void *)cmd_diag_flash,
			 "", ""),
};

int
do_debug_flash(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	cmd_tbl_t *c;

	if (argc < 2)
		return CMD_RET_USAGE;

	/* Strip off leading 'watchdog' command argument */
	argc--;
	argv++;

	c = find_cmd_tbl(argv[0], &cmd_debug_flash_sub[0],
			 ARRAY_SIZE(cmd_debug_flash_sub));

	if (c)
		return c->cmd(cmdtp, flag, argc, argv);
	else
		return CMD_RET_USAGE;
}

#define DISP_LINE_LEN 16
#define DISP_WIDTH 1		/* byte */
static int
cmd_read_flash(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	const void *addr;
	int length;
	uint8_t buf[16];

	if (argc < 3) {
		printf("read ADDR LENGTH\n");
		printf("# ex) read 0x2000 0x100\n");
		printf("# 8bit access only\n");
		return 1;
	}
	addr = (const void *) strtoul(argv[1], NULL, 16);
	if ((length = strtoul(argv[2], NULL, 16)) == 0) {
		printf("0x%x invalid length\n", length);
		return 2;
	}

	while (length > 0) {
		int rlen = length > sizeof(buf) ? sizeof(buf) : length;
		int result;

		result = flread(buf, addr, rlen);
		if (result <= 0) {
			printf("read error\n");
			break;
		}
		if (print_buffer((ulong)addr, (void*)buf, DISP_WIDTH,
				 rlen, DISP_LINE_LEN / DISP_WIDTH)) {
			break;	/* abort */
		}
		length -= rlen;
		addr += rlen;
		if (key_abort()) {
			break;
		}
		WATCHDOG_RESET();
	}
	return 0;
}

static int
cmd_write_flash(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	uint8_t *addr;
	uint8_t data[16];
	int length;
	int i;

	if (argc < 3) {
		printf("write ADDR DATA ...\n");
		printf("# ex) write 0x2000 0x10 0x20 0x30\n");
		printf("# 8bit access only\n");
		return 1;
	}
	addr = (void *)strtoul(argv[1], NULL, 16);
	for (i = 2, length = 0; i < argc; i++) {
		data[length++] = (uint8_t) strtoul(argv[i], NULL, 16);
		if (length >= sizeof(data)) {
			if (flwrite(addr,
				    data, sizeof(data)) != sizeof(data)) {
				printf("write error\n");
				length = 0;
				break;
			}
			length = 0;
			addr += sizeof(data);
		}
	}
	if (length > 0) {
		if (flwrite(addr, data, length) != length) {
			printf("write error\n");
		}
	}
	return 0;
}

/* SIZE : 指定されたサイズ以下の SECTOR を ERASE する。 */
static int
cmd_erase_flash(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	void *addr;
	size_t size;

	if (argc < 3) {
		printf("erase SECTOR_ADDR SIZE\n");
		printf("# ex) erase 0x002c0000 0x10000\n");
		return 1;
	}
	addr = (void *)strtoul(argv[1], NULL, 16);
	size = strtoul(argv[2], NULL, 16);

	if (flerase(addr, size) == 0) {
		printf("ok\n");
	} else {
		printf("error\n");
	}
	tty_flush();

	return 0;
}

/*
 * 検出, install した flash の情報を出力する
 */
static int
cmd_info_flash(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{

	printf("  Flash : ");
	print_size(flsize(), "\n");
	printf("    Device         : %s\n", fldev_name(NULL));
	printf("    Sector size    : ");
	print_size(flbank_size(NULL), "\n");
	return 0;
}

static uint32_t
erase_scan(uint32_t sector)
{
	uint8_t buf[128];
	uint32_t length;
	uint32_t addr;
	unsigned int banksize;

	if ((banksize = flbank_size(NULL)) == 0)
		banksize = BANK_SIZE;
	for (length = banksize, addr = sector; length > 0;) {
		uint32_t len = (length < sizeof(buf)) ? length : sizeof(buf);
		int i;

		memset(buf, 0, sizeof(buf));
		flread(buf, (void *) addr, len);

		for (i = 0; i < len; i++) {
			if (buf[i] != 0xff) {
				return addr + i;
			}
		}
		addr += len;
		length -= len;
	}
	return 0;
}

static uint32_t
fill_scan(uint32_t sector)
{
	uint8_t buf[64];
	uint32_t length;
	uint32_t addr;
	unsigned int banksize;

	if ((banksize = flbank_size(NULL)) == 0)
		banksize = BANK_SIZE;
	for (length = banksize, addr = sector; length > 0;) {
		uint32_t len =
		    (length < sizeof(buf)) ? length : sizeof(buf);
		int i;

		memset(buf, 0, sizeof(buf));
		flread(buf, (void *) addr, len);

		for (i = 0; i < len; i++) {
			if (buf[i] != ((addr + i) & 0xff)) {
				return addr + i;
			}
		}
		addr += len;
		length -= len;
	}
	return 0;
}

static int
diag_flash_erase_sector(uint32_t sector)
{
	int ret;
	uint32_t addr;

	printf("erase 0x%x ...", sector);
	ret = flerase((void *)sector, flbank_size(NULL));
	tty_flush();
	printf(" %s(%d)", ret == 0 ? "OK" : "NG", ret);
	if (ret != 0) {
		printf("\n");
		return 1;
	}
	if ((addr = erase_scan(sector)) == 0) {
		printf(" erase scan OK\n");
	} else {
		printf(" erase scan NG (0x%x)\n", addr);
		return 2;
	}
	return 0;		/* OK */
}

static uint32_t
random_value(uint32_t value)
{
	return value * 17 + 1;
}

#define RND(value) ((value) = random_value(value))

/*
 * write random length
 */
static int
diag_flash_fill_sector(uint32_t sector)
{
	int ret;
	uint32_t rnd;
	uint32_t length;
	uint32_t addr;
	uint8_t buf[128];

	rnd = (uint32_t)get_timer(0);

	printf("write 0x%x ...", sector);
	tty_flush();
	addr = sector;
	if ((length = flbank_size(NULL)) == 0)
		length = BANK_SIZE;
	while (length > 0) {
		uint32_t offset = RND(rnd) % (sizeof(buf) / 4);
		uint32_t len = RND(rnd) % (sizeof(buf) + 1);
		int i, j;

		len = (len + offset) > sizeof(buf) ?
			sizeof(buf) - offset : len;
		len = (len < length) ? len : length;

		memset(buf, 0, sizeof(buf));
		for (i = offset, j = 0; j < len; i++, j++) {
			buf[i] = (addr + j) & 0xff;
		}
		if ((ret = flwrite((void *)addr, buf + offset, len)) != len) {
			printf("write invalid ret=%d"
			       " addr=0x%x offset=%d len=%d\n",
			       ret, addr, offset, len);
			return 1;
		}
		length -= len;
		addr += len;
	}
	if ((addr = fill_scan(sector)) == 0) {
		printf(" read scan OK\n");
	} else {
		printf(" read scan NG (0x%x)\n", addr);
		return 2;
	}
	return 0;
}

/*
 * read random length
 */
static int
diag_flash_scan_sector(uint32_t sector)
{
	int ret;
	uint32_t rnd;
	uint32_t length;
	uint32_t addr;
	uint8_t buf[128];

	rnd = (uint32_t)get_timer(0);

	printf("read 0x%x ...", sector);
	tty_flush();
	addr = sector;
	if ((length = flbank_size(NULL)) == 0)
		length = BANK_SIZE;
	while (length > 0) {
		uint32_t offset = RND(rnd) % (sizeof(buf) / 4);
		uint32_t len = RND(rnd) % (sizeof(buf) + 1);
		int i, j;

		len = (len + offset) >
			sizeof(buf) ? sizeof(buf) - offset : len;
		len = (len < length) ? len : length;

		memset(buf, 0, sizeof(buf));
		if ((ret = flread(buf + offset, (void *) addr, len)) != len) {
			printf("read invalid ret=%d"
			       " addr=0x%x offset=%d len=%d\n",
			       ret, addr, offset, len);
			return 1;
		}
		for (i = offset, j = 0; j < len; i++, j++) {
			if (buf[i] != ((addr + j) & 0xff)) {
				printf("read fail addr 0x%x"
				       " data:0x%x buf:0x%x\n",
				       addr + j, (addr + j) & 0xff, buf[i]);
				printf("NG\n");
				return 1;
			}
		}
		length -= len;
		addr += len;
	}
	printf("OK\n");
	return 0;
}

static int
diag_flash_sector(uint32_t sector, uint32_t opt)
{
	int ret;

	printf("== diag sector 0x%x ==\n", sector);
	if ((ret = diag_flash_erase_sector(sector)) != 0)
		return ret;
	if (key_abort()) {
		printf("key abort\n");
		return 1;
	}
	if ((ret = diag_flash_fill_sector(sector)) != 0)
		return ret;
	if (key_abort()) {
		printf("key abort\n");
		return 1;
	}
	if ((ret = diag_flash_scan_sector(sector)) != 0)
		return ret;
	if (opt & DIAG_FLASH_OPT_FILL)
		return 0;
	if (key_abort()) {
		printf("key abort\n");
		return 1;
	}
	if ((ret = diag_flash_erase_sector(sector)) != 0)
		return ret;
	return 0;
}

static int
cmd_diag_flash(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	uint32_t flash_top_addr = FLASH_TOP_ADDR;
	unsigned long total_size = flsize();
	int i;
	int ret;
	unsigned int banksize;
	uint32_t opt;

	if (argc < 2) {
		printf("diag [fill] SECTOR_ADDRESS ...\n");
		printf("SECTOR_ADDRESS = all is all sectors "
		       "(except boot, config, tp, tiny area)\n");
		printf("sector area is 0x%x - 0x%x\n",
		       flash_top_addr, flash_top_addr + (uint32_t)total_size);
		return 1;
	}
	opt = 0;		/* all */
	if ((argc > 2) && (strcmp(argv[1], "fill") == 0)) {
		opt |= DIAG_FLASH_OPT_FILL;
		argc--;
		argv++;
	}
	if ((banksize = flbank_size(NULL)) == 0)
		banksize = BANK_SIZE;
	if (strcmp(argv[1], "all") == 0) {
		uint32_t start_addr, end_addr;

		start_addr = DIAG_START_ADDR;
		end_addr = flash_top_addr + total_size;
		while (start_addr < end_addr) {
			switch (start_addr) {
			default:
				ret = diag_flash_sector(start_addr, opt);
				if (ret != 0) {
					return 1;
				}
			}
			start_addr += banksize;
			if (key_abort()) {
				printf("key abort\n");
				break;
			}
		}
		printf("diag test complete\n");
		return 0;
	}
	for (i = 1; i < argc; i++) {
		uint32_t addr = strtoul(argv[i], NULL, 16);

		addr &= (~(banksize - 1));
		if ((addr < flash_top_addr) ||
		    (addr >= (flash_top_addr + total_size))) {
			printf("invalid address 0x%x\n", addr);
			continue;
		}
		ret = diag_flash_sector(addr, opt);
		if (ret != 0) {
			return 1;
		}
		if (key_abort()) {
			printf("key abort\n");
			break;
		}
	}
	printf("diag test complete\n");
	return 0;
}
