#include <common.h>
#include <command.h>
#include <watchdog.h>
#include <malloc.h>

DECLARE_GLOBAL_DATA_PTR;

#define UDP_PORT 9
#define DATA_SIZE_MIN 4
#define PKT_ALIGN 128
#define PKT_MAX_ALIGN (PKT_ALIGN * 12) /* 1536 */
#if defined(CONFIG_ETH_TXBUF)
#define PKT_MAX_NUMS CONFIG_ETH_TXBUF
#else
#define PKT_MAX_NUMS 1
#endif

/*
 * 使用例
 *	boot> setenv ipaddr 192.168.0.1
 *	boot> debug pps send mac=a4:12:42:00:35:3b ip=192.168.0.2
 *      Ctrl+C押す
 *	sent packets=3031040, elapse=3567, rate=849744pps
 */
static int
cmd_pps_send(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	uchar ether[6], *pkt;
	IPaddr_t ip = 0;
	int i, size = DATA_SIZE_MIN, cur_buf, wdt = 0;
	const char *s;
	char *t;
	void *bufmem = NULL;
	uchar *pktbuf[PKT_MAX_NUMS];
	u32 seqno, *seqno_p;
	ulong cnt = 0, total, ms;
	u64 rate;

	memcpy(ether, NetEtherNullAddr, sizeof(ether));
	for (i = 1; i < argc; i++) {
		s = argv[i];
		if (strncmp(s, "mac=", 4) == 0) {
			ether[0] = simple_strtoul(&s[4], &t, 16);
			ether[1] = simple_strtoul(++t, &t, 16);
			ether[2] = simple_strtoul(++t, &t, 16);
			ether[3] = simple_strtoul(++t, &t, 16);
			ether[4] = simple_strtoul(++t, &t, 16);
			ether[5] = simple_strtoul(++t, NULL, 16);
		} else if (strncmp(s, "ip=", 3) == 0)
			ip = string_to_ip(&s[3]);
		else if (strncmp(s, "cnt=", 4) == 0)
			cnt = simple_strtoul(&s[4], NULL, 10);
		else if (strncmp(s, "size=", 5) == 0)
			size = simple_strtoul(&s[5], NULL, 10);
	}
	if (ip == 0)
		return CMD_RET_USAGE;
	else if (memcmp(ether, NetEtherNullAddr, sizeof(ether)) == 0)
		return CMD_RET_USAGE;
	else if (size < DATA_SIZE_MIN)
		return CMD_RET_USAGE;
	else if ((size + ETHER_HDR_SIZE + IP_UDP_HDR_SIZE) > PKTSIZE)
		return CMD_RET_USAGE;

	bufmem = memalign(PKT_ALIGN, PKT_MAX_ALIGN * PKT_MAX_NUMS);
	if (bufmem == NULL) {
		printf("no memory\n");
		return CMD_RET_FAILURE;
	}
	for (i = 0; i < PKT_MAX_NUMS; i++) {
		pkt = pktbuf[i] = (uchar *)((ulong)bufmem + PKT_MAX_ALIGN * i);
		NetSetEther(pkt, ether, PROT_IP);
		net_set_udp_header(pkt + ETHER_HDR_SIZE, ip, UDP_PORT,
							UDP_PORT, size);
	}

	net_init();
	eth_halt();
	eth_set_current();
	if (eth_init(gd->bd) < 0) {
		printf("eth_init failed\n");
		eth_halt();
		free(bufmem);
		return CMD_RET_FAILURE;
	}

	cur_buf = 0;
	seqno = 0;
	total = 0;
	ms = get_timer(0);
	while (1) {
		if ((cnt == 0) && (wdt % 16384) == 0) {
			if (ctrlc())
				break;
		}
		if ((wdt++ % 16384) == 0)
			WATCHDOG_RESET();

		seqno_p = (u32 *)&pktbuf[cur_buf][ETHER_HDR_SIZE +
							IP_UDP_HDR_SIZE];
		*seqno_p = htonl(seqno);
		if (!eth_send(pktbuf[cur_buf], ETHER_HDR_SIZE +
						IP_UDP_HDR_SIZE + size)) {
			total++;
			seqno++;
			cur_buf = (cur_buf + 1) % PKT_MAX_NUMS;
			if (cnt > 0) {
				if (--cnt == 0)
					break;
			}
		}
	}
	ms = get_timer(ms);
	printf("sent packets=%lu, elapse=%lums", total, ms);
	if (ms != 0) {
		rate = ((u64)total * CONFIG_SYS_HZ) / ms;
		printf(", rate=%llupps", rate);
	}
	printf("\n");

	mdelay(1 * PKT_MAX_NUMS);
	eth_halt();
	free(bufmem);
	return CMD_RET_SUCCESS;
}

/*
 * 使用例
 *	boot> setenv ipaddr 192.168.0.2
 *	boot> ping 192.168.0.10 (L2SWに学習させる)
 *	Using eth1 device
 *	host 192.168.1.10 is alive
 *	boot> debug pps recv
 *      Ctrl+C押す
 *	received packets=3031040, fail=0
 */
static int
cmd_pps_recv(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	u32 seqno = (0 - 1), *seqno_p, seqno_r;
	ulong total = 0, fail = 0, cnt = 0;
	void *buf;
	int i, wdt = 0;
	char *s;

	for (i = 1; i < argc; i++) {
		s = argv[i];
		if (strncmp(s, "cnt=", 4) == 0)
			cnt = simple_strtoul(&s[4], NULL, 10);
	}

	net_init();
	eth_halt();
	eth_set_current();
	if (eth_init(gd->bd) < 0) {
		printf("eth_init failed\n");
		eth_halt();
		return CMD_RET_FAILURE;
	}

	while (1) {
		if ((cnt == 0) && (wdt % 16384) == 0) {
			if (ctrlc())
				break;
		}
		if ((wdt++ % 16384) == 0)
			WATCHDOG_RESET();

		if (eth_raw_rx(&buf) > 0) {
			total++;
			seqno_p = (u32 *)((ulong)buf + ETHER_HDR_SIZE +
							IP_UDP_HDR_SIZE);
			if ((seqno_r = ntohl(*seqno_p)) != (seqno + 1))
				fail++;
			seqno = seqno_r;

			if (cnt > 0) {
				if (--cnt == 0)
					break;
			}
		}
	}

	printf("received packets=%lu, fail=%lu\n", total, fail);

	eth_halt();
	return CMD_RET_SUCCESS;
}

static cmd_tbl_t cmd_debug_pps_sub[] = {
	U_BOOT_CMD_MKENT(send, 5, 1, (void *)cmd_pps_send,
	"PPS send test",
	"debug pps send mac=REMOTE_MAC_ADDR\n"
	"               ip=REMOTE_IP_ADDR\n"
	"               [size=DATAGRAM_SIZE]\n"
	"               [cnt=PACKET_COUNT] (when 0, use ctrl+c)\n"),
	U_BOOT_CMD_MKENT(recv, 0, 1, (void *)cmd_pps_recv,
	"PPS receive test",
	"debug pps recv [size=DATAGRAM_SIZE]\n"
	"               [cnt=PACKET_COUNT] (when 0, use ctrl+c)\n"),
};

int
do_debug_pps(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	cmd_tbl_t *c;

	if (argc < 2)
		return CMD_RET_USAGE;

	argc--;
	argv++;

	c = find_cmd_tbl(argv[0], &cmd_debug_pps_sub[0],
			 ARRAY_SIZE(cmd_debug_pps_sub));

	if (c)
		return c->cmd(cmdtp, flag, argc, argv);
	else
		return CMD_RET_USAGE;
}
