#include <common.h>
#include <nand.h>
#include <linux/ctype.h>
#include <linux/types.h>

#include <linux/flashmng.h>

#if defined(CONFIG_SPI_FLASH)
#include <spi_flash.h>
#endif	/* CONFIG_SPI_FLASH */

/*
 * flash read cache 機能
 *   flash read 関数が特定のサイズのリードを必ず行う仕組みの場合に、無駄な
 *   リードが発生するのを防ぐために driver 内に cache を用意する。
 *   e.g.) ipq_spi_read() では、read length に関わらず sector size 単位の
 *         read を行っている。
 */
#if defined(FLASHMNG_FLCACHE)
static int flcache_init(void);
static int flcache_read(void *, const uint8_t *, int);
static int flcache_invalidate(const uint8_t *, int);
#else  /* FLASHMNG_FLCACHE */
static int flcache_init(void) {return 0;}
static int flcache_read(void *buf, const uint8_t *flash, int len) {return 0;}
static int flcache_invalidate(const uint8_t *flash, int len) {return 0;}
#endif	/* FLASHMNG_FLCACHE */

static int flash_type = -1;

void
flmng_init(int dev_num)
{
	if (dev_num != -1) {
		flash_type = dev_num;
	}
	flcache_init();
}

int
flread(void *buf, const void *flash, int len)
{
	size_t rlen = len;

	if (len <= 0) {
		return 0;
	}

	if ((len = flcache_read(buf, flash, len)) > 0) {
		return len;
	}

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return 0;
	}

	nand_read(&nand_info[flash_type], (uint32_t)flash, &rlen, buf);

	return rlen;
}

int
flwrite(void *flash, const void *buf, int len)
{
	size_t rlen = len;

	if (len <= 0) {
		return 0;
	}

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return 0;
	}

	flcache_invalidate(flash, len);
	nand_write(&nand_info[flash_type],
		   (uint32_t)flash, &rlen, (u_char *)buf);

	return rlen;
}

int
flerase(void *flash, int len)
{

	if (len <= 0) {
		return 0;
	}

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return -1;
	}

	flcache_invalidate(flash, len);
	nand_erase(&nand_info[flash_type], (uint32_t)flash, len);

	return 0;
}

const char *
fldev_name(void *flash_addr)
{
	const char *name = NULL;

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return NULL;
	}

#if defined(CONFIG_SPI_FLASH)
	{
		const struct mtd_info *mtd = &nand_info[flash_type];
		const struct spi_flash *flash = mtd->priv;

		name = flash->name;
	}
#endif	/* CONFIG_SPI_FLASH */

	return name;
}

/* 装置搭載 flash の total サイズを返す。 */
unsigned long
flsize(void)
{
	u32 size = 0;

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return 0;
	}

#if defined(CONFIG_SPI_FLASH)
	{
		const struct mtd_info *mtd = &nand_info[flash_type];
		const struct spi_flash *flash = mtd->priv;

		size = flash->size;
	}
#endif	/* CONFIG_SPI_FLASH */

	return size;
}

/* 装置搭載 flash の sector サイズを返す。 */
unsigned long
flbank_size(void *flash_addr)
{
	u32 size = 0;

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return 0;
	}

#if defined(CONFIG_SPI_FLASH)
	{
		const struct mtd_info *mtd = &nand_info[flash_type];
		const struct spi_flash *flash = mtd->priv;

		size = flash->sector_size;
	}
#endif	/* CONFIG_SPI_FLASH */

	return size;
}

#if defined(FLASHMNG_FLCACHE)

struct flcache_t {
	const uint8_t *addr;	/* flash address */
	int len;
	u_char buf[FLASH_BLOCK_SIZE]; /* data cache */
};
static struct flcache_t flcache_storage;

static int flcache_has_cache(const struct flcache_t *, const uint8_t *, int);
static int flcache_need_invalidate(
	const struct flcache_t *, const uint8_t *, int);

static int
flcache_init(void)
{
	struct flcache_t *flcache = &flcache_storage;

	flcache->addr = NULL;
	return 0;
}

/* cache にデータが有るか確認する。
 *   0: cache に該当データが無い。
 *   1: cache に該当データが有る。
 */
static int
flcache_has_cache(
	const struct flcache_t *flcache, const uint8_t *flash, int len)
{

	if (flcache->addr == NULL) { /* cache に有効なデータが無い。 */
		return 0;
	}
	/*  cache に hit しない部分が有る。
	 *    request
	 *    |-----|    cache
	 *        |-----------------|    */
	if (flash < flcache->addr) {
		return 0;
	}
	/*  cache に hit しない部分が有る。
	 *                       request
	 *               cache   |-----|
	 *        |-----------------|    */
	if ((flcache->addr + flcache->len) < (flash + len)) {
		return 0;
	}
	return 1;		/* cache hit */
}

static int
flcache_read(void *buf, const uint8_t *flash, int len)
{
	struct flcache_t *flcache = &flcache_storage;
	size_t rlen = len;
	int offset;

	/* cache size を上回る場合。 */
	if (sizeof(flcache->buf) < len) {
		return 0;
	}

	if (flash_type == -1) {
		printf("Unknown flash type\n");
		return 0;
	}

	/* cache にデータが無い場合は flash からリードする。 */
	if (!flcache_has_cache(flcache, flash, len)) {
		flcache->addr = NULL;
		rlen = sizeof(flcache->buf);
		nand_read(&nand_info[flash_type],
			  (uint32_t)flash, &rlen, flcache->buf);
		flcache->len = rlen;
		flcache->addr = flash;
	}

	offset = flash - flcache->addr;
	rlen = min(rlen, len);
	memcpy(buf, flcache->buf + offset, rlen);

	return rlen;
}

/* flash write/erase する場合に、cache invalidate が必要か確認する。
 *   0: cache invalidate 不要。
 *   1: cache invalidate 必要。
 */
static int
flcache_need_invalidate(
	const struct flcache_t *flcache, const uint8_t *flash, int len)
{

	if (flcache->addr == NULL) { /* invalidate 済み */
		return 0;
	}
	/* cache に影響しない領域なので invalidate 不要。0
	 *    request
	 *    |-----|        cache
	 *            |-----------------|     */
	if ((flash + len) <= flcache->addr) {
		return 0;
	}
	/* cache に影響しない領域なので invalidate 不要。0
	 *                            request
	 *               cache        |-----|
	 *        |-----------------|         */
	if ((flcache->addr + flcache->len) <= flash) {
		return 0;
	}
	return 1;		/* need invalidate */
}

/* cache を invalidate(無効) にする。 */
static int
flcache_invalidate(const uint8_t *flash, int len)
{
	struct flcache_t *flcache = &flcache_storage;

	if (flcache_need_invalidate(flcache, flash, len)) {
		flcache->addr = NULL; /* cache invalidate */
	}

	return 0;
}

#endif /* FLASHMNG_FLCACHE */
