/* (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2006 Netfilter Core Team <coreteam@netfilter.org>
 * (C) 2008 Patrick McHardy <kaber@trash.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/types.h>
#include <linux/random.h>
#include <linux/netfilter.h>
#include <linux/export.h>

#include <net/netfilter/nf_nat.h>
#include <net/netfilter/nf_nat_core.h>
#include <net/netfilter/nf_nat_l3proto.h>
#include <net/netfilter/nf_nat_l4proto.h>

#ifdef CONFIG_SNAT_MAPE
#include <net/map.h>    //---   2019-0221   for netfilter SNAT info.
#endif  //---   CONFIG_SNAT_MAPE end

bool nf_nat_l4proto_in_range(const struct nf_conntrack_tuple *tuple,
			     enum nf_nat_manip_type maniptype,
			     const union nf_conntrack_man_proto *min,
			     const union nf_conntrack_man_proto *max)
{
	__be16 port;

	if (maniptype == NF_NAT_MANIP_SRC)
		port = tuple->src.u.all;
	else
		port = tuple->dst.u.all;

	return ntohs(port) >= ntohs(min->all) &&
	       ntohs(port) <= ntohs(max->all);
}
EXPORT_SYMBOL_GPL(nf_nat_l4proto_in_range);

#ifdef CONFIG_SNAT_MAPE
//---   2019-0221   Add counter to check how many packets are finding unique tuple.
static u16 enter_unique_tuple_counter = 0;  //---   2019-0221   To fix SNAT unstable issue.
#endif  //---   CONFIG_SNAT_MAPE end

void nf_nat_l4proto_unique_tuple(const struct nf_nat_l3proto *l3proto,
				 struct nf_conntrack_tuple *tuple,
				 const struct nf_nat_range *range,
				 enum nf_nat_manip_type maniptype,
				 const struct nf_conn *ct,
				 u16 *rover)
{
	unsigned int range_size, min, i;
	__be16 *portptr;
	u_int16_t off;
#ifdef CONFIG_SNAT_MAPE
    //---   2019-0221   To fix SNAT unstable issue.
    u16 cur_counter = 0;
    //---   2019-0221   Merge Map-E SNAT flow to here.
    unsigned int j;
    bool mape_result;
#endif  //---   CONFIG_SNAT_MAPE end
#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
    bool         use_time_wait;
    unsigned int min_time_stamp;
	u_int16_t    min_time_stamp_off;
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

#ifdef CONFIG_SNAT_MAPE
    //---   2019-0221   To fix SNAT unstable issue.
    enter_unique_tuple_counter++;
#endif  //---   CONFIG_SNAT_MAPE end
	if (maniptype == NF_NAT_MANIP_SRC)
		portptr = &tuple->src.u.all;
	else
		portptr = &tuple->dst.u.all;

	/* If no range specified... */
	if (!(range->flags & NF_NAT_RANGE_PROTO_SPECIFIED)) {
		/* If it's dst rewrite, can't change port */
		if (maniptype == NF_NAT_MANIP_DST)
        {
#ifdef CONFIG_SNAT_MAPE
            //---   2019-0221   To fix SNAT unstable issue.
            enter_unique_tuple_counter--;
#endif  //---   CONFIG_SNAT_MAPE end
			return;
        }

		if (ntohs(*portptr) < 1024) {
			/* Loose convention: >> 512 is credential passing */
			if (ntohs(*portptr) < 512) {
				min = 1;
				range_size = 511 - min + 1;
			} else {
				min = 600;
				range_size = 1023 - min + 1;
			}
		} else {
			min = 1024;
			range_size = 65535 - 1024 + 1;
		}
	} else {
		min = ntohs(range->min_proto.all);
		range_size = ntohs(range->max_proto.all) - min + 1;
	}

	if (range->flags & NF_NAT_RANGE_PROTO_RANDOM)
		off = l3proto->secure_port(tuple, maniptype == NF_NAT_MANIP_SRC
						  ? tuple->dst.u.all
						  : tuple->src.u.all);
	else
		off = *rover;

#ifdef CONFIG_SNAT_MAPE
    //---   2019-0221   To fix SNAT unstable issue.
    cur_counter = enter_unique_tuple_counter;
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
    min_time_stamp = -1;        //--- initial min_time_stamp = max unsigned int.
    use_time_wait = false;
    min_time_stamp_off = 0;
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

#ifdef CONFIG_SNAT_MAPE
    //---   2019-0221   To fix SNAT unstable issue.
    for( i = 0; i < range_size; i++) {
#else
	for (i = 0; ; ++off) {
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_SNAT_MAPE
        //---   2019-0221   To fix SNAT unstable issue.
		*portptr = htons(min + off % range_size);
        off++;  //--- 2019-0221   move offset increment to here.
#else
		*portptr = htons(min + off % range_size);
#endif  //---   CONFIG_SNAT_MAPE end
        
    #ifdef CONFIG_SNAT_MAPE
        //---   2019-0221 Move Map-E SNAT process from nf_nat_core to here.
        //---   2019-0201 check Map-E port set.
        if( ct->mark == MAP_NAPT_MARK )
        {
            write_lock_bh(&port_range_for_nf_lock);
            mape_result = false;        
            for( j = 0; (j < port_range_length_for_nf) && !mape_result; j++)
            {   //---   if current port is less then port range, then skip to next port range.
                if( ntohs(*portptr) > port_range_for_nf[j].max)
                {   continue;}
                if( ntohs(*portptr) < port_range_for_nf[j].min)
                {   break;}
                mape_result = true;
            }
            write_unlock_bh(&port_range_for_nf_lock);
            //---   if current port isn't in Map-E port set, skip to next port.
            if( mape_result == false)
                continue;
        }
    #endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_SNAT_MAPE
        //---   2019-0221   To fix SNAT unstable issue.
        //---   don't increase counter i here.
		if (nf_nat_used_tuple(tuple, ct))
        {
#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
                //---   2019-0926 If current tuple is used by other, check if it is in TIME_WAIT case,
                //---   If yes, then save the tuple with the smallest timestamp as target tuple.
                if ((tuple->dst.protonum == IPPROTO_TCP) && 
                        nf_conntrack_tuple_time_wait_reusable(tuple, ct, &use_time_wait, &min_time_stamp))
                {
                    min_time_stamp_off = off - 1;
                }
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end
			continue;
        }
#else
		if (++i != range_size && nf_nat_used_tuple(tuple, ct))
        {
#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
                //---   2019-0926 If current tuple is used by other, check if it is in TIME_WAIT case,
                //---   If yes, then save the tuple with the smallest timestamp as target tuple.
                if ((tuple->dst.protonum == IPPROTO_TCP) && 
                        nf_conntrack_tuple_time_wait_reusable(tuple, ct, &use_time_wait, &min_time_stamp))
                {
                    min_time_stamp_off = off;
                }
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end
            continue;
        }
#endif  //---   CONFIG_SNAT_MAPE end


#ifdef CONFIG_SNAT_MAPE
        //---   2019-0221   To fix SNAT unstable issue.
        --cur_counter;
        //---   2019-0221   If there are more then one packet finding tuple before it,
        //---   skip this tuple one and try to find next one.
        if(cur_counter!=0)
        {   continue;}
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
        //---   2019-0926 If we check every port, but none are available.
        //---   Check whether there is a TIME_WAIT conntrack that can be used.
        if( i == range_size && use_time_wait)
        {   
            //---   2019-0926 Set target port and reset off for next round.
            *portptr = htons(min + min_time_stamp_off % range_size);
            off = min_time_stamp_off++;
        }
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

		if (!(range->flags & NF_NAT_RANGE_PROTO_RANDOM))
			*rover = off;
#ifdef CONFIG_SNAT_MAPE
        //---   2019-0221   To fix SNAT unstable issue.
        enter_unique_tuple_counter--;
#endif  //---   CONFIG_SNAT_MAPE end
		return;
	}

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
    //---   2019-0926 If we check every port, but none are available.
    //---   Check whether there is a TIME_WAIT conntrack that can be used.
    if( i == range_size && use_time_wait)
    {   
        //---   2019-0926 Set target port and reset off for next round.
        *portptr = htons(min + min_time_stamp_off % range_size);
        off = min_time_stamp_off++;
    }
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

#ifdef CONFIG_SNAT_MAPE
#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
    //---   2019-0926   Set Next round start port.
    if (!(range->flags & NF_NAT_RANGE_PROTO_RANDOM))
        *rover = off;
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end
    //---   2019-0221   To fix SNAT unstable issue.
    enter_unique_tuple_counter--;
#endif  //---   CONFIG_SNAT_MAPE end
	return;
}
EXPORT_SYMBOL_GPL(nf_nat_l4proto_unique_tuple);

#if defined(CONFIG_NF_CT_NETLINK) || defined(CONFIG_NF_CT_NETLINK_MODULE)
int nf_nat_l4proto_nlattr_to_range(struct nlattr *tb[],
				   struct nf_nat_range *range)
{
	if (tb[CTA_PROTONAT_PORT_MIN]) {
		range->min_proto.all = nla_get_be16(tb[CTA_PROTONAT_PORT_MIN]);
		range->max_proto.all = range->min_proto.all;
		range->flags |= NF_NAT_RANGE_PROTO_SPECIFIED;
	}
	if (tb[CTA_PROTONAT_PORT_MAX]) {
		range->max_proto.all = nla_get_be16(tb[CTA_PROTONAT_PORT_MAX]);
		range->flags |= NF_NAT_RANGE_PROTO_SPECIFIED;
	}
	return 0;
}
EXPORT_SYMBOL_GPL(nf_nat_l4proto_nlattr_to_range);
#endif
