/*	$KAME: dhcp6c.c,v 1.164 2006/01/10 02:46:09 jinmei Exp $	*/
/*
 * Copyright (C) 1998 and 1999 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/queue.h>
#include <errno.h>
#include <limits.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <net/if.h>
#ifdef __FreeBSD__
#include <net/if_var.h>
#endif

#include <netinet/in.h>
#ifdef __KAME__
#include <net/if_dl.h>
#include <netinet6/in6_var.h>
#endif

#include <arpa/inet.h>
#include <netdb.h>

#include <signal.h>
#include <stdio.h>
#include <stdarg.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <err.h>
#include <ifaddrs.h>

#include <dhcp6.h>
#include <config.h>
#include <common.h>
#include <timer.h>
#include <dhcp6c.h>
#include <control.h>
#include <dhcp6_ctl.h>
#include <dhcp6c_ia.h>
#include <prefixconf.h>
#include <auth.h>

#include <stdarg.h>

static int debug = 0;
static int exit_ok = 0;
static sig_atomic_t sig_flags = 0;
#define SIGF_TERM 0x1
#define SIGF_HUP 0x2

#ifdef DHCPV6_LOGO
#define SIGF_USR1 0x4
#define SIGF_USR2 0x8
#endif


int sock;	/* inbound/outbound udp port */
int rtsock;	/* routing socket */
static int ctlsock = -1;		/* control TCP port */
static char *ctladdr = DEFAULT_CLIENT_CONTROL_ADDR;
static char *ctlport = DEFAULT_CLIENT_CONTROL_PORT;

#define DEFAULT_KEYFILE SYSCONFDIR "/dhcp6cctlkey"
#define CTLSKEW 300

static char *conffile = DHCP6C_CONF;

static const struct sockaddr_in6 *sa6_allagent;
static struct duid client_duid;
static char *pid_file = DHCP6C_PIDFILE;

static char *ctlkeyfile = DEFAULT_KEYFILE;
static struct keyinfo *ctlkey = NULL;
static int ctldigestlen;

static int infreq_mode = 0;

static inline int get_val32 __P((char **, int *, u_int32_t *));
static inline int get_ifname __P((char **, int *, char *, int));

static void usage __P((void));
static void client6_init __P((void));
static void client6_startall __P((int));
static void free_resources __P((struct dhcp6_if *));
static void client6_mainloop __P((void));
static int client6_do_ctlcommand __P((char *, ssize_t));
static void client6_reload __P((void));
static int client6_ifctl __P((char *ifname, u_int16_t));
static void check_exit __P((void));
static void process_signals __P((void));
static struct dhcp6_serverinfo *find_server __P((struct dhcp6_event *,
						 struct duid *));
static struct dhcp6_serverinfo *select_server __P((struct dhcp6_event *));
static void client6_recv __P((void));
static int client6_recvadvert __P((struct dhcp6_if *, struct dhcp6 *,
				   ssize_t, struct dhcp6_optinfo *));
static int client6_recvreply __P((struct dhcp6_if *, struct dhcp6 *,
				  ssize_t, struct dhcp6_optinfo *,
				  struct sockaddr_storage *));
#ifdef DHCPV6_LOGO
	//for UNH-IOL (tom, 20110502)
static int client6_recvreconf __P((struct dhcp6_if *, struct dhcp6 *,
				  ssize_t, struct dhcp6_optinfo *));

extern char reconfig_key[16];//add by rbj
#endif //DHCPV6_LOGO
extern u_int64_t g_prevrd;//add by rbj
#ifdef DHCPV6_LOGO
extern dhcp6_mode_t dhcp6_mode;

#include <sys/types.h>
#include <sys/stat.h>
#endif //DHCPV6_LOGO
#include <fcntl.h>
#ifdef DHCPV6_LOGO
#include <stdarg.h>

#endif //DHCPV6_LOGO


#ifdef DO_DLOG
	#define dlog(args...) _dlog2_(__FILE__,__FUNCTION__,__LINE__,##args)
	void _dlog2_(const char *file, const char *func, int line, const char * format, ...)
	{
	        char *buf;
	        int fd,n;
	        va_list list;
	        fd = open("/dev/console", O_RDWR);
	        buf = (char *)calloc(1,1024);
	        va_start(list, format);
	        sprintf(buf,"[%s:%s:%d]\n\t",file,func,line);
	        vsprintf(buf+strlen(buf), format, list);
	        sprintf(buf+strlen(buf), "%c", '\n');
	        n = write(fd,buf,strlen(buf));
	        va_end(list);
	        free(buf);
	        close(fd);
	}
#else
	#define dlog(args...)
#endif

static void client6_signal __P((int));
static struct dhcp6_event *find_event_withid __P((struct dhcp6_if *,
						  u_int32_t));
static int construct_confdata __P((struct dhcp6_if *, struct dhcp6_event *));
static int construct_reqdata __P((struct dhcp6_if *, struct dhcp6_optinfo *,
    struct dhcp6_event *));
static void destruct_iadata __P((struct dhcp6_eventdata *));
static void tv_sub __P((struct timeval *, struct timeval *, struct timeval *));
static struct dhcp6_timer *client6_expire_refreshtime __P((void *));
static int process_auth __P((struct authparam *, struct dhcp6 *dh6, ssize_t,
    struct dhcp6_optinfo *));
static int set_auth __P((struct dhcp6_event *, struct dhcp6_optinfo *));

struct dhcp6_timer *client6_timo __P((void *));
int client6_start __P((struct dhcp6_if *));
static void info_printf __P((const char *, ...));
static void check_config_file __P((void));
static void output_wan_nexthop __P((struct sockaddr_storage *));

static int read_v6jdg_result __P((void));
static int check_dhcp_status_code __P((struct dhcp6_optinfo *));
static int check_ia_address __P((iatype_t, struct dhcp6_list *));

extern int client6_script __P((char *, int, struct dhcp6_optinfo *));
extern u_int32_t arc4random __P((void));

#define MAX_ELAPSED_TIME 0xffff

extern char *argv_ifname;//rbj
extern int duid_type;//rbj
extern char *oifname;//rbj
extern char *xifname;//rbj
int ra_pfxlen;//rbj
#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
extern int got_ac_ipv6;
extern char str_ac_ipv6[16];
#endif

static int mainloop_end_flg = 0;	/* 1: Solicit or Info-Req retry over */

extern int ngn_pd_mode;
extern int info_req_mode;
extern int duid_retry_cnt;
extern int iana_addr_nothing;
extern int iana_addr_delete;

int
dhcp6c_main(argc, argv)
	int argc;
	char **argv;
{
	int ch, pid;
	char *progname;
	FILE *pidfp;
	struct dhcp6_if *ifp;
	char cmdbuf[64] = "";

	ra_pfxlen = 0;//rbj
	dhcp6_mode = DHCP6_MODE_CLIENT;

#ifndef HAVE_ARC4RANDOM
	srandom(time(NULL) & getpid());
#endif

	if ((progname = strrchr(*argv, '/')) == NULL)
		progname = *argv;
	else
		progname++;

	//while ((ch = getopt(argc, argv, "c:dDfik:p:")) != -1) {
	while ((ch = getopt(argc, argv, "c:dDfik:l:p:t:r:o:n:")) != -1) { //rbj
		switch (ch) {
		case 'c':
			conffile = optarg;
			break;
		case 'd':
			debug = 1;
			break;
		case 'D':
			debug = 2;
			break;
		case 'f':
			foreground++;
			break;
		case 'i':
			infreq_mode = 1;
			break;
		case 'k':
			ctlkeyfile = optarg;
			break;
		case 'l'://rbj
			ra_pfxlen = atoi(optarg);
			//fprintf(stderr,"[%s] ra prefix len is %d\n",__FUNCTION__,ra_pfxlen);//rbj
			break;
		case 'p':
			pid_file = optarg;
			break;
		case 't': //rbj
			//fprintf(stderr,"[%s] check duid type\n",__FUNCTION__);//rbj
			if(strcasecmp(optarg,"LL") == 0)
				duid_type = 3;
			else if(strcasecmp(optarg,"EN") == 0)
				duid_type = 1; //not implement
			else
				duid_type = 1;
			//duid_type = *optarg;
			fprintf(stderr,"[DHCP6C]duid_type: %d\n",duid_type);//rbj
			break;
		case 'r':
			/* DUID switching retry count */
			/*
			 * Notify Info-Req (Solicit) with the DUID specified by "-t"
			 * for the number of times specified by "-r",
			 * and terminate dhcp6c if no response is returned.
			 */
			duid_retry_cnt = atoi(optarg);
			fprintf(stderr,"[DHCP6C]:duid_retry_cnt %d\n",duid_retry_cnt);
			break;
		case  'o'://rbj
			oifname = optarg;
			break;
		case  'n'://rbj
			xifname = optarg;
			//fprintf(stderr,"[DHCP6C]xmldb ifname: %s\n",xifname);//rbj
			break;
		default:
			usage();
			exit(0);
		}
	}
	argc -= optind;
	argv += optind;

	if (argc == 0) {
		usage();
		exit(0);
	}

	if (foreground == 0)
		openlog(progname, LOG_NDELAY|LOG_PID, LOG_DAEMON);

	setloglevel(debug);
	
	argv_ifname = argv[0];//rbj
	//fprintf(stderr,"[%s]argv_ifname: %s\n",__FUNCTION__,argv_ifname);//rbj
	client6_init();
	while (argc-- > 0) { 
		//fprintf(stderr,"[%s]argv: %s\n",__FUNCTION__,argv[0]);//rbj
		if ((ifp = ifinit(argv[0])) == NULL) {
			debug_printf(LOG_ERR, FNAME, "failed to initialize %s",
			    argv[0]);
			exit(1);
		}
		argv++;
	}
	//fprintf(stderr,"[%s]ifname: %s\n",__FUNCTION__,ifp->ifname);//rbj

	if (infreq_mode == 0 && (cfparse(conffile)) != 0) {
		debug_printf(LOG_ERR, FNAME, "failed to parse configuration file");
		exit(1);
	}
#if 1 //traveller add ,do not run in daemon ,can not see debug infomation
	if (foreground == 0 && infreq_mode == 0) {
		if (daemon(0, 0) < 0)
			err(1, "daemon");
	}
#endif

	/* Info-Req and NGN check */
	check_config_file();

	/* dump current PID */
	pid = getpid();
	if ((pidfp = fopen(pid_file, "w")) != NULL) {
		fprintf(pidfp, "%d\n", pid);
		fclose(pidfp);
	}

	client6_startall(0);
	client6_mainloop();

	/* Solicit or Info-Req retry over check */
	if (mainloop_end_flg) {
		/* It notifies termination to "nwctld". */
		if (info_req_mode) {
			/* Info-Req retry over, use_prefix check (on / not change) */
			snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli use_prefix check %d", duid_type);
			system(cmdbuf);
			memset(&cmdbuf, 0, sizeof(cmdbuf));

			snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ng RA %d %s", duid_type, argv_ifname);
		} else if (ngn_pd_mode) {
			snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ng NGNPD %d %s", duid_type, argv_ifname);
		} else {
			snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ng PD %d %s", duid_type, argv_ifname);
		}
		system(cmdbuf);
	}
	/* dhcp6c end code. */
	exit_ok = 1;
	free_resources(NULL);
	unlink(pid_file);
	check_exit();

	exit(0);
}

static void
usage()
{

	fprintf(stderr, "usage: dhcp6c [-c configfile] [-dDfi] "
	    "[-p pid-file] [-t type] [-r retry cnt] [-o interface(for ppp6)] interface [interfaces...]\n");
}

/*------------------------------------------------------------*/

void
client6_init()
{
	struct addrinfo hints, *res;
	static struct sockaddr_in6 sa6_allagent_storage;
	int error, on = 1;
	char *duid_file;

	if (duid_type == 1)
		duid_file = DHCP6C_DUID1_FILE;
	else if (duid_type == 3)
		duid_file = DHCP6C_DUID3_FILE;
	else {
		debug_printf(LOG_ERR, FNAME, "duid type error: duid type=%d",
		    duid_type);
		exit(1);
	}

	/* get our DUID */
	if (get_duid(duid_file, &client_duid)) {
		debug_printf(LOG_ERR, FNAME, "failed to get a DUID");
		exit(1);
	}
	//fprintf(stderr,"[%s]DUID file: %s\n",__FUNCTION__,duid_file);//rbj

	if (dhcp6_ctl_authinit(ctlkeyfile, &ctlkey, &ctldigestlen) != 0) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed initialize control message authentication");
		/* run the server anyway */
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_INET6;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_protocol = IPPROTO_UDP;
	hints.ai_flags = AI_PASSIVE;
	error = getaddrinfo(NULL, DH6PORT_DOWNSTREAM, &hints, &res);
	if (error) {
		debug_printf(LOG_ERR, FNAME, "getaddrinfo: %s",
		    gai_strerror(error));
		exit(1);
	}
	sock = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (sock < 0) {
		debug_printf(LOG_ERR, FNAME, "socket");
		exit(1);
	}
	if (fcntl(sock, F_SETFD, FD_CLOEXEC) == -1) {
		debug_printf(LOG_ERR, FNAME,
		    "fcntl(): %s", strerror(errno));
		exit(1);
	}
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEPORT,
		       &on, sizeof(on)) < 0) {
		debug_printf(LOG_ERR, FNAME,
		    "setsockopt(SO_REUSEPORT): %s", strerror(errno));
		exit(1);
	}
#ifdef IPV6_RECVPKTINFO
	if (setsockopt(sock, IPPROTO_IPV6, IPV6_RECVPKTINFO, &on,
		       sizeof(on)) < 0) {
		debug_printf(LOG_ERR, FNAME,
			"setsockopt(IPV6_RECVPKTINFO): %s",
			strerror(errno));
		exit(1);
	}
#else
	if (setsockopt(sock, IPPROTO_IPV6, IPV6_PKTINFO, &on,
		       sizeof(on)) < 0) {
		debug_printf(LOG_ERR, FNAME,
		    "setsockopt(IPV6_PKTINFO): %s",
		    strerror(errno));
		exit(1);
	}
#endif
	if (setsockopt(sock, IPPROTO_IPV6, IPV6_MULTICAST_LOOP, &on,
		       sizeof(on)) < 0) {
		debug_printf(LOG_ERR, FNAME,
		    "setsockopt(sock, IPV6_MULTICAST_LOOP): %s",
		    strerror(errno));
		exit(1);
	}
#ifdef IPV6_V6ONLY
	if (setsockopt(sock, IPPROTO_IPV6, IPV6_V6ONLY,
	    &on, sizeof(on)) < 0) {
		debug_printf(LOG_ERR, FNAME, "setsockopt(IPV6_V6ONLY): %s",
		    strerror(errno));
		exit(1);
	}
#endif

	/*
	 * According RFC3315 2.2, only the incoming port should be bound to UDP
	 * port 546.  However, to have an interoperability with some servers,
	 * the outgoing port is also bound to the DH6PORT_DOWNSTREAM.
	 */
	if (bind(sock, res->ai_addr, res->ai_addrlen) < 0) {
		debug_printf(LOG_ERR, FNAME, "bind: %s", strerror(errno));
		exit(1);
	}
	freeaddrinfo(res);

	/* open a routing socket to watch the routing table */
	if ((rtsock = socket(PF_ROUTE, SOCK_RAW, 0)) < 0) {
		debug_printf(LOG_ERR, FNAME, "open a routing socket: %s",
		    strerror(errno));
		exit(1);
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_INET6;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_protocol = IPPROTO_UDP;
	error = getaddrinfo(DH6ADDR_ALLAGENT, DH6PORT_UPSTREAM, &hints, &res);
	if (error) {
		debug_printf(LOG_ERR, FNAME, "getaddrinfo: %s",
		    gai_strerror(error));
		exit(1);
	}
	memcpy(&sa6_allagent_storage, res->ai_addr, res->ai_addrlen);
	sa6_allagent = (const struct sockaddr_in6 *)&sa6_allagent_storage;
	freeaddrinfo(res);

	/* set up control socket */
	if (ctlkey == NULL)
		debug_printf(LOG_NOTICE, FNAME, "skip opening control port");
	else if (dhcp6_ctl_init(ctladdr, ctlport,
	    DHCP6CTL_DEF_COMMANDQUEUELEN, &ctlsock)) {
		debug_printf(LOG_ERR, FNAME,
		    "failed to initialize control channel");
		exit(1);
	}

	if (signal(SIGHUP, client6_signal) == SIG_ERR) {
		debug_printf(LOG_WARNING, FNAME, "failed to set signal: %s",
		    strerror(errno));
		exit(1);
	}
	if (signal(SIGTERM, client6_signal) == SIG_ERR) {
		debug_printf(LOG_WARNING, FNAME, "failed to set signal: %s",
		    strerror(errno));
		exit(1);
	}
#ifdef DHCPV6_LOGO
	if (signal(SIGUSR1, client6_signal) == SIG_ERR) {
		debug_printf(LOG_WARNING, FNAME, "failed to set signal: %s",
		    strerror(errno));
		exit(1);
	}
	if (signal(SIGUSR2, client6_signal) == SIG_ERR) {
		debug_printf(LOG_WARNING, FNAME, "failed to set signal: %s",
		    strerror(errno));
		exit(1);
	}
#endif
}

int
client6_start(ifp)
	struct dhcp6_if *ifp;
{
	struct dhcp6_event *ev;

	dlog(" [START] ==[%s]-[%d] ......  \n", __func__, __LINE__);
	
	/* make sure that the interface does not have a timer */
	if (ifp->timer != NULL) {
		debug_printf(LOG_DEBUG, FNAME,
		    "removed existing timer on %s", ifp->ifname);
		dhcp6_remove_timer(&ifp->timer);
	}

	/* create an event for the initial delay */
	if ((ev = dhcp6_create_event(ifp, DHCP6S_INIT)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to create an event");
		return (-1);
	}
	TAILQ_INSERT_TAIL(&ifp->event_list, ev, link);

	if ((ev->authparam = new_authparam(ifp->authproto,
	    ifp->authalgorithm, ifp->authrdm)) == NULL) {
		debug_printf(LOG_WARNING, FNAME, "failed to allocate "
		    "authentication parameters");
		dhcp6_remove_event(ev);
		return (-1);
	}

	if ((ev->timer = dhcp6_add_timer(client6_timo, ev)) == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to add a timer for %s",
		    ifp->ifname);
		dhcp6_remove_event(ev);
		return (-1);
	}
	dhcp6_reset_timer(ev);

	return (0);
}

static void
client6_startall(isrestart)
	int isrestart;
{
	struct dhcp6_if *ifp;

	for (ifp = dhcp6_if; ifp; ifp = ifp->next) {
		if (isrestart &&ifreset(ifp)) {
			debug_printf(LOG_NOTICE, FNAME, "failed to reset %s",
			    ifp->ifname);
			continue; /* XXX: try to recover? */
		}
		if (client6_start(ifp))
			exit(1); /* initialization failure.  we give up. */
	}
}

static void
free_resources(freeifp)
	struct dhcp6_if *freeifp;
{
	struct dhcp6_if *ifp;

	for (ifp = dhcp6_if; ifp; ifp = ifp->next) {
		struct dhcp6_event *ev, *ev_next;

		if (freeifp != NULL && freeifp != ifp)
			continue;

		/* release all IAs as well as send RELEASE message(s) */
		release_all_ia(ifp);

		/*
		 * Cancel all outstanding events for each interface except
		 * ones being released.
		 */
		for (ev = TAILQ_FIRST(&ifp->event_list); ev; ev = ev_next) {
			ev_next = TAILQ_NEXT(ev, link);

			if (ev->state == DHCP6S_RELEASE)
				continue; /* keep it for now */

			dhcp6_remove_event(ev);
		}
	}
}

static void
check_exit()
{
	struct dhcp6_if *ifp;
	int fd, rc, cnt, pid_value;
	char buff[16] = {};

	if (!exit_ok)
		return;

	for (ifp = dhcp6_if; ifp; ifp = ifp->next) {
		/*
		 * Check if we have an outstanding event.  If we do, we cannot
		 * exit for now.
		 */
		if (!TAILQ_EMPTY(&ifp->event_list))
			return;
	}

	/* background shell check */
	fd = open(DHCP6C_SCRIPT_BG_PIDFILE, O_RDONLY);
	if (fd != -1) {
		rc = read(fd, buff, 15);

		if (rc != -1) {
			pid_value = 0;
			for (cnt = 0; cnt < rc; cnt++) {
				pid_value = pid_value * 10 + (buff[cnt] - '0');
			}
			if (pid_value != 0) {
				kill(pid_value, SIGKILL);
			}
		}
		close(fd);
		unlink(DHCP6C_SCRIPT_BG_PIDFILE);
	}

	/* We have no existing event.  Do exit. */
	debug_printf(LOG_INFO, FNAME, "exiting");

	exit(0);
}

#ifdef DHCPV6_LOGO
void confirm_all_ia(struct dhcp6_if *ifp);
static void confirm_all_if(void)
{
	//struct dhcp6_if *ifp = find_ifconfbyname("eth3");
	struct dhcp6_if *ifp = find_ifconfbyname(argv_ifname);//rbj
	if(ifp)
		confirm_all_ia(ifp);
}

static void release_all_if(void)
{
	//struct dhcp6_if *ifp = find_ifconfbyname("eth3");
	struct dhcp6_if *ifp = find_ifconfbyname(argv_ifname);//rbj
	if(ifp)
		release_all_ia(ifp);
}
#endif

static void
process_signals()
{
	if ((sig_flags & SIGF_TERM)) {
		exit_ok = 1;
		free_resources(NULL);
		unlink(pid_file);
		check_exit();
	}
	if ((sig_flags & SIGF_HUP)) {
		debug_printf(LOG_INFO, FNAME, "restarting");
		free_resources(NULL);
		client6_startall(1);
	}

#ifdef DHCPV6_LOGO
	if(sig_flags & SIGF_USR1)
	{
		//send DHCP confirm message
		confirm_all_if();
	}

	if(sig_flags & SIGF_USR2)
	{
		//send DHCP release message
#ifndef DHCPV6_IOL
		release_all_if();
#endif
		system("event WAN.RESTART");//IOL test
	}
#endif

	sig_flags = 0;
}

static void
client6_mainloop()
{
	struct timeval *w;
	int ret, maxsock;
	fd_set r;

	while(1) {
		if (sig_flags)
			process_signals();

		w = dhcp6_check_timer();
		/* Solicit or Info-Req retry over check */
		if (mainloop_end_flg) {
			break;
		}

		FD_ZERO(&r);
		FD_SET(sock, &r);
		maxsock = sock;
		if (ctlsock >= 0) {
			FD_SET(ctlsock, &r);
			maxsock = (sock > ctlsock) ? sock : ctlsock;
			(void)dhcp6_ctl_setreadfds(&r, &maxsock);
		}

		ret = select(maxsock + 1, &r, NULL, NULL, w);

		switch (ret) {
		case -1:
			if (errno != EINTR) {
				debug_printf(LOG_ERR, FNAME, "select: %s",
				    strerror(errno));
				exit(1);
			}
			continue;
		case 0:	/* timeout */
			break;	/* dhcp6_check_timer() will treat the case */
		default:
			break;
		}
		if (FD_ISSET(sock, &r))
			client6_recv();
		if (ctlsock >= 0) {
			if (FD_ISSET(ctlsock, &r)) {
				(void)dhcp6_ctl_acceptcommand(ctlsock,
				    client6_do_ctlcommand);
			}
			(void)dhcp6_ctl_readcommand(&r);
		}
	}
}

static inline int
get_val32(bpp, lenp, valp)
	char **bpp;
	int *lenp;
	u_int32_t *valp;
{
	char *bp = *bpp;
	int len = *lenp;
	u_int32_t i32;

	if (len < sizeof(*valp))
		return (-1);

	memcpy(&i32, bp, sizeof(i32));
	*valp = ntohl(i32);

	*bpp = bp + sizeof(*valp);
	*lenp = len - sizeof(*valp);

	return (0);
}

static inline int
get_ifname(bpp, lenp, ifbuf, ifbuflen)
	char **bpp;
	int *lenp;
	char *ifbuf;
	int ifbuflen;
{
	char *bp = *bpp;
	int len = *lenp, ifnamelen;
	u_int32_t i32;

	if (get_val32(bpp, lenp, &i32))
		return (-1);
	ifnamelen = (int)i32;

	if (*lenp < ifnamelen || ifnamelen > ifbuflen)
		return (-1);

	memset(ifbuf, 0, sizeof(ifbuf));
	memcpy(ifbuf, *bpp, ifnamelen);
	if (ifbuf[ifbuflen - 1] != '\0')
		return (-1);	/* not null terminated */

	*bpp = bp + sizeof(i32) + ifnamelen;
	*lenp = len - (sizeof(i32) + ifnamelen);

	return (0);
}

static int
client6_do_ctlcommand(buf, len)
	char *buf;
	ssize_t len;
{
	struct dhcp6ctl *ctlhead;
	u_int16_t command, version;
	u_int32_t p32, ts, ts0;
	int commandlen;
	char *bp;
	char ifname[IFNAMSIZ];
	time_t now;

	memset(ifname, 0, sizeof(ifname));

	ctlhead = (struct dhcp6ctl *)buf;

	command = ntohs(ctlhead->command);
	commandlen = (int)(ntohs(ctlhead->len));
	version = ntohs(ctlhead->version);
	if (len != sizeof(struct dhcp6ctl) + commandlen) {
		debug_printf(LOG_ERR, FNAME,
		    "assumption failure: command length mismatch");
		return (DHCP6CTL_R_FAILURE);
	}

	/* replay protection and message authentication */
	if ((now = time(NULL)) < 0) {
		debug_printf(LOG_ERR, FNAME, "failed to get current time: %s",
		    strerror(errno));
		return (DHCP6CTL_R_FAILURE);
	}
	ts0 = (u_int32_t)now;
	ts = ntohl(ctlhead->timestamp);
	if (ts + CTLSKEW < ts0 || (ts - CTLSKEW) > ts0) {
		debug_printf(LOG_INFO, FNAME, "timestamp is out of range");
		return (DHCP6CTL_R_FAILURE);
	}

	if (ctlkey == NULL) {	/* should not happen!! */
		debug_printf(LOG_ERR, FNAME, "no secret key for control channel");
		return (DHCP6CTL_R_FAILURE);
	}
	if (dhcp6_verify_mac(buf, len, DHCP6CTL_AUTHPROTO_UNDEF,
	    DHCP6CTL_AUTHALG_HMACMD5, sizeof(*ctlhead), ctlkey) != 0) {
		debug_printf(LOG_INFO, FNAME, "authentication failure");
		return (DHCP6CTL_R_FAILURE);
	}

	bp = buf + sizeof(*ctlhead) + ctldigestlen;
	commandlen -= ctldigestlen;

	if (version > DHCP6CTL_VERSION) {
		debug_printf(LOG_INFO, FNAME, "unsupported version: %d", version);
		return (DHCP6CTL_R_FAILURE);
	}

	switch (command) {
	case DHCP6CTL_COMMAND_RELOAD:
		if (commandlen != 0) {
			debug_printf(LOG_INFO, FNAME, "invalid command length "
			    "for reload: %d", commandlen);
			return (DHCP6CTL_R_DONE);
		}
		client6_reload();
		break;
	case DHCP6CTL_COMMAND_START:
		if (get_val32(&bp, &commandlen, &p32))
			return (DHCP6CTL_R_FAILURE);
		switch (p32) {
		case DHCP6CTL_INTERFACE:
			if (get_ifname(&bp, &commandlen, ifname,
			    sizeof(ifname))) {
				return (DHCP6CTL_R_FAILURE);
			}
			if (client6_ifctl(ifname, DHCP6CTL_COMMAND_START))
				return (DHCP6CTL_R_FAILURE);
			break;
		default:
			debug_printf(LOG_INFO, FNAME,
			    "unknown start target: %ul", p32);
			return (DHCP6CTL_R_FAILURE);
		}
		break;
	case DHCP6CTL_COMMAND_STOP:
		if (commandlen == 0) {
			exit_ok = 1;
			free_resources(NULL);
			unlink(pid_file);
			check_exit();
		} else {
			if (get_val32(&bp, &commandlen, &p32))
				return (DHCP6CTL_R_FAILURE);

			switch (p32) {
			case DHCP6CTL_INTERFACE:
				if (get_ifname(&bp, &commandlen, ifname,
				    sizeof(ifname))) {
					return (DHCP6CTL_R_FAILURE);
				}
				if (client6_ifctl(ifname,
				    DHCP6CTL_COMMAND_STOP)) {
					return (DHCP6CTL_R_FAILURE);
				}
				break;
			default:
				debug_printf(LOG_INFO, FNAME,
				    "unknown start target: %ul", p32);
				return (DHCP6CTL_R_FAILURE);
			}
		}
		break;
	default:
		debug_printf(LOG_INFO, FNAME,
		    "unknown control command: %d (len=%d)",
		    (int)command, commandlen);
		return (DHCP6CTL_R_FAILURE);
	}

  	return (DHCP6CTL_R_DONE);
}

static void
client6_reload()
{
	/* reload the configuration file */
	if (cfparse(conffile) != 0) {
		debug_printf(LOG_WARNING, FNAME,
		    "failed to reload configuration file");
		return;
	}

	debug_printf(LOG_NOTICE, FNAME, "client reloaded");

	return;
}

static int
client6_ifctl(ifname, command)
	char *ifname;
	u_int16_t command;
{
	struct dhcp6_if *ifp;

	if ((ifp = find_ifconfbyname(ifname)) == NULL) {
		debug_printf(LOG_INFO, FNAME,
		    "failed to find interface configuration for %s",
		    ifname);
		return (-1);
	}

	debug_printf(LOG_DEBUG, FNAME, "%s interface %s",
	    command == DHCP6CTL_COMMAND_START ? "start" : "stop", ifname);

	switch(command) {
	case DHCP6CTL_COMMAND_START:
		free_resources(ifp);
		if (client6_start(ifp)) {
			debug_printf(LOG_NOTICE, FNAME, "failed to restart %s",
			    ifname);
			return (-1);
		}
		break;
	case DHCP6CTL_COMMAND_STOP:
		free_resources(ifp);
		if (ifp->timer != NULL) {
			debug_printf(LOG_DEBUG, FNAME,
			    "removed existing timer on %s", ifp->ifname);
			dhcp6_remove_timer(&ifp->timer);
		}
		break;
	default:		/* impossible case, should be a bug */
		debug_printf(LOG_ERR, FNAME, "unknown command: %d", (int)command);
		break;
	}

	return (0);
}

static struct dhcp6_timer *
client6_expire_refreshtime(arg)
	void *arg;
{
	struct dhcp6_if *ifp = arg;

	debug_printf(LOG_DEBUG, FNAME,
	    "information refresh time on %s expired", ifp->ifname);

	dhcp6_remove_timer(&ifp->timer);
	client6_start(ifp);

	return (NULL);
}

struct dhcp6_timer *
client6_timo(arg)
	void *arg;
{
	struct dhcp6_event *ev = (struct dhcp6_event *)arg;
	struct dhcp6_if *ifp;
	int state = ev->state;

	ifp = ev->ifp;
	ev->timeouts++;

	/*
	 * Unless MRC is zero, the message exchange fails once the client has
	 * transmitted the message MRC times.
	 * [RFC3315 14.]
	 */
	if (ev->max_retrans_cnt && ev->timeouts >= ev->max_retrans_cnt) {
		debug_printf(LOG_INFO, FNAME, "no responses were received");
		dhcp6_remove_event(ev);
#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
        got_ac_ipv6 = 0;
        debug_printf(LOG_INFO, FNAME, "AC: timeout reset got_ac_ipv6 to 0\n");
#endif
		if (state == DHCP6S_RELEASE)
			check_exit();

		if ((state == DHCP6S_SOLICIT) || (state == DHCP6S_INFOREQ)) {
			/* Solicit or Info-Req retry over, flg on */
			mainloop_end_flg = 1;
		}

		return (NULL);
	}

#ifdef DHCPV6_LOGO
	if (ev->max_retrans_dur)
	{
		struct timeval now, tv_diff;
		gettimeofday(&now, NULL);
		tv_sub(&now, &ev->tv_start, &tv_diff);

		if(tv_diff.tv_sec * 1000 > ev->max_retrans_dur)
		{
			debug_printf(LOG_INFO, FNAME, "no responses were received");
			dhcp6_remove_event(ev);
			return NULL;
		}
	}
#endif

	switch(ev->state) {
	case DHCP6S_INIT:
		ev->timeouts = 0; /* indicate to generate a new XID. */
		if ((ifp->send_flags & DHCIFF_INFO_ONLY) || infreq_mode)
			ev->state = DHCP6S_INFOREQ;
		else {
			ev->state = DHCP6S_SOLICIT;
#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
			got_ac_ipv6 = 0;
			debug_printf(LOG_ERR, FNAME, "AC: timeout reset got_ac_ipv6 to 0\n");
#endif
			if (construct_confdata(ifp, ev)) {
				debug_printf(LOG_ERR, FNAME, "can't send solicit");
				exit(1); /* XXX */
			}
		}
		
		dlog(" == %s == %d: [case DHCP6S_INIT]: goto dhcp6_set_timeoparam()...", __func__, __LINE__);
		dhcp6_set_timeoparam(ev); /* XXX */
		
		/* fall through */
	case DHCP6S_REQUEST:
	case DHCP6S_RELEASE:
	case DHCP6S_INFOREQ:
		client6_send(ev);
		break;
		
	case DHCP6S_RENEW:
	case DHCP6S_REBIND:
#ifdef DHCPV6_LOGO
	case DHCP6S_CNF:
	case DHCP6S_DECLINE:
#endif
		if (!TAILQ_EMPTY(&ev->data_list))
			client6_send(ev);
		else {
			debug_printf(LOG_INFO, FNAME,
			    "all information to be updated was canceled");
			dhcp6_remove_event(ev);
			return (NULL);
		}
		break;
	case DHCP6S_SOLICIT:
		if (ev->servers) {
			/*
			 * Send a Request to the best server.
			 * Note that when we set Rapid-commit in Solicit,
			 * but a direct Reply has been delayed (very much),
			 * the transition to DHCP6S_REQUEST (and the change of
			 * transaction ID) will invalidate the reply even if it
			 * ever arrives.
			 */
			ev->current_server = select_server(ev);
			if (ev->current_server == NULL) {
				/* this should not happen! */
				debug_printf(LOG_NOTICE, FNAME,
				    "can't find a server");
				exit(1); /* XXX */
			}
			if (duidcpy(&ev->serverid,
			    &ev->current_server->optinfo.serverID)) {
				debug_printf(LOG_NOTICE, FNAME,
				    "can't copy server ID");
				return (NULL); /* XXX: better recovery? */
			}
			ev->timeouts = 0;				
			ev->state = DHCP6S_REQUEST;

			dlog(" == %s == %d: [case DHCP6S_SOLICIT], optinfo.sol_max_rt= %d \n", __func__, __LINE__, 
    	   			             ev->current_server->optinfo.sol_max_rt);		// run here enter dhcp6_set_timeoparam().

			dhcp6_set_timeoparam(ev);

			if (ev->authparam != NULL)
				free(ev->authparam);
			ev->authparam = ev->current_server->authparam;
			ev->current_server->authparam = NULL;

			if (construct_reqdata(ifp,
			    &ev->current_server->optinfo, ev)) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to construct request data");
				break;
			}
		}
		client6_send(ev);
		break;
	}

	dhcp6_reset_timer(ev);

	return (ev->timer);
}

static int
construct_confdata(ifp, ev)
	struct dhcp6_if *ifp;
	struct dhcp6_event *ev;
{
	struct ia_conf *iac;
	struct dhcp6_eventdata *evd = NULL;
	struct dhcp6_list *ial = NULL, pl;
	struct dhcp6_ia iaparam;

	TAILQ_INIT(&pl);	/* for safety */

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac;
	    iac = TAILQ_NEXT(iac, link)) {
		/* ignore IA config currently used */
		if (!TAILQ_EMPTY(&iac->iadata))
			continue;

		evd = NULL;
		if ((evd = malloc(sizeof(*evd))) == NULL) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to create a new event data");
			goto fail;
		}
		memset(evd, 0, sizeof(evd));

		memset(&iaparam, 0, sizeof(iaparam));
		iaparam.iaid = iac->iaid;
		switch (iac->type) {
		case IATYPE_PD:
			ial = NULL;
			if ((ial = malloc(sizeof(*ial))) == NULL)
				goto fail;
			TAILQ_INIT(ial);

			TAILQ_INIT(&pl);
			dhcp6_copy_list(&pl,
			    &((struct iapd_conf *)iac)->iapd_prefix_list);
			if (dhcp6_add_listval(ial, DHCP6_LISTVAL_IAPD,
			    &iaparam, &pl) == NULL) {
				goto fail;
			}
			dhcp6_clear_list(&pl);

			evd->type = DHCP6_EVDATA_IAPD;
			evd->data = ial;
			evd->event = ev;
			evd->destructor = destruct_iadata;
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);
			break;
		case IATYPE_NA:
			ial = NULL;
			if ((ial = malloc(sizeof(*ial))) == NULL)
				goto fail;
			TAILQ_INIT(ial);

			TAILQ_INIT(&pl);
			dhcp6_copy_list(&pl,
			    &((struct iana_conf *)iac)->iana_address_list);
			if (dhcp6_add_listval(ial, DHCP6_LISTVAL_IANA,
			    &iaparam, &pl) == NULL) {
				goto fail;
			}
			dhcp6_clear_list(&pl);

			evd->type = DHCP6_EVDATA_IANA;
			evd->data = ial;
			evd->event = ev;
			evd->destructor = destruct_iadata;
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);
			break;
		default:
			debug_printf(LOG_ERR, FNAME, "internal error");
			exit(1);
		}
	}

	return (0);

  fail:
	if (evd)
		free(evd);
	if (ial)
		free(ial);
	dhcp6_remove_event(ev);	/* XXX */
	
	return (-1);
}

static int
construct_reqdata(ifp, optinfo, ev)
	struct dhcp6_if *ifp;
	struct dhcp6_optinfo *optinfo;
	struct dhcp6_event *ev;
{
	struct ia_conf *iac;
	struct dhcp6_eventdata *evd = NULL;
	struct dhcp6_list *ial = NULL;
	struct dhcp6_ia iaparam;

	/* discard previous event data */
	dhcp6_remove_evdata(ev);

	if (optinfo == NULL)
		return (0);

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac;
	    iac = TAILQ_NEXT(iac, link)) {
		struct dhcp6_listval *v;

		/* ignore IA config currently used */
		if (!TAILQ_EMPTY(&iac->iadata))
			continue;

		memset(&iaparam, 0, sizeof(iaparam));
		iaparam.iaid = iac->iaid;

		ial = NULL;
		evd = NULL;

		switch (iac->type) {
		case IATYPE_PD:
			if ((v = dhcp6_find_listval(&optinfo->iapd_list,
			    DHCP6_LISTVAL_IAPD, &iaparam, 0)) == NULL)
				continue;

			if ((ial = malloc(sizeof(*ial))) == NULL)
				goto fail;

			TAILQ_INIT(ial);
			if (dhcp6_add_listval(ial, DHCP6_LISTVAL_IAPD,
			    &iaparam, &v->sublist) == NULL) {
				goto fail;
			}

			if ((evd = malloc(sizeof(*evd))) == NULL)
				goto fail;
			memset(evd, 0, sizeof(*evd));
			evd->type = DHCP6_EVDATA_IAPD;
			evd->data = ial;
			evd->event = ev;
			evd->destructor = destruct_iadata;
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);
			break;
		case IATYPE_NA:
			if ((v = dhcp6_find_listval(&optinfo->iana_list,
			    DHCP6_LISTVAL_IANA, &iaparam, 0)) == NULL)
				continue;

			if ((ial = malloc(sizeof(*ial))) == NULL)
				goto fail;

			TAILQ_INIT(ial);
			if (dhcp6_add_listval(ial, DHCP6_LISTVAL_IANA,
			    &iaparam, &v->sublist) == NULL) {
				goto fail;
			}

			if ((evd = malloc(sizeof(*evd))) == NULL)
				goto fail;
			memset(evd, 0, sizeof(*evd));
			evd->type = DHCP6_EVDATA_IANA;
			evd->data = ial;
			evd->event = ev;
			evd->destructor = destruct_iadata;
			TAILQ_INSERT_TAIL(&ev->data_list, evd, link);
			break;
		default:
			debug_printf(LOG_ERR, FNAME, "internal error");
			exit(1);
		}
	}

	return (0);

  fail:
	if (evd)
		free(evd);
	if (ial)
		free(ial);
	dhcp6_remove_event(ev);	/* XXX */
	
	return (-1);
}

static void
destruct_iadata(evd)
	struct dhcp6_eventdata *evd;
{
	struct dhcp6_list *ial;

	if (evd->type != DHCP6_EVDATA_IAPD && evd->type != DHCP6_EVDATA_IANA) {
		debug_printf(LOG_ERR, FNAME, "assumption failure %d", evd->type);
		exit(1);
	}

	ial = (struct dhcp6_list *)evd->data;
	dhcp6_clear_list(ial);
	free(ial);
}

static struct dhcp6_serverinfo *
select_server(ev)
	struct dhcp6_event *ev;
{
	struct dhcp6_serverinfo *s;

	/*
	 * pick the best server according to RFC3315 Section 17.1.3.
	 * XXX: we currently just choose the one that is active and has the
	 * highest preference.
	 */
	for (s = ev->servers; s; s = s->next) {
		if (s->active) {
			debug_printf(LOG_DEBUG, FNAME, "picked a server (ID: %s)",
			    duidstr(&s->optinfo.serverID));
			return (s);
		}
	}

	return (NULL);
}

static void
client6_signal(sig)
	int sig;
{

	switch (sig) {
	case SIGTERM:
		sig_flags |= SIGF_TERM;
		break;
	case SIGHUP:
		sig_flags |= SIGF_HUP;
		break;
#ifdef DHCPV6_LOGO
	case SIGUSR1:
		sig_flags |= SIGF_USR1;
		break;
	case SIGUSR2:
		sig_flags |= SIGF_USR2;
		break;
#endif
	}
}

void
client6_send(ev)
	struct dhcp6_event *ev;
{
	struct dhcp6_if *ifp;
	char buf[BUFSIZ];
	struct sockaddr_in6 dst;
	struct dhcp6 *dh6;
	struct dhcp6_optinfo optinfo;
	ssize_t optlen, len;
	struct dhcp6_eventdata *evd;

	ifp = ev->ifp;

	dh6 = (struct dhcp6 *)buf;
	memset(dh6, 0, sizeof(*dh6));

	switch(ev->state) {
	case DHCP6S_SOLICIT:
		dh6->dh6_msgtype = DH6_SOLICIT;
		break;
	case DHCP6S_REQUEST:
		dh6->dh6_msgtype = DH6_REQUEST;
		break;
	case DHCP6S_RENEW:
		dh6->dh6_msgtype = DH6_RENEW;
		break;
	case DHCP6S_REBIND:
		dh6->dh6_msgtype = DH6_REBIND;
		break;
	case DHCP6S_RELEASE:
		dh6->dh6_msgtype = DH6_RELEASE;
		break;
	case DHCP6S_INFOREQ:
		dh6->dh6_msgtype = DH6_INFORM_REQ;
		break;
#ifdef DHCPV6_LOGO
	case DHCP6S_CNF:
		dh6->dh6_msgtype = DH6_CONFIRM;
		break;
	case DHCP6S_DECLINE:
		dh6->dh6_msgtype = DH6_DECLINE;
		break;
#endif
	default:
		debug_printf(LOG_ERR, FNAME, "unexpected state");
		exit(1);	/* XXX */
	}

	if (ev->timeouts == 0) {
		/*
		 * A client SHOULD generate a random number that cannot easily
		 * be guessed or predicted to use as the transaction ID for
		 * each new message it sends.
		 *
		 * A client MUST leave the transaction-ID unchanged in
		 * retransmissions of a message. [RFC3315 15.1]
		 */
#ifdef HAVE_ARC4RANDOM
		ev->xid = arc4random() & DH6_XIDMASK;
#else
		ev->xid = random() & DH6_XIDMASK;
#endif
		debug_printf(LOG_DEBUG, FNAME, "a new XID (%x) is generated",
		    ev->xid);
	}
	dh6->dh6_xid &= ~ntohl(DH6_XIDMASK);
	dh6->dh6_xid |= htonl(ev->xid);
	len = sizeof(*dh6);

	/*
	 * construct options
	 */
	dhcp6_init_options(&optinfo);
#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
	if (ev->state == DHCP6S_SOLICIT|| ev->state ==DHCP6S_REQUEST)
		optinfo.enterprise_code = DHCP_ENTERPRISE_CODE;
#endif
	/* server ID */
	switch (ev->state) {
	case DHCP6S_REQUEST:
	case DHCP6S_RENEW:
	case DHCP6S_RELEASE:
#ifdef DHCPV6_LOGO
	case DHCP6S_DECLINE:
#endif
		if (duidcpy(&optinfo.serverID, &ev->serverid)) {
			debug_printf(LOG_ERR, FNAME, "failed to copy server ID");
			goto end;
		}
		break;
	}

	/* client ID */
	if (duidcpy(&optinfo.clientID, &client_duid)) {
		debug_printf(LOG_ERR, FNAME, "failed to copy client ID");
		goto end;
	}

	/* rapid commit (in Solicit only) */
	if (ev->state == DHCP6S_SOLICIT &&
	    (ifp->send_flags & DHCIFF_RAPID_COMMIT)) {
		optinfo.rapidcommit = 1;
	}

	/* elapsed time */
	if (ev->timeouts == 0) {
		gettimeofday(&ev->tv_start, NULL);
		optinfo.elapsed_time = 0;
	} else {
		struct timeval now, tv_diff;
		long et;

		gettimeofday(&now, NULL);
		tv_sub(&now, &ev->tv_start, &tv_diff);

		/*
		 * The client uses the value 0xffff to represent any elapsed
		 * time values greater than the largest time value that can be
		 * represented in the Elapsed Time option.
		 * [RFC3315 22.9.]
		 */
		if (tv_diff.tv_sec >= (MAX_ELAPSED_TIME / 100) + 1) {
			/*
			 * Perhaps we are nervous too much, but without this
			 * additional check, we would see an overflow in 248
			 * days (of no responses). 
			 */
			et = MAX_ELAPSED_TIME;
		} else {
			et = tv_diff.tv_sec * 100 + tv_diff.tv_usec / 10000;
			if (et >= MAX_ELAPSED_TIME)
				et = MAX_ELAPSED_TIME;
		}
		optinfo.elapsed_time = (int32_t)et;
	}

	/* option request options */
	if (ev->state != DHCP6S_RELEASE &&
	    dhcp6_copy_list(&optinfo.reqopt_list, &ifp->reqopt_list)) {
		debug_printf(LOG_ERR, FNAME, "failed to copy requested options");
		goto end;
	}

	/* configuration information specified as event data */
	for (evd = TAILQ_FIRST(&ev->data_list); evd;
	     evd = TAILQ_NEXT(evd, link)) {
		switch(evd->type) {
		case DHCP6_EVDATA_IAPD:
			if (dhcp6_copy_list(&optinfo.iapd_list,
			    (struct dhcp6_list *)evd->data)) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to add an IAPD");
				goto end;
			}
			break;
		case DHCP6_EVDATA_IANA:
			if (dhcp6_copy_list(&optinfo.iana_list,
			    (struct dhcp6_list *)evd->data)) {
				debug_printf(LOG_NOTICE, FNAME,
				    "failed to add an IAPD");
				goto end;
			}
			break;
		default:
			debug_printf(LOG_ERR, FNAME, "unexpected event data (%d)",
			    evd->type);
			exit(1);
		}
	}

	/* authentication information */
	if (set_auth(ev, &optinfo)) {
		debug_printf(LOG_INFO, FNAME,
		    "failed to set authentication option");
		goto end;
	}

	/* set options in the message */
	if ((optlen = dhcp6_set_options(dh6->dh6_msgtype,
	    (struct dhcp6opt *)(dh6 + 1),
	    (struct dhcp6opt *)(buf + sizeof(buf)), &optinfo)) < 0) {
		debug_printf(LOG_INFO, FNAME, "failed to construct options");
		goto end;
	}
	len += optlen;
	/* calculate MAC if necessary, and put it to the message */
	if (ev->authparam != NULL) {
		switch (ev->authparam->authproto) {
		case DHCP6_AUTHPROTO_DELAYED:
			if (ev->authparam->key == NULL)
				break;

			if (dhcp6_calc_mac((char *)dh6, len,
			    optinfo.authproto, optinfo.authalgorithm,
			    optinfo.delayedauth_offset + sizeof(*dh6),
			    ev->authparam->key)) {
				debug_printf(LOG_WARNING, FNAME,
				    "failed to calculate MAC");
				goto end;
			}
			break;
		default:
			break;	/* do nothing */
		}
	}

	/*
	 * Unless otherwise specified in this document or in a document that
	 * describes how IPv6 is carried over a specific type of link (for link
	 * types that do not support multicast), a client sends DHCP messages
	 * to the All_DHCP_Relay_Agents_and_Servers.
	 * [RFC3315 Section 13.]
	 */
	dst = *sa6_allagent;
	dst.sin6_scope_id = ifp->linkid;

	if (sendto(sock, buf, len, 0, (struct sockaddr *)&dst,
	    sysdep_sa_len((struct sockaddr *)&dst)) == -1) {
		debug_printf(LOG_ERR, FNAME,
		    "transmit failed: %s", strerror(errno));
	debug_printf(LOG_DEBUG, FNAME, "send %s to %s error",
	    dhcp6msgstr(dh6->dh6_msgtype), addr2str((struct sockaddr *)&dst));
		goto end;
	}

	debug_printf(LOG_DEBUG, FNAME, "send %s to %s",
	    dhcp6msgstr(dh6->dh6_msgtype), addr2str((struct sockaddr *)&dst));

  end:
	dhcp6_clear_options(&optinfo);
	return;
}

/* result will be a - b */
static void
tv_sub(a, b, result)
	struct timeval *a, *b, *result;
{
	if (a->tv_sec < b->tv_sec ||
	    (a->tv_sec == b->tv_sec && a->tv_usec < b->tv_usec)) {
		result->tv_sec = 0;
		result->tv_usec = 0;

		return;
	}

	result->tv_sec = a->tv_sec - b->tv_sec;
	if (a->tv_usec < b->tv_usec) {
		result->tv_usec = a->tv_usec + 1000000 - b->tv_usec;
		result->tv_sec -= 1;
	} else
		result->tv_usec = a->tv_usec - b->tv_usec;

	return;
}

static void
client6_recv()
{
	char rbuf[BUFSIZ], cmsgbuf[BUFSIZ];
	struct msghdr mhdr;
	struct iovec iov;
	struct sockaddr_storage from;
	struct dhcp6_if *ifp;
	struct dhcp6opt *p, *ep;
	struct dhcp6_optinfo optinfo;
	ssize_t len;
	struct dhcp6 *dh6;
	struct cmsghdr *cm;
	struct in6_pktinfo *pi = NULL;

	int pd_ra_mode;
	int dhcpv6_stcode = DH6OPT_STCODE_SUCCESS;
	int ia_na_flg = 0;	/* 0:IA-NA Address nothing */
	int ia_pd_flg = 0;	/* 0:IA-PD Prefix nothing */
	char cmdbuf[64] = "";

	dlog("  ****** Enter client6_recv [Advert] -> [reply] ......\n");

	memset(&iov, 0, sizeof(iov));
	memset(&mhdr, 0, sizeof(mhdr));

	iov.iov_base = (caddr_t)rbuf;
	iov.iov_len = sizeof(rbuf);
	mhdr.msg_name = (caddr_t)&from;
	mhdr.msg_namelen = sizeof(from);
	mhdr.msg_iov = &iov;
	mhdr.msg_iovlen = 1;
	mhdr.msg_control = (caddr_t)cmsgbuf;
	mhdr.msg_controllen = sizeof(cmsgbuf);
	if ((len = recvmsg(sock, &mhdr, 0)) < 0) {
		debug_printf(LOG_ERR, FNAME, "recvmsg: %s", strerror(errno));
		return;
	}

	/* detect receiving interface */
	for (cm = (struct cmsghdr *)CMSG_FIRSTHDR(&mhdr); cm;
	     cm = (struct cmsghdr *)CMSG_NXTHDR(&mhdr, cm)) {
		if (cm->cmsg_level == IPPROTO_IPV6 &&
		    cm->cmsg_type == IPV6_PKTINFO &&
		    cm->cmsg_len == CMSG_LEN(sizeof(struct in6_pktinfo))) {
			pi = (struct in6_pktinfo *)(CMSG_DATA(cm));
		}
	}
	if (pi == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to get packet info");
		return;
	}

	if ((ifp = find_ifconfbyid((unsigned int)pi->ipi6_ifindex)) == NULL) {
		debug_printf(LOG_INFO, FNAME, "unexpected interface (%d)",
		    (unsigned int)pi->ipi6_ifindex);
		return;
	}

	if (len < sizeof(*dh6)) {
		debug_printf(LOG_INFO, FNAME, "short packet (%d bytes)", len);
		return;
	}

	dh6 = (struct dhcp6 *)rbuf;

	debug_printf(LOG_DEBUG, FNAME, "receive %s from %s on %s",
	    dhcp6msgstr(dh6->dh6_msgtype),
	    addr2str((struct sockaddr *)&from), ifp->ifname);

	/* get options */
	dhcp6_init_options(&optinfo);
	p = (struct dhcp6opt *)(dh6 + 1);
	ep = (struct dhcp6opt *)((char *)dh6 + len);
	
	if (dhcp6_get_options(p, ep, &optinfo) < 0) {
		debug_printf(LOG_INFO, FNAME, "failed to parse options");
		return;
	}

	/* v6jdg.result get*/
	pd_ra_mode = read_v6jdg_result();

	/* Status code check */
	if (!TAILQ_EMPTY(&optinfo.stcode_list)) {
		/*
		 * If the Status Code option does not appear in a message
		 * in which the option could appear,
		 * the status of the message is assumed to be Success.
		 * [RFC 8415 Section 21.13 Status Code Option]
		 */
		dhcpv6_stcode = check_dhcp_status_code(&optinfo);

		/* NOADDRSAVAIL & NOPREFIXAVAIL -> IA-NA / IA-PD check */
		if ((dhcpv6_stcode != DH6OPT_STCODE_SUCCESS) &&
		    (dhcpv6_stcode != DH6OPT_STCODE_NOADDRSAVAIL) &&
		    (dhcpv6_stcode != DH6OPT_STCODE_NOPREFIXAVAIL)) {
			debug_printf(LOG_ERR, FNAME, "Status code error(%d)", dhcpv6_stcode);
			goto dhcp6_clear;
		}
	}

	/* IA-NA / IA-PD check */
	if (pd_ra_mode != V6_NDP) {
		/* IA-NA check and Status code check */
		if (dhcpv6_stcode != DH6OPT_STCODE_NOADDRSAVAIL) {
			if (!TAILQ_EMPTY(&optinfo.iana_list)) {
				ia_na_flg = check_ia_address(IATYPE_NA, &optinfo.iana_list);
			}
		}

		/* IA-PD check and Status code check */
		if (dhcpv6_stcode != DH6OPT_STCODE_NOPREFIXAVAIL) {
			if (!TAILQ_EMPTY(&optinfo.iapd_list)) {
				ia_pd_flg = check_ia_address(IATYPE_PD, &optinfo.iapd_list);
			}
		}

		/* MODE=judging check later */
		if ((pd_ra_mode != V6_JUDGING) && (ia_na_flg == 0) && (ia_pd_flg == 0)) {
			debug_printf(LOG_NOTICE, FNAME, "IA-PD and IA-NA nothing.");
			goto dhcp6_clear;
		}
	}

	/* MODE=judging Advertise/Reply Error -> nwctld notice */
	if (pd_ra_mode == V6_JUDGING) {
		/* IA-NA / IA-PD check */
		if ((ia_na_flg == 0) && (ia_pd_flg == 0)) {
			if (ngn_pd_mode) {
				snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ng NGNPD %d %s", duid_type, ifp->ifname);
			} else {
				snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ng PD %d %s", duid_type, ifp->ifname);
			}

			system(cmdbuf);

			exit_ok = 1;
			free_resources(NULL);
			unlink(pid_file);
			check_exit();
		}
	}

	switch(dh6->dh6_msgtype) 
	{
	case DH6_ADVERTISE:
		(void)client6_recvadvert(ifp, dh6, len, &optinfo);
		break;
#ifdef DHCPV6_LOGO
	//for UNH-IOL (tom, 20110502)
	case DH6_RECONFIGURE:
		dlog("before client6_recvreconf\n");
		(void)client6_recvreconf(ifp, dh6, len, &optinfo);
		break;
#endif //DHCPV6_LOGO
	case DH6_REPLY:
		(void)client6_recvreply(ifp, dh6, len, &optinfo, &from);
		break;
	default:
		debug_printf(LOG_INFO, FNAME, "received an unexpected message (%s) "
		    "from %s", dhcp6msgstr(dh6->dh6_msgtype),
		    addr2str((struct sockaddr *)&from));
		break;
	}

dhcp6_clear:
	dhcp6_clear_options(&optinfo);
	return;
}

static int
client6_recvadvert(ifp, dh6, len, optinfo)
	struct dhcp6_if *ifp;
	struct dhcp6 *dh6;
	ssize_t len;
	struct dhcp6_optinfo *optinfo;
{
	struct dhcp6_serverinfo *newserver, **sp;
	struct dhcp6_event *ev;						//Defined in config.h
	struct dhcp6_eventdata *evd;
	struct authparam *authparam = NULL, authparam0;

	/* find the corresponding event based on the received xid */
	ev = find_event_withid(ifp, ntohl(dh6->dh6_xid) & DH6_XIDMASK);
	if (ev == NULL) {
		debug_printf(LOG_INFO, FNAME, "XID mismatch");
		return (-1);
	}

	/* packet validation based on Section 15.3 of RFC3315. */
	if (optinfo->serverID.duid_len == 0) {
		debug_printf(LOG_INFO, FNAME, "no server ID option");
		return (-1);
	} else {
		debug_printf(LOG_DEBUG, FNAME, "server ID: %s, pref=%d",
		    duidstr(&optinfo->serverID),
		    optinfo->pref);
	}
	if (optinfo->clientID.duid_len == 0) {
		debug_printf(LOG_INFO, FNAME, "no client ID option");
		return (-1);
	}
	if (duidcmp(&optinfo->clientID, &client_duid)) {
		debug_printf(LOG_INFO, FNAME, "client DUID mismatch");
		return (-1);
	}

	/* validate authentication */
	authparam0 = *ev->authparam;
	if (process_auth(&authparam0, dh6, len, optinfo)) {
		debug_printf(LOG_INFO, FNAME, "failed to process authentication");
		return (-1);
	}

	/*
	 * The requesting router MUST ignore any Advertise message that
	 * includes a Status Code option containing the value NoPrefixAvail
	 * [RFC3633 Section 11.1].
	 * Likewise, the client MUST ignore any Advertise message that includes
	 * a Status Code option containing the value NoAddrsAvail. 
	 * [RFC3315 Section 17.1.3].
	 * We only apply this when we are going to request an address or
	 * a prefix.
	 */
	/* Since the Status Code is confirmed by "client6_recv()", skip it. */
#if 0	/* skip */

#if 1 // HuanYao, Bug fix for IANA not available. 2015/03/25
	unsigned char have_pd = 0, iana_fail = 0;
#endif
	for (evd = TAILQ_FIRST(&ev->data_list); evd;
	    evd = TAILQ_NEXT(evd, link)) {
		u_int16_t stcode;
		char *stcodestr;
		struct dhcp6_listval *iana, *iapd;

		switch (evd->type) {
		case DHCP6_EVDATA_IAPD:
			stcode = DH6OPT_STCODE_NOPREFIXAVAIL;
			stcodestr = "NoPrefixAvail";
#if 1 // HuanYao, Bug fix for IANA not available. 2015/03/25
			have_pd = 1;
#endif
#ifdef DHCPV6_LOGO
			//we should check status code in sublist too
			TAILQ_FOREACH(iapd , &optinfo->iapd_list , link)
			{
				if(dhcp6_find_listval(&iapd->sublist , DHCP6_LISTVAL_STCODE , &stcode , 0))
				{
					debug_printf(LOG_INFO, FNAME, "advertise contains %s status", stcodestr);
					dlog("advertise contains %s status", stcodestr);
					return (-1);
				}
			}
#endif
			break;
		case DHCP6_EVDATA_IANA:
			stcode = DH6OPT_STCODE_NOADDRSAVAIL;
			stcodestr = "NoAddrsAvail";
/* Because we need to go on the process if we got the IAPD */
#if 1
#ifdef DHCPV6_LOGO
			//we should check status code in sublist too
			TAILQ_FOREACH(iana , &optinfo->iana_list , link)
			{
				if(dhcp6_find_listval(&iana->sublist , DHCP6_LISTVAL_STCODE , &stcode , 0))
				{
					debug_printf(LOG_INFO, FNAME, "advertise contains %s status", stcodestr);
					dlog("advertise contains %s status", stcodestr);
#if 1 // HuanYao, Bug fix for IANA not available. 2015/03/25
					iana_fail = 1;
#endif
				}
			}
#endif
#endif
			break;
		default:
			continue;
		}
		if (dhcp6_find_listval(&optinfo->stcode_list,
		    DHCP6_LISTVAL_STCODE, &stcode, 0)) {
			debug_printf(LOG_INFO, FNAME,
			    "advertise contains %s status", stcodestr);
			return (-1);
		}
	}
#if 1 // HuanYao, Bug fix for IANA not available. 2015/03/25
	if (have_pd == 0 && iana_fail == 1)
	{
		return (-1);
	}
#endif
#endif	/* skip */

	if (ev->state != DHCP6S_SOLICIT ||
	    (ifp->send_flags & DHCIFF_RAPID_COMMIT) || infreq_mode) {
		/*
		 * We expected a reply message, but do actually receive an
		 * Advertise message.  The server should be configured not to
		 * allow the Rapid Commit option.
		 * We process the message as if we expected the Advertise.
		 * [RFC3315 Section 17.1.4]
		 */
		debug_printf(LOG_INFO, FNAME, "unexpected advertise");
		/* proceed anyway */
	}

	/* ignore the server if it is known */
	if (find_server(ev, &optinfo->serverID)) {
		debug_printf(LOG_INFO, FNAME, "duplicated server (ID: %s)",
		    duidstr(&optinfo->serverID));
		return (-1);
	}

	/* keep the server */
	if ((newserver = malloc(sizeof(*newserver))) == NULL) {
		debug_printf(LOG_WARNING, FNAME,
		    "memory allocation failed for server");
		return (-1);
	}
	memset(newserver, 0, sizeof(*newserver));

	/* remember authentication parameters */
	newserver->authparam = ev->authparam;
	newserver->authparam->flags = authparam0.flags;
	newserver->authparam->prevrd = authparam0.prevrd;
	newserver->authparam->key = authparam0.key;

	/* allocate new authentication parameter for the soliciting event */
	if ((authparam = new_authparam(ev->authparam->authproto,
	    ev->authparam->authalgorithm, ev->authparam->authrdm)) == NULL) {
		debug_printf(LOG_WARNING, FNAME, "memory allocation failed "
		    "for authentication parameters");
		free(newserver);
		return (-1);
	}
	ev->authparam = authparam;

	/* copy options */
	dhcp6_init_options(&newserver->optinfo);
	
	if (dhcp6_copy_options(&newserver->optinfo, optinfo)) {
		debug_printf(LOG_ERR, FNAME, "failed to copy options");
		if (newserver->authparam != NULL)
			free(newserver->authparam);
		free(newserver);
		return (-1);
	}
	if (optinfo->pref != DH6OPT_PREF_UNDEF)
		newserver->pref = optinfo->pref;
	newserver->active = 1;
	for (sp = &ev->servers; *sp; sp = &(*sp)->next) {
		if ((*sp)->pref != DH6OPT_PREF_MAX &&
		    (*sp)->pref < newserver->pref) {
			break;
		}
	}
	newserver->next = *sp;
	*sp = newserver;

	if (newserver->pref == DH6OPT_PREF_MAX) {
		/*
		 * If the client receives an Advertise message that includes a
		 * Preference option with a preference value of 255, the client
		 * immediately begins a client-initiated message exchange.
		 * [RFC3315 Section 17.1.2]
		 */
		ev->current_server = newserver;
		if (duidcpy(&ev->serverid,
		    &ev->current_server->optinfo.serverID)) {
			debug_printf(LOG_NOTICE, FNAME, "can't copy server ID");
			return (-1); /* XXX: better recovery? */
		}
		if (construct_reqdata(ifp, &ev->current_server->optinfo, ev)) {
			debug_printf(LOG_NOTICE, FNAME,
			    "failed to construct request data");
			return (-1); /* XXX */
		}

		ev->timeouts = 0;
		ev->state = DHCP6S_REQUEST;

		free(ev->authparam);
		ev->authparam = newserver->authparam;
		newserver->authparam = NULL;

		client6_send(ev);

		dhcp6_set_timeoparam(ev);
		dhcp6_reset_timer(ev);
	} else if (ev->servers->next == NULL) {
		struct timeval *rest, elapsed, tv_rt, tv_irt, timo;

		/*
		 * If this is the first advertise, adjust the timer so that
		 * the client can collect other servers until IRT elapses.
		 * XXX: we did not want to do such "low level" timer
		 *      calculation here.
		 */
		rest = dhcp6_timer_rest(ev->timer);
		tv_rt.tv_sec = (ev->retrans * 1000) / 1000000;
		tv_rt.tv_usec = (ev->retrans * 1000) % 1000000;
		tv_irt.tv_sec = (ev->init_retrans * 1000) / 1000000;
		tv_irt.tv_usec = (ev->init_retrans * 1000) % 1000000;
		timeval_sub(&tv_rt, rest, &elapsed);
		if (TIMEVAL_LEQ(elapsed, tv_irt))
			timeval_sub(&tv_irt, &elapsed, &timo);
		else
			timo.tv_sec = timo.tv_usec = 0;

		debug_printf(LOG_DEBUG, FNAME, "reset timer for %s to %d.%06d",
		    ifp->ifname, (int)timo.tv_sec, (int)timo.tv_usec);

		dhcp6_set_timer(&timo, ev->timer);
	}
	
#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
    if (optinfo->enterprise_code == DHCP_ENTERPRISE_CODE)
    {
        char zeros[16];
        memset(zeros, 0, 16);
        if (0 != memcmp(zeros, optinfo->ac_ipv6, 16))
        {
            got_ac_ipv6 = 1;
            memcpy(str_ac_ipv6, optinfo->ac_ipv6, 16);
            debug_printf(LOG_DEBUG, FNAME, "AC: set ac's ipv6 addr from advertise to str_ac_ipv6\n");
        }
    }
#endif
	return (0);
}

static struct dhcp6_serverinfo *
find_server(ev, duid)
	struct dhcp6_event *ev;
	struct duid *duid;
{
	struct dhcp6_serverinfo *s;

	for (s = ev->servers; s; s = s->next) {
		if (duidcmp(&s->optinfo.serverID, duid) == 0)
			return (s);
	}

	return (NULL);
}

#ifdef DHCPV6_LOGO
static int all_iana_in_reply(struct dhcp6_event *ev , struct dhcp6_list *reply_ianalist)
{
	struct dhcp6_eventdata *evd;

	TAILQ_FOREACH(evd , &ev->data_list , link)
	{
		if(evd->type == DHCP6_EVDATA_IANA)
		{
			//check our current requests in list
			struct dhcp6_list *req_ianalist = (struct dhcp6_list*)evd->data;
			struct dhcp6_listval *iav;
						
			TAILQ_FOREACH(iav , req_ianalist , link)
			{
				//console_printf(LOC_FMT"searching IANA: %d\n" , LOC_ARG , iav->val_ia.iaid);
				if(dhcp6_find_listval(reply_ianalist , DHCP6_LISTVAL_IANA , &iav->val_ia , 0) == NULL)
					return 0;
			}
		}
	}

	return 1;
}

void reset_ia_timer(struct dhcp6_listval *iav , iatype_t iatype , struct dhcp6_if *ifp , struct duid *serverid);
static int scan_all_iapds(struct dhcp6_event *ev , struct dhcp6_list *reply_iapdlist , struct dhcp6_if *ifp , struct duid *serverid)
{
	struct dhcp6_eventdata *evd;
	int found = 1;

	TAILQ_FOREACH(evd , &ev->data_list , link)
	{
		if(evd->type == DHCP6_EVDATA_IAPD)
		{
			//check our current requests in list
			struct dhcp6_list *req_iapdlist = (struct dhcp6_list*)evd->data;
			struct dhcp6_listval *iav;

			TAILQ_FOREACH(iav , req_iapdlist , link)
			{
				//console_printf(LOC_FMT"searching IAPD: %d\n" , LOC_ARG , iav->val_ia.iaid);
				if(dhcp6_find_listval(reply_iapdlist , DHCP6_LISTVAL_IAPD , &iav->val_ia , 0) == NULL)
				{
					reset_ia_timer(iav , IATYPE_PD , ifp , serverid);
					found = 0;
				}
			}
		}
	}

	return found;
}

static void release_ia_in_event(struct dhcp6_event *ev , struct dhcp6_if *ifp , struct duid *serverid)
{
	struct dhcp6_eventdata *evd;

	TAILQ_FOREACH(evd , &ev->data_list , link)
	{
		struct dhcp6_list *ianalist = (struct dhcp6_list*)evd->data;

		if(evd->type == DHCP6_EVDATA_IANA)
			release_choose_ia(IATYPE_NA, ianalist, ifp, serverid);
	}
}

//add by rbj
void 
send_info_req(ifp)
	struct dhcp6_if *ifp;
{
	struct dhcp6_event *ev;
	int dhcpstate;

	dlog("enter send_info_req\n");
	dhcpstate = DHCP6S_INFOREQ;

        if ((ev = dhcp6_create_event(ifp, dhcpstate)) == NULL) {
                debug_printf(LOG_NOTICE, FNAME, "failed to create a new event");
                goto fail;
        }
	dhcp6_set_timeoparam(ev);
	client6_send(ev);
	return;
fail:
        if (ev)
                dhcp6_remove_event(ev);

}

static int
client6_recvreconf(ifp, dh6, len, optinfo)
	struct dhcp6_if *ifp;
	struct dhcp6 *dh6;
	ssize_t len;
	struct dhcp6_optinfo *optinfo;
{
	struct authparam *authparam;
	struct keyinfo *keyinfo;

	/* A Reply message must contain a Server ID option */
	dlog("check server id\n");
	if (optinfo->serverID.duid_len == 0) {
		debug_printf(LOG_INFO, FNAME, "no server ID option");
		return (-1);
	}

	dlog("check client id\n");
	/*
	 * DUID in the Client ID option (which must be contained for our
	 * client implementation) must match ours.
	 */
	if (optinfo->clientID.duid_len == 0) {
		debug_printf(LOG_INFO, FNAME, "no client ID option");
		return (-1);
	}
	if (duidcmp(&optinfo->clientID, &client_duid)) {
		debug_printf(LOG_INFO, FNAME, "client DUID mismatch");
		return (-1);
	}

	dlog("Reconfigure message type : %d", optinfo->reconf_type);
	dlog("Reconfigure auth proto : %d", optinfo->authproto);
	dlog("Reconfigure auth algorithm : %d", optinfo->authalgorithm);

	if((authparam = malloc(sizeof(*authparam))) == NULL)
	{
		dlog("memory alloc fail");
		return (-1);
	}
	memset(authparam, 0 , sizeof(*authparam));

	authparam->authproto = optinfo->authproto; 
	authparam->authalgorithm = optinfo->authalgorithm;
	
	if((authparam->key = malloc(sizeof(struct keyinfo))) == NULL)
	{
		dlog("memory alloc fail");
		free(authparam);
		return (-1);
	}
	memset(authparam->key, 0, sizeof(struct keyinfo));
	authparam->key->secretlen = sizeof(reconfig_key);
	authparam->key->secret = reconfig_key;
	authparam->prevrd = g_prevrd;

	//validate authentication
	//Reconfigure key authentication
	if(process_auth(authparam, dh6, len, optinfo))
	{
		dlog("authentication fail !");
		free(authparam);
		free(authparam->key);
		return (-1);
	}

	if(optinfo->reconf_type == DH6OPT_RECONF_TYPE_RENEW)
	{
		dlog("renew all IA\n");
		/* renew stateful configuration information */
		//renew_choose_ia(IATYPE_PD, &optinfo->iapd_list, ifp,
	    	//&optinfo->serverID);

		//renew_choose_ia(IATYPE_NA, &optinfo->iana_list, ifp,
	    	//&optinfo->serverID);
	    	renew_all_ia(ifp); 
	}
	else if(optinfo->reconf_type == DH6OPT_RECONF_TYPE_REBIND)
	{
		dlog("rebind all IA\n");
		rebind_all_ia(ifp);
	}
	else if(optinfo->reconf_type == DH6OPT_RECONF_TYPE_INFOREQ)
	{
		/* not yet */
		dlog("send info req\n");
		send_info_req(ifp);
	}

	free(authparam);
	free(authparam->key);
	return (0);
}

#endif //DHCPV6_LOGO

static int
client6_recvreply(ifp, dh6, len, optinfo, src)
	struct dhcp6_if *ifp;
	struct dhcp6 *dh6;
	ssize_t len;
	struct dhcp6_optinfo *optinfo;
	struct sockaddr_storage *src;
{
	struct dhcp6_listval *lv;
	struct dhcp6_event *ev;
	int state;
	char cmdbuf[64] = "";
	pid_t bg_pid;

	dlog("  ==> Enter client6_recvreply ......\n");

	/* find the corresponding event based on the received xid */
	ev = find_event_withid(ifp, ntohl(dh6->dh6_xid) & DH6_XIDMASK);
	if (ev == NULL) {
		debug_printf(LOG_INFO, FNAME, "XID mismatch");
		return (-1);
	}

	state = ev->state;
	if (state != DHCP6S_INFOREQ &&
	    state != DHCP6S_REQUEST &&
	    state != DHCP6S_RENEW &&
	    state != DHCP6S_REBIND &&
	    state != DHCP6S_RELEASE &&
#ifdef DHCPV6_LOGO
		state != DHCP6S_CNF &&
		state != DHCP6S_DECLINE &&
#endif
	    (state != DHCP6S_SOLICIT ||
	     !(ifp->send_flags & DHCIFF_RAPID_COMMIT))) {
		debug_printf(LOG_INFO, FNAME, "unexpected reply");
		return (-1);
	}

	/* A Reply message must contain a Server ID option */
	if (optinfo->serverID.duid_len == 0) {
		debug_printf(LOG_INFO, FNAME, "no server ID option");
		return (-1);
	}

	/*
	 * DUID in the Client ID option (which must be contained for our
	 * client implementation) must match ours.
	 */
	if (optinfo->clientID.duid_len == 0) {
		debug_printf(LOG_INFO, FNAME, "no client ID option");
		return (-1);
	}
	if (duidcmp(&optinfo->clientID, &client_duid)) {
		debug_printf(LOG_INFO, FNAME, "client DUID mismatch");
		return (-1);
	}

	/* validate authentication */
	if (process_auth(ev->authparam, dh6, len, optinfo)) {
		dlog("authentication fail!!\n");
		debug_printf(LOG_INFO, FNAME, "failed to process authentication");
		return (-1);
	}

	/*
	 * If the client included a Rapid Commit option in the Solicit message,
	 * the client discards any Reply messages it receives that do not
	 * include a Rapid Commit option.
	 * (should we keep the server otherwise?)
	 * [RFC3315 Section 17.1.4]
	 */
	if (state == DHCP6S_SOLICIT &&
	    (ifp->send_flags & DHCIFF_RAPID_COMMIT) &&
	    !optinfo->rapidcommit) {
		debug_printf(LOG_INFO, FNAME, "no rapid commit");
		return (-1);
	}

	/*
	 * The client MAY choose to report any status code or message from the
	 * status code option in the Reply message.
	 * [RFC3315 Section 18.1.8]
	 */
	for (lv = TAILQ_FIRST(&optinfo->stcode_list); lv;
	     lv = TAILQ_NEXT(lv, link)) {
#ifdef DHCPV6_LOGO
		//we should check status code, not ignore it
		if(lv->val_num16 == 1)
			return -1; //unspec failure

		if(lv->val_num16 == 5)
			return -1; //use multicast, because we always use multicast, just resend

		if(lv->val_num16 == 4 && ev->state == DHCP6S_CNF)
		{
			release_choose_ia(IATYPE_NA, &optinfo->iana_list, ifp, &optinfo->serverID);
			dhcp6_remove_event(ev);
			return -1;//not on link
		}
		
		if(lv->val_num16 == 0 && ev->state == DHCP6S_DECLINE)
		{
			release_ia_in_event(ev , ifp , &optinfo->serverID);
			dhcp6_remove_event(ev);
			return 0;
		}
#endif
		debug_printf(LOG_INFO, FNAME, "status code: %s",
		    dhcp6_stcodestr(lv->val_num16));
	}

	if (!TAILQ_EMPTY(&optinfo->dns_list)) {
		struct dhcp6_listval *d;
		int i = 0;

		for (d = TAILQ_FIRST(&optinfo->dns_list); d;
		     d = TAILQ_NEXT(d, link), i++) {
			info_printf("nameserver[%d] %s",
			    i, in6addr2str(&d->val_addr6, 0));
		}
	}

	if (!TAILQ_EMPTY(&optinfo->dnsname_list)) {
		struct dhcp6_listval *d;
		int i = 0;

		for (d = TAILQ_FIRST(&optinfo->dnsname_list); d;
		     d = TAILQ_NEXT(d, link), i++) {
			info_printf("Domain search list[%d] %s",
			    i, d->val_vbuf.dv_buf);
		}
	}

	if (!TAILQ_EMPTY(&optinfo->ntp_list)) {
		struct dhcp6_listval *d;
		int i = 0;

		for (d = TAILQ_FIRST(&optinfo->ntp_list); d;
		     d = TAILQ_NEXT(d, link), i++) {
			info_printf("NTP server[%d] %s",
			    i, in6addr2str(&d->val_addr6, 0));
		}
	}

	if (!TAILQ_EMPTY(&optinfo->sip_list)) {
		struct dhcp6_listval *d;
		int i = 0;

		for (d = TAILQ_FIRST(&optinfo->sip_list); d;
		     d = TAILQ_NEXT(d, link), i++) {
			info_printf("SIP server address[%d] %s",
			    i, in6addr2str(&d->val_addr6, 0));
		}
	}

	if (!TAILQ_EMPTY(&optinfo->sipname_list)) {
		struct dhcp6_listval *d;
		int i = 0;

		for (d = TAILQ_FIRST(&optinfo->sipname_list); d;
		     d = TAILQ_NEXT(d, link), i++) {
			info_printf("SIP domain name[%d] %s",
			    i, d->val_vbuf.dv_buf);
		}
	}

	/* IA-PD/IA-NA binding check (receive Request) */
	if (state == DHCP6S_REQUEST) {
		check_binding_ia(optinfo, ifp);
	}

	if ((!TAILQ_EMPTY(&optinfo->iapd_list)) && (ngn_pd_mode == 1)) {
		/* NGN-PD: Need to add DHCPv6 server address as default route */
		output_wan_nexthop(src);

		/* Reply get, RS send stop (hostrd to down send). */
		snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/hostrd_netlink %s down", ifp->ifname);
		system(cmdbuf);
		memset(&cmdbuf, 0, sizeof(cmdbuf));
	}

	/* dhcp6c result ==> nwctld */
	if (state == DHCP6S_INFOREQ) {
		system("/usr/sbin/nwctl_cli use_prefix on");
		snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ok RA %d %s", duid_type, ifp->ifname);
	} else if (state == DHCP6S_REQUEST) {
		if (ngn_pd_mode) {
			snprintf(cmdbuf, sizeof(cmdbuf), "/usr/sbin/nwctl_cli dhcpv6_rep ok NGNPD %d %s", duid_type, ifp->ifname);
		} else {
			snprintf(cmdbuf, sizeof(cmdbuf), "usr/sbin/nwctl_cli dhcpv6_rep ok PD %d %s", duid_type, ifp->ifname);
		}
	}
	system(cmdbuf);

	/* Since normal Reply was received, DUID is fixed. */
	duid_retry_cnt = 0;

	/* move to the last because we want to get ip before running script */
#if 0   
	/*
	 * Call the configuration script, if specified, to handle various
	 * configuration parameters.
	 */
	if (ifp->scriptpath != NULL && strlen(ifp->scriptpath) != 0) {
		debug_printf(LOG_DEBUG, FNAME, "executes %s", ifp->scriptpath);
		client6_script(ifp->scriptpath, state, optinfo);
	}
#endif
	/*
	 * Set refresh timer for configuration information specified in
	 * information-request.  If the timer value is specified by the server
	 * in an information refresh time option, use it; use the protocol
	 * default otherwise.
	 */
	if (state == DHCP6S_INFOREQ) {
		int64_t refreshtime = DHCP6_IRT_DEFAULT;

		if (optinfo->refreshtime != DH6OPT_REFRESHTIME_UNDEF)
			refreshtime = optinfo->refreshtime;

		ifp->timer = dhcp6_add_timer(client6_expire_refreshtime, ifp);
		if (ifp->timer == NULL) {
			debug_printf(LOG_WARNING, FNAME,
			    "failed to add timer for refresh time");
		} else {
			struct timeval tv;

			tv.tv_sec = (long)refreshtime;
			tv.tv_usec = 0;

			if (tv.tv_sec < 0) {
				/*
				 * XXX: tv_sec can overflow for an
				 * unsigned 32bit value.
				 */
				debug_printf(LOG_WARNING, FNAME,
				    "refresh time is too large: %lu",
				    (u_int32_t)refreshtime);
				tv.tv_sec = 0x7fffffff;	/* XXX */
			}

			dhcp6_set_timer(&tv, ifp->timer);
		}
	} else if (optinfo->refreshtime != DH6OPT_REFRESHTIME_UNDEF) {
		/*
		 * draft-ietf-dhc-lifetime-02 clarifies that refresh time
		 * is only used for information-request and reply exchanges.
		 */
		debug_printf(LOG_INFO, FNAME,
		    "unexpected information refresh time option (ignored)");
	}

	/* update stateful configuration information */
#ifdef DHCPV6_LOGO
	if(state == DHCP6S_RENEW || state == DHCP6S_REBIND || state == DHCP6S_CNF)
	{
		if(all_iana_in_reply(ev , &optinfo->iana_list) == 0)
		{
			//console_printf(LOC_FMT"ianas not found\n" , LOC_ARG);
#ifndef DHCPV6_IOL
			return -1; // not all iana in reply
#endif //!DHCPV6_IOL
		}

		if(scan_all_iapds(ev , &optinfo->iapd_list , ifp , &optinfo->serverID) == 0)
		{
			//console_printf(LOC_FMT"iapds not found\n" , LOC_ARG);
		}
	}
#endif

	if (state != DHCP6S_RELEASE) {
		int iapd_exist = 0;
		int iana_exist = 0;

		if (!TAILQ_EMPTY(&optinfo->iapd_list))
			iapd_exist = 1;

		if (!TAILQ_EMPTY(&optinfo->iana_list)) {
			iana_exist = 1;
		} else {
			iana_addr_nothing = 1;
		}

		if ((iapd_exist) && (iana_exist)) {
			update_ia_pdna(optinfo, ifp, ev->authparam);
		} else {

#ifdef DHCPV6_LOGO
			if(update_ia(IATYPE_PD, &optinfo->iapd_list, ifp,
			    &optinfo->serverID, ev->authparam) < 0)
			{
				return -1; //ignore this reply
			}
#else
			update_ia(IATYPE_PD, &optinfo->iapd_list, ifp,
			    &optinfo->serverID, ev->authparam);
#endif
			update_ia(IATYPE_NA, &optinfo->iana_list, ifp,
			    &optinfo->serverID, ev->authparam);
		}
		/* IA-NA Delete flag default */
		iana_addr_delete = 0;
	}

#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
        if (optinfo->enterprise_code == DHCP_ENTERPRISE_CODE)
        {
            char zeros[16];
            memset(zeros, 0, 16);
            if (0 != memcmp(zeros, optinfo->ac_ipv6, 16))
            {
                got_ac_ipv6 = 1;
                memcpy(str_ac_ipv6, optinfo->ac_ipv6, 16);
                debug_printf(LOG_DEBUG, FNAME, "AC: set ac's ipv6 addr from reply to str_ac_ipv6");
            }
        }
#endif
#ifdef ELBOX_PROGS_PRIV_APMODULE_FOR_ZOOM_NETWORKS
        if (got_ac_ipv6)
        {
            char cmd[256];
            debug_printf(LOG_DEBUG, FNAME, "AC: set ac's ipv6 addr to xmldb");
            sprintf(cmd, "rgdb -i -s /runtime/wan/inf:1/ac_ip \"%s\"",in6addr2str((void*)str_ac_ipv6, 0));
            system(cmd);
        }
#endif
	dhcp6_remove_event(ev);

	if (state == DHCP6S_RELEASE) {
		/*
		 * When the client receives a valid Reply message in response
		 * to a Release message, the client considers the Release event
		 * completed, regardless of the Status Code option(s) returned
		 * by the server.
		 * [RFC3315 Section 18.1.8]
		 */
		check_exit();
	}

	debug_printf(LOG_DEBUG, FNAME, "got an expected reply, sleeping.");

	if (infreq_mode) {
		exit_ok = 1;
		free_resources(NULL);
		unlink(pid_file);
		check_exit();
	}
#if 1
	/*
	 * Call the configuration script, if specified, to handle various
	 * configuration parameters.
	 */
	if (state == DHCP6S_REQUEST || state == DHCP6S_RENEW ||
           state == DHCP6S_REBIND || state == DHCP6S_INFOREQ) {
		if (ifp->scriptpath != NULL && strlen(ifp->scriptpath) != 0) {
			debug_printf(LOG_DEBUG, FNAME, "executes %s", ifp->scriptpath);
			client6_script(ifp->scriptpath, state, optinfo);
		}
	}

	/* Save "dhcp6c_script_bg" process id */
	if (check_process("dhcp6c-script_bg", &bg_pid) == 0) {
		memset(&cmdbuf, 0, sizeof(cmdbuf));
		snprintf(cmdbuf, sizeof(cmdbuf), "echo -n %d > %s", bg_pid, DHCP6C_SCRIPT_BG_PIDFILE);
		system(cmdbuf);
	}
#endif
	return (0);
}

static struct dhcp6_event *
find_event_withid(ifp, xid)
	struct dhcp6_if *ifp;
	u_int32_t xid;
{
	struct dhcp6_event *ev;

	for (ev = TAILQ_FIRST(&ifp->event_list); ev;
	     ev = TAILQ_NEXT(ev, link)) {
		if (ev->xid == xid)
			return (ev);
	}

	return (NULL);
}

static int
process_auth(authparam, dh6, len, optinfo)
	struct authparam *authparam;
	struct dhcp6 *dh6;
	ssize_t len;
	struct dhcp6_optinfo *optinfo;
{
	struct keyinfo *key = NULL;
	int authenticated = 0;

	dlog("  ++ Enter process_auth, authproto: %d\n", optinfo->authproto);
	
	switch (optinfo->authproto) {
	case DHCP6_AUTHPROTO_UNDEF:
		/* server did not provide authentication option */
		break;
	case DHCP6_AUTHPROTO_DELAYED:
		if ((optinfo->authflags & DHCP6OPT_AUTHFLAG_NOINFO)) {
			debug_printf(LOG_INFO, FNAME, "server did not include "
			    "authentication information");
			break;
		}

		if (optinfo->authalgorithm != DHCP6_AUTHALG_HMACMD5) {
			debug_printf(LOG_INFO, FNAME, "unknown authentication "
			    "algorithm (%d)", optinfo->authalgorithm);
			break;
		}

		if (optinfo->authrdm != DHCP6_AUTHRDM_MONOCOUNTER) {
			debug_printf(LOG_INFO, FNAME,"unknown RDM (%d)",
			    optinfo->authrdm);
			break;
		}

		/*
		 * Replay protection.  If we do not know the previous RD value,
		 * we accept the message anyway (XXX).
		 */
		if ((authparam->flags & AUTHPARAM_FLAGS_NOPREVRD)) {
			debug_printf(LOG_WARNING, FNAME, "previous RD value is "
			    "unknown (accept it)");
		} else {
			if (dhcp6_auth_replaycheck(optinfo->authrdm,
			    authparam->prevrd, optinfo->authrd)) {
				debug_printf(LOG_INFO, FNAME,
				    "possible replay attack detected");
				break;
			}
		}

		/* identify the secret key */
		if ((key = authparam->key) != NULL) {
			/*
			 * If we already know a key, its identification should
			 * match that contained in the received option.
			 * (from Section 21.4.5.1 of RFC3315)
			 */
			if (optinfo->delayedauth_keyid != key->keyid ||
			    optinfo->delayedauth_realmlen != key->realmlen ||
			    memcmp(optinfo->delayedauth_realmval, key->realm,
			    key->realmlen) != 0) {
				debug_printf(LOG_INFO, FNAME,
				    "authentication key mismatch");
				break;
			}
		} else {
			key = find_key(optinfo->delayedauth_realmval,
			    optinfo->delayedauth_realmlen,
			    optinfo->delayedauth_keyid);
			if (key == NULL) {
				debug_printf(LOG_INFO, FNAME, "failed to find key "
				    "provided by the server (ID: %x)",
				    optinfo->delayedauth_keyid);
				break;
			} else {
				debug_printf(LOG_DEBUG, FNAME, "found key for "
				    "authentication: %s", key->name);
			}
			authparam->key = key;
		}

		/* check for the key lifetime */
		if (dhcp6_validate_key(key)) {
			debug_printf(LOG_INFO, FNAME, "key %s has expired",
			    key->name);
			break;
		}

		/* validate MAC */
		if (dhcp6_verify_mac((char *)dh6, len, optinfo->authproto,
		    optinfo->authalgorithm,
		    optinfo->delayedauth_offset + sizeof(*dh6), key) == 0) {
			debug_printf(LOG_DEBUG, FNAME, "message authentication "
			    "validated");
			authenticated = 1;
		} else {
			debug_printf(LOG_INFO, FNAME, "invalid message "
			    "authentication");
		}

		break;
	case DHCP6_AUTHPROTO_RECONFIG:
		if ((optinfo->authflags & DHCP6OPT_AUTHFLAG_NOINFO)) {
			debug_printf(LOG_INFO, FNAME, "server did not include "
			    "authentication information");
			dlog("server did not include "
			    "authentication information");
			break;
		}

		if (optinfo->authalgorithm != DHCP6_AUTHALG_HMACMD5) {
			debug_printf(LOG_INFO, FNAME, "unknown authentication "
			    "algorithm (%d)", optinfo->authalgorithm);
			dlog("unknown authentication "
			    "algorithm (%d)", optinfo->authalgorithm);
			break;
		}

		if (optinfo->authrdm != DHCP6_AUTHRDM_MONOCOUNTER) {
			debug_printf(LOG_INFO, FNAME,"unknown RDM (%d)",
			    optinfo->authrdm);
			break;
		}

		dlog("reconfig auth type is %d",optinfo->reconfigauth_type);
		if(optinfo->reconfigauth_type == 1) 
		{
			g_prevrd = optinfo->authrd;//rbj
			return (0);
		}

		/*
		 * Replay protection.  If we do not know the previous RD value,
		 * we accept the message anyway (XXX).
		 */
		if (dhcp6_auth_replaycheck(optinfo->authrdm,
		    authparam->prevrd, optinfo->authrd)) {
			debug_printf(LOG_INFO, FNAME,
			    "possible replay attack detected");
			dlog("possible replay attack detected");
			break;
		}


		//optinfo->reconfigauth_offset+=1;//for test
		/* validate MAC */
		if (dhcp6_verify_mac((char *)dh6, len, optinfo->authproto,
		    optinfo->authalgorithm,
		    optinfo->reconfigauth_offset + sizeof(*dh6), authparam->key) == 0) {
			debug_printf(LOG_DEBUG, FNAME, "message authentication "
			    "validated");
			dlog("message authentication "
			    "validated");
			authenticated = 1;
		} else {
			debug_printf(LOG_INFO, FNAME, "invalid message "
			    "authentication");
			dlog("invalid message "
			    "authentication");
		}

		break;
	default:
		debug_printf(LOG_INFO, FNAME, "server sent unsupported "
		    "authentication protocol (%d)", optinfo->authproto);
		break;
	}

	if (authenticated == 0) {
		if (authparam->authproto != DHCP6_AUTHPROTO_UNDEF) {
			debug_printf(LOG_INFO, FNAME, "message not authenticated "
			    "while authentication required");

			/*
			 * Right now, we simply discard unauthenticated
			 * messages.
			 */
			return (-1);
		}
	} else {
		/* if authenticated, update the "previous" RD value */
		authparam->prevrd = optinfo->authrd;
		authparam->flags &= ~AUTHPARAM_FLAGS_NOPREVRD;
		g_prevrd = optinfo->authrd;//rbj
	}

	return (0);
}

static int
set_auth(ev, optinfo)
	struct dhcp6_event *ev;
	struct dhcp6_optinfo *optinfo;
{
	struct authparam *authparam = ev->authparam;

	if (authparam == NULL)
		return (0);

	optinfo->authproto = authparam->authproto;
	optinfo->authalgorithm = authparam->authalgorithm;
	optinfo->authrdm = authparam->authrdm;

	switch (authparam->authproto) {
	case DHCP6_AUTHPROTO_UNDEF: /* we simply do not need authentication */
		return (0);
	case DHCP6_AUTHPROTO_DELAYED:
		if (ev->state == DHCP6S_INFOREQ) {
			/*
			 * In the current implementation, delayed
			 * authentication for Information-request and Reply
			 * exchanges doesn't work.  Specification is also
			 * unclear on this usage.
			 */
			debug_printf(LOG_WARNING, FNAME, "delayed authentication "
			    "cannot be used for Information-request yet");
			return (-1);
		}

		if (ev->state == DHCP6S_SOLICIT) {
			optinfo->authflags |= DHCP6OPT_AUTHFLAG_NOINFO;
			return (0); /* no auth information is needed */
		}

		if (authparam->key == NULL) {
			debug_printf(LOG_INFO, FNAME,
			    "no authentication key for %s",
			    dhcp6_event_statestr(ev));
			return (-1);
		}

		if (dhcp6_validate_key(authparam->key)) {
			debug_printf(LOG_INFO, FNAME, "key %s is invalid",
			    authparam->key->name);
			return (-1);
		}

		if (get_rdvalue(optinfo->authrdm, &optinfo->authrd,
		    sizeof(optinfo->authrd))) {
			debug_printf(LOG_ERR, FNAME, "failed to get a replay "
			    "detection value");
			return (-1);
		}

		optinfo->delayedauth_keyid = authparam->key->keyid;
		optinfo->delayedauth_realmlen = authparam->key->realmlen;
		optinfo->delayedauth_realmval =
		    malloc(optinfo->delayedauth_realmlen);
		if (optinfo->delayedauth_realmval == NULL) {
			debug_printf(LOG_ERR, FNAME, "failed to allocate memory "
			    "for authentication realm");
			return (-1);
		}
		memcpy(optinfo->delayedauth_realmval, authparam->key->realm,
		    optinfo->delayedauth_realmlen);

		break;
	default:
		debug_printf(LOG_ERR, FNAME, "unsupported authentication protocol "
		    "%d", authparam->authproto);
		return (-1);
	}

	return (0);
}

static void
info_printf(const char *fmt, ...)
{
	va_list ap;
	char logbuf[LINE_MAX];

	va_start(ap, fmt);
	vsnprintf(logbuf, sizeof(logbuf), fmt, ap);

	debug_printf(LOG_DEBUG, FNAME, "%s", logbuf);
	if (infreq_mode)
		printf("%s\n", logbuf);

	return;
}

static void
check_config_file()
{
	struct dhcp6_if *ifp;
	struct ia_conf  *iac;
	int type_pd = 0;
	int type_na = 0;

	/* config file search(argv_ifname = WAN I/F) */
	ifp = find_ifconfbyname(argv_ifname);

	if (ifp == NULL) {
		debug_printf(LOG_INFO, FNAME, "Failed to config file check");
		return;
	}

	/* Info-Req Check */
	if (ifp->send_flags & DHCIFF_INFO_ONLY) {
		info_req_mode = 1;
	}

	for (iac = TAILQ_FIRST(&ifp->iaconf_list); iac; iac = TAILQ_NEXT(iac, link)) {

		/* Send IATYPE Check */
		if (iac->type == IATYPE_PD) {
			type_pd = 1;
		}
		if (iac->type == IATYPE_NA) {
			type_na = 1;
		}
	}

	/* NGN Check */
	/* type = IATYPE_PD send & type = IATYPE_NA not send  */
	if (type_pd == 1 && type_na == 0) {
		ngn_pd_mode = 1;
	}
	return;
}

static void
output_wan_nexthop(from)
	struct sockaddr_storage *from;
{
	FILE *fp;
	char *str_from, *set_str;
	char ifname_buf[INET6_ADDRSTRLEN + IFNAMSIZ + 1] = {0};
	char str_buf[INET6_ADDRSTRLEN + IFNAMSIZ + 1] = {0};

	fp = fopen("/var/dhcp6c_nexthop", "w+");
	if (NULL == fp) {
		debug_printf(LOG_INFO, FNAME, "failed to open nexthop file");
		return;
	}

	str_from = addr2str((struct sockaddr *)from);
	set_str = str_from;
	if ((strstr(str_from, "%")) == NULL) {
		if (if_indextoname(((struct sockaddr_in6 *)from)->sin6_scope_id, ifname_buf) == NULL) {
			debug_printf(LOG_INFO, FNAME, "failed to if_indextoname");
			goto end;
		}

		sprintf(str_buf, "%s%%%s", str_from, ifname_buf);
		set_str = str_buf;
	}

	fprintf(fp, "%s", set_str);

end:
	fclose(fp);

	return;
}

static int
read_v6jdg_result()
{
	FILE *fp;
	char buf[BUFSIZ];
	char *ptr = NULL;
	int mode;

	fp = fopen(DHCP6C_V6JUDGE_FILE, "r");
	if (fp == NULL) {
		debug_printf(LOG_ERR, FNAME, "v6jdg_result file open error.");
		return -1;
	}

	if (fgets(buf, BUFSIZ, fp) != NULL) {
		if ((ptr = strstr(buf, DHCP6C_V6JUDGE_MODE)) != NULL) {
			ptr += strlen(DHCP6C_V6JUDGE_MODE);
		}
	}
	fclose(fp);

	if (ptr == NULL) {
		debug_printf(LOG_ERR, FNAME, "v6jdg_result file read error.");
		return -1;
	}

	if (strncmp(ptr, DHCP6C_V6JUDGE_NGN, strlen(DHCP6C_V6JUDGE_NGN)) == 0) {
		mode = V6_NGN;
	} else if (strncmp(ptr, DHCP6C_V6JUDGE_ROUTING, strlen(DHCP6C_V6JUDGE_ROUTING)) == 0) {
		mode = V6_ROUTING;
	} else if (strncmp(ptr, DHCP6C_V6JUDGE_NDP, strlen(DHCP6C_V6JUDGE_NDP)) == 0) {
		mode = V6_NDP;
	} else if (strncmp(ptr, DHCP6C_V6JUDGE_JUDGING, strlen(DHCP6C_V6JUDGE_JUDGING)) == 0) {
		mode = V6_JUDGING;
	} else {
		debug_printf(LOG_ERR, FNAME, "v6jdg.result not agree[%s]", ptr);
		return -1;
	}

	return mode;
}

static int
check_dhcp_status_code(optinfo)
	struct dhcp6_optinfo *optinfo;
{
	struct dhcp6_listval *stcode;
	int status_code = DH6OPT_STCODE_UNSPECFAIL;

	for (stcode = TAILQ_FIRST(&optinfo->stcode_list); stcode; stcode = TAILQ_NEXT(stcode, link)) {
		if (stcode->type != DHCP6_LISTVAL_STCODE) {
			continue;
		}

		status_code = (int)(stcode->val_num16);
		if ((status_code == DH6OPT_STCODE_NOPREFIXAVAIL) ||
		    (status_code == DH6OPT_STCODE_SUCCESS)) {
			return status_code;
		}
	}

	return status_code;
}

/*
 * check_ia_address(): Check IA-NA and IA-PD in the received DHCPv6 message.
 * return 0: Success (Exist IA address or IA prefix. And Status code is Success or NoBinding)
 *        1: Failure (Status code is not Success and NoBinding)
 */
static int
check_ia_address(iatype, ialist)
	iatype_t iatype;
	struct dhcp6_list *ialist;
{
	struct dhcp6_listval *iav, *siav;
	int rtn = 0;

	for (iav = TAILQ_FIRST(ialist); iav; iav = TAILQ_NEXT(iav, link)) {
		if (((iatype == IATYPE_NA) && (iav->type != DHCP6_LISTVAL_IANA)) ||
		    ((iatype == IATYPE_PD) && (iav->type != DHCP6_LISTVAL_IAPD))) {
			continue;
		}
		for (siav = TAILQ_FIRST(&iav->sublist); siav; siav = TAILQ_NEXT(siav, link)) {
			switch (siav->type) {
			case DHCP6_LISTVAL_PREFIX6:
				if (iatype == IATYPE_PD)
					rtn = 1;
				break;
			case DHCP6_LISTVAL_STATEFULADDR6:
				if (iatype == IATYPE_NA)
					rtn = 1;
				break;
			case DHCP6_LISTVAL_STCODE:
				if ((siav->val_num16 != DH6OPT_STCODE_SUCCESS) &&
				    (siav->val_num16 != DH6OPT_STCODE_NOBINDING)) {
					rtn = 0;
				} else {
					rtn = 1;
				}
				break;
			default:
				break;
			}
		}
	}

	return rtn;
}
