/*	$KAME: prefixconf.c,v 1.33 2005/09/16 11:30:15 suz Exp $	*/

/*
 * Copyright (C) 2002 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/queue.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <net/if.h>
#ifdef __FreeBSD__
#include <net/if_var.h>
#endif

#include <net/route.h>
#include <netinet/in.h>

#ifdef __KAME__
#include <netinet6/in6_var.h>
#include <netinet6/nd6.h>
#endif

#include <arpa/inet.h>

#include <errno.h>
#include <syslog.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include "dhcp6.h"
#include "config.h"
#include "common.h"
#include "timer.h"
#include "dhcp6c_ia.h"
#include "prefixconf.h"

/* move to prefixconf.h (External reference) */
#if 0
TAILQ_HEAD(siteprefix_list, siteprefix);
struct iactl_pd {
	struct iactl common;
	struct pifc_list *pifc_head;
	struct siteprefix_list siteprefix_head;
};
#define iacpd_ia common.iactl_ia
#define iacpd_callback common.callback
#define iacpd_isvalid common.isvalid
#define iacpd_duration common.duration
#define iacpd_renew_data common.renew_data
#define iacpd_rebind_data common.rebind_data
#define iacpd_reestablish_data common.reestablish_data
#define iacpd_release_data common.release_data
#define iacpd_cleanup common.cleanup

struct siteprefix {
	TAILQ_ENTRY (siteprefix) link;

	struct dhcp6_prefix prefix;
	time_t updatetime;
	struct dhcp6_timer *timer;
	struct iactl_pd *ctl;
	TAILQ_HEAD(, dhcp6_ifprefix) ifprefix_list; /* interface prefixes */
};

struct dhcp6_ifprefix {
	TAILQ_ENTRY(dhcp6_ifprefix) plink;

	/* interface configuration */
	struct prefix_ifconf *ifconf;

	/* interface prefix parameters */
	struct sockaddr_in6 paddr;
	int plen;

	/* address assigned on the interface based on the prefix */
	struct sockaddr_in6 ifaddr;
};
#endif

static struct siteprefix *find_siteprefix __P((struct siteprefix_list *,
    struct dhcp6_prefix *, int));
static void remove_siteprefix __P((struct siteprefix *));
static int isvalid __P((struct iactl *));
static u_int32_t duration __P((struct iactl *));
static void cleanup __P((struct iactl *));
static int renew_prefix __P((struct iactl *, struct dhcp6_ia *,
    struct dhcp6_eventdata **, struct dhcp6_eventdata *));
static void renew_data_free __P((struct dhcp6_eventdata *));

static struct dhcp6_timer *siteprefix_timo __P((void *));

#ifdef USE_NWCTL
static int add_ifprefix_nwctl __P((struct iactl_pd *,
    struct siteprefix *, struct dhcp6_prefix *, char *, int));
#endif  /* USE_NWCTL */
static int add_ifprefix __P((struct siteprefix *,
    struct dhcp6_prefix *, struct prefix_ifconf *));
static int make_ifprefix __P((struct dhcp6_prefix *,
    struct prefix_ifconf *, struct dhcp6_ifprefix *));
#ifdef RTK_AP_IPV6
static int create_Dhcp6pdCfgFile __P((struct in6_addr *,
    struct dhcp6_prefix *, int));
#endif	/* RTK_AP_IPV6 */
static int create_lan_address_file __P((struct in6_addr *,
    struct dhcp6_prefix *, int));
static void delete_lan_address_file __P((void));

extern struct dhcp6_timer *client6_timo __P((void *));
static int pd_ifaddrconf __P((ifaddrconf_cmd_t, struct dhcp6_ifprefix *ifpfx));
static int pd_ipv6_route __P((int));
static int pd_chk_wan_update __P((void));

enum { IPv6_ROUTE_ADD, IPv6_ROUTE_DEL };

extern int ngn_pd_mode;
extern int iana_addr_nothing;
extern int iana_addr_delete;
extern char *argv_ifname;

void
call_remove_siteprefix(sp)
	struct siteprefix *sp;
{
	remove_siteprefix(sp);
}

int
update_prefix(ia, pinfo, pifc, dhcpifp, ctlp, callback, p_sp, timer_flg)
	struct ia *ia;
	struct dhcp6_prefix *pinfo;
	struct pifc_list *pifc;
	struct dhcp6_if *dhcpifp;
	struct iactl **ctlp;
	void (*callback)__P((struct ia *));
	struct siteprefix **p_sp;
	int timer_flg;		/* PREFIX_TIMER_SET_LATER or PREFIX_TIMER_SET_NOW */
{
	struct iactl_pd *iac_pd = (struct iactl_pd *)*ctlp;
	struct siteprefix *sp;
	struct prefix_ifconf *pif;
	int spcreate = 0;
	struct timeval timo;

	/*
	 * A client discards any addresses for which the preferred
         * lifetime is greater than the valid lifetime.
	 * [RFC3315 22.6] 
	 */
	if (pinfo->vltime != DHCP6_DURATION_INFINITE &&
	    (pinfo->pltime == DHCP6_DURATION_INFINITE ||
	    pinfo->pltime > pinfo->vltime)) {
		debug_printf(LOG_INFO, FNAME, "invalid prefix %s/%d: "
		    "pltime (%lu) is larger than vltime (%lu)",
		    in6addr2str(&pinfo->addr, 0), pinfo->plen,
		    pinfo->pltime, pinfo->vltime);
		return (-1);
	}

	if (iac_pd == NULL) {
		if ((iac_pd = malloc(sizeof(*iac_pd))) == NULL) {
			debug_printf(LOG_NOTICE, FNAME, "memory allocation failed");
			return (-1);
		}
		memset(iac_pd, 0, sizeof(*iac_pd));
		iac_pd->iacpd_ia = ia;
		iac_pd->iacpd_callback = callback;
		iac_pd->iacpd_isvalid = isvalid;
		iac_pd->iacpd_duration = duration;
		iac_pd->iacpd_cleanup = cleanup;
		iac_pd->iacpd_renew_data =
		    iac_pd->iacpd_rebind_data =
		    iac_pd->iacpd_release_data =
		    iac_pd->iacpd_reestablish_data = renew_prefix;

		iac_pd->pifc_head = pifc;
		TAILQ_INIT(&iac_pd->siteprefix_head);
		*ctlp = (struct iactl *)iac_pd;
	}

	/* search for the given prefix, and make a new one if it fails */
	if ((sp = find_siteprefix(&iac_pd->siteprefix_head, pinfo, 1)) == NULL) {
		if ((sp = malloc(sizeof(*sp))) == NULL) {
			debug_printf(LOG_NOTICE, FNAME, "memory allocation failed");
			return (-1);
		}
		memset(sp, 0, sizeof(*sp));
		sp->prefix.addr = pinfo->addr;
		sp->prefix.plen = pinfo->plen;
		sp->ctl = iac_pd;
		TAILQ_INIT(&sp->ifprefix_list);

		TAILQ_INSERT_TAIL(&iac_pd->siteprefix_head, sp, link);

		spcreate = 1;
	}

	/* update the timestamp of update */
	sp->updatetime = time(NULL);

	/* update the prefix according to pinfo */
	sp->prefix.pltime = pinfo->pltime;
	sp->prefix.vltime = pinfo->vltime;
	debug_printf(LOG_DEBUG, FNAME, "%s a prefix %s/%d pltime=%lu, vltime=%lu",
	    spcreate ? "create" : "update",
	    in6addr2str(&pinfo->addr, 0), pinfo->plen,
	    pinfo->pltime, pinfo->vltime);

	/* update prefix interfaces if necessary */
#ifdef USE_NWCTL
	if (sp->prefix.vltime != 0) {
		/*
		 * The requesting router MUST NOT assign any delegated
		 * prefixes or subnets from the delegated prefix(es) to
		 * the link through which it received the DHCP message
		 * from the delegating router.
		 * [RFC3633 Section 12.1]
		 */
		if (pd_chk_wan_update()) {
			/* WAN: IA-PD prefix update */
			add_ifprefix_nwctl(iac_pd, sp, pinfo, dhcpifp->ifname, spcreate);
		}
		/* LAN: IA-PD prefix update */
		add_ifprefix_nwctl(iac_pd, sp, pinfo, IFNAME_LAN, spcreate);
	}
#else	/* USE_NWCTL */
	if (sp->prefix.vltime != 0 && spcreate) {
		for (pif = TAILQ_FIRST(iac_pd->pifc_head); pif;
		    pif = TAILQ_NEXT(pif, link)) {
			/*
			 * The requesting router MUST NOT assign any delegated
			 * prefixes or subnets from the delegated prefix(es) to
			 * the link through which it received the DHCP message
			 * from the delegating router.
			 * [RFC3633 Section 12.1]
			 */
			if ((strcmp(pif->ifname, dhcpifp->ifname) == 0) &&
			    (!pd_chk_wan_update())) {
				debug_printf(LOG_INFO, FNAME,
				    "skip %s as a prefix interface",
				    dhcpifp->ifname);
				continue;
			}

			add_ifprefix(sp, pinfo, pif);
		}
	}
#endif	/* USE_NWCTL */

	/* timer_flg = PREFIX_TIMER_SET_LATER: other place set up or update the associated timer */
	if (timer_flg == PREFIX_TIMER_SET_NOW) {
		/*
		 * If the new vltime is 0, this prefix immediately expires.
		 * Otherwise, set up or update the associated timer.
		 */
		switch (sp->prefix.vltime) {
		case 0:
			remove_siteprefix(sp);
			return (0);
		case DHCP6_DURATION_INFINITE:
			if (sp->timer)
				dhcp6_remove_timer(&sp->timer);
			break;
		default:
			if (sp->timer == NULL) {
				sp->timer = dhcp6_add_timer(siteprefix_timo, sp);
				if (sp->timer == NULL) {
					debug_printf(LOG_NOTICE, FNAME,
					    "failed to add prefix timer");
					remove_siteprefix(sp); /* XXX */
					return (-1);
				}
			}
			/* update the timer */
			timo.tv_sec = sp->prefix.vltime;
			timo.tv_usec = 0;

			dhcp6_set_timer(&timo, sp->timer);
			break;
		}
	} else {
		/* If the new vltime is 0, this prefix immediately expires. */
		if (sp->prefix.vltime == 0) {
			if (g_sp == sp)
				g_sp = NULL;

			remove_siteprefix(sp);
			return (0);
		}
	}

	*p_sp = sp;
	return (0);
}

static struct siteprefix *
find_siteprefix(head, prefix, match_plen)
	struct siteprefix_list *head;
	struct dhcp6_prefix *prefix;
	int match_plen;
{
	struct siteprefix *sp;

	for (sp = TAILQ_FIRST(head); sp; sp = TAILQ_NEXT(sp, link)) {
		if (!IN6_ARE_ADDR_EQUAL(&sp->prefix.addr, &prefix->addr))
			continue;
		if (match_plen == 0 || sp->prefix.plen == prefix->plen)
			return (sp);
	}

	return (NULL);
}

static void
remove_siteprefix(sp)
	struct siteprefix *sp;
{
	struct dhcp6_ifprefix *ip;

	debug_printf(LOG_DEBUG, FNAME, "remove a site prefix %s/%d",
	    in6addr2str(&sp->prefix.addr, 0), sp->prefix.plen);

	if (sp->timer)
		dhcp6_remove_timer(&sp->timer);

	/* remove all interface prefixes */
	while ((ip = TAILQ_FIRST(&sp->ifprefix_list)) != NULL) {
		TAILQ_REMOVE(&sp->ifprefix_list, ip, plink);
#ifdef USE_NWCTL
		char ifaddr_buf[64] = "";
		char cmdbuf[256] = "";

		inet_ntop(AF_INET6, &(ip->ifaddr.sin6_addr), ifaddr_buf, sizeof(ifaddr_buf));
		/* nwctl_cli v6addr del called. */
		snprintf(cmdbuf, sizeof(cmdbuf),
			"/usr/sbin/nwctl_cli v6addr del %s %s/%d -origin dhcpv6",
			ip->ifconf->ifname, ifaddr_buf, ip->plen);
		system(cmdbuf);
#else	/* USE_NWCTL */
		pd_ifaddrconf(IFADDRCONF_REMOVE, ip);
#endif	/* USE_NWCTL */

		/* LAN GUA address infomation file delete */
		if (strcmp(ip->ifconf->ifname, IFNAME_LAN) == 0) {
			delete_lan_address_file();
		}
		free(ip);
	}

	/* remove default route -> only ngn-pd */
	if (ngn_pd_mode == 1) {
#ifdef USE_NWCTL
		/* default route add use nwctld only */
		/* (If nwctld is not used, additional processing is required.) */
		pd_ipv6_route(IPv6_ROUTE_DEL);
#endif  /* USE_NWCTL */
	}

	TAILQ_REMOVE(&sp->ctl->siteprefix_head, sp, link);
	free(sp);
}

static int
isvalid(iac)
	struct iactl *iac;
{
	struct iactl_pd *iac_pd = (struct iactl_pd *)iac;

	if (TAILQ_EMPTY(&iac_pd->siteprefix_head))
		return (0);	/* this IA is invalid */
	return (1);
}

static u_int32_t
duration(iac)
	struct iactl *iac;
{
	struct iactl_pd *iac_pd = (struct iactl_pd *)iac;
	struct siteprefix *sp;
	u_int32_t base = DHCP6_DURATION_INFINITE, pltime, passed;
	time_t now;

	/* Determine the smallest period until pltime expires. */
	now = time(NULL);
	for (sp = TAILQ_FIRST(&iac_pd->siteprefix_head); sp;
	    sp = TAILQ_NEXT(sp, link)) {
		passed = now > sp->updatetime ?
		    (u_int32_t)(now - sp->updatetime) : 0;
		pltime = sp->prefix.pltime > passed ?
		    sp->prefix.pltime - passed : 0;

		if (base == DHCP6_DURATION_INFINITE || pltime < base)
			base = pltime;
	}

	return (base);
}

static void
cleanup(iac)
	struct iactl *iac;
{
	struct iactl_pd *iac_pd = (struct iactl_pd *)iac;
	struct siteprefix *sp;

	while ((sp = TAILQ_FIRST(&iac_pd->siteprefix_head)) != NULL) {
		TAILQ_REMOVE(&iac_pd->siteprefix_head, sp, link);
		remove_siteprefix(sp);
	}

	free(iac);
}

static int
renew_prefix(iac, iaparam, evdp, evd)
	struct iactl *iac;
	struct dhcp6_ia *iaparam;
	struct dhcp6_eventdata **evdp, *evd;
{
	struct iactl_pd *iac_pd = (struct iactl_pd *)iac;
	struct siteprefix *sp;
	struct dhcp6_list *ial = NULL, pl;

	TAILQ_INIT(&pl);
	for (sp = TAILQ_FIRST(&iac_pd->siteprefix_head); sp;
	    sp = TAILQ_NEXT(sp, link)) {
		if (dhcp6_add_listval(&pl, DHCP6_LISTVAL_PREFIX6,
		    &sp->prefix, NULL) == NULL)
			goto fail;
	}

	if ((ial = malloc(sizeof(*ial))) == NULL)
		goto fail;
	TAILQ_INIT(ial);
	if (dhcp6_add_listval(ial, DHCP6_LISTVAL_IAPD, iaparam, &pl) == NULL)
		goto fail;
	dhcp6_clear_list(&pl);

	evd->type = DHCP6_EVDATA_IAPD;
	evd->data = (void *)ial;
	evd->privdata = (void *)evdp;
	evd->destructor = renew_data_free;

	return (0);

  fail:
	dhcp6_clear_list(&pl);
	if (ial)
		free(ial);
	return (-1);
}

static void
renew_data_free(evd)
	struct dhcp6_eventdata *evd;
{
	struct dhcp6_list *ial;

	if (evd->type != DHCP6_EVDATA_IAPD) {
		debug_printf(LOG_ERR, FNAME, "assumption failure");
		exit(1);
	}

	if (evd->privdata)
		*(struct dhcp6_eventdata **)evd->privdata = NULL;
	ial = (struct dhcp6_list *)evd->data;
	dhcp6_clear_list(ial);
	free(ial);
}

static struct dhcp6_timer *
siteprefix_timo(arg)
	void *arg;
{
	struct siteprefix *sp = (struct siteprefix *)arg;
	struct ia *ia;
	void (*callback)__P((struct ia *));

	debug_printf(LOG_DEBUG, FNAME, "prefix timeout for %s/%d",
	    in6addr2str(&sp->prefix.addr, 0), sp->prefix.plen);

	ia = sp->ctl->iacpd_ia;
	callback = sp->ctl->iacpd_callback;

	if (sp->timer)
		dhcp6_remove_timer(&sp->timer);

	remove_siteprefix(sp);

	(*callback)(ia);

	return (NULL);
}

#ifdef USE_NWCTL
static int
add_ifprefix_nwctl(iac_pd, sp, pinfo, ifname, spcreate)
	struct iactl_pd *iac_pd;
	struct siteprefix *sp;
	struct dhcp6_prefix *pinfo;
	char *ifname;
	int spcreate;
{
	struct prefix_ifconf *pif;
	struct dhcp6_ifprefix ifpfx;
	char ifaddr_buf[64] = "";
	char cmdbuf[256] = "";
	struct stat sts;

	memset(&ifpfx, 0, sizeof(ifpfx));

	for (pif = TAILQ_FIRST(iac_pd->pifc_head); pif;
	     pif = TAILQ_NEXT(pif, link)) {

		/* ifname check */
		if (strcmp(pif->ifname, ifname) == 0) {
			if (spcreate) {
				add_ifprefix(sp, pinfo, pif);
			} else {
				/* IA-PD update */
				if (make_ifprefix(pinfo, pif, &ifpfx) == 0) {
					/* spcreate = 0 (update): nwctl called  */
					inet_ntop(AF_INET6, &(ifpfx.ifaddr.sin6_addr),
					    ifaddr_buf, sizeof(ifaddr_buf));

					/* nwctl_cli v6addr add called (update) */
					snprintf(cmdbuf, sizeof(cmdbuf),
						"/usr/sbin/nwctl_cli v6addr add %s %s/%d -vltime %d -origin dhcpv6",
						ifpfx.ifconf->ifname, ifaddr_buf, ifpfx.plen, pinfo->vltime);
					system(cmdbuf);
				}
			}
		}
	}
	return 0;
}
#endif  /* USE_NWCTL */

static int
add_ifprefix(siteprefix, prefix, pconf)
	struct siteprefix *siteprefix;
	struct dhcp6_prefix *prefix;
	struct prefix_ifconf *pconf;
{
	struct dhcp6_ifprefix *ifpfx = NULL;

	if ((ifpfx = malloc(sizeof(*ifpfx))) == NULL) {
		debug_printf(LOG_NOTICE, FNAME,
		    "failed to allocate memory for ifprefix");
		return (-1);
	}
	memset(ifpfx, 0, sizeof(*ifpfx));

	/* Part of the processing of add_ifprefix() has been moved to make_ifprefix(). */
	if (make_ifprefix(prefix, pconf, ifpfx)) {
		goto bad;
	}

#ifdef USE_NWCTL
	char ifaddr_buf[64] = "";
	char cmdbuf[256] = "";
	struct stat sts;

	/* spcreate = 1 (create): LAN GUA add skip */
	/*   => IA-NA Address update and WAN GUA delete: nwctld send */
	if ((strcmp(pconf->ifname, IFNAME_LAN) != 0) ||
	    ((!pd_chk_wan_update()) && (iana_addr_delete == 0))) {
		inet_ntop(AF_INET6, &(ifpfx->ifaddr.sin6_addr), ifaddr_buf, sizeof(ifaddr_buf));
		/* nwctl_cli v6addr add called. */
		snprintf(cmdbuf, sizeof(cmdbuf),
			"/usr/sbin/nwctl_cli v6addr add %s %s/%d -vltime %d -origin dhcpv6",
			ifpfx->ifconf->ifname, ifaddr_buf, ifpfx->plen, prefix->vltime);
		system(cmdbuf);
	}
#else	/* USE_NWCTL */
	if (pd_ifaddrconf(IFADDRCONF_ADD, ifpfx))
		goto bad;
#endif	/* USE_NWCTL */

	/* add default route -> only ngn-pd */
	if ((ngn_pd_mode == 1) && (strcmp(pconf->ifname, argv_ifname) == 0)) {
#ifdef USE_NWCTL
		/* default route add use nwctld only */
		/* (If nwctld is not used, additional processing is required.) */
		if (pd_ipv6_route(IPv6_ROUTE_ADD) == -1)
			goto bad;
#endif  /* USE_NWCTL */
	}

	/* TODO: send a control message for other processes */

	TAILQ_INSERT_TAIL(&siteprefix->ifprefix_list, ifpfx, plink);

	return (0);

  bad:
	if (ifpfx)
		free(ifpfx);
	return (-1);
}

static int
make_ifprefix(prefix, pconf, ifpfx)
	struct dhcp6_prefix *prefix;
	struct prefix_ifconf *pconf;
	struct dhcp6_ifprefix *ifpfx;
{
	struct in6_addr *a;
	u_long sla_id;
	char *sp;
	int b, i;

	ifpfx->ifconf = pconf;

	ifpfx->paddr.sin6_family = AF_INET6;
#ifdef HAVE_SA_LEN
	ifpfx->paddr.sin6_len = sizeof(struct sockaddr_in6);
#endif
	ifpfx->paddr.sin6_addr = prefix->addr;
#ifdef VEXT_IPV6	/* Vendor Extension IPv6 */
	int wk_sla_len;

	if (prefix->plen > 64) {
		debug_printf(LOG_ERR, FNAME,
		    "prefix is %d, over 64!",prefix->plen);
		return (-1);
	}
	ifpfx->plen = 64;

	/* ifpfx->plen(64) - prefix->plen = pconf->sla_len */
	wk_sla_len = ifpfx->plen - prefix->plen;

	if (wk_sla_len < 0)
		pconf->sla_len = 0;
	else
		pconf->sla_len = wk_sla_len;

	/* copy prefix */
	a = &ifpfx->paddr.sin6_addr;
	b = prefix->plen;
	for (i = 0, b = prefix->plen; b > 0; b -= 8, i++)
		a->s6_addr[i] = prefix->addr.s6_addr[i];

	/* SLA ID */
	sla_id = htonl(pconf->sla_id);
	sp = ((char *)&sla_id + 3);
	i = (128 - pconf->ifid_len) / 8;
	for (b = pconf->sla_len; b > 7; b -= 8, sp--)
		a->s6_addr[--i] = *sp;
	if (b)
		a->s6_addr[--i] |= *sp;

	/* configure the corresponding address */
	ifpfx->ifaddr = ifpfx->paddr;
	for (i = 15; i >= pconf->ifid_len / 8; i--)
		ifpfx->ifaddr.sin6_addr.s6_addr[i] = pconf->ifid[i];

#ifdef RTK_AP_IPV6
	/* create file-"/var/dhcp6pd.conf" to record prefix for delegation */
	if (strcmp(pconf->ifname, IFNAME_LAN) == 0)
		create_Dhcp6pdCfgFile(&ifpfx->paddr.sin6_addr, prefix,pconf->sla_len);
#endif	/* RTK_AP_IPV6 */

	/* LAN GUA address infomation file */
	if (strcmp(pconf->ifname, IFNAME_LAN) == 0) {
		create_lan_address_file(&ifpfx->ifaddr.sin6_addr, prefix, ifpfx->plen);
	}

#else			/* Vendor Extension IPv6 */
	ifpfx->plen = prefix->plen + pconf->sla_len;
	/*
	 * XXX: our current implementation assumes ifid len is a multiple of 8
	 */
	if ((pconf->ifid_len % 8) != 0) {
		debug_printf(LOG_ERR, FNAME,
		    "assumption failure on the length of interface ID");
		return (-1);
	}
	if (ifpfx->plen + pconf->ifid_len < 0 ||
	    ifpfx->plen + pconf->ifid_len > 128) {
		debug_printf(LOG_INFO, FNAME,
			"invalid prefix length %d + %d + %d",
			prefix->plen, pconf->sla_len, pconf->ifid_len);
		return (-1);
	}

	/* copy prefix and SLA ID */
	a = &ifpfx->paddr.sin6_addr;
	b = prefix->plen;
	for (i = 0, b = prefix->plen; b > 0; b -= 8, i++)
		a->s6_addr[i] = prefix->addr.s6_addr[i];
	sla_id = htonl(pconf->sla_id);
	sp = ((char *)&sla_id + 3);
	i = (128 - pconf->ifid_len) / 8;
	for (b = pconf->sla_len; b > 7; b -= 8, sp--)
		a->s6_addr[--i] = *sp;
	if (b)
		a->s6_addr[--i] |= *sp;

	/* configure the corresponding address */
	ifpfx->ifaddr = ifpfx->paddr;
	for (i = 15; i >= pconf->ifid_len / 8; i--)
		ifpfx->ifaddr.sin6_addr.s6_addr[i] = pconf->ifid[i];
#endif			/* Vendor Extension IPv6 */
	return (0);
}

#ifdef RTK_AP_IPV6
static int
create_Dhcp6pdCfgFile(addr, prefix, sla_len)
	struct in6_addr *addr;
	struct dhcp6_prefix *prefix;
	int sla_len;
{
	FILE *fp;		/*for dhcp6pd.conf*/
	struct in6_addr paddr;
	int plen;
	unsigned int	ptime;
	unsigned int	vtime;
	char tmpBuf[256] = {0};
	char tmpBufNew[256] = {0};
	int pid;
	int size;
	char *oldStr = NULL;

	/* open /var/dhcp6pd.conf */
	fp = fopen(DHCP6PD_CONF_FILE, "r");
	if (fp != NULL) {
		fgets(tmpBuf, sizeof(tmpBuf), fp);
		fclose(fp);
	}

	/*order is prefix,length,prefer time, valid time*/
	sprintf(tmpBufNew,"%04x%04x%04x%04x%04x%04x%04x%04x %d	%u	%u",
			addr->s6_addr16[0],addr->s6_addr16[1],addr->s6_addr16[2],
			addr->s6_addr16[3],addr->s6_addr16[4],addr->s6_addr16[5],
			addr->s6_addr16[6],addr->s6_addr16[7],prefix->plen+sla_len,prefix->pltime,prefix->vltime);
	if (strcmp(tmpBufNew,tmpBuf) == 0) {	//need not change
		return 0;
	} else {
		fp = fopen(DHCP6PD_CONF_FILE, "w+");
		if (NULL == fp) {
			return -1;
		}
		fprintf(fp,"%s",tmpBufNew);
		fclose(fp);
		//return 0;
		system("echo \"not update\">/var/dhcp6pd_need_update");
	}
	return 0;
}
#endif	/* RTK_AP_IPV6 */

static int
create_lan_address_file(addr, prefix, plen)
	struct in6_addr *addr;
	struct dhcp6_prefix *prefix;
	int plen;
{
	FILE *fp;
	char ifaddr_buf[64] = "";
	char tmpBuf[256] = {0};
	char tmpBufNew[256] = {0};

	/*
	 * /tmp/new_ipv6_lan_address
	 *   => [LAN ipv6 address] [Prefix length] [Valid Lifetime] [Preferred Lifetime]
	 */
	fp = fopen(PD_LAN_ADDRESS_FILE, "r");
	if (NULL != fp) {
		fgets(tmpBuf, sizeof(tmpBuf), fp);
		fclose(fp);
	}

	inet_ntop(AF_INET6, addr, ifaddr_buf, sizeof(ifaddr_buf));
	sprintf(tmpBufNew, "%s %d %lu %lu", ifaddr_buf, plen, prefix->vltime, prefix->pltime);

	if (strcmp(tmpBufNew,tmpBuf) == 0) {	//need not change
		return 0;
	} else {
		fp = fopen(PD_LAN_ADDRESS_FILE, "w+");
		if (NULL == fp) {
			return -1;
		}
		fprintf(fp, "%s", tmpBufNew);
		fclose(fp);
	}
	return 0;
}

static void
delete_lan_address_file(void)
{
	unlink(PD_LAN_ADDRESS_FILE);
	return;
}

#ifndef ND6_INFINITE_LIFETIME
#define ND6_INFINITE_LIFETIME 0xffffffff
#endif

static int
pd_ifaddrconf(cmd, ifpfx)
	ifaddrconf_cmd_t cmd;
	struct dhcp6_ifprefix *ifpfx;
{
	struct prefix_ifconf *pconf;

	pconf = ifpfx->ifconf;
	return (ifaddrconf(cmd, pconf->ifname, &ifpfx->ifaddr, ifpfx->plen, 
	    ND6_INFINITE_LIFETIME, ND6_INFINITE_LIFETIME));
}

static int
pd_ipv6_route(cmd)
	int cmd;
{
	FILE *fp;
	struct in6_rtmsg rt;
	struct in6_addr dst = IN6ADDR_ANY_INIT;
	char *line = NULL;
	char *tp;
	size_t len = 0;
	int ret;
	char cmdbuf[256] = "";
	char if_buf[64] = "";
	char dst_buf[64] = "";
	char gw_buf[64] = "";

	/* Initialize */
	memset(&rt, 0, sizeof(rt));

	/* flags */
	rt.rtmsg_flags = RTF_UP;

	/* Destination (::/0) */
	memcpy(&(rt.rtmsg_dst), &dst, sizeof(struct in6_addr));
	rt.rtmsg_dst_len = 0;
	rt.rtmsg_flags |= RTF_DEFAULT;

	fp = fopen("/var/dhcp6c_nexthop", "r+");
	if (fp == NULL) {
		debug_printf(LOG_NOTICE, FNAME, "failed to fopen nexthop file");
		return (-1);
	}

	if (getline(&line, &len, fp) != -1) {
		/* NextHop%Interface */
		tp = strtok(line, "%");

		/* NextHop */
		if (inet_pton(AF_INET6, tp, &(rt.rtmsg_gateway)) != 1) {
			debug_printf(LOG_NOTICE, FNAME, "failed to inet_pton nexthop");
			fclose(fp);
			if (line)
				free(line);
			return (-1);
		}
		rt.rtmsg_flags |= RTF_GATEWAY;

		tp = strtok(NULL, "%");

		snprintf(if_buf, sizeof(if_buf), "%s", tp);

		/* Interface */
		if (tp != NULL)
			rt.rtmsg_ifindex = if_nametoindex(tp);
	}
	fclose(fp);
	if (line)
		free(line);

	/* Metric */
	rt.rtmsg_metric = 1024;

	inet_ntop(AF_INET6, &(rt.rtmsg_dst), dst_buf, sizeof(dst_buf));
	inet_ntop(AF_INET6, &(rt.rtmsg_gateway), gw_buf, sizeof(gw_buf));
	if (cmd == IPv6_ROUTE_ADD) {
		/* nwctl_cli v6route add called. */
		snprintf(cmdbuf, sizeof(cmdbuf),
			 "/usr/sbin/nwctl_cli v6route add %s/%d -gw %s -if %s -metric %d -origin dhcpv6",
			 dst_buf, rt.rtmsg_dst_len, gw_buf, if_buf, rt.rtmsg_metric);
	} else if (cmd == IPv6_ROUTE_DEL) {
		/* nwctl_cli v6route del called. */
		snprintf(cmdbuf, sizeof(cmdbuf),
			 "/usr/sbin/nwctl_cli v6route del %s/%d -gw %s -if %s -origin dhcpv6",
			 dst_buf, rt.rtmsg_dst_len, gw_buf, if_buf);
	} else {
		debug_printf(LOG_NOTICE, FNAME, "failed to default route setting");
		return (-1);
	}

	system(cmdbuf);

	return (0);
}

static int
pd_chk_wan_update()
{
	/* WAN GUA update check */
	/* return = 1: PD Prefix update */
	/* return = 0: IA Address update */
	/* Caution! : if "wan_ra_prefix_mode" used -> wan_ra_prefix_mode check need */

	if (ngn_pd_mode) {
		return (1);
	} else if (iana_addr_nothing) {
		return (1);
	} else {
		return (0);
	}
}
