#include <linux/ip.h>
#include <linux/ipv6.h>
#include <linux/inet.h>
#include <net/tcp.h>
#include <net/vext_netlog.h>
#include <net/secfilter_log.h>

static DEFINE_RATELIMIT_STATE(ratelimit, 5*HZ, 10);

static int
sflt_log_ip6(struct sk_buff *skb, u32 logid, const char *dir)
{
	const struct ipv6hdr *ip6h;
	__u16 sport = 0, dport = 0;
	char id_str[8];
	char prot_str[8];
	char sport_str[8];
	char dport_str[8];
	char saddr_str[INET6_ADDRSTRLEN];
	char daddr_str[INET6_ADDRSTRLEN];

	if (skb->protocol != htons(ETH_P_IPV6))
		return 0;

	ip6h = ipv6_hdr(skb);

	switch (ip6h->nexthdr) {
	case IPPROTO_TCP: {
		const struct tcphdr *th;
		unsigned int plen = skb_transport_offset(skb) + sizeof(struct tcphdr);

		if (unlikely(!pskb_may_pull(skb, plen)))
			break;

		th = (struct tcphdr *)(ip6h + 1);
		sport = ntohs(th->source);
		dport = ntohs(th->dest);
	}
		break;
	case IPPROTO_UDP: {
		const struct udphdr *uh;
		unsigned int plen = skb_transport_offset(skb) + sizeof(struct udphdr);

		if (unlikely(!pskb_may_pull(skb, plen)))
			break;

		uh = (struct udphdr *)(ip6h + 1);
		sport = ntohs(uh->source);
		dport = ntohs(uh->dest);
	}
		break;
	case IPPROTO_ICMPV6: {
		sport = 0;
		dport = 0;
	}
		break;
	}

	/* logid */
	snprintf(id_str, sizeof(id_str), "%u", logid);
	/* protocol */
	snprintf(prot_str, sizeof(prot_str), "%u", ip6h->nexthdr);
	/* src addr */
	snprintf(saddr_str, sizeof(saddr_str), "%pI6", &ip6h->saddr);
	/* src port */
	snprintf(sport_str, sizeof(sport_str), "%u", sport);
	/* dst addr */
	snprintf(daddr_str, sizeof(daddr_str), "%pI6", &ip6h->daddr);
	/* dst port */
	snprintf(dport_str, sizeof(dport_str), "%u", dport);

	return vext_netlog_output(SFLT_ARGS, SFLT_EXEC, id_str, prot_str, saddr_str, sport_str, daddr_str, dport_str, dir, skb->dev->name);
}

static int
sflt_log_ip(struct sk_buff *skb, u32 logid, const char *dir)
{
	const struct iphdr *iph;
	__u16 sport = 0, dport = 0;
	char id_str[8];
	char prot_str[8];
	char sport_str[8];
	char dport_str[8];
	char saddr_str[INET_ADDRSTRLEN];
	char daddr_str[INET_ADDRSTRLEN];

	if (skb->protocol != htons(ETH_P_IP))
		return 0;

	iph = ip_hdr(skb);

	switch (iph->protocol) {
	case IPPROTO_TCP: {
		const struct tcphdr *th;
		unsigned int plen = skb_transport_offset(skb) + sizeof(struct tcphdr);

		if (unlikely(!pskb_may_pull(skb, plen)))
			break;

		th = (struct tcphdr *)(iph + 1);
		sport = ntohs(th->source);
		dport = ntohs(th->dest);
	}
		break;
	case IPPROTO_UDP: {
		const struct udphdr *uh;
		unsigned int plen = skb_transport_offset(skb) + sizeof(struct udphdr);

		if (unlikely(!pskb_may_pull(skb, plen)))
			break;

		uh = (struct udphdr *)(iph + 1);
		sport = ntohs(uh->source);
		dport = ntohs(uh->dest);
	}
		break;
	case IPPROTO_ICMP: {
		sport = 0;
		dport = 0;
	}
		break;
	}

	/* logid */
	snprintf(id_str, sizeof(id_str), "%u", logid);
	/* protocol */
	snprintf(prot_str, sizeof(prot_str), "%u", iph->protocol);
	/* src addr */
	snprintf(saddr_str, sizeof(saddr_str), "%pI4", &iph->saddr);
	/* src port */
	snprintf(sport_str, sizeof(sport_str), "%u", sport);
	/* dst addr */
	snprintf(daddr_str, sizeof(daddr_str), "%pI4", &iph->daddr);
	/* dst port */
	snprintf(dport_str, sizeof(dport_str), "%u", dport);

	return vext_netlog_output(SFLT_ARGS, SFLT_EXEC, id_str, prot_str, saddr_str, sport_str, daddr_str, dport_str, dir, skb->dev->name);
}

int
sflt_log_output(struct sk_buff *skb, const u32 logid, const char *dir)
{
	int ret = 0;

	if (!__ratelimit(&ratelimit))
		return 0;

	switch(logid) {
	case IP_LAND :
	case IP_SPOOF :
	case IP_SMURF :
		ret = sflt_log_ip(skb, logid, dir);
		if (ret < 0)
			pr_err("%s:%d sflt_log_ip", __func__, __LINE__);
		break;
	case IP6_LAND :
	case IP6_SPOOF :
		ret = sflt_log_ip6(skb, logid, dir);
		if (ret < 0)
			pr_err("%s:%d sflt_log_ip6", __func__, __LINE__);
		break;
	default:
		pr_err("%s:%d Unknown logid %u", __func__, __LINE__, logid);
		ret = -EINVAL;
	}

	return ret;
}
EXPORT_SYMBOL(sflt_log_output);
