/*
 * ----------------------------------------------------------------------------
 * "THE BEER-WARE LICENSE" (Revision 42):
 * <m-asama@ginzado.co.jp> wrote this file. As long as you retain this notice
 * you can do whatever you want with this stuff. If we meet some day, and you
 * think this stuff is worth it, you can buy me a beer in return Masakazu Asama
 * ----------------------------------------------------------------------------
 */
/*
 * ipmap.c		"ip map"
 * 
 * Authors:	Masakazu Asama, <m-asama@ginzado.ne.jp>
 *
 * (C) 2020 Albert Chan <Albert_Chan@alphanetworks.com>
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <linux/ip.h>
#include <linux/if_map.h>

#include "utils.h"
#include "ip_common.h"

static void usage(void) __attribute__((noreturn));

static void usage(void)
{
	fprintf(stderr, "Usage: ip map { add-rule | del-rule | show-rule } [ NAME ]\n");
	fprintf(stderr, "          [ ipv6-prefix ADDR ] [ ipv4-prefix ADDR ] [ ea-length NUMBER ]\n");
	fprintf(stderr, "          [ psid-offset NUMBER ] [ dev PHYS_DEV ]\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Where: NAME := STRING\n");
	fprintf(stderr, "       ADDR := IP_ADDRESS\n");
	exit(-1);
}

static inline int ipv6_addr_any(const struct in6_addr *a)
{
	return (a->s6_addr32[0] | a->s6_addr32[1] | a->s6_addr32[2] | a->s6_addr32[3]) == 0;
}

static inline void ipv6_addr_prefix(struct in6_addr *pfx, const struct in6_addr *addr, int plen)
{
	int o = plen >> 3, b = plen & 0x7;
	memset(pfx->s6_addr, 0, sizeof(pfx->s6_addr));
	memcpy(pfx->s6_addr, addr, o);
	if (b != 0)
		pfx->s6_addr[o] = addr->s6_addr[o] & (0xff00 >> b);
}

static inline int ipv6_addr_equal(const struct in6_addr *a1, const struct in6_addr *a2)
{
	return ((a1->s6_addr32[0] ^ a2->s6_addr32[0]) |
		(a1->s6_addr32[1] ^ a2->s6_addr32[1]) |
		(a1->s6_addr32[2] ^ a2->s6_addr32[2]) |
		(a1->s6_addr32[3] ^ a2->s6_addr32[3])) == 0;
}

static int map_ioctl(int cmd, const char *basedev, const char *name, void *p)
{
	struct ifreq ifr;
	int fd;
	int err;

	strncpy(ifr.ifr_name, name, IFNAMSIZ - 1);

	ifr.ifr_ifru.ifru_data = p;
	fd = socket(AF_INET6, SOCK_DGRAM, 0);
	err = ioctl(fd, cmd, &ifr);
	if (err)
	{
		fprintf(stderr, "%s ioctl failed: %s\n",
				name, strerror(errno));
	}
	close(fd);
	return err;
}

static int parse_args(int argc, char **argv, int cmd, char *name, struct ip6_tnl_fmr *rp)
{
	int count = 0;
	inet_prefix prefix;

	memset(name, 0, IFNAMSIZ - 1);
	memset(rp, 0, sizeof(*rp));

	/* default settings */
	rp->offset = 6;

	while( argc > 0)
	{
		if( strcmp(*argv, "ipv6-prefix") == 0)
		{
			NEXT_ARG();
			if( get_prefix(&prefix, *argv, AF_INET6))
			{
				fprintf(stderr, "Invalid ipv6-prefix: %s.\n", *argv);
				exit(-1);
			}
			memcpy(&rp->ip6_prefix, prefix.data, 16);
			rp->ip6_prefix_len = prefix.bitlen;
		}
		else if( strcmp(*argv, "ipv4-prefix") == 0)
		{
			NEXT_ARG();
			if( get_prefix(&prefix, *argv, AF_INET))
			{
				fprintf(stderr, "Invalid ipv4-prefix: %s.\n", *argv);
				exit(-1);
			}
			memcpy(&rp->ip4_prefix, prefix.data, 4);
			rp->ip4_prefix_len = prefix.bitlen;
		}
		else if( strcmp(*argv, "ea-length") == 0)
		{
			unsigned uval;
			NEXT_ARG();
			if( get_unsigned(&uval, *argv, 0))
			{
				fprintf(stderr, "Invalid ea-length: %s.\n", *argv);
				exit(-1);
			}
			if( uval > 48)
			{
				fprintf(stderr, "Invalid ea-length: %s.\n", *argv);
				exit(-1);
			}
			rp->ea_len = uval;
		}
		else if( strcmp(*argv, "psid-offset") == 0)
		{
			unsigned uval;
			NEXT_ARG();
			if( get_unsigned(&uval, *argv, 0))
			{
				fprintf(stderr, "Invalid psid-offset: %s.\n", *argv);
				exit(-1);
			}
			if( uval > 16)
			{
				fprintf(stderr, "Invalid psid-offset: %s.\n", *argv);
				exit(-1);
			}
			rp->offset = uval;
		}
		else if( strcmp(*argv, "dev") == 0)
		{
			NEXT_ARG();
			strncpy( name, *argv, IFNAMSIZ - 1);
		}
		else
		{
			printf("[ipmap]argv: %s; name: %s\n", *argv, name);
			if( strcmp(*argv, "name") == 0)
			{	NEXT_ARG();}
			else if( matches(*argv, "help") == 0)
			{	usage();}
			if( name[0])
			{
				fprintf(stderr, "Duplicated name:%s.\n", *argv);
				exit(-1);
			}
		}
		count++;
		argc--; argv++;
	}

	if( !name[0])
	{
		fprintf(stderr, "name or dev required.\n");
		exit(-1);
	}

	if( cmd == SIOCADDMAPRULES)
	{
		struct in6_addr t;
		__u32 amask;
		/*
		 * ipv6-prefix validation.
		 */
		if( ipv6_addr_any(&rp->ip6_prefix))
		{
			fprintf(stderr, "ipv6-prefix required or invalid.\n");
			exit(-1);
		}
		ipv6_addr_prefix(&t, &rp->ip6_prefix, rp->ip6_prefix_len);
		if( !ipv6_addr_equal(&t, &rp->ip6_prefix))
		{
			fprintf(stderr, "ipv6-prefix invalid.\n");
			exit(-1);
		}
		/*
		 * ipv4-prefix validation.
		 */
		if( rp->ip4_prefix.s_addr == htonl(INADDR_ANY)
			|| rp->ip4_prefix_len == 0)
		{
			fprintf(stderr, "ipv4-prefix required or invalid.\n");
			exit(-1);
		}
		amask = 0xffffffff;
		if( rp->ip4_prefix_len < 32)
		{	amask = ~((1 << (32 - rp->ip4_prefix_len)) - 1);}
		if( (ntohl(rp->ip4_prefix.s_addr) & amask) != ntohl(rp->ip4_prefix.s_addr))
		{
			fprintf(stderr, "ipv4-prefix invalid.\n");
			exit(-1);
		}
		/*
		 * other validations.
		 */
		if( rp->ea_len > 48)
		{
			fprintf(stderr, "ea-length required or invalid.\n");
			exit(-1);
		}
		if( rp->offset > 16)
		{
			fprintf(stderr, "psid-offset invalid.\n");
			exit(-1);
		}
		if( rp->ip6_prefix_len + rp->ea_len > 64)
		{
			fprintf(stderr, "ipv6-prefix length or ea-length invalid.\n");
			exit(-1);
		}
		if( rp->ip4_prefix_len + rp->ea_len + rp->offset > 48)
		{
			fprintf(stderr, "ipv4-prefix length or psid-prefix length or ea-length or psid-offset invalid.\n");
			exit(-1);
		}
	}

	if( cmd == SIOCDELMAPRULES)
	{
		if( ipv6_addr_any(&rp->ip6_prefix))
		{
			fprintf(stderr, "ipv6-prefix required or invalid.\n");
			exit(-1);
		}
		if( rp->ip4_prefix.s_addr == htonl(INADDR_ANY))
		{
			fprintf(stderr, "ipv4-prefix required or invalid.\n");
			exit(-1);
		}
	}
	return 0;
}

static int do_show_rule(int argc, char **argv)
{
	struct map_parm p, *pptr = NULL;
	struct ip6_tnl_fmr rp, *rpptr = NULL;
	char name[IFNAMSIZ];
	char s1[1024];
	char s2[1024];
	unsigned int rule_num = 0;
	int i = 0;

	if( parse_args(argc, argv, SIOCGETMAPRULES, name, &rp) < 0)
	{	return -1;}

	if (map_ioctl( SIOCGETMAP, "ip6tnl0", name, &p))
	{	return -1;}
	rule_num = p.rule_num;

	pptr = malloc(sizeof(struct map_parm) + sizeof(struct ip6_tnl_fmr) * rule_num);
	if( !pptr)
	{	return -1;}
	if (map_ioctl( SIOCGETMAPRULES, "ip6tnl0", name, pptr))
	{	return -1;}
	printf("\n");
	printf(" IPv6 prefix, IPv4 prefix, EA-bits length, PSID offset.\n");
	for (i = 0; i < pptr->rule_num; i++)
	{
		rpptr = &pptr->rule[i];
		printf(" %7d: %26s/%-3d %15s/%-2d %2d %2d\n",
				i,
				inet_ntop(AF_INET6, &rpptr->ip6_prefix, s1, sizeof(s1)),
				rpptr->ip6_prefix_len,
				format_host(AF_INET, 4, &rpptr->ip4_prefix, s2, sizeof(s2)),
				rpptr->ip4_prefix_len,
				rpptr->ea_len,
				rpptr->offset);
	}
	printf("\n");
	free(pptr);

	return 0;
}

#define ARRAY_MARGIN	(256)

static int do_add_rule(int argc, char **argv)
{
	struct map_parm *pptr;
	struct ip6_tnl_fmr *rpptr;
	char name[IFNAMSIZ];

	pptr = malloc(sizeof(struct map_parm) + sizeof(struct ip6_tnl_fmr));
	if( !pptr)
	{	return -1;}
	rpptr = &pptr->rule[0];

	//printf("pptr = %p, rpptr = %p, sizeof(*pptr) = %d.\n", pptr, rpptr, sizeof(*pptr));

	if( parse_args(argc, argv, SIOCADDMAPRULES, name, rpptr) < 0)
	{	return -1;}

	pptr->rule_num = 1;

	if (map_ioctl( SIOCADDMAPRULES, "ip6tnl0", name, pptr))
	{	return -1;}
	
	return 0;
}

static int do_del_rule(int argc, char **argv)
{
	struct map_parm *pptr;
	struct ip6_tnl_fmr *rpptr;
	char name[IFNAMSIZ];

	pptr = malloc(sizeof(struct map_parm) + sizeof(struct ip6_tnl_fmr));
	if (!pptr)
	{	return -1;}
	rpptr = &pptr->rule[0];

	if (parse_args(argc, argv, SIOCDELMAPRULES, name, rpptr) < 0)
	{	return -1;}

	pptr->rule_num = 1;

	if (map_ioctl( SIOCDELMAPRULES, "ip6tnl0", name, pptr))
	{	return -1;}

	return 0;
}

int do_ipmap(int argc, char **argv)
{
	if (argc > 0)
	{
		if (matches(*argv, "add-rule") == 0)
		{	return do_add_rule(argc - 1, argv + 1);}
		if (matches(*argv, "del-rule") == 0)
		{	return do_del_rule(argc - 1, argv + 1);}
		if (matches(*argv, "show-rule") == 0)
			return do_show_rule(argc - 1, argv + 1);
		if (matches(*argv, "help") == 0)
		{	usage();}
	}
	fprintf(stderr, "Command \"%s\" is unknown, try \"ip map help\".\n", *argv);
	exit(-1);
}
