/*
 * "Copyright (c) 2018 Intel Corporation
 * DISTRIBUTABLE AS SAMPLE SOURCE SOFTWARE
 * This Distributable As Sample Source Software is subject to the terms
 * and conditions of the Intel Software License Agreement for the Intel(R)
 * Cable GW Software Development Kit"
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "ppv4_regs.h"

#define PP_REG_ADDR_FPGA_MASK           0x0FFFFFFF

/**
 * DDR_BUFF_LIST_ADDR should contain DDR_NUM_BUFFERS of 2048
 * bytes. DDR_PTR_LIST_ADDR should contain DDR_NUM_BUFFERS
 * pointers pointing to DDR_BUFF_LIST_ADDR shifted right by 6
 * bits
 *
 * Example:
 * 0x0  ==> 0x100000, 0x20 ==> 0x100004, 0x40 ==> 0x100008...
 */

#define DDR_PTR_LIST_ADDR               0x100000  /* 32 * 4 = 128 Bytes */
#define DDR_NUM_BUFFERS                 32
#define DDR_BUFF_LIST_ADDR              0x0
#define DDR_QM_BASE                     0x00331100 /* 524,288 Bytes */
#define DDR_SI_BASE                     0x002F1100 /* 262,144 Bytes */
#define DDR_HT1_BASE                    0x00201100 /* 4096 Bytes */
#define DDR_HT2_BASE                    0x00202100 /* 4096 Bytes */

#define PORT0_RING_SIZE                 1
#define PORT0_RING_ADDR_HIGH            0x00       /* Highest byte */
#define PORT0_RING_ADDR_LOW             0xF4000000 /* Lower 4 bytes */

#define PORT1_RING_SIZE                 1
#define PORT1_RING_ADDR_HIGH            0x00       /* Highest byte */
#define PORT1_RING_ADDR_LOW             0x00000000 /* Lower 4 bytes */

#define PORT2_RING_SIZE                 1
#define PORT2_RING_ADDR_HIGH            0x00       /* Highest byte */
#define PORT2_RING_ADDR_LOW             0x00000000 /* Lower 4 bytes */

#define PORT3_RING_SIZE                 1
#define PORT3_RING_ADDR_HIGH            0x00       /* Highest byte */
#define PORT3_RING_ADDR_LOW             0x00000000 /* Lower 4 bytes */

#define REG_WR_32_PRINT(addr, data)                  \
	(printf("echo \"0x%X 0x%X\" > write32\n",    \
		addr & PP_REG_ADDR_FPGA_MASK, data))

//#define SLEEP_PRINT(sec)          \
//	printf("sleep %d\n", sec)

#define SLEEP_PRINT(sec)

struct port_config {
	unsigned int port_id;       /* 0 - 3 */
	unsigned int headroom_size; /* 0 - 255 */
	unsigned int tailroom_size; /* 0 - 255 */
};

struct session_config {
	unsigned int ingress_port_id; /* 0 - 3 */
	unsigned int ingress_tc;      /* 0 - 3 */
	unsigned int egress_color;    /* 0 - 2 */
	unsigned int egress_port_id;  /* 0 - 3 */
};

static unsigned int num_sessions;

void add_port(struct port_config *port_cfg)
{
	unsigned int reg_val = 0;

	printf("# Adding port %d\n", port_cfg->port_id);

	reg_val |= 1 << 27;
	reg_val |= port_cfg->tailroom_size << 16;
	reg_val |= port_cfg->headroom_size;
	REG_WR_32_PRINT(RX_DMA_PORT_CFG_REG(port_cfg->port_id), reg_val);
}

void add_session(struct session_config *session_cfg)
{
	unsigned int new_session_addr = 0xf1940000 + (num_sessions * 32);
	unsigned int cam_ram_offset = ((session_cfg->ingress_port_id << 2) |
				       (session_cfg->ingress_tc)) * 4;

	printf("# Adding exception session %d\n", num_sessions);

	/* Exception session X:
	 * 000000ff  0 session index, ff recipe index
	 * 00001c00  00001c ps offset, 0 base policy
	 * 00000000  0 color, 00 tdox, 000 dest Q(=egress port), 00 egress Q
	 * 10004000  1 msb of checker flags, 4 UD size ,means 32 bytes
	 * 00000000  SGC
	 * 00000000  SGC
	 * 01000000  TBM
	 * 41005bf3  0 segA, 10 BslPriority, 1 policy len bitmap, 5cf3 TBM
	 */
	/*RAM CHK_ALT_SI_RAM data*/
	REG_WR_32_PRINT(new_session_addr, 0x000000ff);
	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 4, 0x00001c00);
	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 8,
			session_cfg->egress_color << 30  |
			session_cfg->egress_port_id << 8 |
			session_cfg->egress_port_id);

	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 12, 0x10004000);
	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 16, 0x00000000);
	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 20, 0x00000000);
	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 24, 0x01000000);
	/*RAM CHK_ALT_SI_RAM*/
	REG_WR_32_PRINT(new_session_addr + 28, 0x41005bf3);

	REG_WR_32_PRINT(0xf1920000 + cam_ram_offset,
			num_sessions << 24 | num_sessions << 16 |
			num_sessions << 8  | num_sessions);

	num_sessions++;
}

void regs_default(void)
{
#ifdef BM_RESET_ALL_REGS
	unsigned int idx, idx2;
#endif

	/* Infra Clocks */
	REG_WR_32_PRINT(0xF0001000, 0x00000003);
	REG_WR_32_PRINT(0xF0001010, 0x00000003);
	REG_WR_32_PRINT(0xF0001020, 0x00000003);
	REG_WR_32_PRINT(0xF0001050, 0x00000003);
	REG_WR_32_PRINT(0xF0000004, 0x00000000);
	REG_WR_32_PRINT(0xF000000C, 0x00000011);

	/* RPB */
	REG_WR_32_PRINT(0xF0018860, 0x00020310);
	REG_WR_32_PRINT(0xF0018868, 0x00020310);
	REG_WR_32_PRINT(0xF0018870, 0x00020310);
	REG_WR_32_PRINT(0xF0018878, 0x00220310);
	REG_WR_32_PRINT(0xF0018880, 0x00020310);
	REG_WR_32_PRINT(0xF0018888, 0x00020310);
	REG_WR_32_PRINT(0xF0018890, 0x00020310);
	REG_WR_32_PRINT(0xF0018898, 0x00020310);
	REG_WR_32_PRINT(0xF001889C, 0x00020310);
	REG_WR_32_PRINT(0xF00188A0, 0x00020310);
	REG_WR_32_PRINT(0xF00188A4, 0x00020310);
	REG_WR_32_PRINT(0xF00187E0, 0x00000010);
	REG_WR_32_PRINT(0xF001883C, 0x00000000);
	REG_WR_32_PRINT(0xF00187E4, 0x00000000);
	REG_WR_32_PRINT(0xF0010080, 0x00000000);
	REG_WR_32_PRINT(0xF0010084, 0xF0080000);
	REG_WR_32_PRINT(0xF0018830, 0x3fa16);
	REG_WR_32_PRINT(0xF0018834, 0x3c4dc);
	REG_WR_32_PRINT(0xF0018844, 0x7ff);
	REG_WR_32_PRINT(0xF0018848, 0x7f6);
	REG_WR_32_PRINT(0xF00187c0, 0x0);
	REG_WR_32_PRINT(0xF0018580, 0x15a16);
	REG_WR_32_PRINT(0xF00186a0, 0x14e42);
	REG_WR_32_PRINT(0xF0018320, 0x0);
	REG_WR_32_PRINT(0xF0018800, 0x1);
	REG_WR_32_PRINT(0xF0018200, 0x1);
	REG_WR_32_PRINT(0xF0018100, 0x1a16);
	REG_WR_32_PRINT(0xF0018220, 0xe42);
	REG_WR_32_PRINT(0xF0018320, 0x2000);
	REG_WR_32_PRINT(0xF0018480, 0x14e42);
	REG_WR_32_PRINT(0xF00185a0, 0x12ac6);
	REG_WR_32_PRINT(0xF0018340, 0x0);
	REG_WR_32_PRINT(0xF0018800, 0x1);
	REG_WR_32_PRINT(0xF0018200, 0x3);
	REG_WR_32_PRINT(0xF0018120, 0x0);
	REG_WR_32_PRINT(0xF0018240, 0x0);
	REG_WR_32_PRINT(0xF0018340, 0x0);
	REG_WR_32_PRINT(0xF00184a0, 0x15a16);
	REG_WR_32_PRINT(0xF00185c0, 0x14e42);
	REG_WR_32_PRINT(0xF0018360, 0x0);
	REG_WR_32_PRINT(0xF0018800, 0x1);
	REG_WR_32_PRINT(0xF0018200, 0x7);
	REG_WR_32_PRINT(0xF0018140, 0x0);
	REG_WR_32_PRINT(0xF0018260, 0x0);
	REG_WR_32_PRINT(0xF0018360, 0x0);
	REG_WR_32_PRINT(0xF00184c0, 0x15a16);
	REG_WR_32_PRINT(0xF00185e0, 0x14e42);
	REG_WR_32_PRINT(0xF0018380, 0x0);
	REG_WR_32_PRINT(0xF0018800, 0x1);
	REG_WR_32_PRINT(0xF0018200, 0xf);
	REG_WR_32_PRINT(0xF0018160, 0x0);
	REG_WR_32_PRINT(0xF0018280, 0x0);
	REG_WR_32_PRINT(0xF0018380, 0x0);
	REG_WR_32_PRINT(0xF00184e0, 0x15a16);
	REG_WR_32_PRINT(0xF0018600, 0x14e42);
	REG_WR_32_PRINT(0xF00187c0, 0x16000);
	REG_WR_32_PRINT(0xF00187c4, 0x0);
	REG_WR_32_PRINT(0xF0018584, 0x15a16);
	REG_WR_32_PRINT(0xF00186a4, 0x14e42);
	REG_WR_32_PRINT(0xF0018324, 0x0);
	REG_WR_32_PRINT(0xF0018804, 0x1);
	REG_WR_32_PRINT(0xF0018204, 0x0);
	REG_WR_32_PRINT(0xF0018104, 0x1a16);
	REG_WR_32_PRINT(0xF0018224, 0xe42);
	REG_WR_32_PRINT(0xF0018324, 0x2000);
	REG_WR_32_PRINT(0xF0018484, 0x14e42);
	REG_WR_32_PRINT(0xF00185a4, 0x12ac6);
	REG_WR_32_PRINT(0xF0018344, 0x0);
	REG_WR_32_PRINT(0xF0018804, 0x1);
	REG_WR_32_PRINT(0xF0018204, 0x0);
	REG_WR_32_PRINT(0xF0018124, 0x0);
	REG_WR_32_PRINT(0xF0018244, 0x0);
	REG_WR_32_PRINT(0xF0018344, 0x0);
	REG_WR_32_PRINT(0xF00184a4, 0x15a16);
	REG_WR_32_PRINT(0xF00185c4, 0x14e42);
	REG_WR_32_PRINT(0xF0018364, 0x0);
	REG_WR_32_PRINT(0xF0018804, 0x1);
	REG_WR_32_PRINT(0xF0018204, 0x0);
	REG_WR_32_PRINT(0xF0018144, 0x0);
	REG_WR_32_PRINT(0xF0018264, 0x0);
	REG_WR_32_PRINT(0xF0018364, 0x0);
	REG_WR_32_PRINT(0xF00184c4, 0x15a16);
	REG_WR_32_PRINT(0xF00185e4, 0x14e42);
	REG_WR_32_PRINT(0xF0018384, 0x0);
	REG_WR_32_PRINT(0xF0018804, 0x1);
	REG_WR_32_PRINT(0xF0018204, 0x0);
	REG_WR_32_PRINT(0xF0018164, 0x0);
	REG_WR_32_PRINT(0xF0018284, 0x0);
	REG_WR_32_PRINT(0xF0018384, 0x0);
	REG_WR_32_PRINT(0xF00184e4, 0x15a16);
	REG_WR_32_PRINT(0xF0018604, 0x14e42);
	REG_WR_32_PRINT(0xF00187c4, 0x16000);
	REG_WR_32_PRINT(0xF00187c8, 0x0);
	REG_WR_32_PRINT(0xF0018588, 0x5f00);
	REG_WR_32_PRINT(0xF00186a8, 0x5d00);
	REG_WR_32_PRINT(0xF0018328, 0x0);
	REG_WR_32_PRINT(0xF0018808, 0x1);
	REG_WR_32_PRINT(0xF0018208, 0x1);
	REG_WR_32_PRINT(0xF0018108, 0x1f00);
	REG_WR_32_PRINT(0xF0018228, 0x1d00);
	REG_WR_32_PRINT(0xF0018328, 0x2000);
	REG_WR_32_PRINT(0xF0018488, 0x5d00);
	REG_WR_32_PRINT(0xF00185a8, 0x5700);
	REG_WR_32_PRINT(0xF0018348, 0x0);
	REG_WR_32_PRINT(0xF0018808, 0x1);
	REG_WR_32_PRINT(0xF0018208, 0x3);
	REG_WR_32_PRINT(0xF0018128, 0x0);
	REG_WR_32_PRINT(0xF0018248, 0x0);
	REG_WR_32_PRINT(0xF0018348, 0x0);
	REG_WR_32_PRINT(0xF00184a8, 0x5f00);
	REG_WR_32_PRINT(0xF00185c8, 0x5d00);
	REG_WR_32_PRINT(0xF0018368, 0x0);
	REG_WR_32_PRINT(0xF0018808, 0x1);
	REG_WR_32_PRINT(0xF0018208, 0x7);
	REG_WR_32_PRINT(0xF0018148, 0x0);
	REG_WR_32_PRINT(0xF0018268, 0x0);
	REG_WR_32_PRINT(0xF0018368, 0x0);
	REG_WR_32_PRINT(0xF00184c8, 0x5f00);
	REG_WR_32_PRINT(0xF00185e8, 0x5d00);
	REG_WR_32_PRINT(0xF0018388, 0x0);
	REG_WR_32_PRINT(0xF0018808, 0x1);
	REG_WR_32_PRINT(0xF0018208, 0xf);
	REG_WR_32_PRINT(0xF0018168, 0x0);
	REG_WR_32_PRINT(0xF0018288, 0x0);
	REG_WR_32_PRINT(0xF0018388, 0x0);
	REG_WR_32_PRINT(0xF00184e8, 0x5f00);
	REG_WR_32_PRINT(0xF0018608, 0x5d00);
	REG_WR_32_PRINT(0xF00187c8, 0x6000);
	REG_WR_32_PRINT(0xF00187cc, 0x0);
	REG_WR_32_PRINT(0xF001858c, 0x5f00);
	REG_WR_32_PRINT(0xF00186ac, 0x5d00);
	REG_WR_32_PRINT(0xF001832c, 0x0);
	REG_WR_32_PRINT(0xF001880c, 0x1);
	REG_WR_32_PRINT(0xF001820c, 0x0);
	REG_WR_32_PRINT(0xF001810c, 0x1f00);
	REG_WR_32_PRINT(0xF001822c, 0x1d00);
	REG_WR_32_PRINT(0xF001832c, 0x2000);
	REG_WR_32_PRINT(0xF001848c, 0x5d00);
	REG_WR_32_PRINT(0xF00185ac, 0x5700);
	REG_WR_32_PRINT(0xF001834c, 0x0);
	REG_WR_32_PRINT(0xF001880c, 0x1);
	REG_WR_32_PRINT(0xF001820c, 0x0);
	REG_WR_32_PRINT(0xF001812c, 0x0);
	REG_WR_32_PRINT(0xF001824c, 0x0);
	REG_WR_32_PRINT(0xF001834c, 0x0);
	REG_WR_32_PRINT(0xF00184ac, 0x5f00);
	REG_WR_32_PRINT(0xF00185cc, 0x5d00);
	REG_WR_32_PRINT(0xF001836c, 0x0);
	REG_WR_32_PRINT(0xF001880c, 0x1);
	REG_WR_32_PRINT(0xF001820c, 0x0);
	REG_WR_32_PRINT(0xF001814c, 0x0);
	REG_WR_32_PRINT(0xF001826c, 0x0);
	REG_WR_32_PRINT(0xF001836c, 0x0);
	REG_WR_32_PRINT(0xF00184cc, 0x5f00);
	REG_WR_32_PRINT(0xF00185ec, 0x5d00);
	REG_WR_32_PRINT(0xF001838c, 0x0);
	REG_WR_32_PRINT(0xF001880c, 0x1);
	REG_WR_32_PRINT(0xF001820c, 0x0);
	REG_WR_32_PRINT(0xF001816c, 0x0);
	REG_WR_32_PRINT(0xF001828c, 0x0);
	REG_WR_32_PRINT(0xF001838c, 0x0);
	REG_WR_32_PRINT(0xF00184ec, 0x5f00);
	REG_WR_32_PRINT(0xF001860c, 0x5d00);
	REG_WR_32_PRINT(0xF00187cc, 0x6000);

	/* Port distributor */
	REG_WR_32_PRINT(0xF0004000, 0x3);
	REG_WR_32_PRINT(0xF0004004, 0x3);
	REG_WR_32_PRINT(0xF0004008, 0x3);
	REG_WR_32_PRINT(0xF000400C, 0x3);

	/* Classifier */
	REG_WR_32_PRINT(0xF1C00004, 0x00010001);
	REG_WR_32_PRINT(0xF1C00008, 0x00000000);
	REG_WR_32_PRINT(0xF1C0008C, 0xfffffffe);
	REG_WR_32_PRINT(0xF1C00090, 0x00000025);
	REG_WR_32_PRINT(0xF1C00094, 0x00000001);
	REG_WR_32_PRINT(0xF1C0021C, 0xFFFFFFFF);
	REG_WR_32_PRINT(0xF1C0031C, DDR_HT1_BASE>>4);
	REG_WR_32_PRINT(0xF1C00320, DDR_HT2_BASE>>4);
	REG_WR_32_PRINT(0xF1C00324, DDR_SI_BASE>>4);
	REG_WR_32_PRINT(0xf1c00328, 0x00000009);

	/* Checker */
	REG_WR_32_PRINT(0xF1800010, 0);
	REG_WR_32_PRINT(0xF1800084, 0);
	REG_WR_32_PRINT(0xF1800000, 0x00000F0E);

	/* Modifier */
	REG_WR_32_PRINT(0xF2201FC0, 0x404DA80A);
	REG_WR_32_PRINT(0xF2201FC4, 0x002EBD81);
	REG_WR_32_PRINT(0xF2213F80, 0x4020ACF6);
	REG_WR_32_PRINT(0xF2213F84, 0xD37D7527);
	REG_WR_32_PRINT(0xF2213F88, 0x0000022B);
	REG_WR_32_PRINT(0xF2213F8C, 0x00000000);
	REG_WR_32_PRINT(0xF2223F80, 0x400033E7);
	REG_WR_32_PRINT(0xF2223F84, 0x260A23CF);
	REG_WR_32_PRINT(0xF2223F88, 0x00000114);
	REG_WR_32_PRINT(0xF2223F8C, 0x00000000);
	REG_WR_32_PRINT(0xF2241FC0, 0x403656C7);
	REG_WR_32_PRINT(0xF2241FC4, 0x000B1F83);
	REG_WR_32_PRINT(0xF2253F80, 0x4020D0B7);
	REG_WR_32_PRINT(0xF2253F84, 0x641A4387);
	REG_WR_32_PRINT(0xF2253F88, 0x00000003);
	REG_WR_32_PRINT(0xF2253F8C, 0x00000000);
	REG_WR_32_PRINT(0xF2261FC0, 0x4014D1CB);
	REG_WR_32_PRINT(0xF2261FC4, 0x00000107);
	REG_WR_32_PRINT(0xF2271FC0, 0x405270BE);
	REG_WR_32_PRINT(0xF2271FC4, 0x000787C9);
	REG_WR_32_PRINT(0xF2281FC0, 0x40575625);
	REG_WR_32_PRINT(0xF2281FC4, 0x0000DE47);
	REG_WR_32_PRINT(0xF23013D8, 0x00005340);
	REG_WR_32_PRINT(0xF2000000, 0x0000000F);

	/* Rx-Dma */
	REG_WR_32_PRINT(0xF0006004, 0x00000800);
	REG_WR_32_PRINT(0xF0006008, 0x00000800);
	REG_WR_32_PRINT(0xF000600C, 0x00000800);
	REG_WR_32_PRINT(0xF0006010, 0x00000800);
	REG_WR_32_PRINT(0xF0006014, 0x00000800);
	REG_WR_32_PRINT(0xF000695C, 0x0F00C000);
	REG_WR_32_PRINT(0xF0006964, 0x0F000800);
	REG_WR_32_PRINT(0xF0006534, 0x0F004000);
	REG_WR_32_PRINT(0xF0006968, 0x0F003000);
	REG_WR_32_PRINT(0xF0006028, 0x00000000);
	REG_WR_32_PRINT(0xF000602C, 0x00000000);
	REG_WR_32_PRINT(0xF0006030, 0x0000003C);
	REG_WR_32_PRINT(0xF0006034, 0x00000040);
	REG_WR_32_PRINT(0xF0006038, 0x00000080);
	REG_WR_32_PRINT(0xF000694C, 0x0000000A);

	/* Buffer Manager */
	REG_WR_32_PRINT(0xF0080000, 0x00000000);
	REG_WR_32_PRINT(0xF0080010, 0x00000000);
	REG_WR_32_PRINT(0xF0080014, 0x00000000);
	REG_WR_32_PRINT(0xF0080008, 0x00000000);

	/* Configure Pool 0 &* Policy 0 */
	REG_WR_32_PRINT(0xF0080020, DDR_NUM_BUFFERS);
	REG_WR_32_PRINT(0xF0080540, DDR_PTR_LIST_ADDR);
	REG_WR_32_PRINT(0xF0080580, 0x0);
	REG_WR_32_PRINT(0xF00805c0, DDR_NUM_BUFFERS);
	REG_WR_32_PRINT(0xF0080400, 0x0);
	REG_WR_32_PRINT(0xF0080440, 0x80);
	REG_WR_32_PRINT(0xF0080480, 0x0);
	REG_WR_32_PRINT(0xF00804c0, 0x1);
	REG_WR_32_PRINT(0xF0080500, 0x70);
	REG_WR_32_PRINT(0xF0080740, 0x200);
	REG_WR_32_PRINT(0xF0080004, 0x1);
	REG_WR_32_PRINT(0xF008000C, 0x0);
	REG_WR_32_PRINT(0xF0080008, 0x10001);
	REG_WR_32_PRINT(0xF008001C, 0x1);
	REG_WR_32_PRINT(0xF008001C, 0x0);
	REG_WR_32_PRINT(0xF0080100, DDR_NUM_BUFFERS);
	REG_WR_32_PRINT(0xF0094000, 0x10);
	REG_WR_32_PRINT(0xF0093000, 0x0);
	REG_WR_32_PRINT(0xF0092000, 0x0);
	REG_WR_32_PRINT(0xF0090000, DDR_NUM_BUFFERS);
	REG_WR_32_PRINT(0xF0091000, 0x8);
	REG_WR_32_PRINT(0xF0080200, 0x8);
	REG_WR_32_PRINT(0xF008001C, 0x1);

	SLEEP_PRINT(1);

	/*TX-MNGR: configure ports 0-3*/
	/*tx scheduler address low*/
	REG_WR_32_PRINT(0xf1010050, 0x00000000);
	/*tx scheduler address low 200000*/
	REG_WR_32_PRINT(0xf1010054, 0x00200000);
	/*uC addressH and size*/
	REG_WR_32_PRINT(0xf1010058, 0x00000900);
	/*uC addressL*/
	REG_WR_32_PRINT(0xf101005c, 0xdeadbe00);

	/*port0 ring size and addressH*/
	REG_WR_32_PRINT(0xf1010060,
			PORT0_RING_SIZE << 8 | PORT0_RING_ADDR_HIGH);
	/*port0 ring addressL */
	REG_WR_32_PRINT(0xf1010064, PORT0_RING_ADDR_LOW);
	/*port1 ring size and addressH*/
	REG_WR_32_PRINT(0xf1010068,
			PORT1_RING_SIZE << 8 | PORT1_RING_ADDR_HIGH);
	/*port1 ring addressL */
	REG_WR_32_PRINT(0xf101006C, PORT1_RING_ADDR_LOW);
	/*port2 ring size and addressH*/
	REG_WR_32_PRINT(0xf1010070,
			PORT2_RING_SIZE << 8 | PORT2_RING_ADDR_HIGH);
	/*port2 ring addressL */
	REG_WR_32_PRINT(0xf1010074, PORT2_RING_ADDR_LOW);
	/*port3 ring size and addressH*/
	REG_WR_32_PRINT(0xf1010078,
			PORT3_RING_SIZE << 8 | PORT3_RING_ADDR_HIGH);
	/*port3 ring addressL */
	REG_WR_32_PRINT(0xf101007C, PORT3_RING_ADDR_LOW);
	/*ctrl register*/
	REG_WR_32_PRINT(0xf101087c, 0x00000303);
	/*port0 override packet credit 0x20*/
	REG_WR_32_PRINT(0xf120040c, 0x00000064); //??
	/*port1 override packet credit 0x20*/
	REG_WR_32_PRINT(0xf120041c, 0x00000064); //??
	/*port2 override packet credit 0x20*/
	REG_WR_32_PRINT(0xf120042c, 0x00000064); //??
	/*port3 override packet credit 0x20*/
	REG_WR_32_PRINT(0xf120043c, 0x00000064); //??
	/*port 0-3 enable*/
	REG_WR_32_PRINT(0xf1010000, 0x0000000F);

	SLEEP_PRINT(1);

	/*TSCD_RGF__GLSCD_NUMOFCONFPORTS*/
	REG_WR_32_PRINT(0xf1000040, 0x00000100);
	/*TSCD_RGF__GLSCD_CMD_DEST_ADDR*/
	REG_WR_32_PRINT(0xf1000044, 0x00200400);
	/*IIL_TSCD_RGF__GLSCD_QUANTA*/
	REG_WR_32_PRINT(0xf1000080, 0x00000000);

	SLEEP_PRINT(1);

	/*IIL_TSCD_RGF__GLSCD_BWLCREDUPDATE*/
	REG_WR_32_PRINT(0xf1000148, 0x000030d4);
	/*IIL_TSCD_RGF__GLSCD_BWLLINESPERARB*/
	REG_WR_32_PRINT(0xf100014c, 0x00000010);

	/*IIL_TSCD_RGF__GLSCD_CREDITSPERQUANTA*/
	REG_WR_32_PRINT(0xf1000144, 0x00000014);
	/*IIL_TSCD_RGF__GLSCD_LANTCBCMDS*/
	REG_WR_32_PRINT(0xf1000154, 0x0000007f);
	/*IIL_TSCD_RGF__GLSCD_IFCTRL*/
	REG_WR_32_PRINT(0xf10000a8, 0x00000008);

	/* QM */
	/*Delete Enable*/
	//REG_WR_32_PRINT(0xf11c0004, 0x0);
	/*DDR_BASE_ADDR*/
	REG_WR_32_PRINT(0xf11c0008, DDR_QM_BASE>>4);
	/*IIL_QMGR_RGF__NUM_OF_PREFETCH_OUTSTANDING*/
	REG_WR_32_PRINT(0xf11c000c, 0x00000013);
	/*IIL_QMGR_RGF__FREE_Q_AEMPTY_THRESHOLD*/
	REG_WR_32_PRINT(0xf11c0014, 0x00000064);
	/*IIL_QMGR_RGF__CACHE_CLEAN_THRESHOLD*/
	REG_WR_32_PRINT(0xf11c0010, 0x000001e0);
	/*IIL_QMGR_RGF__EMPTY_POP_Q_CNT*/
	REG_WR_32_PRINT(0xf11c003c, 0x00000150);

	/*IIL_QMGR_RGF__FAST_QUEUE*/
	REG_WR_32_PRINT(0xf11c0300, 0xc6104711);

	/*IIL_QMGR_RGF__RTC_EN*/
	REG_WR_32_PRINT(0xf11c0400, 0xb0054bf6);

	/*enable and set 0x20 pages*/
	REG_WR_32_PRINT(0xf11c002c, 0x08000020);

	SLEEP_PRINT(1);

	/*total available resources*/
	REG_WR_32_PRINT(0xf1020004, 0x00001388);
	/*avg Q size P param*/
	REG_WR_32_PRINT(0xf1020008, 0x00000220);
	/*1 - prioirtized pop*/
	REG_WR_32_PRINT(0xf1020000, 0x00000001);
	/* 4 bits Q size bytes counter is 20 bits wide for calculations,
	 * with extra 5 bits. this parameter reduces the resolution of this
	 * counter, by up to 11 bits
	 */
	REG_WR_32_PRINT(0xf102000c, 0x0000000a);

	/* wred disable bits: 0-disable wred, 1-disbl min max,
	 * 2-disable qm_full, 3-use fixed prob
	 */
	REG_WR_32_PRINT(0xf1020084, 0x5);

	/*32bits used as threshold for queue size MTU*/
	REG_WR_32_PRINT(0xf102010c, 0x000005ea);

	SLEEP_PRINT(1);

	// Add Port 0
	REG_WR_32_PRINT(0xf1024000, 0x00000000);
	REG_WR_32_PRINT(0xf1010040, 0x1);
	REG_WR_32_PRINT(0xf106000c, 0x2);
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x21);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x80002);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x1);
	REG_WR_32_PRINT(0xf1022400, 0xffffffff);
	REG_WR_32_PRINT(0xf1022000, 0xffffffff);
	SLEEP_PRINT(1);

	// Add Port 1
	REG_WR_32_PRINT(0xf1010040, 0x3);
	REG_WR_32_PRINT(0xf106001c, 0x2);
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x121);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x80102);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x101);
	REG_WR_32_PRINT(0xf1022404, 0xffffffff);
	REG_WR_32_PRINT(0xf1022004, 0xffffffff);
	SLEEP_PRINT(1);

	// Add Port 2
	REG_WR_32_PRINT(0xf1010040, 0x7);
	REG_WR_32_PRINT(0xf106002c, 0x2);
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x221);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x80202);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x201);
	REG_WR_32_PRINT(0xf1022408, 0xffffffff);
	REG_WR_32_PRINT(0xf1022008, 0xffffffff);
	SLEEP_PRINT(1);

	// Add Port 3
	REG_WR_32_PRINT(0xf1010040, 0xF);
	REG_WR_32_PRINT(0xf106003c, 0x2);
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x321);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x80302);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x301);
	REG_WR_32_PRINT(0xf102240C, 0xffffffff);
	REG_WR_32_PRINT(0xf102200C, 0xffffffff);
	SLEEP_PRINT(1);

	/* Set Port - Suspend port 0 */
	REG_WR_32_PRINT(0xf10000a0, 0x0);
	REG_WR_32_PRINT(0xf100009c, 0x80006);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Update scheduler about */
	/* Q0 to be added to port 0          */
	REG_WR_32_PRINT(0xf10000a0, 0x100596);
	REG_WR_32_PRINT(0xf100009c, 0x3);
	REG_WR_32_PRINT(0xf10000a0, 0x10058b);
	REG_WR_32_PRINT(0xf100009c, 0x3);
	REG_WR_32_PRINT(0xf10000a0, 0x1005a3);
	REG_WR_32_PRINT(0xf100009c, 0x3);
	REG_WR_32_PRINT(0xf10000a0, 0x1005af);
	REG_WR_32_PRINT(0xf100009c, 0x3);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Add Queue 0 */
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10021);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10011);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x90002);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10001);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x10000100);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x51);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1023000, 0x0);
	REG_WR_32_PRINT(0xf10200bc, 0x0);
	REG_WR_32_PRINT(0xf1020080, 0x1);
	REG_WR_32_PRINT(0xf1020088, 0x0);
	REG_WR_32_PRINT(0xf102008c, 0x0);
	REG_WR_32_PRINT(0xf1020090, 0x0);
	REG_WR_32_PRINT(0xf1020094, 0x0);
	REG_WR_32_PRINT(0xf1020098, 0x0);
	REG_WR_32_PRINT(0xf10200a0, 0x0);
	REG_WR_32_PRINT(0xf10200a4, 0x0);
	REG_WR_32_PRINT(0xf10200a8, 0x0);
	REG_WR_32_PRINT(0xf10200b0, 0x0);
	REG_WR_32_PRINT(0xf10200b4, 0x40);
	REG_WR_32_PRINT(0xf10200c0, 0x3);
	SLEEP_PRINT(1);

	/* Set Port - Resume port 0 */
	REG_WR_32_PRINT(0xf10000a0, 0x2400);
	REG_WR_32_PRINT(0xf100009c, 0x3);
	REG_WR_32_PRINT(0xf10000a0, 0x0);
	REG_WR_32_PRINT(0xf100009c, 0x100006);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Suspend port 1 */
	REG_WR_32_PRINT(0xf10000a0, 0x1000);
	REG_WR_32_PRINT(0xf100009c, 0x80106);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Update scheduler about */
	/* Q1 to be added to port 1          */
	REG_WR_32_PRINT(0xf10000a0, 0x101596);
	REG_WR_32_PRINT(0xf100009c, 0x103);
	REG_WR_32_PRINT(0xf10000a0, 0x10158b);
	REG_WR_32_PRINT(0xf100009c, 0x103);
	REG_WR_32_PRINT(0xf10000a0, 0x1015a3);
	REG_WR_32_PRINT(0xf100009c, 0x103);
	REG_WR_32_PRINT(0xf10000a0, 0x1015af);
	REG_WR_32_PRINT(0xf100009c, 0x103);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Add Queue 1 */
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10121);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x1);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10111);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x400500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x90102);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10101);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x10100101);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x151);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1023004, 0x1);
	REG_WR_32_PRINT(0xf10200bc, 0x1);
	REG_WR_32_PRINT(0xf1020080, 0x1);
	REG_WR_32_PRINT(0xf1020088, 0x0);
	REG_WR_32_PRINT(0xf102008c, 0x0);
	REG_WR_32_PRINT(0xf1020090, 0x0);
	REG_WR_32_PRINT(0xf1020094, 0x0);
	REG_WR_32_PRINT(0xf1020098, 0x0);
	REG_WR_32_PRINT(0xf10200a0, 0x0);
	REG_WR_32_PRINT(0xf10200a4, 0x0);
	REG_WR_32_PRINT(0xf10200a8, 0x0);
	REG_WR_32_PRINT(0xf10200b0, 0x0);
	REG_WR_32_PRINT(0xf10200b4, 0x40);
	REG_WR_32_PRINT(0xf10200c0, 0x3);
	SLEEP_PRINT(1);

	/* Set Port - Resume port 1*/
	REG_WR_32_PRINT(0xf10000a0, 0x2400);
	REG_WR_32_PRINT(0xf100009c, 0x103);
	REG_WR_32_PRINT(0xf10000a0, 0x1000);
	REG_WR_32_PRINT(0xf100009c, 0x100106);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Suspend port 2 */
	REG_WR_32_PRINT(0xf10000a0, 0x2000);
	REG_WR_32_PRINT(0xf100009c, 0x80206);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Update scheduler about */
	/* Q2 to be added to port 2          */
	REG_WR_32_PRINT(0xf10000a0, 0x102596);
	REG_WR_32_PRINT(0xf100009c, 0x203);
	REG_WR_32_PRINT(0xf10000a0, 0x10258b);
	REG_WR_32_PRINT(0xf100009c, 0x203);
	REG_WR_32_PRINT(0xf10000a0, 0x1025a3);
	REG_WR_32_PRINT(0xf100009c, 0x203);
	REG_WR_32_PRINT(0xf10000a0, 0x1025af);
	REG_WR_32_PRINT(0xf100009c, 0x203);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Add Queue 2 */
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10221);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x2);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10211);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x800500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008c, 0x90202);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10201);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x10200102);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x251);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1023008, 0x2);
	REG_WR_32_PRINT(0xf10200bc, 0x2);
	REG_WR_32_PRINT(0xf1020080, 0x1);
	REG_WR_32_PRINT(0xf1020088, 0x0);
	REG_WR_32_PRINT(0xf102008c, 0x0);
	REG_WR_32_PRINT(0xf1020090, 0x0);
	REG_WR_32_PRINT(0xf1020094, 0x0);
	REG_WR_32_PRINT(0xf1020098, 0x0);
	REG_WR_32_PRINT(0xf10200a0, 0x0);
	REG_WR_32_PRINT(0xf10200a4, 0x0);
	REG_WR_32_PRINT(0xf10200a8, 0x0);
	REG_WR_32_PRINT(0xf10200b0, 0x0);
	REG_WR_32_PRINT(0xf10200b4, 0x40);
	REG_WR_32_PRINT(0xf10200c0, 0x3);
	SLEEP_PRINT(1);

	/* Set Port - Resume port 2 */
	REG_WR_32_PRINT(0xf10000a0, 0x2400);
	REG_WR_32_PRINT(0xf100009c, 0x203);
	REG_WR_32_PRINT(0xf10000a0, 0x2000);
	REG_WR_32_PRINT(0xf100009c, 0x100206);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Suspend port 3 */
	REG_WR_32_PRINT(0xf10000a0, 0x3000);
	REG_WR_32_PRINT(0xf100009c, 0x80306);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Set Port - Update scheduler about */
	/* Q3 to be added to port 3          */
	REG_WR_32_PRINT(0xf10000a0, 0x103596);
	REG_WR_32_PRINT(0xf100009c, 0x303);
	REG_WR_32_PRINT(0xf10000a0, 0x10358b);
	REG_WR_32_PRINT(0xf100009c, 0x303);
	REG_WR_32_PRINT(0xf10000a0, 0x1035a3);
	REG_WR_32_PRINT(0xf100009c, 0x303);
	REG_WR_32_PRINT(0xf10000a0, 0x1035af);
	REG_WR_32_PRINT(0xf100009c, 0x303);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/* Add Queue 3 */
	REG_WR_32_PRINT(0xf1000084, 0x0);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10321);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x3);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10311);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0xc00500);
	REG_WR_32_PRINT(0xf1000088, 0x4004000);
	REG_WR_32_PRINT(0xf100008C, 0x90302);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x10301);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf1000084, 0x10300103);
	REG_WR_32_PRINT(0xf1000088, 0x0);
	REG_WR_32_PRINT(0xf100008c, 0x0);
	REG_WR_32_PRINT(0xf1000090, 0x0);
	REG_WR_32_PRINT(0xf1000094, 0x351);
	SLEEP_PRINT(1);

	REG_WR_32_PRINT(0xf102300c, 0x3);
	REG_WR_32_PRINT(0xf10200bc, 0x3);
	REG_WR_32_PRINT(0xf1020080, 0x1);
	REG_WR_32_PRINT(0xf1020088, 0x0);
	REG_WR_32_PRINT(0xf102008c, 0x0);
	REG_WR_32_PRINT(0xf1020090, 0x0);
	REG_WR_32_PRINT(0xf1020094, 0x0);
	REG_WR_32_PRINT(0xf1020098, 0x0);
	REG_WR_32_PRINT(0xf10200a0, 0x0);
	REG_WR_32_PRINT(0xf10200a4, 0x0);
	REG_WR_32_PRINT(0xf10200a8, 0x0);
	REG_WR_32_PRINT(0xf10200b0, 0x0);
	REG_WR_32_PRINT(0xf10200b4, 0x40);
	REG_WR_32_PRINT(0xf10200c0, 0x3);
	SLEEP_PRINT(1);

	/* Set Port - Resume Port 3 */
	REG_WR_32_PRINT(0xf10000a0, 0x2400);
	REG_WR_32_PRINT(0xf100009c, 0x303);
	REG_WR_32_PRINT(0xf10000a0, 0x3000);
	REG_WR_32_PRINT(0xf100009c, 0x100306);
	REG_WR_32_PRINT(0xf100009c, 0x6);
	REG_WR_32_PRINT(0xf10000a8, 0x19);
	SLEEP_PRINT(1);

	/*This write is needed to overcome FPGA bug*/
	REG_WR_32_PRINT(0xf1014010, 0x00000000);
}

int main(int argc, char *argv[])
{
	struct port_config cfg;
	struct session_config session_cfg;

	regs_default();

	cfg.port_id = 0;
	cfg.headroom_size = 16;
	cfg.tailroom_size = 0;
	add_port(&cfg);

	cfg.port_id = 1;
	cfg.headroom_size = 16;
	cfg.tailroom_size = 0;
	add_port(&cfg);

	session_cfg.ingress_port_id = 1;
	session_cfg.ingress_tc = 0;
	session_cfg.egress_color = 0;
	session_cfg.egress_port_id = 1;
	add_session(&session_cfg);

	session_cfg.ingress_port_id = 0;
	session_cfg.ingress_tc = 2;
	session_cfg.egress_color = 0;
	session_cfg.egress_port_id = 0;
	add_session(&session_cfg);

	return 0;
}
