/*
 *	Forwarding decision
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/err.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/netpoll.h>
#include <linux/skbuff.h>
#include <linux/if_vlan.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
//---	need to check if destination address is broadcast or not.
unsigned char bcast_mac_addr[6] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
#endif

#if IS_ENABLED(CONFIG_MCAST_HELPER)
void (*five_tuple_br_info_ptr)(struct sk_buff *skb) = NULL;
void (*five_tuple_br_info_hook)(struct sk_buff *skb) = NULL;
EXPORT_SYMBOL(five_tuple_br_info_ptr);
EXPORT_SYMBOL(five_tuple_br_info_hook);
int mch_br_capture_pkt = 0;
EXPORT_SYMBOL(mch_br_capture_pkt);
#endif
#ifdef CONFIG_MCAST_SNOOPING
#define IS_MCAST_ADDR 0x1
#endif

#ifdef CONFIG_BRIDGE_GUEST_ZONE
typedef int (*gz_block_cb)(const struct net_bridge_port *, const struct sk_buff *);
gz_block_cb gz_block=NULL;

void register_gz_block_init_callback(gz_block_cb funa)
{
	gz_block = funa;
}	

void unregister_gz_block_init_callback(void)
{
	gz_block = NULL;
}

EXPORT_SYMBOL(register_gz_block_init_callback);
EXPORT_SYMBOL(unregister_gz_block_init_callback);
#endif

/* Don't forward packets to originating port or forwarding disabled */
static inline int should_deliver(const struct net_bridge_port *p,
				 const struct sk_buff *skb)
{
	struct net_bridge_vlan_group *vg;

	vg = nbp_vlan_group_rcu(p);
#ifdef CONFIG_BRIDGE_GUEST_ZONE
	if (!(((p->flags & BR_HAIRPIN_MODE) || skb->dev != p->dev) &&
		br_allowed_egress(vg, skb) && p->state == BR_STATE_FORWARDING &&
		nbp_switchdev_allowed_egress(p, skb)))
		return 0;

	if (gz_block)
	{
		if (gz_block(p, skb))
			return 0;
	}
	
	return 1;
#else
	return ((p->flags & BR_HAIRPIN_MODE) || skb->dev != p->dev) &&
		br_allowed_egress(vg, skb) && p->state == BR_STATE_FORWARDING &&
		nbp_switchdev_allowed_egress(p, skb);
#endif
}

int br_dev_queue_push_xmit(struct net *net, struct sock *sk, struct sk_buff *skb)
{
	skb_push(skb, ETH_HLEN);
	if (!is_skb_forwardable(skb->dev, skb))
		goto drop;

	br_drop_fake_rtable(skb);

	if (skb->ip_summed == CHECKSUM_PARTIAL &&
	    (skb->protocol == htons(ETH_P_8021Q) ||
	     skb->protocol == htons(ETH_P_8021AD))) {
		int depth;

		if (!__vlan_get_protocol(skb, skb->protocol, &depth))
			goto drop;

		skb_set_network_header(skb, depth);
	}

	dev_queue_xmit(skb);

	return 0;

drop:
	kfree_skb(skb);
	return 0;
}
EXPORT_SYMBOL_GPL(br_dev_queue_push_xmit);

int br_forward_finish(struct net *net, struct sock *sk, struct sk_buff *skb)
{
	return NF_HOOK(NFPROTO_BRIDGE, NF_BR_POST_ROUTING,
		       net, sk, skb, NULL, skb->dev,
		       br_dev_queue_push_xmit);

}
EXPORT_SYMBOL_GPL(br_forward_finish);

static void __br_forward(const struct net_bridge_port *to,
			 struct sk_buff *skb, bool local_orig)
{
	struct net_bridge_vlan_group *vg;
	struct net_device *indev;
	struct net *net;
	int br_hook;

	vg = nbp_vlan_group_rcu(to);
	skb = br_handle_vlan(to->br, vg, skb);
	if (!skb)
		return;

	indev = skb->dev;
	skb->dev = to->dev;
	if (!local_orig) {
		if (skb_warn_if_lro(skb)) {
			kfree_skb(skb);
			return;
		}
		br_hook = NF_BR_FORWARD;
		skb_forward_csum(skb);
		net = dev_net(indev);
	} else {
		if (unlikely(netpoll_tx_running(to->br->dev))) {
			skb_push(skb, ETH_HLEN);
			if (!is_skb_forwardable(skb->dev, skb))
				kfree_skb(skb);
			else
				br_netpoll_send_skb(to, skb);
			return;
		}
		br_hook = NF_BR_LOCAL_OUT;
		net = dev_net(skb->dev);
		indev = NULL;
	}

	NF_HOOK(NFPROTO_BRIDGE, br_hook,
		net, NULL, skb, indev, skb->dev,
		br_forward_finish);
}

static int deliver_clone(const struct net_bridge_port *prev,
			 struct sk_buff *skb, bool local_orig)
{
	struct net_device *dev = BR_INPUT_SKB_CB(skb)->brdev;
#if IS_ENABLED(CONFIG_MCAST_HELPER)
	const unsigned char *dest = eth_hdr(skb)->h_dest;
#endif

	skb = skb_clone(skb, GFP_ATOMIC);
	if (!skb) {
		dev->stats.tx_dropped++;
		return -ENOMEM;
	}

#if IS_ENABLED(CONFIG_MCAST_HELPER)
	/* Send five tuple info to mcast helper */
	if (mch_br_capture_pkt == 1) {
		if ((ip_hdr(skb)->protocol == IPPROTO_UDP) || (ipv6_hdr(skb)->nexthdr == IPPROTO_UDP)) {
			if (five_tuple_br_info_ptr != NULL) {
				five_tuple_br_info_ptr(skb);
			}
		}
	}

	/* hook the UDP multicast to mcast helper */
	if (bridge_lanserver_hook == 1) {
		if (!is_broadcast_ether_addr(dest) && is_multicast_ether_addr(dest) && ((ip_hdr(skb)->protocol == IPPROTO_UDP) || (ipv6_hdr(skb)->nexthdr == IPPROTO_UDP))) {
			if (five_tuple_br_info_hook != NULL) {
					five_tuple_br_info_hook(skb);
				}
		}
	}
#endif

	__br_forward(prev, skb, local_orig);
	return 0;
}

/**
 * br_forward - forward a packet to a specific port
 * @to: destination port
 * @skb: packet being forwarded
 * @local_rcv: packet will be received locally after forwarding
 * @local_orig: packet is locally originated
 *
 * Should be called with rcu_read_lock.
 */
void br_forward(const struct net_bridge_port *to,
		struct sk_buff *skb, bool local_rcv, bool local_orig)
{
	if (to->flags & BR_ISOLATE_MODE)
		to = NULL;

	if (to && should_deliver(to, skb)) {
		if (local_rcv)
			deliver_clone(to, skb, local_orig);
		else
			__br_forward(to, skb, local_orig);
		return;
	}

	if (!local_rcv)
		kfree_skb(skb);
}
EXPORT_SYMBOL_GPL(br_forward);

static struct net_bridge_port *maybe_deliver(
	struct net_bridge_port *prev, struct net_bridge_port *p,
	struct sk_buff *skb, bool local_orig)
{
	int err;

	if (!should_deliver(p, skb))
		return prev;

	if (!prev)
		goto out;

#ifdef CONFIG_MCAST_SNOOPING
	if ((bridge_igmp_snooping || bridge_mld_snooping) &&
			(eth_hdr(skb)->h_dest[0] & IS_MCAST_ADDR) &&
			(br_selective_flood(prev, skb) == 0)) {
				prev = p;
				return p;
			}
#endif

	err = deliver_clone(prev, skb, local_orig);
	if (err)
		return ERR_PTR(err);

out:
	return p;
}

static void maybe_deliver_addr(struct net_bridge_port *p, struct sk_buff *skb,
			       const unsigned char *addr, bool local_orig)
{
	struct net_device *dev = BR_INPUT_SKB_CB(skb)->brdev;
	const unsigned char *src = eth_hdr(skb)->h_source;

	if (!should_deliver(p, skb))
		return;

	/* Even with hairpin, no soliloquies - prevent breaking IPv6 DAD */
	if (skb->dev == p->dev && ether_addr_equal(src, addr))
		return;

	skb = skb_copy(skb, GFP_ATOMIC);
	if (!skb) {
		dev->stats.tx_dropped++;
		return;
	}

	memcpy(eth_hdr(skb)->h_dest, addr, ETH_ALEN);
	__br_forward(p, skb, local_orig);
}

/* called under rcu_read_lock */
void br_flood(struct net_bridge *br, struct sk_buff *skb,
	      enum br_pkt_type pkt_type, bool local_rcv, bool local_orig)
{
	u8 igmp_type = br_multicast_igmp_type(skb);
	struct net_bridge_port *prev = NULL;
	struct net_bridge_port *p;

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	//---	if following status all exist, then do alphanetworks's multicast.
	//---	1.	bridge is snooping
	//---	2.	current packet has a multicast Destination MAC Address
	//---	3.	current packet doesn't have a boardcast destination MAC Address
	//---	4.	packet's protocol is ETH_P_IP(0x0800) or ETH_P_IPV6(0x86DD)
	if( br->snooping && is_multicast_ether_addr(dest) &&
		(memcmp(eth_hdr(skb)->h_dest, bcast_mac_addr, 6) != 0) && // non-broadcast packet ?!
		( (eth_hdr(skb)->h_proto == htons(ETH_P_IP)) ||
		(eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)) ) ) // either IPv4 or IPv6
	{
		if (local_rcv)
		{
			struct sk_buff *skb2;
			if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) {
				br->dev->stats.tx_dropped++;
				goto out;
			}
			do_alpha_multicast(br, skb2, local_orig);
		} else {
			do_alpha_multicast(br, skb, local_orig);
		}
		return;
	}
#endif

	list_for_each_entry_rcu(p, &br->port_list, list) {
		if (!local_orig && (p->flags & BR_ISOLATE_MODE))
			continue;
		/* Do not flood unicast traffic to ports that turn it off */
		if (pkt_type == BR_PKT_UNICAST && !(p->flags & BR_FLOOD))
			continue;
		/* Do not flood if mc off, except for traffic we originate */
		if (pkt_type == BR_PKT_MULTICAST &&
		    !(p->flags & BR_MCAST_FLOOD) && skb->dev != br->dev)
			continue;

		/* Do not flood to ports that enable proxy ARP */
		if (p->flags & BR_PROXYARP)
			continue;
		if ((p->flags & BR_PROXYARP_WIFI) &&
		    BR_INPUT_SKB_CB(skb)->proxyarp_replied)
			continue;

		prev = maybe_deliver(prev, p, skb, local_orig);
		if (IS_ERR(prev))
			goto out;
		if (prev == p)
			br_multicast_count(p->br, p, skb, igmp_type,
					   BR_MCAST_DIR_TX);
	}

	if (!prev)
		goto out;

	if (local_rcv) {
#ifdef CONFIG_MCAST_SNOOPING
	if ((bridge_igmp_snooping || bridge_mld_snooping) &&
			(eth_hdr(skb)->h_dest[0] & IS_MCAST_ADDR) &&
			(br_selective_flood(prev, skb) == 0)) {}
	//kfree_skb(skb);
	else
#endif
		deliver_clone(prev, skb, local_orig);
	} else {
#ifdef CONFIG_MCAST_SNOOPING
	if ((bridge_igmp_snooping || bridge_mld_snooping) &&
			(eth_hdr(skb)->h_dest[0] & IS_MCAST_ADDR) &&
			(br_selective_flood(prev, skb) == 0))
		kfree_skb(skb);
	else
#endif
		__br_forward(prev, skb, local_orig);
	}
	return;

out:
	if (!local_rcv)
		kfree_skb(skb);
}

#ifdef CONFIG_BRIDGE_IGMP_SNOOPING
/* called with rcu_read_lock */
void br_multicast_flood(struct net_bridge_mdb_entry *mdst,
			struct sk_buff *skb,
			bool local_rcv, bool local_orig)
{
	struct net_device *dev = BR_INPUT_SKB_CB(skb)->brdev;
	u8 igmp_type = br_multicast_igmp_type(skb);
	struct net_bridge *br = netdev_priv(dev);
	struct net_bridge_port *prev = NULL;
	struct net_bridge_port_group *p;
	struct hlist_node *rp;

	rp = rcu_dereference(hlist_first_rcu(&br->router_list));
	p = mdst ? rcu_dereference(mdst->ports) : NULL;
	while (p || rp) {
		struct net_bridge_port *port, *lport, *rport;

		lport = p ? p->port : NULL;
		rport = rp ? hlist_entry(rp, struct net_bridge_port, rlist) :
			     NULL;

		if ((unsigned long)lport > (unsigned long)rport) {
			port = lport;

			if (p->flags & MDB_PG_FLAGS_MCAST_TO_UCAST) {
				maybe_deliver_addr(lport, skb, p->eth_addr,
						   local_orig);
				goto delivered;
			}
		} else {
			port = rport;
		}

		prev = maybe_deliver(prev, port, skb, local_orig);
delivered:
		if (IS_ERR(prev))
			goto out;
		if (prev == port)
			br_multicast_count(port->br, port, skb, igmp_type,
					   BR_MCAST_DIR_TX);

		if ((unsigned long)lport >= (unsigned long)port)
			p = rcu_dereference(p->next);
		if ((unsigned long)rport >= (unsigned long)port)
			rp = rcu_dereference(hlist_next_rcu(rp));
	}

	if (!prev)
		goto out;

	if (local_rcv)
		deliver_clone(prev, skb, local_orig);
	else
		__br_forward(prev, skb, local_orig);
	return;

out:
	if (!local_rcv)
		kfree_skb(skb);
}
#endif

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
//---	IPv4 flooding list:
//---	1.	if ip header is incomplete									=>	don't flood
//---	2.	check protocol IGMP 										=>	flood it.
//---	3.	check ip UPNP												=>	flood it.
//---	4.	check ip NEC ADP(Aterm Discovery Protocol, 239.255.252.132)	=>	flood it.
//---	5.	check ip control packets address range(224.0.0.0/24)		=>	flood it.
static int in_ipv4_flooding_list(struct sk_buff *skb)
{
	struct iphdr *iph;
	unsigned long prefix;

	if(!pskb_may_pull(skb , sizeof(struct iphdr)))
		return 0; //no ip header, should i flood it? i'm not sure

	iph = ip_hdr(skb);
	
	if (iph->protocol == IPPROTO_IGMP) return 1;

	prefix = ntohl(iph->daddr) & 0xffffffff;

	//flooding UPNP 239.255.255.250
	if(prefix == 0xeffffffa)
		return 1;
		
	//flooding NEC ADP deamon: 239.255.252.132
	if(prefix == 0xeffffc84)
		return 1;
		
	prefix = ntohl(iph->daddr) & 0xffffff00;

	//no wireless enhance for multicast management ip.

	//flooding 224.0.0.0/24
	if(prefix == 0xe0000000)
		return 1;

	return 0;	
}

//---	IPv6 flooding list:
//---	1.	if ip header is incomplete									=>	don't flood
//---	2.	check ip control packets address range(ff02::/16)			=>	flood it.
static int in_ipv6_flooding_list(struct sk_buff *skb)
{
	struct ipv6hdr *ipv6h;
	unsigned short prefix;

	if(!pskb_may_pull(skb , sizeof(struct ipv6hdr)))
		return 0; //no ipv6 header, should i flood it? i'm not sure

	ipv6h = ipv6_hdr(skb);

	prefix = ntohs(ipv6h->daddr.s6_addr16[0]);

	//flooding ff02::/16
	if(prefix == 0xff02)
		return 1;

	return 0;
}

#define PKT_DEFAULT_BUDGET (50)
u64 pkt_time_range[2] = {0ULL , 0ULL}; //jiffies_64
unsigned long available_pkt = PKT_DEFAULT_BUDGET;

//---	reset counter per time range
static void housekeeping(void)
{
	u64 current_time;
	current_time = get_jiffies_64();

	if(!time_after_eq64(current_time , pkt_time_range[0]) ||
			!time_before_eq64(current_time , pkt_time_range[1]))
	{
		//reset our available_pkt
		pkt_time_range[0] = current_time;
		pkt_time_range[1] = current_time + HZ;
		available_pkt = PKT_DEFAULT_BUDGET;
	}	
}

//---	control flooding procedure through port by counter.
static int accept_flooding_by_bw(void)
{
	housekeeping();

	if(available_pkt == 0)
		return 0; //no budget for flooding

	--available_pkt;
	return 1;
}

//---	check if packet should flood no matter what it is.
static int should_flood(struct sk_buff *skb)
{
	struct ethhdr *dest = eth_hdr(skb);

	if(dest->h_proto == htons(ETH_P_IP))
		return in_ipv4_flooding_list(skb);

	if(dest->h_proto == htons(ETH_P_IPV6))
		return in_ipv6_flooding_list(skb);

	return 0;
}

//---	alpha network customized multicast.
//---	Step 1: check all port list, specified there interface and flood flag
//---	Step 2: if packet is wireless and don't need to flood, do wireless enhance case
//---	Step 3: if packet is wire or wireless and sould flood, do flooding procedure.
void do_alpha_multicast(struct net_bridge *br, struct sk_buff *skb, bool local_orig)
{
	struct net_bridge_port *p;
#if defined(CONFIG_BRIDGE_MULTICAST_FLOODING) || defined(CONFIG_BRIDGE_SOFTWARE_SNOOPING)
	int group_exist=0;
	group_exist = do_groupexist(br, skb);
#endif

	list_for_each_entry_rcu(p, &br->port_list, list) 
	{
		struct sk_buff *skb2;
		int wireless = atomic_read(&p->wireless_interface); //wireless interface
#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
		int forward = atomic_read(&p->multicast_forward);
#endif

		if((wireless == 1) && (!should_flood(skb))) //for some traffic, we always need to flood it (tom, 20111228)
		{
#ifdef CONFIG_BRIDGE_MULTICAST_FLOODING
			if(group_exist == 1)
#endif
				do_enhance(p, br, skb, local_orig);
#ifdef CONFIG_BRIDGE_MULTICAST_FLOODING
			else
			{
				//flooding the packet		
				if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) {
					br->dev->stats.tx_dropped++;
					kfree_skb(skb);
					return;
				}
				if (should_deliver(p,skb2))
					__br_forward(p, skb2, local_orig);
				else
					kfree_skb(skb2);
			}
#endif
		}
		else 
		{
#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
			//--- 	Flood in following case.
			//---	1.	not in flooding list, wire, and IPv4
			//---	2.	not in flooding list, wire, IPv6, and multicast forward on
			//---	3.	In Flooding list.
			if((!should_flood(skb))
				&& ((eth_hdr(skb)->h_proto == htons(ETH_P_IP)) || ((eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)) && (forward == 1))))
#else
			//for wired interface, we flood the traffic
			//for wireless interface, we use bandwidth limit to flood it
			if(wireless != 1 || accept_flooding_by_bw())
#endif
			{
#ifdef CONFIG_BRIDGE_SOFTWARE_SNOOPING
				if(group_exist == 1)
					do_snooping(p, br, skb, local_orig);
				else
				{
#endif
					//flooding the packet		
					if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) {
						br->dev->stats.tx_dropped++;
						kfree_skb(skb);
						return;
					}
					if (should_deliver(p,skb2))
						__br_forward(p, skb2, local_orig);
					else
						kfree_skb(skb2);
#ifdef CONFIG_BRIDGE_SOFTWARE_SNOOPING
				}
#endif
			}
#if defined(CONFIG_BRIDGE_MULTICAST_FORWARDING) && defined(CONFIG_BRIDGE_MULTICAST_FLOODING)
			else if(should_flood(skb) || (group_exist != 1))
#elif CONFIG_BRIDGE_MULTICAST_FORWARDING
			else if(should_flood(skb))
#endif
#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
			{
				//flooding the packet		
				if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) {
					br->dev->stats.tx_dropped++;
					kfree_skb(skb);
					return;
				}
				if (should_deliver(p,skb2))
					__br_forward(p, skb2, local_orig);
				else
					kfree_skb(skb2);
			}
#endif
		}
	} //list_f	
	kfree_skb(skb);
}

void do_enhance(struct net_bridge_port *p, struct net_bridge *br, struct sk_buff *skb, bool local_orig)
{
	struct port_group_mac *g;
	struct sk_buff *skb2;
	int found =0;
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	int i;
	unsigned char mac_addr[6]={0,0,0,0,0,0};
#endif
	/*  does group address stored in table ? */
	rcu_read_lock();
	list_for_each_entry_rcu(g, &p->igmp_group_list, list)
	{
		struct ethhdr * dest;
		struct port_member_mac *m;
		dest = eth_hdr(skb);
		//---	check if destination is stored in group mac list.
		if(!memcmp( dest->h_dest, g->grpmac, 6))
		{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			for (i=0; i<6; ++i)
				mac_addr[i] = 0;
#endif
			//---	if destination is stored, then check group member list.
			list_for_each_entry_rcu(m, &g->member_list, list)
			{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
				if(memcmp(m->member_mac,mac_addr, 6))
				{
#endif
				//---	if destination and group member both are stored in list
				//---	copy the socket buffer skb to skb2.
				if ((skb2 = skb_copy(skb, GFP_ATOMIC)) == NULL)
				{
					//---	the memory isn't enough, drop this packet.
					br->dev->stats.tx_dropped++;
					//kfree_skb(skb);
					rcu_read_unlock();
					return;
				}
				//--- get the destination address, check if it should be deliver to output port.
				dest = eth_hdr(skb2);					
				memcpy(dest->h_dest, m->member_mac, sizeof(uint8_t)*6);
				if (should_deliver(p, skb2))
				{
					__br_forward(p, skb2, local_orig);
					found=1;
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
					memcpy(mac_addr, m->member_mac, 6);
#endif
				}
				else
					kfree_skb(skb2);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
				}
#endif
			}
		}
	}
	rcu_read_unlock();
}
#endif

#if defined(CONFIG_BRIDGE_MULTICAST_FLOODING) || defined(CONFIG_BRIDGE_SOFTWARE_SNOOPING)
int do_groupexist(struct net_bridge *br, struct sk_buff *skb)
{
	struct net_bridge_port *p;
	struct port_group_mac *g;

	rcu_read_lock();
	list_for_each_entry_rcu(p, &br->port_list, list) 
	{
		if (p->state == BR_STATE_DISABLED ||
		    p->state == BR_STATE_BLOCKING)
			continue;

		/*  does group address stored in table ? */
		list_for_each_entry_rcu(g, &p->igmp_group_list, list)
		{
			struct ethhdr * dest;
			dest = eth_hdr(skb);
			if(!memcmp( dest->h_dest, g->grpmac, 6))
			{
				rcu_read_unlock();
				return 1;
			}
		}
	}
	rcu_read_unlock();
	return 0;
}

#ifdef CONFIG_BRIDGE_SOFTWARE_SNOOPING
void do_snooping(struct net_bridge_port *p, struct net_bridge *br, struct sk_buff *skb, bool local_orig)
{
	struct port_group_mac *g;
	struct sk_buff *skb2;
	int found =0;
	/*  does group address stored in table ? */
	rcu_read_lock();
	list_for_each_entry_rcu(g, &p->igmp_group_list, list)
	{
		struct ethhdr * dest;
		//struct port_member_mac *m;
		dest = eth_hdr(skb);
		if(!memcmp( dest->h_dest, g->grpmac, 6))
		{
			if ((skb2 = skb_copy(skb, GFP_ATOMIC)) == NULL)
			{
				br->dev->stats.tx_dropped++;
				//kfree_skb(skb);
				rcu_read_unlock();
				return;
			}

			if (should_deliver(p, skb2))
			{
				__br_forward(p, skb2, local_orig);
				found=1;
			}
			else
				kfree_skb(skb2);
		}
	}
	rcu_read_unlock();
}
#endif

#endif