/* (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2006 Netfilter Core Team <coreteam@netfilter.org>
 * (C) 2008 Patrick McHardy <kaber@trash.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/types.h>
#include <linux/random.h>
#include <linux/netfilter.h>
#include <linux/export.h>

#include <net/netfilter/nf_nat.h>
#include <net/netfilter/nf_nat_core.h>
#include <net/netfilter/nf_nat_l3proto.h>
#include <net/netfilter/nf_nat_l4proto.h>

bool nf_nat_l4proto_in_range(const struct nf_conntrack_tuple *tuple,
			     enum nf_nat_manip_type maniptype,
			     const union nf_conntrack_man_proto *min,
			     const union nf_conntrack_man_proto *max)
{
	__be16 port;

	if (maniptype == NF_NAT_MANIP_SRC)
		port = tuple->src.u.all;
	else
		port = tuple->dst.u.all;

	return ntohs(port) >= ntohs(min->all) &&
	       ntohs(port) <= ntohs(max->all);
}
EXPORT_SYMBOL_GPL(nf_nat_l4proto_in_range);

#ifdef CONFIG_SNAT_MAPE
//---   2020-0521 Add counter to check how many packets are finding unique tuple.
static u16 enter_unique_tuple_counter = 0;  //---   2019-0221 To fix SNAT unstable issue.
#endif  //---   CONFIG_SNAT_MAPE end

void nf_nat_l4proto_unique_tuple(const struct nf_nat_l3proto *l3proto,
				 struct nf_conntrack_tuple *tuple,
				 const struct nf_nat_range *range,
				 enum nf_nat_manip_type maniptype,
				 const struct nf_conn *ct,
				 u16 *rover)
{
	unsigned int range_size, min, max, i;
	__be16 *portptr;
	u_int16_t off;
#ifdef CONFIG_SNAT_MAPE
	//---   2020-0103 For Map-E SNAPT flow.
	__be16 mask = 0x0000, tmp_psid = 0x0000;
	//---   2020-0521 To fix SNAT unstable issue.
	u16 cur_counter = 0;
#endif  //---   CONFIG_SNAT_MAPE end
#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
	bool	use_time_wait;
	unsigned int	min_time_stamp;
	u_int16_t	min_time_stamp_off;
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

#ifdef CONFIG_SNAT_MAPE
	//---   2020-0521 To fix SNAT unstable issue.
	enter_unique_tuple_counter++;
#endif  //---   CONFIG_SNAT_MAPE end
	if (maniptype == NF_NAT_MANIP_SRC)
		portptr = &tuple->src.u.all;
	else
		portptr = &tuple->dst.u.all;

	/* If no range specified... */
	if (!(range->flags & NF_NAT_RANGE_PROTO_SPECIFIED)) {
		/* If it's dst rewrite, can't change port */
		if (maniptype == NF_NAT_MANIP_DST)
		{
#ifdef CONFIG_SNAT_MAPE
			//---   2020-0521 To fix SNAT unstable issue.
			enter_unique_tuple_counter--;
#endif  //---   CONFIG_SNAT_MAPE end
			return;
		}

		if (ntohs(*portptr) < 1024) {
			/* Loose convention: >> 512 is credential passing */
			if (ntohs(*portptr) < 512) {
				min = 1;
				range_size = 511 - min + 1;
			} else {
				min = 600;
				range_size = 1023 - min + 1;
			}
		} else {
			min = 1024;
			range_size = 65535 - 1024 + 1;
		}
	} else {
		min = ntohs(range->min_proto.all);
		max = ntohs(range->max_proto.all);
		if (unlikely(max < min))
			swap(max, min);
		range_size = max - min + 1;
	}

	if (range->flags & NF_NAT_RANGE_PROTO_RANDOM) {
		off = l3proto->secure_port(tuple, maniptype == NF_NAT_MANIP_SRC
						  ? tuple->dst.u.all
						  : tuple->src.u.all);
	} else if (range->flags & NF_NAT_RANGE_PROTO_RANDOM_FULLY) {
		off = prandom_u32();
	} else {
		off = *rover;
	}

#ifdef CONFIG_SNAT_MAPE
	//---   2020-0521 To fix SNAT unstable issue.
	cur_counter = enter_unique_tuple_counter;
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
	min_time_stamp = -1; //--- initial min_time_stamp = max unsigned int.
	use_time_wait = false;
	min_time_stamp_off = 0;
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

#ifdef CONFIG_SNAT_MAPE
	//---   2020-0521 To fix SNAT unstable issue.
	for( i = 0; i < range_size; i++) {
#else
	for (i = 0; ; ++off) {
#endif  //---   CONFIG_SNAT_MAPE end
		*portptr = htons(min + off % range_size);

#ifdef CONFIG_SNAT_MAPE
		//---   2020-0521 To fix SNAT unstable issue.
		//---	2020-0521 move offset increment to here.
		off++;
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_SNAT_MAPE
		//--- 2020-0103 check Map-E port set.
		if( ct->mark == MAP_NAPT_MARK && mape_enable)
		{
			//--- Check port set status.
			mask = ( 1 << mape_psid_length) - 1;
			mask <<= 16 - mape_psid_offset - mape_psid_length;
			tmp_psid = mape_psid;
			tmp_psid <<= 16 - mape_psid_offset - mape_psid_length;
			//--- if current port isn't in Map-E port set, skip to next port.
			if(( ntohs(*portptr) & mask) != tmp_psid)
			{	continue;}
		}
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_SNAT_MAPE
		//---   2020-0521 To fix SNAT unstable issue.
		//---   don't increase counter i here.
		if (nf_nat_used_tuple(tuple, ct))
#else
		if (++i != range_size && nf_nat_used_tuple(tuple, ct))
#endif  //---   CONFIG_SNAT_MAPE end
		{
#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
			//---   2020-0521 If current tuple is used by other, check if it is in TIME_WAIT case,
			//---   If yes, then save the tuple with the smallest timestamp as target tuple.
			if ((tuple->dst.protonum == IPPROTO_TCP) && 
				nf_conntrack_tuple_time_wait_reusable(tuple, ct, &use_time_wait, &min_time_stamp))
			{
#ifdef CONFIG_SNAT_MAPE
				min_time_stamp_off = off - 1;
#else
				min_time_stamp_off = off;
#endif  //---   CONFIG_SNAT_MAPE end
			}
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end
			continue;
		}

#ifdef CONFIG_SNAT_MAPE
		//---   2020-0521 To fix SNAT unstable issue.
		--cur_counter;
		//---   2020-0521 If there are more then one packet finding tuple before it,
		//---   skip this tuple one and try to find next one.
		if(cur_counter!=0)
		{   continue;}
#endif  //---   CONFIG_SNAT_MAPE end

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
		//---   2020-0521 If we check every port, but none are available.
		//---   Check whether there is a TIME_WAIT conntrack that can be used.
		if( i == range_size && use_time_wait)
		{
			//---   2020-0521 Set target port and reset off for next round.
			*portptr = htons(min + min_time_stamp_off % range_size);
			off = min_time_stamp_off++;
		}
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

		if (!(range->flags & NF_NAT_RANGE_PROTO_RANDOM_ALL))
			*rover = off;

#ifdef CONFIG_SNAT_MAPE
		//---   2020-0521   To fix SNAT unstable issue.
		enter_unique_tuple_counter--;
#endif  //---   CONFIG_SNAT_MAPE end

		return;
	}

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
	//---   2020-0521 If we check every port, but none are available.
	//---   Check whether there is a TIME_WAIT conntrack that can be used.
	if( i == range_size && use_time_wait)
	{
		//---   2020-0521 Set target port and reset off for next round.
		*portptr = htons(min + min_time_stamp_off % range_size);
		off = min_time_stamp_off++;
 	}
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end

#ifdef CONFIG_NF_NAT_TIME_WAIT_PATCH
    //---   2020-0521   Set Next round start port.
    if (!(range->flags & NF_NAT_RANGE_PROTO_RANDOM))
        *rover = off;
#endif  //---   CONFIG_NF_NAT_TIME_WAIT_PATCH end
#ifdef CONFIG_SNAT_MAPE
	//---   2020-0521   To fix SNAT unstable issue.
	enter_unique_tuple_counter--;
#endif  //---   CONFIG_SNAT_MAPE end
}
EXPORT_SYMBOL_GPL(nf_nat_l4proto_unique_tuple);

#if IS_ENABLED(CONFIG_NF_CT_NETLINK)
int nf_nat_l4proto_nlattr_to_range(struct nlattr *tb[],
				   struct nf_nat_range *range)
{
	if (tb[CTA_PROTONAT_PORT_MIN]) {
		range->min_proto.all = nla_get_be16(tb[CTA_PROTONAT_PORT_MIN]);
		range->max_proto.all = range->min_proto.all;
		range->flags |= NF_NAT_RANGE_PROTO_SPECIFIED;
	}
	if (tb[CTA_PROTONAT_PORT_MAX]) {
		range->max_proto.all = nla_get_be16(tb[CTA_PROTONAT_PORT_MAX]);
		range->flags |= NF_NAT_RANGE_PROTO_SPECIFIED;
	}
	return 0;
}
EXPORT_SYMBOL_GPL(nf_nat_l4proto_nlattr_to_range);
#endif
