/*
 *	Generic parts
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/llc.h>
#include <net/llc.h>
#include <net/stp.h>
#include <net/switchdev.h>

#include "br_private.h"

#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#endif

#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
extern int32 getGroupNum(struct net_bridge *br, uint32 ipVersion);
static unsigned int mCastQueryTimerCnt=0;
#endif

#ifdef CONFIG_BRIDGE_IGMP_QUERY
#include <linux/inetdevice.h>
int igmpVersion=2;
int igmpQueryEnabled=1;

/*igmpv3 general query*/
static unsigned char igmpV3QueryBuf[64]={   0x01,0x00,0x5e,0x00,0x00,0x01,  /*destination mac*/
                                            0x00,0x00,0x00,0x00,0x00,0x00,  /*offset:6,	source mac*/
                                            0x08,0x00,                      /*offset:12, Ethernet II Type: IPv4*/
                                            0x46,0x00,0x00,0x24,            /*offset:14, IP version(0x4), Header length(0x6[24bytes]), total length(0x0024[36])*/
                                            0x00,0x00,0x40,0x00,            /*offset:18*/
                                            0x01,0x02,0x00,0x00,            /*offset:22, time to live(0x01), IGMP(0x02), IPv4's header checksum(0x0000)*/
                                            0x00,0x00,0x00,0x00,            /*offset:26, source ip*/
                                            0xe0,0x00,0x00,0x01,            /*offset:30, destination ip*/
                                            0x94,0x04,0x00,0x00,            /*offset:34, router alert option (New from IGMPv3)*/
                                            0x11,0x64,0x00,0x00,            /*offset:38, Type(Membership Query 0x11), Max Response Time(0x64), IGMP's Checksum(0x0cfa)*/
                                            0x00,0x00,0x00,0x00,            /*offset:42, queried multicast ip address*/
                                            0x0a,0x3c,0x00,0x00,            /*offset:46*/
                                            0x00,0x00,0x00,0x00,            /*offset:50*/
                                            0x00,0x00,0x00,0x00,            /*offset:54*/
                                            0x00,0x00,0x00,0x00,            /*offset:58*/
                                            0x00,0x00                       /*offset:62*/									
                                        };			

/*igmpv2 general query*/
static unsigned char igmpV2QueryBuf[64]={   0x01,0x00,0x5e,0x00,0x00,0x01,  /*destination mac*/
                                            0x00,0x00,0x00,0x00,0x00,0x00,  /*offset:6,	source mac*/
                                            0x08,0x00,                      /*offset:12, Ethernet II Type: IPv4*/
                                            0x45,0x00,0x00,0x1c,            /*offset:14, IP version: 4, Header length(0x5[20bytes]), total length: 0x001c(28)*/
                                            0x00,0x00,0x40,0x00,            /*offset:18*/
                                            0x01,0x02,0x00,0x00,            /*offset:22, time to live(0x01), IGMP(0x02), IPv4's header checksum(0x0000)*/
                                            0x00,0x00,0x00,0x00,            /*offset:26, source IP*/
                                            0xe0,0x00,0x00,0x01,            /*offset:30, Destination IP*/
                                            0x11,0x64,0x0c,0xfa,            /*offset:34, Type(Membership Query 0x11), Max Response Time(0x64), IGMP's Checksum(0x0cfa)*/
                                            0x00,0x00,0x00,0x00,            /*offset:38, Group IP Address*/
                                            0x00,0x00,0x00,0x00,            /*offset:42*/
                                            0x00,0x00,0x00,0x00,            /*offset:46*/
                                            0x00,0x00,0x00,0x00,            /*offset:50*/
                                            0x00,0x00,0x00,0x00,            /*offset:54*/
                                            0x00,0x00,0x00,0x00,            /*offset:58*/
                                            0x00,0x00                       /*offset:62*/									
								};

//---	calculate ipv4 check sum column.(Total 4 Bytes)
static unsigned short  br_ipv4Checksum(unsigned char *pktBuf, unsigned int pktLen)
{
	/*note: the first bytes of  packetBuf should be two bytes aligned*/
	unsigned int  checksum=0;
	unsigned int  count=pktLen;
	unsigned short   *ptr= (unsigned short *)pktBuf;	
	
	while(count>1)
	{
		checksum+= ntohs(*ptr);
		ptr++;
		count -= 2;
	}
	
	if(count>0)
	{
		checksum+= *(pktBuf+pktLen-1)<<8; /*the last odd byte is treated as bit 15~8 of unsigned short*/
	}

	/* Roll over carry bits */
	checksum = (checksum >> 16) + (checksum & 0xffff);
	checksum += (checksum >> 16);

	/* Return checksum */
	return ((unsigned short) ~ checksum);

}

//---	Set IGMP Membership Query Packet.
//--- 	Step 1: Set Source MAC Addr, offset[6]
//---	Step 2: Set Source IP Addr, offset[26]
//---	Step 3: Set IPv4's header checksum, offset[24, 25]
//---	Step 4: Set IGMP's header checksum, v2 offset[36, 37], v3 offset[40, 41]
static unsigned char* br_generateIgmpQuery(struct net_bridge * br)
{
	struct net_device* brDev = NULL;
	unsigned short checkSum=0;
	struct in_device *in_dev;	
	struct net_device *landev;
	struct in_ifaddr *ifap = NULL;
	
      
	if(br==NULL)
	{
		return NULL;
	}
	
	brDev = br->dev;
	if(igmpVersion==3)
	{
		memcpy(&igmpV3QueryBuf[6],brDev->dev_addr,6);			/*set source mac address*/
	}
	else
	{
		memcpy(&igmpV2QueryBuf[6],brDev->dev_addr,6);			/*set source mac address*/
	}
	
	/*set source ip address*/
	if (landev = brDev){
		in_dev=(struct in_device*)(landev->ip_ptr);
		if (in_dev != NULL) {
			//printk("[%s:%d]\n", __FUNCTION__, __LINE__);
			for (ifap=in_dev->ifa_list; ifap != NULL; ifap=ifap->ifa_next) {
				//printk("[%s:%d]\n", __FUNCTION__, __LINE__);
				if (strcmp(br->dev->name, ifap->ifa_label) == 0){
					//printk("src ip:0x%x, [%s:%d]\n", ifap->ifa_address, __FUNCTION__, __LINE__);
					if(igmpVersion==3)
					{
						memcpy(&igmpV3QueryBuf[26],&ifap->ifa_address,4);
					}
					else
					{
						memcpy(&igmpV2QueryBuf[26],&ifap->ifa_address,4);
					}
				}
			}
		}
		else
		{
			//printk("[%s:%d]\n", __FUNCTION__, __LINE__);
			return NULL;
		}
	}
	else
	{
		//printk("[%s:%d]\n", __FUNCTION__, __LINE__);
		return NULL;
	}
	
	//---	initial IPv4's Header Checksum
	if(igmpVersion==3)
	{
		igmpV3QueryBuf[24]=0;
		igmpV3QueryBuf[25]=0;
	}
	else
	{
		igmpV2QueryBuf[24]=0;
		igmpV2QueryBuf[25]=0;
	}
	
	//---	calculate ipv4's Header checksum
	if(igmpVersion==3)
	{
		checkSum=br_ipv4Checksum(&igmpV3QueryBuf[14],24);
	}
	else
	{
		checkSum=br_ipv4Checksum(&igmpV2QueryBuf[14],20);
	}
	
	//---	set ipv4's Header checksum
	if(igmpVersion==3)
	{
		igmpV3QueryBuf[24]=(checkSum&0xff00)>>8;
		igmpV3QueryBuf[25]=(checkSum&0x00ff);

	}
	else
	{
		igmpV2QueryBuf[24]=(checkSum&0xff00)>>8;
		igmpV2QueryBuf[25]=(checkSum&0x00ff);

	}
	
	//---	set ipv4's Header checksum
	if(igmpVersion==3)
	{
		igmpV3QueryBuf[40]=0;
		igmpV3QueryBuf[41]=0;
		checkSum=br_ipv4Checksum(&igmpV3QueryBuf[38],12);
		igmpV3QueryBuf[40]=(checkSum&0xff00)>>8;
		igmpV3QueryBuf[41]=(checkSum&0x00ff);
	}
	else
	{
		igmpV2QueryBuf[36]=0;
		igmpV2QueryBuf[37]=0;
		checkSum=br_ipv4Checksum(&igmpV2QueryBuf[34],8);
		igmpV2QueryBuf[36]=(checkSum&0xff00)>>8;
		igmpV2QueryBuf[37]=(checkSum&0x00ff);
	}

	if(igmpVersion==3)
	{
		return igmpV3QueryBuf;
	}
	else
	{
		return igmpV2QueryBuf;
	}
	
	return NULL;
}

//---	check current group status and send igmp query to specific ports
void br_igmpQueryTimerExpired(unsigned long arg)
{
	struct net_bridge *br = (struct net_bridge*) arg;
	unsigned char *igmpBuf=NULL;
	struct sk_buff *skb;
	struct sk_buff *skb2;
	struct net_bridge_port *p, *n;
	struct net_bridge_port *prev;
	unsigned int fwdCnt=0;
	//---	if query is disabled, we don't need to check further more.
	if(igmpQueryEnabled==0)
	{
		return;
	}

	//---	if it's in router mode but no group member, we don't need to check either.
	if(br->mCastMode==1 && getGroupNum(br, IP_VERSION4)==0)
	{
		//printk("return! [%s:%d]\n", __FUNCTION__, __LINE__);
		return;
	}
	
	//---	allocate a buffer for igmp. if allocation fails, stop procedure and return.
	skb=dev_alloc_skb(1024);
	if(skb==NULL)
	{
		return;
	}
	
	//---	initial skb's pre 64 bytes.
	//---	generate IGMP Query and use igmpBuf to point it.
	//---	if pointer points to NULL, means we didn't success generating IGMP Querym, stop and return.
	memset(skb->data,64,0);
	igmpBuf=br_generateIgmpQuery(br);
	if(igmpBuf==NULL)
	{
		return;
	}
 
	//---	copy generated igmp query to socket buffer.
	memcpy(skb->data,igmpBuf,64);

	//---	extend the used data area of socket buffer.
	skb->len = 0;
	if(igmpVersion==3)
	{
		skb_put(skb, 50);
	}
	else
	{
		skb_put(skb, 42);
	}

	//---	set socket buffer device to bridge.
	skb->dev=br->dev;
	
	prev = NULL;
	fwdCnt=0;
	list_for_each_entry_safe(p, n, &br->port_list, list) 
	{ 
		//--- if current network port is forwarding mode, and it's not a public ethernet or public wlan.
		if ((p->state == BR_STATE_FORWARDING) && (strncmp(p->dev->name, "peth",4)!=0) && (strncmp(p->dev->name, "pwlan",5)!=0)) 
		{
			if (prev != NULL) 
			{   
				//--- if this is the second port that matchs.
				//--- clone the socket buffer structure and transmit from previous port by ndo_start_xmit.
				if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) 
				{
					br->dev->stats.tx_dropped++;
					kfree_skb(skb);
					return;
				} 
				skb2->dev=prev->dev;
				prev->dev->netdev_ops->ndo_start_xmit(skb2,prev->dev);
				fwdCnt++;
			}
			
			//--- save current port as prev.                                                  
			prev = p;
		}
	}

	//--- transmit socket from the last matched port by ndo_start_xmit.
	if (prev != NULL) 
	{
		skb->dev=prev->dev;
		prev->dev->netdev_ops->ndo_start_xmit(skb,prev->dev);
 		fwdCnt++;
	}

	if(fwdCnt==0)
	{
		/*to avoid memory leak*/
		kfree_skb(skb);
	}
	return;
}
#endif

#ifdef CONFIG_BRIDGE_MLD_QUERY
int mldVersion=2;
int mldQueryEnabled=1;

/*IPv6 MLD general query*/
static unsigned char mldQueryBuf[90]={  0x33,0x33,0x00,0x00,0x00,0x01,  /*destination mac*/
                                        0x00,0x00,0x00,0x00,0x00,0x00,  /*offset: 6, source mac*/
                                        0x86,0xdd,                      /*offset:12, ether type*/
                                        0x60,0x00,0x00,0x00,            /*offset:14, version(1 byte)-traffic cliass(1 byte)- flow label(2 bytes)*/
                                        0x00,0x20,0x00,0x01,            /*offset:18, payload length(2 bytes)-next header(1 byte)-hop limit(value:1 1byte)*/
                                        0xfe,0x80,0x00,0x00,            /*offset:22, source address*/
                                        0x00,0x00,0x00,0x00,            /*offset:26, be zero*/
                                        0x00,0x00,0x00,                 /*offset:30, upper 3 bytes mac address |0x02*/
                                        0xff,0xfe,                      /*offset:33, fixed*/
                                        0x00,0x00,0x00,                 /*offset:35, lowert 3 bytes mac address*/
                                        0xff,0x02,0x00,0x00,            /*offset:38, destination address is fixed as FF02::1*/
                                        0x00,0x00,0x00,0x00,			/*offset:42*/
                                        0x00,0x00,0x00,0x00,            /*offset:46*/
                                        0x00,0x00,0x00,0x01,            /*offset:50*/
                                        0x3a,0x00,                      /*offset:54, icmp type(1 byte)-length(1 byte)*/
                                        0x05,0x02,0x00,0x00,            /*offset:56, router alert option*/
                                        0x01,0x00,                      /*offset:60, padN*/
                                        0x82,0x00,                      /*offset:62, type(query:0x82)-code(0)*/
                                        0x00,0x00,                      /*offset:64, checksum*/
                                        0x00,0x0a,                      /*offset:66, maximum reponse code*/
                                        0x00,0x00,                      /*offset:68, reserved*/
                                        0x00,0x00,0x00,0x00,            /*offset:70, multicast address,fixed as 0*/
                                        0x00,0x00,0x00,0x00,			
                                        0x00,0x00,0x00,0x00,			
                                        0x00,0x00,0x00,0x00,
                                        0x0a,0x3c,0x00,0x00
                                    };			

static unsigned char ipv6PseudoHdrBuf[40]=	{	
                                                0xfe,0x80,0x00,0x00,    /*source address*/
                                                0x00,0x00,0x00,0x00,
                                                0x00,0x00,0x00,0xff,
                                                0xfe,0x00,0x00,0x00,
                                                0xff,0x02,0x00,0x00,    /*destination address*/
                                                0x00,0x00,0x00,0x00,
                                                0x00,0x00,0x00,0x00,
                                                0x00,0x00,0x00,0x01,
                                                0x00,0x00,0x00,0x18,    /*upper layer packet length*/
                                                0x00,0x00,0x00,0x3a     /*zero padding(3 bytes)-next header(1 byte)*/
                                            };

//---	calculate ipv6 check sum column.(Total 4 Bytes)
static unsigned short br_ipv6Checksum(unsigned char *pktBuf, unsigned int pktLen, unsigned char  *ipv6PseudoHdrBuf)
{
	unsigned int  checksum=0;
	unsigned int count=pktLen;
	unsigned short   *ptr;

	/*compute ipv6 pseudo-header checksum*/
	ptr= (unsigned short  *) (ipv6PseudoHdrBuf);	
	for(count=0; count<20; count++) /*the pseudo header is 40 bytes long*/
	{
		checksum+= ntohs(*ptr);
		ptr++;
	}
	
	/*compute the checksum of mld buffer*/
	count=pktLen;
	ptr=(unsigned short  *) (pktBuf);	
	while(count>1)
	{
		checksum+= ntohs(*ptr);
		ptr++;
		count -= 2;
	}
	 
	if(count>0)
	{
		checksum+= *(pktBuf+pktLen-1)<<8; /*the last odd byte is treated as bit 15~8 of unsigned short*/
	}

	/* Roll over carry bits */
	checksum = (checksum >> 16) + (checksum & 0xffff);
	checksum += (checksum >> 16);

	/* Return checksum */
	return ((uint16) ~ checksum);	
}

//---	Set ICMPv6 MLD Membership Query Packet.
//--- 	Step 1: Set MLD Source MAC Addr, offset[6]
//---	Step 2: Set MLD Source IP Addr, offset[22~37]
//---	Step 3: Set IPv6 pseudo-header's source ip, offset[0, 15]
//---	Step 4: Set MLD checksum, offset[64, 65]
static unsigned char* br_generateMldQuery(struct net_bridge * br)
{
	struct net_device* brDev = NULL;
	unsigned short checkSum=0;
	if(br==NULL)
	{
		return NULL;
	}
	
	brDev = br->dev;
	
	memcpy(&mldQueryBuf[6],brDev->dev_addr,6);			/*set source mac address*/
	
	//---	mld source ip address example: 
	//---	if mac addr: c0:25:a2:02:95:cf
	//---	ip: fe80:: c'X'25:a2'ff:fe'02:95cf
	//---	copy c0:25:a2 to [30~32], add ffef to [33~34], copy "02:95:cf" to [35~37]
	//---	second Byte do "or operation" with 0x2
	memcpy(&mldQueryBuf[30],brDev->dev_addr,3);		/*set  mld query packet source ip address*/
	mldQueryBuf[30]=mldQueryBuf[30]|0x02;		
	memcpy(&mldQueryBuf[35],&brDev->dev_addr[3],3);		
	//---	copy mld's source ip to pseudo-header. MLD[22~37] ->pseudo-header[0~15]
	memcpy(ipv6PseudoHdrBuf,&mldQueryBuf[22],16);			/*set pseudo-header source ip*/
	if(mldVersion==2)
	{
		mldQueryBuf[19]=	0x24;
	}
	else
	{
		mldQueryBuf[19]=	0x20;
	}

	mldQueryBuf[64]=0;/*reset checksum*/
	mldQueryBuf[65]=0;
	if(mldVersion==2)
	{
		ipv6PseudoHdrBuf[35]=28;
		checkSum=br_ipv6Checksum(&mldQueryBuf[62],28,ipv6PseudoHdrBuf);
	}
	else
	{
		ipv6PseudoHdrBuf[35]=24;
		checkSum=br_ipv6Checksum(&mldQueryBuf[62],24,ipv6PseudoHdrBuf);
	}
		
	mldQueryBuf[64]=(checkSum&0xff00)>>8;
	mldQueryBuf[65]=(checkSum&0x00ff);
	return mldQueryBuf;
}

//---	check current group status and send icmpv6 MLD query to specific ports
void br_mldQueryTimerExpired(unsigned long arg)
{
	struct net_bridge *br = (struct net_bridge*) arg;
	struct sk_buff *skb;
	struct sk_buff *skb2;
	struct net_bridge_port *p, *n;
	struct net_bridge_port *prev;
	unsigned int fwdCnt=0;
	unsigned char *mldBuf=NULL;

	//---	if query is disabled, we don't need to check further more.
	if(mldQueryEnabled==0)
	{
		return;
	}

	//---	if it's in router mode but no group member, we don't need to check either.
	if(br->mCastMode==1 && getGroupNum(br, IP_VERSION6)==0)
		return;

	//---	allocate a buffer for igmp. if allocation fails, stop procedure and return.
	skb=dev_alloc_skb(1024);
	if(skb==NULL)
	{
		return;
	}
	
	//---	initial skb's pre 86 bytes.
	//---	generate ICMPv6 MLD Query and use mldBuf to point it.
	//---	if pointer points to NULL, means we didn't generate IGMP Query successfully, stop and return.
	memset(skb->data,86,0);
	mldBuf=br_generateMldQuery(br);
	if(mldBuf==NULL)
	{
		return;
	}
	
	//---	copy generated igmp query to socket buffer.
	//---	extend the used data area of socket buffer.
	if(mldVersion==2)
	{
		memcpy(skb->data,mldBuf,90);
		skb->len = 0;
		skb_put(skb, 90);
	}
	else
	{
		memcpy(skb->data,mldBuf,86);
		skb->len = 0;
		skb_put(skb, 86);
	}

	prev = NULL;
	fwdCnt=0;

	list_for_each_entry_safe(p, n, &br->port_list, list) 
	{ 
		//--- if current network port is forwarding mode, and it's not a public ethernet or public wlan.
		if ((p->state == BR_STATE_FORWARDING) && (strncmp(p->dev->name, "peth",4)!=0) && (strncmp(p->dev->name, "pwlan",5)!=0)) 
		{
			if (prev != NULL) 
			{                                                                                       
				//--- if this is the second port that matchs.
				//--- clone the socket buffer structure and transmit from previous port by ndo_start_xmit.                                         
				if ((skb2 = skb_clone(skb, GFP_ATOMIC)) == NULL) 
				{
					br->dev->stats.tx_dropped++;
					kfree_skb(skb);
					return;
				} 
				skb2->dev=prev->dev;
				prev->dev->netdev_ops->ndo_start_xmit(skb2,prev->dev);
				fwdCnt++;
			}
			//--- save current port as prev. 
			prev = p;
		}
	}
	//--- transmit socket from the last matched port by ndo_start_xmit.
	if (prev != NULL) 
	{
		skb->dev=prev->dev;
		prev->dev->netdev_ops->ndo_start_xmit(skb,prev->dev);
		fwdCnt++;
	}

	if(fwdCnt==0)
	{
		/*to avoid memory leak*/
		kfree_skb(skb);
	}
	
	return;
}
#endif

#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
//---	check if client didn't respounse the multicast query for over a period of time.
//---	ever timer expired, check igmp and mld query in rotation.
void br_mCastQueryTimerExpired(unsigned long arg)
{
	struct net_bridge *br = (struct net_bridge*) arg;
	//---	modify mCastQueryTimer's timeout per MCAST_QUERY_INTERVAL second.
	mod_timer(&br->mCastQuerytimer, jiffies+MCAST_QUERY_INTERVAL*HZ);

	//printk("[%s:%d]\n", __FUNCTION__, __LINE__);

	if(mCastQueryTimerCnt%2==0)
	{
#ifdef CONFIG_BRIDGE_IGMP_QUERY
		br_igmpQueryTimerExpired(arg);
#endif
	}
	else
	{
#ifdef CONFIG_BRIDGE_MLD_QUERY
		br_mldQueryTimerExpired(arg);
#endif
	}
	mCastQueryTimerCnt++;

	return;
}
#endif

/*
 * Handle changes in state of network devices enslaved to a bridge.
 *
 * Note: don't care about up/down if bridge itself is down, because
 *     port state is checked when bridge is brought up.
 */
static int br_device_event(struct notifier_block *unused, unsigned long event, void *ptr)
{
	struct net_device *dev = netdev_notifier_info_to_dev(ptr);
	struct net_bridge_port *p;
	struct net_bridge *br;
	bool changed_addr;
	int err;

	/* register of bridge completed, add sysfs entries */
	if ((dev->priv_flags & IFF_EBRIDGE) && event == NETDEV_REGISTER) {
		br_sysfs_addbr(dev);
		return NOTIFY_DONE;
	}

	/* not a port of a bridge */
	p = br_port_get_rtnl(dev);
	if (!p)
		return NOTIFY_DONE;

	br = p->br;

	switch (event) {
	case NETDEV_CHANGEMTU:
		dev_set_mtu(br->dev, br_min_mtu(br));
		break;

	case NETDEV_CHANGEADDR:
		spin_lock_bh(&br->lock);
		br_fdb_changeaddr(p, dev->dev_addr);
		changed_addr = br_stp_recalculate_bridge_id(br);
		spin_unlock_bh(&br->lock);

		if (changed_addr)
			call_netdevice_notifiers(NETDEV_CHANGEADDR, br->dev);

		break;

	case NETDEV_CHANGE:
		br_port_carrier_check(p);
		break;

	case NETDEV_FEAT_CHANGE:
		netdev_update_features(br->dev);
		break;

	case NETDEV_DOWN:
		spin_lock_bh(&br->lock);
		if (br->dev->flags & IFF_UP)
			br_stp_disable_port(p);
		spin_unlock_bh(&br->lock);
		break;

	case NETDEV_UP:
		if (netif_running(br->dev) && netif_oper_up(dev)) {
			spin_lock_bh(&br->lock);
			br_stp_enable_port(p);
			spin_unlock_bh(&br->lock);
		}
		break;

	case NETDEV_UNREGISTER:
		br_del_if(br, dev);
		break;

	case NETDEV_CHANGENAME:
		err = br_sysfs_renameif(p);
		if (err)
			return notifier_from_errno(err);
		break;

	case NETDEV_PRE_TYPE_CHANGE:
		/* Forbid underlaying device to change its type. */
		return NOTIFY_BAD;

	case NETDEV_RESEND_IGMP:
		/* Propagate to master device */
		call_netdevice_notifiers(event, br->dev);
		break;
	}

	/* Events that may cause spanning tree to refresh */
	if (event == NETDEV_CHANGEADDR || event == NETDEV_UP ||
	    event == NETDEV_CHANGE || event == NETDEV_DOWN)
		br_ifinfo_notify(RTM_NEWLINK, p);

	return NOTIFY_DONE;
}

static struct notifier_block br_device_notifier = {
	.notifier_call = br_device_event
};

/* called with RTNL */
static int br_switchdev_event(struct notifier_block *unused,
			      unsigned long event, void *ptr)
{
	struct net_device *dev = switchdev_notifier_info_to_dev(ptr);
	struct net_bridge_port *p;
	struct net_bridge *br;
	struct switchdev_notifier_fdb_info *fdb_info;
	int err = NOTIFY_DONE;

	p = br_port_get_rtnl(dev);
	if (!p)
		goto out;

	br = p->br;

	switch (event) {
	case SWITCHDEV_FDB_ADD:
		fdb_info = ptr;
		err = br_fdb_external_learn_add(br, p, fdb_info->addr,
						fdb_info->vid);
		if (err)
			err = notifier_from_errno(err);
		break;
	case SWITCHDEV_FDB_DEL:
		fdb_info = ptr;
		err = br_fdb_external_learn_del(br, p, fdb_info->addr,
						fdb_info->vid);
		if (err)
			err = notifier_from_errno(err);
		break;
	}

out:
	return err;
}

static struct notifier_block br_switchdev_notifier = {
	.notifier_call = br_switchdev_event,
};

static void __net_exit br_net_exit(struct net *net)
{
	struct net_device *dev;
	LIST_HEAD(list);

	rtnl_lock();
	for_each_netdev(net, dev)
		if (dev->priv_flags & IFF_EBRIDGE)
			br_dev_delete(dev, &list);

	unregister_netdevice_many(&list);
	rtnl_unlock();

}

static struct pernet_operations br_net_ops = {
	.exit	= br_net_exit,
};

static const struct stp_proto br_stp_proto = {
	.rcv	= br_stp_rcv,
};

static int __init br_init(void)
{
	int err;

	BUILD_BUG_ON(sizeof(struct br_input_skb_cb) > FIELD_SIZEOF(struct sk_buff, cb));

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	//---	Construct /proc/alpha folder.
	struct proc_dir_entry *alpha_dir=NULL;	
	alpha_dir = proc_mkdir("alpha", NULL);
	if(alpha_dir == NULL) {
		return -ENOMEM;
        }
#endif

	err = stp_proto_register(&br_stp_proto);
	if (err < 0) {
		pr_err("bridge: can't register sap for STP\n");
		return err;
	}

	err = br_fdb_init();
	if (err)
		goto err_out;

	err = register_pernet_subsys(&br_net_ops);
	if (err)
		goto err_out1;

	err = br_nf_core_init();
	if (err)
		goto err_out2;

	err = register_netdevice_notifier(&br_device_notifier);
	if (err)
		goto err_out3;

	err = register_switchdev_notifier(&br_switchdev_notifier);
	if (err)
		goto err_out4;

	err = br_netlink_init();
	if (err)
		goto err_out5;

	brioctl_set(br_ioctl_deviceless_stub);

#if IS_ENABLED(CONFIG_ATM_LANE)
	br_fdb_test_addr_hook = br_fdb_test_addr;
#endif

#ifdef CONFIG_MCAST_SNOOPING
	br_mcast_snoop_init();
#endif

#if IS_MODULE(CONFIG_BRIDGE_NETFILTER)
	pr_info("bridge: filtering via arp/ip/ip6tables is no longer available "
		"by default. Update your scripts to load br_netfilter if you "
		"need this.\n");
#endif

	return 0;

err_out5:
	unregister_switchdev_notifier(&br_switchdev_notifier);
err_out4:
	unregister_netdevice_notifier(&br_device_notifier);
err_out3:
	br_nf_core_fini();
err_out2:
	unregister_pernet_subsys(&br_net_ops);
err_out1:
	br_fdb_fini();
err_out:
	stp_proto_unregister(&br_stp_proto);
	return err;
}

static void __exit br_deinit(void)
{
#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	//---	Destruct /proc/alpha folder.
	remove_proc_entry("alpha", 0);
#endif

	stp_proto_unregister(&br_stp_proto);
	br_netlink_fini();
	unregister_switchdev_notifier(&br_switchdev_notifier);
	unregister_netdevice_notifier(&br_device_notifier);
	brioctl_set(NULL);
	unregister_pernet_subsys(&br_net_ops);

	rcu_barrier(); /* Wait for completion of call_rcu()'s */

	br_nf_core_fini();
#if IS_ENABLED(CONFIG_ATM_LANE)
	br_fdb_test_addr_hook = NULL;
#endif
	br_fdb_fini();
#ifdef CONFIG_MCAST_SNOOPING
	br_mcast_snoop_deinit();
#endif
}

module_init(br_init)
module_exit(br_deinit)
MODULE_LICENSE("GPL");
MODULE_VERSION(BR_VERSION);
MODULE_ALIAS_RTNL_LINK("bridge");
