/*
 *	Userspace interface
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/netpoll.h>
#include <linux/ethtool.h>
#include <linux/if_arp.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/rtnetlink.h>
#include <linux/if_ether.h>
#include <linux/slab.h>
#include <net/sock.h>
#include <linux/if_vlan.h>
#include <net/switchdev.h>

#include "br_private.h"

#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#endif

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
//---	SVN REV 24143 & 24152 & 24154 & 24158 & 24159: Add callbacks for switch api.
//---	Not sure if this is really needed for this use case.
typedef void (*add_member_cb)(unsigned char *, unsigned char *);
typedef void (*del_member_cb)(unsigned char *,unsigned char *);
typedef void (*clear_group_cb)(unsigned char *);
typedef void (*snoop_init_cb)(void);
typedef void (*snoop_deinit_cb)(void);
add_member_cb add_member=NULL;
del_member_cb del_member=NULL;
clear_group_cb clear_group=NULL;
snoop_init_cb snoop_init=NULL;
snoop_deinit_cb snoop_deinit=NULL;

void register_igmp_callbacks(add_member_cb fun1, del_member_cb fun2, clear_group_cb fun3)
{
	add_member = fun1;
	del_member = fun2;
	clear_group = fun3;	
}

void unregister_igmp_callbacks(void)
{	
	add_member = NULL;
	del_member = NULL;
	clear_group = NULL;		
}

void register_snoop_init_callback (snoop_init_cb funa,snoop_deinit_cb funb)
{
	snoop_init = funa;
	snoop_deinit = funb;
}	

void unregister_snoop_init_callback(void)
{
	snoop_init = NULL;
	snoop_deinit =NULL;
}

EXPORT_SYMBOL(register_snoop_init_callback);
EXPORT_SYMBOL(unregister_snoop_init_callback);
EXPORT_SYMBOL(register_igmp_callbacks);
EXPORT_SYMBOL(unregister_igmp_callbacks);

#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
extern void br_mCastQueryTimerExpired(unsigned long arg);
#endif

#endif

/*
 * Determine initial path cost based on speed.
 * using recommendations from 802.1d standard
 *
 * Since driver might sleep need to not be holding any locks.
 */
static int port_cost(struct net_device *dev)
{
	struct ethtool_link_ksettings ecmd;

	if (!__ethtool_get_link_ksettings(dev, &ecmd)) {
		switch (ecmd.base.speed) {
		case SPEED_10000:
			return 2;
		case SPEED_1000:
			return 4;
		case SPEED_100:
			return 19;
		case SPEED_10:
			return 100;
		}
	}

	/* Old silly heuristics based on name */
	if (!strncmp(dev->name, "lec", 3))
		return 7;

	if (!strncmp(dev->name, "plip", 4))
		return 2500;

	return 100;	/* assume old 10Mbps */
}


/* Check for port carrier transitions. */
void br_port_carrier_check(struct net_bridge_port *p)
{
	struct net_device *dev = p->dev;
	struct net_bridge *br = p->br;

	if (!(p->flags & BR_ADMIN_COST) &&
	    netif_running(dev) && netif_oper_up(dev))
		p->path_cost = port_cost(dev);

	if (!netif_running(br->dev))
		return;

	spin_lock_bh(&br->lock);
	if (netif_running(dev) && netif_oper_up(dev)) {
		if (p->state == BR_STATE_DISABLED)
			br_stp_enable_port(p);
	} else {
		if (p->state != BR_STATE_DISABLED)
			br_stp_disable_port(p);
	}
	spin_unlock_bh(&br->lock);
}

static void br_port_set_promisc(struct net_bridge_port *p)
{
	int err = 0;

	if (br_promisc_port(p))
		return;

	err = dev_set_promiscuity(p->dev, 1);
	if (err)
		return;

	br_fdb_unsync_static(p->br, p);
	p->flags |= BR_PROMISC;
}

static void br_port_clear_promisc(struct net_bridge_port *p)
{
	int err;

	/* Check if the port is already non-promisc or if it doesn't
	 * support UNICAST filtering.  Without unicast filtering support
	 * we'll end up re-enabling promisc mode anyway, so just check for
	 * it here.
	 */
	if (!br_promisc_port(p) || !(p->dev->priv_flags & IFF_UNICAST_FLT))
		return;

	/* Since we'll be clearing the promisc mode, program the port
	 * first so that we don't have interruption in traffic.
	 */
	err = br_fdb_sync_static(p->br, p);
	if (err)
		return;

	dev_set_promiscuity(p->dev, -1);
	p->flags &= ~BR_PROMISC;
}

/* When a port is added or removed or when certain port flags
 * change, this function is called to automatically manage
 * promiscuity setting of all the bridge ports.  We are always called
 * under RTNL so can skip using rcu primitives.
 */
void br_manage_promisc(struct net_bridge *br)
{
	struct net_bridge_port *p;
	bool set_all = false;

	/* If vlan filtering is disabled or bridge interface is placed
	 * into promiscuous mode, place all ports in promiscuous mode.
	 */
	if ((br->dev->flags & IFF_PROMISC) || !br_vlan_enabled(br))
		set_all = true;

	list_for_each_entry(p, &br->port_list, list) {
		if (set_all) {
			br_port_set_promisc(p);
		} else {
			/* If the number of auto-ports is <= 1, then all other
			 * ports will have their output configuration
			 * statically specified through fdbs.  Since ingress
			 * on the auto-port becomes forwarding/egress to other
			 * ports and egress configuration is statically known,
			 * we can say that ingress configuration of the
			 * auto-port is also statically known.
			 * This lets us disable promiscuous mode and write
			 * this config to hw.
			 */
			if (br->auto_cnt == 0 ||
			    (br->auto_cnt == 1 && br_auto_port(p)))
				br_port_clear_promisc(p);
			else
				br_port_set_promisc(p);
		}
	}
}

static void nbp_update_port_count(struct net_bridge *br)
{
	struct net_bridge_port *p;
	u32 cnt = 0;

	list_for_each_entry(p, &br->port_list, list) {
		if (br_auto_port(p))
			cnt++;
	}
	if (br->auto_cnt != cnt) {
		br->auto_cnt = cnt;
		br_manage_promisc(br);
	}
}

static void nbp_delete_promisc(struct net_bridge_port *p)
{
	/* If port is currently promiscuous, unset promiscuity.
	 * Otherwise, it is a static port so remove all addresses
	 * from it.
	 */
	dev_set_allmulti(p->dev, -1);
	if (br_promisc_port(p))
		dev_set_promiscuity(p->dev, -1);
	else
		br_fdb_unsync_static(p->br, p);
}

static void release_nbp(struct kobject *kobj)
{
	struct net_bridge_port *p
		= container_of(kobj, struct net_bridge_port, kobj);
	kfree(p);
}

static struct kobj_type brport_ktype = {
#ifdef CONFIG_SYSFS
	.sysfs_ops = &brport_sysfs_ops,
#endif
	.release = release_nbp,
};

static void destroy_nbp(struct net_bridge_port *p)
{
	struct net_device *dev = p->dev;

	p->br = NULL;
	p->dev = NULL;
	dev_put(dev);

	kobject_put(&p->kobj);
}

static void destroy_nbp_rcu(struct rcu_head *head)
{
	struct net_bridge_port *p =
			container_of(head, struct net_bridge_port, rcu);
	destroy_nbp(p);
}

static unsigned get_max_headroom(struct net_bridge *br)
{
	unsigned max_headroom = 0;
	struct net_bridge_port *p;

	list_for_each_entry(p, &br->port_list, list) {
		unsigned dev_headroom = netdev_get_fwd_headroom(p->dev);

		if (dev_headroom > max_headroom)
			max_headroom = dev_headroom;
	}

	return max_headroom;
}

static void update_headroom(struct net_bridge *br, int new_hr)
{
	struct net_bridge_port *p;

	list_for_each_entry(p, &br->port_list, list)
		netdev_set_rx_headroom(p->dev, new_hr);

	br->dev->needed_headroom = new_hr;
}

/* Delete port(interface) from bridge is done in two steps.
 * via RCU. First step, marks device as down. That deletes
 * all the timers and stops new packets from flowing through.
 *
 * Final cleanup doesn't occur until after all CPU's finished
 * processing packets.
 *
 * Protected from multiple admin operations by RTNL mutex
 */
static void del_nbp(struct net_bridge_port *p)
{
	struct net_bridge *br = p->br;
	struct net_device *dev = p->dev;

#ifdef CONFIG_MCAST_SNOOPING
	br_mcast_port_cleanup(p);
#endif

	sysfs_remove_link(br->ifobj, p->dev->name);

	nbp_delete_promisc(p);

	spin_lock_bh(&br->lock);
	br_stp_disable_port(p);
	spin_unlock_bh(&br->lock);

	br_ifinfo_notify(RTM_DELLINK, p);

	list_del_rcu(&p->list);
	if (netdev_get_fwd_headroom(dev) == br->dev->needed_headroom)
		update_headroom(br, get_max_headroom(br));
	netdev_reset_rx_headroom(dev);

	nbp_vlan_flush(p);
	br_fdb_delete_by_port(br, p, 0, 1);
	switchdev_deferred_process();

	nbp_update_port_count(br);

	netdev_upper_dev_unlink(dev, br->dev);

	dev->priv_flags &= ~IFF_BRIDGE_PORT;

	netdev_rx_handler_unregister(dev);

	br_multicast_del_port(p);

	kobject_uevent(&p->kobj, KOBJ_REMOVE);
	kobject_del(&p->kobj);

	br_netpoll_disable(p);

	call_rcu(&p->rcu, destroy_nbp_rcu);
}

/* Delete bridge device */
void br_dev_delete(struct net_device *dev, struct list_head *head)
{
	struct net_bridge *br = netdev_priv(dev);
	struct net_bridge_port *p, *n;

	list_for_each_entry_safe(p, n, &br->port_list, list) {
		del_nbp(p);
	}

	br_fdb_delete_by_port(br, NULL, 0, 1);

	br_vlan_flush(br);
	br_multicast_dev_del(br);
	del_timer_sync(&br->gc_timer);

	br_sysfs_delbr(br->dev);
	unregister_netdevice_queue(br->dev, head);
}

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
//---	check if input MAC(char *s) is a legel MAC.
static int check_mac(char *s)
{
	char * accept = MAC_ACCEPT_CHAR;		//---	legel char.
	if(!s || !(*s))	return (-1);			//---	if s is null or doesn't point to any char list
	if ( strlen(s) == strspn(s, accept) )	//---	check if s's chars only appear in accept.
		return 0;
	return (-1);
}

//---	search device'name that matched
//---	called under bridge lock
static struct net_bridge_port * search_device(struct net_bridge * br, char* name)
{
	struct net_bridge_port *p;
	list_for_each_entry(p, &br->port_list, list) {
		if (strcmp(p->dev->name, name) == 0 ){
			return p;
		}
	}
	return NULL;
}

//---	convert hex number to decimal
static uint8_t * hex2dec(char *ch)
{
	if ( *ch >= '0' && *ch <= '9') *ch=*ch-'0';
	else if ( *ch >= 'A' && *ch <= 'F')  *ch=*ch-'A' +10;
	else if ( *ch >= 'a' && *ch <= 'f')  *ch=*ch-'a' +10;
	return ch;
}

//---	?
static void split_MAC(unsigned char * mac_addr, char * token)
{
	char *macDelim = MAC_DELIM;
	char **pMAC = &token;
	char * macField_char[6];
	int i;
	for (i=0; i<6; ++i)
	{
		int j;
		char temp[2];
		macField_char[i] = strsep(pMAC, macDelim);
		/* copy each char byte and convert to dec number */
		for(j=0; j<2; ++j) {
			memcpy(&temp[j],macField_char[i]+j, sizeof(char));
			hex2dec(&temp[j]);
		}

		temp[0] = temp[0] << 4;
		*(mac_addr + i)= (temp[0]^temp[1]);
	}
}

#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
//---	called under bridge lock
static int table_setmode(struct net_bridge *br, int type)
{
	switch(type)
	{
		case RT_MODE :
			br->mCastMode = 1;
			break;
		case BR_MODE :
			br->mCastMode = 2;
			break;
	}
	return 0;
}
#endif

#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
//---	called under bridge lock
static int table_setmf(struct net_bridge *br, char * name, int type)
{
	struct net_bridge_port *hit_port;
	hit_port = search_device(br, name);
	if (hit_port != NULL){
		if(type==ENABLE)
			atomic_set(&hit_port->multicast_forward, 1);
		else 
			atomic_set(&hit_port->multicast_forward, 0);
		return 0;
	}else
		return (-1);
}
#endif

//---	called under bridge lock
//---	???
static int table_setsnoop(struct net_bridge *br, int type)
{
	switch(type)
	{
		case ENABLE :
			br->snooping = 1;
			if(snoop_init) 
				snoop_init();
			else 
			{
				printk("No snooping implementation. Please check !! \n");
				return (-1);
			}
			break;
		case DISABLE : 
			br->snooping = 0;
			if(snoop_deinit) 
				snoop_deinit();
			else
			{
				printk("No snooping implementation. Please check !! \n");
				return (-1);
			}
			break;
	}
	return 0;
}

//---	set wireless identifier
//---	called under bridge lock
//---	Set wireless status. if this is a wireless or wire interface
static int table_setwl(struct net_bridge *br, char * name, int type)
{
	struct net_bridge_port *hit_port;
	hit_port = search_device(br, name);
	if (hit_port != NULL){
		if(type==ENABLE)
			atomic_set(&hit_port->wireless_interface, 1);
		else 
			atomic_set(&hit_port->wireless_interface, 0);
		return 0;
	}else
		return (-1);
}

#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
static void member_add_ordered(struct list_head *new, struct list_head *head, unsigned char * macaddr)
{
	struct port_member_mac *pmentry = NULL;

	list_for_each_entry(pmentry, head, list)
	{
		if(!memcmp(pmentry->member_mac,macaddr, 6))
			break;
	}

	__list_add(new, pmentry->list.prev, &pmentry->list);
}
#endif

//---	[add igmp table]
//---	check if new user's multicast group is in the igmp table.
//---	if group doesn't exsist, then create a new one and save it into the port group mac.
//---	and then check if the group has any old client still follow the multicast group.
//---	if no then create new one and attached it to this group-mac linked list
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
static void table_add(  struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr, int ip_version, uint32 *member_ip)
#else
static void table_add(  struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr)
#endif
{
	int found=0;
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	int found_member=0;
#endif
	//unsigned long flags;
	//spinlock_t lock;
	struct port_group_mac *pgentry;
	struct port_member_mac *pcentry;

#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	if(group_addr==NULL || member_addr==NULL || member_ip==NULL)
		return;	
#else
	if(group_addr==NULL || member_addr==NULL)
		return;
#endif

	//spin_lock_irqsave(&lock,flags);

	//1. find old group if exist
	list_for_each_entry(pgentry, &p->igmp_group_list, list)
	{
		if(!memcmp(pgentry->grpmac,group_addr, 6))
		{
			found = 1;
			break;
		}
	}

	if(!found)	//create new group
	{
		pgentry = (struct port_group_mac *)kmalloc(sizeof(struct port_group_mac), GFP_ATOMIC);
		INIT_LIST_HEAD(&pgentry->list);
		INIT_LIST_HEAD(&pgentry->member_list);
		list_add(&pgentry->list, &p->igmp_group_list);
		memcpy(pgentry->grpmac , group_addr , 6);
		print_debug("brg : Create new group 0x%02x%02x%02x%02x%02x%02x\n", 
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5]);
	}
	//2. find old client mac if exist
	found = 0;
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	found_member = 0;
#endif
	list_for_each_entry(pcentry, &pgentry->member_list, list)
	{
		if(!memcmp(pcentry->member_mac,member_addr, 6))
		{	/* member already exist, do nothing ~*/
#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
			//---	reset aging time to default.
			pcentry->aging_time = groupMemberAgingTime;
#endif
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			found_member = 1;
			if(!memcmp(pcentry->member_ipaddr,member_ip, 16))
			{
				found = 1;
				break;
			}
#else
			found = 1;
			break;
#endif
		}
	}
	if(!found)
	{	/* member NOT exist, create NEW ONE and attached it to this group-mac linked list ~*/
		pcentry	= (struct port_member_mac *)kmalloc(sizeof(struct port_member_mac), GFP_ATOMIC);
		INIT_LIST_HEAD(&pcentry->list);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		if(found_member)
		{
			member_add_ordered(&pcentry->list, &pgentry->member_list, member_addr);
		}
		else
		{
#endif
		list_add(&pcentry->list, &pgentry->member_list);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		}
#endif
		memcpy( pcentry->member_mac ,member_addr , 6);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		pcentry->ipVersion = ip_version;
		memcpy( pcentry->member_ipaddr ,member_ip , 16);
#endif
#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
		//---	initial aging time to default.
		pcentry->aging_time = groupMemberAgingTime;
#endif 
		print_debug("brg : Added client mac 0x%02x%02x%02x%02x%02x%02x to group 0x%02x%02x%02x%02x%02x%02x\n", 
				pcentry->member_mac[0],pcentry->member_mac[1],pcentry->member_mac[2],pcentry->member_mac[3],pcentry->member_mac[4],pcentry->member_mac[5],
				pgentry->grpmac[0],pgentry->grpmac[1],pgentry->grpmac[2],pgentry->grpmac[3],pgentry->grpmac[4],pgentry->grpmac[5]
				);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		print_debug("brg : Added client ipaddr %08x-%08x-%08x-%08x\n", 
				pcentry->member_ipaddr[0],pcentry->member_ipaddr[1],pcentry->member_ipaddr[2],pcentry->member_ipaddr[3]);
#endif
	}
	//spin_unlock_irqrestore (&lock, flags);
}

//---	[remove igmp table]
//---	remove client member's mac from the group list.
//---	if group's member list is also empty, delete it at the same time.
//---	1. find old group 
//---	2. find old client mac 
//---	3. if group is empty, delete group
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
static void table_remove(struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr, int ip_version, uint32 *member_ip)
#else
static void table_remove(struct net_bridge_port *p, unsigned char * group_addr, unsigned char * member_addr)
#endif
{
	struct port_group_mac *pgentry;
	struct port_member_mac *pcentry;
	int found = 0;

	//0. sanity check
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	if(group_addr==NULL || member_addr==NULL || member_ip==NULL)
		return;
#else
	if(group_addr==NULL || member_addr==NULL)
		return;
#endif

	//1. find old group 
	rcu_read_lock();
	list_for_each_entry_rcu(pgentry, &p->igmp_group_list, list)
	{
		if(!memcmp(pgentry->grpmac,group_addr, 6))
		{
			found = 1;
			break;
		}
	}
	rcu_read_unlock();
	
	if(!found)
	{
		print_debug("dbg : Can't delete 0x%02x%02x%02x%02x%02x%02x, group NOT FOUND.\n", 
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );
		return;
	}

	//2. find old client mac 
	found = 0;
	rcu_read_lock();
	list_for_each_entry_rcu(pcentry, &pgentry->member_list, list)
	{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		//--- we identify member with both mac and ip address.
		if(!memcmp(pcentry->member_mac,member_addr, 6) && !memcmp(pcentry->member_ipaddr,member_ip, 16))
#else
		if(!memcmp(pcentry->member_mac,member_addr, 6))
#endif
		{	
			found = 1;
			break;
		}
	}
	rcu_read_unlock();

	if(found)
	{
		/* member to be deleted FOUND, DELETE IT ! */
		list_del_rcu(&pcentry->list);
		kfree_rcu(pcentry, rcu);
		print_debug("dbg : Delete client 0x%02x%02x%02x%02x%02x%02x in group 0x%02x%02x%02x%02x%02x%02x\n", 
				member_addr[0],member_addr[1],member_addr[2],member_addr[3],member_addr[4],member_addr[5],
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
		print_debug("brg : Delete client ipaddr %08x-%08x-%08x-%08x\n", 
				member_ip[0],member_ip[1],member_ip[2],member_ip[3]);
#endif
	}else
	{	/* do nothing, just debug */
		print_debug("dbg : Can't delete client 0x%02x%02x%02x%02x%02x%02x, client NOT FOUND in group 0x%02x%02x%02x%02x%02x%02x\n", 
				member_addr[0],member_addr[1],member_addr[2],member_addr[3],member_addr[4],member_addr[5],
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );
	}

	//3. if group is empty, delete group
	if(list_empty(&pgentry->member_list))
	{
		list_del_rcu(&pgentry->member_list);
		list_del_rcu(&pgentry->list);
		kfree_rcu(pgentry, rcu);
		//remove group mac from port_list
		print_debug("dbg : Delete group 0x%02x%02x%02x%02x%02x%02x since its empty \n", 
				group_addr[0],group_addr[1],group_addr[2],group_addr[3],group_addr[4],group_addr[5] );	
		return;
	}
}

//---	function to parse user space read proc file command.
//---	ex: cat /proc/tmp
//---	proc_dir_entry->read_proc has been removed since Linux Kernel 3.8,
//---	so we need to refactor the function declaration to following format.
//---	ssize_t (*read) (struct file *, char __user *, size_t, loff_t *);
//---   before: 
//---		static int proc_read_alpha_multicast (char *buf, char **start, off_t offset,
//---				int len, int *eof, void *data)
static ssize_t proc_read_alpha_multicast (struct file *file, char *buf,
		size_t len, loff_t *offset)
{
	int count =0;
	char *data = PDE_DATA(file_inode(file));
	if(!(data)){
		printk(KERN_INFO "Null data");
		return count;
	}
	struct net_bridge *br = (struct net_bridge *) data;
	struct net_bridge_port *p;
	struct port_group_mac *pgentry;
	struct port_member_mac *pmentry;	

	spin_lock_bh(&br->lock); // bridge lock
	printk( "**********************************************************************\n");
	printk( "* bridge name    : %s\n",br->dev->name);
#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
	printk( "* mCastMode      : %d\n",br->mCastMode);
#endif
	printk( "* snooping       : %d\n",br->snooping);
	printk( "**********************************************************************\n");
	list_for_each_entry_rcu(p, &br->port_list, list) {
		printk( "* ==============================================================\n");
		printk( "* <%d> port name : %s\n", p->port_no, p->dev->name);
		printk( "* <%d> wireless_interface -> %d\n", p->port_no, atomic_read(&p->wireless_interface) );
#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
		printk( "* <%d> multicast_forward -> %d\n", p->port_no, atomic_read(&p->multicast_forward) );
#endif
		//traverse through all group list, list all the members inside 
		list_for_each_entry(pgentry, &p->igmp_group_list, list)
		{
			printk(" Group Mac  0x%02x%02x%02x%02x%02x%02x\n",pgentry->grpmac[0],
					pgentry->grpmac[1],
					pgentry->grpmac[2],
					pgentry->grpmac[3],
					pgentry->grpmac[4],
					pgentry->grpmac[5]);

			list_for_each_entry(pmentry, &pgentry->member_list, list)
			{
#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
				//--- print current member's mac and it's multicast listener repot timeout counter.
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
				printk("   membermac 0x%02x%02x%02x%02x%02x%02x\n   ipaddr %08x-%08x-%08x-%08x aging:%ds\n",pmentry->member_mac[0],
#else
				printk("   membermac 0x%02x%02x%02x%02x%02x%02x aging:%ds\n",pmentry->member_mac[0],
#endif
						pmentry->member_mac[1],
						pmentry->member_mac[2],
						pmentry->member_mac[3],
						pmentry->member_mac[4],
						pmentry->member_mac[5],
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
						pmentry->member_ipaddr[0],
						pmentry->member_ipaddr[1],
						pmentry->member_ipaddr[2],
						pmentry->member_ipaddr[3],
#endif
						pmentry->aging_time);
#else
				printk("   membermac 0x%02x%02x%02x%02x%02x%02x\n",pmentry->member_mac[0],
						pmentry->member_mac[1],
						pmentry->member_mac[2],
						pmentry->member_mac[3],
						pmentry->member_mac[4],
						pmentry->member_mac[5]);
#endif
			}
		}
		printk( "* ==============================================================\n");
	} // list_for_each_entry_rcu() - END
	printk( "**********************************************************************\n");
	spin_unlock_bh(&br->lock); // bridge unlock

	return count;
}

//---	function to parse user space write string into proc file command.
//---	Currently we didn't use the "add" and "remove" in this customization.
//---	ex: echo "TEST" > /proc/tmp
//---	proc_dir_entry->write_proc has been removed since Linux Kernel 3.8,
//---	so we need to refactor the function declaration to following format.
//---	ssize_t (*write) (struct file *, const char __user *, size_t, loff_t *);
//---   before: 
//---       static int proc_write_alpha_multicast (struct file *file, const char *buf,
//---   	    unsigned long count, void *data)
static int proc_write_alpha_multicast (struct file *file, const char *buf,
		size_t count, loff_t *offset)
{
	int len = MESSAGE_LENGTH+1;
	char message[len];
	char *msgDelim = MESSAGE_DELIM;
	char *pmesg;
	char *action_token, *action;
	struct net_bridge *br;

	if(count > MESSAGE_LENGTH) {len = MESSAGE_LENGTH;}
	else {len = count; }
	if(copy_from_user(message, buf, len))
		return -EFAULT;
	message[len-1] = '\0';

	/* split input message that get from user space
	 * token[0] => action token --> add or remove
	 * token[1] => multicast group mac address
	 * token[2] => member MAC address of host
	 */
	pmesg = message ;

	action_token = strsep(&pmesg, msgDelim);

	char *data = PDE_DATA(file_inode(file));
	if(!(data)){
		printk(KERN_INFO "Null data");
		return 0;
	}
	br = (struct net_bridge *) data;
	
#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
	/* ============================  set router mode =====================*/
	action = ACTION_RT_MODE;
	if (memcmp(action_token, action, sizeof(ACTION_RT_MODE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmode(br, RT_MODE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SET mode FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}
	
	/* ============================  set bridge mode =====================*/
	action = ACTION_BR_MODE;
	if (memcmp(action_token, action, sizeof(ACTION_BR_MODE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmode(br, BR_MODE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SET mode FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}
#endif

#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
	/* ============================  set multicastforward  =====================*/
	action = ACTION_SET_MCASTFWD;
	if (memcmp(action_token, action, sizeof(ACTION_SET_MCASTFWD) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmf(br,pmesg, ENABLE) != 0){
			print_debug("[BR_IGMPP_PROC]->WARNING SETMF FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  unset multicastforward  ===================*/
	action = ACTION_UNSET_MCASTFWD;
	if ( memcmp(action_token, action, sizeof(ACTION_UNSET_MCASTFWD) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setmf(br,pmesg, DISABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SETMF FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}
#endif

	/* ============================  set wireless enhance =====================*/
	action = ACTION_SET_ENHANCE;
	if (memcmp(action_token, action, sizeof(ACTION_SET_ENHANCE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setwl(br,pmesg, ENABLE) != 0){
			print_debug("[BR_IGMPP_PROC]->WARNING SETWL FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  unset wireless enhance  ===================*/
	action = ACTION_UNSET_ENHANCE;
	if ( memcmp(action_token, action, sizeof(ACTION_UNSET_ENHANCE) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setwl(br,pmesg, DISABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SETWL FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  set snooping ============================*/
	action = ACTION_SET_SNOOPING;
	if (memcmp(action_token, action, sizeof(ACTION_SET_SNOOPING) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setsnoop(br, ENABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING SET snooping FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  unset snooping ==========================*/
	action = ACTION_UNSET_SNOOPING;
	if ( memcmp(action_token, action, sizeof(ACTION_UNSET_SNOOPING) )== 0){
		spin_lock_bh(&br->lock); // bridge lock
		if (table_setsnoop(br, DISABLE) != 0){
			print_debug(KERN_INFO "[BR_IGMPP_PROC]->WARNING UNSET snooping FAILURE-> %s\n",pmesg);
		}
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		goto proc_write_br_igmpp_out;
	}

	/* ============================  add - START =====================================*/
#ifndef CONFIG_BRIDGE_SNOOPING_IP_LIST
	action = ACTION_ADD;
	if ( memcmp(action_token, action, sizeof(ACTION_ADD) )== 0){
		/********** add - START of processing input string **********/
		char *token[2]={0,0};
		int i;
		unsigned char mac_addr[6];
		unsigned char grp_mac_addr[6];	
		struct net_bridge_fdb_entry *hit_fdb_entry;

		for(i=0; i<2; ++i)
			token[i] = strsep(&pmesg, msgDelim);

		/* Only accept MAC, split host MAC address */
		if ( check_mac(token[0]) == -1 || check_mac(token[1]) == -1)
		{
			print_debug(KERN_INFO "[BR_IGMPP_PROC]-> Host MAC address: %s,%s is illegal !!\n",
					(token[0])?(token[0]):"null",
					(token[1])?(token[1]):"null");
			goto proc_write_br_igmpp_out;
		}

		split_MAC(grp_mac_addr, token[0]);
		split_MAC(mac_addr, token[1]);

		print_debug("brg : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);

		spin_lock_bh(&br->lock); // bridge lock
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
		//--- __br_fdb_get has third parameter in kernel 3.10
		hit_fdb_entry = __br_fdb_get(br, mac_addr, 0);

		if (hit_fdb_entry != NULL)
		{
			table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}

		rcu_read_unlock();
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		//do snoop if implemented in switch
		if(add_member) add_member(grp_mac_addr,mac_addr);

		goto proc_write_br_igmpp_out;
	}


	action = ACTION_REMOVE;
	if ( memcmp(action_token, action, sizeof(ACTION_REMOVE) ) == 0)
	{
		char *token[2]={0,0};
		int i;
		unsigned char mac_addr[6];
		struct net_bridge_fdb_entry *hit_fdb_entry;
		unsigned char grp_mac_addr[6];

		for(i=0; i<2; ++i)
			token[i] = strsep(&pmesg, msgDelim);

		/* Only accept MAC, split host MAC address */
		if (check_mac(token[0]) == -1 || check_mac(token[1]) == -1)
		{
			print_debug(KERN_INFO "[BR_IGMPP_PROC]-> Host MAC address: %s,%s is illegal !!\n",
					(token[0])?(token[0]):"null",
					(token[1])?(token[1]):"null");
			goto proc_write_br_igmpp_out;
		}

		split_MAC(grp_mac_addr, token[0]);
		split_MAC(mac_addr, token[1]);

		print_debug("brg : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);

		spin_lock_bh(&br->lock); // bridge lock			
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
		//--- __br_fdb_get has third parameter in kernel 3.10
		hit_fdb_entry = __br_fdb_get(br, mac_addr, 0);
		rcu_read_unlock();

		if (hit_fdb_entry != NULL)
		{
			table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}

		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out

		//do snoop if implemented in switch
		if(del_member) del_member(grp_mac_addr, mac_addr);

		goto proc_write_br_igmpp_out;
	}
#endif
	/* ============================= remove - END ======================================*/

proc_write_br_igmpp_out:
	return len;
}

#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
//---	read members from fdb and update to igmp_group_list and member_list
void write_alpha_multicast (unsigned char op, struct net_bridge *br, unsigned char *dest, struct sk_buff *skb)
{
	unsigned char *src;
	src=(unsigned char*)(skb_mac_header(skb)+ETH_ALEN);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	struct ethhdr *ethh = eth_hdr(skb);
	struct iphdr *iph;
	struct ipv6hdr *ipv6h;
#endif
	struct net_bridge_fdb_entry *hit_fdb_entry;
	unsigned char mac_addr[6];
	unsigned char grp_mac_addr[6];
	memcpy(grp_mac_addr, dest, 6);
	memcpy(mac_addr, src, 6);
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
	uint32 ip_addr[4]={0,0,0,0};
#endif
	
	if (op == 1) /* add */
	{
		print_debug("add : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);
		
		spin_lock_bh(&br->lock); // bridge lock
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
        //--- __br_fdb_get has third parameter in kernel 3.10
		hit_fdb_entry = __br_fdb_get(br, mac_addr, 0);

		if (hit_fdb_entry != NULL)
		{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			if(ethh->h_proto == htons(ETH_P_IP))
			{
				iph = ip_hdr(skb);
				ip_addr[0] = ntohl(iph->saddr);
				table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION4, ip_addr);
			}
			else if(ethh->h_proto == htons(ETH_P_IPV6))
			{
				ipv6h = ipv6_hdr(skb);
				ip_addr[0] = ntohl(ipv6h->saddr.s6_addr32[0]);
				ip_addr[1] = ntohl(ipv6h->saddr.s6_addr32[1]);
				ip_addr[2] = ntohl(ipv6h->saddr.s6_addr32[2]);
				ip_addr[3] = ntohl(ipv6h->saddr.s6_addr32[3]);
				table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION6, ip_addr);
			}
#else
			table_add(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
#endif
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}

		rcu_read_unlock();
		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out
		//do snoop if implemented in switch
		if(add_member) add_member(grp_mac_addr,mac_addr);

		goto write_br_igmpp_out;	
	}
	else if (op == 2) /* delete */
	{
		print_debug("del : group 0x%02x%02x%02x%02x%02x%02x member 0x%02x%02x%02x%02x%02x%02x\n", 
				grp_mac_addr[0],grp_mac_addr[1],grp_mac_addr[2],grp_mac_addr[3],grp_mac_addr[4],grp_mac_addr[5],
				mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);

		spin_lock_bh(&br->lock); // bridge lock			
		/* use rcu lock to protect __br_fdb_get. remove fdb_delete because we not use br_fdb_get */
		rcu_read_lock();

		/* searching bridge_fdb_entry */
        //--- __br_fdb_get has third parameter in kernel 3.10
		hit_fdb_entry = __br_fdb_get(br, mac_addr, 0);
		rcu_read_unlock();

		if (hit_fdb_entry != NULL)
		{
#ifdef CONFIG_BRIDGE_SNOOPING_IP_LIST
			if(ethh->h_proto == htons(ETH_P_IP))
			{
				iph = ip_hdr(skb);
				ip_addr[0] = ntohl(iph->saddr);
				table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION4, ip_addr);
			}
			else if(ethh->h_proto == htons(ETH_P_IPV6))
			{
				ipv6h = ipv6_hdr(skb);
				ip_addr[0] = ntohl(ipv6h->saddr.s6_addr32[0]);
				ip_addr[1] = ntohl(ipv6h->saddr.s6_addr32[1]);
				ip_addr[2] = ntohl(ipv6h->saddr.s6_addr32[2]);
				ip_addr[3] = ntohl(ipv6h->saddr.s6_addr32[3]);
				table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr, IP_VERSION6, ip_addr);
			}
#else
			table_remove(hit_fdb_entry->dst, grp_mac_addr, mac_addr);
#endif
		}
		else
		{
			print_debug(KERN_INFO "The return value of __br_fdb_get() is NULL -> MAC: %X:%X:%X:%X:%X:%X \n",
					mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5] );
		}

		spin_unlock_bh(&br->lock); // bridge unlock for goto proc_write_br_igmpp_out

		//do snoop if implemented in switch
		if(del_member) del_member(grp_mac_addr, mac_addr);

		goto write_br_igmpp_out;
	}

write_br_igmpp_out:
	return;
}

//---	check each ethernet port's igmp group member
//---	if member's timer timeout, delete it; otherwise minus 1.
//---	delete multicast group if it is empty
//---	run this procedure per second.(now: jiffies -> next time: jiffies+HZ)
void br_mCastAgingTimerExpired(unsigned long arg)
{
	struct net_bridge *br = (struct net_bridge*) arg;
	struct net_bridge_port *p;
	struct port_group_mac *pgentry;
	struct port_member_mac *pmentry;
	struct port_group_mac *next_pgentry;
	struct port_member_mac *next_pmentry;
		
	spin_lock_bh(&br->lock); // bridge lock
	list_for_each_entry_rcu(p, &br->port_list, list) {
		if (p->state == BR_STATE_DISABLED ||
		    p->state == BR_STATE_BLOCKING)
			continue;

		list_for_each_entry_safe(pgentry, next_pgentry, &p->igmp_group_list, list)
		{
			list_for_each_entry_safe(pmentry, next_pmentry, &pgentry->member_list, list)
			{
				/* member to be deleted TIMEOUT, DELETE IT ! */
				if(pmentry->aging_time <= 0)
				{
					/* member to be deleted FOUND, DELETE IT ! */
					print_debug("dbg : Delete client 0x%02x%02x%02x%02x%02x%02x in group 0x%02x%02x%02x%02x%02x%02x\n", 
						pmentry->member_mac[0],pmentry->member_mac[1],pmentry->member_mac[2],pmentry->member_mac[3],pmentry->member_mac[4],pmentry->member_mac[5],
						pgentry->grpmac[0],pgentry->grpmac[1],pgentry->grpmac[2],pgentry->grpmac[3],pgentry->grpmac[4],pgentry->grpmac[5] );	
					list_del(&pmentry->list);
					kfree(pmentry);
				}
				else
					pmentry->aging_time = pmentry->aging_time - 1;
			}

			// if group is empty, delete group
			if(list_empty(&pgentry->member_list))
			{
				//remove group mac from port_list
				print_debug("dbg : Delete group 0x%02x%02x%02x%02x%02x%02x since its empty \n", 
					pgentry->grpmac[0],pgentry->grpmac[1],pgentry->grpmac[2],pgentry->grpmac[3],pgentry->grpmac[4],pgentry->grpmac[5] );	
				list_del(&pgentry->member_list);
				list_del(&pgentry->list);
				kfree(pgentry);
			}
		}
	}

	spin_unlock_bh(&br->lock); // bridge unlock
	mod_timer(&br->mCastAgingtimer, jiffies+HZ);
	
	return;
}
#endif

#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
//---	check how many menbers are in the group.
int32 getGroupNum(struct net_bridge *br, uint32 ipVersion)
{
	int32 groupCnt=0;
	struct net_bridge_port *p;
	struct port_group_mac *pgentry;
	
	if(ipVersion ==IP_VERSION4)
	{
		groupCnt=0;
		
		spin_lock_bh(&br->lock); // bridge lock
		list_for_each_entry_rcu(p, &br->port_list, list) {
			if (p->state == BR_STATE_DISABLED ||
			    p->state == BR_STATE_BLOCKING)
				continue;

			list_for_each_entry(pgentry, &p->igmp_group_list, list)
			{
				if(MULTICAST_MAC(pgentry->grpmac))
				{
					groupCnt++; 
				}
			}
		}
		spin_unlock_bh(&br->lock); // bridge unlock
	}
	else if(ipVersion ==IP_VERSION6)
	{
		groupCnt=0;
		
		spin_lock_bh(&br->lock); // bridge lock
		list_for_each_entry_rcu(p, &br->port_list, list) {
			if (p->state == BR_STATE_DISABLED ||
			    p->state == BR_STATE_BLOCKING)
				continue;

			list_for_each_entry(pgentry, &p->igmp_group_list, list)
			{
				if(IPV6_MULTICAST_MAC(pgentry->grpmac))
				{
					groupCnt++; 
				}
			}
		}
		spin_unlock_bh(&br->lock); // bridge unlock
	}
	
	return groupCnt;
}
#endif

#endif

/* find an available port number */
static int find_portno(struct net_bridge *br)
{
	int index;
	struct net_bridge_port *p;
	unsigned long *inuse;

	inuse = kcalloc(BITS_TO_LONGS(BR_MAX_PORTS), sizeof(unsigned long),
			GFP_KERNEL);
	if (!inuse)
		return -ENOMEM;

	set_bit(0, inuse);	/* zero is reserved */
	list_for_each_entry(p, &br->port_list, list) {
		set_bit(p->port_no, inuse);
	}
	index = find_first_zero_bit(inuse, BR_MAX_PORTS);
	kfree(inuse);

	return (index >= BR_MAX_PORTS) ? -EXFULL : index;
}

/* called with RTNL but without bridge lock */
static struct net_bridge_port *new_nbp(struct net_bridge *br,
				       struct net_device *dev)
{
	struct net_bridge_port *p;
	int index, err;

	index = find_portno(br);
	if (index < 0)
		return ERR_PTR(index);

	p = kzalloc(sizeof(*p), GFP_KERNEL);
	if (p == NULL)
		return ERR_PTR(-ENOMEM);

	p->br = br;
	dev_hold(dev);
	p->dev = dev;
	p->path_cost = port_cost(dev);
	p->priority = 0x8000 >> BR_PORT_BITS;
	p->port_no = index;
	p->flags = BR_LEARNING | BR_FLOOD | BR_MCAST_FLOOD;
	br_init_port(p);

#ifdef CONFIG_MCAST_SNOOPING
	br_mcast_port_init(p);
	spin_lock_init(&p->mghash_lock);
#endif

	br_set_state(p, BR_STATE_DISABLED);
	br_stp_port_timer_init(p);
	err = br_multicast_add_port(p);
	if (err) {
		dev_put(dev);
		kfree(p);
		p = ERR_PTR(err);
	}

	return p;
}

int br_add_bridge(struct net *net, const char *name)
{
	struct net_device *dev;
	int res;
#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	char alpha_proc_name[32];
#endif

	dev = alloc_netdev(sizeof(struct net_bridge), name, NET_NAME_UNKNOWN,
			   br_dev_setup);

	if (!dev)
		return -ENOMEM;

	dev_net_set(dev, net);
	dev->rtnl_link_ops = &br_link_ops;

	res = register_netdev(dev);
	if (res)
		free_netdev(dev);

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	//---	create a new bridge device and add into kernel network device list
	//---	while creating a new bridge, add a proc file "multicast_%s" in "/proc/alpha/" to record the customized status
	//---	be aware that /proc/alpha/ was been created while bridge module is initialized.
	if (res)
        return res;
			
	struct net_bridge *bridge0=NULL;
	bridge0=netdev_priv(dev);
	if(bridge0!=NULL)
	{
		snprintf(alpha_proc_name, sizeof(alpha_proc_name), "alpha/multicast_%s", name);

        //--- We can't directly modify proc_dir_entry since it has been moved 
        //--- from include/linux/proc_fs.h to /fs/proc/internal.h in kernel 3.10.
        //--- Starting from kernel 3.8, we need to use API proc_create() instead, and pass
        //--- proc_dir_entry entry into proc_create_data by file_operations data structure.        
        static struct file_operations proc_fops=
        {
            .read = proc_read_alpha_multicast,
            .write = proc_write_alpha_multicast,
            .owner = THIS_MODULE,
        };
        
        bridge0->alpha_multicast_proc = proc_create_data (alpha_proc_name, 0644, NULL, &proc_fops, bridge0);
//	bridge0->alpha_multicast_proc = create_proc_entry (alpha_proc_name, 0644, NULL);
		if (bridge0->alpha_multicast_proc == NULL)
		{
			printk("create proc FAILED %s\n", name);
			return ERR_PTR(-ENOMEM);
		}
        //--- in kernel 3.10, proc_create will set data as NULL.
//		bridge0->alpha_multicast_proc->data = (void*)bridge0;
//		bridge0->alpha_multicast_proc->read_proc = proc_read_alpha_multicast;
//		bridge0->alpha_multicast_proc->write_proc = proc_write_alpha_multicast;
		bridge0->snooping = 0;

#if defined(CONFIG_BRIDGE_IGMP_QUERY) || defined(CONFIG_BRIDGE_MLD_QUERY)
        //---	also initial multicast query timer
		bridge0->mCastMode = 0;
		//---	initial IGMP/MLD multicast query time, expire in one second.
		init_timer(&bridge0->mCastQuerytimer);
		bridge0->mCastQuerytimer.data=(unsigned long)bridge0;
		bridge0->mCastQuerytimer.expires=jiffies+1*HZ;
		bridge0->mCastQuerytimer.function=(void*)br_mCastQueryTimerExpired;
		add_timer(&bridge0->mCastQuerytimer);
#endif

#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
        //---	also initial multicast member aging timer
		init_timer(&bridge0->mCastAgingtimer);
		bridge0->mCastAgingtimer.data=(unsigned long)bridge0;
		bridge0->mCastAgingtimer.expires=jiffies+HZ;
		bridge0->mCastAgingtimer.function=(void*)br_mCastAgingTimerExpired;
		add_timer(&bridge0->mCastAgingtimer);
#endif

	}
#endif

	return res;
}

int br_del_bridge(struct net *net, const char *name)
{
	struct net_device *dev;
	int ret = 0;

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	char alpha_proc_name[32];
#endif

	rtnl_lock();
	dev = __dev_get_by_name(net, name);
	if (dev == NULL)
		ret =  -ENXIO; 	/* Could not find device */

	else if (!(dev->priv_flags & IFF_EBRIDGE)) {
		/* Attempt to delete non bridge device! */
		ret = -EPERM;
	}

	else if (dev->flags & IFF_UP) {
		/* Not shutdown yet. */
		ret = -EBUSY;
	}

	else
		br_dev_delete(dev, NULL);

	rtnl_unlock();

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	//---	remove bridge device and delete the proc file at the same time.
	print_debug("remove proc entry %s\n", name);
	snprintf(alpha_proc_name, sizeof(alpha_proc_name), "alpha/multicast_%s", name);
	remove_proc_entry(alpha_proc_name, 0);
#endif

	return ret;
}

/* MTU of the bridge pseudo-device: ETH_DATA_LEN or the minimum of the ports */
int br_min_mtu(const struct net_bridge *br)
{
	const struct net_bridge_port *p;
	int mtu = 0;

	ASSERT_RTNL();

	if (list_empty(&br->port_list))
		mtu = ETH_DATA_LEN;
	else {
		list_for_each_entry(p, &br->port_list, list) {
			if (!mtu  || p->dev->mtu < mtu)
				mtu = p->dev->mtu;
		}
	}
	return mtu;
}

static void br_set_gso_limits(struct net_bridge *br)
{
	unsigned int gso_max_size = GSO_MAX_SIZE;
	u16 gso_max_segs = GSO_MAX_SEGS;
	const struct net_bridge_port *p;

	list_for_each_entry(p, &br->port_list, list) {
		gso_max_size = min(gso_max_size, p->dev->gso_max_size);
		gso_max_segs = min(gso_max_segs, p->dev->gso_max_segs);
	}
	br->dev->gso_max_size = gso_max_size;
	br->dev->gso_max_segs = gso_max_segs;
}

/*
 * Recomputes features using slave's features
 */
netdev_features_t br_features_recompute(struct net_bridge *br,
	netdev_features_t features)
{
	struct net_bridge_port *p;
	netdev_features_t mask;

	if (list_empty(&br->port_list))
		return features;

	mask = features;
	features &= ~NETIF_F_ONE_FOR_ALL;

	list_for_each_entry(p, &br->port_list, list) {
		features = netdev_increment_features(features,
						     p->dev->features, mask);
	}
	features = netdev_add_tso_features(features, mask);

	return features;
}

/* called with RTNL */
int br_add_if(struct net_bridge *br, struct net_device *dev)
{
	struct net_bridge_port *p;
	int err = 0;
	unsigned br_hr, dev_hr;
	bool changed_addr;

	/* Don't allow bridging non-ethernet like devices, or DSA-enabled
	 * master network devices since the bridge layer rx_handler prevents
	 * the DSA fake ethertype handler to be invoked, so we do not strip off
	 * the DSA switch tag protocol header and the bridge layer just return
	 * RX_HANDLER_CONSUMED, stopping RX processing for these frames.
	 */
	if ((dev->flags & IFF_LOOPBACK) ||
	    dev->type != ARPHRD_ETHER || dev->addr_len != ETH_ALEN ||
	    !is_valid_ether_addr(dev->dev_addr) ||
	    netdev_uses_dsa(dev))
		return -EINVAL;

	/* No bridging of bridges */
	if (dev->netdev_ops->ndo_start_xmit == br_dev_xmit)
		return -ELOOP;

	/* Device has master upper dev */
	if (netdev_master_upper_dev_get(dev))
		return -EBUSY;

	/* No bridging devices that dislike that (e.g. wireless) */
	if (dev->priv_flags & IFF_DONT_BRIDGE)
		return -EOPNOTSUPP;

	p = new_nbp(br, dev);
	if (IS_ERR(p))
		return PTR_ERR(p);

	call_netdevice_notifiers(NETDEV_JOIN, dev);

	err = dev_set_allmulti(dev, 1);
	if (err) {
		kfree(p);	/* kobject not yet init'd, manually free */
		goto err1;
	}

	err = kobject_init_and_add(&p->kobj, &brport_ktype, &(dev->dev.kobj),
				   SYSFS_BRIDGE_PORT_ATTR);
	if (err)
		goto err2;

	err = br_sysfs_addif(p);
	if (err)
		goto err2;

	err = br_netpoll_enable(p);
	if (err)
		goto err3;

	err = netdev_rx_handler_register(dev, br_handle_frame, p);
	if (err)
		goto err4;

	dev->priv_flags |= IFF_BRIDGE_PORT;

	err = netdev_master_upper_dev_link(dev, br->dev, NULL, NULL);
	if (err)
		goto err5;

	err = nbp_switchdev_mark_set(p);
	if (err)
		goto err6;

	dev_disable_lro(dev);

	list_add_rcu(&p->list, &br->port_list);

	nbp_update_port_count(br);

	netdev_update_features(br->dev);

	br_hr = br->dev->needed_headroom;
	dev_hr = netdev_get_fwd_headroom(dev);
	if (br_hr < dev_hr)
		update_headroom(br, dev_hr);
	else
		netdev_set_rx_headroom(dev, br_hr);

	if (br_fdb_insert(br, p, dev->dev_addr, 0))
		netdev_err(dev, "failed insert local address bridge forwarding table\n");

	err = nbp_vlan_init(p);
	if (err) {
		netdev_err(dev, "failed to initialize vlan filtering on this port\n");
		goto err7;
	}

	spin_lock_bh(&br->lock);

#ifdef CONFIG_BRIDGE_ALPHA_MULTICAST_SNOOP
	//---	initial igmp group list.
	//---	set wireless interface flag to zero.
	INIT_LIST_HEAD(&p->igmp_group_list);	
	atomic_set(&p->wireless_interface, 0);
#endif

#ifdef CONFIG_BRIDGE_MULTICAST_FORWARDING
	atomic_set(&p->multicast_forward, 1);
#endif

	changed_addr = br_stp_recalculate_bridge_id(br);

	if (netif_running(dev) && netif_oper_up(dev) &&
	    (br->dev->flags & IFF_UP))
		br_stp_enable_port(p);
	spin_unlock_bh(&br->lock);

	br_ifinfo_notify(RTM_NEWLINK, p);

	if (changed_addr)
		call_netdevice_notifiers(NETDEV_CHANGEADDR, br->dev);

	dev_set_mtu(br->dev, br_min_mtu(br));
	br_set_gso_limits(br);

	kobject_uevent(&p->kobj, KOBJ_ADD);

#ifdef CONFIG_MCAST_SNOOPING
	br_ifinfo_notify(RTM_NEWLINK, p);
#endif

	return 0;

err7:
	list_del_rcu(&p->list);
	br_fdb_delete_by_port(br, p, 0, 1);
	nbp_update_port_count(br);
err6:
	netdev_upper_dev_unlink(dev, br->dev);
err5:
	dev->priv_flags &= ~IFF_BRIDGE_PORT;
	netdev_rx_handler_unregister(dev);
err4:
	br_netpoll_disable(p);
err3:
	sysfs_remove_link(br->ifobj, p->dev->name);
err2:
	kobject_put(&p->kobj);
	dev_set_allmulti(dev, -1);
err1:
	dev_put(dev);
	return err;
}

/* called with RTNL */
int br_del_if(struct net_bridge *br, struct net_device *dev)
{
	struct net_bridge_port *p;
	bool changed_addr;

	p = br_port_get_rtnl(dev);
	if (!p || p->br != br)
		return -EINVAL;

	/* Since more than one interface can be attached to a bridge,
	 * there still maybe an alternate path for netconsole to use;
	 * therefore there is no reason for a NETDEV_RELEASE event.
	 */
	del_nbp(p);

	dev_set_mtu(br->dev, br_min_mtu(br));
	br_set_gso_limits(br);

	spin_lock_bh(&br->lock);
	changed_addr = br_stp_recalculate_bridge_id(br);
	spin_unlock_bh(&br->lock);

	if (changed_addr)
		call_netdevice_notifiers(NETDEV_CHANGEADDR, br->dev);

	netdev_update_features(br->dev);

#ifdef CONFIG_MCAST_SNOOPING
	br_ifinfo_notify(RTM_DELLINK, p);
#endif

	return 0;
}

void br_port_flags_change(struct net_bridge_port *p, unsigned long mask)
{
	struct net_bridge *br = p->br;

	if (mask & BR_AUTO_MASK)
		nbp_update_port_count(br);
}
