// SPDX-License-Identifier:	GPL-2.0+
/*
 * Copyright (C) 2019 MediaTek Incorporation. All Rights Reserved.
 *
 * Author: Sam Shih <sam.shih@mediatek.com>
 */
#include <common.h>

DECLARE_GLOBAL_DATA_PTR;

struct mtk_bootmenu_entry {
	const char *desc;
	const char *cmd;
};

struct mtk_flash_partition {
	const char *desc;
	const unsigned long start;
	const unsigned long end;
};

#ifdef CONFIG_ENABLE_NAND_NMBM
#define NAND_READ		"nmbm nmbm0 read "
#define NAND_WRITE		"nmbm nmbm0 write "
#define NAND_ERASE		"nmbm nmbm0 erase "
#else
#define NAND_READ		"mtd read spi-nand0 "
#define NAND_WRITE		"mtd write spi-nand0 "
#define NAND_ERASE		"mtd erase spi-nand0 "
#endif

struct mtk_bootmenu_entry pre_release_bootmenu_entries[] = {
	{
		.desc = "Pre-release U-boot Boot Menu",
		.cmd = "setenv mtkautoboot_type snand;"
		       "mtkautoboot"
	}, {
		.desc = "SecureBoot U-boot Boot Menu",
		.cmd = "setenv mtkautoboot_type sb_snand;"
		       "mtkautoboot"
	}
};

/*
 * Bootmenu and partition table with secure boot support for spi-nand 
 * Expect bl2 and bl31 image from ATF2.1 
 * Not compatible with MTK U-boot arm 2014
 */
struct mtk_bootmenu_entry sb_snand_bootmenu_entries[] = {
	{
		.desc = "Boot Linux Kernel via SNAND Flash",
		.cmd = "mw.l 0x10212000 0x22000000;"
		       "setenv check_fw_info 1;"
		       "bootm;"
	}, {
		.desc = "Boot System1 via SNAND Flash",
		.cmd = "mw.l 0x10212000 0x22000000;"
		       "setenv check_fw_info;"
		       "run boot_system1;"
	}, {
		.desc = "Load Linux Kernel then write to System1 SNAND Flash via TFTP",
		.cmd = "tftp $(kernel_img);"
		       NAND_ERASE "${kernel0_offset} ${kernel0_maxsize};"
		       NAND_WRITE "${loadaddr} ${kernel0_offset} ${kernel0_maxsize};"
		       "mw.l 0x10212000 0x22000000;"
		       "setenv check_fw_info;"
		       "run boot_system1;"
	}, {
		.desc = "Load FIP then write to SNAND Flash via TFTP (Default Filename: fip.bin)",
		.cmd = "tftp $(fip_img);"
		       NAND_ERASE "${fip_offset} ${fip_maxsize};"
		       NAND_WRITE "${loadaddr} ${fip_offset} ${fip_maxsize}"
	}, {
		.desc = "Load BL2 then write to SNAND Flash via TFTP (Default Filename: bl2.img)",
		.cmd = "tftp $(bl2_img);"
		       NAND_ERASE "${bl2_offset} ${bl2_maxsize};"
		       NAND_WRITE "${loadaddr} ${bl2_offset} ${bl2_maxsize}"
	}
};

const struct mtk_flash_partition sb_snand_flash_partitions[] = {
	{
		.desc  = "bl2",
		.start = 0x00000000,
		.end   = 0x00080000
	}, {
		.desc  = "fip",
		.start = 0x00080000,
		.end   = 0x00280000
	}, {
		.desc  = "nvram",
		.start = 0x00280000,
		.end   = 0x00300000
	}, {
		.desc  = "factory",
		.start = 0x00300000,
		.end   = 0x00400000
	}, {
		.desc  = "bootseq",
		.start = 0x00400000,
		.end   = 0x00440000
	}, {
		.desc  = "kernel0",
		.start = 0x01500000,
		.end   = 0x03500000
	}, {
		.desc  = "kernel1",
		.start = 0x04000000,
		.end   = 0x06000000
	}
};

/*
 * Bootmenu and partition table without secure boot support for spi-nand 
 * Compatible with MTK U-boot arm 2014
 */
struct mtk_bootmenu_entry snand_bootmenu_entries[] = {
	{
		.desc = "Boot Linux Kernel via SNAND Flash",
		.cmd = NAND_READ "${loadaddr} ${kernel_offset} ${kernel_maxsize};"
		       "bootm"
	}, {
		.desc = "Load Linux Kernel then write to SNAND Flash via TFTP",
		.cmd = "tftp $(kernel_img);"
		       NAND_ERASE "${kernel_offset} ${kernel_maxsize};"
		       NAND_WRITE "${loadaddr} ${kernel_offset} ${kernel_maxsize}"
	}, {
		.desc = "Load U-boot then write to SNAND Flash via TFTP",
		.cmd = "tftp $(uboot_img);"
		       NAND_ERASE "${uboot_offset} ${uboot_maxsize};"
		       NAND_WRITE "${loadaddr} ${uboot_offset} ${uboot_maxsize}"
	}, {
		.desc = "Load Preloader then write to SNAND Flash via TFTP",
		.cmd = "tftp $(pl_img);"
		       NAND_ERASE "${pl_offset} ${pl_maxsize};"
		       NAND_WRITE "${loadaddr} ${pl_offset} ${pl_maxsize}"
	}, {
		.desc = "Load ATF then write to SNAND Flash via TFTP",
		.cmd = "tftp $(atf_img);"
		       NAND_ERASE "${atf_offset} ${atf_maxsize};"
		       NAND_WRITE "${loadaddr} ${atf_offset} ${atf_maxsize}"
	}, {
		.desc = "Load U-boot then write to SNAND Flash via Serial (Kermit)",
		.cmd = "loadb;"
		       NAND_ERASE "${uboot_offset} ${uboot_maxsize};"
		       NAND_WRITE "${loadaddr} ${uboot_offset} ${uboot_maxsize}"
	}, {
		.desc = "Load Preloader then write to SNAND Flash via Serial (Kermit)",
		.cmd = "loadb;"
		       NAND_ERASE "${pl_offset} ${pl_maxsize};"
		       NAND_WRITE "${loadaddr} ${pl_offset} ${pl_maxsize}"
	}, {
		.desc = "Load ATF then write to SNAND Flash via Serial (Kermit)",
		.cmd = "loadb;"
		       NAND_ERASE "${atf_offset} ${atf_maxsize};"
		       NAND_WRITE "${loadaddr} ${atf_offset} ${atf_maxsize}"
	}
};

/* New U-boot image is too big, due to pre-release using 
 * u-boot arm 2014 partition layout, so we need to merge
 * nvram and u-boot partition to fit new u-boot
 */
#define PRE_RELEASE_PARTITION_WORKAROUND

const struct mtk_flash_partition snand_flash_partitions[] = {
	{
		.desc  = "pl",
		.start = 0x00000000,
		.end   = 0x00080000
	}, {
		.desc  = "atf",
		.start = 0x00080000,
		.end   = 0x000C0000
	}, {
		.desc  = "uboot",
		.start = 0x000C0000,
#ifdef PRE_RELEASE_PARTITION_WORKAROUND
		.end   = 0x001C0000
#else
		.end   = 0x00140000
#endif
	}, {
		.desc  = "nvram",
#ifdef PRE_RELEASE_PARTITION_WORKAROUND
		.start = 0x00140000,
#else
		.start = 0x001C0000,
#endif
		.end   = 0x001C0000
	}, {
		.desc  = "factory",
		.start = 0x001C0000,
		.end   = 0x002C0000
	}, {
		.desc  = "kernel",
		.start = 0x002C0000,
		.end   = 0x016C0000
	}
};

const struct mtk_flash_partition snor_flash_partitions[] = {
	{
		.desc  = "pl",
		.start = 0x00000000,
		.end   = 0x00040000
	}, {
		.desc  = "atf",
		.start = 0x00040000,
		.end   = 0x00060000
	}, {
		.desc  = "uboot",
		.start = 0x00060000,
		.end   = 0x000A0000
	}, {
		.desc  = "nvram",
		.start = 0x000A0000,
		.end   = 0x000C0000
	}, {
		.desc  = "factory",
		.start = 0x000C0000,
		.end   = 0x00140000
	}, {
		.desc  = "kernel",
		.start = 0x00140000,
		.end   = 0x01540000
	}
};


static void set_partition_table(const struct mtk_flash_partition *partition, int size, int block_size)
{
	int i;
	char key[256];
	char val[4096];
	unsigned long div_to_block;
	
	if (block_size != 0)
		div_to_block = block_size;
	else
		div_to_block = 1;

	for (i = 0; i < size; i++) {
		memset(key, 0x0, sizeof(key));
		memset(val, 0x0, sizeof(val));
		snprintf(key, sizeof(key), "%s_offset", partition->desc);
		snprintf(val, sizeof(val), "0x%lx", 
			 partition->start / div_to_block);
		env_set(key, val);
		memset(key, 0x0, sizeof(key));
		memset(val, 0x0, sizeof(val));
		snprintf(key, sizeof(key), "%s_maxsize", partition->desc);
		snprintf(val, sizeof(val), "0x%lx", 
			 (partition->end - partition->start) / div_to_block);
		env_set(key, val);
		partition++;
	}
}

static int do_mtkautoboot(cmd_tbl_t *cmdtp, int flag, int argc,
	char *const argv[])
{
	int i;
	char *menu_type;
	char *sep;
	char key[256];
	char val[4096];
	char cmdbuf[4096];

	struct mtk_bootmenu_entry *mtk_bootmenu_entry = NULL;
	int mtk_bootmenu_entry_size;

	menu_type = env_get("mtkautoboot_type");
	if (menu_type == NULL) {
		printf("You must specify menu type by provide mtkautoboot_type"
		       "to environment variable\n"
		       "Avaliable values: snand\n");
		return 0;
	}
	if (strcmp(menu_type,"sb_snand")==0) {
		mtk_bootmenu_entry = sb_snand_bootmenu_entries;
		mtk_bootmenu_entry_size = ARRAY_SIZE(sb_snand_bootmenu_entries);
		set_partition_table(sb_snand_flash_partitions, 
				    ARRAY_SIZE(sb_snand_flash_partitions), 0);
		// default image name
		env_set("kernel_img", "soleil-for-boot-code.bin");
		env_set("fip_img", "fip.bin");
		env_set("bl2_img", "bl2.img");
		env_set("check_fw_info", "1");
		env_set("boot_rd_kernel0", "nmbm nmbm0 read ${loadaddr} ${kernel0_offset} ${kernel0_size};bootm start;source ${loadaddr}:script");
		env_set("boot_rd_kernel1", "nmbm nmbm0 read ${loadaddr} ${kernel1_offset} ${kernel1_size};bootm start;source ${loadaddr}:script");
		env_set("boot_rd_rootfs0", "nmbm nmbm0 read ${loadaddr} ${rootfs0_offset} ${rootfs0_size}");
		env_set("boot_rd_rootfs1", "nmbm nmbm0 read ${loadaddr} ${rootfs1_offset} ${rootfs1_size}");
		env_set("boot_rd_bootseq", "nmbm nmbm0 read ${loadaddr} ${bootseq_offset} ${bootseq_maxsize}");
		env_set("boot_wd_bootseq", "nmbm nmbm0 write ${loadaddr} ${bootseq_offset} ${bootseq_maxsize}");
		env_set("boot_system1", "nmbm nmbm0 read ${loadaddr} ${kernel0_offset} ${kernel0_maxsize};setenv check_fw_info;bootm start;source ${loadaddr}:script;mw.l 0x10212000 0x22000000;setenv bootargs ${bootargs_system1};bootm ${loadaddr}#config@1;");
		env_set("boot_system2", "nmbm nmbm0 read ${loadaddr} ${kernel1_offset} ${kernel1_maxsize};setenv check_fw_info;bootm start;source ${loadaddr}:script;mw.l 0x10212000 0x22000000;setenv bootargs ${bootargs_system2};bootm ${loadaddr}#config@1;");
	} else if (strcmp(menu_type,"snand")==0) {
		mtk_bootmenu_entry = snand_bootmenu_entries;
		mtk_bootmenu_entry_size = ARRAY_SIZE(snand_bootmenu_entries);
		set_partition_table(snand_flash_partitions, 
				    ARRAY_SIZE(snand_flash_partitions), 0);
		// default image name
		env_set("kernel_img", "kernel.bin");
		env_set("uboot_img", "u-boot-mtk.bin");
		env_set("pl_img", "preloader_rfb1_7622_64_forspinand.bin");
		env_set("atf_img", "mt7622_ATF_32_64_release.img");
	} else if (strcmp(menu_type,"pre_release")==0) {
		mtk_bootmenu_entry = pre_release_bootmenu_entries;
		mtk_bootmenu_entry_size = ARRAY_SIZE(pre_release_bootmenu_entries);
	} else {
		printf("Unrecognized mtkautoboot_type '%s'\n", menu_type);
		printf("You must specify menu type by provide mtkautoboot_type"
		       "to environment variable\n"
		       "Avaliable values: snand\n");
		return 0;
	}

	/*
	 * Run boot commands one by one in the Ctrl+C was disabled environment.
	 * If execute one of commands fail or returned from run_command_list(),
	 * it will cause the system hang in Uboot.
	 *
	 * This feature is for secure boot, you can add below dts node
	 * in arch/arm/dts/mt7622-rfb.dts, then the system will directly
	 * run command - mtkautoboot to run boot commands.
	 *
	 *	config {
	 *		bootcmd = "mtkautoboot";
	 *		bootsecure = <1>;
	 *	};
	 */
	if (fdtdec_get_config_int(gd->fdt_blob, "bootsecure", 0) != 0) {
		/*
		 * Copy boot commands to buffer for later use.
		 */
		snprintf(cmdbuf, sizeof(cmdbuf), "%s", mtk_bootmenu_entry->cmd);

		/*
		 * To run boot commands one by one and terminate on error,
		 * We need to replace command seperate symbol ';' as '\n', then
		 * call run_command_list().
		 */
		for (sep = strchr(cmdbuf, ';'); sep != NULL; sep = strchr(sep + 1, ';'))
			*sep = '\n';

		return run_command_list(cmdbuf, strlen(cmdbuf), 0);
	}

	for (i = 0; i < mtk_bootmenu_entry_size; i++) {
		snprintf(key, sizeof(key), "bootmenu_%d", i);
		snprintf(val, sizeof(val), "%s=%s",
			 mtk_bootmenu_entry->desc, 
			 mtk_bootmenu_entry->cmd);
		env_set(key, val);
		mtk_bootmenu_entry++;
	}
	/*
	 * Remove possibly existed `next entry` to force bootmenu command to
	 * stop processing
	 */
	snprintf(key, sizeof(key), "bootmenu_%d", i);
	env_set(key, NULL);

	run_command("bootmenu", 0);	
	return 0;
}

U_BOOT_CMD(mtkautoboot, 1, 0, do_mtkautoboot,
	"Display MediaTek bootmenu", ""
);
