// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020 MediaTek Inc. All Rights Reserved.
 *
 * Author: Weijie Gao <weijie.gao@mediatek.com>
 */

#include <common.h>
#include <clk.h>
#include <cpu_func.h>
#include <dm.h>
#include <mapmem.h>
#include <misc.h>
#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/iopoll.h>
#include <linux/string.h>

#define ECC_ENCCON		0x00
#define   ENC_EN		BIT(0)
#define ECC_ENCCNFG		0x04
#define   ENC_MS_SHIFT		16
#define   ENC_BURST_EN		BIT(8)
#define ECC_ENCDIADDR		0x08
#define ECC_ENCIDLE		0x0C
#define ECC_ENCPAR00		0x10
#define ECC_ENCSTA		0x2c
#define   ENC_FSM_M		0x07
#define   ENC_FSM_DONE		4


#define ECC_DECCON		0x100
#define   DEC_EN		BIT(0)
#define ECC_DECCNFG		0x104
#define   DEC_EMPTY_EN		BIT(31)
#define   DEC_MS_SHIFT		16
#define   DEC_CNFG_EL		(0x2 << 12)
#define   DEC_BURST_EN		BIT(8)
#define ECC_DECDIADDR		0x108
#define ECC_DECIDLE		0x10C
#define ECC_DECENUM0		0x114
#define   ERRNUM0_M		0x1f
#define ECC_DECDONE		0x11c
#define ECC_DECEL0		0x120
#define   EL_ODD_S		16
#define   EL_M			0x1fff

/* ECC_ENCIDLE & ECC_DECIDLE */
#define ECC_IDLE		BIT(0)

#define ECC_TIMEOUT		500000
#define NFI_FDM_MAX_SIZE	8

struct mtk_ecc_caps {
	u32 err_mask;
	const u8 *ecc_strength;
	u8 num_ecc_strength;
	u8 ecc_mode_shift;
	u32 parity_bits;
	const u8 *spare_size;
	u8 num_spare_size;
	u32 msg_block_size;
	u32 fdm_ecc_size;
};

struct mtk_ecc {
	struct udevice *dev;
	const struct mtk_ecc_caps *caps;
	void __iomem *regs;
	struct clk clk;

	u8 *eccdata;
	u32 eccdata_size;

	u8 *oobbuf;
	u32 oobbuf_size;

	u8 ecc_size;
	u8 extra_bytes;
	u16 sector_size;

	u8 ecc_steps;
	u8 spare_size;
	u8 ecc_strength;
	u8 ecc_bits_idx;
};


/* ecc strength that each IP supports */
static const u8 ecc_strength_mt7622[] = {
	4, 6, 8, 10, 12
};

static const u8 spare_size_mt7622[] = {
	16, 26, 27, 28
};

int mtk_ecc_config(struct udevice *dev, u32 writesize, u32 oobsize)
{
	struct mtk_ecc *ecc = dev_get_priv(dev);
	u32 ecc_bytes, spare_per_sector;
	u8 *ptr;
	int i;

	if (!dev || !writesize || !oobsize)
		return -EINVAL;

	if (writesize % oobsize)
		return -EINVAL;

	ecc->ecc_steps = writesize / ecc->caps->msg_block_size;
	spare_per_sector = oobsize / ecc->ecc_steps;

	pr_debug("ecc steps = %u, spare_per_sector = %u\n", ecc->ecc_steps,
		spare_per_sector);

	for (i = ecc->caps->num_spare_size; i > 0; i--) {
		if (ecc->caps->spare_size[i - 1] <= spare_per_sector)
			break;
	}

	if (!i) {
		dev_err(dev, "invalid spare size %d\n", spare_per_sector);
		return -EINVAL;
	}

	ecc->spare_size = ecc->caps->spare_size[i - 1];
	ecc->extra_bytes = oobsize - ecc->ecc_steps * ecc->spare_size;

	pr_debug("spare_size = %u, extra_bytes = %u\n", ecc->spare_size, ecc->extra_bytes);

	ecc_bytes = ecc->spare_size - NFI_FDM_MAX_SIZE;
	ecc->ecc_strength = ecc_bytes * 8 / ecc->caps->parity_bits;

	for (i = ecc->caps->num_ecc_strength; i > 0; i--) {
		if (ecc->caps->ecc_strength[i - 1] <= ecc->ecc_strength)
			break;
	}

	if (!i) {
		dev_err(dev, "invalid ecc strength %d\n", ecc->ecc_strength);
		return -EINVAL;
	}

	ecc->ecc_bits_idx = i - 1;
	ecc->ecc_strength = ecc->caps->ecc_strength[ecc->ecc_bits_idx];
	ecc->ecc_size = (ecc->ecc_strength * ecc->caps->parity_bits +7) >> 3;

	pr_debug("ecc strength[%u] = %u, bytes = %u\n", ecc->ecc_bits_idx,
		ecc->ecc_strength, ecc->ecc_size);

	ecc->sector_size = ecc->caps->msg_block_size + ecc->spare_size;

	pr_debug("ecc sector size = %u\n", ecc->sector_size);

	if (!ecc->oobbuf) {
		ecc->oobbuf = malloc(oobsize);
		ecc->oobbuf_size = oobsize;
	} else if (ecc->oobbuf_size < oobsize) {
		ptr = realloc(ecc->oobbuf, oobsize);
		if (!ptr) {
			free(ecc->oobbuf);
			ecc->oobbuf = NULL;
		} else {
			ecc->oobbuf = ptr;
			ecc->oobbuf_size = oobsize;
		}
	}

	if (!ecc->oobbuf)
		return -ENOMEM;

	return 0;
}

static int mtk_ecc_correct_check(struct mtk_ecc *ecc, u8 *data)
{
	u32 decnum, num_error_bits;
	u32 error_locations, error_bit_loc;
	u32 error_byte_pos, error_bit_pos_in_byte;
	int bitflips = 0;
	u32 i;

	decnum = readl(ecc->regs + ECC_DECENUM0);
	num_error_bits = decnum & ERRNUM0_M;

	if (!num_error_bits)
		return 0;

	if (num_error_bits == ERRNUM0_M)
		return -1;

	for (i = 0; i < num_error_bits; i++) {
		error_locations = readl(ecc->regs + ECC_DECEL0 + (i / 2) * 4);
		error_bit_loc = (error_locations >> ((i % 2) * EL_ODD_S)) &
				EL_M;
		error_byte_pos = error_bit_loc / 8;
		error_bit_pos_in_byte = error_bit_loc % 8;

		data[error_byte_pos] ^= (1 << error_bit_pos_in_byte);

		bitflips++;
	}

	return bitflips;
}

static int mtk_ecc_wait_idle(struct mtk_ecc *ecc, u32 reg)
{
	u32 val;
	int ret;

	ret = readw_poll_timeout(ecc->regs + reg, val, (val & ECC_IDLE),
				 ECC_TIMEOUT);

	if (ret) {
		dev_warn(ecc->dev, "ECC engine is busy\n");
		return -EBUSY;
	}

	return 0;
}

static int mtk_ecc_decode_sector(struct mtk_ecc *ecc, void *data,
				 u32 parity_offset)
{
	u32 reg, val, sector_size, decode_size;
	int ret;

	ret = mtk_ecc_wait_idle(ecc, ECC_DECIDLE);
	if (ret)
		return ret;

	sector_size = ecc->caps->msg_block_size + ecc->caps->fdm_ecc_size;
	decode_size = ecc->ecc_strength * ecc->caps->parity_bits +
		      sector_size * 8;

	reg = DEC_EMPTY_EN | (decode_size << DEC_MS_SHIFT) | DEC_CNFG_EL |
	      DEC_BURST_EN | ecc->ecc_bits_idx;
	writel(reg, ecc->regs + ECC_DECCNFG);

	memcpy(ecc->eccdata, data, sector_size);
	memcpy(ecc->eccdata + sector_size, data + parity_offset, ecc->ecc_size);

	writel(lower_32_bits((size_t)ecc->eccdata), ecc->regs + ECC_DECDIADDR);
	writel(DEC_EN, ecc->regs + ECC_DECCON);

	ret = mtk_ecc_wait_idle(ecc, ECC_DECIDLE);
	if (ret)
		return ret;

	ret = readw_poll_timeout(ecc->regs + ECC_DECDONE, val, (val & 1),
				 ECC_TIMEOUT);
	if (ret) {
		dev_warn(ecc->dev, "ECC deoder is busy\n");
		return ret;
	}

	ret = mtk_ecc_correct_check(ecc, data);

	writel(0, ecc->regs + ECC_DECCON);

	if (ret < 0)
		return -EBADMSG;

	return 0;
}

int mtk_ecc_decode_page(struct udevice *dev, void *data, bool ecc_on)
{
	struct mtk_ecc *ecc = dev_get_priv(dev);
	int i, rc, eccbytes, ret = 0;
	void *oobptr, *eccptr;

	if (!dev || !data)
		return -EINVAL;

	if (!ecc_on)
		goto reorder_data;

	for (i = 0; i < ecc->ecc_steps; i++) {
		rc = mtk_ecc_decode_sector(ecc, data + i * ecc->sector_size,
			ecc->caps->msg_block_size + NFI_FDM_MAX_SIZE);
		if (rc < 0)
			ret = -EBADMSG;
		else if (ret >= 0)
			ret += rc;
	}

reorder_data:
	for (i = 0; i < ecc->ecc_steps; i++) {
		memcpy(ecc->oobbuf + i * ecc->spare_size,
		       data + i * ecc->sector_size + ecc->caps->msg_block_size,
		       ecc->spare_size);
	}

	for (i = 0; i < ecc->ecc_steps; i++) {
		memmove(data + i * ecc->caps->msg_block_size,
			data + i * ecc->sector_size, ecc->caps->msg_block_size);
	}

	oobptr = data + ecc->ecc_steps * ecc->caps->msg_block_size;
	eccptr = oobptr + ecc->ecc_steps * NFI_FDM_MAX_SIZE;
	eccbytes = ecc->spare_size - NFI_FDM_MAX_SIZE;

	for (i = 0; i < ecc->ecc_steps; i++) {
		memcpy(oobptr + i * NFI_FDM_MAX_SIZE,
		       ecc->oobbuf + i * ecc->spare_size, NFI_FDM_MAX_SIZE);
		memcpy(eccptr + i * eccbytes,
		       ecc->oobbuf + i * ecc->spare_size + NFI_FDM_MAX_SIZE,
		       eccbytes);
	}

	return ret;
}

static int mtk_ecc_encode_sector(struct mtk_ecc *ecc, void *data,
				 u32 parity_offset)
{
	u32 reg, val, sector_size, encode_size;
	u8 *p = data + parity_offset;
	int i, ret;

	ret = mtk_ecc_wait_idle(ecc, ECC_ENCIDLE);
	if (ret)
		return ret;

	sector_size = ecc->caps->msg_block_size + ecc->caps->fdm_ecc_size;
	encode_size = sector_size * 8;

	reg = (encode_size << ENC_MS_SHIFT) | ENC_BURST_EN | ecc->ecc_bits_idx;
	writel(reg, ecc->regs + ECC_ENCCNFG);

	memcpy(ecc->eccdata, data, sector_size);

	writel(lower_32_bits((size_t)ecc->eccdata), ecc->regs + ECC_ENCDIADDR);
	writel(ENC_EN, ecc->regs + ECC_ENCCON);

	ret = mtk_ecc_wait_idle(ecc, ECC_ENCIDLE);
	if (ret)
		return ret;

	ret = readw_poll_timeout(ecc->regs + ECC_ENCSTA, val,
				 ((val & ENC_FSM_M) != ENC_FSM_DONE),
				 ECC_TIMEOUT);
	if (ret) {
		dev_warn(ecc->dev, "ECC encoder is busy\n");
		return ret;
	}

	for (i = 0; i < ecc->ecc_size; i++) {
		if ((i % 4) == 0)
			val = readl(ecc->regs + ECC_ENCPAR00 + i);
		p[i] = (val >> ((i % 4) * 8)) & 0xff;
	}

	writel(0, ecc->regs + ECC_ENCCON);

	return 0;
}

int mtk_ecc_encode_page(struct udevice *dev, void *data, bool ecc_on)
{
	struct mtk_ecc *ecc = dev_get_priv(dev);
	void *oobptr, *eccptr;
	int i, ret = 0, rc, eccbytes;

	if (!dev || !data)
		return -EINVAL;

	memcpy(ecc->oobbuf, data + ecc->ecc_steps * ecc->caps->msg_block_size, 
	       ecc->ecc_steps * ecc->spare_size);

	for (i = ecc->ecc_steps; i > 0; i--) {
		memmove(data + (i - 1) * ecc->sector_size,
			data + (i - 1) * ecc->caps->msg_block_size,
			ecc->caps->msg_block_size);
	}

	eccptr = ecc->oobbuf + ecc->ecc_steps * NFI_FDM_MAX_SIZE;
	eccbytes = ecc->spare_size - NFI_FDM_MAX_SIZE;

	for (i = 0; i < ecc->ecc_steps; i++) {
		oobptr = data + i * ecc->sector_size +
			 ecc->caps->msg_block_size;
		memcpy(oobptr, ecc->oobbuf + i * NFI_FDM_MAX_SIZE,
		       NFI_FDM_MAX_SIZE);
		memcpy(oobptr + NFI_FDM_MAX_SIZE, eccptr + i * eccbytes,
		       eccbytes);
	}

	memset(data + ecc->ecc_steps * ecc->sector_size, 0xff,
	       ecc->extra_bytes);

	if (!ecc_on)
		return 0;

	for (i = 0; i < ecc->ecc_steps; i++) {
		rc = mtk_ecc_encode_sector(ecc, data + i * ecc->sector_size,
			ecc->caps->msg_block_size + NFI_FDM_MAX_SIZE);
		if (rc)
			ret = rc;
	}

	return ret;
}

int mtk_ecc_ooblayout_free(struct udevice *dev, int section, uint32_t *offset,
			   uint32_t *length)
{
	struct mtk_ecc *ecc = dev_get_priv(dev);

	if (section >= ecc->ecc_steps)
		return -ERANGE;

	if (offset)
		*offset = section * NFI_FDM_MAX_SIZE + 1;

	if (length)
		*length = NFI_FDM_MAX_SIZE - 1;

	return 0;
}

int mtk_ecc_ooblayout_ecc(struct udevice *dev, int section, uint32_t *offset,
			  uint32_t *length)
{
	struct mtk_ecc *ecc = dev_get_priv(dev);

	if (section)
		return -ERANGE;

	if (offset)
		*offset = ecc->ecc_steps * NFI_FDM_MAX_SIZE;

	if (length)
		*length = ecc->ecc_steps * ecc->spare_size + ecc->extra_bytes;

	return 0;
}

static int mtk_ecc_probe(struct udevice *dev)
{
	struct mtk_ecc *ecc = dev_get_priv(dev);
	u32 max_eccdata_size;
	fdt_addr_t base;
	int ret;

	ecc->caps = (struct mtk_ecc_caps *)dev_get_driver_data(dev);
	if (!ecc->caps)
		return -EINVAL;

	base = dev_read_addr(dev);
	if (base == FDT_ADDR_T_NONE)
		return -EINVAL;
	ecc->regs = map_sysmem(base, 0);

	ret = clk_get_by_index(dev, 0, &ecc->clk);
	if (ret < 0)
		return ret;

	max_eccdata_size = ecc->caps->num_ecc_strength - 1;
	max_eccdata_size = ecc->caps->ecc_strength[max_eccdata_size];
	max_eccdata_size = (max_eccdata_size * ecc->caps->parity_bits + 7) >> 3;
	max_eccdata_size = round_up(max_eccdata_size, 4);
	ecc->eccdata_size = ecc->caps->msg_block_size + max_eccdata_size +
			    NFI_FDM_MAX_SIZE;
	ecc->eccdata = (void *)noncached_alloc(ecc->eccdata_size,
					       ARCH_DMA_MINALIGN);
	if (!ecc->eccdata)
		return -ENOMEM;

	clk_enable(&ecc->clk);

	ecc->dev = dev;

	return 0;
}

static const struct mtk_ecc_caps mtk_ecc_caps_mt7622 = {
	.err_mask = 0x3f,
	.ecc_strength = ecc_strength_mt7622,
	.num_ecc_strength = ARRAY_SIZE(ecc_strength_mt7622),
	.ecc_mode_shift = 4,
	.parity_bits = 13,
	.spare_size = spare_size_mt7622,
	.num_spare_size = ARRAY_SIZE(spare_size_mt7622),
	.msg_block_size = 512,
	.fdm_ecc_size = 1,
};

static struct misc_ops mtk_ecc_ops = {
};

static const struct udevice_id mtk_ecc_ids[] = {
	{
		.compatible = "mediatek,mt7622-ecc",
		.data = (ulong)&mtk_ecc_caps_mt7622
	},
	{ }
};

U_BOOT_DRIVER(mtk_ecc) = {
	.name = "mtk_ecc",
	.id = UCLASS_MISC,
	.of_match = mtk_ecc_ids,
	.probe = mtk_ecc_probe,
	.ops = &mtk_ecc_ops,
	.priv_auto_alloc_size = sizeof(struct mtk_ecc),
};
