// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (C) 2019 MediaTek Inc. All Rights Reserved.
 *
 * Author: Weijie Gao <weijie.gao@mediatek.com>
 */

#include <common.h>
#include <clk.h>
#include <dm.h>
#include <errno.h>
#include <spi.h>
#include <spi-mem.h>
#include <stdbool.h>
#include <watchdog.h>
#include <dm/pinctrl.h>
#include <linux/bitops.h>
#include <linux/io.h>
#include <linux/iopoll.h>

#define SNFI_MAC_CTL			0x500
#define MAC_XIO_SEL			BIT(4)
#define SF_MAC_EN			BIT(3)
#define SF_TRIG				BIT(2)
#define WIP_READY			BIT(1)
#define WIP				BIT(0)

#define SNFI_MAC_OUTL			0x504
#define SNFI_MAC_INL			0x508

#define SNFI_MISC_CTL			0x538
#define SW_RST				BIT(28)
#define FIFO_RD_LTC_SHIFT		25
#define FIFO_RD_LTC			GENMASK(26, 25)
#define LATCH_LAT_SHIFT			8
#define LATCH_LAT			GENMASK(9, 8)
#define CS_DESELECT_CYC_SHIFT		0
#define CS_DESELECT_CYC			GENMASK(4, 0)

#define SNF_STA_CTL1			0x550
#define SPI_STATE			GENMASK(3, 0)

#define SNFI_GPRAM_OFFSET		0x800
#define SNFI_GPRAM_SIZE			0xa0

#define SNFI_POLL_INTERVAL		500000
#define SNFI_RST_POLL_INTERVAL		1000000

struct mtk_snfi_priv {
	void __iomem *base;

	struct clk nfi_clk;
	struct clk pad_clk;

	bool fixed_pinctrl;
	bool quad_enabled;
};

static int mtk_snfi_check_io_width(struct mtk_snfi_priv *priv, int curr,
				   int width)
{
	/* Only SIO and QIO are supported */

	switch (width) {
	case 0:
		return curr;
	case 1:
		return max_t(int, curr, 1);
	case 4:
		if (priv->quad_enabled)
			return 4;
	}

	return -ENOTSUPP;
}

static int mtk_snfi_get_io_width(struct mtk_snfi_priv *priv,
				 const struct spi_mem_op *op)
{
	int width = 1;

	/*
	 * Get the maximum bus width.
	 * The combinations of these 4 parts are acceptable if their
	 * buswidth are only 1 and/or 4.
	 */

	width = mtk_snfi_check_io_width(priv, width, op->cmd.buswidth);
	if (width < 0)
		return width;

	width = mtk_snfi_check_io_width(priv, width, op->addr.buswidth);
	if (width < 0)
		return width;

	width = mtk_snfi_check_io_width(priv, width, op->dummy.buswidth);
	if (width < 0)
		return width;

	width = mtk_snfi_check_io_width(priv, width, op->data.buswidth);

	return width;
}

static u32 mtk_snfi_data_to_gpram_size(u32 size, u32 master_iowidth,
				       u32 slave_iowidth)
{
	/*
	 * If SNFI transfer mode is 4-bit but data needs to be transferred
	 * in 1-bit mode, e.g. the cmd byte, each bytes of the data must be
	 * converted to represent 1-bit transfer in 4-bit mode.
	 *
	 * Each byte of the data will be converted into 4 bytes.
	 */

	if (master_iowidth == 4 && slave_iowidth == 1)
		return size * 4;

	return size;
}

static u32 mtk_snfi_gpram_to_data_size(u32 size, u32 master_iowidth,
				       u32 slave_iowidth)
{
	/*
	 * If SNFI transfer mode is 4-bit but data needs to be received
	 * in 1-bit mode, every 4 bytes will be converted into 1 byte.
	 *
	 * This is unlikely to be used.
	 */

	if (master_iowidth == 4 && slave_iowidth == 1)
		return size / 4;

	return size;
}

/*
 * In Quad I/O mode, data bits are transferred in the following sequence:
 * --------------------------------------
 * Byte :   0   |   1   |   2   |   3   |
 * --------------------------------------
 * DI/D0:<4>|<0>|<4>|<0>|<4>|<0>|<4>|<0>|
 * DO/D1:<5>|<1>|<5>|<1>|<5>|<1>|<5>|<1>|
 * D2   :<6>|<2>|<6>|<2>|<6>|<2>|<6>|<2>|
 * D3   :<7>|<3>|<7>|<3>|<7>|<3>|<7>|<3>|
 * --------------------------------------
 *
 * To represent Single I/O byte in Quad I/O mode, data byte bits can only
 * be set in bit0/4 in each byte in Quad I/O mode. Other bits are set to fixed
 * values:
 *   - DO/D1 is not cared
 *   - D2 is WP, should be pulled high
 *   - D3 is HOLD, should be pulled high
 * --------------------------------------
 * Byte :   0   |   1   |   2   |   3   |
 * --------------------------------------
 * DI/D0:<7>|<6>|<5>|<4>|<3>|<2>|<1>|<0>|
 * DO/D1: 0 | 0 | 0 | 0 | 0 | 0 | 0 | 0 |
 * D2   : 1 | 1 | 1 | 1 | 1 | 1 | 1 | 1 |
 * D3   : 1 | 1 | 1 | 1 | 1 | 1 | 1 | 1 |
 * --------------------------------------
 */

static u32 mtk_snfi_sio_byte_to_quadio(u8 byte, u8 *data)
{
	data[0] = (((byte >> 7) & 0x01) << 4) | ((byte >> 6) & 0x01) | 0xcc;
	data[1] = (((byte >> 5) & 0x01) << 4) | ((byte >> 4) & 0x01) | 0xcc;
	data[2] = (((byte >> 3) & 0x01) << 4) | ((byte >> 2) & 0x01) | 0xcc;
	data[3] = (((byte >> 1) & 0x01) << 4) | (byte & 0x01) | 0xcc;

	return 4;
}

static u8 mtk_snfi_quadio_byte_to_sio(const u8 *data)
{
	return (((data[0] >> 4) & 0x01) << 7) | ((data[0] & 0x01) << 6) |
	       (((data[1] >> 4) & 0x01) << 5) | ((data[1] & 0x01) << 4) |
	       (((data[2] >> 4) & 0x01) << 3) | ((data[2] & 0x01) << 2) |
	       (((data[3] >> 4) & 0x01) << 1) | (data[3] & 0x01);
}

static int mtk_snfi_adjust_op_size(struct spi_slave *slave,
				   struct spi_mem_op *op)
{
	struct udevice *bus = dev_get_parent(slave->dev);
	struct mtk_snfi_priv *priv = dev_get_priv(bus);
	u32 nbytes, datasize;
	int width;

	width = mtk_snfi_get_io_width(priv, op);
	if (width < 0)
		return -ENOTSUPP;

	/*
	 * When there is input data, it will be appended after the output
	 * data in the GPRAM. So the total size of either pure output data
	 * or the output+input data must not exceed the GPRAM size.
	 */

	nbytes = mtk_snfi_data_to_gpram_size(sizeof(op->cmd.opcode), width,
					     op->cmd.buswidth);

	nbytes += mtk_snfi_data_to_gpram_size(op->addr.nbytes, width,
					      op->addr.buswidth);

	nbytes += mtk_snfi_data_to_gpram_size(op->dummy.nbytes, width,
					      op->dummy.buswidth);

	datasize = mtk_snfi_data_to_gpram_size(op->data.nbytes, width,
					       op->data.buswidth);

	if (nbytes + datasize <= SNFI_GPRAM_SIZE)
		return 0;

	if (nbytes >= SNFI_GPRAM_SIZE)
		return -ENOTSUPP;

	datasize = SNFI_GPRAM_SIZE - nbytes;

	op->data.nbytes = mtk_snfi_gpram_to_data_size(datasize, width,
						      op->data.buswidth);
	if (!op->data.nbytes)
		return -ENOTSUPP;

	return 0;
}

static bool mtk_snfi_supports_op(struct spi_slave *slave,
				 const struct spi_mem_op *op)
{
	struct udevice *bus = dev_get_parent(slave->dev);
	struct mtk_snfi_priv *priv = dev_get_priv(bus);
	int ret;

	ret = mtk_snfi_get_io_width(priv, op);
	if (ret < 0)
		return false;

	return true;
}

static int mtk_snfi_mac_trigger(struct mtk_snfi_priv *priv, struct udevice *bus,
				u32 outlen, u32 inlen, bool quadio)
{
	int ret;
	u32 val, macctl = 0;

#ifdef CONFIG_PINCTRL
	if (!priv->fixed_pinctrl)
		pinctrl_select_state(bus, "snfi");
#endif

	if (quadio)
		macctl = MAC_XIO_SEL;

	writel(macctl | SF_MAC_EN, priv->base + SNFI_MAC_CTL);
	writel(outlen, priv->base + SNFI_MAC_OUTL);
	writel(inlen, priv->base + SNFI_MAC_INL);

	writel(macctl | SF_MAC_EN | SF_TRIG, priv->base + SNFI_MAC_CTL);

	ret = readl_poll_timeout(priv->base + SNFI_MAC_CTL, val,
				 val & WIP_READY, SNFI_POLL_INTERVAL);
	if (ret) {
		printf("%s: timed out waiting for WIP_READY\n", __func__);
		goto cleanup;
	}

	ret = readl_poll_timeout(priv->base + SNFI_MAC_CTL, val,
				 !(val & WIP), SNFI_POLL_INTERVAL);
	if (ret)
		printf("%s: timed out waiting for WIP cleared\n", __func__);

	writel(0, priv->base + SNFI_MAC_CTL);

cleanup:
#ifdef CONFIG_PINCTRL
	if (!priv->fixed_pinctrl)
		pinctrl_select_state(bus, "default");
#endif

	return ret;
}

static int mtk_snfi_mac_reset(struct mtk_snfi_priv *priv)
{
	int ret;
	u32 val;

	setbits_32(priv->base + SNFI_MISC_CTL, SW_RST);

	ret = readl_poll_timeout(priv->base + SNF_STA_CTL1, val,
				 !(val & SPI_STATE), SNFI_POLL_INTERVAL);
	if (ret)
		printf("%s: failed to reset snfi mac\n", __func__);

	writel((2 << FIFO_RD_LTC_SHIFT) |
		(10 << CS_DESELECT_CYC_SHIFT),
		priv->base + SNFI_MISC_CTL);

	return ret;
}

static u32 mtk_snfi_write_gpram_cache(const u8 *data, size_t len,
				      u8 *gpram_cache, u32 master_iowidth,
				      u32 slave_iowidth)
{
	u32 i, size = 0;

	if (master_iowidth == 4 && slave_iowidth == 1) {
		for (i = 0; i < len; i++) {
			size += mtk_snfi_sio_byte_to_quadio(data[i],
							   gpram_cache + size);
		}

		return size;
	}

	memcpy(gpram_cache, data, len);
	return len;
}

static void mtk_snfi_copy_to_gpram(struct mtk_snfi_priv *priv,
				   const void *data, size_t len)
{
	void __iomem *gpram = priv->base + SNFI_GPRAM_OFFSET;
	size_t i, n = (len + sizeof(u32) - 1) / sizeof(u32);
	const u32 *buff = data;

	/*
	 * The output data will always be copied to the beginning of
	 * the GPRAM. Uses word write for better performace.
	 *
	 * Trailing bytes in the last word are not cared.
	 */

	for (i = 0; i < n; i++)
		writel(buff[i], gpram + i * sizeof(u32));
}

static u32 mtk_snfi_copy_from_gpram(struct mtk_snfi_priv *priv, u8 *cache,
				     size_t pos, size_t len)
{
	void __iomem *gpram = priv->base + SNFI_GPRAM_OFFSET;
	u32 *buff = (u32 *)cache;
	size_t i, off, end;

	/* Start position in the buffer */
	off = pos & (sizeof(u32) - 1);

	/* End position for copy */
	end = (len + pos + sizeof(u32) - 1) & (~(sizeof(u32) - 1));

	/* Start position for copy */
	pos &= ~(sizeof(u32) - 1);

	/*
	 * Read aligned data from GPRAM to buffer first.
	 * Uses word read for better performace.
	 */
	i = 0;
	while (pos < end) {
		buff[i++] = readl(gpram + pos);
		pos += sizeof(u32);
	}

	/* Return offset in GPRAM */
	return off;
}

static int mtk_snfi_exec_op(struct spi_slave *slave,
			    const struct spi_mem_op *op)
{
	struct udevice *bus = dev_get_parent(slave->dev);
	struct mtk_snfi_priv *priv = dev_get_priv(bus);
	u8 ch, gpram_cache[SNFI_GPRAM_SIZE], *p;
	u32 i, len = 0, inlen = 0, off;
	int addr_sh, width, ret;

	WATCHDOG_RESET();

	ret = mtk_snfi_mac_reset(priv);
	if (ret)
		return ret;

	/* Choose the bus width to be used */
	width = mtk_snfi_get_io_width(priv, op);
	if (width < 0)
		return -ENOTSUPP;

	/* Put opcode */
	len += mtk_snfi_write_gpram_cache((u8 *)&op->cmd.opcode, 1, gpram_cache,
					  width, op->cmd.buswidth);

	/* Put address */
	addr_sh = (op->addr.nbytes - 1) * 8;
	while (addr_sh >= 0) {
		ch = (op->addr.val >> addr_sh) & 0xff;
		len += mtk_snfi_write_gpram_cache(&ch, 1, gpram_cache + len,
						  width, op->addr.buswidth);
		addr_sh -= 8;
	}

	/* Put dummy bytes */
	for (i = 0; i < op->dummy.nbytes; i++) {
		ch = 0;
		len += mtk_snfi_write_gpram_cache(&ch, 1, gpram_cache + len,
						  width, op->dummy.buswidth);
	}

	/* Put output data */
	if (op->data.nbytes && op->data.dir == SPI_MEM_DATA_OUT) {
		len += mtk_snfi_write_gpram_cache(op->data.buf.out,
						  op->data.nbytes,
						  gpram_cache + len,
						  width, op->data.buswidth);
	}

	/* Copy final output data to GPRAM */
	mtk_snfi_copy_to_gpram(priv, gpram_cache, len);

	/* Start one SPI transaction */
	if (op->data.nbytes && op->data.dir == SPI_MEM_DATA_IN) {
		inlen = mtk_snfi_gpram_to_data_size(op->data.nbytes, width,
						    op->data.buswidth);
	}

	ret = mtk_snfi_mac_trigger(priv, bus, len, inlen, width == 4);
	if (ret)
		return ret;

	if (!inlen)
		return 0;

	/* Copy input data from GPRAM */
	off = mtk_snfi_copy_from_gpram(priv, gpram_cache, len, inlen);

	if (unlikely(width == 4 && op->data.buswidth == 1)) {
		i = 0;
		p = op->data.buf.in;

		/* Convert SIO bytes in QIO mode */
		while (inlen >= 4) {
			p[i] = mtk_snfi_quadio_byte_to_sio(
				gpram_cache + off + i * 4);
			i++;
			inlen -= 4;
		}
	} else {
		memcpy(op->data.buf.in, gpram_cache + off, inlen);
	}

	return 0;
}

static int mtk_snfi_spi_probe(struct udevice *bus)
{
	struct mtk_snfi_priv *priv = dev_get_priv(bus);
	int ret;

	priv->base = (void __iomem *)devfdt_get_addr(bus);
	if (!priv->base)
		return -EINVAL;

	ret = clk_get_by_name(bus, "nfi_clk", &priv->nfi_clk);
	if (ret < 0)
		return ret;

	ret = clk_get_by_name(bus, "pad_clk", &priv->pad_clk);
	if (ret < 0)
		return ret;

	clk_enable(&priv->nfi_clk);
	clk_enable(&priv->pad_clk);

	priv->fixed_pinctrl = dev_read_bool(bus, "pinctrl-fixed");
	priv->quad_enabled = dev_read_bool(bus, "quad-io");

#ifdef CONFIG_PINCTRL
	if (priv->fixed_pinctrl)
		pinctrl_select_state(bus, "snfi");
#endif

	return 0;
}

static int mtk_snfi_set_speed(struct udevice *bus, uint speed)
{
	/*
	 * The SNFI does not have a bus clock divider.
	 * The bus clock is set in dts (pad_clk, UNIVPLL2_D8 = 50MHz).
	 */

	return 0;
}

static int mtk_snfi_set_mode(struct udevice *bus, uint mode)
{
	/* The SNFI supports only mode 0 */

	if (mode)
		return -EINVAL;

	return 0;
}

static const struct spi_controller_mem_ops mtk_snfi_mem_ops = {
	.adjust_op_size = mtk_snfi_adjust_op_size,
	.supports_op = mtk_snfi_supports_op,
	.exec_op = mtk_snfi_exec_op,
};

static const struct dm_spi_ops mtk_snfi_spi_ops = {
	.mem_ops	= &mtk_snfi_mem_ops,
	.set_speed	= mtk_snfi_set_speed,
	.set_mode	= mtk_snfi_set_mode,
};

static const struct udevice_id mtk_snfi_spi_ids[] = {
	{ .compatible = "mediatek,mtk-snfi-spi" },
	{ }
};

U_BOOT_DRIVER(mtk_snfi_spi) = {
	.name			= "mtk_snfi_spi",
	.id			= UCLASS_SPI,
	.of_match		= mtk_snfi_spi_ids,
	.ops			= &mtk_snfi_spi_ops,
	.priv_auto_alloc_size	= sizeof(struct mtk_snfi_priv),
	.probe			= mtk_snfi_spi_probe,
};
