// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2022 Alphanetwroks Inc. All Rights Reserved.
 *
 * Author: Jerry5566 <jerry_lai@alphanetworks.com>
 *
 * Upgrading firmware command for specific board WRG-AX26
 */

#include <command.h>
#include <env.h>
#include <image.h>
#include <linux/types.h>
#include <seama.h>

#include "load_data.h"
#include "colored_print.h"
#include "upgrade_helper.h"

#define ENV_NAME	"bootfileALPHA"

static unsigned int DstAddr;
static unsigned int DstOffset;

static int do_ALPHAupgrade(struct cmd_tbl *cmdtp, int flag, int argc,
			 char *const argv[])
{
	ulong data_load_addr;
	size_t data_size = 0;
	int ret = 0;
	char file_size[16];

	printf("\n");
	cprintln(PROMPT, "*** Upgrading SEAMA firmware ***");
	printf("\n");
	/* Set load address */
#if defined(CONFIG_SYS_LOAD_ADDR)
	data_load_addr = CONFIG_SYS_LOAD_ADDR;
#elif defined(CONFIG_LOADADDR)
	data_load_addr = CONFIG_LOADADDR;
#endif
	/* Load data */
	if (load_data(data_load_addr, &data_size, ENV_NAME))
		return CMD_RET_FAILURE;
	printf("\n");
	cprintln(PROMPT, "*** Loaded %zd (0x%zx) bytes at 0x%08lx ***",
		 data_size, data_size, data_load_addr);
	printf("\n");
	/* Validate data */
	DstAddr = data_load_addr;
	DstOffset = data_size;

	verify_seama((uchar *)DstAddr,DstOffset);
	if(SEAMA_valid == 1)
	{
		if(SEAMA_noheader)
		{
			DstAddr = (ulong)SEAMA_rawimage;
			DstOffset = ntohl(SEAMA_image->size);
		}
		else
		{
			DstAddr = (ulong)SEAMA_image;
			DstOffset = SEAMA_size;
		}
	}
	else
	{
		return CMD_RET_FAILURE;
	}
	cprintln(PROMPT, "*** DstAddr=0x%08x ,DstOffset=0x%x ***", DstAddr, DstOffset);
	printf("\n");
	/* clean flash before write data */
	run_command("mtd erase ubi_0", 0);
#ifdef CONFIG_DUAL_IMAGE
	run_command("mtd erase ubi_1", 0);
#endif
	/* Write data */
#ifdef CONFIG_DUAL_IMAGE
	ret = ubi_upgrade_image((const void *)DstAddr, DstOffset, 0);
	if (!ret)
		ret = ubi_upgrade_image((const void *)DstAddr, DstOffset, 1);
#else
	ret = ubi_upgrade_image((const void *)DstAddr, DstOffset);
#endif
	if(ret!=0) {
		cprintln(ERROR, "*** Upgrade failed ***");
		printf("\n");
		return CMD_RET_FAILURE;
	}

	snprintf(file_size, 16, "%ld", data_size);
#ifdef CONFIG_DUAL_IMAGE
	env_set("filesize_a", file_size);
	env_set("filesize_b", file_size);
#else
	env_set("filesize_a", file_size);
#endif
	env_save();
	printf("\n\n");
	cprintln(PROMPT, "*** Firmware upgrade completed! ***");
	printf("\n\n");
	if(confirm_yes("Reboot device? (Y/n):")) {
		return do_reset(NULL , 0 , 0 ,NULL);
	}
	else {
		return CMD_RET_SUCCESS;
	}
}

U_BOOT_CMD(upgradeFW, 1, 0, do_ALPHAupgrade, "ALPHA firmware upgrading utility", NULL);
